use std::fmt;
use std::fmt::Display;

#[cfg(feature = "read_write")]
use crate::read_write::component::Component;

#[derive(Debug, Clone, Eq, PartialEq, Default, Ord, PartialOrd)]
/// Simple struct to store a name
pub struct Name {
    pub given: Option<String>,
    pub additional: Option<String>,
    pub family: Option<String>,
    pub prefixes: Option<String>,
    pub suffixes: Option<String>,
}

impl Name {
    /// Creates a new Name
    #[must_use]
    pub const fn new(
        given: Option<String>,
        additional: Option<String>,
        family: Option<String>,
        prefixes: Option<String>,
        suffixes: Option<String>,
    ) -> Self {
        Self {
            given,
            additional,
            family,
            prefixes,
            suffixes,
        }
    }
}

// For .to_string()
impl Display for Name {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut string = "".to_string();

        if self.prefixes != None {
            string.push(' ');
            string.push_str(self.prefixes.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.given != None {
            string.push(' ');
            string.push_str(self.given.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.additional != None {
            string.push(' ');
            string.push_str(self.additional.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.family != None {
            string.push(' ');
            string.push_str(self.family.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.suffixes != None {
            string.push(' ');
            string.push_str(self.suffixes.as_ref().unwrap());
            string = string.trim().to_string();
        }

        write!(f, "{}", string)
    }
}

#[cfg(feature = "read_write")]
impl From<Name> for Component {
    fn from(name: Name) -> Self {
        Self {
            name: "N".to_string(),
            values: vec![
                name.family
                    .map(|x| x.split(' ').map(ToString::to_string).collect())
                    .unwrap_or_default(),
                name.given
                    .map(|x| x.split(' ').map(ToString::to_string).collect())
                    .unwrap_or_default(),
                name.additional
                    .map(|x| x.split(' ').map(ToString::to_string).collect())
                    .unwrap_or_default(),
                name.prefixes
                    .map(|x| x.split(' ').map(ToString::to_string).collect())
                    .unwrap_or_default(),
                name.suffixes
                    .map(|x| x.split(' ').map(ToString::to_string).collect())
                    .unwrap_or_default(),
            ],
            ..Self::default()
        }
    }
}

#[cfg(feature = "read_write")]
impl From<Component> for Name {
    fn from(mut comp: Component) -> Self {
        Self {
            family: comp.values.get_mut(0).and_then(|x| {
                x.pop().and_then(|x| match x.as_str() {
                    "" => None,
                    _ => Some(x),
                })
            }),
            given: comp.values.get_mut(1).and_then(|x| {
                x.pop().and_then(|x| match x.as_str() {
                    "" => None,
                    _ => Some(x),
                })
            }),
            additional: comp.values.get_mut(2).and_then(|x| {
                x.pop().and_then(|x| match x.as_str() {
                    "" => None,
                    _ => Some(x),
                })
            }),
            prefixes: comp.values.get_mut(3).and_then(|x| {
                x.pop().and_then(|x| match x.as_str() {
                    "" => None,
                    _ => Some(x),
                })
            }),
            suffixes: comp.values.get_mut(4).and_then(|x| {
                x.pop().and_then(|x| match x.as_str() {
                    "" => None,
                    _ => Some(x),
                })
            }),
        }
    }
}
