//! # `VCard`
//!
//! This contains the `VCard` struct which is used to be turned to and from
//! a text/vcard file.

use super::escape;
use crate::read_write::component::Component;
use std::fmt;
use std::str::FromStr;

#[derive(Clone, Default, Debug, Eq, PartialEq)]
pub struct VCard(pub Vec<Component>);

impl VCard {
    /// Creates a new `VCard`
    #[must_use]
    pub fn new(vec: Vec<Component>) -> Self {
        Self(vec)
    }
}

impl fmt::Display for VCard {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "BEGIN:VCARD\r\nVERSION:4.0\r\n")?;
        for comp in &self.0 {
            comp.fmt(f)?;
        }
        write!(f, "END:VCARD\r\n")
    }
}

impl FromStr for VCard {
    type Err = super::component::error::ComponentParseError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        // Unfold all the lines, so that we know what to give the components
        let unfolded = escape::unfold_line(s);

        // Vec to store the components
        let mut components: Vec<Component> = Vec::new();

        // Split it by lines
        for line in unfolded.lines() {
            components.push(line.parse()?);
        }

        // Remove the BEGIN and VERSION components.
        components.retain(|x| {
            x.name != "VERSION" && x.name != "BEGIN" && x.name != "END"
        });

        // Create a VCard
        Ok(Self(components))
    }
}
