//! # Contack
//! 
//! [![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
//! [![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)
//! 
//! Contack is a contact library for rust. Rather than following the RFCs exactly, it gives up some compatibility for ease of use. For example, instead of being able to express any number of job roles, Contack gives you the option of only 1, which intern is much easier to work with.
//! 
//! With the `read_write` feature you can have native support for serialisation and deserialisation, to VCard. This is achieved as follows:
//! 
//! ```rust
//! use contack::Contact;
//! use contack::read_write::vcard::VCard;
//! use std::fs::File;
//! use std::io::prelude::*;
//! use std::convert::TryInto;
//! 
//! fn main() -> Result<(), Box<dyn Error>> {
//!     // Load a VCard file
//!     let vcard = String::new();
//!     File::open("my_card.vcard")?.read_to_string(&mut vcard)?;
//!     
//!     // Serialise it
//!     let vcard: VCard = vcard.parse()?;
//!     
//!     // Convert it to a contact
//!     let contact: Contact = vcard.try_into()?;
//!     
//!     // Make some changes
//!     contact.role = Some("Being a handy example.");
//!     
//!     // Convert it to a VCard representation.
//!     let vcard: VCard = vcard.into();
//!     
//!     // Print it to the stdout
//!     println!("{}", vcard.to_string());
//! }
//! ```
//! 
//! It also supports the following external libraries:
//! 
//!  * [VCard](https://crates.io/crates/vcard) `vcard`. Note this only support serialisation, and is a stricter alternative to the inbuilt `read_write`
//!  * [Diesel](https://diesel.rs) `diesel_support`. This supports both serialisation and deserialisation.
//! 
//! 

#![warn(clippy::all, clippy::pedantic, clippy::nursery, clippy::cargo)]
#![allow(clippy::doc_markdown, clippy::module_name_repetitions)]

pub mod address;
pub mod contact_information;
pub mod contact_platform;
pub mod date_time;
pub mod name;
pub mod org;
pub mod uri;
#[cfg(feature = "to_vcard")]
pub mod vcard;

#[cfg(feature = "lazy_static")]
#[macro_use]
extern crate lazy_static;

#[cfg_attr(feature = "diesel-derive-enum", macro_use)]
#[cfg(feature = "diesel-derive-enum")]
extern crate diesel_derive_enum;

#[cfg(feature = "diesel_support")]
#[macro_use]
extern crate diesel;

#[cfg(feature = "diesel_support")]
#[macro_use]
extern crate diesel_migrations;

#[cfg(feature = "diesel_support")]
pub mod schema;

#[cfg(feature = "diesel_support")]
pub mod diesel_support;

#[cfg(feature = "diesel_support")]
use schema::*;

#[cfg(feature = "diesel_support")]
pub(crate) use diesel_support::DieselConversionError;

#[cfg(feature = "read_write")]
pub mod read_write;

#[cfg(feature = "read_write")]
use crate::read_write::error::FromComponentError;
#[cfg(feature = "read_write")]
use crate::read_write::component::Component;
#[cfg(feature = "read_write")]
use std::convert::{TryFrom, TryInto};

pub use {
    address::{Address, Geo},
    contact_information::{ContactInformation, Type},
    contact_platform::ContactPlatform,
    date_time::DateTime,
    name::Name,
    org::Org,
    uri::Uri,
};

/// This is the base structure to hold all contact details
#[derive(Debug, Clone, PartialEq, PartialOrd, Default)]
pub struct Contact {
    /*
     * Explanatory Properties
     */
    /// The Contact's Uid
    pub uid: String,

    /*
     * Identification Property
     */
    /// The Contacts Name
    pub name: name::Name,

    /// The Contact's Nickname
    pub nickname: Option<String>,

    /// The Contact's Anniversary
    pub anniversary: Option<date_time::DateTime>,

    /// The Contact's Birthday
    pub bday: Option<date_time::DateTime>,

    /// The Contact's Photo
    pub photo: Option<uri::Uri>,

    /*
     * Organisational Properties
     */
    /// The Contact's Job Title
    pub title: Option<String>,

    /// The Contact's Job Role
    pub role: Option<String>,

    /// The Contact's Organization
    pub org: Option<org::Org>,

    /// The Contact's Organization's Logo
    pub logo: Option<uri::Uri>,

    /*
     * Communication Properties
     */
    /// The Contact's contact information
    pub contact_information: Vec<contact_information::ContactInformation>,

    /*
     * Deliver Addressing Properties
     */
    /// The Contact's Work Address
    pub work_address: Option<address::Address>,

    /// The Contact's Home Address
    pub home_address: Option<address::Address>,
}

impl Contact {
    pub fn gen_uid(&mut self) {
        // Generates a UID from the UUID Value
        self.uid = uuid::Uuid::new_v4().to_string();
    }

    #[must_use]
    pub fn new(name: name::Name) -> Self {
        Self {
            name,
            nickname: None,
            anniversary: None,
            bday: None,
            photo: None,
            title: None,
            role: None,
            org: None,
            logo: None,
            contact_information: Vec::new(),
            work_address: None,
            home_address: None,
            uid: uuid::Uuid::new_v4().to_string(),
        }
    }
}

#[cfg(feature = "to_vcard")]
impl std::convert::TryInto<vcard::vcard::VCard> for Contact {
    type Error = Box<dyn std::error::Error>;

    fn try_into(self) -> Result<vcard::vcard::VCard, Self::Error> {
        vcard::contack_to_vcard(&self)
    }
}

#[cfg(feature = "read_write")]
impl From<Contact> for read_write::vcard::VCard {
    fn from(c: Contact) -> Self {
        // Vec to store propertiess
        let mut vec = vec![

            // Insert the UID
            Component {
                name: "UID".to_string(),
                values: vec![vec![c.uid]],
                ..Component::default()
            },

            // And the name (FN followed by N)
            Component {
                name: "FN".to_string(),
                values: vec![vec![c.name.to_string()]],
                ..Component::default()
            },
            c.name.into()
        ];

        // Nickname next
        if let Some(nickname) = c.nickname {
            vec.push(Component {
                name: "NICKNAME".to_string(),
                values: vec![vec![nickname]],
                ..Component::default()
            })
        }

        // Anniversary
        if let Some(anniversary) = c.anniversary {
            let mut anniversary: Component = anniversary.into();
            anniversary.name = "ANNIVERSARY".to_string();
            vec.push(anniversary)
        }

        // Bithday
        if let Some(bday) = c.bday {
            let mut bday: Component = bday.into();
            bday.name = "BDAY".to_string();
            vec.push(bday)
        }

        // Photo
        if let Some(photo) = c.photo {
            let mut photo: Component = photo.into();
            photo.name = "PHOTO".to_string();
            vec.push(photo)
        }

        // Title
        if let Some(title) = c.title {
            vec.push(Component {
                name: "TITLE".to_string(),
                values: vec![vec![title]],
                ..Component::default()
            })
        }

        // Role
        if let Some(role) = c.role {
            vec.push(Component {
                name: "ROLE".to_string(),
                values: vec![vec![role]],
                ..Component::default()
            })
        }

        // Org
        if let Some(org) = c.org {
            vec.push(org.into());
        }

        // Logo
        if let Some(logo) = c.logo {
            let mut logo: Component = logo.into();
            logo.name = "LOGO".to_string();
            vec.push(logo)
        }

        // Contact information
        for ci in c.contact_information {
            vec.push(ci.into());
        }

        // Work Address
        if let Some(adr) = c.work_address {
            let mut adr: Component = adr.into();
            adr.parameters
                .insert("TYPE".to_string(), "work".to_string());
            vec.push(adr);
        }

        // Home Address
        if let Some(adr) = c.home_address {
            let mut adr: Component = adr.into();
            adr.parameters
                .insert("TYPE".to_string(), "home".to_string());
            vec.push(adr);
        }

        Self::new(vec)
    }
}

#[cfg(feature = "read_write")]
impl TryFrom<read_write::vcard::VCard> for Contact {
    type Error = Box<dyn std::error::Error>;

    fn try_from(
        vcard: read_write::vcard::VCard,
    ) -> Result<Self, Self::Error> {
        // Create a contact - with no name (for now)
        let mut contact = Self::new(Name::default());

        // Loop through all the components
        for mut comp in vcard.0 {
            match comp.name.as_str() {
                "UID" => {
                    if let Some(uid) =
                        comp.values.pop().and_then(|mut x| x.pop())
                    {
                        contact.uid = uid
                    }
                }

                "FN" => {
                    if contact.name == Name::default() {
                        if let Some(name) =
                            comp.values.pop().and_then(|mut x| x.pop())
                        {
                            contact.name.given = Some(name);
                        }
                    }
                }

                "N" => contact.name = comp.into(),

                "NICKNAME" => {
                    contact.nickname = Some(
                        comp.values
                            .pop()
                            .and_then(|mut x| x.pop())
                            .ok_or(FromComponentError::NotEnoughValues)?,
                    )
                }

                "ANNIVERSARY" => contact.anniversary = Some(comp.try_into()?),

                "BDAY" => contact.bday = Some(comp.try_into()?),

                "PHOTO" => contact.photo = Some(comp.try_into()?),

                "TITLE" => {
                    contact.title = Some(
                        comp.values
                            .pop()
                            .and_then(|mut x| x.pop())
                            .ok_or(FromComponentError::NotEnoughValues)?,
                    )
                }

                "ROLE" => {
                    contact.role = Some(
                        comp.values
                            .pop()
                            .and_then(|mut x| x.pop())
                            .ok_or(FromComponentError::NotEnoughValues)?,
                    )
                }

                "ORG" => contact.org = Some(Org::from(comp)),

                "LOGO" => contact.logo = Some(comp.try_into()?),

                "EMAIL" | "TEL" | "X-DISCORD" | "X-MATRIX" | "X-SKYPE"
                | "X-AIM" | "X-JABBER" | "X-ICQ" | "X-GROUPWISE"
                | "X-GADUGADU" | "IMPP" => {
                    contact.contact_information.push(comp.try_into()?)
                }

                // Addresses.
                //
                // Note that if ambigouous it is assumed that an address
                // is home.
                "ADR" => match comp
                    .parameters
                    .get("TYPE")
                    .map_or("home", String::as_str)
                {
                    "work" => contact.work_address = Some(comp.try_into()?),
                    _ => contact.home_address = Some(comp.try_into()?),
                },
                // Values which we don't know shall be scrapped.
                _ => (),
            }
        }

        Ok(contact)
    }
}
