//! # Diesel Support.
//!
//! This is the main module for diesel support. `DieselContact` can store
//! all features of a normal Contact, par `contact_information`, which
//! can be found in [`DieselContactInformation`](crate::diesel_contact_information::DieselContactInformation.)
//!
//! Migrations are stored in an `embeded_migrations` variable.

use crate::*;
use diesel_migrations::embed_migrations;
use std::convert::{TryFrom, TryInto};
pub mod contact_information;
pub mod error;
pub use error::*;

/// This is a struct, mirroring Contact, but with all its parts 'exploded',
/// aside from ContactInformation which has to come separately.
#[derive(
    Insertable, Queryable, Identifiable, Default, Clone, PartialEq, Debug, AsChangeset
)]
#[primary_key(uid)]
#[table_name = "contacts"]
pub struct DieselContact {
    pub uid: String,
    pub name_given: Option<String>,
    pub name_additional: Option<String>,
    pub name_family: Option<String>,
    pub name_prefixes: Option<String>,
    pub name_suffixes: Option<String>,
    pub nickname: Option<String>,
    pub anniversary: Option<chrono::NaiveDateTime>,
    pub bday: Option<chrono::NaiveDateTime>,
    pub photo_uri: Option<String>,
    pub photo_bin: Option<Vec<u8>>,
    pub photo_mime: Option<String>,
    pub title: Option<String>,
    pub role: Option<String>,
    pub org_org: Option<String>,
    pub org_unit: Option<String>,
    pub org_office: Option<String>,
    pub logo_uri: Option<String>,
    pub logo_bin: Option<Vec<u8>>,
    pub logo_mime: Option<String>,
    pub home_address_street: Option<String>,
    pub home_address_locality: Option<String>,
    pub home_address_region: Option<String>,
    pub home_address_code: Option<String>,
    pub home_address_country: Option<String>,
    pub home_address_geo_longitude: Option<f64>,
    pub home_address_geo_latitude: Option<f64>,
    pub work_address_street: Option<String>,
    pub work_address_locality: Option<String>,
    pub work_address_region: Option<String>,
    pub work_address_code: Option<String>,
    pub work_address_country: Option<String>,
    pub work_address_geo_longitude: Option<f64>,
    pub work_address_geo_latitude: Option<f64>,
}

impl std::convert::TryFrom<DieselContact> for Contact {
    type Error = Box<dyn std::error::Error>;

    fn try_from(them: DieselContact) -> Result<Self, Self::Error> {
        Ok(Contact {
            uid: them.uid,

            // Name
            name: Name::new(
                them.name_given,
                them.name_additional,
                them.name_family,
                them.name_prefixes,
                them.name_suffixes,
            ),

            // Nickname
            nickname: them.nickname,

            // Anniversary
            anniversary: match them.anniversary {
                None => None,
                Some(x) => Some(DateTime::try_from(x)?),
            },

            // Birthday
            bday: match them.bday {
                None => None,
                Some(x) => Some(DateTime::try_from(x)?),
            },

            // Photo
            photo: Uri::try_from_sql_raw((
                them.photo_uri,
                them.photo_bin,
                them.photo_mime,
                String::from("photo"),
            ))?,

            // Title
            title: them.title,

            // Role
            role: them.role,

            // Org
            org: {
                if them.org_org.is_some()
                    && them.org_unit.is_some()
                    && them.org_office.is_some()
                {
                    Some(Org::new(
                        them.org_org.unwrap(),
                        them.org_unit.unwrap(),
                        them.org_office.unwrap(),
                    ))
                } else {
                    if them.org_org.is_none()
                        && them.org_unit.is_none()
                        && them.org_office.is_none()
                    {
                        None
                    } else {
                        return Err(Box::new(DieselConversionError::InvalidProperty(String::from("Some org propties are null, while some are not null."))));
                    }
                }
            },

            // Logo
            logo: Uri::try_from_sql_raw((
                them.logo_uri,
                them.logo_bin,
                them.logo_mime,
                String::from("logo"),
            ))?,

            home_address: address::AddressRaw::new(
                them.home_address_street,
                them.home_address_locality,
                them.home_address_region,
                them.home_address_code,
                them.home_address_country,
                them.home_address_geo_longitude,
                them.home_address_geo_latitude,
            )
            .try_from_sql_raw("home_address")?,
            work_address: address::AddressRaw::new(
                them.work_address_street,
                them.work_address_locality,
                them.work_address_region,
                them.work_address_code,
                them.work_address_country,
                them.work_address_geo_longitude,
                them.work_address_geo_latitude,
            )
            .try_from_sql_raw("work_address")?,
            contact_information: vec![],
        })
    }
}
impl TryFrom<Contact> for DieselContact {
    type Error = Box<dyn std::error::Error>;

    fn try_from(them: Contact) -> Result<Self, Self::Error> {
        // Get the photo properties.
        let (photo_uri, photo_bin, photo_mime) = Uri::to_sql_raw(them.photo);

        // Get the logo properties.
        let (logo_uri, logo_bin, logo_mime) = Uri::to_sql_raw(them.logo);

        // Get the org properties.
        let (org_org, org_unit, org_office) = match them.org {
            Some(org) => (Some(org.org), Some(org.unit), Some(org.office)),
            None => (None, None, None),
        };

        // Get the address properties.
        let (
            home_address_street,
            home_address_locality,
            home_address_region,
            home_address_code,
            home_address_country,
            (home_address_geo_longitude, home_address_geo_latitude),
        ) = Address::to_sql_raw(them.home_address);

        let (
            work_address_street,
            work_address_locality,
            work_address_region,
            work_address_code,
            work_address_country,
            (work_address_geo_longitude, work_address_geo_latitude),
        ) = Address::to_sql_raw(them.work_address);

        Ok(Self {
            // Get the Uid
            uid: them.uid,

            // Get the names
            name_given: them.name.given,
            name_additional: them.name.additional,
            name_family: them.name.family,
            name_prefixes: them.name.prefixes,
            name_suffixes: them.name.suffixes,
            nickname: them.nickname,

            // Get the time based properties
            anniversary: match them.anniversary {
                Some(x) => x.try_into()?,
                None => None,
            },
            bday: match them.bday {
                Some(x) => x.try_into()?,
                None => None,
            },

            // Gets the photo properties.
            photo_uri,
            photo_bin,
            photo_mime,

            // Title
            title: them.title,

            // Role
            role: them.role,

            // Org
            org_org,
            org_unit,
            org_office,

            // Logo Properties
            logo_uri,
            logo_bin,
            logo_mime,

            // Home Address
            home_address_street,
            home_address_locality,
            home_address_region,
            home_address_code,
            home_address_country,
            home_address_geo_longitude,
            home_address_geo_latitude,

            // Work Address
            work_address_street,
            work_address_locality,
            work_address_region,
            work_address_code,
            work_address_country,
            work_address_geo_longitude,
            work_address_geo_latitude,
        })
    }
}

/// This is a struct with the bare minimum fields for contact, being
/// name and uid. This can be inserted into an sql database and then
/// converted into a DieselContact, which intern can be converted into
/// a [crate::Contact](`Contact`)
#[derive(Insertable, Clone, Eq, Debug, PartialEq)]
#[table_name = "contacts"]
pub struct NewDieselContact {
    uid: String,
    name_given: Option<String>,
    name_additional: Option<String>,
    name_family: Option<String>,
    name_prefixes: Option<String>,
    name_suffixes: Option<String>,
}

impl NewDieselContact {
    pub fn new(name: Name) -> Self {
        NewDieselContact {
            uid: uuid::Uuid::new_v4().to_string(),
            name_given: name.given,
            name_additional: name.additional,
            name_family: name.family,
            name_prefixes: name.prefixes,
            name_suffixes: name.suffixes,
        }
    }
}

impl From<NewDieselContact> for DieselContact {
    fn from(them: NewDieselContact) -> Self {
        let mut contact = DieselContact::default();

        contact.uid = them.uid;
        contact.name_given = them.name_given;
        contact.name_additional = them.name_additional;
        contact.name_family = them.name_family;
        contact.name_prefixes = them.name_prefixes;
        contact.name_suffixes = them.name_suffixes;

        contact
    }
}

embed_migrations!("migrations/");
pub mod migrations {
    pub use super::embedded_migrations::*;
}
