#[cfg(feature = "diesel")]
use crate::DieselConversionError;
use std::vec::Vec;

/// A struct which can either store byte information or a url
#[cfg_attr(
    feature = "diesel",
    derive(FromSqlRow, AsExpression),
    sql_type = "diesel_mod::PgUri"
)]
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum Uri {
    Url { url: String },
    Bytes { val: Vec<u8>, mime: String },
}

impl Uri {
    /// Creates a new url from a string
    #[must_use]
    pub const fn new_url(string: String) -> Self {
        Self::Url { url: string }
    }

    /// Creates a new bytes from a vec
    #[must_use]
    pub const fn new_bytes(bytes: Vec<u8>, mime: String) -> Self {
        Self::Bytes { val: bytes, mime }
    }

    /// Creates a new bytes from a &[u8]
    #[must_use]
    pub fn new_bytes_array(bytes: &[u8], mime: String) -> Self {
        Self::Bytes {
            val: bytes.to_vec(),
            mime,
        }
    }
}

#[cfg(feature = "diesel_support")]
impl Uri {
    /// Converts an Uri into something which can be used by a `DieselContact`
    pub(crate) fn to_sql_raw(
        this: Option<Self>,
    ) -> (Option<String>, Option<Vec<u8>>, Option<String>) {
        match this {
            Some(photo) => match photo {
                Uri::Url { url } => (Some(url), None, None),
                Uri::Bytes { val, mime } => (None, Some(val), Some(mime)),
            },
            None => (None, None, None),
        }
    }

    /// This should only be used internally.
    ///
    /// The first `Option<String>` is the url, the `Option<Vec<u8>>` a binary
    /// value and the final `Option<String>` is the mime. The last `String` is
    /// for path, when given an error.
    pub(crate) fn try_from_sql_raw(
        tup: (Option<String>, Option<Vec<u8>>, Option<String>, String),
    ) -> Result<Option<Uri>, DieselConversionError> {
        match tup.0 {
            // Standard URL
            Some(val) => Ok(Some(Uri::new_url(val))),

            // Binary Type
            None => match tup.1 {
                Some(bin) => match tup.2 {
                    Some(mime) => Ok(Some(Uri::new_bytes(bin, mime))),
                    None => Err(DieselConversionError::InvalidProperty(format!(
                        "{3}(Uri) {} is set to null, \n\
                            but {3}(Uri) {} has the value of {:#?}. A Uri url value has also not been \n\
                            given and so the enum is invalid.",
                        "mime", "bin", bin, tup.3
                    ))),
                },
                None => match tup.2 {
                    None => return Ok(None),
                    Some(mime) => Err(DieselConversionError::InvalidProperty(format!(
                        "{3}(Uri) {} is set to null, \n\
                            but {3}(Uri) {} has the value of {:#?}. A Uri url value has also not been \n\
                            given and so the enum is invalid.",
                        "bin", "mime", mime, tup.3
                    ))),
                },
            },
        }
    }
}
