use std::fmt;
use std::fmt::Display;

#[cfg_attr(
    feature = "diesel",
    derive(FromSqlRow, AsExpression),
    sql_type = "diesel_mod::PgName"
)]
#[derive(Clone, Debug, Eq, PartialEq)]
/// Simple struct to store a name
pub struct Name {
    pub given: Option<String>,
    pub additional: Option<String>,
    pub family: Option<String>,
    pub prefixes: Option<String>,
    pub suffixes: Option<String>,
}

impl Name {
    /// Creates a new Name
    #[must_use]
    pub const fn new(
        given: Option<String>,
        additional: Option<String>,
        family: Option<String>,
        prefixes: Option<String>,
        suffixes: Option<String>,
    ) -> Self {
        Self {
            given,
            additional,
            family,
            prefixes,
            suffixes,
        }
    }
}

// For .to_string()
impl Display for Name {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut string = "".to_string();

        if self.prefixes != None {
            string.push(' ');
            string.push_str(self.prefixes.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.given != None {
            string.push(' ');
            string.push_str(self.given.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.additional != None {
            string.push(' ');
            string.push_str(self.additional.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.family != None {
            string.push(' ');
            string.push_str(self.family.as_ref().unwrap());
            string = string.trim().to_string();
        }

        if self.suffixes != None {
            string.push(' ');
            string.push_str(self.suffixes.as_ref().unwrap());
            string = string.trim().to_string();
        }

        write!(f, "{}", string)
    }
}

#[cfg(feature = "diesel")]
pub mod diesel_mod {
    // Imports
    use super::*;
    use diesel::{
        deserialize,
        deserialize::FromSql,
        pg::{types::sql_types::Record, Pg},
        serialize,
        serialize::{Output, ToSql, WriteTuple},
        sql_types::{Nullable, Text},
    };
    use std::io::Write;

    // Postgres Name
    #[derive(SqlType)]
    #[postgres(type_name = "CONTACT_NAME")]
    pub struct PgName;

    impl FromSql<PgName, Pg> for Name {
        fn from_sql(bytes: Option<&[u8]>) -> deserialize::Result<Self> {
            let (given, additional, family, prefixes, suffixes) =
                FromSql::<
                    Record<(
                        Nullable<Text>,
                        Nullable<Text>,
                        Nullable<Text>,
                        Nullable<Text>,
                        Nullable<Text>,
                    )>,
                    Pg,
                >::from_sql(bytes)?;

            Ok(Name::new(given, additional, family, prefixes, suffixes))
        }
    }

    impl ToSql<PgName, Pg> for Name {
        fn to_sql<W: Write>(
            &self,
            out: &mut Output<W, Pg>,
        ) -> serialize::Result {
            WriteTuple::<(
                Nullable<Text>,
                Nullable<Text>,
                Nullable<Text>,
                Nullable<Text>,
                Nullable<Text>,
            )>::write_tuple(
                &(
                    self.given.clone(),
                    self.additional.clone(),
                    self.family.clone(),
                    self.prefixes.clone(),
                    self.suffixes.clone(),
                ),
                out,
            )
        }
    }
}
