#[cfg(feature = "chrono")]
use chrono::{Datelike, NaiveDate, NaiveDateTime, Timelike};
use std::convert::TryFrom;
use std::convert::TryInto;

#[cfg_attr(
    feature = "diesel",
    derive(FromSqlRow, AsExpression),
    sql_type = "diesel::sql_types::Timestamp"
)]
#[derive(Eq, PartialEq, Clone, Debug, Ord, PartialOrd)]
pub struct DateTime {
    pub year: i32,
    pub month: u8,
    pub day: u8,
    pub hour: u8,
    pub minute: u8,
    pub second: u8,
}

impl DateTime {
    #[must_use]
    pub const fn new(
        year: i32,
        month: u8,
        day: u8,
        hour: u8,
        minute: u8,
        second: u8,
    ) -> Self {
        Self {
            year,
            month,
            day,
            hour,
            minute,
            second,
        }
    }
}

pub struct DateTimeError(());

// To Convert to Chrono UTC
#[cfg(feature = "chrono")]
impl Into<Option<chrono::NaiveDateTime>> for DateTime {
    fn into(self) -> Option<chrono::NaiveDateTime> {
        NaiveDate::from_ymd_opt(
            self.year.into(),
            self.month.into(),
            self.day.into(),
        )
        .map(|x| {
            x.and_hms(self.hour.into(), self.minute.into(), self.second.into())
        })
    }
}

// To Convert from Chrono UTC
#[cfg(feature = "chrono")]
impl TryFrom<chrono::NaiveDateTime> for DateTime {
    type Error = std::num::TryFromIntError;

    fn try_from(dt: chrono::NaiveDateTime) -> Result<Self, Self::Error> {
        Ok(Self::new(
            dt.year().try_into()?,
            dt.month().try_into()?,
            dt.day().try_into()?,
            dt.hour().try_into()?,
            dt.minute().try_into()?,
            dt.second().try_into()?,
        ))
    }
}

// For Diesel Support
#[cfg(feature = "diesel")]
pub mod diesel_mod {
    // Imports
    use super::*;
    use diesel::{
        backend::Backend,
        deserialize,
        deserialize::FromSql,
        pg::Pg,
        serialize,
        serialize::{Output, ToSql},
        sql_types::Timestamp,
    };

    use std::io::Write;

    impl FromSql<Timestamp, Pg> for DateTime {
        fn from_sql(bytes: Option<&[u8]>) -> deserialize::Result<Self> {
            let datetime: chrono::NaiveDateTime =
                FromSql::<Timestamp, Pg>::from_sql(bytes)?;

            Ok(datetime.try_into()?)
        }
    }

    impl<DB> ToSql<Timestamp, DB> for DateTime
    where
        DB: Backend,
        NaiveDateTime: ToSql<Timestamp, DB>,
    {
        fn to_sql<W: Write>(
            &self,
            out: &mut Output<W, DB>,
        ) -> serialize::Result {
            let dt: Option<NaiveDateTime> = self.clone().into();
            let dt: NaiveDateTime = dt.unwrap();
            dt.to_sql(out)
        }
    }
}
