use crate::contact_platform::ContactPlatform;
use std::convert::TryFrom;

#[cfg_attr(
    feature = "diesel",
    derive(FromSqlRow, AsExpression),
    sql_type = "diesel_mod::PgContactInformation"
)]
#[derive(PartialEq, Eq, Debug, Clone)]
pub struct ContactInformation {
    // The pid, used for representing this
    pub pid: String,

    /// The Preference Value
    pub pref: u8,

    /// The Value (for example johndoe@example.com)
    pub value: String,

    /// The platform which this is on
    pub platform: ContactPlatform,

    /// The type
    pub typ: Option<Type>,
}

#[derive(Eq, PartialEq, Debug, Copy, Clone)]
#[cfg_attr(
    feature = "diesel-derive-enum",
    derive(DbEnum),
    DieselType = "Contack_contact_information_type",
    db_rename = "Contack_contact_information_type"
)]
pub enum Type {
    Work,
    Home,
}

impl ContactInformation {
    /// Creates a new Contact Information
    #[must_use]
    pub fn new(string: String, platform: ContactPlatform) -> Self {
        Self {
            pref: 0,
            value: string,
            pid: uuid::Uuid::new_v4().to_string(),
            platform,
            typ: None,
        }
    }
}

#[cfg(feature = "diesel")]
pub mod diesel_mod {
    // Imports
    use super::*;
    use crate::contact_platform::ContactPlatformMapping;
    use diesel::{
        deserialize,
        deserialize::FromSql,
        pg::{types::sql_types::Record, Pg},
        serialize,
        serialize::{Output, ToSql, WriteTuple},
        sql_types::{Integer, Nullable, Text},
    };
    use std::io::Write;

    // Postgres Name
    #[derive(SqlType)]
    #[postgres(type_name = "CONTACT_CONTACT_INFORMATION")]
    pub struct PgContactInformation;

    impl FromSql<PgContactInformation, Pg> for ContactInformation {
        fn from_sql(bytes: Option<&[u8]>) -> deserialize::Result<Self> {
            let (pref, value, pid, platform, typ): (i32, _, _, _, _) =
                FromSql::<
                    Record<(
                        Integer,
                        Text,
                        Text,
                        ContactPlatformMapping,
                        Nullable<TypeMapping>,
                    )>,
                    Pg,
                >::from_sql(bytes)?;

            Ok(ContactInformation {
                pref: u8::try_from(pref)?,
                value,
                pid,
                platform,
                typ,
            })
        }
    }

    impl ToSql<PgContactInformation, Pg> for ContactInformation {
        fn to_sql<W: Write>(
            &self,
            out: &mut Output<W, Pg>,
        ) -> serialize::Result {
            WriteTuple::<(
                Integer,
                Text,
                Text,
                ContactPlatformMapping,
                Nullable<TypeMapping>,
            )>::write_tuple(
                &(
                    i32::from(self.pref),
                    self.value.clone(),
                    self.pid.clone(),
                    self.platform.clone(),
                    self.typ.clone(),
                ),
                out,
            )
        }
    }
}
