//! # Address
//!
//! This modules contains structures to do with addresses, positions and
//! suchlike.

#[cfg(feature = "diesel")]
use crate::DieselConversionError;

#[derive(PartialEq)]
#[non_exhaustive]
/// A simple structure to hold address information
pub struct Address {
    /// This is the street where the address points to.
    pub street: Option<String>,

    /// This is the locality/city where the address points to.
    pub locality: Option<String>,

    /// This is the region/county where the address points to.
    pub region: Option<String>,

    /// This is the code where the address is. This may be a zip/postal code.
    pub code: Option<String>,

    /// This is the country where ths address points to.
    pub country: Option<String>,

    /// This is the longitude and latitude of the address.
    pub geo: Option<Geo>,
}

#[cfg_attr(
    feature = "diesel",
    derive(FromSqlRow, AsExpression),
    sql_type = "diesel_mod::PgGeo"
)]
#[derive(PartialOrd, PartialEq, Clone, Debug)]
#[non_exhaustive]
/// A simple structure to hold Geo location.
pub struct Geo {
    /// This is the longitude for the geo-location.
    pub longitude: f64,

    /// This is the latitude for the geo-location.
    pub latitude: f64,
}

impl Address {
    /// Creates a new address
    #[must_use]
    pub const fn new(
        street: Option<String>,
        locality: Option<String>,
        region: Option<String>,
        code: Option<String>,
        country: Option<String>,
    ) -> Self {
        Self {
            street,
            locality,
            region,
            code,
            country,
            geo: None,
        }
    }
}

impl Geo {
    #[must_use]
    pub const fn new(longitude: f64, latitude: f64) -> Self {
        Self {
            longitude,
            latitude,
        }
    }
}

#[cfg(feature = "diesel_support")]
pub(crate) struct AddressRaw {
    pub street: Option<String>,
    pub locality: Option<String>,
    pub region: Option<String>,
    pub code: Option<String>,
    pub country: Option<String>,
    pub geo_long: Option<f64>,
    pub geo_lat: Option<f64>,
}

#[cfg(feature = "diesel_support")]
impl AddressRaw {
    /// Creates a new `AddressRaw`
    pub(crate) fn new(
        street: Option<String>,
        locality: Option<String>,
        region: Option<String>,
        code: Option<String>,
        country: Option<String>,
        geo_long: Option<f64>,
        geo_lat: Option<f64>,
    ) -> Self {
        Self {
            street,
            locality,
            region,
            code,
            country,
            geo_long,
            geo_lat,
        }
    }
}

#[cfg(feature = "diesel_support")]
impl Address {
    /// Converts an address to something that can be used by a DieselContact.
    pub(crate) fn to_sql_raw(
        this: Option<Self>,
    ) -> (
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
        Option<String>,
        (Option<f64>, Option<f64>),
    ) {
        match this {
            None => (None, None, None, None, None, (None, None)),
            Some(adr) => (
                adr.street,
                adr.locality,
                adr.region,
                adr.code,
                adr.country,
                match adr.geo {
                    None => (None, None),
                    Some(geo) => (Some(geo.longitude), Some(geo.latitude)),
                },
            ),
        }
    }
}

#[cfg(feature = "diesel_support")]
impl AddressRaw {
    /// This should only be used internally.
    pub(crate) fn try_from_sql_raw(
        self,
        field_name: &'static str,
    ) -> Result<Option<Address>, DieselConversionError> {
        let address = Address {
            // Set up the basic fields
            street: self.street,
            locality: self.locality,
            region: self.region,
            code: self.code,
            country: self.country,

            // Set up the Geo
            geo: {
                // Checks if the Geo is none
                if self.geo_long.is_none() && self.geo_lat.is_none() {
                    None
                } else {
                    // Otherwise make sure that it is some.
                    if self.geo_long.is_some() {
                        if self.geo_lat.is_some() {
                            // Creates the Geo
                            Some(Geo::new(
                                self.geo_long.unwrap(),
                                self.geo_lat.unwrap(),
                            ))
                        } else {
                            return Err(
                                DieselConversionError::InvalidProperty(
                                    String::from(format!(
                                        "Incomplete Geo in {}, \
                                            missing latitude",
                                        field_name
                                    )),
                                ),
                            );
                        }
                    } else {
                        if self.geo_lat.is_some() {
                            return Err(
                                DieselConversionError::InvalidProperty(
                                    String::from(format!(
                                        "Incomplete Geo in {}, \
                                            missing longitude",
                                        field_name
                                    )),
                                ),
                            );
                        } else {
                            None
                        }
                    }
                }
            },
        };

        // Check if the address has all fields.
        if address.street.is_none()
            && address.locality.is_none()
            && address.region.is_none()
            && address.code.is_none()
            && address.country.is_none()
            && address.geo.is_none()
        {
            Ok(None)
        } else {
            Ok(Some(address))
        }
    }
}
