// Constellation
//
// Pluggable authoritative DNS server
// Copyright: 2018, Valerian Saliou <valerian@valeriansaliou.name>
// License: Mozilla Public License v2.0 (MPL v2.0)

use serde_derive::{Deserialize, Serialize};

use super::region::RegionCode;

//- Source: https://github.com/lukes/ISO-3166-Countries-with-Regional-Codes/blob/master/all/all.csv
//- Last update: 13 August 2018

#[derive(Serialize, Deserialize, Clone, Debug, Hash, Eq, PartialEq)]
pub enum CountryCode {
    AF,
    AX,
    AL,
    DZ,
    AS,
    AD,
    AO,
    AI,
    AG,
    AR,
    AM,
    AW,
    AU,
    AT,
    AZ,
    BS,
    BH,
    BD,
    BB,
    BY,
    BE,
    BZ,
    BJ,
    BM,
    BT,
    BO,
    BQ,
    BA,
    BW,
    BV,
    BR,
    IO,
    BN,
    BG,
    BF,
    BI,
    CV,
    KH,
    CM,
    CA,
    KY,
    CF,
    TD,
    CL,
    CN,
    CX,
    CC,
    CO,
    KM,
    CG,
    CD,
    CK,
    CR,
    CI,
    HR,
    CU,
    CW,
    CY,
    CZ,
    DK,
    DJ,
    DM,
    DO,
    EC,
    EG,
    SV,
    GQ,
    ER,
    EE,
    SZ,
    ET,
    FK,
    FO,
    FJ,
    FI,
    FR,
    GF,
    PF,
    TF,
    GA,
    GM,
    GE,
    DE,
    GH,
    GI,
    GR,
    GL,
    GD,
    GP,
    GU,
    GT,
    GG,
    GN,
    GW,
    GY,
    HT,
    HM,
    VA,
    HN,
    HK,
    HU,
    IS,
    IN,
    ID,
    IR,
    IQ,
    IE,
    IM,
    IL,
    IT,
    JM,
    JP,
    JE,
    JO,
    KZ,
    KE,
    KI,
    KP,
    KR,
    KW,
    KG,
    LA,
    LV,
    LB,
    LS,
    LR,
    LY,
    LI,
    LT,
    LU,
    MO,
    MK,
    MG,
    MW,
    MY,
    MV,
    ML,
    MT,
    MH,
    MQ,
    MR,
    MU,
    YT,
    MX,
    FM,
    MD,
    MC,
    MN,
    ME,
    MS,
    MA,
    MZ,
    MM,
    NA,
    NR,
    NP,
    NL,
    NC,
    NZ,
    NI,
    NE,
    NG,
    NU,
    NF,
    MP,
    NO,
    OM,
    PK,
    PW,
    PS,
    PA,
    PG,
    PY,
    PE,
    PH,
    PN,
    PL,
    PT,
    PR,
    QA,
    RE,
    RO,
    RU,
    RW,
    BL,
    SH,
    KN,
    LC,
    MF,
    PM,
    VC,
    WS,
    SM,
    ST,
    SA,
    SN,
    RS,
    SC,
    SL,
    SG,
    SX,
    SK,
    SI,
    SB,
    SO,
    ZA,
    GS,
    SS,
    ES,
    LK,
    SD,
    SR,
    SJ,
    SE,
    CH,
    SY,
    TW,
    TJ,
    TZ,
    TH,
    TL,
    TG,
    TK,
    TO,
    TT,
    TN,
    TR,
    TM,
    TC,
    TV,
    UG,
    UA,
    AE,
    GB,
    US,
    UM,
    UY,
    UZ,
    VU,
    VE,
    VN,
    VG,
    VI,
    WF,
    EH,
    YE,
    ZM,
    ZW,
}

impl CountryCode {
    pub fn from_str(code: &str) -> Option<CountryCode> {
        match code {
            "AF" => Some(CountryCode::AF),
            "AX" => Some(CountryCode::AX),
            "AL" => Some(CountryCode::AL),
            "DZ" => Some(CountryCode::DZ),
            "AS" => Some(CountryCode::AS),
            "AD" => Some(CountryCode::AD),
            "AO" => Some(CountryCode::AO),
            "AI" => Some(CountryCode::AI),
            "AG" => Some(CountryCode::AG),
            "AR" => Some(CountryCode::AR),
            "AM" => Some(CountryCode::AM),
            "AW" => Some(CountryCode::AW),
            "AU" => Some(CountryCode::AU),
            "AT" => Some(CountryCode::AT),
            "AZ" => Some(CountryCode::AZ),
            "BS" => Some(CountryCode::BS),
            "BH" => Some(CountryCode::BH),
            "BD" => Some(CountryCode::BD),
            "BB" => Some(CountryCode::BB),
            "BY" => Some(CountryCode::BY),
            "BE" => Some(CountryCode::BE),
            "BZ" => Some(CountryCode::BZ),
            "BJ" => Some(CountryCode::BJ),
            "BM" => Some(CountryCode::BM),
            "BT" => Some(CountryCode::BT),
            "BO" => Some(CountryCode::BO),
            "BQ" => Some(CountryCode::BQ),
            "BA" => Some(CountryCode::BA),
            "BW" => Some(CountryCode::BW),
            "BV" => Some(CountryCode::BV),
            "BR" => Some(CountryCode::BR),
            "IO" => Some(CountryCode::IO),
            "BN" => Some(CountryCode::BN),
            "BG" => Some(CountryCode::BG),
            "BF" => Some(CountryCode::BF),
            "BI" => Some(CountryCode::BI),
            "CV" => Some(CountryCode::CV),
            "KH" => Some(CountryCode::KH),
            "CM" => Some(CountryCode::CM),
            "CA" => Some(CountryCode::CA),
            "KY" => Some(CountryCode::KY),
            "CF" => Some(CountryCode::CF),
            "TD" => Some(CountryCode::TD),
            "CL" => Some(CountryCode::CL),
            "CN" => Some(CountryCode::CN),
            "CX" => Some(CountryCode::CX),
            "CC" => Some(CountryCode::CC),
            "CO" => Some(CountryCode::CO),
            "KM" => Some(CountryCode::KM),
            "CG" => Some(CountryCode::CG),
            "CD" => Some(CountryCode::CD),
            "CK" => Some(CountryCode::CK),
            "CR" => Some(CountryCode::CR),
            "CI" => Some(CountryCode::CI),
            "HR" => Some(CountryCode::HR),
            "CU" => Some(CountryCode::CU),
            "CW" => Some(CountryCode::CW),
            "CY" => Some(CountryCode::CY),
            "CZ" => Some(CountryCode::CZ),
            "DK" => Some(CountryCode::DK),
            "DJ" => Some(CountryCode::DJ),
            "DM" => Some(CountryCode::DM),
            "DO" => Some(CountryCode::DO),
            "EC" => Some(CountryCode::EC),
            "EG" => Some(CountryCode::EG),
            "SV" => Some(CountryCode::SV),
            "GQ" => Some(CountryCode::GQ),
            "ER" => Some(CountryCode::ER),
            "EE" => Some(CountryCode::EE),
            "SZ" => Some(CountryCode::SZ),
            "ET" => Some(CountryCode::ET),
            "FK" => Some(CountryCode::FK),
            "FO" => Some(CountryCode::FO),
            "FJ" => Some(CountryCode::FJ),
            "FI" => Some(CountryCode::FI),
            "FR" => Some(CountryCode::FR),
            "GF" => Some(CountryCode::GF),
            "PF" => Some(CountryCode::PF),
            "TF" => Some(CountryCode::TF),
            "GA" => Some(CountryCode::GA),
            "GM" => Some(CountryCode::GM),
            "GE" => Some(CountryCode::GE),
            "DE" => Some(CountryCode::DE),
            "GH" => Some(CountryCode::GH),
            "GI" => Some(CountryCode::GI),
            "GR" => Some(CountryCode::GR),
            "GL" => Some(CountryCode::GL),
            "GD" => Some(CountryCode::GD),
            "GP" => Some(CountryCode::GP),
            "GU" => Some(CountryCode::GU),
            "GT" => Some(CountryCode::GT),
            "GG" => Some(CountryCode::GG),
            "GN" => Some(CountryCode::GN),
            "GW" => Some(CountryCode::GW),
            "GY" => Some(CountryCode::GY),
            "HT" => Some(CountryCode::HT),
            "HM" => Some(CountryCode::HM),
            "VA" => Some(CountryCode::VA),
            "HN" => Some(CountryCode::HN),
            "HK" => Some(CountryCode::HK),
            "HU" => Some(CountryCode::HU),
            "IS" => Some(CountryCode::IS),
            "IN" => Some(CountryCode::IN),
            "ID" => Some(CountryCode::ID),
            "IR" => Some(CountryCode::IR),
            "IQ" => Some(CountryCode::IQ),
            "IE" => Some(CountryCode::IE),
            "IM" => Some(CountryCode::IM),
            "IL" => Some(CountryCode::IL),
            "IT" => Some(CountryCode::IT),
            "JM" => Some(CountryCode::JM),
            "JP" => Some(CountryCode::JP),
            "JE" => Some(CountryCode::JE),
            "JO" => Some(CountryCode::JO),
            "KZ" => Some(CountryCode::KZ),
            "KE" => Some(CountryCode::KE),
            "KI" => Some(CountryCode::KI),
            "KP" => Some(CountryCode::KP),
            "KR" => Some(CountryCode::KR),
            "KW" => Some(CountryCode::KW),
            "KG" => Some(CountryCode::KG),
            "LA" => Some(CountryCode::LA),
            "LV" => Some(CountryCode::LV),
            "LB" => Some(CountryCode::LB),
            "LS" => Some(CountryCode::LS),
            "LR" => Some(CountryCode::LR),
            "LY" => Some(CountryCode::LY),
            "LI" => Some(CountryCode::LI),
            "LT" => Some(CountryCode::LT),
            "LU" => Some(CountryCode::LU),
            "MO" => Some(CountryCode::MO),
            "MK" => Some(CountryCode::MK),
            "MG" => Some(CountryCode::MG),
            "MW" => Some(CountryCode::MW),
            "MY" => Some(CountryCode::MY),
            "MV" => Some(CountryCode::MV),
            "ML" => Some(CountryCode::ML),
            "MT" => Some(CountryCode::MT),
            "MH" => Some(CountryCode::MH),
            "MQ" => Some(CountryCode::MQ),
            "MR" => Some(CountryCode::MR),
            "MU" => Some(CountryCode::MU),
            "YT" => Some(CountryCode::YT),
            "MX" => Some(CountryCode::MX),
            "FM" => Some(CountryCode::FM),
            "MD" => Some(CountryCode::MD),
            "MC" => Some(CountryCode::MC),
            "MN" => Some(CountryCode::MN),
            "ME" => Some(CountryCode::ME),
            "MS" => Some(CountryCode::MS),
            "MA" => Some(CountryCode::MA),
            "MZ" => Some(CountryCode::MZ),
            "MM" => Some(CountryCode::MM),
            "NA" => Some(CountryCode::NA),
            "NR" => Some(CountryCode::NR),
            "NP" => Some(CountryCode::NP),
            "NL" => Some(CountryCode::NL),
            "NC" => Some(CountryCode::NC),
            "NZ" => Some(CountryCode::NZ),
            "NI" => Some(CountryCode::NI),
            "NE" => Some(CountryCode::NE),
            "NG" => Some(CountryCode::NG),
            "NU" => Some(CountryCode::NU),
            "NF" => Some(CountryCode::NF),
            "MP" => Some(CountryCode::MP),
            "NO" => Some(CountryCode::NO),
            "OM" => Some(CountryCode::OM),
            "PK" => Some(CountryCode::PK),
            "PW" => Some(CountryCode::PW),
            "PS" => Some(CountryCode::PS),
            "PA" => Some(CountryCode::PA),
            "PG" => Some(CountryCode::PG),
            "PY" => Some(CountryCode::PY),
            "PE" => Some(CountryCode::PE),
            "PH" => Some(CountryCode::PH),
            "PN" => Some(CountryCode::PN),
            "PL" => Some(CountryCode::PL),
            "PT" => Some(CountryCode::PT),
            "PR" => Some(CountryCode::PR),
            "QA" => Some(CountryCode::QA),
            "RE" => Some(CountryCode::RE),
            "RO" => Some(CountryCode::RO),
            "RU" => Some(CountryCode::RU),
            "RW" => Some(CountryCode::RW),
            "BL" => Some(CountryCode::BL),
            "SH" => Some(CountryCode::SH),
            "KN" => Some(CountryCode::KN),
            "LC" => Some(CountryCode::LC),
            "MF" => Some(CountryCode::MF),
            "PM" => Some(CountryCode::PM),
            "VC" => Some(CountryCode::VC),
            "WS" => Some(CountryCode::WS),
            "SM" => Some(CountryCode::SM),
            "ST" => Some(CountryCode::ST),
            "SA" => Some(CountryCode::SA),
            "SN" => Some(CountryCode::SN),
            "RS" => Some(CountryCode::RS),
            "SC" => Some(CountryCode::SC),
            "SL" => Some(CountryCode::SL),
            "SG" => Some(CountryCode::SG),
            "SX" => Some(CountryCode::SX),
            "SK" => Some(CountryCode::SK),
            "SI" => Some(CountryCode::SI),
            "SB" => Some(CountryCode::SB),
            "SO" => Some(CountryCode::SO),
            "ZA" => Some(CountryCode::ZA),
            "GS" => Some(CountryCode::GS),
            "SS" => Some(CountryCode::SS),
            "ES" => Some(CountryCode::ES),
            "LK" => Some(CountryCode::LK),
            "SD" => Some(CountryCode::SD),
            "SR" => Some(CountryCode::SR),
            "SJ" => Some(CountryCode::SJ),
            "SE" => Some(CountryCode::SE),
            "CH" => Some(CountryCode::CH),
            "SY" => Some(CountryCode::SY),
            "TW" => Some(CountryCode::TW),
            "TJ" => Some(CountryCode::TJ),
            "TZ" => Some(CountryCode::TZ),
            "TH" => Some(CountryCode::TH),
            "TL" => Some(CountryCode::TL),
            "TG" => Some(CountryCode::TG),
            "TK" => Some(CountryCode::TK),
            "TO" => Some(CountryCode::TO),
            "TT" => Some(CountryCode::TT),
            "TN" => Some(CountryCode::TN),
            "TR" => Some(CountryCode::TR),
            "TM" => Some(CountryCode::TM),
            "TC" => Some(CountryCode::TC),
            "TV" => Some(CountryCode::TV),
            "UG" => Some(CountryCode::UG),
            "UA" => Some(CountryCode::UA),
            "AE" => Some(CountryCode::AE),
            "GB" => Some(CountryCode::GB),
            "US" => Some(CountryCode::US),
            "UM" => Some(CountryCode::UM),
            "UY" => Some(CountryCode::UY),
            "UZ" => Some(CountryCode::UZ),
            "VU" => Some(CountryCode::VU),
            "VE" => Some(CountryCode::VE),
            "VN" => Some(CountryCode::VN),
            "VG" => Some(CountryCode::VG),
            "VI" => Some(CountryCode::VI),
            "WF" => Some(CountryCode::WF),
            "EH" => Some(CountryCode::EH),
            "YE" => Some(CountryCode::YE),
            "ZM" => Some(CountryCode::ZM),
            "ZW" => Some(CountryCode::ZW),
            _ => None,
        }
    }

    pub fn to_str(&self) -> &'static str {
        match *self {
            CountryCode::AF => "AF",
            CountryCode::AX => "AX",
            CountryCode::AL => "AL",
            CountryCode::DZ => "DZ",
            CountryCode::AS => "AS",
            CountryCode::AD => "AD",
            CountryCode::AO => "AO",
            CountryCode::AI => "AI",
            CountryCode::AG => "AG",
            CountryCode::AR => "AR",
            CountryCode::AM => "AM",
            CountryCode::AW => "AW",
            CountryCode::AU => "AU",
            CountryCode::AT => "AT",
            CountryCode::AZ => "AZ",
            CountryCode::BS => "BS",
            CountryCode::BH => "BH",
            CountryCode::BD => "BD",
            CountryCode::BB => "BB",
            CountryCode::BY => "BY",
            CountryCode::BE => "BE",
            CountryCode::BZ => "BZ",
            CountryCode::BJ => "BJ",
            CountryCode::BM => "BM",
            CountryCode::BT => "BT",
            CountryCode::BO => "BO",
            CountryCode::BQ => "BQ",
            CountryCode::BA => "BA",
            CountryCode::BW => "BW",
            CountryCode::BV => "BV",
            CountryCode::BR => "BR",
            CountryCode::IO => "IO",
            CountryCode::BN => "BN",
            CountryCode::BG => "BG",
            CountryCode::BF => "BF",
            CountryCode::BI => "BI",
            CountryCode::CV => "CV",
            CountryCode::KH => "KH",
            CountryCode::CM => "CM",
            CountryCode::CA => "CA",
            CountryCode::KY => "KY",
            CountryCode::CF => "CF",
            CountryCode::TD => "TD",
            CountryCode::CL => "CL",
            CountryCode::CN => "CN",
            CountryCode::CX => "CX",
            CountryCode::CC => "CC",
            CountryCode::CO => "CO",
            CountryCode::KM => "KM",
            CountryCode::CG => "CG",
            CountryCode::CD => "CD",
            CountryCode::CK => "CK",
            CountryCode::CR => "CR",
            CountryCode::CI => "CI",
            CountryCode::HR => "HR",
            CountryCode::CU => "CU",
            CountryCode::CW => "CW",
            CountryCode::CY => "CY",
            CountryCode::CZ => "CZ",
            CountryCode::DK => "DK",
            CountryCode::DJ => "DJ",
            CountryCode::DM => "DM",
            CountryCode::DO => "DO",
            CountryCode::EC => "EC",
            CountryCode::EG => "EG",
            CountryCode::SV => "SV",
            CountryCode::GQ => "GQ",
            CountryCode::ER => "ER",
            CountryCode::EE => "EE",
            CountryCode::SZ => "SZ",
            CountryCode::ET => "ET",
            CountryCode::FK => "FK",
            CountryCode::FO => "FO",
            CountryCode::FJ => "FJ",
            CountryCode::FI => "FI",
            CountryCode::FR => "FR",
            CountryCode::GF => "GF",
            CountryCode::PF => "PF",
            CountryCode::TF => "TF",
            CountryCode::GA => "GA",
            CountryCode::GM => "GM",
            CountryCode::GE => "GE",
            CountryCode::DE => "DE",
            CountryCode::GH => "GH",
            CountryCode::GI => "GI",
            CountryCode::GR => "GR",
            CountryCode::GL => "GL",
            CountryCode::GD => "GD",
            CountryCode::GP => "GP",
            CountryCode::GU => "GU",
            CountryCode::GT => "GT",
            CountryCode::GG => "GG",
            CountryCode::GN => "GN",
            CountryCode::GW => "GW",
            CountryCode::GY => "GY",
            CountryCode::HT => "HT",
            CountryCode::HM => "HM",
            CountryCode::VA => "VA",
            CountryCode::HN => "HN",
            CountryCode::HK => "HK",
            CountryCode::HU => "HU",
            CountryCode::IS => "IS",
            CountryCode::IN => "IN",
            CountryCode::ID => "ID",
            CountryCode::IR => "IR",
            CountryCode::IQ => "IQ",
            CountryCode::IE => "IE",
            CountryCode::IM => "IM",
            CountryCode::IL => "IL",
            CountryCode::IT => "IT",
            CountryCode::JM => "JM",
            CountryCode::JP => "JP",
            CountryCode::JE => "JE",
            CountryCode::JO => "JO",
            CountryCode::KZ => "KZ",
            CountryCode::KE => "KE",
            CountryCode::KI => "KI",
            CountryCode::KP => "KP",
            CountryCode::KR => "KR",
            CountryCode::KW => "KW",
            CountryCode::KG => "KG",
            CountryCode::LA => "LA",
            CountryCode::LV => "LV",
            CountryCode::LB => "LB",
            CountryCode::LS => "LS",
            CountryCode::LR => "LR",
            CountryCode::LY => "LY",
            CountryCode::LI => "LI",
            CountryCode::LT => "LT",
            CountryCode::LU => "LU",
            CountryCode::MO => "MO",
            CountryCode::MK => "MK",
            CountryCode::MG => "MG",
            CountryCode::MW => "MW",
            CountryCode::MY => "MY",
            CountryCode::MV => "MV",
            CountryCode::ML => "ML",
            CountryCode::MT => "MT",
            CountryCode::MH => "MH",
            CountryCode::MQ => "MQ",
            CountryCode::MR => "MR",
            CountryCode::MU => "MU",
            CountryCode::YT => "YT",
            CountryCode::MX => "MX",
            CountryCode::FM => "FM",
            CountryCode::MD => "MD",
            CountryCode::MC => "MC",
            CountryCode::MN => "MN",
            CountryCode::ME => "ME",
            CountryCode::MS => "MS",
            CountryCode::MA => "MA",
            CountryCode::MZ => "MZ",
            CountryCode::MM => "MM",
            CountryCode::NA => "NA",
            CountryCode::NR => "NR",
            CountryCode::NP => "NP",
            CountryCode::NL => "NL",
            CountryCode::NC => "NC",
            CountryCode::NZ => "NZ",
            CountryCode::NI => "NI",
            CountryCode::NE => "NE",
            CountryCode::NG => "NG",
            CountryCode::NU => "NU",
            CountryCode::NF => "NF",
            CountryCode::MP => "MP",
            CountryCode::NO => "NO",
            CountryCode::OM => "OM",
            CountryCode::PK => "PK",
            CountryCode::PW => "PW",
            CountryCode::PS => "PS",
            CountryCode::PA => "PA",
            CountryCode::PG => "PG",
            CountryCode::PY => "PY",
            CountryCode::PE => "PE",
            CountryCode::PH => "PH",
            CountryCode::PN => "PN",
            CountryCode::PL => "PL",
            CountryCode::PT => "PT",
            CountryCode::PR => "PR",
            CountryCode::QA => "QA",
            CountryCode::RE => "RE",
            CountryCode::RO => "RO",
            CountryCode::RU => "RU",
            CountryCode::RW => "RW",
            CountryCode::BL => "BL",
            CountryCode::SH => "SH",
            CountryCode::KN => "KN",
            CountryCode::LC => "LC",
            CountryCode::MF => "MF",
            CountryCode::PM => "PM",
            CountryCode::VC => "VC",
            CountryCode::WS => "WS",
            CountryCode::SM => "SM",
            CountryCode::ST => "ST",
            CountryCode::SA => "SA",
            CountryCode::SN => "SN",
            CountryCode::RS => "RS",
            CountryCode::SC => "SC",
            CountryCode::SL => "SL",
            CountryCode::SG => "SG",
            CountryCode::SX => "SX",
            CountryCode::SK => "SK",
            CountryCode::SI => "SI",
            CountryCode::SB => "SB",
            CountryCode::SO => "SO",
            CountryCode::ZA => "ZA",
            CountryCode::GS => "GS",
            CountryCode::SS => "SS",
            CountryCode::ES => "ES",
            CountryCode::LK => "LK",
            CountryCode::SD => "SD",
            CountryCode::SR => "SR",
            CountryCode::SJ => "SJ",
            CountryCode::SE => "SE",
            CountryCode::CH => "CH",
            CountryCode::SY => "SY",
            CountryCode::TW => "TW",
            CountryCode::TJ => "TJ",
            CountryCode::TZ => "TZ",
            CountryCode::TH => "TH",
            CountryCode::TL => "TL",
            CountryCode::TG => "TG",
            CountryCode::TK => "TK",
            CountryCode::TO => "TO",
            CountryCode::TT => "TT",
            CountryCode::TN => "TN",
            CountryCode::TR => "TR",
            CountryCode::TM => "TM",
            CountryCode::TC => "TC",
            CountryCode::TV => "TV",
            CountryCode::UG => "UG",
            CountryCode::UA => "UA",
            CountryCode::AE => "AE",
            CountryCode::GB => "GB",
            CountryCode::US => "US",
            CountryCode::UM => "UM",
            CountryCode::UY => "UY",
            CountryCode::UZ => "UZ",
            CountryCode::VU => "VU",
            CountryCode::VE => "VE",
            CountryCode::VN => "VN",
            CountryCode::VG => "VG",
            CountryCode::VI => "VI",
            CountryCode::WF => "WF",
            CountryCode::EH => "EH",
            CountryCode::YE => "YE",
            CountryCode::ZM => "ZM",
            CountryCode::ZW => "ZW",
        }
    }

    pub fn to_name(&self) -> &'static str {
        match *self {
            CountryCode::AF => "Afghanistan",
            CountryCode::AX => "Åland Islands",
            CountryCode::AL => "Albania",
            CountryCode::DZ => "Algeria",
            CountryCode::AS => "American Samoa",
            CountryCode::AD => "Andorra",
            CountryCode::AO => "Angola",
            CountryCode::AI => "Anguilla",
            CountryCode::AG => "Antigua and Barbuda",
            CountryCode::AR => "Argentina",
            CountryCode::AM => "Armenia",
            CountryCode::AW => "Aruba",
            CountryCode::AU => "Australia",
            CountryCode::AT => "Austria",
            CountryCode::AZ => "Azerbaijan",
            CountryCode::BS => "Bahamas",
            CountryCode::BH => "Bahrain",
            CountryCode::BD => "Bangladesh",
            CountryCode::BB => "Barbados",
            CountryCode::BY => "Belarus",
            CountryCode::BE => "Belgium",
            CountryCode::BZ => "Belize",
            CountryCode::BJ => "Benin",
            CountryCode::BM => "Bermuda",
            CountryCode::BT => "Bhutan",
            CountryCode::BO => "Bolivia (Plurinational State of)",
            CountryCode::BQ => "Bonaire, Sint Eustatius and Saba",
            CountryCode::BA => "Bosnia and Herzegovina",
            CountryCode::BW => "Botswana",
            CountryCode::BV => "Bouvet Island",
            CountryCode::BR => "Brazil",
            CountryCode::IO => "British Indian Ocean Territory",
            CountryCode::BN => "Brunei Darussalam",
            CountryCode::BG => "Bulgaria",
            CountryCode::BF => "Burkina Faso",
            CountryCode::BI => "Burundi",
            CountryCode::CV => "Cabo Verde",
            CountryCode::KH => "Cambodia",
            CountryCode::CM => "Cameroon",
            CountryCode::CA => "Canada",
            CountryCode::KY => "Cayman Islands",
            CountryCode::CF => "Central African Republic",
            CountryCode::TD => "Chad",
            CountryCode::CL => "Chile",
            CountryCode::CN => "China",
            CountryCode::CX => "Christmas Island",
            CountryCode::CC => "Cocos (Keeling) Islands",
            CountryCode::CO => "Colombia",
            CountryCode::KM => "Comoros",
            CountryCode::CG => "Congo",
            CountryCode::CD => "Congo (Democratic Republic of the)",
            CountryCode::CK => "Cook Islands",
            CountryCode::CR => "Costa Rica",
            CountryCode::CI => "Côte d'Ivoire",
            CountryCode::HR => "Croatia",
            CountryCode::CU => "Cuba",
            CountryCode::CW => "Curaçao",
            CountryCode::CY => "Cyprus",
            CountryCode::CZ => "Czechia",
            CountryCode::DK => "Denmark",
            CountryCode::DJ => "Djibouti",
            CountryCode::DM => "Dominica",
            CountryCode::DO => "Dominican Republic",
            CountryCode::EC => "Ecuador",
            CountryCode::EG => "Egypt",
            CountryCode::SV => "El Salvador",
            CountryCode::GQ => "Equatorial Guinea",
            CountryCode::ER => "Eritrea",
            CountryCode::EE => "Estonia",
            CountryCode::SZ => "Eswatini",
            CountryCode::ET => "Ethiopia",
            CountryCode::FK => "Falkland Islands (Malvinas)",
            CountryCode::FO => "Faroe Islands",
            CountryCode::FJ => "Fiji",
            CountryCode::FI => "Finland",
            CountryCode::FR => "France",
            CountryCode::GF => "French Guiana",
            CountryCode::PF => "French Polynesia",
            CountryCode::TF => "French Southern Territories",
            CountryCode::GA => "Gabon",
            CountryCode::GM => "Gambia",
            CountryCode::GE => "Georgia",
            CountryCode::DE => "Germany",
            CountryCode::GH => "Ghana",
            CountryCode::GI => "Gibraltar",
            CountryCode::GR => "Greece",
            CountryCode::GL => "Greenland",
            CountryCode::GD => "Grenada",
            CountryCode::GP => "Guadeloupe",
            CountryCode::GU => "Guam",
            CountryCode::GT => "Guatemala",
            CountryCode::GG => "Guernsey",
            CountryCode::GN => "Guinea",
            CountryCode::GW => "Guinea-Bissau",
            CountryCode::GY => "Guyana",
            CountryCode::HT => "Haiti",
            CountryCode::HM => "Heard Island and McDonald Islands",
            CountryCode::VA => "Holy See",
            CountryCode::HN => "Honduras",
            CountryCode::HK => "Hong Kong",
            CountryCode::HU => "Hungary",
            CountryCode::IS => "Iceland",
            CountryCode::IN => "India",
            CountryCode::ID => "Indonesia",
            CountryCode::IR => "Iran (Islamic Republic of)",
            CountryCode::IQ => "Iraq",
            CountryCode::IE => "Ireland",
            CountryCode::IM => "Isle of Man",
            CountryCode::IL => "Israel",
            CountryCode::IT => "Italy",
            CountryCode::JM => "Jamaica",
            CountryCode::JP => "Japan",
            CountryCode::JE => "Jersey",
            CountryCode::JO => "Jordan",
            CountryCode::KZ => "Kazakhstan",
            CountryCode::KE => "Kenya",
            CountryCode::KI => "Kiribati",
            CountryCode::KP => "Korea (Democratic People's Republic of)",
            CountryCode::KR => "Korea (Republic of)",
            CountryCode::KW => "Kuwait",
            CountryCode::KG => "Kyrgyzstan",
            CountryCode::LA => "Lao People's Democratic Republic",
            CountryCode::LV => "Latvia",
            CountryCode::LB => "Lebanon",
            CountryCode::LS => "Lesotho",
            CountryCode::LR => "Liberia",
            CountryCode::LY => "Libya",
            CountryCode::LI => "Liechtenstein",
            CountryCode::LT => "Lithuania",
            CountryCode::LU => "Luxembourg",
            CountryCode::MO => "Macao",
            CountryCode::MK => "Macedonia (the former Yugoslav Republic of)",
            CountryCode::MG => "Madagascar",
            CountryCode::MW => "Malawi",
            CountryCode::MY => "Malaysia",
            CountryCode::MV => "Maldives",
            CountryCode::ML => "Mali",
            CountryCode::MT => "Malta",
            CountryCode::MH => "Marshall Islands",
            CountryCode::MQ => "Martinique",
            CountryCode::MR => "Mauritania",
            CountryCode::MU => "Mauritius",
            CountryCode::YT => "Mayotte",
            CountryCode::MX => "Mexico",
            CountryCode::FM => "Micronesia (Federated States of)",
            CountryCode::MD => "Moldova (Republic of)",
            CountryCode::MC => "Monaco",
            CountryCode::MN => "Mongolia",
            CountryCode::ME => "Montenegro",
            CountryCode::MS => "Montserrat",
            CountryCode::MA => "Morocco",
            CountryCode::MZ => "Mozambique",
            CountryCode::MM => "Myanmar",
            CountryCode::NA => "Namibia",
            CountryCode::NR => "Nauru",
            CountryCode::NP => "Nepal",
            CountryCode::NL => "Netherlands",
            CountryCode::NC => "New Caledonia",
            CountryCode::NZ => "New Zealand",
            CountryCode::NI => "Nicaragua",
            CountryCode::NE => "Niger",
            CountryCode::NG => "Nigeria",
            CountryCode::NU => "Niue",
            CountryCode::NF => "Norfolk Island",
            CountryCode::MP => "Northern Mariana Islands",
            CountryCode::NO => "Norway",
            CountryCode::OM => "Oman",
            CountryCode::PK => "Pakistan",
            CountryCode::PW => "Palau",
            CountryCode::PS => "Palestine, State of",
            CountryCode::PA => "Panama",
            CountryCode::PG => "Papua New Guinea",
            CountryCode::PY => "Paraguay",
            CountryCode::PE => "Peru",
            CountryCode::PH => "Philippines",
            CountryCode::PN => "Pitcairn",
            CountryCode::PL => "Poland",
            CountryCode::PT => "Portugal",
            CountryCode::PR => "Puerto Rico",
            CountryCode::QA => "Qatar",
            CountryCode::RE => "Réunion",
            CountryCode::RO => "Romania",
            CountryCode::RU => "Russian Federation",
            CountryCode::RW => "Rwanda",
            CountryCode::BL => "Saint Barthélemy",
            CountryCode::SH => "Saint Helena, Ascension and Tristan da Cunha",
            CountryCode::KN => "Saint Kitts and Nevis",
            CountryCode::LC => "Saint Lucia",
            CountryCode::MF => "Saint Martin (French part)",
            CountryCode::PM => "Saint Pierre and Miquelon",
            CountryCode::VC => "Saint Vincent and the Grenadines",
            CountryCode::WS => "Samoa",
            CountryCode::SM => "San Marino",
            CountryCode::ST => "Sao Tome and Principe",
            CountryCode::SA => "Saudi Arabia",
            CountryCode::SN => "Senegal",
            CountryCode::RS => "Serbia",
            CountryCode::SC => "Seychelles",
            CountryCode::SL => "Sierra Leone",
            CountryCode::SG => "Singapore",
            CountryCode::SX => "Sint Maarten (Dutch part)",
            CountryCode::SK => "Slovakia",
            CountryCode::SI => "Slovenia",
            CountryCode::SB => "Solomon Islands",
            CountryCode::SO => "Somalia",
            CountryCode::ZA => "South Africa",
            CountryCode::GS => "South Georgia and the South Sandwich Islands",
            CountryCode::SS => "South Sudan",
            CountryCode::ES => "Spain",
            CountryCode::LK => "Sri Lanka",
            CountryCode::SD => "Sudan",
            CountryCode::SR => "Suriname",
            CountryCode::SJ => "Svalbard and Jan Mayen",
            CountryCode::SE => "Sweden",
            CountryCode::CH => "Switzerland",
            CountryCode::SY => "Syrian Arab Republic",
            CountryCode::TW => "Taiwan, Province of China",
            CountryCode::TJ => "Tajikistan",
            CountryCode::TZ => "Tanzania, United Republic of",
            CountryCode::TH => "Thailand",
            CountryCode::TL => "Timor-Leste",
            CountryCode::TG => "Togo",
            CountryCode::TK => "Tokelau",
            CountryCode::TO => "Tonga",
            CountryCode::TT => "Trinidad and Tobago",
            CountryCode::TN => "Tunisia",
            CountryCode::TR => "Turkey",
            CountryCode::TM => "Turkmenistan",
            CountryCode::TC => "Turks and Caicos Islands",
            CountryCode::TV => "Tuvalu",
            CountryCode::UG => "Uganda",
            CountryCode::UA => "Ukraine",
            CountryCode::AE => "United Arab Emirates",
            CountryCode::GB => "United Kingdom of Great Britain and Northern Ireland",
            CountryCode::US => "United States of America",
            CountryCode::UM => "United States Minor Outlying Islands",
            CountryCode::UY => "Uruguay",
            CountryCode::UZ => "Uzbekistan",
            CountryCode::VU => "Vanuatu",
            CountryCode::VE => "Venezuela (Bolivarian Republic of)",
            CountryCode::VN => "Viet Nam",
            CountryCode::VG => "Virgin Islands (British)",
            CountryCode::VI => "Virgin Islands (U.S.)",
            CountryCode::WF => "Wallis and Futuna",
            CountryCode::EH => "Western Sahara",
            CountryCode::YE => "Yemen",
            CountryCode::ZM => "Zambia",
            CountryCode::ZW => "Zimbabwe",
        }
    }
}

impl CountryCode {
    pub fn to_region_code(&self) -> RegionCode {
        match *self {
            CountryCode::AF => RegionCode::SEAS,
            CountryCode::AX => RegionCode::CEU,
            CountryCode::AL => RegionCode::EEU,
            CountryCode::DZ => RegionCode::NAF,
            CountryCode::AS => RegionCode::OC,
            CountryCode::AD => RegionCode::WEU,
            CountryCode::AO => RegionCode::SAF,
            CountryCode::AI => RegionCode::NSAM,
            CountryCode::AG => RegionCode::NSAM,
            CountryCode::AR => RegionCode::SSAM,
            CountryCode::AM => RegionCode::SEAS,
            CountryCode::AW => RegionCode::NSAM,
            CountryCode::AU => RegionCode::OC,
            CountryCode::AT => RegionCode::CEU,
            CountryCode::AZ => RegionCode::SEAS,
            CountryCode::BS => RegionCode::NSAM,
            CountryCode::BH => RegionCode::ME,
            CountryCode::BD => RegionCode::SEAS,
            CountryCode::BB => RegionCode::NSAM,
            CountryCode::BY => RegionCode::EEU,
            CountryCode::BE => RegionCode::WEU,
            CountryCode::BZ => RegionCode::NSAM,
            CountryCode::BJ => RegionCode::MAF,
            CountryCode::BM => RegionCode::SNAM,
            CountryCode::BT => RegionCode::SEAS,
            CountryCode::BO => RegionCode::SSAM,
            CountryCode::BQ => RegionCode::NSAM,
            CountryCode::BA => RegionCode::EEU,
            CountryCode::BW => RegionCode::SAF,
            CountryCode::BV => RegionCode::SSAM,
            CountryCode::BR => RegionCode::SSAM,
            CountryCode::IO => RegionCode::MAF,
            CountryCode::BN => RegionCode::SEAS,
            CountryCode::BG => RegionCode::EEU,
            CountryCode::BF => RegionCode::MAF,
            CountryCode::BI => RegionCode::MAF,
            CountryCode::CV => RegionCode::MAF,
            CountryCode::KH => RegionCode::SEAS,
            CountryCode::CM => RegionCode::MAF,
            CountryCode::CA => RegionCode::NNAM,
            CountryCode::KY => RegionCode::NSAM,
            CountryCode::CF => RegionCode::MAF,
            CountryCode::TD => RegionCode::MAF,
            CountryCode::CL => RegionCode::SSAM,
            CountryCode::CN => RegionCode::NEAS,
            CountryCode::CX => RegionCode::OC,
            CountryCode::CC => RegionCode::OC,
            CountryCode::CO => RegionCode::NSAM,
            CountryCode::KM => RegionCode::SAF,
            CountryCode::CG => RegionCode::MAF,
            CountryCode::CD => RegionCode::MAF,
            CountryCode::CK => RegionCode::OC,
            CountryCode::CR => RegionCode::NSAM,
            CountryCode::CI => RegionCode::MAF,
            CountryCode::HR => RegionCode::CEU,
            CountryCode::CU => RegionCode::NSAM,
            CountryCode::CW => RegionCode::NSAM,
            CountryCode::CY => RegionCode::ME,
            CountryCode::CZ => RegionCode::CEU,
            CountryCode::DK => RegionCode::CEU,
            CountryCode::DJ => RegionCode::MAF,
            CountryCode::DM => RegionCode::NSAM,
            CountryCode::DO => RegionCode::NSAM,
            CountryCode::EC => RegionCode::NSAM,
            CountryCode::EG => RegionCode::ME,
            CountryCode::SV => RegionCode::NSAM,
            CountryCode::GQ => RegionCode::MAF,
            CountryCode::ER => RegionCode::MAF,
            CountryCode::EE => RegionCode::EEU,
            CountryCode::SZ => RegionCode::SAF,
            CountryCode::ET => RegionCode::MAF,
            CountryCode::FK => RegionCode::SSAM,
            CountryCode::FO => RegionCode::WEU,
            CountryCode::FJ => RegionCode::OC,
            CountryCode::FI => RegionCode::EEU,
            CountryCode::FR => RegionCode::WEU,
            CountryCode::GF => RegionCode::NSAM,
            CountryCode::PF => RegionCode::OC,
            CountryCode::TF => RegionCode::SAF,
            CountryCode::GA => RegionCode::MAF,
            CountryCode::GM => RegionCode::MAF,
            CountryCode::GE => RegionCode::SEAS,
            CountryCode::DE => RegionCode::CEU,
            CountryCode::GH => RegionCode::MAF,
            CountryCode::GI => RegionCode::WEU,
            CountryCode::GR => RegionCode::EEU,
            CountryCode::GL => RegionCode::NNAM,
            CountryCode::GD => RegionCode::NSAM,
            CountryCode::GP => RegionCode::NSAM,
            CountryCode::GU => RegionCode::OC,
            CountryCode::GT => RegionCode::NSAM,
            CountryCode::GG => RegionCode::WEU,
            CountryCode::GN => RegionCode::MAF,
            CountryCode::GW => RegionCode::MAF,
            CountryCode::GY => RegionCode::NSAM,
            CountryCode::HT => RegionCode::NSAM,
            CountryCode::HM => RegionCode::OC,
            CountryCode::VA => RegionCode::WEU,
            CountryCode::HN => RegionCode::NSAM,
            CountryCode::HK => RegionCode::NEAS,
            CountryCode::HU => RegionCode::CEU,
            CountryCode::IS => RegionCode::WEU,
            CountryCode::IN => RegionCode::IN,
            CountryCode::ID => RegionCode::SEAS,
            CountryCode::IR => RegionCode::ME,
            CountryCode::IQ => RegionCode::ME,
            CountryCode::IE => RegionCode::WEU,
            CountryCode::IM => RegionCode::WEU,
            CountryCode::IL => RegionCode::ME,
            CountryCode::IT => RegionCode::WEU,
            CountryCode::JM => RegionCode::NSAM,
            CountryCode::JP => RegionCode::NEAS,
            CountryCode::JE => RegionCode::WEU,
            CountryCode::JO => RegionCode::ME,
            CountryCode::KZ => RegionCode::SEAS,
            CountryCode::KE => RegionCode::MAF,
            CountryCode::KI => RegionCode::OC,
            CountryCode::KP => RegionCode::NEAS,
            CountryCode::KR => RegionCode::NEAS,
            CountryCode::KW => RegionCode::ME,
            CountryCode::KG => RegionCode::SEAS,
            CountryCode::LA => RegionCode::SEAS,
            CountryCode::LV => RegionCode::EEU,
            CountryCode::LB => RegionCode::ME,
            CountryCode::LS => RegionCode::SAF,
            CountryCode::LR => RegionCode::MAF,
            CountryCode::LY => RegionCode::NAF,
            CountryCode::LI => RegionCode::CEU,
            CountryCode::LT => RegionCode::EEU,
            CountryCode::LU => RegionCode::CEU,
            CountryCode::MO => RegionCode::NEAS,
            CountryCode::MK => RegionCode::EEU,
            CountryCode::MG => RegionCode::SAF,
            CountryCode::MW => RegionCode::SAF,
            CountryCode::MY => RegionCode::SEAS,
            CountryCode::MV => RegionCode::SEAS,
            CountryCode::ML => RegionCode::MAF,
            CountryCode::MT => RegionCode::WEU,
            CountryCode::MH => RegionCode::OC,
            CountryCode::MQ => RegionCode::NSAM,
            CountryCode::MR => RegionCode::MAF,
            CountryCode::MU => RegionCode::SAF,
            CountryCode::YT => RegionCode::SAF,
            CountryCode::MX => RegionCode::NSAM,
            CountryCode::FM => RegionCode::OC,
            CountryCode::MD => RegionCode::EEU,
            CountryCode::MC => RegionCode::WEU,
            CountryCode::MN => RegionCode::NEAS,
            CountryCode::ME => RegionCode::EEU,
            CountryCode::MS => RegionCode::NSAM,
            CountryCode::MA => RegionCode::NAF,
            CountryCode::MZ => RegionCode::SAF,
            CountryCode::MM => RegionCode::SEAS,
            CountryCode::NA => RegionCode::SAF,
            CountryCode::NR => RegionCode::OC,
            CountryCode::NP => RegionCode::SEAS,
            CountryCode::NL => RegionCode::WEU,
            CountryCode::NC => RegionCode::OC,
            CountryCode::NZ => RegionCode::OC,
            CountryCode::NI => RegionCode::NSAM,
            CountryCode::NE => RegionCode::MAF,
            CountryCode::NG => RegionCode::MAF,
            CountryCode::NU => RegionCode::OC,
            CountryCode::NF => RegionCode::OC,
            CountryCode::MP => RegionCode::OC,
            CountryCode::NO => RegionCode::CEU,
            CountryCode::OM => RegionCode::ME,
            CountryCode::PK => RegionCode::SEAS,
            CountryCode::PW => RegionCode::OC,
            CountryCode::PS => RegionCode::ME,
            CountryCode::PA => RegionCode::NSAM,
            CountryCode::PG => RegionCode::OC,
            CountryCode::PY => RegionCode::SSAM,
            CountryCode::PE => RegionCode::NSAM,
            CountryCode::PH => RegionCode::SEAS,
            CountryCode::PN => RegionCode::OC,
            CountryCode::PL => RegionCode::CEU,
            CountryCode::PT => RegionCode::WEU,
            CountryCode::PR => RegionCode::NSAM,
            CountryCode::QA => RegionCode::ME,
            CountryCode::RE => RegionCode::SAF,
            CountryCode::RO => RegionCode::EEU,
            CountryCode::RU => RegionCode::RU,
            CountryCode::RW => RegionCode::MAF,
            CountryCode::BL => RegionCode::NSAM,
            CountryCode::SH => RegionCode::SAF,
            CountryCode::KN => RegionCode::NSAM,
            CountryCode::LC => RegionCode::NSAM,
            CountryCode::MF => RegionCode::NSAM,
            CountryCode::PM => RegionCode::NNAM,
            CountryCode::VC => RegionCode::NSAM,
            CountryCode::WS => RegionCode::OC,
            CountryCode::SM => RegionCode::WEU,
            CountryCode::ST => RegionCode::MAF,
            CountryCode::SA => RegionCode::ME,
            CountryCode::SN => RegionCode::MAF,
            CountryCode::RS => RegionCode::EEU,
            CountryCode::SC => RegionCode::MAF,
            CountryCode::SL => RegionCode::MAF,
            CountryCode::SG => RegionCode::SEAS,
            CountryCode::SX => RegionCode::NSAM,
            CountryCode::SK => RegionCode::CEU,
            CountryCode::SI => RegionCode::CEU,
            CountryCode::SB => RegionCode::OC,
            CountryCode::SO => RegionCode::MAF,
            CountryCode::ZA => RegionCode::SAF,
            CountryCode::GS => RegionCode::SSAM,
            CountryCode::SS => RegionCode::MAF,
            CountryCode::ES => RegionCode::WEU,
            CountryCode::LK => RegionCode::SEAS,
            CountryCode::SD => RegionCode::NAF,
            CountryCode::SR => RegionCode::NSAM,
            CountryCode::SJ => RegionCode::CEU,
            CountryCode::SE => RegionCode::CEU,
            CountryCode::CH => RegionCode::CEU,
            CountryCode::SY => RegionCode::ME,
            CountryCode::TW => RegionCode::NEAS,
            CountryCode::TJ => RegionCode::SEAS,
            CountryCode::TZ => RegionCode::MAF,
            CountryCode::TH => RegionCode::SEAS,
            CountryCode::TL => RegionCode::SEAS,
            CountryCode::TG => RegionCode::MAF,
            CountryCode::TK => RegionCode::OC,
            CountryCode::TO => RegionCode::OC,
            CountryCode::TT => RegionCode::NSAM,
            CountryCode::TN => RegionCode::NAF,
            CountryCode::TR => RegionCode::ME,
            CountryCode::TM => RegionCode::SEAS,
            CountryCode::TC => RegionCode::NSAM,
            CountryCode::TV => RegionCode::OC,
            CountryCode::UG => RegionCode::MAF,
            CountryCode::UA => RegionCode::EEU,
            CountryCode::AE => RegionCode::ME,
            CountryCode::GB => RegionCode::WEU,
            CountryCode::US => RegionCode::SNAM,
            CountryCode::UM => RegionCode::OC,
            CountryCode::UY => RegionCode::SSAM,
            CountryCode::UZ => RegionCode::SEAS,
            CountryCode::VU => RegionCode::OC,
            CountryCode::VE => RegionCode::NSAM,
            CountryCode::VN => RegionCode::SEAS,
            CountryCode::VG => RegionCode::NSAM,
            CountryCode::VI => RegionCode::NSAM,
            CountryCode::WF => RegionCode::OC,
            CountryCode::EH => RegionCode::NAF,
            CountryCode::YE => RegionCode::ME,
            CountryCode::ZM => RegionCode::SAF,
            CountryCode::ZW => RegionCode::SAF,
        }
    }
}
