use core::str::FromStr;
use num_bigint::BigInt;
use num_bigint::BigUint;
use num_bigint::Sign;
use proc_macro::TokenStream;
use quote::quote;
use syn::Path;
use syn::__private::Span;
use syn::{parse_macro_input, Lit, LitInt};
use syn::{Expr, ExprLit,token::Comma,ExprPath,ExprGroup,punctuated::Punctuated};

#[proc_macro]
pub fn bigint_with_crate(input: TokenStream) -> TokenStream {
    let params:Punctuated<Expr, Comma> = parse_macro_input!(input with Punctuated::parse_terminated);
    let crate_name = match &params[0] {
        Expr::Path(ExprPath{path:Path{segments,..},..})=>segments[0].ident.to_owned(),
        _ => panic!("except crate name"),
    };
    let expr1 = match &params[1] {
        Expr::Group(ExprGroup{ expr ,..})=> expr.to_owned(),
        other => panic!("except literal string {} ", quote! {#other}.to_string()),
    };
    let s = match expr1.as_ref() {
        Expr::Lit(ExprLit{lit:Lit::Str(s),..})=>s,
        other => panic!("except literal string {} ", quote! {#other}.to_string()),
    };
    let (sign, array32) = BigInt::from_str(&s.value()).unwrap().to_u32_digits();
    let elems: Vec<Expr> = array32
        .into_iter()
        .map(|value32| {
            Expr::Lit(ExprLit {
                attrs: Default::default(),
                lit: Lit::Int(LitInt::new(&value32.to_string(), Span::call_site())),
            })
        })
        .collect();

    let arr = quote! {
        #(#elems),*
    };
    let sign_lit = match sign {
        Sign::Minus => quote! {#crate_name::Sign::Minus},
        Sign::NoSign => quote! {#crate_name::Sign::NoSign},
        Sign::Plus => quote! {#crate_name::Sign::Plus},
    };
    let expanded = quote! {
        {
            const BIGINT_CONST_VALUE : &'static BigInt = & #crate_name::ConstBigInt::new(#sign_lit,&#crate_name::from_slice!(#arr)).to_bigint();
            BIGINT_CONST_VALUE
        }
    };
    TokenStream::from(expanded)
}

#[proc_macro]
pub fn biguint_with_crate(input: TokenStream) -> TokenStream {
    let params:Punctuated<Expr, Comma> = parse_macro_input!(input with Punctuated::parse_terminated);
    let crate_name = match &params[0] {
        Expr::Path(ExprPath{path:Path{segments,..},..})=>segments[0].ident.to_owned(),
        _ => panic!("except crate name"),
    };
    let expr1 = match &params[1] {
        Expr::Group(ExprGroup{ expr ,..})=> expr.to_owned(),
        other => panic!("except literal string {} ", quote! {#other}.to_string()),
    };
    let s = match expr1.as_ref() {
        Expr::Lit(ExprLit{lit:Lit::Str(s),..})=>s,
        other => panic!("except literal string {} ", quote! {#other}.to_string()),
    };
    let array32 = BigUint::from_str(&s.value()).unwrap().to_u32_digits();
    let elems: Vec<Expr> = array32
        .into_iter()
        .map(|value32| {
            Expr::Lit(ExprLit {
                attrs: Default::default(),
                lit: Lit::Int(LitInt::new(&value32.to_string(), Span::call_site())),
            })
        })
        .collect();

    let arr = quote! {
        #(#elems),*
    };
    let expanded = quote! {
        {
            const BIGUINT_CONST_VALUE : &'static BigUint = & #crate_name::ConstBigUint::new(&#crate_name::from_slice!(#arr)).to_biguint();
            BIGUINT_CONST_VALUE
        }
    };
    TokenStream::from(expanded)
}
