// Copyright 2021 Palantir Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//! Support for double collection keys.
use ordered_float::OrderedFloat;
use serde::{Deserialize, Deserializer, Serialize, Serializer};
use std::cmp::Ordering;
use std::fmt::{self, Display, Formatter};
use std::hash::{Hash, Hasher};
use std::ops::{Deref, DerefMut};

/// A trait implemented by `f64` and aliases wrapping an `f64`.
pub trait AsDouble {
    /// Returns the inner `f64`.
    fn as_double(&self) -> f64;
}

impl AsDouble for f64 {
    #[inline]
    fn as_double(&self) -> f64 {
        *self
    }
}

/// A wrapper type allowing `f64` to be used as a key in collection types.
///
/// Conjure allows `map<double, T>` and `set<double>`, but Rust's `f64` type does not implement `Eq` and `Ord`,
/// preventing the direct translations of `BTreeMap<f64, T>` and `BTreeSet<f64>` from compiling. This wrapper type is
/// used to provide suitable trait implementations. The code generated by `conjure-codegen` will use this type,
/// resulting in `BTreeMap<DoubleKey<f64>, T>` and `BTreeSet<DoubleKey<f64>>`.
///
/// All trait implementations delegate directly to the inner type, with the exception of the `PartialEq`, `Eq`,
/// `PartialOrd`, and `Ord` methods.
#[derive(Debug, Copy, Clone, Default)]
pub struct DoubleKey<T>(pub T);

impl<T> Display for DoubleKey<T>
where
    T: Display,
{
    #[inline]
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        self.0.fmt(f)
    }
}

impl<T> Deref for DoubleKey<T> {
    type Target = T;

    #[inline]
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl<T> DerefMut for DoubleKey<T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl<T> PartialOrd for DoubleKey<T>
where
    T: AsDouble,
{
    #[inline]
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        OrderedFloat(self.as_double()).partial_cmp(&OrderedFloat(other.as_double()))
    }
}

impl<T> PartialEq for DoubleKey<T>
where
    T: AsDouble,
{
    #[inline]
    fn eq(&self, other: &Self) -> bool {
        OrderedFloat(self.as_double()) == OrderedFloat(other.as_double())
    }
}

impl<T> Ord for DoubleKey<T>
where
    T: AsDouble,
{
    #[inline]
    fn cmp(&self, other: &Self) -> Ordering {
        OrderedFloat(self.as_double()).cmp(&OrderedFloat(other.as_double()))
    }
}

impl<T> Eq for DoubleKey<T> where T: AsDouble {}

impl<T> Hash for DoubleKey<T>
where
    T: AsDouble,
{
    #[inline]
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        OrderedFloat(self.as_double()).hash(state)
    }
}

impl<'de, T> Deserialize<'de> for DoubleKey<T>
where
    T: Deserialize<'de>,
{
    #[inline]
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        T::deserialize(deserializer).map(DoubleKey)
    }
}

impl<T> Serialize for DoubleKey<T>
where
    T: Serialize,
{
    #[inline]
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        self.0.serialize(serializer)
    }
}
