const DATA_U64_SIZE: usize = 4;
const DATA_U8_SIZE: usize = 32;
const ONE_I128: i128 = 0x1_0000_0000_0000_0000;
const ONE_F64: f64 = ONE_I128 as f64;

use rand_core::{OsRng, RngCore};
use serde::{Deserialize, Serialize};
use sha2::Digest;
use sha2::Sha256;
use std::fmt::Formatter;
use std::num::Wrapping;

/// CKey implements a consistent hask key.
/// Theory here: https://en.wikipedia.org/wiki/Consistent_hashing
///
/// # Examples
///
/// ```
/// use conflictdb::CKey;
///
/// let k1 = CKey::new_rand();
/// let k2 = k1.next();
/// print!("k1:{} k2:{}\n", k1, k2);
/// ```
#[derive(Debug, Default, Clone, Copy, Serialize, Deserialize, PartialEq, Eq)]
pub struct CKey {
    data: [u64; DATA_U64_SIZE],
}

impl CKey {
    /// Make a digest from data and returns a new key from it.
    /// Typical usage is to create a a CKey from a string or an object content.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_digest("/a/unique/path");
    /// assert_eq!("3b818742b0f27cfc10f4de1e5ba5594c975d580c412a31011ad58d36a2b17cdc", format!("{}",k));
    /// ```
    pub fn new_digest(data: impl AsRef<[u8]>) -> Self {
        let mut hasher = Sha256::default();
        hasher.update(data);
        let hash = hasher.finalize();
        bincode::deserialize(&hash).unwrap()
    }

    /// Generate a random key. This can be time consuming as it is using
    /// the OS random generation, which is better from a cryptographic
    /// point of view, but possibly slower than a standard prng.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_rand();
    /// print!("k: {}", k);
    /// ```
    pub fn new_rand() -> Self {
        let mut hash = [0u8; DATA_U8_SIZE];
        OsRng.fill_bytes(&mut hash);
        bincode::deserialize(&hash).unwrap()
    }

    /// Generate a key with the value 0.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_zero();
    /// assert_eq!("0000000000000000000000000000000000000000000000000000000000000000", format!("{}", k));
    /// ```
    pub fn new_zero() -> Self {
        Self::default()
    }

    /// Generate a key with the value 1, the smallest key possible just after zero.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_unit();
    /// assert_eq!("0000000000000000000000000000000000000000000000000000000000000001", format!("{}", k));
    /// ```
    pub fn new_unit() -> Self {
        Self { data: [0, 0, 0, 1] }
    }

    /// Generate a key with the last, highest possible value.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_last();
    /// assert_eq!("ffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff", format!("{}", k));
    /// ```
    pub fn new_last() -> Self {
        Self {
            data: [
                0xffffffffffffffff,
                0xffffffffffffffff,
                0xffffffffffffffff,
                0xffffffffffffffff,
            ],
        }
    }

    /// Generate a key which is exactly in the middle of the ring.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_half();
    /// assert_eq!("8000000000000000000000000000000000000000000000000000000000000000", format!("{}", k));
    /// ```
    pub fn new_half() -> Self {
        Self {
            data: [
                0x8000000000000000,
                0x0000000000000000,
                0x0000000000000000,
                0x0000000000000000,
            ],
        }
    }

    /// Returns true if self is inside lower and upper.
    /// Lower limit is excluded, upper is included.
    /// This is typically used in a ring to know if a given
    /// key is handled by a node. You would ask if key
    /// is inside previous node (lower) and this node (upper)
    /// and if true -> yes, this is the right node to handle it.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k1 = CKey::from(0.1);
    /// let k2 = CKey::from(0.3);
    /// let k3 = CKey::from(0.9);
    /// assert!(k2.inside(k1, k3));
    /// ```
    pub fn inside(self, lower: CKey, upper: CKey) -> bool {
        if lower == upper {
            // If lower and upper are equal, consider it matches all the ring,
            // so any value is inside the interval.
            return true;
        }
        // Upper limit is included within the interval. Lower is not.
        if self == lower {
            return false;
        }
        if self == upper {
            return true;
        }
        // Now compare member by member, since we made a diff initially,
        // self and upper are correctly ordered, according to lower.
        let lower_to_self = self - lower;
        let lower_to_upper = upper - lower;
        for i in 0..DATA_U64_SIZE {
            if lower_to_self.data[i] > lower_to_upper.data[i] {
                return false;
            }
            if lower_to_self.data[i] < lower_to_upper.data[i] {
                return true;
            }
        }
        // This should be unreachable code as to get here, we need to
        // have self == upper and this is a special case was tested before.
        true
    }

    /// Increment current key by one.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let mut k = CKey::new_zero();
    /// k.incr();
    /// assert_eq!(CKey::new_unit(), k);
    /// ```
    pub fn incr(&mut self) {
        *self = self.next();
    }

    /// Decrement current key by one.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let mut k = CKey::new_zero();
    /// k.decr();
    /// assert_eq!(CKey::new_last(), k);
    /// ```
    pub fn decr(&mut self) {
        *self = self.prev();
    }

    /// Return current key plus one.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_zero();
    /// assert_eq!(CKey::new_unit(), k.next());
    /// ```
    pub fn next(self) -> CKey {
        self + Self::new_unit()
    }

    /// Return current key minus one.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_zero();
    /// assert_eq!(CKey::new_last(), k.prev());
    /// ```
    pub fn prev(self) -> CKey {
        self - Self::new_unit()
    }

    /// Return a short ID in an u64. Only the lower 60 bits are used.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::from(0.03);
    /// assert_eq!("007ae147ae147ae0", format!("{:016x}", k.short_id_u64()));
    /// ```
    pub fn short_id_u64(self) -> u64 {
        self.data[0] >> 4
    }

    /// Return a short ID in an u32. Only the lower 28 bits are used.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::from(0.03);
    /// assert_eq!("007ae147", format!("{:08x}", k.short_id_u32()));
    /// ```
    pub fn short_id_u32(self) -> u32 {
        (self.data[0] >> 36) as u32
    }

    /// Return a short ID in a string. Only 7 chars (28 bits).
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::from(0.03);
    /// assert_eq!("07ae147", k.short_id_str());
    /// ```
    pub fn short_id_str(self) -> String {
        format!("{:07x}", self.short_id_u32())
    }

    /// Return the position of the key within the ring.
    ///
    /// # Examples
    /// ```
    /// use conflictdb::CKey;
    ///
    /// let k = CKey::new_half();
    /// assert_eq!(0.5, k.ring_pos());
    /// ```
    pub fn ring_pos(self) -> f64 {
        self.into()
    }
}

impl From<f64> for CKey {
    fn from(f: f64) -> CKey {
        let f_mod_1: f64 = if f < 0.0 || f >= 1.0 {
            f.rem_euclid(1.0)
        } else {
            f
        };
        let v = (ONE_F64 * f_mod_1) as u64;
        CKey { data: [v, v, v, v] }
    }
}

impl From<CKey> for f64 {
    // We implement From and not Into as recommended by rust doc,
    // the Into implementation is deduced from From.
    fn from(k: CKey) -> f64 {
        (k.data[0] as f64) / ONE_F64
    }
}

impl std::convert::TryFrom<String> for CKey {
    type Error = hex::FromHexError;

    fn try_from(value: String) -> Result<CKey, Self::Error> {
        if value.len() != 64 {
            return Err(hex::FromHexError::InvalidStringLength);
        }
        let hash = hex::decode(value)?;
        Ok(bincode::deserialize(&hash).unwrap())
    }
}

impl From<CKey> for String {
    fn from(k: CKey) -> String {
        format!("{}", k)
    }
}

impl std::ops::Add<CKey> for CKey {
    type Output = Self;

    fn add(self, other: CKey) -> CKey {
        let mut old_carry = 0;
        let mut ret: CKey = CKey::default();
        for i in 0..DATA_U64_SIZE {
            let j = DATA_U64_SIZE - i - 1;
            let (v, new_carry) = match self.data[j].checked_add(other.data[j]) {
                Some(v1) => match v1.checked_add(old_carry) {
                    Some(v2) => (v2, 0),
                    None => ((Wrapping(v1) + Wrapping(old_carry)).0, 1),
                },
                None => (
                    (Wrapping(self.data[j]) + Wrapping(other.data[j]) + Wrapping(old_carry)).0,
                    1,
                ),
            };
            ret.data[j] = v;
            old_carry = new_carry;
        }
        ret
    }
}

impl std::ops::Sub<CKey> for CKey {
    type Output = Self;

    fn sub(self, other: CKey) -> CKey {
        let mut old_carry = 0;
        let mut ret: CKey = CKey::default();
        for i in 0..DATA_U64_SIZE {
            let j = DATA_U64_SIZE - i - 1;
            let (v, new_carry) = match self.data[j].checked_sub(other.data[j]) {
                Some(v1) => match v1.checked_sub(old_carry) {
                    Some(v2) => (v2, 0),
                    None => ((Wrapping(v1) - Wrapping(old_carry)).0, 1),
                },
                None => (
                    (Wrapping(self.data[j]) - Wrapping(other.data[j]) - Wrapping(old_carry)).0,
                    1,
                ),
            };
            ret.data[j] = v;
            old_carry = new_carry;
        }
        ret
    }
}

impl std::ops::Add<f64> for CKey {
    type Output = Self;

    fn add(self, delta: f64) -> CKey {
        let other = CKey::from(delta);
        self + other
    }
}

impl std::ops::Sub<f64> for CKey {
    type Output = Self;

    fn sub(self, delta: f64) -> CKey {
        let other = CKey::from(delta);
        self - other
    }
}

impl std::fmt::Display for CKey {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "{:016x}{:016x}{:016x}{:016x}",
            self.data[0], self.data[1], self.data[2], self.data[3]
        )
    }
}

#[cfg(test)]
mod tests {
    use super::CKey;

    #[test]
    fn test_ckey_fmt_display() {
        let k = CKey {
            data: [3, 7, 11, 15],
        };
        assert_eq!(
            "00000000000000030000000000000007000000000000000b000000000000000f",
            format!("{}", k)
        );
    }
}
