//! Usage :
//! ```
//! #[derive(Serialize, Deserialize, Debug, Default)]
//! #[serde(default)]
//! struct Config {
//!     title: String,
//!
//! }
//!
//! fn main() {
//!     let mut cfg = Config::from_toml_file("config.toml").unwrap();
//!     cfg.title = String::from("Jean");
//!     cfg.save_to_file("config.toml").unwrap();
//! }
//!
//! ```
use serde::{de::DeserializeOwned, Serialize};
use std::{fs, io::Result};

/// This trait represent anything that can be parsed from toml file
pub trait FromTomlFile {
    fn from_toml_file(filename: &str) -> Result<Self>
    where
        Self: Sized;
    fn save_to_file(&self, filename: &str) -> Result<()>;
}
impl<T: DeserializeOwned + Serialize> FromTomlFile for T {
    /// Implementation of FromTomFile for any Deserializable data type
    fn from_toml_file(filename: &str) -> Result<Self> {
        let content = fs::read_to_string(filename)?;
        let result: Self = toml::from_str(&content)?;
        Ok(result)
    }
    fn save_to_file(&self, filename: &str) -> Result<()> {
        let data = toml::to_string_pretty(&self).unwrap();
        fs::write(filename, data)
    }
}

#[cfg(test)]
mod tests {
    use super::FromTomlFile;
    use serde::{Deserialize, Serialize};
    use std::fs;

    #[derive(Deserialize, Serialize, Debug, Default)]
    #[serde(default)]
    struct Config {
        title: String,
        blank: String,
    }

    #[test]
    fn test_config() {
        const CONF_FILE: &str = "rust_test_config.toml";
        fs::write(CONF_FILE, "title='Jean'").expect("Failed to create mock config file.");
        let cfg = Config::from_toml_file(CONF_FILE).unwrap();
        assert_eq!(cfg.title, "Jean");
        assert_eq!(cfg.blank, "");
        fs::remove_file(CONF_FILE).expect("Can't remove file");
    }
}
