//! Usage :
//! ```
//! #[derive(Deserialize, Debug, Default)]
//! #[serde(default)]
//! struct Config {
//!     title: String,
//!     blank: String,
//! }
//!
//! fn main() {
//!     let cfg = Config::from_toml_file("config.toml").unwrap();
//! }
//!
//! ```
use serde::de::DeserializeOwned;
use std::{fs, io::Result};

/// This trait represent anything that can be parsed from toml file
pub trait FromTomlFile {
    fn from_toml_file(filename: &str) -> Result<Self>
    where
        Self: Sized;
}
impl<T: DeserializeOwned> FromTomlFile for T {
    /// Implementation of FromTomFile for any Deserializable data type
    fn from_toml_file(filename: &str) -> Result<Self> {
        let content = fs::read_to_string(filename)?;
        println!("{}", content);
        let result: Self = toml::from_str(&content)?;
        Ok(result)
    }
}

#[cfg(test)]
mod tests {
    use super::FromTomlFile;
    use serde::Deserialize;
    use std::fs;

    #[derive(Deserialize, Debug, Default)]
    #[serde(default)]
    struct Config {
        title: String,
        blank: String,
    }

    #[test]
    fn test_config() {
        const CONF_FILE: &str = "rust_test_config.toml";
        fs::write(CONF_FILE, "title='Jean'").expect("Failed to create mock config file.");
        let cfg = Config::from_toml_file(CONF_FILE).unwrap();
        assert_eq!(cfg.title, "Jean");
        assert_eq!(cfg.blank, "");
        fs::remove_file(CONF_FILE).expect("Can't remove file");
    }
}
