use crate::ckey::CKey;
use serde::{Deserialize, Serialize};
use std::fmt::Formatter;
use url::Url;

/// EP stands for End Point.
///
/// It contains all the informations required to, typically,
/// contact a node. It's composed of a Key and a URL. Neither the Key or the URL
/// could be enough. Obviously, when you only have the Key, you don't
/// know where to call the node. And if you only have the URL, you still
/// need the Key because a given URL might respond for several different
/// virtual nodes.
#[derive(Debug, Clone, Serialize, Deserialize, PartialEq, Eq)]
pub struct EP {
    key: CKey,
    urls: Vec<Url>,
}

pub const LOCAL: &str = "local";

impl EP {
    fn parse_urls_str(urls: Vec<&str>) -> Result<Vec<Url>, Box<dyn std::error::Error>> {
        let mut parsed_urls = Vec::<Url>::new();
        for url in urls.into_iter() {
            let parsed_url = Url::parse(url)?;
            parsed_urls.push(parsed_url);
        }
        Ok(parsed_urls)
    }

    fn parse_urls_string(urls: Vec<String>) -> Result<Vec<Url>, Box<dyn std::error::Error>> {
        let mut parsed_urls = Vec::<Url>::new();
        for url in urls.into_iter() {
            let parsed_url = Url::parse(url.as_str())?;
            parsed_urls.push(parsed_url);
        }
        Ok(parsed_urls)
    }

    /// Generate a new local endpoint.
    ///
    /// # Examples
    /// ```
    /// use confitdb::EP;
    ///
    /// let e = EP::new_local();
    /// print!("e: {:?}", e);
    /// ```
    pub fn new_local() -> EP {
        let key = CKey::new_rand();
        let urls = vec![format!("{}:{}", LOCAL, key)];
        EP {
            key,
            urls: Self::parse_urls_string(urls).unwrap(),
        }
    }

    pub fn parse(key: &str, urls: Vec<&str>) -> Result<EP, Box<dyn std::error::Error>> {
        let parsed_key = CKey::parse(key)?;
        let parsed_urls = Self::parse_urls_str(urls)?;
        Ok(EP {
            key: parsed_key,
            urls: parsed_urls,
        })
    }
}

impl std::fmt::Display for EP {
    /// Pretty-print an endpoint.
    ///
    /// # Examples
    /// ```
    /// use confitdb::EP;
    /// use std::convert::TryFrom;
    ///
    /// let e = EP::parse("12345678ffffffff01234567fffffffff00123456ffffffff0012345ffffffff",vec!["http://just-a-test.com"]).unwrap();
    /// assert_eq!("{\"key\":\"0.071111111\",\"urls\":[\"http://just-a-test.com/\"]", format!("{}", e));
    /// ```
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{{\"key\":\"{}\",\"urls\":[", self.key)?;
        for i in 0..self.urls.len() {
            if i > 0 {
                write!(f, ",")?;
            }
            write!(f, "\"{}\"", self.urls[i].as_str())?;
        }
        write!(f, "]")
    }
}
