# What we want

- TOML/YAML Template (would be great if that could be configured)
- An easy to understand API
- Good error message in deserialization
- Maybe auto reloading
- Maybe Meta information where values came from


## Use cases for loading files

- Path with known extension that identifies file type -> &Path
- Path without extension -> manually specify file type
- Path with unknown extension, we want to allow exactly one of the supported file types
- only one of a list of sources


## Differences between leaf and inner nodes

- empty
    - leaf: None
    - obj: ty::empty()
- default_values()
    - leaf: None or deserialize
    - obj: ty::default_values
- with_fallback
    - leaf: Option::or
    - obj: with_fallback
- partial struct
    - leaf: Make Option
    - obj: never Option
- from_partial
    - leaf: ok_or/copy
    - obj: ty::from_partial


## TODO
- [x] meta
- [x] generate TOML
- [x] Convenience methods for trait
- [x] `Partial::is_empty`
- [x] `Partial::is_complete`
- [x] Optional with `default` makes no sense!
- [x] Think about whether optional nested makes sense
- [x] `Config::from_file`
- [x] env
- [x] `yaml::format`
- [x] LICENSE
- [x] docs
- [x] CI
- [x] Remove extra traits
- [x] change path dependency
- [x] Cargo Metadata
- [x] README

### Soon
- [ ] `format` should mention env variables
- [ ] Add `from` attribute to deserialize into another type and then use `From` or `TryFrom`
    - Or sth like `parse`
- [ ] Validation
- [ ] Rename?
- [ ] Forward other derives??? (or maybe not, just derive Debug for Partial)
    - We cannot see other derives, only in special circumstances
    - Thus we probably want `#[config(derive_for_partial = Debug)]` or sth
    - Maybe a more general attribute forward?

### Later
- [ ] more `toml::format` options
- [ ] Tests
- [ ] Make sure error messages when misusing proc macro are good. Also when fields don't implement `Deserialize` and stuff.
- [ ] Arrays as default
- [ ] `FirstFile` or sth like that? -> Probably not at all, we don't have to offer logic like that
- [ ] `config.{toml,yaml}` kind of syntax
- [ ] think about env deserialization again






-----------------


# Trait with `&self`
- Preloaded requires cloning
+ Works with slices
- Trait objects meh
- Potentially overlapping impls

```rust
Config::from_sources(&[
    &confique::Env,
    &Path::new("tobira.toml"),
    &Path::new("/etc/tobira/config.toml"),
]);
```


# Trait with `self`
+ Preloaded works fine
- Other things need `for &T` impl or sth
- No dyanmic dispatch
- Slice doesnt work, need tuples -> ugly API -> cannot build dynamic
- Potentially overlapping impls

```rust
Config::from_sources((
    confique::Env,
    Path::new("tobira.toml"),
    Path::new("/etc/tobira/config.toml"),
));
```


# Enum
- Closed polymorphism
- Has potentially many generic parameters, shitty

```rust
Config::from_sources([
    Source::Env,
    Source::File(Path::new("tobira.toml")),
    Source::File(Path::new("/etc/tobira/config.toml")),
));
```

# Builder
- First name questionable: `loader` or `builder` are fine, but `with_fallback` then? Seems strange
- Kind of the same job as `Partial`

## Without trait
- Long names `with_fallback_file`

```rust
Config::builder()
    .env()
    .optional_file("tobira.toml")
    .required_file("/etc/tobira/config.toml")
    .load()
```


## With trait
-
- Potentially overlapping impls

```rust
Config::builder()
    .env()
    .with_fallback(File::new("tobira.toml"))
    .with_fallback(File::with_formats("/etc/tobira", [FileFormat::Toml, FileFormat::Yaml]))
    .load()
```

## Macro
- cannot be build dynamically
- more magic






------------------


- Low level: `Partial::with_fallback` and `Deserialize`
- Mid level: ???
    - Some `File` stuff?
- High Level: ???
    - Super specific:
        - `Config::from_file`
        - `Config::from_env`
    - Builder or "list" (slice/tuple)
        - Not important: preloaded partials
        - Not important: dynamically building lists of sources
        - cloning does not matter! As long as it's not evident from the API
        - We problably still want to be explicit about files, i.e. use `File` not just `Path`


```rust
Config::from_sources(&[
    &Env::new("TOBIRA_"),
    &File::new("config.toml"),
    &File::new("/etc/tobira/config.toml"),
])

Config::from_sources(&[
    &confique::Env::new("TOBIRA_"),
    &confique::File::new("config.toml"),
    &confique::File::new("/etc/tobira/config.toml"),
])

Config::from_sources(&[
    &confique::env("TOBIRA_"),
    &confique::file("config.toml"),
    &confique::file("/etc/tobira/config.toml"),
])

Config::builder()
    .env("TOBIRA_")
    .file("config.toml")
    .file("/etc/tobira/config.toml")
    .load()
```
