use std::fs::{create_dir_all, File};
use std::path::{Path, PathBuf};

/// Reexport of Attribute Macros
pub use configr_derive::Configr;
use snafu::{OptionExt, ResultExt};
pub use toml;

const CONFIG_FILE_NAME: &str = "config.toml";

/// List of error categories
#[derive(snafu::Snafu, Debug)]
pub enum ConfigError {
	/// Loading the config.toml file failed.
	#[snafu(display("Unable to read configuration file from {}: {}", path.display(), source))]
	ReadConfig { source: std::io::Error, path: PathBuf },
	/// Creating the directory or file failed.
	#[snafu(display("Unable to create configuration file or directory {}: {}", path.display(), source))]
	CreateFs { source: std::io::Error, path: PathBuf },
	/// TOML parsing failed in some way.
	#[snafu(display("Unable to parse TOML\n{}\n```\n{}```{}", path.display(), toml, source))]
	Deserialize {
		source: toml::de::Error,
		path: PathBuf,
		toml: String,
	},
	/// Unable to get the configuration directory, possibly because of
	/// an unsupported OS.
	#[snafu(display(
		"Unable to get config directory from OS, if you believe this is an error please file an issue on \
		 the `dirs` crate"
	))]
	ConfigDir,
}

type Result<T, E = ConfigError> = std::result::Result<T, E>;

/// This is the main trait that you implement on your struct, either
/// manually or using the [`Configr`][configr_derive::Configr]
/// attribute macro
///
/// ```no_run
/// use configr::{Config, Configr};
/// #[derive(Configr, Default, serde::Serialize, serde::Deserialize)]
/// pub struct BotConfig {
///     bot_username: String,
///     client_id: String,
///     client_secret: String,
///     channel: String,
/// }
///
/// let config = BotConfig::load("bot-app", true).unwrap();
/// ```
pub trait Config<C>
where
	C: serde::ser::Serialize + serde::de::DeserializeOwned + Config<C> + Default,
{
	/// Load the config from the config file located in the OS
	/// specific config directory\
	/// This is a wrapper around
	/// [`load_custom`][Self::load_custom] which just
	/// takes the system configuration directory, instead of a custom
	/// path.\
	/// Read [`load_custom`][Self::load_custom] for more
	/// informationg about failure and config folder structure
	///
	/// # Failures
	/// this will contains the same failure possibilities as
	/// [`load_custom`][Self::load_custom] in addtion this can
	/// also fail due to the user configuration path not being found,
	/// if this is the case, you should switch to using
	/// `load_custom` or [`load_specific`][Self::load_specific] with a
	/// custom path
	///
	/// # Notes
	/// This should in almost every case be prefered over supplying
	/// your own configuration directory.
	///
	/// The `force_user_dir` option makes sure the fuction always
	/// prefers the user configuration path, compared to using /etc on
	/// UNIX systems and besides the executable on other systems, if
	/// the user configuration file is not found
	///
	/// The configuration directory is as follows\
	/// Linux: `$XDG_CONFIG_HOME/`\
	/// Windows: `%APPDATA%/`\
	/// Mac OS: `$HOME/Library/Application Support/`
	fn load(
		app_name: &str,
		force_user_dir: bool,
	) -> Result<C> {
		let mut dir = dirs::config_dir().context(ConfigDir)?;
		dir.push(app_name.replace(" ", "-").to_ascii_lowercase());
		dir.push(CONFIG_FILE_NAME);
		if !force_user_dir && !dir.exists() {
			if let Ok(c) = if cfg!(target_family = "unix") {
				Self::load_custom(app_name, &mut PathBuf::from("/etc"))
			} else {
				Self::load_custom(app_name, &mut PathBuf::from("./"))
			} {
				return Ok(c);
			}
		};
		Self::load_specific(&dir)
	}

	/// Load the config from the config file located in the app
	/// specific config directory which is
	/// `config_dir/app-name/config.toml`
	///
	/// # Notes
	/// This should only be used in the case you are running this on a
	/// system which you know doesn't have a configuration directory.
	///
	/// The app_name will be converted to lowercase-kebab-case
	///
	/// # Failures
	/// This function will Error under the following circumstances\
	/// * If the config.toml or the app-name directory could not be
	///   created\
	/// * If the config.toml could not be read properly\
	/// * If the config.toml is not valid toml data
	fn load_custom(
		app_name: &str,
		config_dir: &mut PathBuf,
	) -> Result<C> {
		// Get the location of the config file, create directories and the
		// file itself if needed.
		let config_dir = {
			config_dir.push(app_name.replace(" ", "-").to_ascii_lowercase());
			if !config_dir.exists() {
				create_dir_all(&config_dir).context(CreateFs { path: &config_dir })?;
			}
			config_dir.push(CONFIG_FILE_NAME);
			if !config_dir.exists() {
				let fd = File::create(&config_dir).context(CreateFs { path: &config_dir })?;
				Self::populate_template(fd).context(CreateFs { path: &config_dir })?;
				return Ok(C::default());
			}
			config_dir
		};

		Self::load_specific(&config_dir)
	}

	/// This is the completely barebones and should almost never be
	/// used over [`load_custom`][Self::load_custom] or
	/// [`load`][Self::load]
	fn load_specific(dir: &Path) -> Result<C> {
		let toml_content = std::fs::read_to_string(&dir).context(ReadConfig { path: &dir })?;

		toml::from_str::<C>(&toml_content).context(Deserialize {
			path: &dir,
			toml: &toml_content,
		})
	}

	fn populate_template(fd: File) -> std::io::Result<()> {
		use std::io::Write;
		let mut writer = std::io::BufWriter::new(fd);
		writer.write_all(
			toml::ser::to_string(&toml::Value::try_from(C::default()).unwrap())
				.unwrap()
				.as_bytes(),
		)?;
		writer.flush()?;
		Ok(())
	}
}

#[cfg(test)]
mod configr_tests {
	use configr::{Config, Configr};
	use serde::{Deserialize, Serialize};

	use crate as configr;

	#[derive(Configr, Serialize, Deserialize, Default, Debug, PartialEq)]
	struct TestConfig {
		a: String,
		b: String,
	}

	#[test]
	fn read_proper_config() {
		std::fs::create_dir("test-config2").unwrap();
		std::fs::write("test-config2/config.toml", b"a=\"test\"\nb=\"test\"\n").unwrap();
		let config = TestConfig::load_custom("Test Config2", &mut std::path::PathBuf::from("."));
		// expect toml parse error with correct fields but no actual values
		assert!(if let Ok(c) = config {
			c == TestConfig {
				a: "test".into(),
				b: "test".into(),
			}
		} else {
			false
		});

		std::fs::remove_dir_all("test-config2").unwrap();
	}

	#[test]
	fn serialized_config() {
		let config = TestConfig::load_custom("Test Config1", &mut std::path::PathBuf::from("."));
		assert!(if let Ok(c) = config {
			c == TestConfig {
				a: Default::default(),
				b: Default::default(),
			}
		} else {
			false
		});
		std::fs::remove_dir_all("test-config1").unwrap();
	}
}
