use crate::conn::Conn;
use std::{collections::HashMap, future::Future, pin::Pin};
use uuid::Uuid;

const DEFAULT_LANGUAGE: &str = "es-MX";

pub fn get_language(c: Conn) -> Pin<Box<dyn Future<Output = Conn>>> {
    Box::pin(get_language_conn(c))
}

pub fn get_session_id(c: Conn) -> Pin<Box<dyn Future<Output = Conn>>> {
    Box::pin(get_session_id_conn(c))
}

pub async fn get_language_conn(conn: Conn) -> Conn {
    let lang: &str = match conn.req_headers.get("accept-language") {
        Some(l) => l,
        None => DEFAULT_LANGUAGE,
    };

    let mut assigns: HashMap<String, String> = conn.assigns.clone();
    assigns.insert("language".to_owned(), lang.to_owned());
    return Conn {
        assigns: assigns,
        ..conn
    };
}

pub async fn get_session_id_conn(conn: Conn) -> Conn {
    match conn.req_headers.get("cookie") {
        Some(cookies) => {
            for c in cookies.split(";") {
                let v: Vec<&str> = c.split("=").collect();
                if v[0] == "confetti_session_id" {
                    let mut assigns: HashMap<String, String> = conn.assigns.clone();
                    assigns.insert("session_id".to_owned(), v[1].to_owned());
                    return Conn {
                        assigns: assigns,
                        ..conn
                    };
                }
            }
            return add_new_session(conn);
        }
        None => {
            return add_new_session(conn);
        }
    };
}

fn generate_session_id() -> String {
    let new_uuid = Uuid::new_v4();
    return new_uuid.to_string();
}

fn add_new_session(conn: Conn) -> Conn {
    let session_id = generate_session_id();
    let mut assigns: HashMap<String, String> = conn.assigns.clone();
    assigns.insert("session_id".to_owned(), session_id.clone());

    let mut resp_headers: HashMap<String, String> = conn.resp_headers.clone();
    resp_headers.insert(
        "Set-Cookie".to_owned(),
        format!("confetti_session_id={};", session_id),
    );
    return Conn {
        assigns: assigns,
        resp_headers: resp_headers,
        ..conn
    };
}
