use crate::web::{Method, Request, Response};
use std::collections::HashMap;
use web_sys::Url;

// Conn to be sent to plugs
pub struct Conn {
    // Request
    pub host: String,
    pub method: Method,
    pub port: u16,
    pub scheme: String,
    pub path: String,
    pub query_string: String,
    pub req_body: String,
    pub req_headers: HashMap<String, String>,
    pub cf: HashMap<String, String>,
    // Response
    pub resp_body: String,
    pub resp_headers: HashMap<String, String>,
    pub status: u16,
    // Connection
    pub assigns: HashMap<String, String>,
    pub halted: bool,
}

pub fn create_conn(request: Request) -> Conn {
    let method: Method = get_method(request.method);
    let url = Url::new(request.url.as_str()).unwrap();
    let port: u16 = url.port().parse::<u16>().unwrap_or(80);
    let host = url.host();
    Conn {
        method,
        port,
        host,
        scheme: url.protocol(),
        assigns: HashMap::new(),
        cf: request.cf,
        halted: false,
        path: url.pathname(),
        query_string: url.search(),
        req_body: request.body,
        req_headers: request.headers,
        resp_body: String::new(),
        resp_headers: HashMap::new(),
        status: 200,
    }
}

pub fn conn_to_response(conn: Conn) -> Response {
    return Response {
        status: conn.status,
        headers: conn.resp_headers,
        body: conn.resp_body,
    };
}

pub fn html(conn: Conn, content: &str) -> Conn {
    let mut headers = conn.resp_headers;
    headers.insert(
        "Content-Type".to_string(),
        "text/html; charset=utf-8".to_string(),
    );
    return Conn {
        resp_headers: headers,
        resp_body: content.to_string(),
        status: 200,
        ..conn
        //
        // host: conn.host.clone(),
        // method: conn.method.clone(),
        // port: conn.port,
        // scheme: conn.scheme.clone(),
        // path: conn.path.clone(),
        // query_string: conn.query_string.clone(),
        // req_body: conn.req_body.clone(),
        // req_headers: conn.req_headers.clone(),
        // cf: conn.cf.clone(),
        // assigns: conn.assigns.clone(),
        // halted: conn.halted,
    };
}

pub fn json(conn: Conn, content: &str) -> Conn {
    let mut headers = conn.resp_headers;
    headers.insert(
        "Content-Type".to_string(),
        "application/json; charset=utf-8".to_string(),
    );
    return Conn {
        resp_headers: headers,
        resp_body: content.to_string(),
        status: 200,
        ..conn
        // //
        // host: conn.host.clone(),
        // method: conn.method.clone(),
        // port: conn.port,
        // scheme: conn.scheme.clone(),
        // path: conn.path.clone(),
        // query_string: conn.query_string.clone(),
        // req_body: conn.req_body.clone(),
        // req_headers: conn.req_headers.clone(),
        // cf: conn.cf.clone(),
        // assigns: conn.assigns.clone(),
        // halted: conn.halted,
    };
}

pub fn xml(conn: Conn, content: &str) -> Conn {
    let mut headers = conn.resp_headers;
    headers.insert(
        "Content-Type".to_string(),
        "application/xml; charset=utf-8".to_string(),
    );
    return Conn {
        resp_headers: headers,
        resp_body: content.to_string(),
        status: 200,
        ..conn
        //
        // host: conn.host.clone(),
        // method: conn.method.clone(),
        // port: conn.port,
        // scheme: conn.scheme.clone(),
        // path: conn.path.clone(),
        // query_string: conn.query_string.clone(),
        // req_body: conn.req_body.clone(),
        // req_headers: conn.req_headers.clone(),
        // cf: conn.cf.clone(),
        // assigns: conn.assigns.clone(),
        // halted: conn.halted,
    };
}

fn get_method(method: String) -> Method {
    match method.to_uppercase().as_str() {
        "GET" => Method::Get,
        "HEAD" => Method::Head,
        "POST" => Method::Post,
        "PUT" => Method::Put,
        "DELETE" => Method::Delete,
        "CONNECT" => Method::Connect,
        "OPTIONS" => Method::Options,
        "TRACE" => Method::Trace,
        "PATCH" => Method::Patch,
        // Default to GET
        _ => Method::Get,
    }
}
