use crate::{aesni, software};
#[cfg(feature = "multithread")]
use rayon::{iter::IndexedParallelIterator, prelude::*};
use std::cmp::Ordering;

#[cfg(test)]
mod test;

#[cfg(all(
    test,
    target_arch = "x86_64",
    target_feature = "aes",
    target_feature = "sse2",
    target_feature = "rdseed"
))]
mod test_aes;

/// A trait for batched generators, i.e. generators that creates 128 bytes of random values at a
/// time.
pub trait AesBatchedGenerator: Clone {
    /// Instantiate a new generator from a secret key.
    fn new(key: Option<AesKey>) -> Self;
    /// Generates the batch corresponding to the given counter.
    fn generate_batch(&mut self, ctr: AesCtr) -> [u8; 128];
}

/// Represents the counter used by the AES block cipher to generate a set of values.
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct AesCtr(pub u128);

/// Represents the counter used to index on the batch-generated bytes.
#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct ByteCtr(pub u8);

/// Represents a key used in the AES ciphertext.
#[derive(Clone, Copy)]
pub struct AesKey(pub u128);

/// Whether the next `increment` call will need a new batch of values or not.
pub enum ShouldGenerateBatch {
    GenerateBatch,
    Wait,
}

/// A state that uniquely defines the next byte outputted by a given `AesCtrGenerator`.
///
/// To construct a generator that can yield a byte at each call to `generate_next`, we must be
/// able to store both the last AES counter used, and the index of the last outputted byte in the
/// batch. This structure contains both, and provides operations to manipulate those two values
/// correctly.
#[derive(Debug, Clone)]
pub struct State {
    aes_ctr: AesCtr,
    byte_ctr: ByteCtr,
}

impl State {
    /// Generate a new state from an aes counter and a byte counter.
    #[allow(dead_code)]
    pub fn new(aes_ctr: AesCtr, byte_ctr: ByteCtr) -> Self {
        debug_assert!(byte_ctr.0 <= 127);
        State { aes_ctr, byte_ctr }
    }

    /// Generates a new state from an AES counter value.
    pub fn from_aes_counter(aes_ctr: AesCtr) -> Self {
        State {
            aes_ctr,
            byte_ctr: ByteCtr(0),
        }
    }

    // Normalizes the current state, to be compatible with the offset of the other one.
    //
    // Two states with the same normalization are faster to compare.
    pub fn normalize_with(&mut self, other: &Self) {
        let State {
            aes_ctr: AesCtr(ref mut self_aes_ctr),
            byte_ctr: ByteCtr(ref mut self_byte_ctr),
        } = self;
        let State {
            aes_ctr: AesCtr(ref other_aes_ctr),
            ..
        } = other;
        let rem = (*self_aes_ctr - *other_aes_ctr) % 16;
        *self_aes_ctr -= rem;
        *self_byte_ctr += rem as u8 * 16;
    }

    // Return a clone of the state in normalized form, e.g. with the aes counter maximized, and
    // the byte counter minimized.
    pub fn to_normalized(&self) -> Self {
        let State {
            aes_ctr: AesCtr(aes_ctr),
            byte_ctr: ByteCtr(byte_ctr),
        } = self;
        let aes_ctr = aes_ctr + (byte_ctr / 16) as u128;
        let byte_ctr = byte_ctr % 16;
        State {
            aes_ctr: AesCtr(aes_ctr),
            byte_ctr: ByteCtr(byte_ctr),
        }
    }

    /// Returns the successor of the current state.
    pub fn successor(&self) -> Self {
        let State {
            byte_ctr: ByteCtr(byte_ctr),
            aes_ctr: AesCtr(aes_ctr),
        } = self;
        if *byte_ctr < 127 {
            State {
                aes_ctr: AesCtr(*aes_ctr),
                byte_ctr: ByteCtr(byte_ctr + 1),
            }
        } else {
            State {
                aes_ctr: AesCtr(*aes_ctr + 8),
                byte_ctr: ByteCtr(0),
            }
        }
    }

    /// Returns the byte counter.
    pub fn get_byte_counter(&self) -> ByteCtr {
        self.byte_ctr
    }

    /// Returns the aes counter.
    pub fn get_aes_counter(&self) -> AesCtr {
        self.aes_ctr
    }

    /// Returns the current index on the batch.
    pub fn get_batch_index(&self) -> usize {
        self.byte_ctr.0 as usize
    }

    /// Increment the state.
    pub fn increment(&mut self) -> ShouldGenerateBatch {
        let State {
            aes_ctr: AesCtr(ref mut aes_ctr),
            byte_ctr: ByteCtr(ref mut byte_ctr),
        } = self;
        if *byte_ctr < 127 {
            *byte_ctr += 1;
            ShouldGenerateBatch::Wait
        } else {
            *aes_ctr += 8;
            *byte_ctr = 0;
            ShouldGenerateBatch::GenerateBatch
        }
    }

    /// Increments the state of several bytes at a time.
    pub fn shift(&mut self, n_bytes: usize) -> ShouldGenerateBatch {
        let State {
            aes_ctr: AesCtr(ref mut aes_ctr),
            byte_ctr: ByteCtr(ref mut byte_ctr),
        } = self;
        let n_bytes = n_bytes + (*byte_ctr as usize);
        let aes_bump = ((n_bytes / 128) as u128) * 8;
        *byte_ctr = (n_bytes % 128) as u8;
        if aes_bump > 0 {
            *aes_ctr += aes_bump;
            ShouldGenerateBatch::GenerateBatch
        } else {
            ShouldGenerateBatch::Wait
        }
    }
}

impl PartialEq for State {
    fn eq(&self, other: &Self) -> bool {
        let self_norm = self.to_normalized();
        let other_norm = other.to_normalized();
        self_norm.aes_ctr == other_norm.aes_ctr && self_norm.byte_ctr == other_norm.byte_ctr
    }
}

impl PartialOrd for State {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        let self_norm = self.to_normalized();
        let other_norm = other.to_normalized();
        if self_norm.aes_ctr < other_norm.aes_ctr {
            Some(Ordering::Less)
        } else if self_norm.aes_ctr > other_norm.aes_ctr {
            Some(Ordering::Greater)
        } else if self_norm.byte_ctr < other_norm.byte_ctr {
            Some(Ordering::Less)
        } else if self_norm.byte_ctr > other_norm.byte_ctr {
            Some(Ordering::Greater)
        } else {
            Some(Ordering::Equal)
        }
    }
}

impl Ord for State {
    fn cmp(&self, other: &Self) -> Ordering {
        self.partial_cmp(other).unwrap()
    }
}

impl Eq for State {}

impl Default for State {
    fn default() -> Self {
        State {
            aes_ctr: AesCtr(0),
            byte_ctr: ByteCtr(0),
        }
    }
}

/// A generator that uses the software implementation.
pub type SoftAesCtrGenerator = AesCtrGenerator<software::Generator>;

/// A generator that uses the hardware implementation.
pub type HardAesCtrGenerator = AesCtrGenerator<aesni::Generator>;

/// A csprng which operates in batch mode.
#[derive(Clone)]
pub struct AesCtrGenerator<G: AesBatchedGenerator> {
    generator: G,
    state: State,
    bound: Option<State>,
    batch: [u8; 128],
}

impl<G: AesBatchedGenerator> AesCtrGenerator<G> {
    /// Generates a new csprng.
    ///
    /// If not given, the key is automatically selected, and the state is set to zero.
    pub fn new(
        key: Option<AesKey>,
        state: Option<State>,
        bound: Option<State>,
    ) -> AesCtrGenerator<G> {
        let mut generator = G::new(key);
        let state = state.unwrap_or_default();
        if let Some(ref actual_bound) = bound {
            debug_assert!(state <= *actual_bound);
        }
        let batch = generator.generate_batch(state.aes_ctr);
        AesCtrGenerator {
            generator,
            state,
            bound,
            batch,
        }
    }

    /// Returns the state of the current generator.
    pub fn get_state(&self) -> &State {
        &self.state
    }

    /// Returns the bound of the generator if any.
    pub fn get_bound(&self) -> Option<&State> {
        self.bound.as_ref()
    }

    /// Returns whether the generator is bounded.
    pub fn is_bounded(&self) -> bool {
        self.bound.is_some()
    }

    /// Returns the number of remaining bytes if bounded
    pub fn remaining_bytes(&self) -> Option<usize> {
        self.bound.as_ref().map(|bound| {
            let res = ((bound.aes_ctr.0 - self.state.aes_ctr.0) as i128) * 16
                + (bound.byte_ctr.0 as i128 - self.state.byte_ctr.0 as i128);
            res as usize
        })
    }

    /// Yields the next random byte.
    pub fn generate_next(&mut self) -> u8 {
        let output = self.batch[self.state.get_batch_index()];
        if let Some(bound) = self.bound.as_ref() {
            let reached_bound =
                self.state.aes_ctr == bound.aes_ctr && self.state.byte_ctr == bound.byte_ctr;
            if reached_bound {
                panic!("Tried to generate a byte outside the generator bound.");
            }
        }
        match self.state.increment() {
            ShouldGenerateBatch::GenerateBatch => {
                self.batch = self.generator.generate_batch(self.state.get_aes_counter());
            }
            ShouldGenerateBatch::Wait => {}
        }
        output
    }

    /// Tries to fork the current generator into `n_child` generators each able to yield
    /// `child_bytes` random bytes.
    ///
    /// If the total number of bytes to be generated exceeds the bound of the current generator,
    /// `None` is returned. Otherwise, we return an iterator over the children generators.
    pub fn try_fork(
        &mut self,
        n_child: ChildCount,
        child_bytes: BytesPerChild,
    ) -> Option<impl Iterator<Item = AesCtrGenerator<G>>> {
        if !self.is_fork_in_bound(n_child, child_bytes) {
            return None;
        }
        let output = (0..n_child.0).map(self.get_creation_closure(child_bytes));
        self.regenerate_batch(n_child, child_bytes);
        Some(output)
    }

    /// Tries to fork the current generator into `n_child` generators each able to yield
    /// `child_bytes` random bytes as a parallel iterator.
    ///
    /// If the total number of bytes to be generated exceeds the bound of the current generator,
    /// `None` is returned. Otherwise, we return a parallel iterator over the children generators.
    ///
    /// # Notes
    ///
    /// This method necessitate the "multithread" feature.
    #[cfg(feature = "multithread")]
    pub fn par_try_fork(
        &mut self,
        n_child: ChildCount,
        child_bytes: BytesPerChild,
    ) -> Option<impl IndexedParallelIterator<Item = AesCtrGenerator<G>>>
    where
        G: Send + Sync,
    {
        if !self.is_fork_in_bound(n_child, child_bytes) {
            return None;
        }
        let output = (0..n_child.0)
            .into_par_iter()
            .map(self.get_creation_closure(child_bytes));
        self.regenerate_batch(n_child, child_bytes);
        Some(output)
    }

    fn regenerate_batch(&mut self, n_child: ChildCount, child_bytes: BytesPerChild) {
        let generate = self.state.shift(child_bytes.0 * n_child.0);
        if let ShouldGenerateBatch::GenerateBatch = generate {
            self.batch = self.generator.generate_batch(self.state.get_aes_counter());
        }
    }

    fn is_fork_in_bound(&self, n_child: ChildCount, child_bytes: BytesPerChild) -> bool {
        if let Some(ref actual_bound) = self.bound {
            let mut end = self.state.clone();
            end.shift(n_child.0 * child_bytes.0);
            if end > *actual_bound {
                return false;
            }
        }
        true
    }

    fn get_creation_closure(
        &self,
        child_bytes: BytesPerChild,
    ) -> impl Fn(usize) -> AesCtrGenerator<G> {
        let state = self.state.clone();
        let generator = self.generator.clone();
        move |i| {
            let mut new_state = state.clone();
            new_state.shift(child_bytes.0 * i);
            let mut new_bound = new_state.clone();
            new_bound.shift(child_bytes.0);
            new_bound.normalize_with(&new_state);
            let mut new_generator = generator.clone();
            let batch = new_generator.generate_batch(new_state.aes_ctr);
            AesCtrGenerator {
                generator: new_generator,
                state: new_state,
                bound: Some(new_bound),
                batch,
            }
        }
    }
}

/// The number of children created when a generator is forked.
#[derive(Debug, Copy, Clone)]
pub struct ChildCount(pub usize);

/// The number of bytes each children can generate, when a generator is forked.
#[derive(Debug, Copy, Clone)]
pub struct BytesPerChild(pub usize);
