#![allow(clippy::excessive_precision, clippy::approx_constant)]

use concrete_fftw::types::*;

use super::Complex64;
use crate::backends::core::private::math::tensor::{tensor_traits, Tensor};

/// Correcting factors for the forward transform.
#[derive(Debug, Clone)]
pub struct ForwardCorrector<Cont> {
    tensor: Tensor<Cont>,
}

tensor_traits!(ForwardCorrector);

/// Correcting factors for the backward transform.
#[derive(Debug, Clone)]
pub struct BackwardCorrector<Cont> {
    tensor: Tensor<Cont>,
}

tensor_traits!(BackwardCorrector);

/// A set of correcting factors allowing to perform the product modulo $(X^N+1)$ with fftw.
#[derive(Debug, Clone)]
pub struct Correctors {
    pub forward: ForwardCorrector<&'static [Complex64]>,
    pub backward: BackwardCorrector<&'static [Complex64]>,
}

impl Correctors {
    pub fn new(big_n: usize) -> Correctors {
        let (forward, backward) = match big_n {
            128 => (&TWIDDLES_128[0..128], &INVERSE_TWIDDLES_128[0..128]),
            256 => (&TWIDDLES_256[0..256], &INVERSE_TWIDDLES_256[0..256]),
            512 => (&TWIDDLES_512[0..512], &INVERSE_TWIDDLES_512[0..512]),
            1024 => (&TWIDDLES_1024[0..1024], &INVERSE_TWIDDLES_1024[0..1024]),
            2048 => (&TWIDDLES_2048[0..2048], &INVERSE_TWIDDLES_2048[0..2048]),
            4096 => (&TWIDDLES_4096[0..4096], &INVERSE_TWIDDLES_4096[0..4096]),
            8192 => (&TWIDDLES_8192[0..8192], &INVERSE_TWIDDLES_8192[0..8192]),
            16384 => (&TWIDDLES_16384[0..16384], &INVERSE_TWIDDLES_16384[0..16384]),
            _ => panic!("Wrong big_n when constructing corrector factors"),
        };
        Correctors {
            forward: ForwardCorrector {
                tensor: Tensor::from_container(forward),
            },
            backward: BackwardCorrector {
                tensor: Tensor::from_container(backward),
            },
        }
    }
}

static TWIDDLES_128: [c64; 128] = [
    c64::new(1.0, 0.0),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.9996988186962042, 0.024541228522912326),
];
static INVERSE_TWIDDLES_128: [c64; 128] = [
    c64::new(0.0078125, -0.0),
    c64::new(0.007810147021064096, -0.00019172834783525225),
    c64::new(0.007803089501602909, -0.00038334120568295324),
    c64::new(0.007791331692802267, -0.0005747231531224018),
    c64::new(0.0077748806771265385, -0.0007657589088246922),
    c64::new(0.007753746364052422, -0.0009563333999938765),
    c64::new(0.007727941484099852, -0.0011463318316825137),
    c64::new(0.007697481581163603, -0.0013356397559398533),
    c64::new(0.007662385003150238, -0.001524143140751002),
    c64::new(0.0076226728909260045, -0.0017117284387255453),
    c64::new(0.007578369165582375, -0.001898282655494249),
    c64::new(0.007529500514026874, -0.0020836934177726435),
    c64::new(0.0074760963729078814, -0.002267849041050487),
    c64::new(0.007418188910883099, -0.00245063859686634),
    c64::new(0.00735581300924235, -0.002631951979626719),
    c64::new(0.007289006240896398, -0.0028116799729295946),
    c64::new(0.007217808847744428, -0.002989714315352264),
    c64::new(0.0071422637164338335, -0.0031659477656639833),
    c64::new(0.007062416352526901, -0.003340274167424079),
    c64::new(0.0069783148530899635, -0.0035125885129266136),
    c64::new(0.006890009877721524, -0.0036827870064531066),
    c64::new(0.006797554618036808, -0.003850767126795188),
    c64::new(0.006701004765627126, -0.004016427689009544),
    c64::new(0.006600418478513337, -0.0041796689053679465),
    c64::new(0.006495856346113635, -0.004340392445465642),
    c64::new(0.006387381352746748, -0.004498501495451917),
    c64::new(0.006275058839692539, -0.004653900816347136),
    c64::new(0.006158956465832862, -0.004806496801411147),
    c64::new(0.006039144166896383, -0.00495619753252848),
    c64::new(0.005915694113331911, -0.005102912835576381),
    c64::new(0.005788680666835619, -0.005246554334742331),
    c64::new(0.005658180335558336, -0.005387035505758335),
    c64::new(0.005524271728019903, -0.005524271728019902),
    c64::new(0.0053870355057583355, -0.005658180335558335),
    c64::new(0.005246554334742332, -0.005788680666835617),
    c64::new(0.005102912835576382, -0.00591569411333191),
    c64::new(0.004956197532528481, -0.006039144166896382),
    c64::new(0.004806496801411148, -0.006158956465832861),
    c64::new(0.0046539008163471365, -0.006275058839692538),
    c64::new(0.004498501495451918, -0.006387381352746748),
    c64::new(0.004340392445465643, -0.006495856346113634),
    c64::new(0.004179668905367948, -0.006600418478513336),
    c64::new(0.004016427689009545, -0.006701004765627126),
    c64::new(0.0038507671267951887, -0.006797554618036807),
    c64::new(0.003682787006453108, -0.006890009877721523),
    c64::new(0.003512588512926614, -0.0069783148530899635),
    c64::new(0.0033402741674240797, -0.007062416352526901),
    c64::new(0.003165947765663983, -0.0071422637164338335),
    c64::new(0.0029897143153522644, -0.007217808847744428),
    c64::new(0.002811679972929596, -0.007289006240896397),
    c64::new(0.002631951979626719, -0.00735581300924235),
    c64::new(0.00245063859686634, -0.007418188910883099),
    c64::new(0.0022678490410504865, -0.007476096372907882),
    c64::new(0.002083693417772644, -0.007529500514026874),
    c64::new(0.0018982826554942499, -0.007578369165582375),
    c64::new(0.001711728438725545, -0.0076226728909260045),
    c64::new(0.0015241431407510024, -0.007662385003150238),
    c64::new(0.0013356397559398543, -0.007697481581163603),
    c64::new(0.0011463318316825137, -0.007727941484099852),
    c64::new(0.0009563333999938772, -0.007753746364052422),
    c64::new(0.0007657589088246935, -0.007774880677126538),
    c64::new(0.000574723153122402, -0.007791331692802267),
    c64::new(0.00038334120568295416, -0.007803089501602909),
    c64::new(0.0001917283478352521, -0.007810147021064096),
    c64::new(4.783776559169348e-19, -0.0078125),
    c64::new(-0.0001917283478352511, -0.007810147021064096),
    c64::new(-0.0003833412056829532, -0.007803089501602909),
    c64::new(-0.0005747231531224011, -0.007791331692802267),
    c64::new(-0.0007657589088246926, -0.0077748806771265385),
    c64::new(-0.0009563333999938763, -0.007753746364052422),
    c64::new(-0.0011463318316825128, -0.007727941484099852),
    c64::new(-0.0013356397559398535, -0.007697481581163603),
    c64::new(-0.0015241431407510015, -0.007662385003150238),
    c64::new(-0.0017117284387255442, -0.0076226728909260045),
    c64::new(-0.001898282655494249, -0.007578369165582375),
    c64::new(-0.002083693417772643, -0.007529500514026874),
    c64::new(-0.002267849041050486, -0.007476096372907882),
    c64::new(-0.0024506385968663395, -0.007418188910883099),
    c64::new(-0.0026319519796267183, -0.00735581300924235),
    c64::new(-0.002811679972929595, -0.007289006240896397),
    c64::new(-0.0029897143153522635, -0.007217808847744428),
    c64::new(-0.0031659477656639824, -0.0071422637164338335),
    c64::new(-0.003340274167424077, -0.007062416352526902),
    c64::new(-0.003512588512926615, -0.006978314853089963),
    c64::new(-0.003682787006453107, -0.006890009877721524),
    c64::new(-0.003850767126795187, -0.006797554618036808),
    c64::new(-0.004016427689009543, -0.006701004765627126),
    c64::new(-0.004179668905367945, -0.006600418478513338),
    c64::new(-0.00434039244546564, -0.006495856346113636),
    c64::new(-0.004498501495451917, -0.006387381352746748),
    c64::new(-0.004653900816347135, -0.006275058839692539),
    c64::new(-0.004806496801411146, -0.006158956465832862),
    c64::new(-0.004956197532528479, -0.006039144166896384),
    c64::new(-0.005102912835576379, -0.005915694113331912),
    c64::new(-0.005246554334742332, -0.005788680666835617),
    c64::new(-0.005387035505758335, -0.005658180335558336),
    c64::new(-0.005524271728019902, -0.005524271728019903),
    c64::new(-0.005658180335558335, -0.005387035505758336),
    c64::new(-0.005788680666835616, -0.0052465543347423324),
    c64::new(-0.0059156941133319116, -0.00510291283557638),
    c64::new(-0.006039144166896383, -0.00495619753252848),
    c64::new(-0.006158956465832861, -0.004806496801411148),
    c64::new(-0.006275058839692538, -0.0046539008163471365),
    c64::new(-0.006387381352746747, -0.004498501495451918),
    c64::new(-0.0064958563461136355, -0.004340392445465642),
    c64::new(-0.006600418478513337, -0.0041796689053679465),
    c64::new(-0.006701004765627126, -0.004016427689009545),
    c64::new(-0.006797554618036807, -0.0038507671267951887),
    c64::new(-0.006890009877721523, -0.003682787006453108),
    c64::new(-0.006978314853089963, -0.003512588512926616),
    c64::new(-0.007062416352526901, -0.0033402741674240784),
    c64::new(-0.0071422637164338335, -0.0031659477656639833),
    c64::new(-0.007217808847744428, -0.0029897143153522644),
    c64::new(-0.007289006240896397, -0.0028116799729295964),
    c64::new(-0.007355813009242349, -0.002631951979626721),
    c64::new(-0.007418188910883099, -0.002450638596866339),
    c64::new(-0.0074760963729078814, -0.002267849041050487),
    c64::new(-0.007529500514026874, -0.0020836934177726443),
    c64::new(-0.007578369165582375, -0.0018982826554942503),
    c64::new(-0.007622672890926004, -0.001711728438725547),
    c64::new(-0.007662385003150238, -0.0015241431407510045),
    c64::new(-0.007697481581163603, -0.001335639755939853),
    c64::new(-0.007727941484099852, -0.001146331831682514),
    c64::new(-0.007753746364052422, -0.0009563333999938777),
    c64::new(-0.007774880677126538, -0.0007657589088246941),
    c64::new(-0.007791331692802267, -0.0005747231531224043),
    c64::new(-0.007803089501602909, -0.0003833412056829529),
    c64::new(-0.007810147021064096, -0.00019172834783525255),
];

static TWIDDLES_256: [c64; 256] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9191138516900578, 0.3939920400610481),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9091679830905224, 0.41642956009763715),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.8986744656939538, 0.43861623853852766),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.8876396204028539, 0.46053871095824),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.8760700941954066, 0.4821837720791227),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8639728561215867, 0.5035383837257176),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8513551931052652, 0.524589682678469),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.838224705554838, 0.5453249884220465),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.8245893027850253, 0.5657318107836131),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.8104571982525948, 0.5857978574564389),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.7958369046088836, 0.6055110414043255),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.7807372285720945, 0.6248594881423863),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.765167265622459, 0.6438315428897914),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7491363945234594, 0.6624157775901718),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.7326542716724128, 0.680600997795453),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7157308252838186, 0.6983762494089729),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.6983762494089729, 0.7157308252838186),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6806009977954531, 0.7326542716724127),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6624157775901718, 0.7491363945234593),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.6438315428897915, 0.7651672656224588),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6248594881423865, 0.7807372285720944),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.6055110414043255, 0.7958369046088835),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5857978574564389, 0.8104571982525948),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5657318107836132, 0.8245893027850253),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5453249884220466, 0.838224705554838),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.524589682678469, 0.8513551931052652),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.5035383837257176, 0.8639728561215867),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.4821837720791229, 0.8760700941954065),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.46053871095824, 0.8876396204028539),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.4386162385385277, 0.8986744656939538),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.4164295600976373, 0.9091679830905224),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.3939920400610481, 0.9191138516900578),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3713171939518376, 0.9285060804732155),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.3484186802494345, 0.937339011912575),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.325310292162263, 0.9456073253805213),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.3020059493192282, 0.9533060403541938),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.27851968938505306, 0.9604305194155658),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.2548656596045146, 0.9669764710448521),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.23105810828067125, 0.9729399522055601),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.20711137619221856, 0.9783173707196277),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.18303988795514103, 0.9831054874312163),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.1588581433338614, 0.9873014181578584),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.13458070850712622, 0.99090263542778),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.11022220729388318, 0.9939069700023561),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.08579731234443988, 0.996312612182778),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.061320736302208655, 0.9981181129001492),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.03680722294135899, 0.9993223845883495),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(0.012271538285719944, 0.9999247018391445),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.012271538285719823, 0.9999247018391445),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.036807222941358866, 0.9993223845883495),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.06132073630220853, 0.9981181129001492),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.08579731234443977, 0.996312612182778),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.11022220729388307, 0.9939069700023561),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.1345807085071261, 0.99090263542778),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.15885814333386128, 0.9873014181578584),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.18303988795514095, 0.9831054874312163),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.20711137619221845, 0.9783173707196277),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.23105810828067117, 0.9729399522055602),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.2548656596045145, 0.9669764710448521),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.27851968938505295, 0.9604305194155659),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.3020059493192281, 0.9533060403541939),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.32531029216226287, 0.9456073253805214),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.3484186802494344, 0.937339011912575),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.37131719395183754, 0.9285060804732156),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.393992040061048, 0.9191138516900578),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.416429560097637, 0.9091679830905225),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.4386162385385274, 0.8986744656939539),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.4605387109582401, 0.8876396204028539),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.4821837720791227, 0.8760700941954066),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.5035383837257175, 0.8639728561215868),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5245896826784687, 0.8513551931052652),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.5453249884220461, 0.8382247055548382),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5657318107836132, 0.8245893027850253),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5857978574564389, 0.8104571982525948),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.6055110414043254, 0.7958369046088836),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.6248594881423862, 0.7807372285720946),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6438315428897913, 0.7651672656224591),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.6624157775901718, 0.7491363945234593),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.680600997795453, 0.7326542716724128),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.6983762494089728, 0.7157308252838187),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7157308252838186, 0.6983762494089729),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7326542716724127, 0.6806009977954532),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.749136394523459, 0.662415777590172),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.765167265622459, 0.6438315428897914),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7807372285720945, 0.6248594881423863),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.7958369046088835, 0.6055110414043255),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8104571982525947, 0.585797857456439),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.8245893027850251, 0.5657318107836135),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.8382247055548381, 0.5453249884220464),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.8513551931052652, 0.524589682678469),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8639728561215867, 0.5035383837257176),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.8760700941954065, 0.4821837720791229),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8876396204028538, 0.4605387109582402),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.8986744656939539, 0.43861623853852755),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9091679830905224, 0.41642956009763715),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.9191138516900578, 0.39399204006104815),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9285060804732155, 0.37131719395183765),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.9373390119125748, 0.3484186802494348),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9456073253805213, 0.3253102921622632),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.9533060403541939, 0.30200594931922803),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9604305194155658, 0.2785196893850531),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.9669764710448521, 0.2548656596045147),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9729399522055601, 0.2310581082806713),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.9783173707196275, 0.20711137619221884),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9831054874312164, 0.1830398879551409),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.9873014181578584, 0.15885814333386145),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.99090263542778, 0.13458070850712628),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.9939069700023561, 0.11022220729388325),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.996312612182778, 0.08579731234444016),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9981181129001492, 0.061320736302208495),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.9993223845883495, 0.03680722294135883),
    c64::new(-0.9996988186962042, 0.024541228522912326),
    c64::new(-0.9999247018391445, 0.012271538285720007),
];

static INVERSE_TWIDDLES_256: [c64; 256] = [
    c64::new(0.00390625, -0.0),
    c64::new(0.003905955866559158, -4.793569642859346e-05),
    c64::new(0.003905073510532048, -9.586417391762613e-05),
    c64::new(0.0039036030647982404, -0.00014377821461468294),
    c64::new(0.0039015447508014547, -0.00019167060284147662),
    c64::new(0.0038988988785162077, -0.00023953412618050226),
    c64::new(0.0038956658464011336, -0.0002873615765612009),
    c64::new(0.0038918461413389766, -0.00033514575134546834),
    c64::new(0.0038874403385632693, -0.0003828794544123461),
    c64::new(0.0038824491015717034, -0.0004305554972417307),
    c64::new(0.003876873182026211, -0.00047816669999693826),
    c64::new(0.0038707134196397657, -0.0005257058926059616),
    c64::new(0.003863970742049926, -0.0005731659158412568),
    c64::new(0.0038566461646791345, -0.0006205396223978963),
    c64::new(0.0038487407905818016, -0.0006678198779699266),
    c64::new(0.0038402558102781886, -0.0007149995623247693),
    c64::new(0.003831192501575119, -0.000762071570375501),
    c64::new(0.0038215522293735455, -0.0008090288132508538),
    c64::new(0.0038113364454630022, -0.0008558642193627726),
    c64::new(0.0038005466883029694, -0.0009025707354713715),
    c64::new(0.0037891845827911874, -0.0009491413277471245),
    c64::new(0.0037772518400189534, -0.000995568982830135),
    c64::new(0.003764750257013437, -0.0010418467088863217),
    c64::new(0.003751681716467054, -0.0010879675366603635),
    c64::new(0.0037380481864539407, -0.0011339245205252435),
    c64::new(0.0037238517201335698, -0.0011797107395282347),
    c64::new(0.0037090944554415495, -0.00122531929843317),
    c64::new(0.0036937786147676612, -0.0012707433287588396),
    c64::new(0.003677906504621175, -0.0013159759898133596),
    c64::new(0.003661480515283496, -0.0013610104697243538),
    c64::new(0.003644503120448199, -0.0014058399864647973),
    c64::new(0.0036269768768484984, -0.0014504577888743654),
    c64::new(0.003608904423872214, -0.001494857157676132),
    c64::new(0.003590288483164288, -0.0015390314064884691),
    c64::new(0.0035711318582169168, -0.0015829738828319916),
    c64::new(0.003551437433947353, -0.0016266779691313951),
    c64::new(0.0035312081762634505, -0.0016701370837120394),
    c64::new(0.003510447131617007, -0.0017133446817911237),
    c64::new(0.0034891574265449817, -0.0017562942564633068),
    c64::new(0.003467342267198648, -0.001798979339680625),
    c64::new(0.003445004938860762, -0.0018413935032265533),
    c64::new(0.003422148805450807, -0.0018835303596840731),
    c64::new(0.003398777309018404, -0.001925383563397594),
    c64::new(0.003374893969224948, -0.0019669468114285843),
    c64::new(0.003350502382813563, -0.002008213844504772),
    c64::new(0.003325606223067442, -0.0020491784479627693),
    c64::new(0.0033002092392566684, -0.0020898344526839733),
    c64::new(0.003274315256073586, -0.002130175736023619),
    c64::new(0.0032479281730568173, -0.002170196222732821),
    c64::new(0.003221051964004005, -0.0022098898858734888),
    c64::new(0.003193690676373374, -0.0022492507477259584),
    c64::new(0.0031658484306741983, -0.0022882728806892143),
    c64::new(0.0031375294198462693, -0.002326950408173568),
    c64::new(0.0031087379086284514, -0.0023652775054856467),
    c64::new(0.003079478232916431, -0.0024032484007055735),
    c64::new(0.003049754799109744, -0.0024408573755561967),
    c64::new(0.0030195720834481914, -0.00247809876626424),
    c64::new(0.0029889346313377303, -0.0025149669644132476),
    c64::new(0.0029578470566659553, -0.0025514564177881905),
    c64::new(0.002926314041107263, -0.0025875616312116085),
    c64::new(0.0028943403334178094, -0.0026232771673711654),
    c64::new(0.0028619307487203626, -0.0026585976476384884),
    c64::new(0.002829090167779168, -0.0026935177528791673),
    c64::new(0.0027958235362649164, -0.0027280322242538005),
    c64::new(0.0027621358640099515, -0.002762135864009951),
    c64::new(0.0027280322242538005, -0.0027958235362649164),
    c64::new(0.0026935177528791677, -0.0028290901677791675),
    c64::new(0.002658597647638489, -0.002861930748720362),
    c64::new(0.002623277167371166, -0.0028943403334178086),
    c64::new(0.0025875616312116085, -0.0029263140411072627),
    c64::new(0.002551456417788191, -0.002957847056665955),
    c64::new(0.002514966964413248, -0.00298893463133773),
    c64::new(0.0024780987662642406, -0.003019572083448191),
    c64::new(0.002440857375556197, -0.0030497547991097437),
    c64::new(0.002403248400705574, -0.0030794782329164306),
    c64::new(0.0023652775054856467, -0.003108737908628451),
    c64::new(0.0023269504081735682, -0.003137529419846269),
    c64::new(0.0022882728806892143, -0.0031658484306741983),
    c64::new(0.002249250747725959, -0.003193690676373374),
    c64::new(0.002209889885873489, -0.003221051964004005),
    c64::new(0.0021701962227328214, -0.003247928173056817),
    c64::new(0.0021301757360236194, -0.003274315256073586),
    c64::new(0.002089834452683974, -0.003300209239256668),
    c64::new(0.0020491784479627693, -0.003325606223067442),
    c64::new(0.0020082138445047725, -0.003350502382813563),
    c64::new(0.0019669468114285843, -0.003374893969224948),
    c64::new(0.0019253835633975943, -0.0033987773090184037),
    c64::new(0.0018835303596840738, -0.0034221488054508066),
    c64::new(0.001841393503226554, -0.0034450049388607615),
    c64::new(0.001798979339680625, -0.003467342267198648),
    c64::new(0.001756294256463307, -0.0034891574265449817),
    c64::new(0.0017133446817911239, -0.003510447131617007),
    c64::new(0.0016701370837120398, -0.0035312081762634505),
    c64::new(0.0016266779691313958, -0.003551437433947353),
    c64::new(0.0015829738828319914, -0.0035711318582169168),
    c64::new(0.0015390314064884691, -0.003590288483164288),
    c64::new(0.0014948571576761322, -0.003608904423872214),
    c64::new(0.0014504577888743656, -0.003626976876848498),
    c64::new(0.001405839986464798, -0.0036445031204481986),
    c64::new(0.0013610104697243536, -0.003661480515283496),
    c64::new(0.0013159759898133596, -0.003677906504621175),
    c64::new(0.0012707433287588398, -0.0036937786147676612),
    c64::new(0.00122531929843317, -0.0037090944554415495),
    c64::new(0.0011797107395282351, -0.0037238517201335693),
    c64::new(0.0011339245205252433, -0.003738048186453941),
    c64::new(0.0010879675366603635, -0.003751681716467054),
    c64::new(0.001041846708886322, -0.003764750257013437),
    c64::new(0.0009955689828301353, -0.0037772518400189534),
    c64::new(0.0009491413277471249, -0.0037891845827911874),
    c64::new(0.0009025707354713721, -0.003800546688302969),
    c64::new(0.0008558642193627725, -0.0038113364454630022),
    c64::new(0.0008090288132508538, -0.0038215522293735455),
    c64::new(0.0007620715703755012, -0.003831192501575119),
    c64::new(0.0007149995623247697, -0.0038402558102781886),
    c64::new(0.0006678198779699272, -0.0038487407905818016),
    c64::new(0.0006205396223978961, -0.0038566461646791345),
    c64::new(0.0005731659158412568, -0.003863970742049926),
    c64::new(0.0005257058926059618, -0.0038707134196397657),
    c64::new(0.0004781666999969386, -0.003876873182026211),
    c64::new(0.0004305554972417312, -0.0038824491015717034),
    c64::new(0.00038287945441234676, -0.003887440338563269),
    c64::new(0.0003351457513454683, -0.0038918461413389766),
    c64::new(0.000287361576561201, -0.0038956658464011336),
    c64::new(0.00023953412618050256, -0.0038988988785162077),
    c64::new(0.00019167060284147708, -0.0039015447508014547),
    c64::new(0.00014377821461468356, -0.0039036030647982404),
    c64::new(9.586417391762604e-05, -0.003905073510532048),
    c64::new(4.793569642859353e-05, -0.003905955866559158),
    c64::new(2.391888279584674e-19, -0.00390625),
    c64::new(-4.793569642859306e-05, -0.003905955866559158),
    c64::new(-9.586417391762556e-05, -0.003905073510532048),
    c64::new(-0.00014377821461468307, -0.0039036030647982404),
    c64::new(-0.0001916706028414766, -0.0039015447508014547),
    c64::new(-0.00023953412618050207, -0.0038988988785162077),
    c64::new(-0.00028736157656120056, -0.0038956658464011336),
    c64::new(-0.00033514575134546785, -0.0038918461413389766),
    c64::new(-0.0003828794544123463, -0.0038874403385632693),
    c64::new(-0.00043055549724173075, -0.0038824491015717034),
    c64::new(-0.00047816669999693816, -0.003876873182026211),
    c64::new(-0.0005257058926059614, -0.0038707134196397657),
    c64::new(-0.0005731659158412564, -0.003863970742049926),
    c64::new(-0.0006205396223978956, -0.0038566461646791345),
    c64::new(-0.0006678198779699267, -0.0038487407905818016),
    c64::new(-0.0007149995623247693, -0.0038402558102781886),
    c64::new(-0.0007620715703755008, -0.003831192501575119),
    c64::new(-0.0008090288132508533, -0.0038215522293735455),
    c64::new(-0.0008558642193627721, -0.0038113364454630022),
    c64::new(-0.0009025707354713717, -0.0038005466883029694),
    c64::new(-0.0009491413277471245, -0.0037891845827911874),
    c64::new(-0.0009955689828301348, -0.0037772518400189534),
    c64::new(-0.0010418467088863215, -0.003764750257013437),
    c64::new(-0.001087967536660363, -0.0037516817164670543),
    c64::new(-0.001133924520525243, -0.003738048186453941),
    c64::new(-0.0011797107395282347, -0.0037238517201335698),
    c64::new(-0.0012253192984331698, -0.0037090944554415495),
    c64::new(-0.0012707433287588393, -0.0036937786147676617),
    c64::new(-0.0013159759898133591, -0.003677906504621175),
    c64::new(-0.0013610104697243531, -0.003661480515283496),
    c64::new(-0.0014058399864647975, -0.0036445031204481986),
    c64::new(-0.0014504577888743654, -0.0036269768768484984),
    c64::new(-0.0014948571576761317, -0.003608904423872214),
    c64::new(-0.0015390314064884687, -0.003590288483164288),
    c64::new(-0.0015829738828319912, -0.0035711318582169168),
    c64::new(-0.0016266779691313945, -0.0035514374339473535),
    c64::new(-0.0016701370837120385, -0.003531208176263451),
    c64::new(-0.0017133446817911226, -0.0035104471316170075),
    c64::new(-0.0017562942564633074, -0.0034891574265449813),
    c64::new(-0.0017989793396806255, -0.003467342267198648),
    c64::new(-0.0018413935032265535, -0.003445004938860762),
    c64::new(-0.0018835303596840731, -0.003422148805450807),
    c64::new(-0.0019253835633975935, -0.003398777309018404),
    c64::new(-0.001966946811428584, -0.0033748939692249485),
    c64::new(-0.0020082138445047717, -0.003350502382813563),
    c64::new(-0.0020491784479627685, -0.003325606223067442),
    c64::new(-0.0020898344526839724, -0.003300209239256669),
    c64::new(-0.0021301757360236177, -0.0032743152560735867),
    c64::new(-0.00217019622273282, -0.003247928173056818),
    c64::new(-0.002209889885873489, -0.003221051964004005),
    c64::new(-0.0022492507477259584, -0.003193690676373374),
    c64::new(-0.0022882728806892143, -0.0031658484306741983),
    c64::new(-0.0023269504081735674, -0.0031375294198462693),
    c64::new(-0.0023652775054856462, -0.0031087379086284514),
    c64::new(-0.002403248400705573, -0.003079478232916431),
    c64::new(-0.0024408573755561962, -0.0030497547991097445),
    c64::new(-0.0024780987662642393, -0.003019572083448192),
    c64::new(-0.002514966964413247, -0.0029889346313377307),
    c64::new(-0.0025514564177881896, -0.002957847056665956),
    c64::new(-0.0025875616312116085, -0.0029263140411072627),
    c64::new(-0.002623277167371166, -0.0028943403334178086),
    c64::new(-0.0026585976476384884, -0.0028619307487203626),
    c64::new(-0.0026935177528791673, -0.002829090167779168),
    c64::new(-0.0027280322242538, -0.002795823536264917),
    c64::new(-0.002762135864009951, -0.0027621358640099515),
    c64::new(-0.0027958235362649164, -0.0027280322242538005),
    c64::new(-0.0028290901677791675, -0.002693517752879168),
    c64::new(-0.002861930748720362, -0.0026585976476384892),
    c64::new(-0.002894340333417808, -0.0026232771673711662),
    c64::new(-0.002926314041107262, -0.0025875616312116094),
    c64::new(-0.0029578470566659558, -0.00255145641778819),
    c64::new(-0.0029889346313377303, -0.0025149669644132476),
    c64::new(-0.0030195720834481914, -0.00247809876626424),
    c64::new(-0.003049754799109744, -0.0024408573755561967),
    c64::new(-0.0030794782329164306, -0.002403248400705574),
    c64::new(-0.003108737908628451, -0.0023652775054856467),
    c64::new(-0.003137529419846269, -0.0023269504081735682),
    c64::new(-0.003165848430674198, -0.0022882728806892147),
    c64::new(-0.0031936906763733734, -0.002249250747725959),
    c64::new(-0.003221051964004004, -0.00220988988587349),
    c64::new(-0.0032479281730568178, -0.002170196222732821),
    c64::new(-0.0032743152560735862, -0.0021301757360236186),
    c64::new(-0.0033002092392566684, -0.0020898344526839733),
    c64::new(-0.003325606223067442, -0.0020491784479627693),
    c64::new(-0.003350502382813563, -0.0020082138445047725),
    c64::new(-0.003374893969224948, -0.0019669468114285843),
    c64::new(-0.0033987773090184037, -0.0019253835633975943),
    c64::new(-0.0034221488054508066, -0.0018835303596840738),
    c64::new(-0.0034450049388607615, -0.001841393503226554),
    c64::new(-0.0034673422671986477, -0.0017989793396806259),
    c64::new(-0.0034891574265449813, -0.001756294256463308),
    c64::new(-0.0035104471316170075, -0.0017133446817911232),
    c64::new(-0.0035312081762634505, -0.0016701370837120392),
    c64::new(-0.003551437433947353, -0.0016266779691313951),
    c64::new(-0.0035711318582169168, -0.0015829738828319916),
    c64::new(-0.003590288483164288, -0.0015390314064884694),
    c64::new(-0.003608904423872214, -0.0014948571576761322),
    c64::new(-0.003626976876848498, -0.0014504577888743658),
    c64::new(-0.0036445031204481986, -0.0014058399864647982),
    c64::new(-0.0036614805152834955, -0.0013610104697243546),
    c64::new(-0.0036779065046211746, -0.0013159759898133604),
    c64::new(-0.0036937786147676612, -0.0012707433287588406),
    c64::new(-0.0037090944554415495, -0.0012253192984331696),
    c64::new(-0.0037238517201335698, -0.0011797107395282345),
    c64::new(-0.0037380481864539407, -0.0011339245205252435),
    c64::new(-0.003751681716467054, -0.0010879675366603637),
    c64::new(-0.003764750257013437, -0.0010418467088863222),
    c64::new(-0.0037772518400189534, -0.0009955689828301355),
    c64::new(-0.0037891845827911874, -0.0009491413277471251),
    c64::new(-0.003800546688302969, -0.0009025707354713723),
    c64::new(-0.003811336445463002, -0.0008558642193627735),
    c64::new(-0.003821552229373545, -0.0008090288132508548),
    c64::new(-0.003831192501575119, -0.0007620715703755023),
    c64::new(-0.003840255810278189, -0.0007149995623247691),
    c64::new(-0.0038487407905818016, -0.0006678198779699265),
    c64::new(-0.0038566461646791345, -0.0006205396223978963),
    c64::new(-0.003863970742049926, -0.000573165915841257),
    c64::new(-0.0038707134196397657, -0.000525705892605962),
    c64::new(-0.003876873182026211, -0.00047816669999693886),
    c64::new(-0.0038824491015717034, -0.00043055549724173146),
    c64::new(-0.003887440338563269, -0.00038287945441234703),
    c64::new(-0.0038918461413389766, -0.00033514575134546937),
    c64::new(-0.0038956658464011336, -0.00028736157656120213),
    c64::new(-0.0038988988785162077, -0.00023953412618050193),
    c64::new(-0.0039015447508014547, -0.00019167060284147646),
    c64::new(-0.0039036030647982404, -0.00014377821461468294),
    c64::new(-0.003905073510532048, -9.586417391762627e-05),
    c64::new(-0.003905955866559158, -4.793569642859378e-05),
];

static TWIDDLES_512: [c64; 512] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999811752826011, 0.006135884649154475),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9998305817958234, 0.01840672990580482),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9995294175010931, 0.030674803176636626),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9990777277526454, 0.04293825693494082),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9984755805732948, 0.05519524434968994),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.9977230666441916, 0.06744391956366405),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.9968202992911657, 0.07968243797143013),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.9957674144676598, 0.09190895649713272),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9945645707342554, 0.10412163387205459),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.9932119492347945, 0.11631863091190475),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.9917097536690995, 0.12849811079379317),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.9900582102622971, 0.1406582393328492),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9882575677307495, 0.15279718525844344),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.9863080972445987, 0.16491312048996992),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.984210092386929, 0.17700422041214875),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9819638691095552, 0.1890686641498062),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9795697656854405, 0.2011046348420919),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.9770281426577544, 0.21311031991609136),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.9743393827855759, 0.22508391135979283),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.9715038909862518, 0.2370236059943672),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9685220942744173, 0.24892760574572015),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9653944416976894, 0.2607941179152755),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9621214042690416, 0.272621355449949),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.9587034748958716, 0.2844075372112719),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9551411683057707, 0.2961508882436238),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9514350209690083, 0.30784964004153487),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9475855910177411, 0.3195020308160157),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9435934581619604, 0.33110630575987643),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.9394592236021899, 0.3426607173119944),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.9351835099389476, 0.35416352542049034),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9307669610789837, 0.36561299780477385),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9262102421383114, 0.37700741021641826),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9215140393420419, 0.38834504669882625),
    c64::new(0.9191138516900578, 0.3939920400610481),
    c64::new(0.9166790599210427, 0.3996241998456468),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9117060320054299, 0.4108431710579039),
    c64::new(0.9091679830905224, 0.41642956009763715),
    c64::new(0.9065957045149153, 0.4220002707997997),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.901348847046022, 0.43309381885315196),
    c64::new(0.8986744656939538, 0.43861623853852766),
    c64::new(0.8959662497561852, 0.4441221445704292),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.8904487232447579, 0.45508358712634384),
    c64::new(0.8876396204028539, 0.46053871095824),
    c64::new(0.8847970984309378, 0.4659764957679662),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.8790122264286335, 0.4767992300633221),
    c64::new(0.8760700941954066, 0.4821837720791227),
    c64::new(0.8730949784182901, 0.487550160148436),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8670462455156926, 0.49822766697278187),
    c64::new(0.8639728561215867, 0.5035383837257176),
    c64::new(0.8608669386377673, 0.508830142543107),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8545579883654005, 0.5193559901655896),
    c64::new(0.8513551931052652, 0.524589682678469),
    c64::new(0.8481203448032972, 0.5298036246862946),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.8415549774368984, 0.5401714727298929),
    c64::new(0.838224705554838, 0.5453249884220465),
    c64::new(0.83486287498638, 0.5504579729366048),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.8280450452577558, 0.560661576197336),
    c64::new(0.8245893027850253, 0.5657318107836131),
    c64::new(0.8211025149911046, 0.5707807458869673),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.8140363297059484, 0.5808139580957645),
    c64::new(0.8104571982525948, 0.5857978574564389),
    c64::new(0.8068475535437993, 0.5907597018588742),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.799537269107905, 0.600616479383869),
    c64::new(0.7958369046088836, 0.6055110414043255),
    c64::new(0.7921065773002124, 0.6103828062763095),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.7845565971555752, 0.6200572117632891),
    c64::new(0.7807372285720945, 0.6248594881423863),
    c64::new(0.7768884656732324, 0.629638238914927),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.7691033376455797, 0.6391244448637757),
    c64::new(0.765167265622459, 0.6438315428897914),
    c64::new(0.7612023854842618, 0.6485144010221124),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7531867990436125, 0.6578066932970786),
    c64::new(0.7491363945234594, 0.6624157775901718),
    c64::new(0.745057785441466, 0.6669999223036375),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.7368165688773698, 0.6760927035753159),
    c64::new(0.7326542716724128, 0.680600997795453),
    c64::new(0.7284643904482252, 0.6850836677727004),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7200025079613817, 0.6939714608896539),
    c64::new(0.7157308252838186, 0.6983762494089729),
    c64::new(0.7114321957452164, 0.7027547444572253),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.7027547444572254, 0.7114321957452163),
    c64::new(0.6983762494089729, 0.7157308252838186),
    c64::new(0.693971460889654, 0.7200025079613817),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6850836677727005, 0.7284643904482252),
    c64::new(0.6806009977954531, 0.7326542716724127),
    c64::new(0.676092703575316, 0.7368165688773698),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6669999223036376, 0.745057785441466),
    c64::new(0.6624157775901718, 0.7491363945234593),
    c64::new(0.6578066932970787, 0.7531867990436124),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.6485144010221126, 0.7612023854842618),
    c64::new(0.6438315428897915, 0.7651672656224588),
    c64::new(0.6391244448637758, 0.7691033376455796),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6296382389149271, 0.7768884656732324),
    c64::new(0.6248594881423865, 0.7807372285720944),
    c64::new(0.6200572117632892, 0.7845565971555752),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.6103828062763095, 0.7921065773002123),
    c64::new(0.6055110414043255, 0.7958369046088835),
    c64::new(0.600616479383869, 0.799537269107905),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5907597018588743, 0.8068475535437992),
    c64::new(0.5857978574564389, 0.8104571982525948),
    c64::new(0.5808139580957646, 0.8140363297059483),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5707807458869674, 0.8211025149911046),
    c64::new(0.5657318107836132, 0.8245893027850253),
    c64::new(0.560661576197336, 0.8280450452577557),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5504579729366049, 0.83486287498638),
    c64::new(0.5453249884220466, 0.838224705554838),
    c64::new(0.540171472729893, 0.8415549774368983),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.5298036246862948, 0.8481203448032971),
    c64::new(0.524589682678469, 0.8513551931052652),
    c64::new(0.5193559901655896, 0.8545579883654005),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.5088301425431071, 0.8608669386377672),
    c64::new(0.5035383837257176, 0.8639728561215867),
    c64::new(0.4982276669727819, 0.8670462455156926),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.4875501601484361, 0.87309497841829),
    c64::new(0.4821837720791229, 0.8760700941954065),
    c64::new(0.47679923006332225, 0.8790122264286334),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.4659764957679661, 0.8847970984309378),
    c64::new(0.46053871095824, 0.8876396204028539),
    c64::new(0.45508358712634384, 0.8904487232447579),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.44412214457042926, 0.8959662497561851),
    c64::new(0.4386162385385277, 0.8986744656939538),
    c64::new(0.433093818853152, 0.901348847046022),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.4220002707997998, 0.9065957045149153),
    c64::new(0.4164295600976373, 0.9091679830905224),
    c64::new(0.4108431710579039, 0.9117060320054299),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.3996241998456468, 0.9166790599210427),
    c64::new(0.3939920400610481, 0.9191138516900578),
    c64::new(0.3883450466988263, 0.9215140393420419),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3770074102164183, 0.9262102421383114),
    c64::new(0.3713171939518376, 0.9285060804732155),
    c64::new(0.36561299780477396, 0.9307669610789837),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.3541635254204905, 0.9351835099389475),
    c64::new(0.3484186802494345, 0.937339011912575),
    c64::new(0.3426607173119944, 0.9394592236021899),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.33110630575987643, 0.9435934581619604),
    c64::new(0.325310292162263, 0.9456073253805213),
    c64::new(0.31950203081601575, 0.9475855910177411),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.307849640041535, 0.9514350209690083),
    c64::new(0.3020059493192282, 0.9533060403541938),
    c64::new(0.29615088824362396, 0.9551411683057707),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.2844075372112718, 0.9587034748958716),
    c64::new(0.27851968938505306, 0.9604305194155658),
    c64::new(0.272621355449949, 0.9621214042690416),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.26079411791527557, 0.9653944416976894),
    c64::new(0.2548656596045146, 0.9669764710448521),
    c64::new(0.24892760574572026, 0.9685220942744173),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.23702360599436734, 0.9715038909862518),
    c64::new(0.23105810828067125, 0.9729399522055601),
    c64::new(0.22508391135979278, 0.9743393827855759),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.21311031991609136, 0.9770281426577544),
    c64::new(0.20711137619221856, 0.9783173707196277),
    c64::new(0.20110463484209193, 0.9795697656854405),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.18906866414980628, 0.9819638691095552),
    c64::new(0.18303988795514103, 0.9831054874312163),
    c64::new(0.17700422041214886, 0.984210092386929),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.16491312048997006, 0.9863080972445987),
    c64::new(0.1588581433338614, 0.9873014181578584),
    c64::new(0.1527971852584434, 0.9882575677307495),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.14065823933284924, 0.9900582102622971),
    c64::new(0.13458070850712622, 0.99090263542778),
    c64::new(0.12849811079379322, 0.9917097536690995),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.11631863091190486, 0.9932119492347945),
    c64::new(0.11022220729388318, 0.9939069700023561),
    c64::new(0.10412163387205473, 0.9945645707342554),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.0919089564971327, 0.9957674144676598),
    c64::new(0.08579731234443988, 0.996312612182778),
    c64::new(0.07968243797143013, 0.9968202992911657),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.0674439195636641, 0.9977230666441916),
    c64::new(0.061320736302208655, 0.9981181129001492),
    c64::new(0.05519524434969003, 0.9984755805732948),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.04293825693494096, 0.9990777277526454),
    c64::new(0.03680722294135899, 0.9993223845883495),
    c64::new(0.030674803176636584, 0.9995294175010931),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(0.01840672990580482, 0.9998305817958234),
    c64::new(0.012271538285719944, 0.9999247018391445),
    c64::new(0.006135884649154516, 0.9999811752826011),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.006135884649154393, 0.9999811752826011),
    c64::new(-0.012271538285719823, 0.9999247018391445),
    c64::new(-0.018406729905804695, 0.9998305817958234),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.03067480317663646, 0.9995294175010931),
    c64::new(-0.036807222941358866, 0.9993223845883495),
    c64::new(-0.042938256934940834, 0.9990777277526454),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.05519524434968991, 0.9984755805732948),
    c64::new(-0.06132073630220853, 0.9981181129001492),
    c64::new(-0.067443919563664, 0.9977230666441916),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.07968243797143001, 0.9968202992911657),
    c64::new(-0.08579731234443977, 0.996312612182778),
    c64::new(-0.09190895649713259, 0.9957674144676598),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.10412163387205461, 0.9945645707342554),
    c64::new(-0.11022220729388307, 0.9939069700023561),
    c64::new(-0.11631863091190475, 0.9932119492347945),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.1284981107937931, 0.9917097536690995),
    c64::new(-0.1345807085071261, 0.99090263542778),
    c64::new(-0.14065823933284913, 0.9900582102622971),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.1527971852584433, 0.9882575677307495),
    c64::new(-0.15885814333386128, 0.9873014181578584),
    c64::new(-0.16491312048996995, 0.9863080972445987),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.17700422041214875, 0.984210092386929),
    c64::new(-0.18303988795514095, 0.9831054874312163),
    c64::new(-0.18906866414980616, 0.9819638691095552),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.20110463484209182, 0.9795697656854405),
    c64::new(-0.20711137619221845, 0.9783173707196277),
    c64::new(-0.21311031991609125, 0.9770281426577544),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.22508391135979267, 0.9743393827855759),
    c64::new(-0.23105810828067117, 0.9729399522055602),
    c64::new(-0.23702360599436723, 0.9715038909862518),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.24892760574572015, 0.9685220942744173),
    c64::new(-0.2548656596045145, 0.9669764710448521),
    c64::new(-0.26079411791527546, 0.9653944416976894),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.27262135544994887, 0.9621214042690416),
    c64::new(-0.27851968938505295, 0.9604305194155659),
    c64::new(-0.2844075372112717, 0.9587034748958716),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.29615088824362384, 0.9551411683057707),
    c64::new(-0.3020059493192281, 0.9533060403541939),
    c64::new(-0.30784964004153487, 0.9514350209690083),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.31950203081601564, 0.9475855910177412),
    c64::new(-0.32531029216226287, 0.9456073253805214),
    c64::new(-0.3311063057598763, 0.9435934581619604),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.34266071731199427, 0.9394592236021899),
    c64::new(-0.3484186802494344, 0.937339011912575),
    c64::new(-0.3541635254204904, 0.9351835099389476),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.36561299780477385, 0.9307669610789837),
    c64::new(-0.37131719395183754, 0.9285060804732156),
    c64::new(-0.3770074102164182, 0.9262102421383114),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.3883450466988262, 0.921514039342042),
    c64::new(-0.393992040061048, 0.9191138516900578),
    c64::new(-0.39962419984564673, 0.9166790599210427),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.4108431710579038, 0.9117060320054299),
    c64::new(-0.416429560097637, 0.9091679830905225),
    c64::new(-0.4220002707997997, 0.9065957045149153),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.43309381885315196, 0.901348847046022),
    c64::new(-0.4386162385385274, 0.8986744656939539),
    c64::new(-0.44412214457042914, 0.8959662497561852),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.4550835871263437, 0.890448723244758),
    c64::new(-0.4605387109582401, 0.8876396204028539),
    c64::new(-0.465976495767966, 0.8847970984309379),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.476799230063322, 0.8790122264286335),
    c64::new(-0.4821837720791227, 0.8760700941954066),
    c64::new(-0.4875501601484357, 0.8730949784182902),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.49822766697278154, 0.8670462455156929),
    c64::new(-0.5035383837257175, 0.8639728561215868),
    c64::new(-0.5088301425431071, 0.8608669386377672),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5193559901655896, 0.8545579883654005),
    c64::new(-0.5245896826784687, 0.8513551931052652),
    c64::new(-0.5298036246862946, 0.8481203448032972),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.5401714727298929, 0.8415549774368984),
    c64::new(-0.5453249884220461, 0.8382247055548382),
    c64::new(-0.5504579729366047, 0.8348628749863801),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5606615761973359, 0.8280450452577559),
    c64::new(-0.5657318107836132, 0.8245893027850253),
    c64::new(-0.5707807458869671, 0.8211025149911048),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5808139580957643, 0.8140363297059485),
    c64::new(-0.5857978574564389, 0.8104571982525948),
    c64::new(-0.590759701858874, 0.8068475535437994),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.6006164793838686, 0.7995372691079052),
    c64::new(-0.6055110414043254, 0.7958369046088836),
    c64::new(-0.6103828062763095, 0.7921065773002123),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.6200572117632892, 0.7845565971555752),
    c64::new(-0.6248594881423862, 0.7807372285720946),
    c64::new(-0.629638238914927, 0.7768884656732324),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6391244448637757, 0.7691033376455797),
    c64::new(-0.6438315428897913, 0.7651672656224591),
    c64::new(-0.6485144010221124, 0.7612023854842619),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.6578066932970785, 0.7531867990436125),
    c64::new(-0.6624157775901718, 0.7491363945234593),
    c64::new(-0.6669999223036374, 0.7450577854414661),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.6760927035753158, 0.73681656887737),
    c64::new(-0.680600997795453, 0.7326542716724128),
    c64::new(-0.6850836677727002, 0.7284643904482253),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.6939714608896538, 0.7200025079613819),
    c64::new(-0.6983762494089728, 0.7157308252838187),
    c64::new(-0.7027547444572251, 0.7114321957452168),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7114321957452165, 0.7027547444572252),
    c64::new(-0.7157308252838186, 0.6983762494089729),
    c64::new(-0.7200025079613817, 0.6939714608896539),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7284643904482252, 0.6850836677727004),
    c64::new(-0.7326542716724127, 0.6806009977954532),
    c64::new(-0.7368165688773698, 0.6760927035753159),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.745057785441466, 0.6669999223036376),
    c64::new(-0.749136394523459, 0.662415777590172),
    c64::new(-0.7531867990436124, 0.6578066932970787),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.7612023854842618, 0.6485144010221126),
    c64::new(-0.765167265622459, 0.6438315428897914),
    c64::new(-0.7691033376455796, 0.6391244448637758),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7768884656732323, 0.6296382389149272),
    c64::new(-0.7807372285720945, 0.6248594881423863),
    c64::new(-0.7845565971555751, 0.6200572117632894),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.7921065773002122, 0.6103828062763097),
    c64::new(-0.7958369046088835, 0.6055110414043255),
    c64::new(-0.799537269107905, 0.6006164793838689),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8068475535437993, 0.5907597018588742),
    c64::new(-0.8104571982525947, 0.585797857456439),
    c64::new(-0.8140363297059484, 0.5808139580957645),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.8211025149911046, 0.5707807458869673),
    c64::new(-0.8245893027850251, 0.5657318107836135),
    c64::new(-0.8280450452577557, 0.560661576197336),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.83486287498638, 0.5504579729366049),
    c64::new(-0.8382247055548381, 0.5453249884220464),
    c64::new(-0.8415549774368983, 0.540171472729893),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.8481203448032971, 0.5298036246862948),
    c64::new(-0.8513551931052652, 0.524589682678469),
    c64::new(-0.8545579883654004, 0.5193559901655898),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8608669386377672, 0.5088301425431073),
    c64::new(-0.8639728561215867, 0.5035383837257176),
    c64::new(-0.8670462455156926, 0.49822766697278176),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.8730949784182902, 0.4875501601484359),
    c64::new(-0.8760700941954065, 0.4821837720791229),
    c64::new(-0.8790122264286335, 0.4767992300633221),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8847970984309378, 0.4659764957679662),
    c64::new(-0.8876396204028538, 0.4605387109582402),
    c64::new(-0.8904487232447579, 0.4550835871263439),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.8959662497561851, 0.4441221445704293),
    c64::new(-0.8986744656939539, 0.43861623853852755),
    c64::new(-0.9013488470460219, 0.43309381885315207),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9065957045149153, 0.42200027079979985),
    c64::new(-0.9091679830905224, 0.41642956009763715),
    c64::new(-0.9117060320054298, 0.41084317105790413),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.9166790599210426, 0.39962419984564707),
    c64::new(-0.9191138516900578, 0.39399204006104815),
    c64::new(-0.9215140393420418, 0.3883450466988266),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9262102421383114, 0.37700741021641815),
    c64::new(-0.9285060804732155, 0.37131719395183765),
    c64::new(-0.9307669610789837, 0.3656129978047738),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.9351835099389476, 0.35416352542049034),
    c64::new(-0.9373390119125748, 0.3484186802494348),
    c64::new(-0.9394592236021899, 0.34266071731199443),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9435934581619604, 0.3311063057598765),
    c64::new(-0.9456073253805213, 0.3253102921622632),
    c64::new(-0.9475855910177411, 0.3195020308160158),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.9514350209690083, 0.30784964004153503),
    c64::new(-0.9533060403541939, 0.30200594931922803),
    c64::new(-0.9551411683057707, 0.296150888243624),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9587034748958715, 0.28440753721127204),
    c64::new(-0.9604305194155658, 0.2785196893850531),
    c64::new(-0.9621214042690416, 0.27262135544994925),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.9653944416976893, 0.26079411791527585),
    c64::new(-0.9669764710448521, 0.2548656596045147),
    c64::new(-0.9685220942744174, 0.2489276057457201),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9715038909862518, 0.23702360599436717),
    c64::new(-0.9729399522055601, 0.2310581082806713),
    c64::new(-0.9743393827855759, 0.22508391135979283),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.9770281426577544, 0.21311031991609142),
    c64::new(-0.9783173707196275, 0.20711137619221884),
    c64::new(-0.9795697656854405, 0.20110463484209198),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9819638691095552, 0.18906866414980633),
    c64::new(-0.9831054874312164, 0.1830398879551409),
    c64::new(-0.984210092386929, 0.17700422041214892),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.9863080972445986, 0.16491312048997012),
    c64::new(-0.9873014181578584, 0.15885814333386145),
    c64::new(-0.9882575677307495, 0.15279718525844369),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.990058210262297, 0.14065823933284952),
    c64::new(-0.99090263542778, 0.13458070850712628),
    c64::new(-0.9917097536690995, 0.12849811079379306),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.9932119492347945, 0.11631863091190471),
    c64::new(-0.9939069700023561, 0.11022220729388325),
    c64::new(-0.9945645707342554, 0.10412163387205457),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.9957674144676598, 0.09190895649713277),
    c64::new(-0.996312612182778, 0.08579731234444016),
    c64::new(-0.9968202992911657, 0.0796824379714302),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9977230666441916, 0.06744391956366418),
    c64::new(-0.9981181129001492, 0.061320736302208495),
    c64::new(-0.9984755805732948, 0.055195244349690094),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.9990777277526454, 0.04293825693494102),
    c64::new(-0.9993223845883495, 0.03680722294135883),
    c64::new(-0.9995294175010931, 0.030674803176636865),
    c64::new(-0.9996988186962042, 0.024541228522912326),
    c64::new(-0.9998305817958234, 0.0184067299058051),
    c64::new(-0.9999247018391445, 0.012271538285720007),
    c64::new(-0.9999811752826011, 0.006135884649154799),
];

static INVERSE_TWIDDLES_512: [c64; 512] = [
    c64::new(0.001953125, -0.0),
    c64::new(0.0019530882329738303, -1.1984149705379835e-05),
    c64::new(0.001952977933279579, -2.396784821429673e-05),
    c64::new(0.0019527941050699676, -3.595064434727504e-05),
    c64::new(0.001952536755266024, -4.793208695881306e-05),
    c64::new(0.0019522058935568225, -5.991172495436841e-05),
    c64::new(0.0019518015323991202, -7.188910730734147e-05),
    c64::new(0.0019513236870168855, -8.386378307605629e-05),
    c64::new(0.0019507723754007274, -9.583530142073831e-05),
    c64::new(0.0019501476183072164, -0.00010780321162048817),
    c64::new(0.0019494494392581039, -0.00011976706309025113),
    c64::new(0.0019486778645394368, -0.00013172640539778135),
    c64::new(0.0019478329232005668, -0.00014368078828060044),
    c64::new(0.001946914647053058, -0.00015562976166294946),
    c64::new(0.0019459230706694883, -0.00016757287567273417),
    c64::new(0.001944858231382148, -0.00017950968065846235),
    c64::new(0.0019437201692816346, -0.00019143972720617305),
    c64::new(0.0019425089272153426, -0.00020336256615635661),
    c64::new(0.0019412245507858517, -0.00021527774862086535),
    c64::new(0.001939867088349208, -0.00022718482599981397),
    c64::new(0.0019384365910131054, -0.00023908334999846913),
    c64::new(0.00193693311263496, -0.0002509728726441273),
    c64::new(0.0019353567098198828, -0.0002628529463029808),
    c64::new(0.001933707441918549, -0.0002747231236969711),
    c64::new(0.001931985371024963, -0.0002865829579206284),
    c64::new(0.00193019056197412, -0.00029843200245789733),
    c64::new(0.0019283230823395673, -0.00031026981119894814),
    c64::new(0.0019263830024308568, -0.0003220959384569725),
    c64::new(0.0019243703952909008, -0.0003339099389849633),
    c64::new(0.0019222853366932208, -0.000345711367992478),
    c64::new(0.0019201279051390943, -0.00035749978116238467),
    c64::new(0.0019178981818546, -0.0003692747346675902),
    c64::new(0.0019155962507875594, -0.0003810357851877505),
    c64::new(0.001913222198604376, -0.00039278248992596074),
    c64::new(0.0019107761146867728, -0.0004045144066254269),
    c64::new(0.0019082580911284266, -0.00041623109358611594),
    c64::new(0.0019056682227315011, -0.0004279321096813863),
    c64::new(0.0019030066070030778, -0.0004396170143745954),
    c64::new(0.0019002733441514847, -0.00045128536773568576),
    c64::new(0.001897468537082523, -0.00046293673045774843),
    c64::new(0.0018945922913955937, -0.00047457066387356225),
    c64::new(0.0018916447153797212, -0.00048618672997210967),
    c64::new(0.0018886259200094767, -0.0004977844914150675),
    c64::new(0.0018855360189407996, -0.0005093635115532725),
    c64::new(0.0018823751285067184, -0.0005209233544431609),
    c64::new(0.0018791433677129718, -0.0005324635848631816),
    c64::new(0.001875840858233527, -0.0005439837683301818),
    c64::new(0.0018724677244059992, -0.0005554834711157654),
    c64::new(0.0018690240932269704, -0.0005669622602626217),
    c64::new(0.0018655100943472083, -0.0005784197036008277),
    c64::new(0.0018619258600667849, -0.0005898553697641174),
    c64::new(0.0018582715253300944, -0.0006012688282061228),
    c64::new(0.0018545472277207748, -0.000612659649216585),
    c64::new(0.0018507531074565256, -0.0006240274039375306),
    c64::new(0.0018468893073838306, -0.0006353716643794198),
    c64::new(0.0018429559729725789, -0.0006466920034372587),
    c64::new(0.0018389532523105875, -0.0006579879949066798),
    c64::new(0.0018348812960980272, -0.000669259213499989),
    c64::new(0.001830740257641748, -0.0006805052348621769),
    c64::new(0.001826530292849507, -0.0006917256355868952),
    c64::new(0.0018222515602240995, -0.0007029199932323987),
    c64::new(0.00181790422085739, -0.0007140878863374489),
    c64::new(0.0018134884384242492, -0.0007252288944371827),
    c64::new(0.0018090043791763894, -0.0007363425980789419),
    c64::new(0.001804452211936107, -0.000747428578838066),
    c64::new(0.0017998321080899256, -0.000758486419333645),
    c64::new(0.001795144241582144, -0.0007695157032442346),
    c64::new(0.0017903887889082865, -0.0007805160153235289),
    c64::new(0.0017855659291084584, -0.0007914869414159958),
    c64::new(0.0017806758437606052, -0.0008024280684724686),
    c64::new(0.0017757187169736765, -0.0008133389845656976),
    c64::new(0.001770694735380694, -0.0008242192789058588),
    c64::new(0.0017656040881317253, -0.0008350685418560197),
    c64::new(0.0017604469668867618, -0.0008458863649475624),
    c64::new(0.0017552235658085035, -0.0008566723408955618),
    c64::new(0.0017499340815550493, -0.0008674260636141195),
    c64::new(0.0017445787132724909, -0.0008781471282316534),
    c64::new(0.0017391576625874177, -0.0008888351311061403),
    c64::new(0.001733671133599324, -0.0008994896698403125),
    c64::new(0.0017281193328729254, -0.000910110343296809),
    c64::new(0.001722502469430381, -0.0009206967516132766),
    c64::new(0.0017168207547434249, -0.000931248496217426),
    c64::new(0.0017110744027254035, -0.0009417651798420366),
    c64::new(0.0017052636297232228, -0.0009522464065399141),
    c64::new(0.001699388654509202, -0.000962691781698797),
    c64::new(0.0016934496982728372, -0.0009731009120562146),
    c64::new(0.001687446984612474, -0.0009834734057142921),
    c64::new(0.0016813807395268893, -0.0009938088721545058),
    c64::new(0.0016752511914067815, -0.001004106922252386),
    c64::new(0.001669058571026173, -0.0010143671682921673),
    c64::new(0.001662803111533721, -0.0010245892239813847),
    c64::new(0.00165648504844394, -0.0010347727044654192),
    c64::new(0.0016501046196283342, -0.0010449172263419866),
    c64::new(0.0016436620653064423, -0.001055022407675572),
    c64::new(0.001637157628036793, -0.0010650878680118095),
    c64::new(0.0016305915527077735, -0.0010751132283918063),
    c64::new(0.0016239640865284087, -0.0010850981113664105),
    c64::new(0.0016172754790190543, -0.001095042141010422),
    c64::new(0.0016105259820020025, -0.0011049449429367444),
    c64::new(0.0016037158495920013, -0.001114806144310483),
    c64::new(0.001596845338186687, -0.0011246253738629792),
    c64::new(0.0015899147064569305, -0.00113440226190579),
    c64::new(0.0015829242153370992, -0.0011441364403446072),
    c64::new(0.001575874128015233, -0.0011538275426931136),
    c64::new(0.0015687647099231347, -0.001163475204086784),
    c64::new(0.001561596228726377, -0.001173079061296619),
    c64::new(0.0015543689543142257, -0.0011826387527428233),
    c64::new(0.0015470831587894773, -0.001192153918508417),
    c64::new(0.0015397391164582155, -0.0012016242003527868),
    c64::new(0.001532337103819483, -0.001211049241725174),
    c64::new(0.001524877399554872, -0.0012204286877780983),
    c64::new(0.0015173602845180321, -0.0012297621853807168),
    c64::new(0.0015097860417240957, -0.00123904938313212),
    c64::new(0.0015021549563390228, -0.001248289931374562),
    c64::new(0.0014944673156688652, -0.0012574834822066238),
    c64::new(0.0014867234091489488, -0.0012666296894963134),
    c64::new(0.0014789235283329777, -0.0012757282088940952),
    c64::new(0.0014710679668820557, -0.0012847786978458567),
    c64::new(0.0014631570205536316, -0.0012937808156058043),
    c64::new(0.0014551909871903632, -0.001302734223249292),
    c64::new(0.0014471701667089047, -0.0013116385836855827),
    c64::new(0.0014390948610886129, -0.001320493561670539),
    c64::new(0.0014309653743601813, -0.0013292988238192442),
    c64::new(0.0014227820125941898, -0.0013380540386185554),
    c64::new(0.001414545083889584, -0.0013467588764395837),
    c64::new(0.0014062548983620735, -0.0013554130095501053),
    c64::new(0.0013979117681324582, -0.0013640161121269002),
    c64::new(0.0013895160073148758, -0.0013725678602680182),
    c64::new(0.0013810679320049757, -0.0013810679320049755),
    c64::new(0.0013725678602680184, -0.0013895160073148756),
    c64::new(0.0013640161121269002, -0.0013979117681324582),
    c64::new(0.0013554130095501055, -0.0014062548983620735),
    c64::new(0.0013467588764395839, -0.0014145450838895838),
    c64::new(0.0013380540386185556, -0.0014227820125941898),
    c64::new(0.0013292988238192444, -0.001430965374360181),
    c64::new(0.0013204935616705391, -0.0014390948610886129),
    c64::new(0.001311638583685583, -0.0014471701667089043),
    c64::new(0.0013027342232492922, -0.0014551909871903632),
    c64::new(0.0012937808156058043, -0.0014631570205536314),
    c64::new(0.001284778697845857, -0.0014710679668820555),
    c64::new(0.0012757282088940954, -0.0014789235283329775),
    c64::new(0.0012666296894963136, -0.0014867234091489488),
    c64::new(0.001257483482206624, -0.001494467315668865),
    c64::new(0.0012482899313745622, -0.0015021549563390226),
    c64::new(0.0012390493831321203, -0.0015097860417240955),
    c64::new(0.001229762185380717, -0.0015173602845180321),
    c64::new(0.0012204286877780985, -0.0015248773995548718),
    c64::new(0.0012110492417251742, -0.001532337103819483),
    c64::new(0.001201624200352787, -0.0015397391164582153),
    c64::new(0.001192153918508417, -0.001547083158789477),
    c64::new(0.0011826387527428233, -0.0015543689543142255),
    c64::new(0.001173079061296619, -0.001561596228726377),
    c64::new(0.0011634752040867841, -0.0015687647099231344),
    c64::new(0.0011538275426931138, -0.0015758741280152329),
    c64::new(0.0011441364403446072, -0.0015829242153370992),
    c64::new(0.0011344022619057903, -0.0015899147064569303),
    c64::new(0.0011246253738629794, -0.001596845338186687),
    c64::new(0.0011148061443104831, -0.0016037158495920013),
    c64::new(0.0011049449429367446, -0.0016105259820020025),
    c64::new(0.001095042141010422, -0.001617275479019054),
    c64::new(0.0010850981113664107, -0.0016239640865284084),
    c64::new(0.0010751132283918065, -0.0016305915527077735),
    c64::new(0.0010650878680118097, -0.001637157628036793),
    c64::new(0.0010550224076755722, -0.001643662065306442),
    c64::new(0.001044917226341987, -0.001650104619628334),
    c64::new(0.0010347727044654196, -0.0016564850484439397),
    c64::new(0.0010245892239813847, -0.001662803111533721),
    c64::new(0.0010143671682921673, -0.001669058571026173),
    c64::new(0.0010041069222523863, -0.0016752511914067815),
    c64::new(0.000993808872154506, -0.001681380739526889),
    c64::new(0.0009834734057142921, -0.001687446984612474),
    c64::new(0.0009731009120562147, -0.0016934496982728372),
    c64::new(0.0009626917816987972, -0.0016993886545092019),
    c64::new(0.0009522464065399143, -0.0017052636297232226),
    c64::new(0.0009417651798420369, -0.0017110744027254033),
    c64::new(0.0009312484962174263, -0.0017168207547434246),
    c64::new(0.000920696751613277, -0.0017225024694303807),
    c64::new(0.0009101103432968088, -0.0017281193328729254),
    c64::new(0.0008994896698403125, -0.001733671133599324),
    c64::new(0.0008888351311061403, -0.0017391576625874177),
    c64::new(0.0008781471282316535, -0.0017445787132724909),
    c64::new(0.0008674260636141196, -0.001749934081555049),
    c64::new(0.0008566723408955619, -0.0017552235658085035),
    c64::new(0.0008458863649475625, -0.0017604469668867618),
    c64::new(0.0008350685418560199, -0.0017656040881317253),
    c64::new(0.000824219278905859, -0.001770694735380694),
    c64::new(0.0008133389845656979, -0.0017757187169736765),
    c64::new(0.0008024280684724686, -0.0017806758437606052),
    c64::new(0.0007914869414159957, -0.0017855659291084584),
    c64::new(0.0007805160153235289, -0.0017903887889082865),
    c64::new(0.0007695157032442346, -0.001795144241582144),
    c64::new(0.0007584864193336451, -0.0017998321080899256),
    c64::new(0.0007474285788380661, -0.001804452211936107),
    c64::new(0.000736342598078942, -0.0018090043791763894),
    c64::new(0.0007252288944371828, -0.001813488438424249),
    c64::new(0.0007140878863374492, -0.00181790422085739),
    c64::new(0.000702919993232399, -0.0018222515602240993),
    c64::new(0.0006917256355868955, -0.0018265302928495068),
    c64::new(0.0006805052348621768, -0.001830740257641748),
    c64::new(0.000669259213499989, -0.0018348812960980272),
    c64::new(0.0006579879949066798, -0.0018389532523105875),
    c64::new(0.0006466920034372587, -0.0018429559729725789),
    c64::new(0.0006353716643794199, -0.0018468893073838306),
    c64::new(0.0006240274039375308, -0.0018507531074565256),
    c64::new(0.000612659649216585, -0.0018545472277207748),
    c64::new(0.000601268828206123, -0.0018582715253300944),
    c64::new(0.0005898553697641176, -0.0018619258600667847),
    c64::new(0.000578419703600828, -0.0018655100943472083),
    c64::new(0.0005669622602626216, -0.0018690240932269706),
    c64::new(0.0005554834711157653, -0.0018724677244059992),
    c64::new(0.0005439837683301818, -0.001875840858233527),
    c64::new(0.0005324635848631816, -0.0018791433677129718),
    c64::new(0.000520923354443161, -0.0018823751285067184),
    c64::new(0.0005093635115532726, -0.0018855360189407996),
    c64::new(0.0004977844914150676, -0.0018886259200094767),
    c64::new(0.0004861867299721099, -0.0018916447153797212),
    c64::new(0.00047457066387356246, -0.0018945922913955937),
    c64::new(0.0004629367304577487, -0.001897468537082523),
    c64::new(0.00045128536773568603, -0.0019002733441514845),
    c64::new(0.00043961701437459527, -0.0019030066070030778),
    c64::new(0.00042793210968138627, -0.0019056682227315011),
    c64::new(0.00041623109358611594, -0.0019082580911284266),
    c64::new(0.0004045144066254269, -0.0019107761146867728),
    c64::new(0.0003927824899259608, -0.001913222198604376),
    c64::new(0.0003810357851877506, -0.0019155962507875594),
    c64::new(0.0003692747346675904, -0.0019178981818546),
    c64::new(0.00035749978116238483, -0.0019201279051390943),
    c64::new(0.00034571136799247824, -0.0019222853366932208),
    c64::new(0.0003339099389849636, -0.0019243703952909008),
    c64::new(0.0003220959384569728, -0.0019263830024308568),
    c64::new(0.00031026981119894803, -0.0019283230823395673),
    c64::new(0.0002984320024578973, -0.00193019056197412),
    c64::new(0.0002865829579206284, -0.001931985371024963),
    c64::new(0.00027472312369697117, -0.001933707441918549),
    c64::new(0.0002628529463029809, -0.0019353567098198828),
    c64::new(0.0002509728726441274, -0.00193693311263496),
    c64::new(0.0002390833499984693, -0.0019384365910131054),
    c64::new(0.00022718482599981419, -0.001939867088349208),
    c64::new(0.0002152777486208656, -0.0019412245507858517),
    c64::new(0.00020336256615635689, -0.0019425089272153426),
    c64::new(0.00019143972720617338, -0.0019437201692816344),
    c64::new(0.0001795096806584623, -0.001944858231382148),
    c64::new(0.00016757287567273414, -0.0019459230706694883),
    c64::new(0.00015562976166294946, -0.001946914647053058),
    c64::new(0.0001436807882806005, -0.0019478329232005668),
    c64::new(0.00013172640539778146, -0.0019486778645394368),
    c64::new(0.00011976706309025128, -0.0019494494392581039),
    c64::new(0.00010780321162048834, -0.0019501476183072164),
    c64::new(9.583530142073854e-05, -0.0019507723754007274),
    c64::new(8.386378307605656e-05, -0.0019513236870168855),
    c64::new(7.188910730734178e-05, -0.0019518015323991202),
    c64::new(5.991172495436833e-05, -0.0019522058935568225),
    c64::new(4.793208695881302e-05, -0.001952536755266024),
    c64::new(3.595064434727504e-05, -0.0019527941050699676),
    c64::new(2.3967848214296767e-05, -0.001952977933279579),
    c64::new(1.1984149705379914e-05, -0.0019530882329738303),
    c64::new(1.195944139792337e-19, -0.001953125),
    c64::new(-1.1984149705379674e-05, -0.0019530882329738303),
    c64::new(-2.396784821429653e-05, -0.001952977933279579),
    c64::new(-3.5950644347274795e-05, -0.0019527941050699676),
    c64::new(-4.793208695881278e-05, -0.001952536755266024),
    c64::new(-5.9911724954368085e-05, -0.0019522058935568225),
    c64::new(-7.188910730734154e-05, -0.0019518015323991202),
    c64::new(-8.386378307605632e-05, -0.0019513236870168855),
    c64::new(-9.58353014207383e-05, -0.0019507723754007274),
    c64::new(-0.00010780321162048811, -0.0019501476183072164),
    c64::new(-0.00011976706309025103, -0.0019494494392581039),
    c64::new(-0.00013172640539778124, -0.0019486778645394368),
    c64::new(-0.00014368078828060028, -0.0019478329232005668),
    c64::new(-0.00015562976166294925, -0.001946914647053058),
    c64::new(-0.00016757287567273392, -0.0019459230706694883),
    c64::new(-0.00017950968065846208, -0.001944858231382148),
    c64::new(-0.00019143972720617316, -0.0019437201692816346),
    c64::new(-0.00020336256615635667, -0.0019425089272153426),
    c64::new(-0.00021527774862086538, -0.0019412245507858517),
    c64::new(-0.00022718482599981397, -0.001939867088349208),
    c64::new(-0.00023908334999846908, -0.0019384365910131054),
    c64::new(-0.0002509728726441272, -0.00193693311263496),
    c64::new(-0.0002628529463029807, -0.0019353567098198828),
    c64::new(-0.00027472312369697095, -0.001933707441918549),
    c64::new(-0.0002865829579206282, -0.001931985371024963),
    c64::new(-0.00029843200245789706, -0.00193019056197412),
    c64::new(-0.0003102698111989478, -0.0019283230823395673),
    c64::new(-0.00032209593845697256, -0.0019263830024308568),
    c64::new(-0.00033390993898496337, -0.0019243703952909008),
    c64::new(-0.000345711367992478, -0.0019222853366932208),
    c64::new(-0.00035749978116238467, -0.0019201279051390943),
    c64::new(-0.00036927473466759017, -0.0019178981818546),
    c64::new(-0.0003810357851877504, -0.0019155962507875594),
    c64::new(-0.0003927824899259606, -0.001913222198604376),
    c64::new(-0.00040451440662542666, -0.0019107761146867728),
    c64::new(-0.0004162310935861157, -0.0019082580911284266),
    c64::new(-0.00042793210968138605, -0.0019056682227315011),
    c64::new(-0.00043961701437459505, -0.0019030066070030778),
    c64::new(-0.00045128536773568587, -0.0019002733441514847),
    c64::new(-0.0004629367304577485, -0.001897468537082523),
    c64::new(-0.00047457066387356225, -0.0018945922913955937),
    c64::new(-0.00048618672997210967, -0.0018916447153797212),
    c64::new(-0.0004977844914150674, -0.0018886259200094767),
    c64::new(-0.0005093635115532724, -0.0018855360189407996),
    c64::new(-0.0005209233544431608, -0.0018823751285067184),
    c64::new(-0.0005324635848631814, -0.0018791433677129718),
    c64::new(-0.0005439837683301815, -0.0018758408582335271),
    c64::new(-0.0005554834711157651, -0.0018724677244059992),
    c64::new(-0.0005669622602626215, -0.0018690240932269706),
    c64::new(-0.0005784197036008278, -0.0018655100943472083),
    c64::new(-0.0005898553697641174, -0.0018619258600667849),
    c64::new(-0.0006012688282061228, -0.0018582715253300944),
    c64::new(-0.0006126596492165849, -0.0018545472277207748),
    c64::new(-0.0006240274039375305, -0.0018507531074565258),
    c64::new(-0.0006353716643794197, -0.0018468893073838308),
    c64::new(-0.0006466920034372584, -0.0018429559729725789),
    c64::new(-0.0006579879949066796, -0.0018389532523105875),
    c64::new(-0.0006692592134999888, -0.0018348812960980272),
    c64::new(-0.0006805052348621766, -0.001830740257641748),
    c64::new(-0.0006917256355868953, -0.001826530292849507),
    c64::new(-0.0007029199932323988, -0.0018222515602240993),
    c64::new(-0.0007140878863374489, -0.00181790422085739),
    c64::new(-0.0007252288944371827, -0.0018134884384242492),
    c64::new(-0.0007363425980789418, -0.0018090043791763894),
    c64::new(-0.0007474285788380659, -0.001804452211936107),
    c64::new(-0.0007584864193336449, -0.0017998321080899258),
    c64::new(-0.0007695157032442344, -0.001795144241582144),
    c64::new(-0.0007805160153235288, -0.0017903887889082865),
    c64::new(-0.0007914869414159956, -0.0017855659291084584),
    c64::new(-0.0008024280684724684, -0.0017806758437606052),
    c64::new(-0.0008133389845656972, -0.0017757187169736767),
    c64::new(-0.0008242192789058588, -0.001770694735380694),
    c64::new(-0.0008350685418560193, -0.0017656040881317255),
    c64::new(-0.0008458863649475624, -0.0017604469668867618),
    c64::new(-0.0008566723408955613, -0.0017552235658085038),
    c64::new(-0.0008674260636141194, -0.0017499340815550493),
    c64::new(-0.0008781471282316537, -0.0017445787132724907),
    c64::new(-0.0008888351311061401, -0.001739157662587418),
    c64::new(-0.0008994896698403127, -0.001733671133599324),
    c64::new(-0.0009101103432968086, -0.0017281193328729256),
    c64::new(-0.0009206967516132768, -0.001722502469430381),
    c64::new(-0.0009312484962174257, -0.0017168207547434249),
    c64::new(-0.0009417651798420366, -0.0017110744027254035),
    c64::new(-0.0009522464065399135, -0.001705263629723223),
    c64::new(-0.0009626917816987967, -0.001699388654509202),
    c64::new(-0.0009731009120562139, -0.0016934496982728376),
    c64::new(-0.000983473405714292, -0.0016874469846124742),
    c64::new(-0.000993808872154506, -0.001681380739526889),
    c64::new(-0.0010041069222523858, -0.0016752511914067815),
    c64::new(-0.0010143671682921673, -0.001669058571026173),
    c64::new(-0.0010245892239813842, -0.001662803111533721),
    c64::new(-0.0010347727044654192, -0.00165648504844394),
    c64::new(-0.0010449172263419862, -0.0016501046196283344),
    c64::new(-0.001055022407675572, -0.0016436620653064423),
    c64::new(-0.0010650878680118088, -0.0016371576280367933),
    c64::new(-0.001075113228391806, -0.0016305915527077737),
    c64::new(-0.00108509811136641, -0.001623964086528409),
    c64::new(-0.0010950421410104217, -0.0016172754790190545),
    c64::new(-0.0011049449429367446, -0.0016105259820020025),
    c64::new(-0.0011148061443104827, -0.0016037158495920015),
    c64::new(-0.0011246253738629792, -0.001596845338186687),
    c64::new(-0.0011344022619057897, -0.0015899147064569307),
    c64::new(-0.0011441364403446072, -0.0015829242153370992),
    c64::new(-0.0011538275426931134, -0.0015758741280152333),
    c64::new(-0.0011634752040867837, -0.0015687647099231347),
    c64::new(-0.0011730790612966184, -0.0015615962287263774),
    c64::new(-0.0011826387527428231, -0.0015543689543142257),
    c64::new(-0.001192153918508417, -0.001547083158789477),
    c64::new(-0.0012016242003527865, -0.0015397391164582155),
    c64::new(-0.0012110492417251742, -0.001532337103819483),
    c64::new(-0.0012204286877780981, -0.0015248773995548723),
    c64::new(-0.0012297621853807168, -0.0015173602845180321),
    c64::new(-0.0012390493831321197, -0.001509786041724096),
    c64::new(-0.001248289931374562, -0.0015021549563390228),
    c64::new(-0.0012574834822066236, -0.0014944673156688654),
    c64::new(-0.0012666296894963134, -0.001486723409148949),
    c64::new(-0.0012757282088940948, -0.001478923528332978),
    c64::new(-0.0012847786978458565, -0.0014710679668820557),
    c64::new(-0.0012937808156058043, -0.0014631570205536314),
    c64::new(-0.0013027342232492917, -0.0014551909871903634),
    c64::new(-0.001311638583685583, -0.0014471701667089043),
    c64::new(-0.0013204935616705387, -0.0014390948610886133),
    c64::new(-0.0013292988238192442, -0.0014309653743601813),
    c64::new(-0.0013380540386185552, -0.00142278201259419),
    c64::new(-0.0013467588764395837, -0.001414545083889584),
    c64::new(-0.001355413009550105, -0.001406254898362074),
    c64::new(-0.0013640161121269, -0.0013979117681324584),
    c64::new(-0.0013725678602680177, -0.0013895160073148765),
    c64::new(-0.0013810679320049755, -0.0013810679320049757),
    c64::new(-0.001389516007314876, -0.001372567860268018),
    c64::new(-0.0013979117681324582, -0.0013640161121269002),
    c64::new(-0.0014062548983620735, -0.0013554130095501053),
    c64::new(-0.0014145450838895838, -0.001346758876439584),
    c64::new(-0.0014227820125941898, -0.0013380540386185554),
    c64::new(-0.001430965374360181, -0.0013292988238192446),
    c64::new(-0.0014390948610886129, -0.001320493561670539),
    c64::new(-0.001447170166708904, -0.0013116385836855831),
    c64::new(-0.0014551909871903632, -0.0013027342232492922),
    c64::new(-0.001463157020553631, -0.0012937808156058047),
    c64::new(-0.0014710679668820555, -0.001284778697845857),
    c64::new(-0.0014789235283329779, -0.001275728208894095),
    c64::new(-0.0014867234091489488, -0.0012666296894963136),
    c64::new(-0.0014944673156688652, -0.0012574834822066238),
    c64::new(-0.0015021549563390226, -0.0012482899313745622),
    c64::new(-0.0015097860417240957, -0.00123904938313212),
    c64::new(-0.001517360284518032, -0.0012297621853807172),
    c64::new(-0.001524877399554872, -0.0012204286877780983),
    c64::new(-0.0015323371038194827, -0.0012110492417251747),
    c64::new(-0.0015397391164582153, -0.001201624200352787),
    c64::new(-0.0015470831587894769, -0.0011921539185084174),
    c64::new(-0.0015543689543142255, -0.0011826387527428233),
    c64::new(-0.001561596228726377, -0.0011730790612966189),
    c64::new(-0.0015687647099231344, -0.0011634752040867841),
    c64::new(-0.001575874128015233, -0.0011538275426931136),
    c64::new(-0.001582924215337099, -0.0011441364403446074),
    c64::new(-0.0015899147064569305, -0.00113440226190579),
    c64::new(-0.0015968453381866867, -0.0011246253738629794),
    c64::new(-0.0016037158495920013, -0.001114806144310483),
    c64::new(-0.001610525982002002, -0.001104944942936745),
    c64::new(-0.001617275479019054, -0.001095042141010422),
    c64::new(-0.0016239640865284089, -0.0010850981113664105),
    c64::new(-0.0016305915527077735, -0.0010751132283918065),
    c64::new(-0.0016371576280367931, -0.0010650878680118093),
    c64::new(-0.001643662065306442, -0.0010550224076755722),
    c64::new(-0.0016501046196283342, -0.0010449172263419866),
    c64::new(-0.0016564850484439397, -0.0010347727044654196),
    c64::new(-0.001662803111533721, -0.0010245892239813847),
    c64::new(-0.0016690585710261727, -0.0010143671682921675),
    c64::new(-0.0016752511914067815, -0.0010041069222523863),
    c64::new(-0.001681380739526889, -0.0009938088721545065),
    c64::new(-0.001687446984612474, -0.0009834734057142921),
    c64::new(-0.0016934496982728372, -0.0009731009120562144),
    c64::new(-0.0016993886545092019, -0.0009626917816987972),
    c64::new(-0.001705263629723223, -0.0009522464065399138),
    c64::new(-0.0017110744027254033, -0.0009417651798420369),
    c64::new(-0.0017168207547434249, -0.000931248496217426),
    c64::new(-0.0017225024694303807, -0.000920696751613277),
    c64::new(-0.0017281193328729254, -0.000910110343296809),
    c64::new(-0.0017336711335993239, -0.0008994896698403129),
    c64::new(-0.0017391576625874177, -0.0008888351311061404),
    c64::new(-0.0017445787132724907, -0.000878147128231654),
    c64::new(-0.001749934081555049, -0.0008674260636141197),
    c64::new(-0.0017552235658085038, -0.0008566723408955616),
    c64::new(-0.0017604469668867616, -0.0008458863649475626),
    c64::new(-0.0017656040881317253, -0.0008350685418560196),
    c64::new(-0.001770694735380694, -0.0008242192789058591),
    c64::new(-0.0017757187169736765, -0.0008133389845656976),
    c64::new(-0.001780675843760605, -0.000802428068472469),
    c64::new(-0.0017855659291084584, -0.0007914869414159958),
    c64::new(-0.0017903887889082863, -0.0007805160153235294),
    c64::new(-0.001795144241582144, -0.0007695157032442347),
    c64::new(-0.0017998321080899254, -0.0007584864193336457),
    c64::new(-0.001804452211936107, -0.0007474285788380661),
    c64::new(-0.0018090043791763894, -0.0007363425980789417),
    c64::new(-0.001813488438424249, -0.0007252288944371829),
    c64::new(-0.00181790422085739, -0.0007140878863374488),
    c64::new(-0.0018222515602240993, -0.0007029199932323991),
    c64::new(-0.001826530292849507, -0.0006917256355868952),
    c64::new(-0.0018307402576417478, -0.0006805052348621773),
    c64::new(-0.0018348812960980272, -0.0006692592134999891),
    c64::new(-0.0018389532523105873, -0.0006579879949066802),
    c64::new(-0.0018429559729725789, -0.0006466920034372588),
    c64::new(-0.0018468893073838306, -0.0006353716643794203),
    c64::new(-0.0018507531074565256, -0.0006240274039375309),
    c64::new(-0.0018545472277207748, -0.0006126596492165848),
    c64::new(-0.0018582715253300944, -0.0006012688282061231),
    c64::new(-0.0018619258600667849, -0.0005898553697641172),
    c64::new(-0.0018655100943472083, -0.0005784197036008281),
    c64::new(-0.0018690240932269704, -0.0005669622602626217),
    c64::new(-0.001872467724405999, -0.0005554834711157657),
    c64::new(-0.001875840858233527, -0.0005439837683301819),
    c64::new(-0.0018791433677129718, -0.0005324635848631821),
    c64::new(-0.0018823751285067184, -0.0005209233544431611),
    c64::new(-0.0018855360189407994, -0.0005093635115532731),
    c64::new(-0.0018886259200094767, -0.0004977844914150677),
    c64::new(-0.0018916447153797214, -0.00048618672997210956),
    c64::new(-0.0018945922913955937, -0.0004745706638735626),
    c64::new(-0.001897468537082523, -0.0004629367304577484),
    c64::new(-0.0019002733441514845, -0.00045128536773568614),
    c64::new(-0.0019030066070030778, -0.0004396170143745954),
    c64::new(-0.001905668222731501, -0.00042793210968138676),
    c64::new(-0.0019082580911284266, -0.00041623109358611605),
    c64::new(-0.0019107761146867725, -0.0004045144066254274),
    c64::new(-0.001913222198604376, -0.0003927824899259609),
    c64::new(-0.0019155962507875594, -0.00038103578518775114),
    c64::new(-0.0019178981818546, -0.0003692747346675905),
    c64::new(-0.0019201279051390945, -0.00035749978116238456),
    c64::new(-0.0019222853366932208, -0.00034571136799247835),
    c64::new(-0.0019243703952909008, -0.00033390993898496326),
    c64::new(-0.0019263830024308566, -0.0003220959384569729),
    c64::new(-0.0019283230823395673, -0.00031026981119894814),
    c64::new(-0.00193019056197412, -0.0002984320024578978),
    c64::new(-0.001931985371024963, -0.0002865829579206285),
    c64::new(-0.0019337074419185489, -0.0002747231236969717),
    c64::new(-0.0019353567098198828, -0.000262852946302981),
    c64::new(-0.00193693311263496, -0.00025097287264412707),
    c64::new(-0.0019384365910131054, -0.00023908334999846943),
    c64::new(-0.001939867088349208, -0.0002271848259998139),
    c64::new(-0.0019412245507858517, -0.00021527774862086573),
    c64::new(-0.0019425089272153426, -0.0002033625661563566),
    c64::new(-0.0019437201692816344, -0.00019143972720617351),
    c64::new(-0.001944858231382148, -0.00017950968065846243),
    c64::new(-0.0019459230706694883, -0.00016757287567273468),
    c64::new(-0.001946914647053058, -0.0001556297616629496),
    c64::new(-0.0019478329232005668, -0.00014368078828060107),
    c64::new(-0.0019486778645394368, -0.0001317264053977816),
    c64::new(-0.0019494494392581039, -0.00011976706309025097),
    c64::new(-0.0019501476183072164, -0.00010780321162048846),
    c64::new(-0.0019507723754007274, -9.583530142073823e-05),
    c64::new(-0.0019513236870168855, -8.386378307605668e-05),
    c64::new(-0.0019518015323991202, -7.188910730734147e-05),
    c64::new(-0.0019522058935568225, -5.991172495436888e-05),
    c64::new(-0.001952536755266024, -4.793208695881314e-05),
    c64::new(-0.0019527941050699676, -3.595064434727559e-05),
    c64::new(-0.001952977933279579, -2.396784821429689e-05),
    c64::new(-0.0019530882329738303, -1.1984149705380466e-05),
];

static TWIDDLES_1024: [c64; 1024] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999952938095762, 0.003067956762965976),
    c64::new(0.9999811752826011, 0.006135884649154475),
    c64::new(0.9999576445519639, 0.00920375478205982),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9998823474542126, 0.0153392062849881),
    c64::new(0.9998305817958234, 0.01840672990580482),
    c64::new(0.9997694053512153, 0.021474080275469508),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9996188224951786, 0.02760814577896574),
    c64::new(0.9995294175010931, 0.030674803176636626),
    c64::new(0.9994306045554617, 0.03374117185137758),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9992047586183639, 0.03987292758773981),
    c64::new(0.9990777277526454, 0.04293825693494082),
    c64::new(0.9989412931868569, 0.04600318213091462),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9986402181802653, 0.052131704680283324),
    c64::new(0.9984755805732948, 0.05519524434968994),
    c64::new(0.9983015449338929, 0.05825826450043575),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.997925286198596, 0.06438263092985747),
    c64::new(0.9977230666441916, 0.06744391956366405),
    c64::new(0.9975114561403035, 0.07050457338961386),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.997060070339483, 0.07662386139203149),
    c64::new(0.9968202992911657, 0.07968243797143013),
    c64::new(0.9965711457905548, 0.08274026454937569),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.996044700901252, 0.0888535525825246),
    c64::new(0.9957674144676598, 0.09190895649713272),
    c64::new(0.9954807554919269, 0.09496349532963899),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9948793307948056, 0.10106986275482782),
    c64::new(0.9945645707342554, 0.10412163387205459),
    c64::new(0.9942404494531879, 0.10717242495680884),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.9935641355205953, 0.11327095217756435),
    c64::new(0.9932119492347945, 0.11631863091190475),
    c64::new(0.9928504144598651, 0.11936521481099135),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.9920993131421918, 0.12545498341154623),
    c64::new(0.9917097536690995, 0.12849811079379317),
    c64::new(0.9913108598461154, 0.13154002870288312),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.9904850842564571, 0.13762012158648604),
    c64::new(0.9900582102622971, 0.1406582393328492),
    c64::new(0.9896220174632009, 0.14369503315029447),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9887216919603238, 0.1497645346773215),
    c64::new(0.9882575677307495, 0.15279718525844344),
    c64::new(0.9877841416445722, 0.15582839765426523),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.9868094018141855, 0.16188639378011183),
    c64::new(0.9863080972445987, 0.16491312048996992),
    c64::new(0.9857975091675675, 0.16793829497473117),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9847485018019042, 0.17398387338746382),
    c64::new(0.984210092386929, 0.17700422041214875),
    c64::new(0.9836624192117303, 0.18002290140569951),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9825393022874412, 0.18605515166344663),
    c64::new(0.9819638691095552, 0.1890686641498062),
    c64::new(0.9813791933137546, 0.19208039704989244),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9801821359681174, 0.19809841071795356),
    c64::new(0.9795697656854405, 0.2011046348420919),
    c64::new(0.9789481753190622, 0.20410896609281687),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.9776773578245099, 0.2101118368804696),
    c64::new(0.9770281426577544, 0.21311031991609136),
    c64::new(0.9763697313300211, 0.21610679707621952),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.9750253450669941, 0.2220936209732035),
    c64::new(0.9743393827855759, 0.22508391135979283),
    c64::new(0.973644249650812, 0.22807208317088573),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.9722264970789363, 0.23404195858354343),
    c64::new(0.9715038909862518, 0.2370236059943672),
    c64::new(0.9707721407289504, 0.2400030224487415),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9692812353565485, 0.2459550503357946),
    c64::new(0.9685220942744173, 0.24892760574572015),
    c64::new(0.9677538370934755, 0.25189781815421697),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9661900034454125, 0.257831102162159),
    c64::new(0.9653944416976894, 0.2607941179152755),
    c64::new(0.9645897932898128, 0.26375467897483135),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9629532668736839, 0.2696683255729151),
    c64::new(0.9621214042690416, 0.272621355449949),
    c64::new(0.9612804858113206, 0.27557181931095814),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.9595715130819845, 0.28146493792575794),
    c64::new(0.9587034748958716, 0.2844075372112719),
    c64::new(0.9578264130275329, 0.2873474595447295),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9560452513499964, 0.29321916269425863),
    c64::new(0.9551411683057707, 0.2961508882436238),
    c64::new(0.9542280951091057, 0.2990798263080405),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9523750127197659, 0.3049292297354024),
    c64::new(0.9514350209690083, 0.30784964004153487),
    c64::new(0.9504860739494817, 0.3107671527496115),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9485613499157303, 0.31659337555616585),
    c64::new(0.9475855910177411, 0.3195020308160157),
    c64::new(0.9466009130832835, 0.32240767880106985),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9446048372614803, 0.3282098435790925),
    c64::new(0.9435934581619604, 0.33110630575987643),
    c64::new(0.9425731976014469, 0.3339996514420094),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.9405060705932683, 0.33977688440682685),
    c64::new(0.9394592236021899, 0.3426607173119944),
    c64::new(0.9384035340631081, 0.3455413249639891),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.9362656671702783, 0.3512927560855671),
    c64::new(0.9351835099389476, 0.35416352542049034),
    c64::new(0.934092550404259, 0.35703096123343),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9318842655816681, 0.3627557243673972),
    c64::new(0.9307669610789837, 0.36561299780477385),
    c64::new(0.9296408958431812, 0.3684668299533723),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9273625256504011, 0.37416406297145793),
    c64::new(0.9262102421383114, 0.37700741021641826),
    c64::new(0.9250492407826776, 0.37984720892405116),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9227011283338786, 0.38551605384391885),
    c64::new(0.9215140393420419, 0.38834504669882625),
    c64::new(0.9203182767091106, 0.39117038430225387),
    c64::new(0.9191138516900578, 0.3939920400610481),
    c64::new(0.9179007756213905, 0.3968099874167103),
    c64::new(0.9166790599210427, 0.3996241998456468),
    c64::new(0.9154487160882678, 0.40243465085941843),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9129621904283982, 0.4080441628649787),
    c64::new(0.9117060320054299, 0.4108431710579039),
    c64::new(0.9104412922580672, 0.41363831223843456),
    c64::new(0.9091679830905224, 0.41642956009763715),
    c64::new(0.9078861164876663, 0.4192168883632239),
    c64::new(0.9065957045149153, 0.4220002707997997),
    c64::new(0.9052967593181188, 0.4247796812091088),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.9026733182372588, 0.4303264813400826),
    c64::new(0.901348847046022, 0.43309381885315196),
    c64::new(0.9000158920161603, 0.4358570799222555),
    c64::new(0.8986744656939538, 0.43861623853852766),
    c64::new(0.8973245807054183, 0.44137126873171667),
    c64::new(0.8959662497561852, 0.4441221445704292),
    c64::new(0.8945994856313827, 0.44686884016237416),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.8918407093923427, 0.4523495872337709),
    c64::new(0.8904487232447579, 0.45508358712634384),
    c64::new(0.8890483558546646, 0.45781330359887723),
    c64::new(0.8876396204028539, 0.46053871095824),
    c64::new(0.8862225301488806, 0.46325978355186015),
    c64::new(0.8847970984309378, 0.4659764957679662),
    c64::new(0.8833633386657316, 0.46868882203582796),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.8804708890521608, 0.47410021465054997),
    c64::new(0.8790122264286335, 0.4767992300633221),
    c64::new(0.8775452902072614, 0.479493757660153),
    c64::new(0.8760700941954066, 0.4821837720791227),
    c64::new(0.8745866522781762, 0.48486924800079106),
    c64::new(0.8730949784182901, 0.487550160148436),
    c64::new(0.871595086655951, 0.49022648328829116),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8685707059713409, 0.49556526182577254),
    c64::new(0.8670462455156926, 0.49822766697278187),
    c64::new(0.8655136240905691, 0.5008853826112407),
    c64::new(0.8639728561215867, 0.5035383837257176),
    c64::new(0.8624239561110405, 0.5061866453451552),
    c64::new(0.8608669386377673, 0.508830142543107),
    c64::new(0.8593018183570085, 0.5114688504379703),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8561473283751945, 0.5167317990176499),
    c64::new(0.8545579883654005, 0.5193559901655896),
    c64::new(0.8529606049303636, 0.5219752929371544),
    c64::new(0.8513551931052652, 0.524589682678469),
    c64::new(0.8497417680008525, 0.5271991347819014),
    c64::new(0.8481203448032972, 0.5298036246862946),
    c64::new(0.8464909387740521, 0.5324031278771979),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.8432082396418454, 0.5375870762956454),
    c64::new(0.8415549774368984, 0.5401714727298929),
    c64::new(0.8398937941959995, 0.5427507848645159),
    c64::new(0.838224705554838, 0.5453249884220465),
    c64::new(0.836547727223512, 0.5478940591731002),
    c64::new(0.83486287498638, 0.5504579729366048),
    c64::new(0.8331701647019132, 0.5530167055800275),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.829761233794523, 0.5581185312205561),
    c64::new(0.8280450452577558, 0.560661576197336),
    c64::new(0.8263210628456635, 0.5631993440138341),
    c64::new(0.8245893027850253, 0.5657318107836131),
    c64::new(0.8228497813758264, 0.5682589526701315),
    c64::new(0.8211025149911046, 0.5707807458869673),
    c64::new(0.819347520076797, 0.5732971666980422),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.8158144108067338, 0.5783137964116556),
    c64::new(0.8140363297059484, 0.5808139580957645),
    c64::new(0.8122505865852039, 0.5833086529376983),
    c64::new(0.8104571982525948, 0.5857978574564389),
    c64::new(0.808656181588175, 0.5882815482226452),
    c64::new(0.8068475535437993, 0.5907597018588742),
    c64::new(0.8050313311429637, 0.5932322950397998),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.8013761717231402, 0.5981607069963424),
    c64::new(0.799537269107905, 0.600616479383869),
    c64::new(0.7976908409433912, 0.6030665985403482),
    c64::new(0.7958369046088836, 0.6055110414043255),
    c64::new(0.7939754775543372, 0.6079497849677736),
    c64::new(0.7921065773002124, 0.6103828062763095),
    c64::new(0.79023022143731, 0.6128100824294097),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.7864552135990858, 0.6176473079378039),
    c64::new(0.7845565971555752, 0.6200572117632891),
    c64::new(0.7826505961665757, 0.62246127937415),
    c64::new(0.7807372285720945, 0.6248594881423863),
    c64::new(0.778816512381476, 0.6272518154951441),
    c64::new(0.7768884656732324, 0.629638238914927),
    c64::new(0.7749531065948738, 0.6320187359398091),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.7710605242618138, 0.6367618612362842),
    c64::new(0.7691033376455797, 0.6391244448637757),
    c64::new(0.7671389119358204, 0.6414810128085832),
    c64::new(0.765167265622459, 0.6438315428897914),
    c64::new(0.7631884172633814, 0.6461760129833163),
    c64::new(0.7612023854842618, 0.6485144010221124),
    c64::new(0.759209188978388, 0.650846684996381),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7552013768965365, 0.6554928529996153),
    c64::new(0.7531867990436125, 0.6578066932970786),
    c64::new(0.7511651319096864, 0.6601143420674205),
    c64::new(0.7491363945234594, 0.6624157775901718),
    c64::new(0.7471006059801801, 0.6647109782033448),
    c64::new(0.745057785441466, 0.6669999223036375),
    c64::new(0.7430079521351217, 0.669282588346636),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.7388873244606151, 0.673829000378756),
    c64::new(0.7368165688773698, 0.6760927035753159),
    c64::new(0.7347388780959634, 0.6783500431298615),
    c64::new(0.7326542716724128, 0.680600997795453),
    c64::new(0.7305627692278276, 0.6828455463852481),
    c64::new(0.7284643904482252, 0.6850836677727004),
    c64::new(0.726359155084346, 0.687315340891759),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7221281939292153, 0.6917592583641577),
    c64::new(0.7200025079613817, 0.6939714608896539),
    c64::new(0.7178700450557317, 0.696177131491463),
    c64::new(0.7157308252838186, 0.6983762494089729),
    c64::new(0.7135848687807935, 0.7005687939432483),
    c64::new(0.7114321957452164, 0.7027547444572253),
    c64::new(0.7092728264388657, 0.7049340803759049),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.704934080375905, 0.7092728264388656),
    c64::new(0.7027547444572254, 0.7114321957452163),
    c64::new(0.7005687939432484, 0.7135848687807935),
    c64::new(0.6983762494089729, 0.7157308252838186),
    c64::new(0.6961771314914631, 0.7178700450557316),
    c64::new(0.693971460889654, 0.7200025079613817),
    c64::new(0.6917592583641579, 0.7221281939292152),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6873153408917592, 0.7263591550843459),
    c64::new(0.6850836677727005, 0.7284643904482252),
    c64::new(0.6828455463852481, 0.7305627692278276),
    c64::new(0.6806009977954531, 0.7326542716724127),
    c64::new(0.6783500431298616, 0.7347388780959634),
    c64::new(0.676092703575316, 0.7368165688773698),
    c64::new(0.6738290003787561, 0.7388873244606151),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6692825883466361, 0.7430079521351216),
    c64::new(0.6669999223036376, 0.745057785441466),
    c64::new(0.6647109782033449, 0.7471006059801801),
    c64::new(0.6624157775901718, 0.7491363945234593),
    c64::new(0.6601143420674206, 0.7511651319096864),
    c64::new(0.6578066932970787, 0.7531867990436124),
    c64::new(0.6554928529996155, 0.7552013768965364),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.650846684996381, 0.759209188978388),
    c64::new(0.6485144010221126, 0.7612023854842618),
    c64::new(0.6461760129833164, 0.7631884172633813),
    c64::new(0.6438315428897915, 0.7651672656224588),
    c64::new(0.6414810128085832, 0.7671389119358203),
    c64::new(0.6391244448637758, 0.7691033376455796),
    c64::new(0.6367618612362843, 0.7710605242618137),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6320187359398091, 0.7749531065948738),
    c64::new(0.6296382389149271, 0.7768884656732324),
    c64::new(0.6272518154951442, 0.7788165123814759),
    c64::new(0.6248594881423865, 0.7807372285720944),
    c64::new(0.6224612793741501, 0.7826505961665756),
    c64::new(0.6200572117632892, 0.7845565971555752),
    c64::new(0.617647307937804, 0.7864552135990858),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.6128100824294098, 0.79023022143731),
    c64::new(0.6103828062763095, 0.7921065773002123),
    c64::new(0.6079497849677737, 0.7939754775543371),
    c64::new(0.6055110414043255, 0.7958369046088835),
    c64::new(0.6030665985403483, 0.797690840943391),
    c64::new(0.600616479383869, 0.799537269107905),
    c64::new(0.5981607069963424, 0.8013761717231401),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5932322950397999, 0.8050313311429635),
    c64::new(0.5907597018588743, 0.8068475535437992),
    c64::new(0.5882815482226453, 0.808656181588175),
    c64::new(0.5857978574564389, 0.8104571982525948),
    c64::new(0.5833086529376984, 0.8122505865852039),
    c64::new(0.5808139580957646, 0.8140363297059483),
    c64::new(0.5783137964116557, 0.8158144108067338),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5732971666980423, 0.8193475200767969),
    c64::new(0.5707807458869674, 0.8211025149911046),
    c64::new(0.5682589526701316, 0.8228497813758263),
    c64::new(0.5657318107836132, 0.8245893027850253),
    c64::new(0.5631993440138342, 0.8263210628456634),
    c64::new(0.560661576197336, 0.8280450452577557),
    c64::new(0.5581185312205562, 0.829761233794523),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5530167055800276, 0.8331701647019132),
    c64::new(0.5504579729366049, 0.83486287498638),
    c64::new(0.5478940591731003, 0.8365477272235119),
    c64::new(0.5453249884220466, 0.838224705554838),
    c64::new(0.542750784864516, 0.8398937941959994),
    c64::new(0.540171472729893, 0.8415549774368983),
    c64::new(0.5375870762956455, 0.8432082396418454),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.532403127877198, 0.846490938774052),
    c64::new(0.5298036246862948, 0.8481203448032971),
    c64::new(0.5271991347819014, 0.8497417680008524),
    c64::new(0.524589682678469, 0.8513551931052652),
    c64::new(0.5219752929371544, 0.8529606049303636),
    c64::new(0.5193559901655896, 0.8545579883654005),
    c64::new(0.51673179901765, 0.8561473283751944),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.5114688504379705, 0.8593018183570083),
    c64::new(0.5088301425431071, 0.8608669386377672),
    c64::new(0.5061866453451555, 0.8624239561110405),
    c64::new(0.5035383837257176, 0.8639728561215867),
    c64::new(0.5008853826112409, 0.865513624090569),
    c64::new(0.4982276669727819, 0.8670462455156926),
    c64::new(0.49556526182577254, 0.8685707059713409),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.49022648328829116, 0.871595086655951),
    c64::new(0.4875501601484361, 0.87309497841829),
    c64::new(0.4848692480007911, 0.8745866522781761),
    c64::new(0.4821837720791229, 0.8760700941954065),
    c64::new(0.47949375766015306, 0.8775452902072612),
    c64::new(0.47679923006332225, 0.8790122264286334),
    c64::new(0.47410021465055, 0.8804708890521608),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.46868882203582796, 0.8833633386657316),
    c64::new(0.4659764957679661, 0.8847970984309378),
    c64::new(0.46325978355186026, 0.8862225301488806),
    c64::new(0.46053871095824, 0.8876396204028539),
    c64::new(0.4578133035988773, 0.8890483558546645),
    c64::new(0.45508358712634384, 0.8904487232447579),
    c64::new(0.452349587233771, 0.8918407093923427),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.4468688401623743, 0.8945994856313826),
    c64::new(0.44412214457042926, 0.8959662497561851),
    c64::new(0.4413712687317166, 0.8973245807054183),
    c64::new(0.4386162385385277, 0.8986744656939538),
    c64::new(0.4358570799222555, 0.9000158920161603),
    c64::new(0.433093818853152, 0.901348847046022),
    c64::new(0.4303264813400826, 0.9026733182372588),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.4247796812091088, 0.9052967593181188),
    c64::new(0.4220002707997998, 0.9065957045149153),
    c64::new(0.41921688836322396, 0.9078861164876662),
    c64::new(0.4164295600976373, 0.9091679830905224),
    c64::new(0.41363831223843456, 0.9104412922580671),
    c64::new(0.4108431710579039, 0.9117060320054299),
    c64::new(0.40804416286497874, 0.9129621904283981),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.40243465085941854, 0.9154487160882678),
    c64::new(0.3996241998456468, 0.9166790599210427),
    c64::new(0.3968099874167104, 0.9179007756213904),
    c64::new(0.3939920400610481, 0.9191138516900578),
    c64::new(0.391170384302254, 0.9203182767091105),
    c64::new(0.3883450466988263, 0.9215140393420419),
    c64::new(0.385516053843919, 0.9227011283338785),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3798472089240511, 0.9250492407826776),
    c64::new(0.3770074102164183, 0.9262102421383114),
    c64::new(0.37416406297145793, 0.9273625256504011),
    c64::new(0.3713171939518376, 0.9285060804732155),
    c64::new(0.3684668299533723, 0.9296408958431812),
    c64::new(0.36561299780477396, 0.9307669610789837),
    c64::new(0.3627557243673972, 0.9318842655816681),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.35703096123343003, 0.9340925504042589),
    c64::new(0.3541635254204905, 0.9351835099389475),
    c64::new(0.35129275608556715, 0.9362656671702783),
    c64::new(0.3484186802494345, 0.937339011912575),
    c64::new(0.34554132496398915, 0.9384035340631081),
    c64::new(0.3426607173119944, 0.9394592236021899),
    c64::new(0.33977688440682696, 0.9405060705932683),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.3339996514420095, 0.9425731976014469),
    c64::new(0.33110630575987643, 0.9435934581619604),
    c64::new(0.32820984357909266, 0.9446048372614803),
    c64::new(0.325310292162263, 0.9456073253805213),
    c64::new(0.32240767880106996, 0.9466009130832835),
    c64::new(0.31950203081601575, 0.9475855910177411),
    c64::new(0.31659337555616585, 0.9485613499157303),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.3107671527496115, 0.9504860739494817),
    c64::new(0.307849640041535, 0.9514350209690083),
    c64::new(0.3049292297354024, 0.9523750127197659),
    c64::new(0.3020059493192282, 0.9533060403541938),
    c64::new(0.2990798263080405, 0.9542280951091057),
    c64::new(0.29615088824362396, 0.9551411683057707),
    c64::new(0.2932191626942587, 0.9560452513499964),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.28734745954472957, 0.9578264130275329),
    c64::new(0.2844075372112718, 0.9587034748958716),
    c64::new(0.28146493792575805, 0.9595715130819845),
    c64::new(0.27851968938505306, 0.9604305194155658),
    c64::new(0.27557181931095825, 0.9612804858113206),
    c64::new(0.272621355449949, 0.9621214042690416),
    c64::new(0.2696683255729152, 0.9629532668736839),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.2637546789748315, 0.9645897932898126),
    c64::new(0.26079411791527557, 0.9653944416976894),
    c64::new(0.25783110216215893, 0.9661900034454126),
    c64::new(0.2548656596045146, 0.9669764710448521),
    c64::new(0.2518978181542169, 0.9677538370934755),
    c64::new(0.24892760574572026, 0.9685220942744173),
    c64::new(0.2459550503357946, 0.9692812353565485),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.2400030224487415, 0.9707721407289504),
    c64::new(0.23702360599436734, 0.9715038909862518),
    c64::new(0.23404195858354346, 0.9722264970789363),
    c64::new(0.23105810828067125, 0.9729399522055601),
    c64::new(0.2280720831708858, 0.9736442496508119),
    c64::new(0.22508391135979278, 0.9743393827855759),
    c64::new(0.2220936209732036, 0.9750253450669941),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.2161067970762196, 0.9763697313300211),
    c64::new(0.21311031991609136, 0.9770281426577544),
    c64::new(0.21011183688046972, 0.9776773578245099),
    c64::new(0.20711137619221856, 0.9783173707196277),
    c64::new(0.204108966092817, 0.9789481753190622),
    c64::new(0.20110463484209193, 0.9795697656854405),
    c64::new(0.19809841071795373, 0.9801821359681173),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.19208039704989238, 0.9813791933137546),
    c64::new(0.18906866414980628, 0.9819638691095552),
    c64::new(0.1860551516634466, 0.9825393022874412),
    c64::new(0.18303988795514103, 0.9831054874312163),
    c64::new(0.18002290140569951, 0.9836624192117303),
    c64::new(0.17700422041214886, 0.984210092386929),
    c64::new(0.17398387338746385, 0.9847485018019042),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.1679382949747312, 0.9857975091675674),
    c64::new(0.16491312048997006, 0.9863080972445987),
    c64::new(0.16188639378011188, 0.9868094018141854),
    c64::new(0.1588581433338614, 0.9873014181578584),
    c64::new(0.15582839765426532, 0.9877841416445722),
    c64::new(0.1527971852584434, 0.9882575677307495),
    c64::new(0.14976453467732162, 0.9887216919603238),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.14369503315029458, 0.9896220174632008),
    c64::new(0.14065823933284924, 0.9900582102622971),
    c64::new(0.13762012158648618, 0.990485084256457),
    c64::new(0.13458070850712622, 0.99090263542778),
    c64::new(0.13154002870288325, 0.9913108598461154),
    c64::new(0.12849811079379322, 0.9917097536690995),
    c64::new(0.1254549834115462, 0.9920993131421918),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.11936521481099134, 0.9928504144598651),
    c64::new(0.11631863091190486, 0.9932119492347945),
    c64::new(0.11327095217756435, 0.9935641355205953),
    c64::new(0.11022220729388318, 0.9939069700023561),
    c64::new(0.10717242495680887, 0.9942404494531879),
    c64::new(0.10412163387205473, 0.9945645707342554),
    c64::new(0.10106986275482786, 0.9948793307948056),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.09496349532963906, 0.9954807554919269),
    c64::new(0.0919089564971327, 0.9957674144676598),
    c64::new(0.08885355258252468, 0.996044700901252),
    c64::new(0.08579731234443988, 0.996312612182778),
    c64::new(0.0827402645493758, 0.9965711457905548),
    c64::new(0.07968243797143013, 0.9968202992911657),
    c64::new(0.07662386139203162, 0.997060070339483),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.07050457338961401, 0.9975114561403035),
    c64::new(0.0674439195636641, 0.9977230666441916),
    c64::new(0.06438263092985741, 0.997925286198596),
    c64::new(0.061320736302208655, 0.9981181129001492),
    c64::new(0.05825826450043573, 0.9983015449338929),
    c64::new(0.05519524434969003, 0.9984755805732948),
    c64::new(0.05213170468028332, 0.9986402181802653),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.046003182130914644, 0.9989412931868569),
    c64::new(0.04293825693494096, 0.9990777277526454),
    c64::new(0.039872927587739845, 0.9992047586183639),
    c64::new(0.03680722294135899, 0.9993223845883495),
    c64::new(0.03374117185137764, 0.9994306045554617),
    c64::new(0.030674803176636584, 0.9995294175010931),
    c64::new(0.02760814577896582, 0.9996188224951786),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(0.02147408027546961, 0.9997694053512153),
    c64::new(0.01840672990580482, 0.9998305817958234),
    c64::new(0.01533920628498822, 0.9998823474542126),
    c64::new(0.012271538285719944, 0.9999247018391445),
    c64::new(0.00920375478205996, 0.9999576445519639),
    c64::new(0.006135884649154516, 0.9999811752826011),
    c64::new(0.003067956762966138, 0.9999952938095762),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.0030679567629660156, 0.9999952938095762),
    c64::new(-0.006135884649154393, 0.9999811752826011),
    c64::new(-0.009203754782059839, 0.9999576445519639),
    c64::new(-0.012271538285719823, 0.9999247018391445),
    c64::new(-0.015339206284988098, 0.9998823474542126),
    c64::new(-0.018406729905804695, 0.9998305817958234),
    c64::new(-0.021474080275469484, 0.9997694053512153),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.027608145778965695, 0.9996188224951786),
    c64::new(-0.03067480317663646, 0.9995294175010931),
    c64::new(-0.03374117185137752, 0.9994306045554617),
    c64::new(-0.036807222941358866, 0.9993223845883495),
    c64::new(-0.03987292758773972, 0.9992047586183639),
    c64::new(-0.042938256934940834, 0.9990777277526454),
    c64::new(-0.04600318213091452, 0.9989412931868569),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.05213170468028319, 0.9986402181802653),
    c64::new(-0.05519524434968991, 0.9984755805732948),
    c64::new(-0.05825826450043561, 0.9983015449338929),
    c64::new(-0.06132073630220853, 0.9981181129001492),
    c64::new(-0.0643826309298573, 0.997925286198596),
    c64::new(-0.067443919563664, 0.9977230666441916),
    c64::new(-0.0705045733896139, 0.9975114561403035),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.0766238613920315, 0.997060070339483),
    c64::new(-0.07968243797143001, 0.9968202992911657),
    c64::new(-0.08274026454937569, 0.9965711457905548),
    c64::new(-0.08579731234443977, 0.996312612182778),
    c64::new(-0.08885355258252457, 0.996044700901252),
    c64::new(-0.09190895649713259, 0.9957674144676598),
    c64::new(-0.09496349532963895, 0.9954807554919269),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.10106986275482775, 0.9948793307948056),
    c64::new(-0.10412163387205461, 0.9945645707342554),
    c64::new(-0.10717242495680876, 0.9942404494531879),
    c64::new(-0.11022220729388307, 0.9939069700023561),
    c64::new(-0.11327095217756424, 0.9935641355205953),
    c64::new(-0.11631863091190475, 0.9932119492347945),
    c64::new(-0.11936521481099123, 0.9928504144598651),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.1254549834115461, 0.9920993131421918),
    c64::new(-0.1284981107937931, 0.9917097536690995),
    c64::new(-0.13154002870288317, 0.9913108598461154),
    c64::new(-0.1345807085071261, 0.99090263542778),
    c64::new(-0.13762012158648607, 0.990485084256457),
    c64::new(-0.14065823933284913, 0.9900582102622971),
    c64::new(-0.14369503315029447, 0.9896220174632009),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.1497645346773215, 0.9887216919603238),
    c64::new(-0.1527971852584433, 0.9882575677307495),
    c64::new(-0.1558283976542652, 0.9877841416445722),
    c64::new(-0.15885814333386128, 0.9873014181578584),
    c64::new(-0.16188639378011177, 0.9868094018141855),
    c64::new(-0.16491312048996995, 0.9863080972445987),
    c64::new(-0.1679382949747311, 0.9857975091675675),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.17398387338746374, 0.9847485018019042),
    c64::new(-0.17700422041214875, 0.984210092386929),
    c64::new(-0.1800229014056994, 0.9836624192117303),
    c64::new(-0.18303988795514095, 0.9831054874312163),
    c64::new(-0.18605515166344652, 0.9825393022874412),
    c64::new(-0.18906866414980616, 0.9819638691095552),
    c64::new(-0.19208039704989227, 0.9813791933137546),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.19809841071795362, 0.9801821359681174),
    c64::new(-0.20110463484209182, 0.9795697656854405),
    c64::new(-0.2041089660928169, 0.9789481753190622),
    c64::new(-0.20711137619221845, 0.9783173707196277),
    c64::new(-0.2101118368804696, 0.9776773578245099),
    c64::new(-0.21311031991609125, 0.9770281426577544),
    c64::new(-0.2161067970762195, 0.9763697313300211),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.22209362097320348, 0.9750253450669941),
    c64::new(-0.22508391135979267, 0.9743393827855759),
    c64::new(-0.22807208317088568, 0.973644249650812),
    c64::new(-0.23105810828067117, 0.9729399522055602),
    c64::new(-0.23404195858354335, 0.9722264970789363),
    c64::new(-0.23702360599436723, 0.9715038909862518),
    c64::new(-0.2400030224487414, 0.9707721407289504),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.24595505033579448, 0.9692812353565485),
    c64::new(-0.24892760574572015, 0.9685220942744173),
    c64::new(-0.2518978181542168, 0.9677538370934755),
    c64::new(-0.2548656596045145, 0.9669764710448521),
    c64::new(-0.2578311021621588, 0.9661900034454126),
    c64::new(-0.26079411791527546, 0.9653944416976894),
    c64::new(-0.2637546789748314, 0.9645897932898128),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.2696683255729151, 0.9629532668736839),
    c64::new(-0.27262135544994887, 0.9621214042690416),
    c64::new(-0.27557181931095814, 0.9612804858113206),
    c64::new(-0.27851968938505295, 0.9604305194155659),
    c64::new(-0.28146493792575794, 0.9595715130819845),
    c64::new(-0.2844075372112717, 0.9587034748958716),
    c64::new(-0.28734745954472946, 0.9578264130275329),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.2932191626942586, 0.9560452513499965),
    c64::new(-0.29615088824362384, 0.9551411683057707),
    c64::new(-0.29907982630804036, 0.9542280951091057),
    c64::new(-0.3020059493192281, 0.9533060403541939),
    c64::new(-0.3049292297354023, 0.9523750127197659),
    c64::new(-0.30784964004153487, 0.9514350209690083),
    c64::new(-0.31076715274961136, 0.9504860739494818),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.31659337555616573, 0.9485613499157304),
    c64::new(-0.31950203081601564, 0.9475855910177412),
    c64::new(-0.3224076788010699, 0.9466009130832835),
    c64::new(-0.32531029216226287, 0.9456073253805214),
    c64::new(-0.32820984357909255, 0.9446048372614803),
    c64::new(-0.3311063057598763, 0.9435934581619604),
    c64::new(-0.3339996514420094, 0.9425731976014469),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.33977688440682685, 0.9405060705932683),
    c64::new(-0.34266071731199427, 0.9394592236021899),
    c64::new(-0.34554132496398904, 0.9384035340631082),
    c64::new(-0.3484186802494344, 0.937339011912575),
    c64::new(-0.35129275608556704, 0.9362656671702783),
    c64::new(-0.3541635254204904, 0.9351835099389476),
    c64::new(-0.3570309612334299, 0.934092550404259),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.3627557243673971, 0.9318842655816681),
    c64::new(-0.36561299780477385, 0.9307669610789837),
    c64::new(-0.3684668299533722, 0.9296408958431813),
    c64::new(-0.37131719395183754, 0.9285060804732156),
    c64::new(-0.3741640629714579, 0.9273625256504011),
    c64::new(-0.3770074102164182, 0.9262102421383114),
    c64::new(-0.379847208924051, 0.9250492407826777),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.3855160538439189, 0.9227011283338785),
    c64::new(-0.3883450466988262, 0.921514039342042),
    c64::new(-0.39117038430225387, 0.9203182767091106),
    c64::new(-0.393992040061048, 0.9191138516900578),
    c64::new(-0.3968099874167103, 0.9179007756213905),
    c64::new(-0.39962419984564673, 0.9166790599210427),
    c64::new(-0.40243465085941843, 0.9154487160882678),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.40804416286497863, 0.9129621904283982),
    c64::new(-0.4108431710579038, 0.9117060320054299),
    c64::new(-0.4136383122384345, 0.9104412922580672),
    c64::new(-0.416429560097637, 0.9091679830905225),
    c64::new(-0.41921688836322407, 0.9078861164876662),
    c64::new(-0.4220002707997997, 0.9065957045149153),
    c64::new(-0.42477968120910875, 0.9052967593181188),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.4303264813400827, 0.9026733182372588),
    c64::new(-0.43309381885315196, 0.901348847046022),
    c64::new(-0.43585707992225536, 0.9000158920161603),
    c64::new(-0.4386162385385274, 0.8986744656939539),
    c64::new(-0.4413712687317167, 0.8973245807054182),
    c64::new(-0.44412214457042914, 0.8959662497561852),
    c64::new(-0.446868840162374, 0.8945994856313828),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.4523495872337709, 0.8918407093923427),
    c64::new(-0.4550835871263437, 0.890448723244758),
    c64::new(-0.457813303598877, 0.8890483558546647),
    c64::new(-0.4605387109582401, 0.8876396204028539),
    c64::new(-0.46325978355186015, 0.8862225301488806),
    c64::new(-0.465976495767966, 0.8847970984309379),
    c64::new(-0.4686888220358277, 0.8833633386657317),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.47410021465054997, 0.8804708890521609),
    c64::new(-0.476799230063322, 0.8790122264286335),
    c64::new(-0.47949375766015306, 0.8775452902072612),
    c64::new(-0.4821837720791227, 0.8760700941954066),
    c64::new(-0.48486924800079095, 0.8745866522781762),
    c64::new(-0.4875501601484357, 0.8730949784182902),
    c64::new(-0.49022648328829116, 0.871595086655951),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.4955652618257723, 0.868570705971341),
    c64::new(-0.49822766697278154, 0.8670462455156929),
    c64::new(-0.5008853826112407, 0.8655136240905691),
    c64::new(-0.5035383837257175, 0.8639728561215868),
    c64::new(-0.5061866453451551, 0.8624239561110407),
    c64::new(-0.5088301425431071, 0.8608669386377672),
    c64::new(-0.5114688504379703, 0.8593018183570085),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5167317990176497, 0.8561473283751946),
    c64::new(-0.5193559901655896, 0.8545579883654005),
    c64::new(-0.5219752929371543, 0.8529606049303637),
    c64::new(-0.5245896826784687, 0.8513551931052652),
    c64::new(-0.5271991347819011, 0.8497417680008527),
    c64::new(-0.5298036246862946, 0.8481203448032972),
    c64::new(-0.5324031278771978, 0.8464909387740521),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.5375870762956455, 0.8432082396418454),
    c64::new(-0.5401714727298929, 0.8415549774368984),
    c64::new(-0.5427507848645158, 0.8398937941959996),
    c64::new(-0.5453249884220461, 0.8382247055548382),
    c64::new(-0.5478940591731002, 0.836547727223512),
    c64::new(-0.5504579729366047, 0.8348628749863801),
    c64::new(-0.5530167055800274, 0.8331701647019133),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5581185312205561, 0.829761233794523),
    c64::new(-0.5606615761973359, 0.8280450452577559),
    c64::new(-0.5631993440138339, 0.8263210628456636),
    c64::new(-0.5657318107836132, 0.8245893027850253),
    c64::new(-0.5682589526701315, 0.8228497813758264),
    c64::new(-0.5707807458869671, 0.8211025149911048),
    c64::new(-0.573297166698042, 0.8193475200767971),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5783137964116555, 0.8158144108067339),
    c64::new(-0.5808139580957643, 0.8140363297059485),
    c64::new(-0.5833086529376984, 0.8122505865852039),
    c64::new(-0.5857978574564389, 0.8104571982525948),
    c64::new(-0.5882815482226452, 0.8086561815881751),
    c64::new(-0.590759701858874, 0.8068475535437994),
    c64::new(-0.5932322950397999, 0.8050313311429635),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.5981607069963422, 0.8013761717231404),
    c64::new(-0.6006164793838686, 0.7995372691079052),
    c64::new(-0.6030665985403482, 0.7976908409433912),
    c64::new(-0.6055110414043254, 0.7958369046088836),
    c64::new(-0.6079497849677735, 0.7939754775543373),
    c64::new(-0.6103828062763095, 0.7921065773002123),
    c64::new(-0.6128100824294097, 0.79023022143731),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.6176473079378038, 0.7864552135990859),
    c64::new(-0.6200572117632892, 0.7845565971555752),
    c64::new(-0.62246127937415, 0.7826505961665757),
    c64::new(-0.6248594881423862, 0.7807372285720946),
    c64::new(-0.6272518154951439, 0.7788165123814762),
    c64::new(-0.629638238914927, 0.7768884656732324),
    c64::new(-0.632018735939809, 0.7749531065948739),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6367618612362843, 0.7710605242618137),
    c64::new(-0.6391244448637757, 0.7691033376455797),
    c64::new(-0.641481012808583, 0.7671389119358205),
    c64::new(-0.6438315428897913, 0.7651672656224591),
    c64::new(-0.6461760129833164, 0.7631884172633813),
    c64::new(-0.6485144010221124, 0.7612023854842619),
    c64::new(-0.6508466849963808, 0.7592091889783882),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.6554928529996153, 0.7552013768965365),
    c64::new(-0.6578066932970785, 0.7531867990436125),
    c64::new(-0.6601143420674203, 0.7511651319096866),
    c64::new(-0.6624157775901718, 0.7491363945234593),
    c64::new(-0.6647109782033448, 0.7471006059801801),
    c64::new(-0.6669999223036374, 0.7450577854414661),
    c64::new(-0.6692825883466358, 0.7430079521351218),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.673829000378756, 0.7388873244606152),
    c64::new(-0.6760927035753158, 0.73681656887737),
    c64::new(-0.6783500431298612, 0.7347388780959637),
    c64::new(-0.680600997795453, 0.7326542716724128),
    c64::new(-0.682845546385248, 0.7305627692278276),
    c64::new(-0.6850836677727002, 0.7284643904482253),
    c64::new(-0.6873153408917592, 0.7263591550843459),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.6917592583641576, 0.7221281939292155),
    c64::new(-0.6939714608896538, 0.7200025079613819),
    c64::new(-0.696177131491463, 0.7178700450557317),
    c64::new(-0.6983762494089728, 0.7157308252838187),
    c64::new(-0.7005687939432482, 0.7135848687807937),
    c64::new(-0.7027547444572251, 0.7114321957452168),
    c64::new(-0.7049340803759049, 0.7092728264388657),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7092728264388655, 0.7049340803759051),
    c64::new(-0.7114321957452165, 0.7027547444572252),
    c64::new(-0.7135848687807935, 0.7005687939432483),
    c64::new(-0.7157308252838186, 0.6983762494089729),
    c64::new(-0.7178700450557316, 0.6961771314914631),
    c64::new(-0.7200025079613817, 0.6939714608896539),
    c64::new(-0.7221281939292152, 0.6917592583641579),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7263591550843458, 0.6873153408917593),
    c64::new(-0.7284643904482252, 0.6850836677727004),
    c64::new(-0.7305627692278276, 0.6828455463852481),
    c64::new(-0.7326542716724127, 0.6806009977954532),
    c64::new(-0.7347388780959636, 0.6783500431298615),
    c64::new(-0.7368165688773698, 0.6760927035753159),
    c64::new(-0.7388873244606151, 0.6738290003787561),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.7430079521351217, 0.669282588346636),
    c64::new(-0.745057785441466, 0.6669999223036376),
    c64::new(-0.74710060598018, 0.664710978203345),
    c64::new(-0.749136394523459, 0.662415777590172),
    c64::new(-0.7511651319096864, 0.6601143420674205),
    c64::new(-0.7531867990436124, 0.6578066932970787),
    c64::new(-0.7552013768965364, 0.6554928529996156),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.759209188978388, 0.650846684996381),
    c64::new(-0.7612023854842618, 0.6485144010221126),
    c64::new(-0.7631884172633812, 0.6461760129833165),
    c64::new(-0.765167265622459, 0.6438315428897914),
    c64::new(-0.7671389119358203, 0.6414810128085832),
    c64::new(-0.7691033376455796, 0.6391244448637758),
    c64::new(-0.7710605242618136, 0.6367618612362844),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7749531065948738, 0.6320187359398091),
    c64::new(-0.7768884656732323, 0.6296382389149272),
    c64::new(-0.7788165123814761, 0.6272518154951441),
    c64::new(-0.7807372285720945, 0.6248594881423863),
    c64::new(-0.7826505961665756, 0.6224612793741501),
    c64::new(-0.7845565971555751, 0.6200572117632894),
    c64::new(-0.7864552135990858, 0.6176473079378039),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.7902302214373099, 0.6128100824294098),
    c64::new(-0.7921065773002122, 0.6103828062763097),
    c64::new(-0.7939754775543372, 0.6079497849677736),
    c64::new(-0.7958369046088835, 0.6055110414043255),
    c64::new(-0.7976908409433909, 0.6030665985403484),
    c64::new(-0.799537269107905, 0.6006164793838689),
    c64::new(-0.8013761717231402, 0.5981607069963424),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8050313311429634, 0.5932322950398),
    c64::new(-0.8068475535437993, 0.5907597018588742),
    c64::new(-0.808656181588175, 0.5882815482226453),
    c64::new(-0.8104571982525947, 0.585797857456439),
    c64::new(-0.8122505865852037, 0.5833086529376985),
    c64::new(-0.8140363297059484, 0.5808139580957645),
    c64::new(-0.8158144108067338, 0.5783137964116557),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.819347520076797, 0.5732971666980421),
    c64::new(-0.8211025149911046, 0.5707807458869673),
    c64::new(-0.8228497813758262, 0.5682589526701317),
    c64::new(-0.8245893027850251, 0.5657318107836135),
    c64::new(-0.8263210628456635, 0.5631993440138341),
    c64::new(-0.8280450452577557, 0.560661576197336),
    c64::new(-0.8297612337945229, 0.5581185312205563),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.8331701647019132, 0.5530167055800275),
    c64::new(-0.83486287498638, 0.5504579729366049),
    c64::new(-0.8365477272235119, 0.5478940591731003),
    c64::new(-0.8382247055548381, 0.5453249884220464),
    c64::new(-0.8398937941959995, 0.5427507848645159),
    c64::new(-0.8415549774368983, 0.540171472729893),
    c64::new(-0.8432082396418453, 0.5375870762956457),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.846490938774052, 0.532403127877198),
    c64::new(-0.8481203448032971, 0.5298036246862948),
    c64::new(-0.8497417680008525, 0.5271991347819013),
    c64::new(-0.8513551931052652, 0.524589682678469),
    c64::new(-0.8529606049303636, 0.5219752929371544),
    c64::new(-0.8545579883654004, 0.5193559901655898),
    c64::new(-0.8561473283751946, 0.5167317990176498),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8593018183570083, 0.5114688504379705),
    c64::new(-0.8608669386377672, 0.5088301425431073),
    c64::new(-0.8624239561110405, 0.5061866453451552),
    c64::new(-0.8639728561215867, 0.5035383837257176),
    c64::new(-0.865513624090569, 0.5008853826112409),
    c64::new(-0.8670462455156926, 0.49822766697278176),
    c64::new(-0.8685707059713409, 0.49556526182577254),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.871595086655951, 0.4902264832882914),
    c64::new(-0.8730949784182902, 0.4875501601484359),
    c64::new(-0.8745866522781761, 0.4848692480007911),
    c64::new(-0.8760700941954065, 0.4821837720791229),
    c64::new(-0.8775452902072611, 0.4794937576601533),
    c64::new(-0.8790122264286335, 0.4767992300633221),
    c64::new(-0.8804708890521608, 0.4741002146505501),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8833633386657317, 0.46868882203582785),
    c64::new(-0.8847970984309378, 0.4659764957679662),
    c64::new(-0.8862225301488805, 0.4632597835518603),
    c64::new(-0.8876396204028538, 0.4605387109582402),
    c64::new(-0.8890483558546646, 0.4578133035988772),
    c64::new(-0.8904487232447579, 0.4550835871263439),
    c64::new(-0.8918407093923426, 0.452349587233771),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.8945994856313827, 0.44686884016237416),
    c64::new(-0.8959662497561851, 0.4441221445704293),
    c64::new(-0.8973245807054182, 0.4413712687317169),
    c64::new(-0.8986744656939539, 0.43861623853852755),
    c64::new(-0.9000158920161603, 0.4358570799222555),
    c64::new(-0.9013488470460219, 0.43309381885315207),
    c64::new(-0.9026733182372587, 0.4303264813400829),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9052967593181187, 0.42477968120910886),
    c64::new(-0.9065957045149153, 0.42200027079979985),
    c64::new(-0.907886116487666, 0.41921688836322424),
    c64::new(-0.9091679830905224, 0.41642956009763715),
    c64::new(-0.9104412922580671, 0.4136383122384346),
    c64::new(-0.9117060320054298, 0.41084317105790413),
    c64::new(-0.9129621904283982, 0.4080441628649786),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.9154487160882677, 0.40243465085941854),
    c64::new(-0.9166790599210426, 0.39962419984564707),
    c64::new(-0.9179007756213905, 0.39680998741671025),
    c64::new(-0.9191138516900578, 0.39399204006104815),
    c64::new(-0.9203182767091105, 0.39117038430225404),
    c64::new(-0.9215140393420418, 0.3883450466988266),
    c64::new(-0.9227011283338786, 0.38551605384391885),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9250492407826775, 0.3798472089240514),
    c64::new(-0.9262102421383114, 0.37700741021641815),
    c64::new(-0.9273625256504011, 0.374164062971458),
    c64::new(-0.9285060804732155, 0.37131719395183765),
    c64::new(-0.9296408958431812, 0.3684668299533726),
    c64::new(-0.9307669610789837, 0.3656129978047738),
    c64::new(-0.9318842655816681, 0.3627557243673973),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.9340925504042588, 0.3570309612334303),
    c64::new(-0.9351835099389476, 0.35416352542049034),
    c64::new(-0.9362656671702783, 0.3512927560855672),
    c64::new(-0.9373390119125748, 0.3484186802494348),
    c64::new(-0.9384035340631082, 0.345541324963989),
    c64::new(-0.9394592236021899, 0.34266071731199443),
    c64::new(-0.9405060705932683, 0.339776884406827),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9425731976014469, 0.3339996514420093),
    c64::new(-0.9435934581619604, 0.3311063057598765),
    c64::new(-0.9446048372614801, 0.3282098435790927),
    c64::new(-0.9456073253805213, 0.3253102921622632),
    c64::new(-0.9466009130832835, 0.32240767880106985),
    c64::new(-0.9475855910177411, 0.3195020308160158),
    c64::new(-0.9485613499157303, 0.31659337555616607),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.9504860739494817, 0.31076715274961153),
    c64::new(-0.9514350209690083, 0.30784964004153503),
    c64::new(-0.9523750127197658, 0.30492922973540265),
    c64::new(-0.9533060403541939, 0.30200594931922803),
    c64::new(-0.9542280951091056, 0.29907982630804053),
    c64::new(-0.9551411683057707, 0.296150888243624),
    c64::new(-0.9560452513499964, 0.29321916269425896),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9578264130275329, 0.2873474595447296),
    c64::new(-0.9587034748958715, 0.28440753721127204),
    c64::new(-0.9595715130819845, 0.2814649379257579),
    c64::new(-0.9604305194155658, 0.2785196893850531),
    c64::new(-0.9612804858113206, 0.2755718193109583),
    c64::new(-0.9621214042690416, 0.27262135544994925),
    c64::new(-0.9629532668736839, 0.26966832557291504),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.9645897932898126, 0.26375467897483157),
    c64::new(-0.9653944416976893, 0.26079411791527585),
    c64::new(-0.9661900034454125, 0.257831102162159),
    c64::new(-0.9669764710448521, 0.2548656596045147),
    c64::new(-0.9677538370934754, 0.2518978181542172),
    c64::new(-0.9685220942744174, 0.2489276057457201),
    c64::new(-0.9692812353565485, 0.24595505033579465),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9707721407289502, 0.24000302244874175),
    c64::new(-0.9715038909862518, 0.23702360599436717),
    c64::new(-0.9722264970789363, 0.23404195858354349),
    c64::new(-0.9729399522055601, 0.2310581082806713),
    c64::new(-0.9736442496508119, 0.22807208317088606),
    c64::new(-0.9743393827855759, 0.22508391135979283),
    c64::new(-0.9750253450669941, 0.22209362097320365),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.9763697313300211, 0.21610679707621944),
    c64::new(-0.9770281426577544, 0.21311031991609142),
    c64::new(-0.9776773578245099, 0.21011183688046978),
    c64::new(-0.9783173707196275, 0.20711137619221884),
    c64::new(-0.9789481753190622, 0.20410896609281684),
    c64::new(-0.9795697656854405, 0.20110463484209198),
    c64::new(-0.9801821359681173, 0.19809841071795378),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9813791933137546, 0.19208039704989244),
    c64::new(-0.9819638691095552, 0.18906866414980633),
    c64::new(-0.9825393022874412, 0.18605515166344688),
    c64::new(-0.9831054874312164, 0.1830398879551409),
    c64::new(-0.9836624192117303, 0.18002290140569957),
    c64::new(-0.984210092386929, 0.17700422041214892),
    c64::new(-0.9847485018019042, 0.1739838733874641),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.9857975091675674, 0.16793829497473126),
    c64::new(-0.9863080972445986, 0.16491312048997012),
    c64::new(-0.9868094018141855, 0.16188639378011171),
    c64::new(-0.9873014181578584, 0.15885814333386145),
    c64::new(-0.9877841416445722, 0.15582839765426537),
    c64::new(-0.9882575677307495, 0.15279718525844369),
    c64::new(-0.9887216919603238, 0.14976453467732145),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.9896220174632008, 0.14369503315029464),
    c64::new(-0.990058210262297, 0.14065823933284952),
    c64::new(-0.9904850842564571, 0.137620121586486),
    c64::new(-0.99090263542778, 0.13458070850712628),
    c64::new(-0.9913108598461154, 0.1315400287028833),
    c64::new(-0.9917097536690995, 0.12849811079379306),
    c64::new(-0.9920993131421918, 0.12545498341154626),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.9928504144598651, 0.11936521481099163),
    c64::new(-0.9932119492347945, 0.11631863091190471),
    c64::new(-0.9935641355205953, 0.11327095217756442),
    c64::new(-0.9939069700023561, 0.11022220729388325),
    c64::new(-0.9942404494531879, 0.10717242495680916),
    c64::new(-0.9945645707342554, 0.10412163387205457),
    c64::new(-0.9948793307948056, 0.10106986275482793),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.9954807554919269, 0.09496349532963891),
    c64::new(-0.9957674144676598, 0.09190895649713277),
    c64::new(-0.996044700901252, 0.08885355258252475),
    c64::new(-0.996312612182778, 0.08579731234444016),
    c64::new(-0.9965711457905548, 0.08274026454937565),
    c64::new(-0.9968202992911657, 0.0796824379714302),
    c64::new(-0.997060070339483, 0.07662386139203169),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9975114561403035, 0.07050457338961386),
    c64::new(-0.9977230666441916, 0.06744391956366418),
    c64::new(-0.997925286198596, 0.0643826309298577),
    c64::new(-0.9981181129001492, 0.061320736302208495),
    c64::new(-0.9983015449338929, 0.058258264500435794),
    c64::new(-0.9984755805732948, 0.055195244349690094),
    c64::new(-0.9986402181802652, 0.0521317046802836),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.9989412931868569, 0.046003182130914706),
    c64::new(-0.9990777277526454, 0.04293825693494102),
    c64::new(-0.9992047586183639, 0.03987292758774013),
    c64::new(-0.9993223845883495, 0.03680722294135883),
    c64::new(-0.9994306045554617, 0.033741171851377705),
    c64::new(-0.9995294175010931, 0.030674803176636865),
    c64::new(-0.9996188224951786, 0.027608145778965656),
    c64::new(-0.9996988186962042, 0.024541228522912326),
    c64::new(-0.9997694053512153, 0.021474080275469667),
    c64::new(-0.9998305817958234, 0.0184067299058051),
    c64::new(-0.9998823474542126, 0.01533920628498806),
    c64::new(-0.9999247018391445, 0.012271538285720007),
    c64::new(-0.9999576445519639, 0.009203754782060022),
    c64::new(-0.9999811752826011, 0.006135884649154799),
    c64::new(-0.9999952938095762, 0.003067956762965977),
];

static INVERSE_TWIDDLES_1024: [c64; 1024] = [
    c64::new(0.0009765625, -0.0),
    c64::new(0.0009765579041109142, -2.996051526333961e-06),
    c64::new(0.0009765441164869151, -5.992074852689917e-06),
    c64::new(0.0009765211372577772, -8.988041779355292e-06),
    c64::new(0.0009764889666397896, -1.1983924107148365e-05),
    c64::new(0.0009764476049357545, -1.4979693637683692e-05),
    c64::new(0.0009763970525349838, -1.797532217363752e-05),
    c64::new(0.0009763373099132962, -2.097078151901319e-05),
    c64::new(0.000976268377633012, -2.396604347940653e-05),
    c64::new(0.0009761902563429479, -2.696107986227123e-05),
    c64::new(0.0009761029467784113, -2.9955862477184205e-05),
    c64::new(0.0009760064497611931, -3.295036313611092e-05),
    c64::new(0.0009759007661995601, -3.5944553653670734e-05),
    c64::new(0.000975785897088246, -3.893840584740216e-05),
    c64::new(0.0009756618435084427, -4.1931891538028145e-05),
    c64::new(0.0009755286066277899, -4.492498254972131e-05),
    c64::new(0.0009753861877003637, -4.7917650710369155e-05),
    c64::new(0.0009752345880666653, -5.090986785183918e-05),
    c64::new(0.0009750738091536082, -5.390160581024408e-05),
    c64::new(0.0009749038524745048, -5.689283642620679e-05),
    c64::new(0.0009747247196290519, -5.9883531545125565e-05),
    c64::new(0.0009745364123033164, -6.287366301743893e-05),
    c64::new(0.0009743389322697184, -6.586320269889067e-05),
    c64::new(0.0009741322813870151, -6.885212245079478e-05),
    c64::new(0.0009739164616002834, -7.184039414030022e-05),
    c64::new(0.0009736914749409013, -7.482798964065575e-05),
    c64::new(0.000973457323526529, -7.781488083147473e-05),
    c64::new(0.0009732140095610887, -8.08010395989997e-05),
    c64::new(0.0009729615353347441, -8.378643783636708e-05),
    c64::new(0.0009726999032238789, -8.677104744387168e-05),
    c64::new(0.000972429115691074, -8.975484032923118e-05),
    c64::new(0.0009721491752850849, -9.273778840785058e-05),
    c64::new(0.0009718600846408173, -9.571986360308653e-05),
    c64::new(0.0009715618464793024, -9.870103784651154e-05),
    c64::new(0.0009712544636076713, -0.00010168128307817831),
    c64::new(0.0009709379389191288, -0.00010466057124688364),
    c64::new(0.0009706122753929258, -0.00010763887431043268),
    c64::new(0.0009702774760943314, -0.00011061616423590268),
    c64::new(0.000969933544174604, -0.00011359241299990698),
    c64::new(0.000969580482870962, -0.00011656759258885874),
    c64::new(0.0009692182955065527, -0.00011954167499923457),
    c64::new(0.0009688469854904217, -0.00012251463223783812),
    c64::new(0.00096846655631748, -0.00012548643632206364),
    c64::new(0.0009680770115684721, -0.0001284570592801593),
    c64::new(0.0009676783549099414, -0.0001314264731514904),
    c64::new(0.0009672705900941964, -0.00013439464998680277),
    c64::new(0.0009668537209592745, -0.00013736156184848556),
    c64::new(0.0009664277514289071, -0.00014032718081083444),
    c64::new(0.0009659926855124815, -0.0001432914789603142),
    c64::new(0.0009655485273050037, -0.00014625442839582179),
    c64::new(0.00096509528098706, -0.00014921600122894867),
    c64::new(0.0009646329508247775, -0.0001521761695842434),
    c64::new(0.0009641615411697836, -0.00015513490559947407),
    c64::new(0.0009636810564591656, -0.00015809218142589045),
    c64::new(0.0009631915012154284, -0.00016104796922848625),
    c64::new(0.0009626928800464526, -0.0001640022411862609),
    c64::new(0.0009621851976454504, -0.00016695496949248166),
    c64::new(0.0009616684587909221, -0.00016990612635494514),
    c64::new(0.0009611426683466104, -0.000172855683996239),
    c64::new(0.0009606078312614553, -0.00017580361465400343),
    c64::new(0.0009600639525695472, -0.00017874989058119233),
    c64::new(0.0009595110373900793, -0.0001816944840463346),
    c64::new(0.0009589490909273, -0.0001846373673337951),
    c64::new(0.0009583781184704634, -0.00018757851274403558),
    c64::new(0.0009577981253937797, -0.00019051789259387524),
    c64::new(0.0009572091171563647, -0.00019345547921675152),
    c64::new(0.000956611099302188, -0.00019639124496298037),
    c64::new(0.0009560040774600217, -0.00019932516220001647),
    c64::new(0.0009553880573433864, -0.00020225720331271344),
    c64::new(0.000954763044750498, -0.0002051873407035836),
    c64::new(0.0009541290455642133, -0.00020811554679305797),
    c64::new(0.0009534860657519738, -0.00021104179401974563),
    c64::new(0.0009528341113657506, -0.00021396605484069316),
    c64::new(0.0009521731885419864, -0.00021688830173164405),
    c64::new(0.0009515033035015389, -0.0002198085071872977),
    c64::new(0.0009508244625496211, -0.0002227266437215681),
    c64::new(0.0009501366720757424, -0.00022564268386784288),
    c64::new(0.0009494399385536487, -0.00022855660017924163),
    c64::new(0.0009487342685412615, -0.00023146836522887422),
    c64::new(0.0009480196686806156, -0.00023437795161009912),
    c64::new(0.0009472961456977968, -0.00023728533193678112),
    c64::new(0.0009465637064028794, -0.0002401904788435494),
    c64::new(0.0009458223576898606, -0.00024309336498605483),
    c64::new(0.0009450721065365972, -0.0002459939630412275),
    c64::new(0.0009443129600047384, -0.00024889224570753376),
    c64::new(0.0009435449252396606, -0.0002517881857052334),
    c64::new(0.0009427680094703998, -0.00025468175577663624),
    c64::new(0.0009419822200095828, -0.00025757292868635874),
    c64::new(0.0009411875642533592, -0.00026046167722158043),
    c64::new(0.0009403840496813319, -0.0002633479741922999),
    c64::new(0.0009395716838564859, -0.0002662317924315908),
    c64::new(0.0009387504744251178, -0.00026911310479585756),
    c64::new(0.0009379204291167635, -0.0002719918841650909),
    c64::new(0.0009370815557441255, -0.000274868103443123),
    c64::new(0.0009362338622029996, -0.0002777417355578827),
    c64::new(0.0009353773564722001, -0.0002806127534616499),
    c64::new(0.0009345120466134852, -0.00028348113013131087),
    c64::new(0.0009336379407714809, -0.00028634683856861194),
    c64::new(0.0009327550471736042, -0.00028920985180041386),
    c64::new(0.000931863374129986, -0.0002920701428789458),
    c64::new(0.0009309629300333924, -0.0002949276848820587),
    c64::new(0.0009300537233591464, -0.0002977824509134789),
    c64::new(0.0009291357626650472, -0.0003006344141030614),
    c64::new(0.0009282090565912907, -0.00030348354760704246),
    c64::new(0.0009272736138603874, -0.0003063298246082925),
    c64::new(0.0009263294432770803, -0.0003091732183165682),
    c64::new(0.0009253765537282628, -0.0003120137019687653),
    c64::new(0.0009244149541828941, -0.0003148512488291698),
    c64::new(0.0009234446536919153, -0.0003176858321897099),
    c64::new(0.0009224656613881643, -0.0003205174253702075),
    c64::new(0.0009214779864862894, -0.0003233460017186293),
    c64::new(0.000920481638282663, -0.0003261715346113373),
    c64::new(0.0009194766261552938, -0.0003289939974533399),
    c64::new(0.0009184629595637386, -0.00033181336367854185),
    c64::new(0.0009174406480490136, -0.0003346296067499945),
    c64::new(0.000916409701233504, -0.0003374427001601456),
    c64::new(0.000915370128820874, -0.00034025261743108844),
    c64::new(0.0009143219405959749, -0.0003430593321148116),
    c64::new(0.0009132651464247535, -0.0003458628177934476),
    c64::new(0.0009121997562541592, -0.00034866304807952146),
    c64::new(0.0009111257801120498, -0.0003514599966161993),
    c64::new(0.0009100432281070978, -0.00035425363707753635),
    c64::new(0.000908952110428695, -0.00035704394316872447),
    c64::new(0.0009078524373468567, -0.00035983088862634016),
    c64::new(0.0009067442192121246, -0.00036261444721859135),
    c64::new(0.0009056274664554698, -0.0003653945927455644),
    c64::new(0.0009045021895881947, -0.00036817129903947096),
    c64::new(0.0009033683992018336, -0.0003709445399648937),
    c64::new(0.0009022261059680535, -0.000373714289419033),
    c64::new(0.0009010753206385533, -0.000376480521331952),
    c64::new(0.0008999160540449628, -0.0003792432096668225),
    c64::new(0.0008987483170987408, -0.0003820023284201698),
    c64::new(0.000897572120791072, -0.0003847578516221173),
    c64::new(0.0008963874761927642, -0.00038750975333663116),
    c64::new(0.0008951943944541433, -0.00039025800766176444),
    c64::new(0.0008939928868049491, -0.0003930025887299008),
    c64::new(0.0008927829645542292, -0.0003957434707079979),
    c64::new(0.0008915646390902326, -0.00039848062779783075),
    c64::new(0.0008903379218803026, -0.0004012140342362343),
    c64::new(0.0008891028244707688, -0.00040394366429534625),
    c64::new(0.0008878593584868383, -0.0004066694922828488),
    c64::new(0.0008866075356324866, -0.00040939149254221084),
    c64::new(0.000885347367690347, -0.0004121096394529294),
    c64::new(0.0008840788665216004, -0.0004148239074307703),
    c64::new(0.0008828020440658626, -0.00041753427092800985),
    c64::new(0.0008815169123410731, -0.0004202407044336744),
    c64::new(0.0008802234834433809, -0.0004229431824737812),
    c64::new(0.0008789217695470315, -0.0004256416796115776),
    c64::new(0.0008776117829042518, -0.0004283361704477809),
    c64::new(0.0008762935358451351, -0.00043102662962081706),
    c64::new(0.0008749670407775246, -0.00043371303180705977),
    c64::new(0.0008736323101868972, -0.0004363953517210685),
    c64::new(0.0008722893566362454, -0.0004390735641158267),
    c64::new(0.0008709381927659597, -0.0004417476437829794),
    c64::new(0.0008695788312937089, -0.00044441756555307015),
    c64::new(0.0008682112850143209, -0.00044708330429577855),
    c64::new(0.000866835566799662, -0.00044974483492015626),
    c64::new(0.0008654516895985162, -0.0004524021323748634),
    c64::new(0.0008640596664364627, -0.0004550551716484045),
    c64::new(0.0008626595104157535, -0.00045770392776936324),
    c64::new(0.0008612512347151905, -0.0004603483758066383),
    c64::new(0.0008598348525900007, -0.0004629884908696777),
    c64::new(0.0008584103773717124, -0.000465624248108713),
    c64::new(0.0008569778224680287, -0.00046825562271499317),
    c64::new(0.0008555372013627018, -0.0004708825899210183),
    c64::new(0.0008540885276154065, -0.0004735051250007725),
    c64::new(0.0008526318148616114, -0.00047612320326995703),
    c64::new(0.0008511670768124521, -0.00047873680008622184),
    c64::new(0.000849694327254601, -0.0004813458908493985),
    c64::new(0.0008482135800501376, -0.000483950451001731),
    c64::new(0.0008467248491364186, -0.0004865504560281073),
    c64::new(0.0008452281485259464, -0.0004891458814562898),
    c64::new(0.000843723492306237, -0.0004917367028571461),
    c64::new(0.000842210894639688, -0.0004943228958448782),
    c64::new(0.0008406903697634446, -0.0004969044360772529),
    c64::new(0.0008391619319892661, -0.0004994812992558304),
    c64::new(0.0008376255957033907, -0.000502053461126193),
    c64::new(0.0008360813753664009, -0.0005046208974781737),
    c64::new(0.0008345292855130865, -0.0005071835841460836),
    c64::new(0.0008329693407523082, -0.0005097414970089398),
    c64::new(0.0008314015557668605, -0.0005122946119906923),
    c64::new(0.0008298259453133326, -0.0005148429050604506),
    c64::new(0.00082824252422197, -0.0005173863522327096),
    c64::new(0.0008266513073965353, -0.0005199249295675761),
    c64::new(0.0008250523098141671, -0.0005224586131709933),
    c64::new(0.0008234455465252397, -0.0005249873791949662),
    c64::new(0.0008218310326532211, -0.000527511203837786),
    c64::new(0.0008202087833945308, -0.0005300300633442538),
    c64::new(0.0008185788140183965, -0.0005325439340059047),
    c64::new(0.0008169411398667109, -0.0005350527921612307),
    c64::new(0.0008152957763538867, -0.0005375566141959031),
    c64::new(0.0008136427389667121, -0.0005400553765429956),
    c64::new(0.0008119820432642043, -0.0005425490556832053),
    c64::new(0.0008103137048774639, -0.0005450376281450743),
    c64::new(0.0008086377395095271, -0.000547521070505211),
    c64::new(0.0008069541629352183, -0.0005499993593885099),
    c64::new(0.0008052629910010013, -0.0005524724714683722),
    c64::new(0.0008035642396248305, -0.0005549403834669253),
    c64::new(0.0008018579247960006, -0.0005574030721552415),
    c64::new(0.0008001440625749971, -0.0005598605143535568),
    c64::new(0.0007984226690933435, -0.0005623126869314896),
    c64::new(0.000796693760553451, -0.0005647595668082574),
    c64::new(0.0007949573532284652, -0.000567201130952895),
    c64::new(0.0007932134634621132, -0.000569637356384471),
    c64::new(0.0007914621076685496, -0.0005720682201723036),
    c64::new(0.0007897033023322021, -0.000574493699436177),
    c64::new(0.0007879370640076165, -0.0005769137713465568),
    c64::new(0.0007861634093193004, -0.0005793284131248045),
    c64::new(0.0007843823549615673, -0.000581737602043392),
    c64::new(0.0007825939176983791, -0.0005841413154261156),
    c64::new(0.0007807981143631885, -0.0005865395306483095),
    c64::new(0.0007789949618587804, -0.0005889322251370588),
    c64::new(0.0007771844771571129, -0.0005913193763714117),
    c64::new(0.0007753666772991574, -0.0005937009618825914),
    c64::new(0.0007735415793947387, -0.0005960769592542085),
    c64::new(0.0007717092006223731, -0.0005984473461224704),
    c64::new(0.0007698695582291078, -0.0006008121001763934),
    c64::new(0.0007680226695303572, -0.0006031711991580116),
    c64::new(0.0007661685519097414, -0.000605524620862587),
    c64::new(0.0007643072228189216, -0.0006078723431388183),
    c64::new(0.000762438699777436, -0.0006102143438890492),
    c64::new(0.0007605630003725351, -0.0006125506010694766),
    c64::new(0.0007586801422590161, -0.0006148810926903584),
    c64::new(0.0007567901431590565, -0.0006172057968162198),
    c64::new(0.0007548930208620478, -0.00061952469156606),
    c64::new(0.0007529887932244276, -0.0006218377551135588),
    c64::new(0.0007510774781695114, -0.000624144965687281),
    c64::new(0.0007491590936873246, -0.000626446301570882),
    c64::new(0.0007472336578344326, -0.0006287417411033119),
    c64::new(0.0007453011887337709, -0.0006310312626790198),
    c64::new(0.0007433617045744744, -0.0006333148447481567),
    c64::new(0.000741415223611707, -0.0006355924658167783),
    c64::new(0.0007394617641664888, -0.0006378641044470476),
    c64::new(0.000737501344625524, -0.0006401297392574369),
    c64::new(0.0007355339834410279, -0.0006423893489229284),
    c64::new(0.0007335596991305531, -0.0006446429121752153),
    c64::new(0.0007315785102768158, -0.0006468904078029021),
    c64::new(0.0007295904355275197, -0.0006491318146517039),
    c64::new(0.0007275954935951816, -0.000651367111624646),
    c64::new(0.0007255937032569548, -0.0006535962776822617),
    c64::new(0.0007235850833544524, -0.0006558192918427913),
    c64::new(0.0007215696527935694, -0.0006580361331823789),
    c64::new(0.0007195474305443064, -0.0006602467808352695),
    c64::new(0.0007175184356405892, -0.0006624512139940053),
    c64::new(0.0007154826871800906, -0.0006646494119096221),
    c64::new(0.0007134402043240504, -0.0006668413538918438),
    c64::new(0.0007113910062970949, -0.0006690270193092777),
    c64::new(0.0007093351123870566, -0.0006712063875896084),
    c64::new(0.000707272541944792, -0.0006733794382197918),
    c64::new(0.0007052033143839994, -0.0006755461507462478),
    c64::new(0.0007031274491810368, -0.0006777065047750526),
    c64::new(0.000701044965874738, -0.0006798604799721318),
    c64::new(0.0006989558840662291, -0.0006820080560634501),
    c64::new(0.0006968602234187437, -0.0006841492128352035),
    c64::new(0.0006947580036574379, -0.0006862839301340091),
    c64::new(0.0006926492445692048, -0.0006884121878670946),
    c64::new(0.0006905339660024879, -0.0006905339660024878),
    c64::new(0.0006884121878670947, -0.0006926492445692047),
    c64::new(0.0006862839301340092, -0.0006947580036574378),
    c64::new(0.0006841492128352036, -0.0006968602234187437),
    c64::new(0.0006820080560634501, -0.0006989558840662291),
    c64::new(0.0006798604799721319, -0.0007010449658747379),
    c64::new(0.0006777065047750527, -0.0007031274491810368),
    c64::new(0.0006755461507462479, -0.0007052033143839993),
    c64::new(0.0006733794382197919, -0.0007072725419447919),
    c64::new(0.0006712063875896086, -0.0007093351123870565),
    c64::new(0.0006690270193092778, -0.0007113910062970949),
    c64::new(0.0006668413538918438, -0.0007134402043240504),
    c64::new(0.0006646494119096222, -0.0007154826871800905),
    c64::new(0.0006624512139940054, -0.0007175184356405892),
    c64::new(0.0006602467808352696, -0.0007195474305443064),
    c64::new(0.000658036133182379, -0.0007215696527935694),
    c64::new(0.0006558192918427914, -0.0007235850833544521),
    c64::new(0.0006535962776822618, -0.0007255937032569547),
    c64::new(0.0006513671116246461, -0.0007275954935951816),
    c64::new(0.000649131814651704, -0.0007295904355275197),
    c64::new(0.0006468904078029021, -0.0007315785102768157),
    c64::new(0.0006446429121752154, -0.0007335596991305531),
    c64::new(0.0006423893489229285, -0.0007355339834410277),
    c64::new(0.000640129739257437, -0.0007375013446255239),
    c64::new(0.0006378641044470477, -0.0007394617641664887),
    c64::new(0.0006355924658167783, -0.000741415223611707),
    c64::new(0.0006333148447481568, -0.0007433617045744744),
    c64::new(0.0006310312626790199, -0.0007453011887337708),
    c64::new(0.000628741741103312, -0.0007472336578344325),
    c64::new(0.000626446301570882, -0.0007491590936873245),
    c64::new(0.0006241449656872811, -0.0007510774781695113),
    c64::new(0.0006218377551135589, -0.0007529887932244274),
    c64::new(0.0006195246915660602, -0.0007548930208620477),
    c64::new(0.0006172057968162198, -0.0007567901431590565),
    c64::new(0.0006148810926903585, -0.0007586801422590161),
    c64::new(0.0006125506010694768, -0.000760563000372535),
    c64::new(0.0006102143438890493, -0.0007624386997774359),
    c64::new(0.0006078723431388184, -0.0007643072228189215),
    c64::new(0.0006055246208625871, -0.0007661685519097414),
    c64::new(0.0006031711991580117, -0.0007680226695303572),
    c64::new(0.0006008121001763935, -0.0007698695582291076),
    c64::new(0.0005984473461224705, -0.0007717092006223731),
    c64::new(0.0005960769592542085, -0.0007735415793947386),
    c64::new(0.0005937009618825915, -0.0007753666772991573),
    c64::new(0.0005913193763714117, -0.0007771844771571127),
    c64::new(0.0005889322251370589, -0.0007789949618587803),
    c64::new(0.0005865395306483095, -0.0007807981143631885),
    c64::new(0.0005841413154261156, -0.000782593917698379),
    c64::new(0.0005817376020433921, -0.0007843823549615672),
    c64::new(0.0005793284131248046, -0.0007861634093193003),
    c64::new(0.0005769137713465569, -0.0007879370640076164),
    c64::new(0.0005744936994361771, -0.0007897033023322021),
    c64::new(0.0005720682201723036, -0.0007914621076685496),
    c64::new(0.0005696373563844711, -0.0007932134634621132),
    c64::new(0.0005672011309528952, -0.0007949573532284651),
    c64::new(0.0005647595668082575, -0.000796693760553451),
    c64::new(0.0005623126869314897, -0.0007984226690933435),
    c64::new(0.000559860514353557, -0.000800144062574997),
    c64::new(0.0005574030721552416, -0.0008018579247960006),
    c64::new(0.0005549403834669254, -0.0008035642396248304),
    c64::new(0.0005524724714683723, -0.0008052629910010013),
    c64::new(0.00054999935938851, -0.0008069541629352182),
    c64::new(0.000547521070505211, -0.000808637739509527),
    c64::new(0.0005450376281450744, -0.0008103137048774639),
    c64::new(0.0005425490556832054, -0.0008119820432642042),
    c64::new(0.0005400553765429957, -0.0008136427389667121),
    c64::new(0.0005375566141959032, -0.0008152957763538867),
    c64::new(0.0005350527921612308, -0.0008169411398667108),
    c64::new(0.0005325439340059049, -0.0008185788140183965),
    c64::new(0.0005300300633442539, -0.0008202087833945307),
    c64::new(0.0005275112038377861, -0.000821831032653221),
    c64::new(0.0005249873791949663, -0.0008234455465252397),
    c64::new(0.0005224586131709935, -0.000825052309814167),
    c64::new(0.0005199249295675762, -0.0008266513073965352),
    c64::new(0.0005173863522327098, -0.0008282425242219698),
    c64::new(0.0005148429050604506, -0.0008298259453133325),
    c64::new(0.0005122946119906923, -0.0008314015557668605),
    c64::new(0.0005097414970089398, -0.0008329693407523082),
    c64::new(0.0005071835841460836, -0.0008345292855130865),
    c64::new(0.0005046208974781738, -0.0008360813753664007),
    c64::new(0.0005020534611261931, -0.0008376255957033907),
    c64::new(0.0004994812992558306, -0.0008391619319892659),
    c64::new(0.000496904436077253, -0.0008406903697634445),
    c64::new(0.0004943228958448784, -0.000842210894639688),
    c64::new(0.0004917367028571461, -0.000843723492306237),
    c64::new(0.00048914588145629, -0.0008452281485259463),
    c64::new(0.00048655045602810735, -0.0008467248491364186),
    c64::new(0.000483950451001731, -0.0008482135800501376),
    c64::new(0.0004813458908493986, -0.0008496943272546009),
    c64::new(0.00047873680008622184, -0.0008511670768124521),
    c64::new(0.00047612320326995714, -0.0008526318148616113),
    c64::new(0.0004735051250007726, -0.0008540885276154064),
    c64::new(0.00047088258992101844, -0.0008555372013627017),
    c64::new(0.0004682556227149932, -0.0008569778224680286),
    c64::new(0.00046562424810871314, -0.0008584103773717123),
    c64::new(0.00046298849086967776, -0.0008598348525900007),
    c64::new(0.0004603483758066385, -0.0008612512347151904),
    c64::new(0.00045770392776936324, -0.0008626595104157535),
    c64::new(0.0004550551716484044, -0.0008640596664364627),
    c64::new(0.00045240213237486354, -0.0008654516895985162),
    c64::new(0.00044974483492015626, -0.000866835566799662),
    c64::new(0.0004470833042957786, -0.0008682112850143208),
    c64::new(0.00044441756555307015, -0.0008695788312937089),
    c64::new(0.0004417476437829795, -0.0008709381927659597),
    c64::new(0.00043907356411582675, -0.0008722893566362454),
    c64::new(0.0004363953517210687, -0.0008736323101868971),
    c64::new(0.0004337130318070598, -0.0008749670407775245),
    c64::new(0.000431026629620817, -0.0008762935358451351),
    c64::new(0.00042833617044778097, -0.0008776117829042518),
    c64::new(0.0004256416796115776, -0.0008789217695470315),
    c64::new(0.00042294318247378126, -0.0008802234834433809),
    c64::new(0.0004202407044336744, -0.0008815169123410731),
    c64::new(0.00041753427092800996, -0.0008828020440658626),
    c64::new(0.0004148239074307703, -0.0008840788665216004),
    c64::new(0.0004121096394529295, -0.000885347367690347),
    c64::new(0.0004093914925422109, -0.0008866075356324865),
    c64::new(0.00040666949228284894, -0.0008878593584868383),
    c64::new(0.00040394366429534625, -0.0008891028244707687),
    c64::new(0.0004012140342362343, -0.0008903379218803026),
    c64::new(0.0003984806277978308, -0.0008915646390902325),
    c64::new(0.00039574347070799786, -0.0008927829645542292),
    c64::new(0.0003930025887299009, -0.0008939928868049491),
    c64::new(0.00039025800766176444, -0.0008951943944541433),
    c64::new(0.00038750975333663127, -0.000896387476192764),
    c64::new(0.0003847578516221173, -0.000897572120791072),
    c64::new(0.0003820023284201699, -0.0008987483170987407),
    c64::new(0.00037924320966682256, -0.0008999160540449628),
    c64::new(0.00037648052133195216, -0.0009010753206385532),
    c64::new(0.00037371428941903304, -0.0009022261059680535),
    c64::new(0.00037094453996489366, -0.0009033683992018336),
    c64::new(0.000368171299039471, -0.0009045021895881947),
    c64::new(0.0003653945927455644, -0.0009056274664554698),
    c64::new(0.0003626144472185914, -0.0009067442192121245),
    c64::new(0.00035983088862634016, -0.0009078524373468567),
    c64::new(0.0003570439431687246, -0.000908952110428695),
    c64::new(0.00035425363707753635, -0.0009100432281070978),
    c64::new(0.0003514599966161995, -0.0009111257801120497),
    c64::new(0.0003486630480795215, -0.000912199756254159),
    c64::new(0.00034586281779344776, -0.0009132651464247534),
    c64::new(0.00034305933211481167, -0.0009143219405959749),
    c64::new(0.0003402526174310884, -0.000915370128820874),
    c64::new(0.00033744270016014565, -0.000916409701233504),
    c64::new(0.0003346296067499945, -0.0009174406480490136),
    c64::new(0.00033181336367854196, -0.0009184629595637386),
    c64::new(0.0003289939974533399, -0.0009194766261552938),
    c64::new(0.0003261715346113374, -0.000920481638282663),
    c64::new(0.0003233460017186293, -0.0009214779864862894),
    c64::new(0.0003205174253702077, -0.0009224656613881643),
    c64::new(0.00031768583218970994, -0.0009234446536919153),
    c64::new(0.0003148512488291699, -0.0009244149541828941),
    c64::new(0.0003120137019687654, -0.0009253765537282628),
    c64::new(0.0003091732183165682, -0.0009263294432770803),
    c64::new(0.0003063298246082925, -0.0009272736138603874),
    c64::new(0.00030348354760704246, -0.0009282090565912907),
    c64::new(0.0003006344141030615, -0.0009291357626650472),
    c64::new(0.0002977824509134789, -0.0009300537233591464),
    c64::new(0.0002949276848820588, -0.0009309629300333923),
    c64::new(0.0002920701428789458, -0.000931863374129986),
    c64::new(0.000289209851800414, -0.0009327550471736042),
    c64::new(0.000286346838568612, -0.0009336379407714809),
    c64::new(0.0002834811301313108, -0.0009345120466134853),
    c64::new(0.00028061275346164997, -0.0009353773564722001),
    c64::new(0.00027774173555788264, -0.0009362338622029996),
    c64::new(0.0002748681034431231, -0.0009370815557441255),
    c64::new(0.0002719918841650909, -0.0009379204291167635),
    c64::new(0.00026911310479585767, -0.0009387504744251178),
    c64::new(0.0002662317924315908, -0.0009395716838564859),
    c64::new(0.0002633479741923, -0.0009403840496813319),
    c64::new(0.0002604616772215805, -0.0009411875642533592),
    c64::new(0.0002575729286863589, -0.0009419822200095827),
    c64::new(0.0002546817557766363, -0.0009427680094703998),
    c64::new(0.00025178818570523333, -0.0009435449252396608),
    c64::new(0.0002488922457075338, -0.0009443129600047384),
    c64::new(0.00024599396304122745, -0.0009450721065365972),
    c64::new(0.00024309336498605494, -0.0009458223576898606),
    c64::new(0.0002401904788435494, -0.0009465637064028794),
    c64::new(0.00023728533193678123, -0.0009472961456977968),
    c64::new(0.00023437795161009912, -0.0009480196686806156),
    c64::new(0.00023146836522887435, -0.0009487342685412615),
    c64::new(0.00022855660017924166, -0.0009494399385536487),
    c64::new(0.00022564268386784302, -0.0009501366720757423),
    c64::new(0.00022272664372156815, -0.000950824462549621),
    c64::new(0.00021980850718729763, -0.0009515033035015389),
    c64::new(0.00021688830173164413, -0.0009521731885419864),
    c64::new(0.00021396605484069313, -0.0009528341113657506),
    c64::new(0.0002110417940197457, -0.0009534860657519738),
    c64::new(0.00020811554679305797, -0.0009541290455642133),
    c64::new(0.0002051873407035837, -0.000954763044750498),
    c64::new(0.00020225720331271344, -0.0009553880573433864),
    c64::new(0.0001993251622000166, -0.0009560040774600217),
    c64::new(0.0001963912449629804, -0.000956611099302188),
    c64::new(0.00019345547921675169, -0.0009572091171563646),
    c64::new(0.0001905178925938753, -0.0009577981253937797),
    c64::new(0.00018757851274403553, -0.0009583781184704634),
    c64::new(0.0001846373673337952, -0.0009589490909273),
    c64::new(0.00018169448404633458, -0.0009595110373900793),
    c64::new(0.00017874989058119242, -0.0009600639525695472),
    c64::new(0.00017580361465400343, -0.0009606078312614553),
    c64::new(0.00017285568399623912, -0.0009611426683466104),
    c64::new(0.00016990612635494517, -0.0009616684587909221),
    c64::new(0.0001669549694924818, -0.0009621851976454504),
    c64::new(0.00016400224118626094, -0.0009626928800464525),
    c64::new(0.0001610479692284864, -0.0009631915012154284),
    c64::new(0.0001580921814258905, -0.0009636810564591654),
    c64::new(0.00015513490559947401, -0.0009641615411697836),
    c64::new(0.00015217616958424347, -0.0009646329508247775),
    c64::new(0.00014921600122894864, -0.00096509528098706),
    c64::new(0.0001462544283958219, -0.0009655485273050037),
    c64::new(0.0001432914789603142, -0.0009659926855124815),
    c64::new(0.00014032718081083455, -0.000966427751428907),
    c64::new(0.00013736156184848558, -0.0009668537209592745),
    c64::new(0.0001343946499868029, -0.0009672705900941963),
    c64::new(0.00013142647315149045, -0.0009676783549099414),
    c64::new(0.00012845705928015943, -0.0009680770115684721),
    c64::new(0.0001254864363220637, -0.00096846655631748),
    c64::new(0.0001225146322378381, -0.0009688469854904217),
    c64::new(0.00011954167499923465, -0.0009692182955065527),
    c64::new(0.00011656759258885873, -0.000969580482870962),
    c64::new(0.00011359241299990709, -0.000969933544174604),
    c64::new(0.00011061616423590268, -0.0009702774760943314),
    c64::new(0.0001076388743104328, -0.0009706122753929258),
    c64::new(0.00010466057124688366, -0.0009709379389191288),
    c64::new(0.00010168128307817844, -0.0009712544636076713),
    c64::new(9.870103784651159e-05, -0.0009715618464793024),
    c64::new(9.571986360308669e-05, -0.0009718600846408172),
    c64::new(9.273778840785065e-05, -0.0009721491752850849),
    c64::new(8.975484032923115e-05, -0.000972429115691074),
    c64::new(8.677104744387176e-05, -0.0009726999032238789),
    c64::new(8.378643783636707e-05, -0.0009729615353347441),
    c64::new(8.080103959899981e-05, -0.0009732140095610887),
    c64::new(7.781488083147473e-05, -0.000973457323526529),
    c64::new(7.482798964065588e-05, -0.0009736914749409013),
    c64::new(7.184039414030025e-05, -0.0009739164616002834),
    c64::new(6.885212245079493e-05, -0.0009741322813870151),
    c64::new(6.586320269889073e-05, -0.0009743389322697184),
    c64::new(6.287366301743888e-05, -0.0009745364123033164),
    c64::new(5.988353154512564e-05, -0.0009747247196290519),
    c64::new(5.689283642620677e-05, -0.0009749038524745048),
    c64::new(5.390160581024417e-05, -0.0009750738091536082),
    c64::new(5.0909867851839176e-05, -0.0009752345880666653),
    c64::new(4.791765071036927e-05, -0.0009753861877003637),
    c64::new(4.492498254972133e-05, -0.0009755286066277899),
    c64::new(4.193189153802828e-05, -0.0009756618435084427),
    c64::new(3.893840584740219e-05, -0.000975785897088246),
    c64::new(3.594455365367089e-05, -0.0009759007661995601),
    c64::new(3.295036313611098e-05, -0.0009760064497611931),
    c64::new(2.9955862477184164e-05, -0.0009761029467784113),
    c64::new(2.6961079862271308e-05, -0.0009761902563429479),
    c64::new(2.396604347940651e-05, -0.000976268377633012),
    c64::new(2.097078151901329e-05, -0.0009763373099132962),
    c64::new(1.797532217363752e-05, -0.0009763970525349838),
    c64::new(1.4979693637683808e-05, -0.0009764476049357545),
    c64::new(1.1983924107148383e-05, -0.0009764889666397896),
    c64::new(8.98804177935543e-06, -0.0009765211372577772),
    c64::new(5.992074852689957e-06, -0.0009765441164869151),
    c64::new(2.996051526334119e-06, -0.0009765579041109142),
    c64::new(5.979720698961686e-20, -0.0009765625),
    c64::new(-2.9960515263339996e-06, -0.0009765579041109142),
    c64::new(-5.992074852689837e-06, -0.0009765441164869151),
    c64::new(-8.988041779355311e-06, -0.0009765211372577772),
    c64::new(-1.1983924107148265e-05, -0.0009764889666397896),
    c64::new(-1.497969363768369e-05, -0.0009764476049357545),
    c64::new(-1.7975322173637398e-05, -0.0009763970525349838),
    c64::new(-2.0970781519013168e-05, -0.0009763373099132962),
    c64::new(-2.396604347940639e-05, -0.000976268377633012),
    c64::new(-2.6961079862271186e-05, -0.0009761902563429479),
    c64::new(-2.9955862477184042e-05, -0.0009761029467784113),
    c64::new(-3.295036313611086e-05, -0.0009760064497611931),
    c64::new(-3.594455365367077e-05, -0.0009759007661995601),
    c64::new(-3.893840584740207e-05, -0.000975785897088246),
    c64::new(-4.193189153802816e-05, -0.0009756618435084427),
    c64::new(-4.492498254972121e-05, -0.0009755286066277899),
    c64::new(-4.791765071036915e-05, -0.0009753861877003637),
    c64::new(-5.0909867851839055e-05, -0.0009752345880666653),
    c64::new(-5.3901605810244056e-05, -0.0009750738091536082),
    c64::new(-5.689283642620665e-05, -0.0009749038524745048),
    c64::new(-5.988353154512552e-05, -0.0009747247196290519),
    c64::new(-6.287366301743877e-05, -0.0009745364123033164),
    c64::new(-6.586320269889062e-05, -0.0009743389322697184),
    c64::new(-6.885212245079482e-05, -0.0009741322813870151),
    c64::new(-7.184039414030014e-05, -0.0009739164616002834),
    c64::new(-7.482798964065577e-05, -0.0009736914749409013),
    c64::new(-7.781488083147462e-05, -0.000973457323526529),
    c64::new(-8.08010395989997e-05, -0.0009732140095610887),
    c64::new(-8.378643783636696e-05, -0.0009729615353347441),
    c64::new(-8.677104744387165e-05, -0.0009726999032238789),
    c64::new(-8.975484032923104e-05, -0.000972429115691074),
    c64::new(-9.273778840785054e-05, -0.0009721491752850849),
    c64::new(-9.571986360308658e-05, -0.0009718600846408173),
    c64::new(-9.870103784651148e-05, -0.0009715618464793024),
    c64::new(-0.00010168128307817833, -0.0009712544636076713),
    c64::new(-0.00010466057124688355, -0.0009709379389191288),
    c64::new(-0.00010763887431043269, -0.0009706122753929258),
    c64::new(-0.00011061616423590257, -0.0009702774760943314),
    c64::new(-0.00011359241299990698, -0.000969933544174604),
    c64::new(-0.00011656759258885862, -0.000969580482870962),
    c64::new(-0.00011954167499923454, -0.0009692182955065527),
    c64::new(-0.00012251463223783798, -0.0009688469854904217),
    c64::new(-0.0001254864363220636, -0.00096846655631748),
    c64::new(-0.00012845705928015935, -0.0009680770115684721),
    c64::new(-0.00013142647315149034, -0.0009676783549099414),
    c64::new(-0.0001343946499868028, -0.0009672705900941963),
    c64::new(-0.00013736156184848548, -0.0009668537209592745),
    c64::new(-0.00014032718081083444, -0.0009664277514289071),
    c64::new(-0.0001432914789603141, -0.0009659926855124815),
    c64::new(-0.00014625442839582179, -0.0009655485273050037),
    c64::new(-0.00014921600122894853, -0.00096509528098706),
    c64::new(-0.00015217616958424336, -0.0009646329508247775),
    c64::new(-0.0001551349055994739, -0.0009641615411697836),
    c64::new(-0.0001580921814258904, -0.0009636810564591656),
    c64::new(-0.00016104796922848628, -0.0009631915012154284),
    c64::new(-0.00016400224118626083, -0.0009626928800464526),
    c64::new(-0.00016695496949248168, -0.0009621851976454504),
    c64::new(-0.00016990612635494506, -0.0009616684587909221),
    c64::new(-0.000172855683996239, -0.0009611426683466104),
    c64::new(-0.00017580361465400332, -0.0009606078312614553),
    c64::new(-0.00017874989058119233, -0.0009600639525695472),
    c64::new(-0.0001816944840463345, -0.0009595110373900793),
    c64::new(-0.00018463736733379508, -0.0009589490909273),
    c64::new(-0.00018757851274403542, -0.0009583781184704634),
    c64::new(-0.0001905178925938752, -0.0009577981253937797),
    c64::new(-0.00019345547921675158, -0.0009572091171563647),
    c64::new(-0.0001963912449629803, -0.000956611099302188),
    c64::new(-0.0001993251622000165, -0.0009560040774600217),
    c64::new(-0.00020225720331271333, -0.0009553880573433864),
    c64::new(-0.0002051873407035836, -0.000954763044750498),
    c64::new(-0.00020811554679305786, -0.0009541290455642133),
    c64::new(-0.0002110417940197456, -0.0009534860657519738),
    c64::new(-0.00021396605484069303, -0.0009528341113657506),
    c64::new(-0.00021688830173164403, -0.0009521731885419864),
    c64::new(-0.00021980850718729752, -0.0009515033035015389),
    c64::new(-0.00022272664372156804, -0.0009508244625496211),
    c64::new(-0.00022564268386784293, -0.0009501366720757424),
    c64::new(-0.00022855660017924155, -0.0009494399385536487),
    c64::new(-0.00023146836522887424, -0.0009487342685412615),
    c64::new(-0.000234377951610099, -0.0009480196686806156),
    c64::new(-0.00023728533193678112, -0.0009472961456977968),
    c64::new(-0.0002401904788435493, -0.0009465637064028794),
    c64::new(-0.00024309336498605483, -0.0009458223576898606),
    c64::new(-0.00024599396304122735, -0.0009450721065365972),
    c64::new(-0.0002488922457075337, -0.0009443129600047384),
    c64::new(-0.0002517881857052332, -0.0009435449252396608),
    c64::new(-0.0002546817557766362, -0.0009427680094703998),
    c64::new(-0.0002575729286863588, -0.0009419822200095828),
    c64::new(-0.0002604616772215804, -0.0009411875642533592),
    c64::new(-0.0002633479741922999, -0.0009403840496813319),
    c64::new(-0.0002662317924315907, -0.0009395716838564859),
    c64::new(-0.00026911310479585756, -0.0009387504744251178),
    c64::new(-0.00027199188416509077, -0.0009379204291167636),
    c64::new(-0.000274868103443123, -0.0009370815557441255),
    c64::new(-0.00027774173555788253, -0.0009362338622029996),
    c64::new(-0.00028061275346164986, -0.0009353773564722001),
    c64::new(-0.00028348113013131076, -0.0009345120466134853),
    c64::new(-0.0002863468385686119, -0.000933637940771481),
    c64::new(-0.0002892098518004139, -0.0009327550471736042),
    c64::new(-0.00029207014287894567, -0.000931863374129986),
    c64::new(-0.0002949276848820587, -0.0009309629300333924),
    c64::new(-0.0002977824509134788, -0.0009300537233591464),
    c64::new(-0.0003006344141030614, -0.0009291357626650472),
    c64::new(-0.00030348354760704235, -0.0009282090565912908),
    c64::new(-0.00030632982460829244, -0.0009272736138603874),
    c64::new(-0.0003091732183165681, -0.0009263294432770804),
    c64::new(-0.00031201370196876527, -0.0009253765537282629),
    c64::new(-0.00031485124882916983, -0.0009244149541828941),
    c64::new(-0.00031768583218970983, -0.0009234446536919154),
    c64::new(-0.00032051742537020757, -0.0009224656613881643),
    c64::new(-0.0003233460017186292, -0.0009214779864862894),
    c64::new(-0.0003261715346113373, -0.000920481638282663),
    c64::new(-0.0003289939974533398, -0.0009194766261552938),
    c64::new(-0.00033181336367854185, -0.0009184629595637386),
    c64::new(-0.0003346296067499944, -0.0009174406480490136),
    c64::new(-0.00033744270016014555, -0.0009164097012335041),
    c64::new(-0.0003402526174310883, -0.000915370128820874),
    c64::new(-0.00034305933211481156, -0.0009143219405959749),
    c64::new(-0.00034586281779344766, -0.0009132651464247535),
    c64::new(-0.0003486630480795214, -0.0009121997562541592),
    c64::new(-0.0003514599966161994, -0.0009111257801120497),
    c64::new(-0.00035425363707753624, -0.0009100432281070978),
    c64::new(-0.00035704394316872447, -0.000908952110428695),
    c64::new(-0.00035983088862634005, -0.0009078524373468568),
    c64::new(-0.00036261444721859135, -0.0009067442192121246),
    c64::new(-0.00036539459274556433, -0.0009056274664554698),
    c64::new(-0.0003681712990394709, -0.0009045021895881947),
    c64::new(-0.00037094453996489355, -0.0009033683992018337),
    c64::new(-0.00037371428941903294, -0.0009022261059680535),
    c64::new(-0.00037648052133195206, -0.0009010753206385532),
    c64::new(-0.00037924320966682245, -0.0008999160540449629),
    c64::new(-0.0003820023284201698, -0.0008987483170987408),
    c64::new(-0.0003847578516221172, -0.000897572120791072),
    c64::new(-0.00038750975333663116, -0.0008963874761927642),
    c64::new(-0.0003902580076617644, -0.0008951943944541433),
    c64::new(-0.0003930025887299008, -0.0008939928868049491),
    c64::new(-0.0003957434707079978, -0.0008927829645542292),
    c64::new(-0.0003984806277978307, -0.0008915646390902326),
    c64::new(-0.0004012140342362342, -0.0008903379218803026),
    c64::new(-0.0004039436642953462, -0.0008891028244707688),
    c64::new(-0.0004066694922828486, -0.0008878593584868384),
    c64::new(-0.000409391492542211, -0.0008866075356324865),
    c64::new(-0.0004121096394529294, -0.000885347367690347),
    c64::new(-0.00041482390743077026, -0.0008840788665216004),
    c64::new(-0.00041753427092800963, -0.0008828020440658627),
    c64::new(-0.00042024070443367453, -0.0008815169123410731),
    c64::new(-0.0004229431824737812, -0.0008802234834433809),
    c64::new(-0.0004256416796115775, -0.0008789217695470315),
    c64::new(-0.00042833617044778065, -0.0008776117829042519),
    c64::new(-0.0004310266296208171, -0.000876293535845135),
    c64::new(-0.0004337130318070597, -0.0008749670407775246),
    c64::new(-0.00043639535172106835, -0.0008736323101868973),
    c64::new(-0.00043907356411582686, -0.0008722893566362453),
    c64::new(-0.0004417476437829794, -0.0008709381927659597),
    c64::new(-0.00044441756555307004, -0.000869578831293709),
    c64::new(-0.00044708330429577833, -0.000868211285014321),
    c64::new(-0.00044974483492015636, -0.000866835566799662),
    c64::new(-0.0004524021323748634, -0.0008654516895985162),
    c64::new(-0.0004550551716484043, -0.0008640596664364628),
    c64::new(-0.00045770392776936297, -0.0008626595104157536),
    c64::new(-0.0004603483758066384, -0.0008612512347151905),
    c64::new(-0.0004629884908696777, -0.0008598348525900008),
    c64::new(-0.00046562424810871287, -0.0008584103773717124),
    c64::new(-0.0004682556227149932, -0.0008569778224680286),
    c64::new(-0.0004708825899210183, -0.0008555372013627018),
    c64::new(-0.0004735051250007724, -0.0008540885276154065),
    c64::new(-0.00047612320326995676, -0.0008526318148616115),
    c64::new(-0.00047873680008622184, -0.0008511670768124521),
    c64::new(-0.00048134589084939837, -0.000849694327254601),
    c64::new(-0.0004839504510017308, -0.0008482135800501377),
    c64::new(-0.00048655045602810697, -0.0008467248491364188),
    c64::new(-0.0004891458814562898, -0.0008452281485259464),
    c64::new(-0.000491736702857146, -0.0008437234923062371),
    c64::new(-0.000494322895844878, -0.0008422108946396882),
    c64::new(-0.000496904436077253, -0.0008406903697634445),
    c64::new(-0.0004994812992558304, -0.0008391619319892661),
    c64::new(-0.0005020534611261929, -0.0008376255957033907),
    c64::new(-0.0005046208974781735, -0.000836081375366401),
    c64::new(-0.0005071835841460836, -0.0008345292855130865),
    c64::new(-0.0005097414970089397, -0.0008329693407523083),
    c64::new(-0.0005122946119906921, -0.0008314015557668605),
    c64::new(-0.0005148429050604503, -0.0008298259453133327),
    c64::new(-0.0005173863522327096, -0.00082824252422197),
    c64::new(-0.000519924929567576, -0.0008266513073965353),
    c64::new(-0.0005224586131709931, -0.0008250523098141672),
    c64::new(-0.0005249873791949663, -0.0008234455465252397),
    c64::new(-0.000527511203837786, -0.0008218310326532211),
    c64::new(-0.0005300300633442537, -0.0008202087833945309),
    c64::new(-0.0005325439340059044, -0.0008185788140183967),
    c64::new(-0.0005350527921612307, -0.0008169411398667109),
    c64::new(-0.000537556614195903, -0.0008152957763538868),
    c64::new(-0.0005400553765429955, -0.0008136427389667122),
    c64::new(-0.000542549055683205, -0.0008119820432642045),
    c64::new(-0.0005450376281450743, -0.0008103137048774639),
    c64::new(-0.0005475210705052109, -0.0008086377395095273),
    c64::new(-0.0005499993593885096, -0.0008069541629352184),
    c64::new(-0.0005524724714683723, -0.0008052629910010013),
    c64::new(-0.0005549403834669253, -0.0008035642396248305),
    c64::new(-0.0005574030721552414, -0.0008018579247960007),
    c64::new(-0.0005598605143535566, -0.0008001440625749972),
    c64::new(-0.0005623126869314896, -0.0007984226690933435),
    c64::new(-0.0005647595668082573, -0.0007966937605534511),
    c64::new(-0.0005672011309528948, -0.0007949573532284654),
    c64::new(-0.0005696373563844711, -0.0007932134634621132),
    c64::new(-0.0005720682201723036, -0.0007914621076685496),
    c64::new(-0.000574493699436177, -0.0007897033023322022),
    c64::new(-0.0005769137713465567, -0.0007879370640076166),
    c64::new(-0.0005793284131248046, -0.0007861634093193003),
    c64::new(-0.0005817376020433918, -0.0007843823549615673),
    c64::new(-0.0005841413154261154, -0.0007825939176983792),
    c64::new(-0.0005865395306483092, -0.0007807981143631887),
    c64::new(-0.0005889322251370588, -0.0007789949618587804),
    c64::new(-0.0005913193763714116, -0.0007771844771571129),
    c64::new(-0.0005937009618825913, -0.0007753666772991575),
    c64::new(-0.0005960769592542085, -0.0007735415793947386),
    c64::new(-0.0005984473461224704, -0.0007717092006223731),
    c64::new(-0.0006008121001763933, -0.0007698695582291078),
    c64::new(-0.0006031711991580115, -0.0007680226695303573),
    c64::new(-0.0006055246208625871, -0.0007661685519097414),
    c64::new(-0.0006078723431388183, -0.0007643072228189216),
    c64::new(-0.0006102143438890491, -0.0007624386997774361),
    c64::new(-0.0006125506010694764, -0.0007605630003725354),
    c64::new(-0.0006148810926903584, -0.0007586801422590161),
    c64::new(-0.0006172057968162197, -0.0007567901431590566),
    c64::new(-0.0006195246915660598, -0.000754893020862048),
    c64::new(-0.0006218377551135589, -0.0007529887932244274),
    c64::new(-0.000624144965687281, -0.0007510774781695114),
    c64::new(-0.0006264463015708819, -0.0007491590936873247),
    c64::new(-0.0006287417411033118, -0.0007472336578344327),
    c64::new(-0.0006310312626790199, -0.0007453011887337708),
    c64::new(-0.0006333148447481567, -0.0007433617045744745),
    c64::new(-0.0006355924658167781, -0.0007414152236117072),
    c64::new(-0.0006378641044470474, -0.000739461764166489),
    c64::new(-0.0006401297392574369, -0.000737501344625524),
    c64::new(-0.0006423893489229282, -0.0007355339834410279),
    c64::new(-0.0006446429121752151, -0.0007335596991305533),
    c64::new(-0.0006468904078029021, -0.0007315785102768157),
    c64::new(-0.0006491318146517039, -0.0007295904355275197),
    c64::new(-0.0006513671116246459, -0.0007275954935951817),
    c64::new(-0.0006535962776822615, -0.0007255937032569549),
    c64::new(-0.0006558192918427914, -0.0007235850833544521),
    c64::new(-0.0006580361331823789, -0.0007215696527935696),
    c64::new(-0.0006602467808352693, -0.0007195474305443067),
    c64::new(-0.0006624512139940051, -0.0007175184356405896),
    c64::new(-0.0006646494119096221, -0.0007154826871800906),
    c64::new(-0.0006668413538918437, -0.0007134402043240504),
    c64::new(-0.0006690270193092776, -0.000711391006297095),
    c64::new(-0.0006712063875896086, -0.0007093351123870565),
    c64::new(-0.0006733794382197918, -0.000707272541944792),
    c64::new(-0.0006755461507462477, -0.0007052033143839995),
    c64::new(-0.0006777065047750525, -0.000703127449181037),
    c64::new(-0.0006798604799721318, -0.000701044965874738),
    c64::new(-0.00068200805606345, -0.0006989558840662292),
    c64::new(-0.0006841492128352033, -0.0006968602234187439),
    c64::new(-0.0006862839301340089, -0.0006947580036574382),
    c64::new(-0.0006884121878670946, -0.0006926492445692048),
    c64::new(-0.0006905339660024878, -0.0006905339660024879),
    c64::new(-0.0006926492445692046, -0.0006884121878670948),
    c64::new(-0.000694758003657438, -0.000686283930134009),
    c64::new(-0.0006968602234187437, -0.0006841492128352035),
    c64::new(-0.0006989558840662291, -0.0006820080560634501),
    c64::new(-0.0007010449658747379, -0.0006798604799721319),
    c64::new(-0.0007031274491810368, -0.0006777065047750526),
    c64::new(-0.0007052033143839993, -0.0006755461507462479),
    c64::new(-0.0007072725419447919, -0.000673379438219792),
    c64::new(-0.0007093351123870564, -0.0006712063875896087),
    c64::new(-0.0007113910062970949, -0.0006690270193092777),
    c64::new(-0.0007134402043240504, -0.0006668413538918438),
    c64::new(-0.0007154826871800905, -0.0006646494119096223),
    c64::new(-0.0007175184356405895, -0.0006624512139940053),
    c64::new(-0.0007195474305443064, -0.0006602467808352695),
    c64::new(-0.0007215696527935694, -0.000658036133182379),
    c64::new(-0.000723585083354452, -0.0006558192918427916),
    c64::new(-0.0007255937032569548, -0.0006535962776822617),
    c64::new(-0.0007275954935951816, -0.0006513671116246461),
    c64::new(-0.0007295904355275196, -0.0006491318146517041),
    c64::new(-0.0007315785102768155, -0.0006468904078029023),
    c64::new(-0.0007335596991305531, -0.0006446429121752153),
    c64::new(-0.0007355339834410277, -0.0006423893489229285),
    c64::new(-0.0007375013446255239, -0.0006401297392574371),
    c64::new(-0.0007394617641664889, -0.0006378641044470475),
    c64::new(-0.000741415223611707, -0.0006355924658167783),
    c64::new(-0.0007433617045744744, -0.0006333148447481568),
    c64::new(-0.0007453011887337707, -0.00063103126267902),
    c64::new(-0.0007472336578344326, -0.0006287417411033119),
    c64::new(-0.0007491590936873245, -0.000626446301570882),
    c64::new(-0.0007510774781695113, -0.0006241449656872811),
    c64::new(-0.0007529887932244273, -0.000621837755113559),
    c64::new(-0.0007548930208620478, -0.00061952469156606),
    c64::new(-0.0007567901431590565, -0.0006172057968162198),
    c64::new(-0.000758680142259016, -0.0006148810926903586),
    c64::new(-0.0007605630003725352, -0.0006125506010694766),
    c64::new(-0.000762438699777436, -0.0006102143438890492),
    c64::new(-0.0007643072228189215, -0.0006078723431388184),
    c64::new(-0.0007661685519097413, -0.0006055246208625873),
    c64::new(-0.0007680226695303572, -0.0006031711991580116),
    c64::new(-0.0007698695582291076, -0.0006008121001763935),
    c64::new(-0.000771709200622373, -0.0005984473461224705),
    c64::new(-0.0007735415793947384, -0.0005960769592542087),
    c64::new(-0.0007753666772991574, -0.0005937009618825914),
    c64::new(-0.0007771844771571127, -0.0005913193763714117),
    c64::new(-0.0007789949618587802, -0.000588932225137059),
    c64::new(-0.0007807981143631885, -0.0005865395306483094),
    c64::new(-0.0007825939176983791, -0.0005841413154261156),
    c64::new(-0.0007843823549615672, -0.0005817376020433921),
    c64::new(-0.0007861634093193002, -0.0005793284131248047),
    c64::new(-0.0007879370640076165, -0.0005769137713465568),
    c64::new(-0.0007897033023322021, -0.0005744936994361771),
    c64::new(-0.0007914621076685495, -0.0005720682201723037),
    c64::new(-0.000793213463462113, -0.0005696373563844712),
    c64::new(-0.0007949573532284652, -0.000567201130952895),
    c64::new(-0.000796693760553451, -0.0005647595668082575),
    c64::new(-0.0007984226690933434, -0.0005623126869314897),
    c64::new(-0.0008001440625749971, -0.0005598605143535567),
    c64::new(-0.0008018579247960006, -0.0005574030721552415),
    c64::new(-0.0008035642396248303, -0.0005549403834669255),
    c64::new(-0.000805262991001001, -0.0005524724714683725),
    c64::new(-0.0008069541629352183, -0.0005499993593885099),
    c64::new(-0.000808637739509527, -0.000547521070505211),
    c64::new(-0.0008103137048774638, -0.0005450376281450745),
    c64::new(-0.0008119820432642044, -0.0005425490556832053),
    c64::new(-0.0008136427389667121, -0.0005400553765429956),
    c64::new(-0.0008152957763538867, -0.0005375566141959032),
    c64::new(-0.0008169411398667108, -0.0005350527921612308),
    c64::new(-0.0008185788140183966, -0.0005325439340059046),
    c64::new(-0.0008202087833945308, -0.0005300300633442538),
    c64::new(-0.000821831032653221, -0.0005275112038377861),
    c64::new(-0.0008234455465252396, -0.0005249873791949665),
    c64::new(-0.0008250523098141671, -0.0005224586131709933),
    c64::new(-0.0008266513073965352, -0.0005199249295675762),
    c64::new(-0.0008282425242219698, -0.0005173863522327098),
    c64::new(-0.0008298259453133326, -0.0005148429050604505),
    c64::new(-0.0008314015557668605, -0.0005122946119906923),
    c64::new(-0.0008329693407523082, -0.0005097414970089398),
    c64::new(-0.0008345292855130864, -0.0005071835841460837),
    c64::new(-0.000836081375366401, -0.0005046208974781736),
    c64::new(-0.0008376255957033907, -0.0005020534611261931),
    c64::new(-0.0008391619319892659, -0.0004994812992558306),
    c64::new(-0.0008406903697634445, -0.0004969044360772532),
    c64::new(-0.000842210894639688, -0.0004943228958448782),
    c64::new(-0.000843723492306237, -0.0004917367028571461),
    c64::new(-0.0008452281485259463, -0.00048914588145629),
    c64::new(-0.0008467248491364186, -0.0004865504560281072),
    c64::new(-0.0008482135800501376, -0.000483950451001731),
    c64::new(-0.0008496943272546009, -0.0004813458908493986),
    c64::new(-0.0008511670768124521, -0.00047873680008622205),
    c64::new(-0.0008526318148616115, -0.0004761232032699569),
    c64::new(-0.0008540885276154064, -0.0004735051250007726),
    c64::new(-0.0008555372013627017, -0.00047088258992101844),
    c64::new(-0.0008569778224680284, -0.00046825562271499344),
    c64::new(-0.0008584103773717124, -0.000465624248108713),
    c64::new(-0.0008598348525900007, -0.0004629884908696778),
    c64::new(-0.0008612512347151904, -0.0004603483758066385),
    c64::new(-0.0008626595104157536, -0.00045770392776936313),
    c64::new(-0.0008640596664364627, -0.0004550551716484045),
    c64::new(-0.0008654516895985161, -0.0004524021323748636),
    c64::new(-0.0008668355667996619, -0.00044974483492015647),
    c64::new(-0.0008682112850143209, -0.0004470833042957785),
    c64::new(-0.0008695788312937089, -0.0004444175655530702),
    c64::new(-0.0008709381927659596, -0.0004417476437829795),
    c64::new(-0.0008722893566362453, -0.000439073564115827),
    c64::new(-0.0008736323101868972, -0.0004363953517210685),
    c64::new(-0.0008749670407775245, -0.0004337130318070599),
    c64::new(-0.000876293535845135, -0.0004310266296208173),
    c64::new(-0.0008776117829042519, -0.0004283361704477808),
    c64::new(-0.0008789217695470315, -0.0004256416796115776),
    c64::new(-0.0008802234834433808, -0.0004229431824737813),
    c64::new(-0.000881516912341073, -0.0004202407044336747),
    c64::new(-0.0008828020440658626, -0.0004175342709280098),
    c64::new(-0.0008840788665216003, -0.00041482390743077037),
    c64::new(-0.000885347367690347, -0.00041210963945292954),
    c64::new(-0.0008866075356324864, -0.00040939149254221117),
    c64::new(-0.0008878593584868383, -0.0004066694922828488),
    c64::new(-0.0008891028244707687, -0.0004039436642953463),
    c64::new(-0.0008903379218803025, -0.0004012140342362345),
    c64::new(-0.0008915646390902326, -0.00039848062779783064),
    c64::new(-0.0008927829645542292, -0.0003957434707079979),
    c64::new(-0.000893992886804949, -0.0003930025887299009),
    c64::new(-0.0008951943944541432, -0.0003902580076617647),
    c64::new(-0.0008963874761927642, -0.0003875097533366311),
    c64::new(-0.000897572120791072, -0.00038475785162211734),
    c64::new(-0.0008987483170987407, -0.00038200232842016996),
    c64::new(-0.0008999160540449627, -0.00037924320966682283),
    c64::new(-0.0009010753206385533, -0.000376480521331952),
    c64::new(-0.0009022261059680535, -0.00037371428941903304),
    c64::new(-0.0009033683992018335, -0.00037094453996489393),
    c64::new(-0.0009045021895881947, -0.00036817129903947085),
    c64::new(-0.0009056274664554698, -0.00036539459274556444),
    c64::new(-0.0009067442192121245, -0.00036261444721859146),
    c64::new(-0.0009078524373468567, -0.00035983088862634043),
    c64::new(-0.000908952110428695, -0.0003570439431687244),
    c64::new(-0.0009100432281070978, -0.0003542536370775364),
    c64::new(-0.0009111257801120497, -0.00035145999661619954),
    c64::new(-0.0009121997562541589, -0.0003486630480795218),
    c64::new(-0.0009132651464247535, -0.0003458628177934476),
    c64::new(-0.0009143219405959749, -0.0003430593321148117),
    c64::new(-0.0009153701288208739, -0.00034025261743108866),
    c64::new(-0.0009164097012335041, -0.0003374427001601455),
    c64::new(-0.0009174406480490136, -0.00033462960674999456),
    c64::new(-0.0009184629595637386, -0.000331813363678542),
    c64::new(-0.0009194766261552936, -0.0003289939974533401),
    c64::new(-0.000920481638282663, -0.00032617153461133723),
    c64::new(-0.0009214779864862894, -0.0003233460017186294),
    c64::new(-0.0009224656613881642, -0.00032051742537020773),
    c64::new(-0.0009234446536919153, -0.00031768583218971016),
    c64::new(-0.0009244149541828941, -0.0003148512488291698),
    c64::new(-0.0009253765537282628, -0.00031201370196876543),
    c64::new(-0.0009263294432770803, -0.0003091732183165684),
    c64::new(-0.0009272736138603874, -0.0003063298246082924),
    c64::new(-0.0009282090565912907, -0.0003034835476070425),
    c64::new(-0.0009291357626650472, -0.00030063441410306156),
    c64::new(-0.0009300537233591463, -0.00029778245091347915),
    c64::new(-0.0009309629300333924, -0.0002949276848820586),
    c64::new(-0.0009318633741299859, -0.00029207014287894583),
    c64::new(-0.0009327550471736042, -0.0002892098518004141),
    c64::new(-0.0009336379407714809, -0.00028634683856861227),
    c64::new(-0.0009345120466134852, -0.00028348113013131087),
    c64::new(-0.0009353773564722001, -0.00028061275346165),
    c64::new(-0.0009362338622029995, -0.00027774173555788285),
    c64::new(-0.0009370815557441255, -0.00027486810344312293),
    c64::new(-0.0009379204291167635, -0.00027199188416509093),
    c64::new(-0.0009387504744251178, -0.0002691131047958577),
    c64::new(-0.0009395716838564859, -0.00026623179243159107),
    c64::new(-0.0009403840496813319, -0.00026334797419229984),
    c64::new(-0.0009411875642533592, -0.00026046167722158054),
    c64::new(-0.0009419822200095827, -0.00025757292868635895),
    c64::new(-0.0009427680094703997, -0.00025468175577663657),
    c64::new(-0.0009435449252396606, -0.0002517881857052334),
    c64::new(-0.0009443129600047384, -0.00024889224570753387),
    c64::new(-0.0009450721065365971, -0.0002459939630412277),
    c64::new(-0.0009458223576898607, -0.00024309336498605478),
    c64::new(-0.0009465637064028794, -0.00024019047884354946),
    c64::new(-0.0009472961456977968, -0.0002372853319367813),
    c64::new(-0.0009480196686806155, -0.00023437795161009936),
    c64::new(-0.0009487342685412615, -0.0002314683652288742),
    c64::new(-0.0009494399385536487, -0.00022855660017924169),
    c64::new(-0.0009501366720757423, -0.00022564268386784307),
    c64::new(-0.000950824462549621, -0.00022272664372156842),
    c64::new(-0.0009515033035015389, -0.0002198085071872977),
    c64::new(-0.0009521731885419864, -0.0002168883017316442),
    c64::new(-0.0009528341113657504, -0.00021396605484069338),
    c64::new(-0.0009534860657519738, -0.00021104179401974554),
    c64::new(-0.0009541290455642133, -0.00020811554679305802),
    c64::new(-0.000954763044750498, -0.00020518734070358377),
    c64::new(-0.0009553880573433863, -0.0002022572033127137),
    c64::new(-0.0009560040774600217, -0.00019932516220001645),
    c64::new(-0.000956611099302188, -0.00019639124496298045),
    c64::new(-0.0009572091171563646, -0.00019345547921675174),
    c64::new(-0.0009577981253937797, -0.00019051789259387557),
    c64::new(-0.0009583781184704634, -0.00018757851274403558),
    c64::new(-0.0009589490909273, -0.00018463736733379525),
    c64::new(-0.0009595110373900793, -0.00018169448404633485),
    c64::new(-0.0009600639525695473, -0.00017874989058119228),
    c64::new(-0.0009606078312614553, -0.0001758036146540035),
    c64::new(-0.0009611426683466104, -0.00017285568399623918),
    c64::new(-0.0009616684587909221, -0.0001699061263549454),
    c64::new(-0.0009621851976454504, -0.00016695496949248163),
    c64::new(-0.0009626928800464525, -0.000164002241186261),
    c64::new(-0.0009631915012154283, -0.00016104796922848644),
    c64::new(-0.0009636810564591656, -0.00015809218142589035),
    c64::new(-0.0009641615411697836, -0.00015513490559947407),
    c64::new(-0.0009646329508247775, -0.00015217616958424353),
    c64::new(-0.00096509528098706, -0.0001492160012289489),
    c64::new(-0.0009655485273050037, -0.00014625442839582173),
    c64::new(-0.0009659926855124815, -0.00014329147896031426),
    c64::new(-0.000966427751428907, -0.0001403271808108346),
    c64::new(-0.0009668537209592744, -0.00013736156184848586),
    c64::new(-0.0009672705900941964, -0.00013439464998680274),
    c64::new(-0.0009676783549099414, -0.0001314264731514905),
    c64::new(-0.0009680770115684721, -0.00012845705928015948),
    c64::new(-0.00096846655631748, -0.00012548643632206353),
    c64::new(-0.0009688469854904217, -0.00012251463223783815),
    c64::new(-0.0009692182955065527, -0.00011954167499923472),
    c64::new(-0.000969580482870962, -0.00011656759258885902),
    c64::new(-0.000969933544174604, -0.00011359241299990694),
    c64::new(-0.0009702774760943314, -0.00011061616423590275),
    c64::new(-0.0009706122753929258, -0.00010763887431043286),
    c64::new(-0.0009709379389191288, -0.00010466057124688395),
    c64::new(-0.0009712544636076713, -0.0001016812830781783),
    c64::new(-0.0009715618464793024, -9.870103784651165e-05),
    c64::new(-0.0009718600846408172, -9.571986360308676e-05),
    c64::new(-0.0009721491752850849, -9.27377884078505e-05),
    c64::new(-0.000972429115691074, -8.975484032923122e-05),
    c64::new(-0.0009726999032238789, -8.677104744387183e-05),
    c64::new(-0.0009729615353347441, -8.378643783636734e-05),
    c64::new(-0.0009732140095610887, -8.080103959899966e-05),
    c64::new(-0.000973457323526529, -7.78148808314748e-05),
    c64::new(-0.0009736914749409013, -7.482798964065594e-05),
    c64::new(-0.0009739164616002834, -7.184039414030053e-05),
    c64::new(-0.0009741322813870151, -6.885212245079478e-05),
    c64::new(-0.0009743389322697184, -6.58632026988908e-05),
    c64::new(-0.0009745364123033164, -6.287366301743916e-05),
    c64::new(-0.0009747247196290519, -5.9883531545125483e-05),
    c64::new(-0.0009749038524745048, -5.689283642620683e-05),
    c64::new(-0.0009750738091536082, -5.390160581024423e-05),
    c64::new(-0.0009752345880666652, -5.0909867851839454e-05),
    c64::new(-0.0009753861877003637, -4.7917650710369115e-05),
    c64::new(-0.0009755286066277899, -4.492498254972139e-05),
    c64::new(-0.0009756618435084427, -4.193189153802834e-05),
    c64::new(-0.000975785897088246, -3.893840584740247e-05),
    c64::new(-0.0009759007661995601, -3.5944553653670734e-05),
    c64::new(-0.0009760064497611931, -3.295036313611104e-05),
    c64::new(-0.0009761029467784113, -2.995586247718444e-05),
    c64::new(-0.0009761902563429479, -2.696107986227115e-05),
    c64::new(-0.000976268377633012, -2.396604347940657e-05),
    c64::new(-0.0009763373099132962, -2.0970781519013347e-05),
    c64::new(-0.0009763970525349838, -1.7975322173637794e-05),
    c64::new(-0.0009764476049357545, -1.4979693637683653e-05),
    c64::new(-0.0009764889666397896, -1.1983924107148444e-05),
    c64::new(-0.0009765211372577772, -8.98804177935549e-06),
    c64::new(-0.0009765441164869151, -5.992074852690233e-06),
    c64::new(-0.0009765579041109142, -2.996051526333962e-06),
];

static TWIDDLES_2048: [c64; 2048] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999988234517019, 0.0015339801862847655),
    c64::new(0.9999952938095762, 0.003067956762965976),
    c64::new(0.9999894110819284, 0.0046019261204485705),
    c64::new(0.9999811752826011, 0.006135884649154475),
    c64::new(0.9999705864309741, 0.007669828739531097),
    c64::new(0.9999576445519639, 0.00920375478205982),
    c64::new(0.9999423496760239, 0.01073765916726449),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9999047010828529, 0.01380538852806039),
    c64::new(0.9998823474542126, 0.0153392062849881),
    c64::new(0.9998576410058239, 0.01687298794728171),
    c64::new(0.9998305817958234, 0.01840672990580482),
    c64::new(0.9998011698878843, 0.01994042855151444),
    c64::new(0.9997694053512153, 0.021474080275469508),
    c64::new(0.9997352882605617, 0.02300768146883937),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9996599967439592, 0.0260747178291039),
    c64::new(0.9996188224951786, 0.02760814577896574),
    c64::new(0.9995752960467492, 0.029141508764193722),
    c64::new(0.9995294175010931, 0.030674803176636626),
    c64::new(0.999481186966167, 0.032208025408304586),
    c64::new(0.9994306045554617, 0.03374117185137758),
    c64::new(0.9993776703880028, 0.03527423889821395),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9992647472865944, 0.038340120373552694),
    c64::new(0.9992047586183639, 0.03987292758773981),
    c64::new(0.9991424187248169, 0.04140564097707674),
    c64::new(0.9990777277526454, 0.04293825693494082),
    c64::new(0.9990106858540734, 0.04447077185493867),
    c64::new(0.9989412931868569, 0.04600318213091462),
    c64::new(0.9988695499142836, 0.0475354841569593),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9987190122338729, 0.05059974903689928),
    c64::new(0.9986402181802653, 0.052131704680283324),
    c64::new(0.9985590742297593, 0.05366353765273052),
    c64::new(0.9984755805732948, 0.05519524434968994),
    c64::new(0.9983897374073402, 0.05672682116690775),
    c64::new(0.9983015449338929, 0.05825826450043575),
    c64::new(0.9982110033604782, 0.05978957074663987),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.9980228737714862, 0.0628517575641614),
    c64::new(0.997925286198596, 0.06438263092985747),
    c64::new(0.9978253504111116, 0.0659133527970038),
    c64::new(0.9977230666441916, 0.06744391956366405),
    c64::new(0.9976184351385196, 0.06897432762826675),
    c64::new(0.9975114561403035, 0.07050457338961386),
    c64::new(0.9974021299012753, 0.07203465324688933),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.9971764367353262, 0.0750943008479213),
    c64::new(0.997060070339483, 0.07662386139203149),
    c64::new(0.9969413577649822, 0.07815324163279423),
    c64::new(0.9968202992911657, 0.07968243797143013),
    c64::new(0.9966968952028961, 0.08121144680959244),
    c64::new(0.9965711457905548, 0.08274026454937569),
    c64::new(0.9964430513500426, 0.08426888759332407),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.996179828595697, 0.08732553520619206),
    c64::new(0.996044700901252, 0.0888535525825246),
    c64::new(0.9959072294174117, 0.09038136087786498),
    c64::new(0.9957674144676598, 0.09190895649713272),
    c64::new(0.9956252563809943, 0.09343633584574779),
    c64::new(0.9954807554919269, 0.09496349532963899),
    c64::new(0.9953339121404823, 0.09649043135525259),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9950331994381186, 0.09954361866006932),
    c64::new(0.9948793307948056, 0.10106986275482782),
    c64::new(0.9947231211043257, 0.10259586902243628),
    c64::new(0.9945645707342554, 0.10412163387205459),
    c64::new(0.9944036800576791, 0.10564715371341062),
    c64::new(0.9942404494531879, 0.10717242495680884),
    c64::new(0.9940748793048794, 0.10869744401313872),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.9937367219407246, 0.11174671121112659),
    c64::new(0.9935641355205953, 0.11327095217756435),
    c64::new(0.9933892111480807, 0.11479492660651008),
    c64::new(0.9932119492347945, 0.11631863091190475),
    c64::new(0.9930323501978514, 0.11784206150832498),
    c64::new(0.9928504144598651, 0.11936521481099135),
    c64::new(0.992666142448948, 0.12088808723577708),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.9922905913482574, 0.12393297511851216),
    c64::new(0.9920993131421918, 0.12545498341154623),
    c64::new(0.9919057004306093, 0.12697669649688587),
    c64::new(0.9917097536690995, 0.12849811079379317),
    c64::new(0.9915114733187439, 0.13001922272223335),
    c64::new(0.9913108598461154, 0.13154002870288312),
    c64::new(0.9911079137232769, 0.13306052515713906),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.9906950254426646, 0.1361005751757062),
    c64::new(0.9904850842564571, 0.13762012158648604),
    c64::new(0.9902728123631691, 0.1391393441638262),
    c64::new(0.9900582102622971, 0.1406582393328492),
    c64::new(0.9898412784588205, 0.14217680351944803),
    c64::new(0.9896220174632009, 0.14369503315029447),
    c64::new(0.9894004277913804, 0.14521292465284746),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.988950264510303, 0.14824767898689603),
    c64::new(0.9887216919603238, 0.1497645346773215),
    c64::new(0.9884907928526966, 0.15128103795733022),
    c64::new(0.9882575677307495, 0.15279718525844344),
    c64::new(0.9880220171432835, 0.1543129730130201),
    c64::new(0.9877841416445722, 0.15582839765426523),
    c64::new(0.9875439417943592, 0.15734345561623825),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.987056571305751, 0.16037245724292828),
    c64::new(0.9868094018141855, 0.16188639378011183),
    c64::new(0.9865599102647754, 0.16339994938297323),
    c64::new(0.9863080972445987, 0.16491312048996992),
    c64::new(0.9860539633461954, 0.1664259035404641),
    c64::new(0.9857975091675675, 0.16793829497473117),
    c64::new(0.9855387353121761, 0.16945029123396796),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9850142310122398, 0.17247308399679595),
    c64::new(0.9847485018019042, 0.17398387338746382),
    c64::new(0.9844804553832209, 0.17549425337727143),
    c64::new(0.984210092386929, 0.17700422041214875),
    c64::new(0.9839374134492189, 0.1785137709389975),
    c64::new(0.9836624192117303, 0.18002290140569951),
    c64::new(0.9833851103215512, 0.18153160826112497),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9828235511987052, 0.18454773693861962),
    c64::new(0.9825393022874412, 0.18605515166344663),
    c64::new(0.9822527413662894, 0.1875621285825296),
    c64::new(0.9819638691095552, 0.1890686641498062),
    c64::new(0.9816726861969831, 0.19057475482025274),
    c64::new(0.9813791933137546, 0.19208039704989244),
    c64::new(0.9810833911504867, 0.1935855872958036),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9804848617734694, 0.19659459767008022),
    c64::new(0.9801821359681174, 0.19809841071795356),
    c64::new(0.9798771036995176, 0.19960175762113097),
    c64::new(0.9795697656854405, 0.2011046348420919),
    c64::new(0.979260122649082, 0.20260703884442113),
    c64::new(0.9789481753190622, 0.20410896609281687),
    c64::new(0.9786339244294232, 0.20561041305309924),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.9779985149345571, 0.20861185197826349),
    c64::new(0.9776773578245099, 0.2101118368804696),
    c64::new(0.9773539001452001, 0.21161132736922755),
    c64::new(0.9770281426577544, 0.21311031991609136),
    c64::new(0.9767000861287118, 0.21460881099378676),
    c64::new(0.9763697313300211, 0.21610679707621952),
    c64::new(0.976037079039039, 0.21760427463848364),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.975364885116657, 0.2205976901088735),
    c64::new(0.9750253450669941, 0.2220936209732035),
    c64::new(0.9746835106885107, 0.22358902922979),
    c64::new(0.9743393827855759, 0.22508391135979283),
    c64::new(0.9739929621679558, 0.22657826384561),
    c64::new(0.973644249650812, 0.22807208317088573),
    c64::new(0.9732932460546982, 0.22956536582051887),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.9725843689347322, 0.23255030703877524),
    c64::new(0.9722264970789363, 0.23404195858354343),
    c64::new(0.9718663374802794, 0.2355330594049755),
    c64::new(0.9715038909862518, 0.2370236059943672),
    c64::new(0.9711391584497251, 0.23851359484431842),
    c64::new(0.9707721407289504, 0.2400030224487415),
    c64::new(0.9704028386875555, 0.24149188530286933),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9696573851242924, 0.24446790274782415),
    c64::new(0.9692812353565485, 0.2459550503357946),
    c64::new(0.9689028047764289, 0.24744161916777327),
    c64::new(0.9685220942744173, 0.24892760574572015),
    c64::new(0.9681391047463624, 0.2504130065729652),
    c64::new(0.9677538370934755, 0.25189781815421697),
    c64::new(0.9673662922223285, 0.25338203699557016),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9665843744783331, 0.2563486824899429),
    c64::new(0.9661900034454125, 0.257831102162159),
    c64::new(0.9657933588740837, 0.25931291513288623),
    c64::new(0.9653944416976894, 0.2607941179152755),
    c64::new(0.9649932528549203, 0.2622747070239136),
    c64::new(0.9645897932898128, 0.26375467897483135),
    c64::new(0.9641840639517458, 0.2652340302855118),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.963365799780954, 0.2681908570634032),
    c64::new(0.9629532668736839, 0.2696683255729151),
    c64::new(0.9625384680443592, 0.271145159526808),
    c64::new(0.9621214042690416, 0.272621355449949),
    c64::new(0.9617020765291225, 0.2740969098687064),
    c64::new(0.9612804858113206, 0.27557181931095814),
    c64::new(0.9608566331076797, 0.2770460803060999),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.960002145737666, 0.2799926430802732),
    c64::new(0.9595715130819845, 0.28146493792575794),
    c64::new(0.9591386224618419, 0.2829365704570554),
    c64::new(0.9587034748958716, 0.2844075372112719),
    c64::new(0.9582660714080177, 0.2858778347270806),
    c64::new(0.9578264130275329, 0.2873474595447295),
    c64::new(0.9573845007889759, 0.2888164082060495),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9564939189023951, 0.29175226323498926),
    c64::new(0.9560452513499964, 0.29321916269425863),
    c64::new(0.9555943341307711, 0.2946853721805143),
    c64::new(0.9551411683057707, 0.2961508882436238),
    c64::new(0.9546857549413383, 0.2976157074350862),
    c64::new(0.9542280951091057, 0.2990798263080405),
    c64::new(0.9537681898859903, 0.30054324141727345),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9528416476011987, 0.3034679465720113),
    c64::new(0.9523750127197659, 0.3049292297354024),
    c64::new(0.9519061368079323, 0.3063897953708609),
    c64::new(0.9514350209690083, 0.30784964004153487),
    c64::new(0.9509616663115751, 0.3093087603122687),
    c64::new(0.9504860739494817, 0.3107671527496115),
    c64::new(0.950008245001843, 0.3122248139218249),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9490458818527006, 0.31513792875252244),
    c64::new(0.9485613499157303, 0.31659337555616585),
    c64::new(0.9480745859222762, 0.31804807738501495),
    c64::new(0.9475855910177411, 0.3195020308160157),
    c64::new(0.9470943663527772, 0.3209552324278752),
    c64::new(0.9466009130832835, 0.32240767880106985),
    c64::new(0.9461052323704034, 0.32385936651785285),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9451071932852606, 0.32676045232013173),
    c64::new(0.9446048372614803, 0.3282098435790925),
    c64::new(0.9441002584912727, 0.3296584625285875),
    c64::new(0.9435934581619604, 0.33110630575987643),
    c64::new(0.9430844374660935, 0.3325533698660442),
    c64::new(0.9425731976014469, 0.3339996514420094),
    c64::new(0.9420597397710173, 0.3354451470845316),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.9410261750508893, 0.3383337669655411),
    c64::new(0.9405060705932683, 0.33977688440682685),
    c64::new(0.939983753034014, 0.34121920232028236),
    c64::new(0.9394592236021899, 0.3426607173119944),
    c64::new(0.9389324835320645, 0.3441014259899388),
    c64::new(0.9384035340631081, 0.3455413249639891),
    c64::new(0.9378723764399899, 0.3469804108459237),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.9368034417359216, 0.3498561297901349),
    c64::new(0.9362656671702783, 0.3512927560855671),
    c64::new(0.9357256894810804, 0.3527285557552107),
    c64::new(0.9351835099389476, 0.35416352542049034),
    c64::new(0.9346391298196808, 0.35559766170478385),
    c64::new(0.934092550404259, 0.35703096123343),
    c64::new(0.9335437729788362, 0.35846342063373654),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9324396292684624, 0.3613258055684543),
    c64::new(0.9318842655816681, 0.3627557243673972),
    c64::new(0.9313267090811804, 0.3641847895670799),
    c64::new(0.9307669610789837, 0.36561299780477385),
    c64::new(0.9302050228922191, 0.3670403457197672),
    c64::new(0.9296408958431812, 0.3684668299533723),
    c64::new(0.9290745812593157, 0.3698924471489341),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9279353948226179, 0.37274106700951576),
    c64::new(0.9273625256504011, 0.37416406297145793),
    c64::new(0.9267874743045817, 0.3755861784892172),
    c64::new(0.9262102421383114, 0.37700741021641826),
    c64::new(0.9256308305098727, 0.37842775480876556),
    c64::new(0.9250492407826776, 0.37984720892405116),
    c64::new(0.9244654743252626, 0.3812657692221624),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9232914167195276, 0.38410019501693504),
    c64::new(0.9227011283338786, 0.38551605384391885),
    c64::new(0.9221086687433451, 0.3869310055143886),
    c64::new(0.9215140393420419, 0.38834504669882625),
    c64::new(0.9209172415291895, 0.3897581740698564),
    c64::new(0.9203182767091106, 0.39117038430225387),
    c64::new(0.9197171462912274, 0.39258167407295147),
    c64::new(0.9191138516900578, 0.3939920400610481),
    c64::new(0.9185083943252123, 0.39540147894781635),
    c64::new(0.9179007756213905, 0.3968099874167103),
    c64::new(0.9172909970083779, 0.39821756215337356),
    c64::new(0.9166790599210427, 0.3996241998456468),
    c64::new(0.9160649657993317, 0.4010298971835756),
    c64::new(0.9154487160882678, 0.40243465085941843),
    c64::new(0.9148303122379462, 0.4038384575676541),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9135870479452508, 0.40664321687036903),
    c64::new(0.9129621904283982, 0.4080441628649787),
    c64::new(0.9123351846233227, 0.4094441486922576),
    c64::new(0.9117060320054299, 0.4108431710579039),
    c64::new(0.9110747340551764, 0.4122412266698829),
    c64::new(0.9104412922580672, 0.41363831223843456),
    c64::new(0.9098057081046522, 0.41503442447608163),
    c64::new(0.9091679830905224, 0.41642956009763715),
    c64::new(0.9085281187163061, 0.41782371582021227),
    c64::new(0.9078861164876663, 0.4192168883632239),
    c64::new(0.9072419779152958, 0.4206090744484025),
    c64::new(0.9065957045149153, 0.4220002707997997),
    c64::new(0.9059472978072685, 0.42339047414379605),
    c64::new(0.9052967593181188, 0.4247796812091088),
    c64::new(0.9046440905782462, 0.4261678887267996),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.9033323684945118, 0.4289412920553295),
    c64::new(0.9026733182372588, 0.4303264813400826),
    c64::new(0.9020121439024932, 0.43171065802505726),
    c64::new(0.901348847046022, 0.43309381885315196),
    c64::new(0.900683429228647, 0.43447596056965565),
    c64::new(0.9000158920161603, 0.4358570799222555),
    c64::new(0.8993462369793416, 0.4372371736610441),
    c64::new(0.8986744656939538, 0.43861623853852766),
    c64::new(0.8980005797407399, 0.43999427130963326),
    c64::new(0.8973245807054183, 0.44137126873171667),
    c64::new(0.8966464701786803, 0.44274722756457),
    c64::new(0.8959662497561852, 0.4441221445704292),
    c64::new(0.8952839210385576, 0.44549601651398174),
    c64::new(0.8945994856313827, 0.44686884016237416),
    c64::new(0.8939129451452033, 0.4482406122852199),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.8925335554027646, 0.45098098904510386),
    c64::new(0.8918407093923427, 0.4523495872337709),
    c64::new(0.8911457647945832, 0.45371712100016387),
    c64::new(0.8904487232447579, 0.45508358712634384),
    c64::new(0.8897495863830729, 0.4564489823968839),
    c64::new(0.8890483558546646, 0.45781330359887723),
    c64::new(0.8883450333095964, 0.4591765475219441),
    c64::new(0.8876396204028539, 0.46053871095824),
    c64::new(0.8869321187943422, 0.46189979070246273),
    c64::new(0.8862225301488806, 0.46325978355186015),
    c64::new(0.8855108561362, 0.4646186863062378),
    c64::new(0.8847970984309378, 0.4659764957679662),
    c64::new(0.884081258712635, 0.46733320874198847),
    c64::new(0.8833633386657316, 0.46868882203582796),
    c64::new(0.8826433399795628, 0.4700433324595956),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.881197113471222, 0.4727490319503428),
    c64::new(0.8804708890521608, 0.47410021465054997),
    c64::new(0.8797425928000474, 0.47545028174715587),
    c64::new(0.8790122264286335, 0.4767992300633221),
    c64::new(0.8782797916565416, 0.478147056424843),
    c64::new(0.8775452902072614, 0.479493757660153),
    c64::new(0.8768087238091457, 0.48083933060033396),
    c64::new(0.8760700941954066, 0.4821837720791227),
    c64::new(0.8753294031041109, 0.48352707893291874),
    c64::new(0.8745866522781762, 0.48486924800079106),
    c64::new(0.8738418434653669, 0.4862102761244864),
    c64::new(0.8730949784182901, 0.487550160148436),
    c64::new(0.8723460588943915, 0.48888889691976317),
    c64::new(0.871595086655951, 0.49022648328829116),
    c64::new(0.8708420634700789, 0.4915629161065499),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8693298713486068, 0.4942323085159597),
    c64::new(0.8685707059713409, 0.49556526182577254),
    c64::new(0.8678094967633033, 0.49689704902265447),
    c64::new(0.8670462455156926, 0.49822766697278187),
    c64::new(0.866280954024513, 0.49955711254508184),
    c64::new(0.8655136240905691, 0.5008853826112407),
    c64::new(0.8647442575194624, 0.5022124740457108),
    c64::new(0.8639728561215867, 0.5035383837257176),
    c64::new(0.8631994217121242, 0.5048631085312676),
    c64::new(0.8624239561110405, 0.5061866453451552),
    c64::new(0.8616464611430813, 0.5075089910529709),
    c64::new(0.8608669386377673, 0.508830142543107),
    c64::new(0.8600853904293901, 0.5101500967067668),
    c64::new(0.8593018183570085, 0.5114688504379703),
    c64::new(0.8585162242644429, 0.512786400633563),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8569389774178288, 0.5154178780194629),
    c64::new(0.8561473283751945, 0.5167317990176499),
    c64::new(0.855353664735196, 0.5180445040959993),
    c64::new(0.8545579883654005, 0.5193559901655896),
    c64::new(0.8537603011381114, 0.5206662541403672),
    c64::new(0.8529606049303636, 0.5219752929371544),
    c64::new(0.8521589016239198, 0.5232831034756564),
    c64::new(0.8513551931052652, 0.524589682678469),
    c64::new(0.8505494812656035, 0.5258950274710846),
    c64::new(0.8497417680008525, 0.5271991347819014),
    c64::new(0.8489320552116396, 0.5285020015422285),
    c64::new(0.8481203448032972, 0.5298036246862946),
    c64::new(0.8473066386858583, 0.531104001151255),
    c64::new(0.8464909387740521, 0.5324031278771979),
    c64::new(0.8456732469872991, 0.533701001807153),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.8440318954900664, 0.5362929790659632),
    c64::new(0.8432082396418454, 0.5375870762956454),
    c64::new(0.8423825996431858, 0.5388799085310084),
    c64::new(0.8415549774368984, 0.5401714727298929),
    c64::new(0.8407253749704581, 0.5414617658531234),
    c64::new(0.8398937941959995, 0.5427507848645159),
    c64::new(0.8390602370703127, 0.5440385267308838),
    c64::new(0.838224705554838, 0.5453249884220465),
    c64::new(0.8373872016156619, 0.5466101669108349),
    c64::new(0.836547727223512, 0.5478940591731002),
    c64::new(0.8357062843537526, 0.5491766621877197),
    c64::new(0.83486287498638, 0.5504579729366048),
    c64::new(0.8340175011060181, 0.5517379884047073),
    c64::new(0.8331701647019132, 0.5530167055800275),
    c64::new(0.8323208677679297, 0.55429412145362),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.8306164003088463, 0.5568450372751601),
    c64::new(0.829761233794523, 0.5581185312205561),
    c64::new(0.8289041147718649, 0.5593907118591361),
    c64::new(0.8280450452577558, 0.560661576197336),
    c64::new(0.8271840272736691, 0.5619311212446895),
    c64::new(0.8263210628456635, 0.5631993440138341),
    c64::new(0.8254561540043776, 0.5644662415205195),
    c64::new(0.8245893027850253, 0.5657318107836131),
    c64::new(0.8237205112273914, 0.5669960488251087),
    c64::new(0.8228497813758264, 0.5682589526701315),
    c64::new(0.8219771152792416, 0.5695205193469471),
    c64::new(0.8211025149911046, 0.5707807458869673),
    c64::new(0.8202259825694347, 0.572039629324757),
    c64::new(0.819347520076797, 0.5732971666980422),
    c64::new(0.8184671295802987, 0.5745533550477158),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.8167005728668278, 0.5770616728556794),
    c64::new(0.8158144108067338, 0.5783137964116556),
    c64::new(0.8149263290565266, 0.5795645591394056),
    c64::new(0.8140363297059484, 0.5808139580957645),
    c64::new(0.8131444148492536, 0.5820619903407754),
    c64::new(0.8122505865852039, 0.5833086529376983),
    c64::new(0.8113548470170637, 0.5845539429530153),
    c64::new(0.8104571982525948, 0.5857978574564389),
    c64::new(0.8095576424040513, 0.587040393520918),
    c64::new(0.808656181588175, 0.5882815482226452),
    c64::new(0.8077528179261904, 0.5895213186410639),
    c64::new(0.8068475535437993, 0.5907597018588742),
    c64::new(0.8059403905711763, 0.591996694962041),
    c64::new(0.8050313311429637, 0.5932322950397998),
    c64::new(0.8041203773982657, 0.5944664991846644),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.8022927955381157, 0.5969307080621965),
    c64::new(0.8013761717231402, 0.5981607069963424),
    c64::new(0.8004576621926228, 0.5993892984005645),
    c64::new(0.799537269107905, 0.600616479383869),
    c64::new(0.7986149946347609, 0.60184224705858),
    c64::new(0.7976908409433912, 0.6030665985403482),
    c64::new(0.7967648102084188, 0.604289530948156),
    c64::new(0.7958369046088836, 0.6055110414043255),
    c64::new(0.794907126328237, 0.6067311270345245),
    c64::new(0.7939754775543372, 0.6079497849677736),
    c64::new(0.7930419604794436, 0.6091670123364532),
    c64::new(0.7921065773002124, 0.6103828062763095),
    c64::new(0.7911693302176901, 0.6115971639264619),
    c64::new(0.79023022143731, 0.6128100824294097),
    c64::new(0.7892892531688857, 0.6140215589310385),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.7874017470290313, 0.6164401745308536),
    c64::new(0.7864552135990858, 0.6176473079378039),
    c64::new(0.7855068295640539, 0.6188529879609763),
    c64::new(0.7845565971555752, 0.6200572117632891),
    c64::new(0.7836045186096383, 0.6212599765110876),
    c64::new(0.7826505961665757, 0.62246127937415),
    c64::new(0.7816948320710595, 0.6236611175256945),
    c64::new(0.7807372285720945, 0.6248594881423863),
    c64::new(0.7797777879230146, 0.6260563884043435),
    c64::new(0.778816512381476, 0.6272518154951441),
    c64::new(0.7778534042094531, 0.6284457666018327),
    c64::new(0.7768884656732324, 0.629638238914927),
    c64::new(0.7759216990434077, 0.6308292296284245),
    c64::new(0.7749531065948738, 0.6320187359398091),
    c64::new(0.7739826906068229, 0.6332067550500572),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.7720363971503845, 0.6355783204885561),
    c64::new(0.7710605242618138, 0.6367618612362842),
    c64::new(0.7700828369933479, 0.637943903621844),
    c64::new(0.7691033376455797, 0.6391244448637757),
    c64::new(0.7681220285233654, 0.6403034821841517),
    c64::new(0.7671389119358204, 0.6414810128085832),
    c64::new(0.7661539901963129, 0.6426570339662269),
    c64::new(0.765167265622459, 0.6438315428897914),
    c64::new(0.7641787405361167, 0.6450045368155439),
    c64::new(0.7631884172633814, 0.6461760129833163),
    c64::new(0.7621962981345789, 0.6473459686365121),
    c64::new(0.7612023854842618, 0.6485144010221124),
    c64::new(0.7602066816512024, 0.6496813073906832),
    c64::new(0.759209188978388, 0.650846684996381),
    c64::new(0.7582099098130153, 0.6520105310969595),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7562060014143945, 0.6543336178318004),
    c64::new(0.7552013768965365, 0.6554928529996153),
    c64::new(0.7541949753168892, 0.656650545729429),
    c64::new(0.7531867990436125, 0.6578066932970786),
    c64::new(0.7521768504490427, 0.6589612929820373),
    c64::new(0.7511651319096864, 0.6601143420674205),
    c64::new(0.7501516458062151, 0.6612658378399923),
    c64::new(0.7491363945234594, 0.6624157775901718),
    c64::new(0.7481193804504036, 0.6635641586120398),
    c64::new(0.7471006059801801, 0.6647109782033448),
    c64::new(0.7460800735100638, 0.6658562336655097),
    c64::new(0.745057785441466, 0.6669999223036375),
    c64::new(0.7440337441799293, 0.6681420414265185),
    c64::new(0.7430079521351217, 0.669282588346636),
    c64::new(0.7419804117208311, 0.6704215603801731),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.7399200954595162, 0.6726947690707729),
    c64::new(0.7388873244606151, 0.673829000378756),
    c64::new(0.737852814788466, 0.674961646102012),
    c64::new(0.7368165688773698, 0.6760927035753159),
    c64::new(0.7357785891657136, 0.6772221701371803),
    c64::new(0.7347388780959634, 0.6783500431298615),
    c64::new(0.7336974381146604, 0.679476319899365),
    c64::new(0.7326542716724128, 0.680600997795453),
    c64::new(0.7316093812238926, 0.6817240741716497),
    c64::new(0.7305627692278276, 0.6828455463852481),
    c64::new(0.729514438146997, 0.6839654117973155),
    c64::new(0.7284643904482252, 0.6850836677727004),
    c64::new(0.7274126286023758, 0.6862003116800386),
    c64::new(0.726359155084346, 0.687315340891759),
    c64::new(0.7253039723730608, 0.6884287527840904),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7231884893065275, 0.6906507141345346),
    c64::new(0.7221281939292153, 0.6917592583641577),
    c64::new(0.7210661993145081, 0.6928661748174246),
    c64::new(0.7200025079613817, 0.6939714608896539),
    c64::new(0.7189371223728045, 0.6950751139800009),
    c64::new(0.7178700450557317, 0.696177131491463),
    c64::new(0.7168012785210995, 0.6972775108308865),
    c64::new(0.7157308252838186, 0.6983762494089729),
    c64::new(0.7146586878627691, 0.6994733446402838),
    c64::new(0.7135848687807935, 0.7005687939432483),
    c64::new(0.7125093705646923, 0.7016625947401685),
    c64::new(0.7114321957452164, 0.7027547444572253),
    c64::new(0.7103533468570624, 0.7038452405244849),
    c64::new(0.7092728264388657, 0.7049340803759049),
    c64::new(0.7081906370331954, 0.7060212614493397),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.7060212614493399, 0.7081906370331953),
    c64::new(0.704934080375905, 0.7092728264388656),
    c64::new(0.703845240524485, 0.7103533468570623),
    c64::new(0.7027547444572254, 0.7114321957452163),
    c64::new(0.7016625947401686, 0.7125093705646923),
    c64::new(0.7005687939432484, 0.7135848687807935),
    c64::new(0.6994733446402839, 0.714658687862769),
    c64::new(0.6983762494089729, 0.7157308252838186),
    c64::new(0.6972775108308866, 0.7168012785210994),
    c64::new(0.6961771314914631, 0.7178700450557316),
    c64::new(0.6950751139800009, 0.7189371223728044),
    c64::new(0.693971460889654, 0.7200025079613817),
    c64::new(0.6928661748174247, 0.721066199314508),
    c64::new(0.6917592583641579, 0.7221281939292152),
    c64::new(0.6906507141345347, 0.7231884893065272),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6884287527840905, 0.7253039723730607),
    c64::new(0.6873153408917592, 0.7263591550843459),
    c64::new(0.6862003116800387, 0.7274126286023757),
    c64::new(0.6850836677727005, 0.7284643904482252),
    c64::new(0.6839654117973155, 0.7295144381469969),
    c64::new(0.6828455463852481, 0.7305627692278276),
    c64::new(0.6817240741716498, 0.7316093812238925),
    c64::new(0.6806009977954531, 0.7326542716724127),
    c64::new(0.6794763198993651, 0.7336974381146603),
    c64::new(0.6783500431298616, 0.7347388780959634),
    c64::new(0.6772221701371804, 0.7357785891657135),
    c64::new(0.676092703575316, 0.7368165688773698),
    c64::new(0.674961646102012, 0.737852814788466),
    c64::new(0.6738290003787561, 0.7388873244606151),
    c64::new(0.672694769070773, 0.7399200954595161),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6704215603801732, 0.741980411720831),
    c64::new(0.6692825883466361, 0.7430079521351216),
    c64::new(0.6681420414265186, 0.7440337441799292),
    c64::new(0.6669999223036376, 0.745057785441466),
    c64::new(0.6658562336655097, 0.7460800735100637),
    c64::new(0.6647109782033449, 0.7471006059801801),
    c64::new(0.6635641586120399, 0.7481193804504035),
    c64::new(0.6624157775901718, 0.7491363945234593),
    c64::new(0.6612658378399923, 0.750151645806215),
    c64::new(0.6601143420674206, 0.7511651319096864),
    c64::new(0.6589612929820373, 0.7521768504490427),
    c64::new(0.6578066932970787, 0.7531867990436124),
    c64::new(0.656650545729429, 0.7541949753168892),
    c64::new(0.6554928529996155, 0.7552013768965364),
    c64::new(0.6543336178318006, 0.7562060014143945),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.6520105310969596, 0.7582099098130153),
    c64::new(0.650846684996381, 0.759209188978388),
    c64::new(0.6496813073906833, 0.7602066816512023),
    c64::new(0.6485144010221126, 0.7612023854842618),
    c64::new(0.6473459686365122, 0.7621962981345789),
    c64::new(0.6461760129833164, 0.7631884172633813),
    c64::new(0.645004536815544, 0.7641787405361167),
    c64::new(0.6438315428897915, 0.7651672656224588),
    c64::new(0.642657033966227, 0.7661539901963128),
    c64::new(0.6414810128085832, 0.7671389119358203),
    c64::new(0.6403034821841518, 0.7681220285233653),
    c64::new(0.6391244448637758, 0.7691033376455796),
    c64::new(0.6379439036218442, 0.7700828369933479),
    c64::new(0.6367618612362843, 0.7710605242618137),
    c64::new(0.6355783204885562, 0.7720363971503844),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6332067550500573, 0.7739826906068228),
    c64::new(0.6320187359398091, 0.7749531065948738),
    c64::new(0.6308292296284246, 0.7759216990434076),
    c64::new(0.6296382389149271, 0.7768884656732324),
    c64::new(0.6284457666018327, 0.777853404209453),
    c64::new(0.6272518154951442, 0.7788165123814759),
    c64::new(0.6260563884043436, 0.7797777879230144),
    c64::new(0.6248594881423865, 0.7807372285720944),
    c64::new(0.6236611175256946, 0.7816948320710594),
    c64::new(0.6224612793741501, 0.7826505961665756),
    c64::new(0.6212599765110877, 0.7836045186096382),
    c64::new(0.6200572117632892, 0.7845565971555752),
    c64::new(0.6188529879609764, 0.7855068295640539),
    c64::new(0.617647307937804, 0.7864552135990858),
    c64::new(0.6164401745308536, 0.7874017470290313),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.6140215589310385, 0.7892892531688855),
    c64::new(0.6128100824294098, 0.79023022143731),
    c64::new(0.611597163926462, 0.7911693302176901),
    c64::new(0.6103828062763095, 0.7921065773002123),
    c64::new(0.6091670123364533, 0.7930419604794436),
    c64::new(0.6079497849677737, 0.7939754775543371),
    c64::new(0.6067311270345246, 0.794907126328237),
    c64::new(0.6055110414043255, 0.7958369046088835),
    c64::new(0.6042895309481561, 0.7967648102084187),
    c64::new(0.6030665985403483, 0.797690840943391),
    c64::new(0.6018422470585801, 0.7986149946347608),
    c64::new(0.600616479383869, 0.799537269107905),
    c64::new(0.5993892984005647, 0.8004576621926227),
    c64::new(0.5981607069963424, 0.8013761717231401),
    c64::new(0.5969307080621965, 0.8022927955381156),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5944664991846645, 0.8041203773982657),
    c64::new(0.5932322950397999, 0.8050313311429635),
    c64::new(0.591996694962041, 0.8059403905711763),
    c64::new(0.5907597018588743, 0.8068475535437992),
    c64::new(0.5895213186410639, 0.8077528179261902),
    c64::new(0.5882815482226453, 0.808656181588175),
    c64::new(0.5870403935209181, 0.8095576424040511),
    c64::new(0.5857978574564389, 0.8104571982525948),
    c64::new(0.5845539429530154, 0.8113548470170637),
    c64::new(0.5833086529376984, 0.8122505865852039),
    c64::new(0.5820619903407755, 0.8131444148492535),
    c64::new(0.5808139580957646, 0.8140363297059483),
    c64::new(0.5795645591394057, 0.8149263290565265),
    c64::new(0.5783137964116557, 0.8158144108067338),
    c64::new(0.5770616728556796, 0.8167005728668277),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5745533550477159, 0.8184671295802987),
    c64::new(0.5732971666980423, 0.8193475200767969),
    c64::new(0.5720396293247572, 0.8202259825694347),
    c64::new(0.5707807458869674, 0.8211025149911046),
    c64::new(0.5695205193469473, 0.8219771152792414),
    c64::new(0.5682589526701316, 0.8228497813758263),
    c64::new(0.5669960488251087, 0.8237205112273913),
    c64::new(0.5657318107836132, 0.8245893027850253),
    c64::new(0.5644662415205195, 0.8254561540043774),
    c64::new(0.5631993440138342, 0.8263210628456634),
    c64::new(0.5619311212446895, 0.827184027273669),
    c64::new(0.560661576197336, 0.8280450452577557),
    c64::new(0.5593907118591361, 0.8289041147718648),
    c64::new(0.5581185312205562, 0.829761233794523),
    c64::new(0.5568450372751602, 0.8306164003088462),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5542941214536201, 0.8323208677679297),
    c64::new(0.5530167055800276, 0.8331701647019132),
    c64::new(0.5517379884047074, 0.834017501106018),
    c64::new(0.5504579729366049, 0.83486287498638),
    c64::new(0.5491766621877198, 0.8357062843537525),
    c64::new(0.5478940591731003, 0.8365477272235119),
    c64::new(0.546610166910835, 0.8373872016156618),
    c64::new(0.5453249884220466, 0.838224705554838),
    c64::new(0.5440385267308839, 0.8390602370703126),
    c64::new(0.542750784864516, 0.8398937941959994),
    c64::new(0.5414617658531236, 0.840725374970458),
    c64::new(0.540171472729893, 0.8415549774368983),
    c64::new(0.5388799085310084, 0.8423825996431858),
    c64::new(0.5375870762956455, 0.8432082396418454),
    c64::new(0.5362929790659632, 0.8440318954900664),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.533701001807153, 0.8456732469872991),
    c64::new(0.532403127877198, 0.846490938774052),
    c64::new(0.5311040011512551, 0.8473066386858583),
    c64::new(0.5298036246862948, 0.8481203448032971),
    c64::new(0.5285020015422285, 0.8489320552116396),
    c64::new(0.5271991347819014, 0.8497417680008524),
    c64::new(0.5258950274710849, 0.8505494812656034),
    c64::new(0.524589682678469, 0.8513551931052652),
    c64::new(0.5232831034756564, 0.8521589016239198),
    c64::new(0.5219752929371544, 0.8529606049303636),
    c64::new(0.5206662541403673, 0.8537603011381113),
    c64::new(0.5193559901655896, 0.8545579883654005),
    c64::new(0.5180445040959994, 0.855353664735196),
    c64::new(0.51673179901765, 0.8561473283751944),
    c64::new(0.5154178780194631, 0.8569389774178287),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.5127864006335631, 0.8585162242644427),
    c64::new(0.5114688504379705, 0.8593018183570083),
    c64::new(0.5101500967067668, 0.8600853904293901),
    c64::new(0.5088301425431071, 0.8608669386377672),
    c64::new(0.507508991052971, 0.8616464611430813),
    c64::new(0.5061866453451555, 0.8624239561110405),
    c64::new(0.5048631085312676, 0.8631994217121242),
    c64::new(0.5035383837257176, 0.8639728561215867),
    c64::new(0.5022124740457109, 0.8647442575194624),
    c64::new(0.5008853826112409, 0.865513624090569),
    c64::new(0.4995571125450819, 0.866280954024513),
    c64::new(0.4982276669727819, 0.8670462455156926),
    c64::new(0.4968970490226547, 0.8678094967633032),
    c64::new(0.49556526182577254, 0.8685707059713409),
    c64::new(0.4942323085159598, 0.8693298713486067),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.4915629161065501, 0.8708420634700789),
    c64::new(0.49022648328829116, 0.871595086655951),
    c64::new(0.4888888969197632, 0.8723460588943914),
    c64::new(0.4875501601484361, 0.87309497841829),
    c64::new(0.4862102761244866, 0.8738418434653668),
    c64::new(0.4848692480007911, 0.8745866522781761),
    c64::new(0.4835270789329188, 0.8753294031041108),
    c64::new(0.4821837720791229, 0.8760700941954065),
    c64::new(0.48083933060033396, 0.8768087238091457),
    c64::new(0.47949375766015306, 0.8775452902072612),
    c64::new(0.4781470564248431, 0.8782797916565415),
    c64::new(0.47679923006332225, 0.8790122264286334),
    c64::new(0.47545028174715587, 0.8797425928000474),
    c64::new(0.47410021465055, 0.8804708890521608),
    c64::new(0.4727490319503429, 0.881197113471222),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.4700433324595956, 0.8826433399795628),
    c64::new(0.46868882203582796, 0.8833633386657316),
    c64::new(0.4673332087419885, 0.884081258712635),
    c64::new(0.4659764957679661, 0.8847970984309378),
    c64::new(0.4646186863062378, 0.8855108561362),
    c64::new(0.46325978355186026, 0.8862225301488806),
    c64::new(0.46189979070246284, 0.8869321187943421),
    c64::new(0.46053871095824, 0.8876396204028539),
    c64::new(0.45917654752194415, 0.8883450333095964),
    c64::new(0.4578133035988773, 0.8890483558546645),
    c64::new(0.45644898239688386, 0.8897495863830729),
    c64::new(0.45508358712634384, 0.8904487232447579),
    c64::new(0.4537171210001639, 0.8911457647945832),
    c64::new(0.452349587233771, 0.8918407093923427),
    c64::new(0.4509809890451038, 0.8925335554027647),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.44824061228522, 0.8939129451452033),
    c64::new(0.4468688401623743, 0.8945994856313826),
    c64::new(0.44549601651398174, 0.8952839210385576),
    c64::new(0.44412214457042926, 0.8959662497561851),
    c64::new(0.44274722756457013, 0.8966464701786802),
    c64::new(0.4413712687317166, 0.8973245807054183),
    c64::new(0.43999427130963326, 0.8980005797407399),
    c64::new(0.4386162385385277, 0.8986744656939538),
    c64::new(0.4372371736610442, 0.8993462369793415),
    c64::new(0.4358570799222555, 0.9000158920161603),
    c64::new(0.4344759605696557, 0.9006834292286469),
    c64::new(0.433093818853152, 0.901348847046022),
    c64::new(0.43171065802505737, 0.9020121439024931),
    c64::new(0.4303264813400826, 0.9026733182372588),
    c64::new(0.42894129205532955, 0.9033323684945118),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.42616788872679956, 0.9046440905782462),
    c64::new(0.4247796812091088, 0.9052967593181188),
    c64::new(0.4233904741437961, 0.9059472978072685),
    c64::new(0.4220002707997998, 0.9065957045149153),
    c64::new(0.4206090744484025, 0.9072419779152958),
    c64::new(0.41921688836322396, 0.9078861164876662),
    c64::new(0.4178237158202124, 0.9085281187163061),
    c64::new(0.4164295600976373, 0.9091679830905224),
    c64::new(0.41503442447608163, 0.9098057081046522),
    c64::new(0.41363831223843456, 0.9104412922580671),
    c64::new(0.412241226669883, 0.9110747340551762),
    c64::new(0.4108431710579039, 0.9117060320054299),
    c64::new(0.4094441486922576, 0.9123351846233227),
    c64::new(0.40804416286497874, 0.9129621904283981),
    c64::new(0.40664321687036914, 0.9135870479452508),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.40383845756765413, 0.9148303122379462),
    c64::new(0.40243465085941854, 0.9154487160882678),
    c64::new(0.4010298971835758, 0.9160649657993316),
    c64::new(0.3996241998456468, 0.9166790599210427),
    c64::new(0.3982175621533736, 0.9172909970083779),
    c64::new(0.3968099874167104, 0.9179007756213904),
    c64::new(0.3954014789478163, 0.9185083943252123),
    c64::new(0.3939920400610481, 0.9191138516900578),
    c64::new(0.3925816740729515, 0.9197171462912274),
    c64::new(0.391170384302254, 0.9203182767091105),
    c64::new(0.3897581740698564, 0.9209172415291895),
    c64::new(0.3883450466988263, 0.9215140393420419),
    c64::new(0.3869310055143887, 0.9221086687433451),
    c64::new(0.385516053843919, 0.9227011283338785),
    c64::new(0.38410019501693504, 0.9232914167195276),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3812657692221625, 0.9244654743252626),
    c64::new(0.3798472089240511, 0.9250492407826776),
    c64::new(0.37842775480876556, 0.9256308305098727),
    c64::new(0.3770074102164183, 0.9262102421383114),
    c64::new(0.3755861784892173, 0.9267874743045817),
    c64::new(0.37416406297145793, 0.9273625256504011),
    c64::new(0.3727410670095158, 0.9279353948226179),
    c64::new(0.3713171939518376, 0.9285060804732155),
    c64::new(0.3698924471489342, 0.9290745812593157),
    c64::new(0.3684668299533723, 0.9296408958431812),
    c64::new(0.36704034571976724, 0.9302050228922191),
    c64::new(0.36561299780477396, 0.9307669610789837),
    c64::new(0.36418478956707984, 0.9313267090811805),
    c64::new(0.3627557243673972, 0.9318842655816681),
    c64::new(0.36132580556845434, 0.9324396292684624),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.35846342063373654, 0.9335437729788362),
    c64::new(0.35703096123343003, 0.9340925504042589),
    c64::new(0.35559766170478396, 0.9346391298196808),
    c64::new(0.3541635254204905, 0.9351835099389475),
    c64::new(0.3527285557552107, 0.9357256894810804),
    c64::new(0.35129275608556715, 0.9362656671702783),
    c64::new(0.34985612979013503, 0.9368034417359216),
    c64::new(0.3484186802494345, 0.937339011912575),
    c64::new(0.3469804108459237, 0.9378723764399899),
    c64::new(0.34554132496398915, 0.9384035340631081),
    c64::new(0.344101425989939, 0.9389324835320645),
    c64::new(0.3426607173119944, 0.9394592236021899),
    c64::new(0.3412192023202824, 0.9399837530340139),
    c64::new(0.33977688440682696, 0.9405060705932683),
    c64::new(0.3383337669655413, 0.9410261750508893),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.33544514708453166, 0.9420597397710173),
    c64::new(0.3339996514420095, 0.9425731976014469),
    c64::new(0.33255336986604417, 0.9430844374660935),
    c64::new(0.33110630575987643, 0.9435934581619604),
    c64::new(0.32965846252858755, 0.9441002584912727),
    c64::new(0.32820984357909266, 0.9446048372614803),
    c64::new(0.32676045232013173, 0.9451071932852606),
    c64::new(0.325310292162263, 0.9456073253805213),
    c64::new(0.32385936651785296, 0.9461052323704033),
    c64::new(0.32240767880106996, 0.9466009130832835),
    c64::new(0.3209552324278752, 0.9470943663527772),
    c64::new(0.31950203081601575, 0.9475855910177411),
    c64::new(0.31804807738501506, 0.9480745859222762),
    c64::new(0.31659337555616585, 0.9485613499157303),
    c64::new(0.31513792875252244, 0.9490458818527006),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.31222481392182505, 0.950008245001843),
    c64::new(0.3107671527496115, 0.9504860739494817),
    c64::new(0.3093087603122687, 0.9509616663115751),
    c64::new(0.307849640041535, 0.9514350209690083),
    c64::new(0.3063897953708611, 0.9519061368079322),
    c64::new(0.3049292297354024, 0.9523750127197659),
    c64::new(0.30346794657201137, 0.9528416476011987),
    c64::new(0.3020059493192282, 0.9533060403541938),
    c64::new(0.3005432414172734, 0.9537681898859903),
    c64::new(0.2990798263080405, 0.9542280951091057),
    c64::new(0.2976157074350863, 0.9546857549413383),
    c64::new(0.29615088824362396, 0.9551411683057707),
    c64::new(0.2946853721805143, 0.9555943341307711),
    c64::new(0.2932191626942587, 0.9560452513499964),
    c64::new(0.2917522632349894, 0.9564939189023951),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.2888164082060495, 0.9573845007889759),
    c64::new(0.28734745954472957, 0.9578264130275329),
    c64::new(0.2858778347270807, 0.9582660714080177),
    c64::new(0.2844075372112718, 0.9587034748958716),
    c64::new(0.2829365704570554, 0.9591386224618419),
    c64::new(0.28146493792575805, 0.9595715130819845),
    c64::new(0.2799926430802734, 0.9600021457376658),
    c64::new(0.27851968938505306, 0.9604305194155658),
    c64::new(0.27704608030609995, 0.9608566331076797),
    c64::new(0.27557181931095825, 0.9612804858113206),
    c64::new(0.27409690986870633, 0.9617020765291225),
    c64::new(0.272621355449949, 0.9621214042690416),
    c64::new(0.27114515952680807, 0.9625384680443592),
    c64::new(0.2696683255729152, 0.9629532668736839),
    c64::new(0.2681908570634031, 0.963365799780954),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.2652340302855119, 0.9641840639517457),
    c64::new(0.2637546789748315, 0.9645897932898126),
    c64::new(0.2622747070239136, 0.9649932528549203),
    c64::new(0.26079411791527557, 0.9653944416976894),
    c64::new(0.25931291513288635, 0.9657933588740837),
    c64::new(0.25783110216215893, 0.9661900034454126),
    c64::new(0.2563486824899429, 0.9665843744783331),
    c64::new(0.2548656596045146, 0.9669764710448521),
    c64::new(0.25338203699557027, 0.9673662922223285),
    c64::new(0.2518978181542169, 0.9677538370934755),
    c64::new(0.2504130065729653, 0.9681391047463624),
    c64::new(0.24892760574572026, 0.9685220942744173),
    c64::new(0.24744161916777344, 0.9689028047764289),
    c64::new(0.2459550503357946, 0.9692812353565485),
    c64::new(0.2444679027478242, 0.9696573851242924),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.24149188530286927, 0.9704028386875555),
    c64::new(0.2400030224487415, 0.9707721407289504),
    c64::new(0.2385135948443185, 0.9711391584497251),
    c64::new(0.23702360599436734, 0.9715038909862518),
    c64::new(0.23553305940497546, 0.9718663374802794),
    c64::new(0.23404195858354346, 0.9722264970789363),
    c64::new(0.23255030703877533, 0.9725843689347322),
    c64::new(0.23105810828067125, 0.9729399522055601),
    c64::new(0.22956536582051887, 0.9732932460546982),
    c64::new(0.2280720831708858, 0.9736442496508119),
    c64::new(0.2265782638456101, 0.9739929621679558),
    c64::new(0.22508391135979278, 0.9743393827855759),
    c64::new(0.22358902922979, 0.9746835106885107),
    c64::new(0.2220936209732036, 0.9750253450669941),
    c64::new(0.22059769010887365, 0.9753648851166569),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.21760427463848367, 0.976037079039039),
    c64::new(0.2161067970762196, 0.9763697313300211),
    c64::new(0.21460881099378692, 0.9767000861287118),
    c64::new(0.21311031991609136, 0.9770281426577544),
    c64::new(0.2116113273692276, 0.9773539001452),
    c64::new(0.21011183688046972, 0.9776773578245099),
    c64::new(0.20861185197826343, 0.9779985149345571),
    c64::new(0.20711137619221856, 0.9783173707196277),
    c64::new(0.20561041305309932, 0.9786339244294231),
    c64::new(0.204108966092817, 0.9789481753190622),
    c64::new(0.2026070388444211, 0.979260122649082),
    c64::new(0.20110463484209193, 0.9795697656854405),
    c64::new(0.19960175762113105, 0.9798771036995176),
    c64::new(0.19809841071795373, 0.9801821359681173),
    c64::new(0.19659459767008022, 0.9804848617734694),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.19358558729580372, 0.9810833911504866),
    c64::new(0.19208039704989238, 0.9813791933137546),
    c64::new(0.19057475482025277, 0.9816726861969831),
    c64::new(0.18906866414980628, 0.9819638691095552),
    c64::new(0.18756212858252974, 0.9822527413662894),
    c64::new(0.1860551516634466, 0.9825393022874412),
    c64::new(0.18454773693861964, 0.9828235511987052),
    c64::new(0.18303988795514103, 0.9831054874312163),
    c64::new(0.18153160826112513, 0.9833851103215512),
    c64::new(0.18002290140569951, 0.9836624192117303),
    c64::new(0.17851377093899756, 0.9839374134492189),
    c64::new(0.17700422041214886, 0.984210092386929),
    c64::new(0.17549425337727137, 0.9844804553832209),
    c64::new(0.17398387338746385, 0.9847485018019042),
    c64::new(0.17247308399679603, 0.9850142310122398),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.16945029123396793, 0.9855387353121761),
    c64::new(0.1679382949747312, 0.9857975091675674),
    c64::new(0.16642590354046422, 0.9860539633461954),
    c64::new(0.16491312048997006, 0.9863080972445987),
    c64::new(0.16339994938297323, 0.9865599102647754),
    c64::new(0.16188639378011188, 0.9868094018141854),
    c64::new(0.1603724572429284, 0.987056571305751),
    c64::new(0.1588581433338614, 0.9873014181578584),
    c64::new(0.15734345561623828, 0.9875439417943592),
    c64::new(0.15582839765426532, 0.9877841416445722),
    c64::new(0.15431297301302024, 0.9880220171432835),
    c64::new(0.1527971852584434, 0.9882575677307495),
    c64::new(0.15128103795733025, 0.9884907928526966),
    c64::new(0.14976453467732162, 0.9887216919603238),
    c64::new(0.1482476789868962, 0.988950264510303),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.14521292465284752, 0.9894004277913804),
    c64::new(0.14369503315029458, 0.9896220174632008),
    c64::new(0.142176803519448, 0.9898412784588205),
    c64::new(0.14065823933284924, 0.9900582102622971),
    c64::new(0.13913934416382628, 0.9902728123631691),
    c64::new(0.13762012158648618, 0.990485084256457),
    c64::new(0.13610057517570617, 0.9906950254426646),
    c64::new(0.13458070850712622, 0.99090263542778),
    c64::new(0.13306052515713918, 0.9911079137232768),
    c64::new(0.13154002870288325, 0.9913108598461154),
    c64::new(0.13001922272223335, 0.9915114733187439),
    c64::new(0.12849811079379322, 0.9917097536690995),
    c64::new(0.12697669649688598, 0.9919057004306093),
    c64::new(0.1254549834115462, 0.9920993131421918),
    c64::new(0.12393297511851219, 0.9922905913482574),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.12088808723577722, 0.992666142448948),
    c64::new(0.11936521481099134, 0.9928504144598651),
    c64::new(0.11784206150832502, 0.9930323501978514),
    c64::new(0.11631863091190486, 0.9932119492347945),
    c64::new(0.11479492660651025, 0.9933892111480807),
    c64::new(0.11327095217756435, 0.9935641355205953),
    c64::new(0.11174671121112666, 0.9937367219407246),
    c64::new(0.11022220729388318, 0.9939069700023561),
    c64::new(0.10869744401313867, 0.9940748793048794),
    c64::new(0.10717242495680887, 0.9942404494531879),
    c64::new(0.1056471537134107, 0.9944036800576791),
    c64::new(0.10412163387205473, 0.9945645707342554),
    c64::new(0.10259586902243627, 0.9947231211043257),
    c64::new(0.10106986275482786, 0.9948793307948056),
    c64::new(0.09954361866006943, 0.9950331994381186),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.09649043135525259, 0.9953339121404823),
    c64::new(0.09496349532963906, 0.9954807554919269),
    c64::new(0.09343633584574791, 0.9956252563809943),
    c64::new(0.0919089564971327, 0.9957674144676598),
    c64::new(0.09038136087786501, 0.9959072294174117),
    c64::new(0.08885355258252468, 0.996044700901252),
    c64::new(0.08732553520619221, 0.9961798285956969),
    c64::new(0.08579731234443988, 0.996312612182778),
    c64::new(0.08426888759332411, 0.9964430513500426),
    c64::new(0.0827402645493758, 0.9965711457905548),
    c64::new(0.08121144680959239, 0.9966968952028961),
    c64::new(0.07968243797143013, 0.9968202992911657),
    c64::new(0.0781532416327943, 0.9969413577649822),
    c64::new(0.07662386139203162, 0.997060070339483),
    c64::new(0.07509430084792128, 0.9971764367353262),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.07203465324688942, 0.9974021299012753),
    c64::new(0.07050457338961401, 0.9975114561403035),
    c64::new(0.06897432762826673, 0.9976184351385196),
    c64::new(0.0674439195636641, 0.9977230666441916),
    c64::new(0.06591335279700392, 0.9978253504111116),
    c64::new(0.06438263092985741, 0.997925286198596),
    c64::new(0.06285175756416142, 0.9980228737714862),
    c64::new(0.061320736302208655, 0.9981181129001492),
    c64::new(0.05978957074664001, 0.9982110033604782),
    c64::new(0.05825826450043573, 0.9983015449338929),
    c64::new(0.05672682116690778, 0.9983897374073402),
    c64::new(0.05519524434969003, 0.9984755805732948),
    c64::new(0.05366353765273068, 0.9985590742297593),
    c64::new(0.05213170468028332, 0.9986402181802653),
    c64::new(0.05059974903689934, 0.9987190122338729),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.04753548415695926, 0.9988695499142836),
    c64::new(0.046003182130914644, 0.9989412931868569),
    c64::new(0.044470771854938744, 0.9990106858540734),
    c64::new(0.04293825693494096, 0.9990777277526454),
    c64::new(0.04140564097707672, 0.9991424187248169),
    c64::new(0.039872927587739845, 0.9992047586183639),
    c64::new(0.03834012037355279, 0.9992647472865944),
    c64::new(0.03680722294135899, 0.9993223845883495),
    c64::new(0.03527423889821395, 0.9993776703880028),
    c64::new(0.03374117185137764, 0.9994306045554617),
    c64::new(0.032208025408304704, 0.999481186966167),
    c64::new(0.030674803176636584, 0.9995294175010931),
    c64::new(0.029141508764193743, 0.9995752960467492),
    c64::new(0.02760814577896582, 0.9996188224951786),
    c64::new(0.02607471782910404, 0.9996599967439592),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(0.02300768146883941, 0.9997352882605617),
    c64::new(0.02147408027546961, 0.9997694053512153),
    c64::new(0.0199404285515146, 0.9998011698878843),
    c64::new(0.01840672990580482, 0.9998305817958234),
    c64::new(0.016872987947281773, 0.9998576410058239),
    c64::new(0.01533920628498822, 0.9998823474542126),
    c64::new(0.013805388528060349, 0.9999047010828529),
    c64::new(0.012271538285719944, 0.9999247018391445),
    c64::new(0.01073765916726457, 0.9999423496760239),
    c64::new(0.00920375478205996, 0.9999576445519639),
    c64::new(0.007669828739531077, 0.9999705864309741),
    c64::new(0.006135884649154516, 0.9999811752826011),
    c64::new(0.004601926120448672, 0.9999894110819284),
    c64::new(0.003067956762966138, 0.9999952938095762),
    c64::new(0.001533980186284766, 0.9999988234517019),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.0015339801862846436, 0.9999988234517019),
    c64::new(-0.0030679567629660156, 0.9999952938095762),
    c64::new(-0.004601926120448549, 0.9999894110819284),
    c64::new(-0.006135884649154393, 0.9999811752826011),
    c64::new(-0.007669828739530954, 0.9999705864309741),
    c64::new(-0.009203754782059839, 0.9999576445519639),
    c64::new(-0.010737659167264449, 0.9999423496760239),
    c64::new(-0.012271538285719823, 0.9999247018391445),
    c64::new(-0.013805388528060228, 0.9999047010828529),
    c64::new(-0.015339206284988098, 0.9998823474542126),
    c64::new(-0.016872987947281648, 0.9998576410058239),
    c64::new(-0.018406729905804695, 0.9998305817958234),
    c64::new(-0.019940428551514476, 0.9998011698878843),
    c64::new(-0.021474080275469484, 0.9997694053512153),
    c64::new(-0.023007681468839285, 0.9997352882605617),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.026074717829103915, 0.9996599967439592),
    c64::new(-0.027608145778965695, 0.9996188224951786),
    c64::new(-0.029141508764193618, 0.9995752960467492),
    c64::new(-0.03067480317663646, 0.9995294175010931),
    c64::new(-0.03220802540830458, 0.999481186966167),
    c64::new(-0.03374117185137752, 0.9994306045554617),
    c64::new(-0.03527423889821382, 0.9993776703880028),
    c64::new(-0.036807222941358866, 0.9993223845883495),
    c64::new(-0.038340120373552666, 0.9992647472865944),
    c64::new(-0.03987292758773972, 0.9992047586183639),
    c64::new(-0.041405640977076594, 0.9991424187248169),
    c64::new(-0.042938256934940834, 0.9990777277526454),
    c64::new(-0.04447077185493862, 0.9990106858540734),
    c64::new(-0.04600318213091452, 0.9989412931868569),
    c64::new(-0.047535484156959136, 0.9988695499142836),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.05059974903689921, 0.9987190122338729),
    c64::new(-0.05213170468028319, 0.9986402181802653),
    c64::new(-0.053663537652730554, 0.9985590742297593),
    c64::new(-0.05519524434968991, 0.9984755805732948),
    c64::new(-0.05672682116690766, 0.9983897374073402),
    c64::new(-0.05825826450043561, 0.9983015449338929),
    c64::new(-0.05978957074663988, 0.9982110033604782),
    c64::new(-0.06132073630220853, 0.9981181129001492),
    c64::new(-0.06285175756416131, 0.9980228737714862),
    c64::new(-0.0643826309298573, 0.997925286198596),
    c64::new(-0.0659133527970038, 0.9978253504111116),
    c64::new(-0.067443919563664, 0.9977230666441916),
    c64::new(-0.06897432762826662, 0.9976184351385196),
    c64::new(-0.0705045733896139, 0.9975114561403035),
    c64::new(-0.0720346532468893, 0.9974021299012753),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.07509430084792117, 0.9971764367353262),
    c64::new(-0.0766238613920315, 0.997060070339483),
    c64::new(-0.07815324163279419, 0.9969413577649822),
    c64::new(-0.07968243797143001, 0.9968202992911657),
    c64::new(-0.08121144680959227, 0.9966968952028961),
    c64::new(-0.08274026454937569, 0.9965711457905548),
    c64::new(-0.084268887593324, 0.9964430513500426),
    c64::new(-0.08579731234443977, 0.996312612182778),
    c64::new(-0.0873255352061921, 0.9961798285956969),
    c64::new(-0.08885355258252457, 0.996044700901252),
    c64::new(-0.0903813608778649, 0.9959072294174117),
    c64::new(-0.09190895649713259, 0.9957674144676598),
    c64::new(-0.0934363358457478, 0.9956252563809943),
    c64::new(-0.09496349532963895, 0.9954807554919269),
    c64::new(-0.09649043135525248, 0.9953339121404823),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.09954361866006932, 0.9950331994381186),
    c64::new(-0.10106986275482775, 0.9948793307948056),
    c64::new(-0.10259586902243616, 0.9947231211043257),
    c64::new(-0.10412163387205461, 0.9945645707342554),
    c64::new(-0.10564715371341059, 0.9944036800576791),
    c64::new(-0.10717242495680876, 0.9942404494531879),
    c64::new(-0.10869744401313856, 0.9940748793048795),
    c64::new(-0.11022220729388307, 0.9939069700023561),
    c64::new(-0.11174671121112655, 0.9937367219407246),
    c64::new(-0.11327095217756424, 0.9935641355205953),
    c64::new(-0.11479492660651014, 0.9933892111480807),
    c64::new(-0.11631863091190475, 0.9932119492347945),
    c64::new(-0.11784206150832491, 0.9930323501978514),
    c64::new(-0.11936521481099123, 0.9928504144598651),
    c64::new(-0.12088808723577711, 0.992666142448948),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.12393297511851208, 0.9922905913482574),
    c64::new(-0.1254549834115461, 0.9920993131421918),
    c64::new(-0.12697669649688587, 0.9919057004306093),
    c64::new(-0.1284981107937931, 0.9917097536690995),
    c64::new(-0.13001922272223324, 0.991511473318744),
    c64::new(-0.13154002870288317, 0.9913108598461154),
    c64::new(-0.13306052515713906, 0.9911079137232769),
    c64::new(-0.1345807085071261, 0.99090263542778),
    c64::new(-0.13610057517570606, 0.9906950254426646),
    c64::new(-0.13762012158648607, 0.990485084256457),
    c64::new(-0.13913934416382617, 0.9902728123631691),
    c64::new(-0.14065823933284913, 0.9900582102622971),
    c64::new(-0.1421768035194479, 0.9898412784588205),
    c64::new(-0.14369503315029447, 0.9896220174632009),
    c64::new(-0.1452129246528474, 0.9894004277913804),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.1482476789868961, 0.988950264510303),
    c64::new(-0.1497645346773215, 0.9887216919603238),
    c64::new(-0.15128103795733014, 0.9884907928526967),
    c64::new(-0.1527971852584433, 0.9882575677307495),
    c64::new(-0.15431297301302013, 0.9880220171432835),
    c64::new(-0.1558283976542652, 0.9877841416445722),
    c64::new(-0.15734345561623816, 0.9875439417943593),
    c64::new(-0.15885814333386128, 0.9873014181578584),
    c64::new(-0.16037245724292828, 0.987056571305751),
    c64::new(-0.16188639378011177, 0.9868094018141855),
    c64::new(-0.16339994938297311, 0.9865599102647755),
    c64::new(-0.16491312048996995, 0.9863080972445987),
    c64::new(-0.1664259035404641, 0.9860539633461954),
    c64::new(-0.1679382949747311, 0.9857975091675675),
    c64::new(-0.16945029123396782, 0.9855387353121761),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.17247308399679592, 0.9850142310122398),
    c64::new(-0.17398387338746374, 0.9847485018019042),
    c64::new(-0.17549425337727126, 0.9844804553832209),
    c64::new(-0.17700422041214875, 0.984210092386929),
    c64::new(-0.17851377093899745, 0.9839374134492189),
    c64::new(-0.1800229014056994, 0.9836624192117303),
    c64::new(-0.18153160826112502, 0.9833851103215512),
    c64::new(-0.18303988795514095, 0.9831054874312163),
    c64::new(-0.18454773693861953, 0.9828235511987053),
    c64::new(-0.18605515166344652, 0.9825393022874412),
    c64::new(-0.18756212858252963, 0.9822527413662894),
    c64::new(-0.18906866414980616, 0.9819638691095552),
    c64::new(-0.19057475482025266, 0.9816726861969831),
    c64::new(-0.19208039704989227, 0.9813791933137546),
    c64::new(-0.1935855872958036, 0.9810833911504867),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.1965945976700801, 0.9804848617734694),
    c64::new(-0.19809841071795362, 0.9801821359681174),
    c64::new(-0.19960175762113094, 0.9798771036995176),
    c64::new(-0.20110463484209182, 0.9795697656854405),
    c64::new(-0.202607038844421, 0.9792601226490821),
    c64::new(-0.2041089660928169, 0.9789481753190622),
    c64::new(-0.2056104130530992, 0.9786339244294232),
    c64::new(-0.20711137619221845, 0.9783173707196277),
    c64::new(-0.20861185197826332, 0.9779985149345571),
    c64::new(-0.2101118368804696, 0.9776773578245099),
    c64::new(-0.2116113273692275, 0.9773539001452001),
    c64::new(-0.21311031991609125, 0.9770281426577544),
    c64::new(-0.2146088109937868, 0.9767000861287118),
    c64::new(-0.2161067970762195, 0.9763697313300211),
    c64::new(-0.21760427463848356, 0.9760370790390391),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.22059769010887353, 0.975364885116657),
    c64::new(-0.22209362097320348, 0.9750253450669941),
    c64::new(-0.2235890292297899, 0.9746835106885107),
    c64::new(-0.22508391135979267, 0.9743393827855759),
    c64::new(-0.22657826384561, 0.9739929621679558),
    c64::new(-0.22807208317088568, 0.973644249650812),
    c64::new(-0.22956536582051876, 0.9732932460546982),
    c64::new(-0.23105810828067117, 0.9729399522055602),
    c64::new(-0.23255030703877522, 0.9725843689347322),
    c64::new(-0.23404195858354335, 0.9722264970789363),
    c64::new(-0.23553305940497535, 0.9718663374802794),
    c64::new(-0.23702360599436723, 0.9715038909862518),
    c64::new(-0.2385135948443184, 0.9711391584497251),
    c64::new(-0.2400030224487414, 0.9707721407289504),
    c64::new(-0.2414918853028692, 0.9704028386875555),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.2444679027478241, 0.9696573851242924),
    c64::new(-0.24595505033579448, 0.9692812353565485),
    c64::new(-0.24744161916777332, 0.9689028047764289),
    c64::new(-0.24892760574572015, 0.9685220942744173),
    c64::new(-0.25041300657296517, 0.9681391047463624),
    c64::new(-0.2518978181542168, 0.9677538370934755),
    c64::new(-0.25338203699557016, 0.9673662922223285),
    c64::new(-0.2548656596045145, 0.9669764710448521),
    c64::new(-0.2563486824899428, 0.9665843744783331),
    c64::new(-0.2578311021621588, 0.9661900034454126),
    c64::new(-0.25931291513288623, 0.9657933588740837),
    c64::new(-0.26079411791527546, 0.9653944416976894),
    c64::new(-0.2622747070239135, 0.9649932528549204),
    c64::new(-0.2637546789748314, 0.9645897932898128),
    c64::new(-0.2652340302855118, 0.9641840639517458),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.26819085706340307, 0.963365799780954),
    c64::new(-0.2696683255729151, 0.9629532668736839),
    c64::new(-0.27114515952680796, 0.9625384680443592),
    c64::new(-0.27262135544994887, 0.9621214042690416),
    c64::new(-0.2740969098687062, 0.9617020765291225),
    c64::new(-0.27557181931095814, 0.9612804858113206),
    c64::new(-0.27704608030609984, 0.9608566331076797),
    c64::new(-0.27851968938505295, 0.9604305194155659),
    c64::new(-0.2799926430802733, 0.9600021457376658),
    c64::new(-0.28146493792575794, 0.9595715130819845),
    c64::new(-0.2829365704570553, 0.959138622461842),
    c64::new(-0.2844075372112717, 0.9587034748958716),
    c64::new(-0.2858778347270806, 0.9582660714080177),
    c64::new(-0.28734745954472946, 0.9578264130275329),
    c64::new(-0.28881640820604937, 0.957384500788976),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.29175226323498926, 0.9564939189023951),
    c64::new(-0.2932191626942586, 0.9560452513499965),
    c64::new(-0.2946853721805142, 0.9555943341307711),
    c64::new(-0.29615088824362384, 0.9551411683057707),
    c64::new(-0.2976157074350862, 0.9546857549413383),
    c64::new(-0.29907982630804036, 0.9542280951091057),
    c64::new(-0.3005432414172733, 0.9537681898859903),
    c64::new(-0.3020059493192281, 0.9533060403541939),
    c64::new(-0.30346794657201126, 0.9528416476011987),
    c64::new(-0.3049292297354023, 0.9523750127197659),
    c64::new(-0.306389795370861, 0.9519061368079323),
    c64::new(-0.30784964004153487, 0.9514350209690083),
    c64::new(-0.30930876031226867, 0.9509616663115751),
    c64::new(-0.31076715274961136, 0.9504860739494818),
    c64::new(-0.31222481392182494, 0.950008245001843),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.31513792875252233, 0.9490458818527006),
    c64::new(-0.31659337555616573, 0.9485613499157304),
    c64::new(-0.31804807738501495, 0.9480745859222762),
    c64::new(-0.31950203081601564, 0.9475855910177412),
    c64::new(-0.3209552324278751, 0.9470943663527772),
    c64::new(-0.3224076788010699, 0.9466009130832835),
    c64::new(-0.32385936651785285, 0.9461052323704034),
    c64::new(-0.32531029216226287, 0.9456073253805214),
    c64::new(-0.3267604523201317, 0.9451071932852606),
    c64::new(-0.32820984357909255, 0.9446048372614803),
    c64::new(-0.32965846252858744, 0.9441002584912727),
    c64::new(-0.3311063057598763, 0.9435934581619604),
    c64::new(-0.3325533698660441, 0.9430844374660935),
    c64::new(-0.3339996514420094, 0.9425731976014469),
    c64::new(-0.33544514708453155, 0.9420597397710174),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.3383337669655412, 0.9410261750508893),
    c64::new(-0.33977688440682685, 0.9405060705932683),
    c64::new(-0.3412192023202823, 0.939983753034014),
    c64::new(-0.34266071731199427, 0.9394592236021899),
    c64::new(-0.34410142598993887, 0.9389324835320645),
    c64::new(-0.34554132496398904, 0.9384035340631082),
    c64::new(-0.34698041084592357, 0.9378723764399899),
    c64::new(-0.3484186802494344, 0.937339011912575),
    c64::new(-0.3498561297901349, 0.9368034417359216),
    c64::new(-0.35129275608556704, 0.9362656671702783),
    c64::new(-0.3527285557552106, 0.9357256894810804),
    c64::new(-0.3541635254204904, 0.9351835099389476),
    c64::new(-0.35559766170478385, 0.9346391298196808),
    c64::new(-0.3570309612334299, 0.934092550404259),
    c64::new(-0.35846342063373643, 0.9335437729788363),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.36132580556845423, 0.9324396292684625),
    c64::new(-0.3627557243673971, 0.9318842655816681),
    c64::new(-0.3641847895670797, 0.9313267090811805),
    c64::new(-0.36561299780477385, 0.9307669610789837),
    c64::new(-0.3670403457197671, 0.9302050228922191),
    c64::new(-0.3684668299533722, 0.9296408958431813),
    c64::new(-0.36989244714893416, 0.9290745812593157),
    c64::new(-0.37131719395183754, 0.9285060804732156),
    c64::new(-0.3727410670095157, 0.9279353948226179),
    c64::new(-0.3741640629714579, 0.9273625256504011),
    c64::new(-0.3755861784892172, 0.9267874743045817),
    c64::new(-0.3770074102164182, 0.9262102421383114),
    c64::new(-0.3784277548087655, 0.9256308305098728),
    c64::new(-0.379847208924051, 0.9250492407826777),
    c64::new(-0.3812657692221624, 0.9244654743252626),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.38410019501693493, 0.9232914167195276),
    c64::new(-0.3855160538439189, 0.9227011283338785),
    c64::new(-0.3869310055143886, 0.9221086687433451),
    c64::new(-0.3883450466988262, 0.921514039342042),
    c64::new(-0.3897581740698563, 0.9209172415291895),
    c64::new(-0.39117038430225387, 0.9203182767091106),
    c64::new(-0.39258167407295147, 0.9197171462912274),
    c64::new(-0.393992040061048, 0.9191138516900578),
    c64::new(-0.3954014789478162, 0.9185083943252123),
    c64::new(-0.3968099874167103, 0.9179007756213905),
    c64::new(-0.39821756215337356, 0.917290997008378),
    c64::new(-0.39962419984564673, 0.9166790599210427),
    c64::new(-0.4010298971835757, 0.9160649657993317),
    c64::new(-0.40243465085941843, 0.9154487160882678),
    c64::new(-0.403838457567654, 0.9148303122379462),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.40664321687036903, 0.9135870479452508),
    c64::new(-0.40804416286497863, 0.9129621904283982),
    c64::new(-0.40944414869225754, 0.9123351846233229),
    c64::new(-0.4108431710579038, 0.9117060320054299),
    c64::new(-0.4122412266698829, 0.9110747340551764),
    c64::new(-0.4136383122384345, 0.9104412922580672),
    c64::new(-0.4150344244760815, 0.9098057081046523),
    c64::new(-0.416429560097637, 0.9091679830905225),
    c64::new(-0.41782371582021227, 0.9085281187163061),
    c64::new(-0.41921688836322407, 0.9078861164876662),
    c64::new(-0.4206090744484024, 0.9072419779152959),
    c64::new(-0.4220002707997997, 0.9065957045149153),
    c64::new(-0.42339047414379577, 0.9059472978072686),
    c64::new(-0.42477968120910875, 0.9052967593181188),
    c64::new(-0.42616788872679967, 0.9046440905782462),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.42894129205532944, 0.9033323684945118),
    c64::new(-0.4303264813400827, 0.9026733182372588),
    c64::new(-0.4317106580250571, 0.9020121439024933),
    c64::new(-0.43309381885315196, 0.901348847046022),
    c64::new(-0.4344759605696558, 0.9006834292286469),
    c64::new(-0.43585707992225536, 0.9000158920161603),
    c64::new(-0.4372371736610441, 0.8993462369793416),
    c64::new(-0.4386162385385274, 0.8986744656939539),
    c64::new(-0.43999427130963314, 0.8980005797407399),
    c64::new(-0.4413712687317167, 0.8973245807054182),
    c64::new(-0.4427472275645698, 0.8966464701786803),
    c64::new(-0.44412214457042914, 0.8959662497561852),
    c64::new(-0.44549601651398185, 0.8952839210385575),
    c64::new(-0.446868840162374, 0.8945994856313828),
    c64::new(-0.4482406122852199, 0.8939129451452033),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.45098098904510375, 0.8925335554027647),
    c64::new(-0.4523495872337709, 0.8918407093923427),
    c64::new(-0.45371712100016365, 0.8911457647945834),
    c64::new(-0.4550835871263437, 0.890448723244758),
    c64::new(-0.456448982396884, 0.8897495863830728),
    c64::new(-0.457813303598877, 0.8890483558546647),
    c64::new(-0.45917654752194403, 0.8883450333095964),
    c64::new(-0.4605387109582401, 0.8876396204028539),
    c64::new(-0.46189979070246256, 0.8869321187943423),
    c64::new(-0.46325978355186015, 0.8862225301488806),
    c64::new(-0.4646186863062379, 0.8855108561361998),
    c64::new(-0.465976495767966, 0.8847970984309379),
    c64::new(-0.46733320874198847, 0.884081258712635),
    c64::new(-0.4686888220358277, 0.8833633386657317),
    c64::new(-0.4700433324595955, 0.8826433399795628),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.47274903195034257, 0.8811971134712222),
    c64::new(-0.47410021465054997, 0.8804708890521609),
    c64::new(-0.475450281747156, 0.8797425928000474),
    c64::new(-0.476799230063322, 0.8790122264286335),
    c64::new(-0.478147056424843, 0.8782797916565416),
    c64::new(-0.47949375766015306, 0.8775452902072612),
    c64::new(-0.4808393306003338, 0.8768087238091458),
    c64::new(-0.4821837720791227, 0.8760700941954066),
    c64::new(-0.4835270789329184, 0.875329403104111),
    c64::new(-0.48486924800079095, 0.8745866522781762),
    c64::new(-0.4862102761244864, 0.8738418434653669),
    c64::new(-0.4875501601484357, 0.8730949784182902),
    c64::new(-0.48888889691976306, 0.8723460588943915),
    c64::new(-0.49022648328829116, 0.871595086655951),
    c64::new(-0.49156291610654973, 0.8708420634700791),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.4942323085159598, 0.8693298713486067),
    c64::new(-0.4955652618257723, 0.868570705971341),
    c64::new(-0.49689704902265447, 0.8678094967633033),
    c64::new(-0.49822766697278154, 0.8670462455156929),
    c64::new(-0.4995571125450817, 0.8662809540245131),
    c64::new(-0.5008853826112407, 0.8655136240905691),
    c64::new(-0.5022124740457106, 0.8647442575194625),
    c64::new(-0.5035383837257175, 0.8639728561215868),
    c64::new(-0.5048631085312676, 0.8631994217121242),
    c64::new(-0.5061866453451551, 0.8624239561110407),
    c64::new(-0.5075089910529708, 0.8616464611430814),
    c64::new(-0.5088301425431071, 0.8608669386377672),
    c64::new(-0.5101500967067666, 0.8600853904293903),
    c64::new(-0.5114688504379703, 0.8593018183570085),
    c64::new(-0.5127864006335627, 0.858516224264443),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5154178780194629, 0.8569389774178288),
    c64::new(-0.5167317990176497, 0.8561473283751946),
    c64::new(-0.5180445040959992, 0.855353664735196),
    c64::new(-0.5193559901655896, 0.8545579883654005),
    c64::new(-0.5206662541403669, 0.8537603011381115),
    c64::new(-0.5219752929371543, 0.8529606049303637),
    c64::new(-0.5232831034756564, 0.8521589016239198),
    c64::new(-0.5245896826784687, 0.8513551931052652),
    c64::new(-0.5258950274710846, 0.8505494812656035),
    c64::new(-0.5271991347819011, 0.8497417680008527),
    c64::new(-0.5285020015422283, 0.8489320552116397),
    c64::new(-0.5298036246862946, 0.8481203448032972),
    c64::new(-0.5311040011512547, 0.8473066386858585),
    c64::new(-0.5324031278771978, 0.8464909387740521),
    c64::new(-0.533701001807153, 0.8456732469872991),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.5362929790659631, 0.8440318954900665),
    c64::new(-0.5375870762956455, 0.8432082396418454),
    c64::new(-0.5388799085310082, 0.842382599643186),
    c64::new(-0.5401714727298929, 0.8415549774368984),
    c64::new(-0.5414617658531232, 0.8407253749704582),
    c64::new(-0.5427507848645158, 0.8398937941959996),
    c64::new(-0.5440385267308838, 0.8390602370703127),
    c64::new(-0.5453249884220461, 0.8382247055548382),
    c64::new(-0.5466101669108347, 0.837387201615662),
    c64::new(-0.5478940591731002, 0.836547727223512),
    c64::new(-0.5491766621877195, 0.8357062843537527),
    c64::new(-0.5504579729366047, 0.8348628749863801),
    c64::new(-0.5517379884047074, 0.834017501106018),
    c64::new(-0.5530167055800274, 0.8331701647019133),
    c64::new(-0.55429412145362, 0.8323208677679297),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5568450372751599, 0.8306164003088464),
    c64::new(-0.5581185312205561, 0.829761233794523),
    c64::new(-0.5593907118591358, 0.8289041147718651),
    c64::new(-0.5606615761973359, 0.8280450452577559),
    c64::new(-0.5619311212446895, 0.8271840272736691),
    c64::new(-0.5631993440138339, 0.8263210628456636),
    c64::new(-0.5644662415205194, 0.8254561540043776),
    c64::new(-0.5657318107836132, 0.8245893027850253),
    c64::new(-0.5669960488251085, 0.8237205112273915),
    c64::new(-0.5682589526701315, 0.8228497813758264),
    c64::new(-0.5695205193469473, 0.8219771152792414),
    c64::new(-0.5707807458869671, 0.8211025149911048),
    c64::new(-0.572039629324757, 0.8202259825694347),
    c64::new(-0.573297166698042, 0.8193475200767971),
    c64::new(-0.5745533550477157, 0.8184671295802988),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5770616728556793, 0.816700572866828),
    c64::new(-0.5783137964116555, 0.8158144108067339),
    c64::new(-0.5795645591394057, 0.8149263290565265),
    c64::new(-0.5808139580957643, 0.8140363297059485),
    c64::new(-0.5820619903407754, 0.8131444148492536),
    c64::new(-0.5833086529376984, 0.8122505865852039),
    c64::new(-0.5845539429530151, 0.811354847017064),
    c64::new(-0.5857978574564389, 0.8104571982525948),
    c64::new(-0.5870403935209177, 0.8095576424040515),
    c64::new(-0.5882815482226452, 0.8086561815881751),
    c64::new(-0.5895213186410639, 0.8077528179261904),
    c64::new(-0.590759701858874, 0.8068475535437994),
    c64::new(-0.5919966949620409, 0.8059403905711764),
    c64::new(-0.5932322950397999, 0.8050313311429635),
    c64::new(-0.5944664991846643, 0.8041203773982659),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.5969307080621965, 0.8022927955381156),
    c64::new(-0.5981607069963422, 0.8013761717231404),
    c64::new(-0.5993892984005645, 0.8004576621926228),
    c64::new(-0.6006164793838686, 0.7995372691079052),
    c64::new(-0.6018422470585799, 0.7986149946347609),
    c64::new(-0.6030665985403482, 0.7976908409433912),
    c64::new(-0.6042895309481559, 0.7967648102084189),
    c64::new(-0.6055110414043254, 0.7958369046088836),
    c64::new(-0.6067311270345246, 0.794907126328237),
    c64::new(-0.6079497849677735, 0.7939754775543373),
    c64::new(-0.6091670123364531, 0.7930419604794438),
    c64::new(-0.6103828062763095, 0.7921065773002123),
    c64::new(-0.6115971639264618, 0.7911693302176903),
    c64::new(-0.6128100824294097, 0.79023022143731),
    c64::new(-0.6140215589310382, 0.7892892531688859),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.6164401745308536, 0.7874017470290313),
    c64::new(-0.6176473079378038, 0.7864552135990859),
    c64::new(-0.6188529879609762, 0.785506829564054),
    c64::new(-0.6200572117632892, 0.7845565971555752),
    c64::new(-0.6212599765110874, 0.7836045186096383),
    c64::new(-0.62246127937415, 0.7826505961665757),
    c64::new(-0.6236611175256946, 0.7816948320710594),
    c64::new(-0.6248594881423862, 0.7807372285720946),
    c64::new(-0.6260563884043435, 0.7797777879230146),
    c64::new(-0.6272518154951439, 0.7788165123814762),
    c64::new(-0.6284457666018326, 0.7778534042094531),
    c64::new(-0.629638238914927, 0.7768884656732324),
    c64::new(-0.6308292296284244, 0.7759216990434078),
    c64::new(-0.632018735939809, 0.7749531065948739),
    c64::new(-0.6332067550500573, 0.7739826906068228),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6355783204885561, 0.7720363971503845),
    c64::new(-0.6367618612362843, 0.7710605242618137),
    c64::new(-0.6379439036218439, 0.7700828369933481),
    c64::new(-0.6391244448637757, 0.7691033376455797),
    c64::new(-0.6403034821841515, 0.7681220285233656),
    c64::new(-0.641481012808583, 0.7671389119358205),
    c64::new(-0.6426570339662269, 0.7661539901963129),
    c64::new(-0.6438315428897913, 0.7651672656224591),
    c64::new(-0.6450045368155439, 0.7641787405361168),
    c64::new(-0.6461760129833164, 0.7631884172633813),
    c64::new(-0.647345968636512, 0.7621962981345791),
    c64::new(-0.6485144010221124, 0.7612023854842619),
    c64::new(-0.6496813073906833, 0.7602066816512023),
    c64::new(-0.6508466849963808, 0.7592091889783882),
    c64::new(-0.6520105310969595, 0.7582099098130153),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.6543336178318004, 0.7562060014143945),
    c64::new(-0.6554928529996153, 0.7552013768965365),
    c64::new(-0.6566505457294288, 0.7541949753168893),
    c64::new(-0.6578066932970785, 0.7531867990436125),
    c64::new(-0.6589612929820373, 0.7521768504490427),
    c64::new(-0.6601143420674203, 0.7511651319096866),
    c64::new(-0.6612658378399922, 0.7501516458062151),
    c64::new(-0.6624157775901718, 0.7491363945234593),
    c64::new(-0.6635641586120397, 0.7481193804504037),
    c64::new(-0.6647109782033448, 0.7471006059801801),
    c64::new(-0.6658562336655094, 0.746080073510064),
    c64::new(-0.6669999223036374, 0.7450577854414661),
    c64::new(-0.6681420414265185, 0.7440337441799293),
    c64::new(-0.6692825883466358, 0.7430079521351218),
    c64::new(-0.670421560380173, 0.7419804117208311),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.6726947690707727, 0.7399200954595163),
    c64::new(-0.673829000378756, 0.7388873244606152),
    c64::new(-0.674961646102012, 0.737852814788466),
    c64::new(-0.6760927035753158, 0.73681656887737),
    c64::new(-0.6772221701371803, 0.7357785891657136),
    c64::new(-0.6783500431298612, 0.7347388780959637),
    c64::new(-0.679476319899365, 0.7336974381146604),
    c64::new(-0.680600997795453, 0.7326542716724128),
    c64::new(-0.6817240741716496, 0.7316093812238927),
    c64::new(-0.682845546385248, 0.7305627692278276),
    c64::new(-0.6839654117973155, 0.729514438146997),
    c64::new(-0.6850836677727002, 0.7284643904482253),
    c64::new(-0.6862003116800386, 0.7274126286023759),
    c64::new(-0.6873153408917592, 0.7263591550843459),
    c64::new(-0.6884287527840903, 0.7253039723730609),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.6906507141345344, 0.7231884893065277),
    c64::new(-0.6917592583641576, 0.7221281939292155),
    c64::new(-0.6928661748174246, 0.7210661993145081),
    c64::new(-0.6939714608896538, 0.7200025079613819),
    c64::new(-0.6950751139800008, 0.7189371223728045),
    c64::new(-0.696177131491463, 0.7178700450557317),
    c64::new(-0.6972775108308864, 0.7168012785210996),
    c64::new(-0.6983762494089728, 0.7157308252838187),
    c64::new(-0.6994733446402839, 0.714658687862769),
    c64::new(-0.7005687939432482, 0.7135848687807937),
    c64::new(-0.7016625947401685, 0.7125093705646923),
    c64::new(-0.7027547444572251, 0.7114321957452168),
    c64::new(-0.7038452405244848, 0.7103533468570624),
    c64::new(-0.7049340803759049, 0.7092728264388657),
    c64::new(-0.7060212614493395, 0.7081906370331955),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7081906370331954, 0.7060212614493397),
    c64::new(-0.7092728264388655, 0.7049340803759051),
    c64::new(-0.7103533468570623, 0.703845240524485),
    c64::new(-0.7114321957452165, 0.7027547444572252),
    c64::new(-0.7125093705646923, 0.7016625947401687),
    c64::new(-0.7135848687807935, 0.7005687939432483),
    c64::new(-0.7146586878627689, 0.699473344640284),
    c64::new(-0.7157308252838186, 0.6983762494089729),
    c64::new(-0.7168012785210995, 0.6972775108308865),
    c64::new(-0.7178700450557316, 0.6961771314914631),
    c64::new(-0.7189371223728044, 0.6950751139800009),
    c64::new(-0.7200025079613817, 0.6939714608896539),
    c64::new(-0.721066199314508, 0.6928661748174249),
    c64::new(-0.7221281939292152, 0.6917592583641579),
    c64::new(-0.7231884893065275, 0.6906507141345346),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7253039723730608, 0.6884287527840904),
    c64::new(-0.7263591550843458, 0.6873153408917593),
    c64::new(-0.7274126286023757, 0.6862003116800387),
    c64::new(-0.7284643904482252, 0.6850836677727004),
    c64::new(-0.7295144381469969, 0.6839654117973156),
    c64::new(-0.7305627692278276, 0.6828455463852481),
    c64::new(-0.7316093812238926, 0.6817240741716497),
    c64::new(-0.7326542716724127, 0.6806009977954532),
    c64::new(-0.7336974381146603, 0.6794763198993651),
    c64::new(-0.7347388780959636, 0.6783500431298615),
    c64::new(-0.7357785891657134, 0.6772221701371806),
    c64::new(-0.7368165688773698, 0.6760927035753159),
    c64::new(-0.7378528147884658, 0.6749616461020123),
    c64::new(-0.7388873244606151, 0.6738290003787561),
    c64::new(-0.7399200954595162, 0.6726947690707729),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.741980411720831, 0.6704215603801732),
    c64::new(-0.7430079521351217, 0.669282588346636),
    c64::new(-0.7440337441799292, 0.6681420414265187),
    c64::new(-0.745057785441466, 0.6669999223036376),
    c64::new(-0.7460800735100639, 0.6658562336655096),
    c64::new(-0.74710060598018, 0.664710978203345),
    c64::new(-0.7481193804504036, 0.6635641586120398),
    c64::new(-0.749136394523459, 0.662415777590172),
    c64::new(-0.750151645806215, 0.6612658378399923),
    c64::new(-0.7511651319096864, 0.6601143420674205),
    c64::new(-0.7521768504490426, 0.6589612929820375),
    c64::new(-0.7531867990436124, 0.6578066932970787),
    c64::new(-0.7541949753168892, 0.656650545729429),
    c64::new(-0.7552013768965364, 0.6554928529996156),
    c64::new(-0.7562060014143945, 0.6543336178318006),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.7582099098130152, 0.6520105310969597),
    c64::new(-0.759209188978388, 0.650846684996381),
    c64::new(-0.7602066816512022, 0.6496813073906834),
    c64::new(-0.7612023854842618, 0.6485144010221126),
    c64::new(-0.7621962981345789, 0.6473459686365121),
    c64::new(-0.7631884172633812, 0.6461760129833165),
    c64::new(-0.7641787405361167, 0.645004536815544),
    c64::new(-0.765167265622459, 0.6438315428897914),
    c64::new(-0.7661539901963128, 0.6426570339662271),
    c64::new(-0.7671389119358203, 0.6414810128085832),
    c64::new(-0.7681220285233654, 0.6403034821841516),
    c64::new(-0.7691033376455796, 0.6391244448637758),
    c64::new(-0.7700828369933479, 0.637943903621844),
    c64::new(-0.7710605242618136, 0.6367618612362844),
    c64::new(-0.7720363971503844, 0.6355783204885562),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7739826906068226, 0.6332067550500574),
    c64::new(-0.7749531065948738, 0.6320187359398091),
    c64::new(-0.7759216990434077, 0.6308292296284245),
    c64::new(-0.7768884656732323, 0.6296382389149272),
    c64::new(-0.777853404209453, 0.6284457666018327),
    c64::new(-0.7788165123814761, 0.6272518154951441),
    c64::new(-0.7797777879230143, 0.6260563884043436),
    c64::new(-0.7807372285720945, 0.6248594881423863),
    c64::new(-0.7816948320710593, 0.6236611175256949),
    c64::new(-0.7826505961665756, 0.6224612793741501),
    c64::new(-0.7836045186096383, 0.6212599765110876),
    c64::new(-0.7845565971555751, 0.6200572117632894),
    c64::new(-0.7855068295640539, 0.6188529879609764),
    c64::new(-0.7864552135990858, 0.6176473079378039),
    c64::new(-0.7874017470290313, 0.6164401745308538),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.7892892531688858, 0.6140215589310384),
    c64::new(-0.7902302214373099, 0.6128100824294098),
    c64::new(-0.7911693302176901, 0.6115971639264619),
    c64::new(-0.7921065773002122, 0.6103828062763097),
    c64::new(-0.7930419604794436, 0.6091670123364533),
    c64::new(-0.7939754775543372, 0.6079497849677736),
    c64::new(-0.7949071263282368, 0.6067311270345247),
    c64::new(-0.7958369046088835, 0.6055110414043255),
    c64::new(-0.7967648102084188, 0.604289530948156),
    c64::new(-0.7976908409433909, 0.6030665985403484),
    c64::new(-0.7986149946347608, 0.6018422470585801),
    c64::new(-0.799537269107905, 0.6006164793838689),
    c64::new(-0.8004576621926227, 0.5993892984005647),
    c64::new(-0.8013761717231402, 0.5981607069963424),
    c64::new(-0.8022927955381156, 0.5969307080621967),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8041203773982657, 0.5944664991846644),
    c64::new(-0.8050313311429634, 0.5932322950398),
    c64::new(-0.8059403905711763, 0.591996694962041),
    c64::new(-0.8068475535437993, 0.5907597018588742),
    c64::new(-0.8077528179261902, 0.589521318641064),
    c64::new(-0.808656181588175, 0.5882815482226453),
    c64::new(-0.8095576424040514, 0.5870403935209179),
    c64::new(-0.8104571982525947, 0.585797857456439),
    c64::new(-0.8113548470170637, 0.5845539429530153),
    c64::new(-0.8122505865852037, 0.5833086529376985),
    c64::new(-0.8131444148492535, 0.5820619903407757),
    c64::new(-0.8140363297059484, 0.5808139580957645),
    c64::new(-0.8149263290565264, 0.579564559139406),
    c64::new(-0.8158144108067338, 0.5783137964116557),
    c64::new(-0.8167005728668278, 0.5770616728556794),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.8184671295802987, 0.5745533550477159),
    c64::new(-0.819347520076797, 0.5732971666980421),
    c64::new(-0.8202259825694346, 0.5720396293247573),
    c64::new(-0.8211025149911046, 0.5707807458869673),
    c64::new(-0.8219771152792414, 0.5695205193469475),
    c64::new(-0.8228497813758262, 0.5682589526701317),
    c64::new(-0.8237205112273914, 0.5669960488251087),
    c64::new(-0.8245893027850251, 0.5657318107836135),
    c64::new(-0.8254561540043774, 0.5644662415205195),
    c64::new(-0.8263210628456635, 0.5631993440138341),
    c64::new(-0.827184027273669, 0.5619311212446896),
    c64::new(-0.8280450452577557, 0.560661576197336),
    c64::new(-0.828904114771865, 0.559390711859136),
    c64::new(-0.8297612337945229, 0.5581185312205563),
    c64::new(-0.8306164003088463, 0.5568450372751601),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.8323208677679296, 0.5542941214536202),
    c64::new(-0.8331701647019132, 0.5530167055800275),
    c64::new(-0.834017501106018, 0.5517379884047077),
    c64::new(-0.83486287498638, 0.5504579729366049),
    c64::new(-0.8357062843537526, 0.5491766621877197),
    c64::new(-0.8365477272235119, 0.5478940591731003),
    c64::new(-0.8373872016156618, 0.546610166910835),
    c64::new(-0.8382247055548381, 0.5453249884220464),
    c64::new(-0.8390602370703126, 0.544038526730884),
    c64::new(-0.8398937941959995, 0.5427507848645159),
    c64::new(-0.8407253749704582, 0.5414617658531233),
    c64::new(-0.8415549774368983, 0.540171472729893),
    c64::new(-0.8423825996431858, 0.5388799085310084),
    c64::new(-0.8432082396418453, 0.5375870762956457),
    c64::new(-0.8440318954900664, 0.5362929790659632),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.845673246987299, 0.5337010018071532),
    c64::new(-0.846490938774052, 0.532403127877198),
    c64::new(-0.8473066386858584, 0.5311040011512549),
    c64::new(-0.8481203448032971, 0.5298036246862948),
    c64::new(-0.8489320552116396, 0.5285020015422285),
    c64::new(-0.8497417680008525, 0.5271991347819013),
    c64::new(-0.8505494812656034, 0.5258950274710849),
    c64::new(-0.8513551931052652, 0.524589682678469),
    c64::new(-0.8521589016239197, 0.5232831034756567),
    c64::new(-0.8529606049303636, 0.5219752929371544),
    c64::new(-0.8537603011381114, 0.5206662541403672),
    c64::new(-0.8545579883654004, 0.5193559901655898),
    c64::new(-0.855353664735196, 0.5180445040959994),
    c64::new(-0.8561473283751946, 0.5167317990176498),
    c64::new(-0.8569389774178287, 0.5154178780194631),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8585162242644429, 0.512786400633563),
    c64::new(-0.8593018183570083, 0.5114688504379705),
    c64::new(-0.8600853904293901, 0.5101500967067668),
    c64::new(-0.8608669386377672, 0.5088301425431073),
    c64::new(-0.8616464611430813, 0.507508991052971),
    c64::new(-0.8624239561110405, 0.5061866453451552),
    c64::new(-0.863199421712124, 0.5048631085312677),
    c64::new(-0.8639728561215867, 0.5035383837257176),
    c64::new(-0.8647442575194624, 0.5022124740457107),
    c64::new(-0.865513624090569, 0.5008853826112409),
    c64::new(-0.866280954024513, 0.4995571125450819),
    c64::new(-0.8670462455156926, 0.49822766697278176),
    c64::new(-0.8678094967633032, 0.4968970490226547),
    c64::new(-0.8685707059713409, 0.49556526182577254),
    c64::new(-0.8693298713486066, 0.49423230851595995),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.8708420634700789, 0.4915629161065499),
    c64::new(-0.871595086655951, 0.4902264832882914),
    c64::new(-0.8723460588943914, 0.4888888969197632),
    c64::new(-0.8730949784182902, 0.4875501601484359),
    c64::new(-0.8738418434653668, 0.4862102761244866),
    c64::new(-0.8745866522781761, 0.4848692480007911),
    c64::new(-0.8753294031041109, 0.48352707893291863),
    c64::new(-0.8760700941954065, 0.4821837720791229),
    c64::new(-0.8768087238091457, 0.48083933060033396),
    c64::new(-0.8775452902072611, 0.4794937576601533),
    c64::new(-0.8782797916565415, 0.4781470564248432),
    c64::new(-0.8790122264286335, 0.4767992300633221),
    c64::new(-0.8797425928000473, 0.4754502817471561),
    c64::new(-0.8804708890521608, 0.4741002146505501),
    c64::new(-0.8811971134712221, 0.47274903195034274),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8826433399795628, 0.4700433324595957),
    c64::new(-0.8833633386657317, 0.46868882203582785),
    c64::new(-0.8840812587126349, 0.4673332087419886),
    c64::new(-0.8847970984309378, 0.4659764957679662),
    c64::new(-0.8855108561361998, 0.4646186863062381),
    c64::new(-0.8862225301488805, 0.4632597835518603),
    c64::new(-0.8869321187943422, 0.46189979070246273),
    c64::new(-0.8876396204028538, 0.4605387109582402),
    c64::new(-0.8883450333095962, 0.4591765475219442),
    c64::new(-0.8890483558546646, 0.4578133035988772),
    c64::new(-0.8897495863830727, 0.45644898239688414),
    c64::new(-0.8904487232447579, 0.4550835871263439),
    c64::new(-0.8911457647945833, 0.45371712100016376),
    c64::new(-0.8918407093923426, 0.452349587233771),
    c64::new(-0.8925335554027646, 0.45098098904510386),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.8939129451452033, 0.44824061228522005),
    c64::new(-0.8945994856313827, 0.44686884016237416),
    c64::new(-0.8952839210385575, 0.44549601651398196),
    c64::new(-0.8959662497561851, 0.4441221445704293),
    c64::new(-0.8966464701786803, 0.44274722756456997),
    c64::new(-0.8973245807054182, 0.4413712687317169),
    c64::new(-0.8980005797407398, 0.4399942713096333),
    c64::new(-0.8986744656939539, 0.43861623853852755),
    c64::new(-0.8993462369793415, 0.43723717366104425),
    c64::new(-0.9000158920161603, 0.4358570799222555),
    c64::new(-0.9006834292286467, 0.434475960569656),
    c64::new(-0.9013488470460219, 0.43309381885315207),
    c64::new(-0.9020121439024932, 0.43171065802505726),
    c64::new(-0.9026733182372587, 0.4303264813400829),
    c64::new(-0.9033323684945118, 0.4289412920553296),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9046440905782461, 0.42616788872679984),
    c64::new(-0.9052967593181187, 0.42477968120910886),
    c64::new(-0.9059472978072685, 0.42339047414379594),
    c64::new(-0.9065957045149153, 0.42200027079979985),
    c64::new(-0.9072419779152958, 0.4206090744484025),
    c64::new(-0.907886116487666, 0.41921688836322424),
    c64::new(-0.9085281187163061, 0.41782371582021244),
    c64::new(-0.9091679830905224, 0.41642956009763715),
    c64::new(-0.9098057081046521, 0.41503442447608185),
    c64::new(-0.9104412922580671, 0.4136383122384346),
    c64::new(-0.9110747340551764, 0.41224122666988283),
    c64::new(-0.9117060320054298, 0.41084317105790413),
    c64::new(-0.9123351846233227, 0.40944414869225765),
    c64::new(-0.9129621904283982, 0.4080441628649786),
    c64::new(-0.9135870479452508, 0.4066432168703692),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.914830312237946, 0.4038384575676544),
    c64::new(-0.9154487160882677, 0.40243465085941854),
    c64::new(-0.9160649657993317, 0.4010298971835756),
    c64::new(-0.9166790599210426, 0.39962419984564707),
    c64::new(-0.9172909970083779, 0.39821756215337367),
    c64::new(-0.9179007756213905, 0.39680998741671025),
    c64::new(-0.9185083943252121, 0.3954014789478165),
    c64::new(-0.9191138516900578, 0.39399204006104815),
    c64::new(-0.9197171462912274, 0.3925816740729514),
    c64::new(-0.9203182767091105, 0.39117038430225404),
    c64::new(-0.9209172415291894, 0.38975817406985647),
    c64::new(-0.9215140393420418, 0.3883450466988266),
    c64::new(-0.9221086687433451, 0.3869310055143887),
    c64::new(-0.9227011283338786, 0.38551605384391885),
    c64::new(-0.9232914167195275, 0.3841001950169353),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9244654743252627, 0.3812657692221623),
    c64::new(-0.9250492407826775, 0.3798472089240514),
    c64::new(-0.9256308305098727, 0.3784277548087656),
    c64::new(-0.9262102421383114, 0.37700741021641815),
    c64::new(-0.9267874743045817, 0.3755861784892174),
    c64::new(-0.9273625256504011, 0.374164062971458),
    c64::new(-0.9279353948226178, 0.37274106700951604),
    c64::new(-0.9285060804732155, 0.37131719395183765),
    c64::new(-0.9290745812593157, 0.3698924471489341),
    c64::new(-0.9296408958431812, 0.3684668299533726),
    c64::new(-0.930205022892219, 0.3670403457197673),
    c64::new(-0.9307669610789837, 0.3656129978047738),
    c64::new(-0.9313267090811803, 0.3641847895670801),
    c64::new(-0.9318842655816681, 0.3627557243673973),
    c64::new(-0.9324396292684625, 0.3613258055684542),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.9335437729788362, 0.3584634206337366),
    c64::new(-0.9340925504042588, 0.3570309612334303),
    c64::new(-0.9346391298196807, 0.355597661704784),
    c64::new(-0.9351835099389476, 0.35416352542049034),
    c64::new(-0.9357256894810803, 0.35272855575521095),
    c64::new(-0.9362656671702783, 0.3512927560855672),
    c64::new(-0.9368034417359217, 0.34985612979013486),
    c64::new(-0.9373390119125748, 0.3484186802494348),
    c64::new(-0.9378723764399898, 0.34698041084592374),
    c64::new(-0.9384035340631082, 0.345541324963989),
    c64::new(-0.9389324835320645, 0.34410142598993904),
    c64::new(-0.9394592236021899, 0.34266071731199443),
    c64::new(-0.9399837530340138, 0.3412192023202827),
    c64::new(-0.9405060705932683, 0.339776884406827),
    c64::new(-0.9410261750508893, 0.3383337669655411),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9420597397710173, 0.3354451470845317),
    c64::new(-0.9425731976014469, 0.3339996514420093),
    c64::new(-0.9430844374660934, 0.33255336986604445),
    c64::new(-0.9435934581619604, 0.3311063057598765),
    c64::new(-0.9441002584912727, 0.32965846252858744),
    c64::new(-0.9446048372614801, 0.3282098435790927),
    c64::new(-0.9451071932852606, 0.3267604523201318),
    c64::new(-0.9456073253805213, 0.3253102921622632),
    c64::new(-0.9461052323704033, 0.323859366517853),
    c64::new(-0.9466009130832835, 0.32240767880106985),
    c64::new(-0.9470943663527771, 0.3209552324278755),
    c64::new(-0.9475855910177411, 0.3195020308160158),
    c64::new(-0.9480745859222762, 0.3180480773850149),
    c64::new(-0.9485613499157303, 0.31659337555616607),
    c64::new(-0.9490458818527006, 0.31513792875252244),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.950008245001843, 0.3122248139218251),
    c64::new(-0.9504860739494817, 0.31076715274961153),
    c64::new(-0.950961666311575, 0.309308760312269),
    c64::new(-0.9514350209690083, 0.30784964004153503),
    c64::new(-0.9519061368079323, 0.3063897953708609),
    c64::new(-0.9523750127197658, 0.30492922973540265),
    c64::new(-0.9528416476011986, 0.3034679465720114),
    c64::new(-0.9533060403541939, 0.30200594931922803),
    c64::new(-0.9537681898859902, 0.3005432414172737),
    c64::new(-0.9542280951091056, 0.29907982630804053),
    c64::new(-0.9546857549413383, 0.29761570743508614),
    c64::new(-0.9551411683057707, 0.296150888243624),
    c64::new(-0.9555943341307711, 0.2946853721805144),
    c64::new(-0.9560452513499964, 0.29321916269425896),
    c64::new(-0.956493918902395, 0.29175226323498943),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9573845007889759, 0.28881640820604976),
    c64::new(-0.9578264130275329, 0.2873474595447296),
    c64::new(-0.9582660714080177, 0.28587783472708056),
    c64::new(-0.9587034748958715, 0.28440753721127204),
    c64::new(-0.9591386224618419, 0.28293657045705545),
    c64::new(-0.9595715130819845, 0.2814649379257579),
    c64::new(-0.9600021457376658, 0.27999264308027344),
    c64::new(-0.9604305194155658, 0.2785196893850531),
    c64::new(-0.9608566331076795, 0.27704608030610023),
    c64::new(-0.9612804858113206, 0.2755718193109583),
    c64::new(-0.9617020765291225, 0.2740969098687064),
    c64::new(-0.9621214042690416, 0.27262135544994925),
    c64::new(-0.9625384680443592, 0.2711451595268081),
    c64::new(-0.9629532668736839, 0.26966832557291504),
    c64::new(-0.9633657997809539, 0.2681908570634034),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.9641840639517458, 0.26523403028551173),
    c64::new(-0.9645897932898126, 0.26375467897483157),
    c64::new(-0.9649932528549203, 0.26227470702391364),
    c64::new(-0.9653944416976893, 0.26079411791527585),
    c64::new(-0.9657933588740836, 0.2593129151328864),
    c64::new(-0.9661900034454125, 0.257831102162159),
    c64::new(-0.966584374478333, 0.25634868248994314),
    c64::new(-0.9669764710448521, 0.2548656596045147),
    c64::new(-0.9673662922223285, 0.2533820369955701),
    c64::new(-0.9677538370934754, 0.2518978181542172),
    c64::new(-0.9681391047463623, 0.25041300657296534),
    c64::new(-0.9685220942744174, 0.2489276057457201),
    c64::new(-0.9689028047764289, 0.2474416191677735),
    c64::new(-0.9692812353565485, 0.24595505033579465),
    c64::new(-0.9696573851242923, 0.24446790274782448),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9704028386875555, 0.24149188530286933),
    c64::new(-0.9707721407289502, 0.24000302244874175),
    c64::new(-0.9711391584497251, 0.23851359484431855),
    c64::new(-0.9715038909862518, 0.23702360599436717),
    c64::new(-0.9718663374802793, 0.23553305940497574),
    c64::new(-0.9722264970789363, 0.23404195858354349),
    c64::new(-0.9725843689347322, 0.23255030703877516),
    c64::new(-0.9729399522055601, 0.2310581082806713),
    c64::new(-0.9732932460546982, 0.22956536582051892),
    c64::new(-0.9736442496508119, 0.22807208317088606),
    c64::new(-0.9739929621679558, 0.22657826384561017),
    c64::new(-0.9743393827855759, 0.22508391135979283),
    c64::new(-0.9746835106885107, 0.22358902922979026),
    c64::new(-0.9750253450669941, 0.22209362097320365),
    c64::new(-0.975364885116657, 0.22059769010887348),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.976037079039039, 0.21760427463848372),
    c64::new(-0.9763697313300211, 0.21610679707621944),
    c64::new(-0.9767000861287117, 0.21460881099378698),
    c64::new(-0.9770281426577544, 0.21311031991609142),
    c64::new(-0.9773539001452, 0.2116113273692279),
    c64::new(-0.9776773578245099, 0.21011183688046978),
    c64::new(-0.9779985149345571, 0.20861185197826349),
    c64::new(-0.9783173707196275, 0.20711137619221884),
    c64::new(-0.9786339244294231, 0.20561041305309938),
    c64::new(-0.9789481753190622, 0.20410896609281684),
    c64::new(-0.979260122649082, 0.20260703884442136),
    c64::new(-0.9795697656854405, 0.20110463484209198),
    c64::new(-0.9798771036995176, 0.1996017576211309),
    c64::new(-0.9801821359681173, 0.19809841071795378),
    c64::new(-0.9804848617734694, 0.19659459767008028),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9810833911504866, 0.19358558729580377),
    c64::new(-0.9813791933137546, 0.19208039704989244),
    c64::new(-0.9816726861969831, 0.19057475482025305),
    c64::new(-0.9819638691095552, 0.18906866414980633),
    c64::new(-0.9822527413662894, 0.18756212858252957),
    c64::new(-0.9825393022874412, 0.18605515166344688),
    c64::new(-0.9828235511987052, 0.1845477369386197),
    c64::new(-0.9831054874312164, 0.1830398879551409),
    c64::new(-0.9833851103215512, 0.1815316082611252),
    c64::new(-0.9836624192117303, 0.18002290140569957),
    c64::new(-0.9839374134492189, 0.1785137709389974),
    c64::new(-0.984210092386929, 0.17700422041214892),
    c64::new(-0.9844804553832209, 0.17549425337727143),
    c64::new(-0.9847485018019042, 0.1739838733874641),
    c64::new(-0.9850142310122398, 0.1724730839967961),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.9855387353121761, 0.1694502912339682),
    c64::new(-0.9857975091675674, 0.16793829497473126),
    c64::new(-0.9860539633461954, 0.16642590354046405),
    c64::new(-0.9863080972445986, 0.16491312048997012),
    c64::new(-0.9865599102647754, 0.16339994938297328),
    c64::new(-0.9868094018141855, 0.16188639378011171),
    c64::new(-0.987056571305751, 0.16037245724292842),
    c64::new(-0.9873014181578584, 0.15885814333386145),
    c64::new(-0.9875439417943592, 0.15734345561623855),
    c64::new(-0.9877841416445722, 0.15582839765426537),
    c64::new(-0.9880220171432835, 0.15431297301302008),
    c64::new(-0.9882575677307495, 0.15279718525844369),
    c64::new(-0.9884907928526966, 0.1512810379573303),
    c64::new(-0.9887216919603238, 0.14976453467732145),
    c64::new(-0.988950264510303, 0.14824767898689625),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.9894004277913804, 0.14521292465284735),
    c64::new(-0.9896220174632008, 0.14369503315029464),
    c64::new(-0.9898412784588205, 0.14217680351944806),
    c64::new(-0.990058210262297, 0.14065823933284952),
    c64::new(-0.9902728123631691, 0.13913934416382634),
    c64::new(-0.9904850842564571, 0.137620121586486),
    c64::new(-0.9906950254426646, 0.13610057517570645),
    c64::new(-0.99090263542778, 0.13458070850712628),
    c64::new(-0.9911079137232769, 0.133060525157139),
    c64::new(-0.9913108598461154, 0.1315400287028833),
    c64::new(-0.9915114733187439, 0.1300192227222334),
    c64::new(-0.9917097536690995, 0.12849811079379306),
    c64::new(-0.9919057004306093, 0.12697669649688603),
    c64::new(-0.9920993131421918, 0.12545498341154626),
    c64::new(-0.9922905913482573, 0.12393297511851248),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.992666142448948, 0.12088808723577707),
    c64::new(-0.9928504144598651, 0.11936521481099163),
    c64::new(-0.9930323501978514, 0.11784206150832509),
    c64::new(-0.9932119492347945, 0.11631863091190471),
    c64::new(-0.9933892111480807, 0.11479492660651032),
    c64::new(-0.9935641355205953, 0.11327095217756442),
    c64::new(-0.9937367219407246, 0.1117467112111265),
    c64::new(-0.9939069700023561, 0.11022220729388325),
    c64::new(-0.9940748793048794, 0.10869744401313874),
    c64::new(-0.9942404494531879, 0.10717242495680916),
    c64::new(-0.9944036800576791, 0.10564715371341077),
    c64::new(-0.9945645707342554, 0.10412163387205457),
    c64::new(-0.9947231211043257, 0.10259586902243656),
    c64::new(-0.9948793307948056, 0.10106986275482793),
    c64::new(-0.9950331994381186, 0.09954361866006928),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.9953339121404823, 0.09649043135525266),
    c64::new(-0.9954807554919269, 0.09496349532963891),
    c64::new(-0.9956252563809943, 0.09343633584574798),
    c64::new(-0.9957674144676598, 0.09190895649713277),
    c64::new(-0.9959072294174117, 0.0903813608778653),
    c64::new(-0.996044700901252, 0.08885355258252475),
    c64::new(-0.996179828595697, 0.08732553520619206),
    c64::new(-0.996312612182778, 0.08579731234444016),
    c64::new(-0.9964430513500426, 0.08426888759332418),
    c64::new(-0.9965711457905548, 0.08274026454937565),
    c64::new(-0.9966968952028961, 0.08121144680959266),
    c64::new(-0.9968202992911657, 0.0796824379714302),
    c64::new(-0.9969413577649822, 0.07815324163279415),
    c64::new(-0.997060070339483, 0.07662386139203169),
    c64::new(-0.9971764367353261, 0.07509430084792135),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9974021299012753, 0.07203465324688949),
    c64::new(-0.9975114561403035, 0.07050457338961386),
    c64::new(-0.9976184351385196, 0.06897432762826702),
    c64::new(-0.9977230666441916, 0.06744391956366418),
    c64::new(-0.9978253504111116, 0.06591335279700376),
    c64::new(-0.997925286198596, 0.0643826309298577),
    c64::new(-0.9980228737714862, 0.06285175756416149),
    c64::new(-0.9981181129001492, 0.061320736302208495),
    c64::new(-0.9982110033604782, 0.05978957074664007),
    c64::new(-0.9983015449338929, 0.058258264500435794),
    c64::new(-0.9983897374073402, 0.05672682116690807),
    c64::new(-0.9984755805732948, 0.055195244349690094),
    c64::new(-0.9985590742297593, 0.05366353765273052),
    c64::new(-0.9986402181802652, 0.0521317046802836),
    c64::new(-0.9987190122338729, 0.0505997490368994),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.9988695499142836, 0.047535484156959545),
    c64::new(-0.9989412931868569, 0.046003182130914706),
    c64::new(-0.9990106858540734, 0.044470771854938584),
    c64::new(-0.9990777277526454, 0.04293825693494102),
    c64::new(-0.9991424187248169, 0.04140564097707678),
    c64::new(-0.9992047586183639, 0.03987292758774013),
    c64::new(-0.9992647472865944, 0.038340120373552854),
    c64::new(-0.9993223845883495, 0.03680722294135883),
    c64::new(-0.9993776703880028, 0.03527423889821423),
    c64::new(-0.9994306045554617, 0.033741171851377705),
    c64::new(-0.999481186966167, 0.032208025408304544),
    c64::new(-0.9995294175010931, 0.030674803176636865),
    c64::new(-0.9995752960467492, 0.029141508764193802),
    c64::new(-0.9996188224951786, 0.027608145778965656),
    c64::new(-0.9996599967439592, 0.0260747178291041),
    c64::new(-0.9996988186962042, 0.024541228522912326),
    c64::new(-0.9997352882605617, 0.02300768146883969),
    c64::new(-0.9997694053512153, 0.021474080275469667),
    c64::new(-0.9998011698878843, 0.019940428551514438),
    c64::new(-0.9998305817958234, 0.0184067299058051),
    c64::new(-0.9998576410058239, 0.016872987947281832),
    c64::new(-0.9998823474542126, 0.01533920628498806),
    c64::new(-0.9999047010828528, 0.013805388528060633),
    c64::new(-0.9999247018391445, 0.012271538285720007),
    c64::new(-0.9999423496760239, 0.01073765916726441),
    c64::new(-0.9999576445519639, 0.009203754782060022),
    c64::new(-0.9999705864309741, 0.007669828739531138),
    c64::new(-0.9999811752826011, 0.006135884649154799),
    c64::new(-0.9999894110819284, 0.004601926120448733),
    c64::new(-0.9999952938095762, 0.003067956762965977),
    c64::new(-0.9999988234517019, 0.0015339801862850493),
];

static INVERSE_TWIDDLES_2048: [c64; 2048] = [
    c64::new(0.00048828125, -0.0),
    c64::new(0.0004882806755135263, -7.490137628343582e-07),
    c64::new(0.0004882789520554571, -1.4980257631669805e-06),
    c64::new(0.00048827607962984785, -2.2470342385002786e-06),
    c64::new(0.00048827205824345757, -2.9960374263449586e-06),
    c64::new(0.0004882668879057491, -3.7450335642241685e-06),
    c64::new(0.0004882605686288886, -4.494020889677646e-06),
    c64::new(0.00048825310042774605, -5.2429976402658645e-06),
    c64::new(0.0004882444833198948, -5.991962053574182e-06),
    c64::new(0.00048823471732561177, -6.740912367216988e-06),
    c64::new(0.0004882238024678772, -7.489846818841846e-06),
    c64::new(0.00048821173877237493, -8.238763646133648e-06),
    c64::new(0.0004881985262674919, -8.98766108681876e-06),
    c64::new(0.0004881841649843185, -9.73653737866916e-06),
    c64::new(0.0004881686549566481, -1.0485390759506596e-05),
    c64::new(0.0004881519962209774, -1.1234219467206723e-05),
    c64::new(0.000488134188816506, -1.1983021739703266e-05),
    c64::new(0.00048811523278513634, -1.2731795814992139e-05),
    c64::new(0.00048809512817147394, -1.3480539931135615e-05),
    c64::new(0.00048807387502282677, -1.4229252326266466e-05),
    c64::new(0.00048805147338920564, -1.4977931238592103e-05),
    c64::new(0.0004880279233233237, -1.5726574906398724e-05),
    c64::new(0.00048800322488059655, -1.647518156805546e-05),
    c64::new(0.000487977378119142, -1.722374946201853e-05),
    c64::new(0.00048795038309978005, -1.7972276826835367e-05),
    c64::new(0.00048792223988603243, -1.8720761901148776e-05),
    c64::new(0.000487892948544123, -1.946920292370108e-05),
    c64::new(0.000487862509142977, -2.021759813333825e-05),
    c64::new(0.00048783092175422137, -2.0965945769014072e-05),
    c64::new(0.00048779818645218427, -2.171424406979427e-05),
    c64::new(0.00048776430331389496, -2.2462491274860656e-05),
    c64::new(0.00048772927241908377, -2.3210685623515284e-05),
    c64::new(0.00048769309385018184, -2.3958825355184578e-05),
    c64::new(0.00048765576769232077, -2.4706908709423477e-05),
    c64::new(0.00048761729403333265, -2.545493392591959e-05),
    c64::new(0.00048757767296374966, -2.6202899244497324e-05),
    c64::new(0.0004875369045768041, -2.695080290512204e-05),
    c64::new(0.0004874949889684278, -2.7698643147904174e-05),
    c64::new(0.0004874519262372524, -2.8446418213103395e-05),
    c64::new(0.0004874077164846085, -2.9194126341132748e-05),
    c64::new(0.00048736235981452597, -2.9941765772562782e-05),
    c64::new(0.0004873158563337335, -3.068933474812569e-05),
    c64::new(0.0004872682061516582, -3.1436831508719465e-05),
    c64::new(0.0004872194093804256, -3.2184254295412014e-05),
    c64::new(0.0004871694661348592, -3.293160134944534e-05),
    c64::new(0.00048711837653248025, -3.367887091223962e-05),
    c64::new(0.00048706614069350755, -3.442606122539739e-05),
    c64::new(0.0004870127587408571, -3.517317053070768e-05),
    c64::new(0.0004869582308001417, -3.592019707015011e-05),
    c64::new(0.000486902556999671, -3.6667139085899075e-05),
    c64::new(0.00048684573747045067, -3.741399482032788e-05),
    c64::new(0.0004867877723461827, -3.816076251601281e-05),
    c64::new(0.0004867286617632645, -3.8907440415737366e-05),
    c64::new(0.0004866684058607891, -3.965402676249631e-05),
    c64::new(0.00048660700478054435, -4.040051979949985e-05),
    c64::new(0.000486544458667013, -4.114691777017777e-05),
    c64::new(0.00048648076766737207, -4.189321891818354e-05),
    c64::new(0.00048641593193149267, -4.2639421487398466e-05),
    c64::new(0.00048634995161193944, -4.338552372193584e-05),
    c64::new(0.00048628282686397057, -4.413152386614501e-05),
    c64::new(0.000486214557845537, -4.487742016461559e-05),
    c64::new(0.0004861451447172824, -4.5623210862181536e-05),
    c64::new(0.00048607458764254245, -4.636889420392529e-05),
    c64::new(0.00048600288678734486, -4.711446843518193e-05),
    c64::new(0.00048593004232040866, -4.7859931801543263e-05),
    c64::new(0.00048585605441314386, -4.860528254886197e-05),
    c64::new(0.0004857809232396512, -4.935051892325577e-05),
    c64::new(0.00048570464897672153, -5.0095639171111465e-05),
    c64::new(0.00048562723180383565, -5.0840641539089154e-05),
    c64::new(0.0004855486719031636, -5.158552427412628e-05),
    c64::new(0.0004854689694595644, -5.233028562344182e-05),
    c64::new(0.00048538812466058563, -5.307492383454039e-05),
    c64::new(0.0004853061376964629, -5.381943715521634e-05),
    c64::new(0.00048522300876011943, -5.4563823833557904e-05),
    c64::new(0.0004851387380471657, -5.530808211795134e-05),
    c64::new(0.00048505332575589876, -5.6052210257085e-05),
    c64::new(0.000484966772087302, -5.679620649995349e-05),
    c64::new(0.00048487907724504463, -5.7540069095861805e-05),
    c64::new(0.000484790241435481, -5.828379629442937e-05),
    c64::new(0.0004847002648676504, -5.902738634559428e-05),
    c64::new(0.00048460914775327635, -5.977083749961728e-05),
    c64::new(0.0004845168903067663, -6.0514148007086015e-05),
    c64::new(0.00048442349274521084, -6.125731611891906e-05),
    c64::new(0.00048432895528838346, -6.200034008637005e-05),
    c64::new(0.00048423327815874, -6.274321816103182e-05),
    c64::new(0.0004841364615814179, -6.34859485948405e-05),
    c64::new(0.00048403850578423606, -6.422852964007965e-05),
    c64::new(0.0004839394109976938, -6.497095954938431e-05),
    c64::new(0.0004838391774549707, -6.57132365757452e-05),
    c64::new(0.0004837378053919261, -6.64553589725128e-05),
    c64::new(0.0004836352950470982, -6.719732499340139e-05),
    c64::new(0.00048353164666170367, -6.793913289249326e-05),
    c64::new(0.00048342686047963727, -6.868078092424278e-05),
    c64::new(0.00048332093674747096, -6.942226734348048e-05),
    c64::new(0.00048321387571445356, -7.016359040541722e-05),
    c64::new(0.00048310567763250995, -7.090474836564818e-05),
    c64::new(0.00048299634275624073, -7.16457394801571e-05),
    c64::new(0.0004828858713429214, -7.238656200532033e-05),
    c64::new(0.00048277426365250185, -7.312721419791089e-05),
    c64::new(0.00048266151994760576, -7.386769431510265e-05),
    c64::new(0.00048254764049353, -7.460800061447433e-05),
    c64::new(0.0004824326255582439, -7.534813135401372e-05),
    c64::new(0.00048231647541238876, -7.60880847921217e-05),
    c64::new(0.00048219919032927697, -7.682785918761633e-05),
    c64::new(0.0004820807705848918, -7.756745279973703e-05),
    c64::new(0.0004819612164578862, -7.830686388814858e-05),
    c64::new(0.0004818405282295828, -7.904609071294523e-05),
    c64::new(0.00048171870618397237, -7.97851315346549e-05),
    c64::new(0.0004815957506077142, -8.052398461424313e-05),
    c64::new(0.0004814716617901345, -8.126264821311724e-05),
    c64::new(0.0004813464400232263, -8.200112059313046e-05),
    c64::new(0.00048122008560164847, -8.273940001658592e-05),
    c64::new(0.0004810925988227252, -8.347748474624083e-05),
    c64::new(0.00048096397998644523, -8.421537304531052e-05),
    c64::new(0.00048083422939546104, -8.495306317747257e-05),
    c64::new(0.00048070334735508835, -8.569055340687081e-05),
    c64::new(0.0004805713341733052, -8.64278419981195e-05),
    c64::new(0.00048043819016075143, -8.716492721630738e-05),
    c64::new(0.00048030391563072766, -8.790180732700172e-05),
    c64::new(0.0004801685108991949, -8.863848059625242e-05),
    c64::new(0.0004800319762847736, -8.937494529059617e-05),
    c64::new(0.0004798943121087428, -9.011119967706036e-05),
    c64::new(0.00047975551869503967, -9.08472420231673e-05),
    c64::new(0.0004796155963702585, -9.158307059693828e-05),
    c64::new(0.00047947454546365, -9.231868366689755e-05),
    c64::new(0.00047933236630712066, -9.305407950207653e-05),
    c64::new(0.0004791890592352317, -9.378925637201779e-05),
    c64::new(0.0004790446245851986, -9.45242125467791e-05),
    c64::new(0.00047889906269688986, -9.525894629693762e-05),
    c64::new(0.00047875237391282685, -9.599345589359386e-05),
    c64::new(0.00047860455857818234, -9.672773960837576e-05),
    c64::new(0.0004784556170407801, -9.746179571344286e-05),
    c64::new(0.000478305549651094, -9.819562248149019e-05),
    c64::new(0.0004781543567622471, -9.89292181857525e-05),
    c64::new(0.00047800203873001084, -9.966258110000824e-05),
    c64::new(0.0004778485959128043, -0.00010039570949858361),
    c64::new(0.0004776940286716932, -0.00010112860165635672),
    c64::new(0.00047753833737038923, -0.00010186125584876147),
    c64::new(0.000477381522375249, -0.0001025936703517918),
    c64::new(0.00047722358405527347, -0.00010332584344200564),
    c64::new(0.00047706452278210664, -0.00010405777339652899),
    c64::new(0.0004769043389300351, -0.00010478945849305994),
    c64::new(0.0004767430328759869, -0.00010552089700987281),
    c64::new(0.0004765806049995308, -0.00010625208722582209),
    c64::new(0.0004764170556828753, -0.00010698302742034658),
    c64::new(0.00047625238531086766, -0.00010771371587347339),
    c64::new(0.0004760865942709932, -0.00010844415086582203),
    c64::new(0.00047591968295337435, -0.00010917433067860839),
    c64::new(0.00047575165175076946, -0.00010990425359364884),
    c64::new(0.0004755825010585722, -0.00011063391789336426),
    c64::new(0.00047541223127481054, -0.00011136332186078405),
    c64::new(0.00047524084280014563, -0.00011209246377955023),
    c64::new(0.0004750683360378712, -0.00011282134193392144),
    c64::new(0.0004748947113939122, -0.00011354995460877697),
    c64::new(0.00047471996927682435, -0.00011427830008962082),
    c64::new(0.0004745441100977927, -0.00011500637666258569),
    c64::new(0.00047436713427063075, -0.00011573418261443711),
    c64::new(0.00047418904221177983, -0.00011646171623257735),
    c64::new(0.0004740098343403078, -0.00011718897580504956),
    c64::new(0.00047382951107790796, -0.00011791595962054167),
    c64::new(0.0004736480728488984, -0.00011864266596839056),
    c64::new(0.0004734655200802209, -0.00011936909313858601),
    c64::new(0.0004732818532014397, -0.0001200952394217747),
    c64::new(0.00047309707264474066, -0.00012082110310926429),
    c64::new(0.0004729111788449303, -0.00012154668249302742),
    c64::new(0.0004727241722394348, -0.00012227197586570568),
    c64::new(0.0004725360532682986, -0.00012299698152061375),
    c64::new(0.00047234682237418384, -0.00012372169775174324),
    c64::new(0.0004721564800023692, -0.00012444612285376688),
    c64::new(0.0004719650266007486, -0.00012517025512204244),
    c64::new(0.0004717724626198303, -0.0001258940928526167),
    c64::new(0.0004715787885127362, -0.0001266176343422296),
    c64::new(0.0004713840047351999, -0.00012734087788831812),
    c64::new(0.00047118811174556656, -0.0001280638217890203),
    c64::new(0.0004709911100047914, -0.00012878646434317937),
    c64::new(0.0004707929999764384, -0.00012950880385034755),
    c64::new(0.0004705937821266796, -0.00013023083861079022),
    c64::new(0.00047039345692429397, -0.00013095256692548983),
    c64::new(0.00047019202484066596, -0.00013167398709614995),
    c64::new(0.00046998948634978475, -0.00013239509742519922),
    c64::new(0.00046978584192824296, -0.0001331158962157954),
    c64::new(0.0004695810920552356, -0.0001338363817718293),
    c64::new(0.0004693752372125589, -0.00013455655239792878),
    c64::new(0.0004691682778846092, -0.00013527640639946284),
    c64::new(0.00046896021455838173, -0.00013599594208254544),
    c64::new(0.0004687510477234697, -0.00013671515775403966),
    c64::new(0.00046854077787206275, -0.0001374340517215615),
    c64::new(0.00046832940549894624, -0.00013815262229348408),
    c64::new(0.0004681169311014998, -0.00013887086777894135),
    c64::new(0.00046790335517969613, -0.00013958878648783233),
    c64::new(0.00046768867823610005, -0.00014030637673082496),
    c64::new(0.0004674729007758671, -0.0001410236368193601),
    c64::new(0.0004672560233067426, -0.00014174056506565544),
    c64::new(0.0004670380463390601, -0.0001424571597827096),
    c64::new(0.00046681897038574043, -0.00014317341928430597),
    c64::new(0.0004665987959622906, -0.00014388934188501676),
    c64::new(0.0004663775235868021, -0.00014460492590020693),
    c64::new(0.00046615515377995036, -0.00014532016964603818),
    c64::new(0.000465931687064993, -0.0001460350714394729),
    c64::new(0.0004657071239677687, -0.00014674962959827805),
    c64::new(0.0004654814650166962, -0.00014746384244102934),
    c64::new(0.0004652547107427728, -0.0001481777082871149),
    c64::new(0.0004650268616795732, -0.00014889122545673944),
    c64::new(0.0004647979183632482, -0.00014960439227092818),
    c64::new(0.0004645678813325236, -0.0001503172070515307),
    c64::new(0.00046433675112869877, -0.00015102966812122496),
    c64::new(0.00046410452829564536, -0.00015174177380352123),
    c64::new(0.00046387121337980615, -0.00015245352242276606),
    c64::new(0.0004636368069301937, -0.00015316491230414625),
    c64::new(0.00046340130949838894, -0.0001538759417736926),
    c64::new(0.00046316472163854017, -0.0001545866091582841),
    c64::new(0.00046292704390736144, -0.00015529691278565183),
    c64::new(0.0004626882768641314, -0.00015600685098438266),
    c64::new(0.000462448421070692, -0.00015671642208392344),
    c64::new(0.00046220747709144704, -0.0001574256244145849),
    c64::new(0.00046196544549336106, -0.00015813445630754534),
    c64::new(0.00046172232684595766, -0.00015884291609485494),
    c64::new(0.00046147812172131865, -0.00015955100210943932),
    c64::new(0.00046123283069408216, -0.00016025871268510376),
    c64::new(0.00046098645434144173, -0.00016096604615653686),
    c64::new(0.0004607389932431447, -0.00016167300085931466),
    c64::new(0.00046049044798149096, -0.0001623795751299044),
    c64::new(0.0004602408191413315, -0.00016308576730566864),
    c64::new(0.00045999010731006705, -0.00016379157572486895),
    c64::new(0.0004597383130776469, -0.00016449699872666995),
    c64::new(0.000459485437036567, -0.00016520203465114313),
    c64::new(0.0004592314797818693, -0.00016590668183927092),
    c64::new(0.00045897644191113967, -0.00016661093863295037),
    c64::new(0.0004587203240245068, -0.00016731480337499726),
    c64::new(0.00045846312672464086, -0.0001680182744091498),
    c64::new(0.000458204850616752, -0.0001687213500800728),
    c64::new(0.0004579454963085888, -0.00016942402873336117),
    c64::new(0.000457685064410437, -0.00017012630871554422),
    c64::new(0.00045742355553511795, -0.00017082818837408932),
    c64::new(0.00045716097029798743, -0.0001715296660574058),
    c64::new(0.0004568973093169338, -0.000172230740114849),
    c64::new(0.00045663257321237676, -0.0001729314088967238),
    c64::new(0.000456366762607266, -0.000173631670754289),
    c64::new(0.0004560998781270796, -0.00017433152403976073),
    c64::new(0.00045583192039982235, -0.00017503096710631667),
    c64::new(0.0004555628900560249, -0.00017572999830809966),
    c64::new(0.0004552927877287414, -0.00017642861600022182),
    c64::new(0.0004550216140535489, -0.00017712681853876818),
    c64::new(0.00045474936966854513, -0.00017782460428080073),
    c64::new(0.0004544760552143475, -0.00017852197158436223),
    c64::new(0.00045420167133409134, -0.00017921891880848007),
    c64::new(0.00045392621867342833, -0.00017991544431317008),
    c64::new(0.00045364969788052527, -0.00018061154645944048),
    c64::new(0.0004533721096060623, -0.00018130722360929568),
    c64::new(0.0004530934545032314, -0.00018200247412574012),
    c64::new(0.0004528137332277349, -0.0001826972963727822),
    c64::new(0.00045253294643778406, -0.0001833916887154381),
    c64::new(0.00045225109479409735, -0.00018408564951973548),
    c64::new(0.0004519681789598988, -0.00018477917715271756),
    c64::new(0.0004516841996009168, -0.00018547226998244686),
    c64::new(0.00045139915738538213, -0.00018616492637800897),
    c64::new(0.00045111305298402673, -0.0001868571447095165),
    c64::new(0.00045082588707008185, -0.0001875489233481128),
    c64::new(0.0004505376603192767, -0.000188240260665976),
    c64::new(0.00045024837340983646, -0.00018893115503632255),
    c64::new(0.0004499580270224814, -0.00018962160483341125),
    c64::new(0.00044966662184042457, -0.00019031160843254708),
    c64::new(0.0004493741585493704, -0.0001910011642100849),
    c64::new(0.00044908063783751336, -0.00019169027054343333),
    c64::new(0.000448786060395536, -0.00019237892581105864),
    c64::new(0.00044849042691660754, -0.00019306712839248845),
    c64::new(0.0004481937380963821, -0.00019375487666831558),
    c64::new(0.000447895994632997, -0.00019444216902020193),
    c64::new(0.00044759719722707163, -0.00019512900383088222),
    c64::new(0.00044729734658170494, -0.00019581537948416778),
    c64::new(0.00044699644340247453, -0.0001965012943649504),
    c64::new(0.00044669448839743467, -0.0001971867468592061),
    c64::new(0.0004463914822771146, -0.00019787173535399896),
    c64::new(0.000446087425754517, -0.00019855625823748488),
    c64::new(0.0004457823195451163, -0.00019924031389891537),
    c64::new(0.0004454761643668568, -0.0001999239007286414),
    c64::new(0.0004451689609401513, -0.00020060701711811714),
    c64::new(0.0004448607099878791, -0.00020128966145990375),
    c64::new(0.0004445514122353844, -0.00020197183214767312),
    c64::new(0.0004442410684104747, -0.00020265352757621173),
    c64::new(0.00044392967924341913, -0.0002033347461414244),
    c64::new(0.00044361724546694635, -0.00020401548624033802),
    c64::new(0.0004433037678162433, -0.00020469574627110542),
    c64::new(0.00044298924702895304, -0.00020537552463300904),
    c64::new(0.0004426736838451735, -0.0002060548197264647),
    c64::new(0.0004423570790074553, -0.00020673362995302541),
    c64::new(0.0004420394332608002, -0.00020741195371538516),
    c64::new(0.0004417207473526593, -0.00020808978941738262),
    c64::new(0.0004414010220329313, -0.00020876713546400492),
    c64::new(0.00044108025805396085, -0.00020944399026139135),
    c64::new(0.00044075845617053654, -0.0002101203522168372),
    c64::new(0.00044043561713988925, -0.0002107962197387975),
    c64::new(0.00044011174172169044, -0.0002114715912368906),
    c64::new(0.0004397868306780503, -0.00021214646512190217),
    c64::new(0.00043946088477351576, -0.0002128208398057888),
    c64::new(0.00043913390477506913, -0.00021349471370168168),
    c64::new(0.0004388058914521259, -0.00021416808522389046),
    c64::new(0.00043847684557653314, -0.00021484095278790686),
    c64::new(0.00043814676792256754, -0.00021551331481040853),
    c64::new(0.0004378156592669337, -0.0002161851697092627),
    c64::new(0.0004374835203887623, -0.00021685651590352988),
    c64::new(0.0004371503520696082, -0.00021752735181346765),
    c64::new(0.0004368161550934486, -0.00021819767586053426),
    c64::new(0.0004364809302466813, -0.00021886748646739252),
    c64::new(0.0004361446783181227, -0.00021953678205791335),
    c64::new(0.00043580740009900614, -0.00022020556105717962),
    c64::new(0.00043546909638297985, -0.0002208738218914897),
    c64::new(0.00043512976796610507, -0.00022154156298836126),
    c64::new(0.00043478941564685443, -0.00022220878277653508),
    c64::new(0.0004344480402261098, -0.00022287547968597848),
    c64::new(0.00043410564250716044, -0.00022354165214788927),
    c64::new(0.00043376222329570135, -0.00022420729859469926),
    c64::new(0.000433417783399831, -0.00022487241746007813),
    c64::new(0.0004330723236300499, -0.00022553700717893688),
    c64::new(0.0004327258447992581, -0.0002262010661874317),
    c64::new(0.0004323783477227539, -0.00022686459292296768),
    c64::new(0.00043202983321823134, -0.00022752758582420224),
    c64::new(0.0004316803021057788, -0.00022819004333104906),
    c64::new(0.00043132975520787675, -0.00022885196388468162),
    c64::new(0.0004309781933493959, -0.00022951334592753692),
    c64::new(0.00043062561735759524, -0.00023017418790331916),
    c64::new(0.0004302720280621201, -0.00023083448825700332),
    c64::new(0.00042991742629500037, -0.00023149424543483885),
    c64::new(0.00042956181289064815, -0.00023215345788435345),
    c64::new(0.0004292051886858562, -0.0002328121240543565),
    c64::new(0.0004288475545197957, -0.00023347024239494288),
    c64::new(0.00042848891123401433, -0.00023412781135749659),
    c64::new(0.0004281292596724344, -0.00023478482939469432),
    c64::new(0.0004277686006813509, -0.00023544129496050914),
    c64::new(0.00042740693510942915, -0.00023609720651021423),
    c64::new(0.00042704426380770323, -0.00023675256250038626),
    c64::new(0.00042668058762957366, -0.00023740736138890938),
    c64::new(0.0004263159074308057, -0.00023806160163497851),
    c64::new(0.0004259502240695271, -0.0002387152816991031),
    c64::new(0.00042558353840622606, -0.00023936840004311092),
    c64::new(0.00042521585130374945, -0.00024002095513015132),
    c64::new(0.0004248471636273005, -0.00024067294542469924),
    c64::new(0.00042447747624443693, -0.00024132436939255843),
    c64::new(0.0004241067900250688, -0.0002419752255008655),
    c64::new(0.0004237351058414567, -0.000242625512218093),
    c64::new(0.0004233624245682093, -0.00024327522801405365),
    c64::new(0.00042298874708228174, -0.00024392437135990324),
    c64::new(0.0004226140742629732, -0.0002445729407281449),
    c64::new(0.000422238406991925, -0.0002452209345926322),
    c64::new(0.0004218617461531185, -0.00024586835142857303),
    c64::new(0.0004214840926328731, -0.000246515189712533),
    c64::new(0.000421105447319844, -0.0002471614479224391),
    c64::new(0.00042072581110502017, -0.00024780712453758343),
    c64::new(0.0004203451848817223, -0.00024845221803862646),
    c64::new(0.00041996356954560065, -0.000249096726907601),
    c64::new(0.00041958096599463304, -0.0002497406496279152),
    c64::new(0.0004191973751291225, -0.0002503839846843569),
    c64::new(0.00041881279785169537, -0.0002510267305630965),
    c64::new(0.0004184272350672992, -0.0002516688857516909),
    c64::new(0.0004180406876832004, -0.00025231044873908685),
    c64::new(0.00041765315660898244, -0.0002529514180156247),
    c64::new(0.00041726464275654323, -0.0002535917920730418),
    c64::new(0.00041687514704009346, -0.00025423156940447615),
    c64::new(0.0004164846703761541, -0.0002548707485044699),
    c64::new(0.0004160932136835546, -0.00025550932786897287),
    c64::new(0.00041570077788343027, -0.00025614730599534617),
    c64::new(0.00041530736389922045, -0.00025678468138236554),
    c64::new(0.0004149129726566663, -0.0002574214525302253),
    c64::new(0.0004145176050838084, -0.00025805761794054125),
    c64::new(0.000414121262110985, -0.0002586931761163548),
    c64::new(0.00041372394467082926, -0.00025932812556213623),
    c64::new(0.00041332565369826764, -0.00025996246478378804),
    c64::new(0.0004129263901305171, -0.0002605961922886489),
    c64::new(0.00041252615490708355, -0.00026122930658549666),
    c64::new(0.000412124948969759, -0.00026186180618455234),
    c64::new(0.00041172277326261984, -0.0002624936895974831),
    c64::new(0.00041131962873202434, -0.00026312495533740646),
    c64::new(0.00041091551632661057, -0.000263755601918893),
    c64::new(0.000410510436997294, -0.00026438562785797043),
    c64::new(0.0004101043916972654, -0.0002650150316721269),
    c64::new(0.00040969738138198864, -0.00026564381188031437),
    c64::new(0.0004092894070091982, -0.0002662719670029524),
    c64::new(0.00040888046953889743, -0.0002668994955619311),
    c64::new(0.00040847056993335547, -0.0002675263960806153),
    c64::new(0.00040805970915710576, -0.0002681526670838475),
    c64::new(0.00040764788817694336, -0.00026877830709795157),
    c64::new(0.0004072351079619229, -0.000269403314650736),
    c64::new(0.00040682136948335605, -0.0002700276882714978),
    c64::new(0.0004064066737148094, -0.0002706514264910254),
    c64::new(0.00040599102163210217, -0.0002712745278416026),
    c64::new(0.00040557441421330386, -0.00027189699085701177),
    c64::new(0.00040515685243873196, -0.00027251881407253716),
    c64::new(0.00040473833729094965, -0.0002731399960249688),
    c64::new(0.00040431886975476357, -0.0002737605352526055),
    c64::new(0.00040389845081722126, -0.00027438043029525853),
    c64::new(0.00040347708146760915, -0.00027499967969425493),
    c64::new(0.00040305476269745, -0.00027561828199244116),
    c64::new(0.00040263149550050063, -0.0002762362357341861),
    c64::new(0.0004022072808727497, -0.0002768535394653851),
    c64::new(0.00040178211981241525, -0.00027747019173346264),
    c64::new(0.00040135601331994217, -0.00027808619108737653),
    c64::new(0.0004009289623980003, -0.00027870153607762073),
    c64::new(0.0004005009680514818, -0.00027931622525622903),
    c64::new(0.00040007203128749854, -0.0002799302571767784),
    c64::new(0.0003996421531153802, -0.00028054363039439246),
    c64::new(0.00039921133454667173, -0.0002811563434657448),
    c64::new(0.0003987795765951308, -0.00028176839494906223),
    c64::new(0.0003983468802767255, -0.0002823797834041287),
    c64::new(0.00039791324660963214, -0.0002829905073922879),
    c64::new(0.0003974786766142326, -0.0002836005654764475),
    c64::new(0.0003970431713131121, -0.00028420995622108176),
    c64::new(0.0003966067317310566, -0.0002848186781922355),
    c64::new(0.00039616935889505065, -0.000285426729957527),
    c64::new(0.0003957310538342748, -0.0002860341100861518),
    c64::new(0.00039529181758010315, -0.0002866408171488857),
    c64::new(0.00039485165116610107, -0.0002872468497180885),
    c64::new(0.00039441055562802264, -0.000287852206367707),
    c64::new(0.00039396853200380827, -0.0002884568856732784),
    c64::new(0.00039352558133358217, -0.0002890608862119341),
    c64::new(0.0003930817046596502, -0.00028966420656240224),
    c64::new(0.0003926369030264969, -0.00029026684530501193),
    c64::new(0.00039219117748078366, -0.000290868801021696),
    c64::new(0.00039174452907134557, -0.0002914700722959944),
    c64::new(0.00039129695884918957, -0.0002920706577130578),
    c64::new(0.0003908484678674916, -0.00029267055585965065),
    c64::new(0.00039039905718159424, -0.00029326976532415477),
    c64::new(0.00038994872784900436, -0.0002938682846965723),
    c64::new(0.0003894974809293902, -0.0002944661125685294),
    c64::new(0.0003890453174845795, -0.0002950632475332793),
    c64::new(0.00038859223857855643, -0.00029565968818570583),
    c64::new(0.0003881382452774595, -0.0002962554331223264),
    c64::new(0.0003876833386495787, -0.0002968504809412957),
    c64::new(0.00038722751976535334, -0.0002974448302424088),
    c64::new(0.00038677078969736933, -0.00029803847962710424),
    c64::new(0.0003863131495203565, -0.00029863142769846773),
    c64::new(0.00038585460031118654, -0.0002992236730612352),
    c64::new(0.00038539514314886995, -0.00029981521432179614),
    c64::new(0.0003849347791145539, -0.0003004060500881967),
    c64::new(0.0003844735092915192, -0.0003009961789701434),
    c64::new(0.0003840113347651786, -0.0003015855995790058),
    c64::new(0.0003835482566230732, -0.00030217431052782047),
    c64::new(0.0003830842759548707, -0.0003027623104312935),
    c64::new(0.00038261939385236246, -0.00030334959790580447),
    c64::new(0.0003821536114094608, -0.00030393617156940916),
    c64::new(0.000381686929722197, -0.00030452203004184303),
    c64::new(0.000381219349888718, -0.0003051071719445246),
    c64::new(0.00038075087300928445, -0.00030569159590055836),
    c64::new(0.00038028150018626757, -0.0003062753005347383),
    c64::new(0.00037981123252414704, -0.00030685828447355113),
    c64::new(0.00037934007112950803, -0.0003074405463451792),
    c64::new(0.0003788680171110389, -0.00030802208477950414),
    c64::new(0.00037839507157952823, -0.0003086028984081099),
    c64::new(0.00037792123564786274, -0.00030918298586428574),
    c64::new(0.0003774465104310239, -0.00030976234578303),
    c64::new(0.0003769708970460862, -0.0003103409768010528),
    c64::new(0.0003764943966122138, -0.0003109188775567794),
    c64::new(0.00037601701025065815, -0.00031149604669035354),
    c64::new(0.0003755387390847557, -0.0003120724828436405),
    c64::new(0.0003750595842399245, -0.0003126481846602303),
    c64::new(0.0003745795468436623, -0.000313223150785441),
    c64::new(0.0003740986280255434, -0.0003137973798663217),
    c64::new(0.0003736168289172163, -0.00031437087055165595),
    c64::new(0.0003731341506524007, -0.0003149436214919648),
    c64::new(0.00037265059436688544, -0.0003155156313395099),
    c64::new(0.00037216616119852485, -0.0003160868987482969),
    c64::new(0.0003716808522872372, -0.00031665742237407834),
    c64::new(0.0003711946687750012, -0.00031722720087435703),
    c64::new(0.0003707076118058535, -0.00031779623290838915),
    c64::new(0.00037021968252588637, -0.00031836451713718726),
    c64::new(0.0003697308820832444, -0.0003189320522235238),
    c64::new(0.00036924121162812233, -0.0003194988368319338),
    c64::new(0.000368750672312762, -0.00032006486962871843),
    c64::new(0.0003682592652914498, -0.0003206301492819478),
    c64::new(0.0003677669917205139, -0.0003211946744614642),
    c64::new(0.00036727385275832163, -0.0003217584438388854),
    c64::new(0.00036677984956527655, -0.00032232145608760766),
    c64::new(0.00036628498330381595, -0.0003228837098828087),
    c64::new(0.0003657892551384079, -0.00032344520390145107),
    c64::new(0.00036529266623554863, -0.00032400593682228504),
    c64::new(0.00036479521776375983, -0.00032456590732585195),
    c64::new(0.00036429691089358583, -0.00032512511409448717),
    c64::new(0.0003637977467975908, -0.000325683555812323),
    c64::new(0.0003632977266503561, -0.0003262412311652922),
    c64::new(0.0003627968516284774, -0.00032679813884113086),
    c64::new(0.00036229512291056204, -0.0003273542775293814),
    c64::new(0.0003617925416772262, -0.00032790964592139567),
    c64::new(0.0003612891091110919, -0.0003284642427103383),
    c64::new(0.0003607848263967847, -0.00032901806659118947),
    c64::new(0.00036027969472093066, -0.00032957111626074806),
    c64::new(0.0003597737152721532, -0.00033012339041763473),
    c64::new(0.0003592668892410711, -0.0003306748877622951),
    c64::new(0.0003587592178202946, -0.00033122560699700267),
    c64::new(0.000358250702204424, -0.0003317755468258618),
    c64::new(0.0003577413435900453, -0.00033232470595481105),
    c64::new(0.0003572311431757288, -0.00033287308309162583),
    c64::new(0.0003567201021620252, -0.0003334206769459219),
    c64::new(0.0003562082217514634, -0.00033396748622915797),
    c64::new(0.00035569550314854746, -0.00033451350965463885),
    c64::new(0.0003551819475597538, -0.00033505874593751884),
    c64::new(0.0003546675561935283, -0.0003356031937948042),
    c64::new(0.0003541523302602836, -0.00033614685194535666),
    c64::new(0.000353636270972396, -0.0003366897191098959),
    c64::new(0.00035311937954420286, -0.0003372317940110032),
    c64::new(0.0003526016571919997, -0.0003377730753731239),
    c64::new(0.00035208310513403716, -0.0003383135619225706),
    c64::new(0.0003515637245905184, -0.0003388532523875263),
    c64::new(0.00035104351678359594, -0.0003393921454980473),
    c64::new(0.000350522482937369, -0.0003399302399860659),
    c64::new(0.00035000062427788063, -0.0003404675345853938),
    c64::new(0.00034947794203311455, -0.00034100402803172506),
    c64::new(0.0003489544374329927, -0.00034153971906263856),
    c64::new(0.00034843011170937184, -0.0003420746064176017),
    c64::new(0.0003479049660960412, -0.0003426086888379729),
    c64::new(0.00034737900182871896, -0.00034314196506700454),
    c64::new(0.00034685222014505, -0.00034367443384984616),
    c64::new(0.0003463246222846024, -0.0003442060939335473),
    c64::new(0.00034579620948886494, -0.0003447369440670604),
    c64::new(0.00034526698300124393, -0.0003452669830012439),
    c64::new(0.0003447369440670605, -0.0003457962094888649),
    c64::new(0.00034420609393354736, -0.00034632462228460233),
    c64::new(0.0003436744338498462, -0.00034685222014504995),
    c64::new(0.0003431419650670046, -0.0003473790018287189),
    c64::new(0.00034260868883797293, -0.0003479049660960412),
    c64::new(0.0003420746064176018, -0.00034843011170937184),
    c64::new(0.0003415397190626386, -0.00034895443743299267),
    c64::new(0.00034100402803172506, -0.00034947794203311455),
    c64::new(0.00034046753458539386, -0.0003500006242778806),
    c64::new(0.00033993023998606597, -0.00035052248293736894),
    c64::new(0.0003393921454980473, -0.0003510435167835959),
    c64::new(0.00033885325238752637, -0.0003515637245905184),
    c64::new(0.0003383135619225707, -0.0003520831051340371),
    c64::new(0.00033777307537312395, -0.0003526016571919996),
    c64::new(0.0003372317940110033, -0.00035311937954420275),
    c64::new(0.00033668971910989597, -0.00035363627097239594),
    c64::new(0.0003361468519453567, -0.0003541523302602835),
    c64::new(0.0003356031937948043, -0.00035466755619352827),
    c64::new(0.0003350587459375189, -0.00035518194755975374),
    c64::new(0.0003345135096546389, -0.00035569550314854746),
    c64::new(0.00033396748622915797, -0.00035620822175146333),
    c64::new(0.0003334206769459219, -0.0003567201021620252),
    c64::new(0.0003328730830916259, -0.00035723114317572877),
    c64::new(0.0003323247059548111, -0.00035774134359004527),
    c64::new(0.00033177554682586185, -0.00035825070220442396),
    c64::new(0.0003312256069970027, -0.0003587592178202946),
    c64::new(0.00033067488776229514, -0.00035926688924107104),
    c64::new(0.0003301233904176348, -0.0003597737152721532),
    c64::new(0.00032957111626074806, -0.00036027969472093066),
    c64::new(0.0003290180665911895, -0.0003607848263967847),
    c64::new(0.00032846424271033836, -0.00036128910911109184),
    c64::new(0.0003279096459213957, -0.00036179254167722607),
    c64::new(0.00032735427752938144, -0.000362295122910562),
    c64::new(0.0003267981388411309, -0.00036279685162847735),
    c64::new(0.00032624123116529227, -0.00036329772665035604),
    c64::new(0.00032568355581232304, -0.0003637977467975908),
    c64::new(0.00032512511409448717, -0.0003642969108935858),
    c64::new(0.000324565907325852, -0.00036479521776375983),
    c64::new(0.0003240059368222851, -0.0003652926662355486),
    c64::new(0.00032344520390145107, -0.00036578925513840784),
    c64::new(0.0003228837098828087, -0.0003662849833038159),
    c64::new(0.0003223214560876077, -0.00036677984956527655),
    c64::new(0.0003217584438388854, -0.00036727385275832163),
    c64::new(0.00032119467446146423, -0.00036776699172051387),
    c64::new(0.0003206301492819478, -0.0003682592652914498),
    c64::new(0.0003200648696287185, -0.00036875067231276193),
    c64::new(0.00031949883683193386, -0.00036924121162812233),
    c64::new(0.00031893205222352386, -0.00036973088208324436),
    c64::new(0.0003183645171371873, -0.00037021968252588637),
    c64::new(0.00031779623290838915, -0.0003707076118058535),
    c64::new(0.0003172272008743571, -0.00037119466877500113),
    c64::new(0.0003166574223740784, -0.0003716808522872372),
    c64::new(0.00031608689874829696, -0.00037216616119852485),
    c64::new(0.00031551563133950996, -0.0003726505943668854),
    c64::new(0.00031494362149196486, -0.0003731341506524007),
    c64::new(0.000314370870551656, -0.00037361682891721623),
    c64::new(0.00031379737986632177, -0.00037409862802554336),
    c64::new(0.000313223150785441, -0.00037457954684366225),
    c64::new(0.00031264818466023036, -0.00037505958423992447),
    c64::new(0.00031207248284364055, -0.00037553873908475566),
    c64::new(0.0003114960466903536, -0.00037601701025065815),
    c64::new(0.00031091887755677945, -0.0003764943966122137),
    c64::new(0.00031034097680105284, -0.00037697089704608614),
    c64::new(0.0003097623457830301, -0.00037744651043102387),
    c64::new(0.0003091829858642858, -0.0003779212356478627),
    c64::new(0.0003086028984081099, -0.00037839507157952823),
    c64::new(0.0003080220847795042, -0.00037886801711103886),
    c64::new(0.00030744054634517925, -0.00037934007112950803),
    c64::new(0.00030685828447355113, -0.000379811232524147),
    c64::new(0.0003062753005347384, -0.0003802815001862675),
    c64::new(0.0003056915959005584, -0.0003807508730092844),
    c64::new(0.00030510717194452464, -0.00038121934988871796),
    c64::new(0.0003045220300418431, -0.00038168692972219697),
    c64::new(0.0003039361715694092, -0.00038215361140946075),
    c64::new(0.0003033495979058045, -0.0003826193938523624),
    c64::new(0.00030276231043129356, -0.0003830842759548707),
    c64::new(0.0003021743105278205, -0.0003835482566230732),
    c64::new(0.00030158559957900585, -0.0003840113347651786),
    c64::new(0.0003009961789701434, -0.0003844735092915192),
    c64::new(0.00030040605008819674, -0.0003849347791145538),
    c64::new(0.00029981521432179614, -0.0003853951431488699),
    c64::new(0.00029922367306123526, -0.00038585460031118654),
    c64::new(0.0002986314276984678, -0.0003863131495203565),
    c64::new(0.00029803847962710424, -0.0003867707896973693),
    c64::new(0.00029744483024240885, -0.00038722751976535334),
    c64::new(0.0002968504809412958, -0.00038768333864957864),
    c64::new(0.00029625543312232646, -0.0003881382452774595),
    c64::new(0.00029565968818570583, -0.0003885922385785564),
    c64::new(0.00029506324753327933, -0.00038904531748457945),
    c64::new(0.00029446611256852943, -0.00038949748092939016),
    c64::new(0.00029386828469657233, -0.0003899487278490043),
    c64::new(0.00029326976532415477, -0.00039039905718159424),
    c64::new(0.0002926705558596507, -0.00039084846786749156),
    c64::new(0.0002920706577130578, -0.0003912969588491895),
    c64::new(0.0002914700722959944, -0.0003917445290713455),
    c64::new(0.00029086880102169603, -0.0003921911774807836),
    c64::new(0.000290266845305012, -0.0003926369030264969),
    c64::new(0.0002896642065624023, -0.00039308170465965017),
    c64::new(0.0002890608862119341, -0.00039352558133358217),
    c64::new(0.00028845688567327845, -0.0003939685320038082),
    c64::new(0.000287852206367707, -0.0003944105556280226),
    c64::new(0.00028724684971808854, -0.00039485165116610107),
    c64::new(0.0002866408171488858, -0.0003952918175801031),
    c64::new(0.0002860341100861518, -0.0003957310538342748),
    c64::new(0.00028542672995752707, -0.00039616935889505065),
    c64::new(0.00028481867819223555, -0.0003966067317310566),
    c64::new(0.0002842099562210818, -0.00039704317131311205),
    c64::new(0.0002836005654764476, -0.00039747867661423257),
    c64::new(0.00028299050739228796, -0.0003979132466096321),
    c64::new(0.00028237978340412876, -0.0003983468802767255),
    c64::new(0.0002817683949490623, -0.00039877957659513073),
    c64::new(0.00028115634346574485, -0.00039921133454667173),
    c64::new(0.0002805436303943925, -0.0003996421531153802),
    c64::new(0.0002799302571767785, -0.0004000720312874985),
    c64::new(0.0002793162252562291, -0.0004005009680514818),
    c64::new(0.0002787015360776208, -0.0004009289623980003),
    c64::new(0.0002780861910873766, -0.0004013560133199421),
    c64::new(0.0002774701917334627, -0.0004017821198124152),
    c64::new(0.0002768535394653851, -0.00040220728087274967),
    c64::new(0.00027623623573418615, -0.00040263149550050063),
    c64::new(0.00027561828199244116, -0.0004030547626974499),
    c64::new(0.000274999679694255, -0.0004034770814676091),
    c64::new(0.00027438043029525853, -0.0004038984508172212),
    c64::new(0.0002737605352526055, -0.0004043188697547635),
    c64::new(0.0002731399960249688, -0.0004047383372909496),
    c64::new(0.0002725188140725372, -0.00040515685243873196),
    c64::new(0.0002718969908570118, -0.0004055744142133038),
    c64::new(0.0002712745278416027, -0.0004059910216321021),
    c64::new(0.00027065142649102545, -0.0004064066737148094),
    c64::new(0.00027002768827149784, -0.00040682136948335605),
    c64::new(0.00026940331465073606, -0.00040723510796192286),
    c64::new(0.0002687783070979516, -0.00040764788817694336),
    c64::new(0.00026815266708384754, -0.0004080597091571057),
    c64::new(0.0002675263960806154, -0.0004084705699333554),
    c64::new(0.00026689949556193114, -0.0004088804695388974),
    c64::new(0.00026627196700295243, -0.0004092894070091982),
    c64::new(0.0002656438118803144, -0.0004096973813819886),
    c64::new(0.00026501503167212695, -0.00041010439169726534),
    c64::new(0.0002643856278579705, -0.0004105104369972939),
    c64::new(0.00026375560191889305, -0.0004109155163266105),
    c64::new(0.00026312495533740646, -0.00041131962873202434),
    c64::new(0.00026249368959748316, -0.00041172277326261984),
    c64::new(0.00026186180618455234, -0.000412124948969759),
    c64::new(0.00026122930658549677, -0.0004125261549070835),
    c64::new(0.0002605961922886489, -0.0004129263901305171),
    c64::new(0.0002599624647837881, -0.0004133256536982676),
    c64::new(0.0002593281255621363, -0.00041372394467082926),
    c64::new(0.0002586931761163549, -0.0004141212621109849),
    c64::new(0.00025805761794054125, -0.0004145176050838084),
    c64::new(0.0002574214525302253, -0.00041491297265666623),
    c64::new(0.00025678468138236565, -0.0004153073638992204),
    c64::new(0.00025614730599534617, -0.00041570077788343027),
    c64::new(0.00025550932786897287, -0.0004160932136835546),
    c64::new(0.0002548707485044699, -0.0004164846703761541),
    c64::new(0.0002542315694044762, -0.0004168751470400934),
    c64::new(0.0002535917920730418, -0.00041726464275654323),
    c64::new(0.00025295141801562473, -0.00041765315660898244),
    c64::new(0.0002523104487390869, -0.00041804068768320037),
    c64::new(0.000251668885751691, -0.00041842723506729915),
    c64::new(0.00025102673056309657, -0.00041881279785169537),
    c64::new(0.00025038398468435697, -0.00041919737512912243),
    c64::new(0.0002497406496279153, -0.00041958096599463294),
    c64::new(0.000249096726907601, -0.00041996356954560065),
    c64::new(0.0002484522180386265, -0.00042034518488172226),
    c64::new(0.0002478071245375835, -0.00042072581110502017),
    c64::new(0.0002471614479224392, -0.000421105447319844),
    c64::new(0.000246515189712533, -0.0004214840926328731),
    c64::new(0.00024586835142857303, -0.0004218617461531185),
    c64::new(0.0002452209345926323, -0.000422238406991925),
    c64::new(0.000244572940728145, -0.00042261407426297313),
    c64::new(0.00024392437135990327, -0.00042298874708228174),
    c64::new(0.00024327522801405367, -0.0004233624245682093),
    c64::new(0.0002426255122180931, -0.00042373510584145665),
    c64::new(0.0002419752255008655, -0.0004241067900250688),
    c64::new(0.0002413243693925585, -0.0004244774762444369),
    c64::new(0.0002406729454246993, -0.00042484716362730046),
    c64::new(0.00024002095513015143, -0.00042521585130374945),
    c64::new(0.00023936840004311092, -0.00042558353840622606),
    c64::new(0.00023871528169910314, -0.00042595022406952707),
    c64::new(0.00023806160163497857, -0.00042631590743080565),
    c64::new(0.00023740736138890947, -0.0004266805876295736),
    c64::new(0.0002367525625003863, -0.0004270442638077032),
    c64::new(0.00023609720651021426, -0.0004274069351094291),
    c64::new(0.00023544129496050922, -0.0004277686006813508),
    c64::new(0.00023478482939469432, -0.0004281292596724344),
    c64::new(0.0002341278113574966, -0.0004284889112340143),
    c64::new(0.00023347024239494293, -0.00042884755451979564),
    c64::new(0.00023281212405435657, -0.00042920518868585616),
    c64::new(0.00023215345788435345, -0.00042956181289064815),
    c64::new(0.00023149424543483888, -0.00042991742629500037),
    c64::new(0.00023083448825700337, -0.0004302720280621201),
    c64::new(0.00023017418790331924, -0.0004306256173575952),
    c64::new(0.00022951334592753692, -0.0004309781933493959),
    c64::new(0.00022885196388468162, -0.00043132975520787675),
    c64::new(0.00022819004333104909, -0.0004316803021057788),
    c64::new(0.0002275275858242022, -0.00043202983321823134),
    c64::new(0.00022686459292296768, -0.0004323783477227539),
    c64::new(0.00022620106618743177, -0.0004327258447992581),
    c64::new(0.00022553700717893693, -0.00043307232363004985),
    c64::new(0.00022487241746007813, -0.000433417783399831),
    c64::new(0.0002242072985946993, -0.00043376222329570135),
    c64::new(0.0002235416521478893, -0.0004341056425071604),
    c64::new(0.00022287547968597845, -0.0004344480402261098),
    c64::new(0.00022220878277653508, -0.00043478941564685443),
    c64::new(0.0002215415629883613, -0.00043512976796610507),
    c64::new(0.00022087382189148975, -0.00043546909638297985),
    c64::new(0.0002202055610571796, -0.0004358074000990062),
    c64::new(0.00021953678205791338, -0.0004361446783181227),
    c64::new(0.00021886748646739258, -0.0004364809302466813),
    c64::new(0.00021819767586053434, -0.00043681615509344853),
    c64::new(0.00021752735181346765, -0.0004371503520696082),
    c64::new(0.0002168565159035299, -0.00043748352038876226),
    c64::new(0.00021618516970926276, -0.00043781565926693367),
    c64::new(0.0002155133148104085, -0.00043814676792256754),
    c64::new(0.00021484095278790686, -0.00043847684557653314),
    c64::new(0.00021416808522389049, -0.0004388058914521259),
    c64::new(0.00021349471370168174, -0.0004391339047750691),
    c64::new(0.0002128208398057888, -0.00043946088477351576),
    c64::new(0.0002121464651219022, -0.0004397868306780502),
    c64::new(0.00021147159123689063, -0.00044011174172169044),
    c64::new(0.00021079621973879754, -0.0004404356171398892),
    c64::new(0.0002101203522168372, -0.00044075845617053654),
    c64::new(0.00020944399026139138, -0.00044108025805396085),
    c64::new(0.00020876713546400498, -0.0004414010220329313),
    c64::new(0.0002080897894173826, -0.0004417207473526593),
    c64::new(0.00020741195371538516, -0.0004420394332608002),
    c64::new(0.00020673362995302544, -0.0004423570790074553),
    c64::new(0.00020605481972646474, -0.0004426736838451735),
    c64::new(0.00020537552463300904, -0.00044298924702895304),
    c64::new(0.00020469574627110545, -0.00044330376781624324),
    c64::new(0.00020401548624033808, -0.00044361724546694635),
    c64::new(0.00020333474614142447, -0.00044392967924341913),
    c64::new(0.00020265352757621173, -0.0004442410684104747),
    c64::new(0.00020197183214767312, -0.00044455141223538434),
    c64::new(0.0002012896614599038, -0.00044486070998787903),
    c64::new(0.00020060701711811714, -0.0004451689609401513),
    c64::new(0.0001999239007286414, -0.0004454761643668568),
    c64::new(0.0001992403138989154, -0.00044578231954511626),
    c64::new(0.00019855625823748493, -0.000446087425754517),
    c64::new(0.00019787173535399893, -0.0004463914822771146),
    c64::new(0.00019718674685920612, -0.00044669448839743467),
    c64::new(0.00019650129436495046, -0.00044699644340247453),
    c64::new(0.00019581537948416787, -0.0004472973465817049),
    c64::new(0.00019512900383088222, -0.00044759719722707163),
    c64::new(0.00019444216902020196, -0.000447895994632997),
    c64::new(0.00019375487666831563, -0.000448193738096382),
    c64::new(0.00019306712839248843, -0.00044849042691660754),
    c64::new(0.00019237892581105864, -0.000448786060395536),
    c64::new(0.00019169027054343336, -0.00044908063783751336),
    c64::new(0.00019100116421008495, -0.00044937415854937035),
    c64::new(0.00019031160843254708, -0.00044966662184042457),
    c64::new(0.00018962160483341128, -0.0004499580270224814),
    c64::new(0.0001889311550363226, -0.00045024837340983646),
    c64::new(0.00018824026066597608, -0.0004505376603192766),
    c64::new(0.0001875489233481128, -0.00045082588707008185),
    c64::new(0.00018685714470951652, -0.00045111305298402673),
    c64::new(0.00018616492637800903, -0.00045139915738538213),
    c64::new(0.00018547226998244683, -0.0004516841996009168),
    c64::new(0.00018477917715271756, -0.0004519681789598988),
    c64::new(0.0001840856495197355, -0.00045225109479409735),
    c64::new(0.00018339168871543815, -0.00045253294643778406),
    c64::new(0.0001826972963727822, -0.0004528137332277349),
    c64::new(0.00018200247412574014, -0.0004530934545032314),
    c64::new(0.0001813072236092957, -0.00045337210960606225),
    c64::new(0.00018061154645944053, -0.00045364969788052527),
    c64::new(0.00017991544431317008, -0.00045392621867342833),
    c64::new(0.0001792189188084801, -0.00045420167133409134),
    c64::new(0.0001785219715843623, -0.0004544760552143475),
    c64::new(0.0001778246042808007, -0.0004547493696685452),
    c64::new(0.00017712681853876818, -0.0004550216140535489),
    c64::new(0.00017642861600022185, -0.0004552927877287414),
    c64::new(0.00017572999830809974, -0.0004555628900560248),
    c64::new(0.00017503096710631667, -0.00045583192039982235),
    c64::new(0.00017433152403976076, -0.0004560998781270795),
    c64::new(0.00017363167075428904, -0.000456366762607266),
    c64::new(0.00017293140889672388, -0.0004566325732123767),
    c64::new(0.000172230740114849, -0.0004568973093169338),
    c64::new(0.00017152966605740583, -0.00045716097029798743),
    c64::new(0.00017082818837408937, -0.00045742355553511795),
    c64::new(0.0001701263087155442, -0.000457685064410437),
    c64::new(0.00016942402873336117, -0.0004579454963085888),
    c64::new(0.00016872135008007283, -0.000458204850616752),
    c64::new(0.0001680182744091499, -0.00045846312672464086),
    c64::new(0.00016731480337499726, -0.0004587203240245068),
    c64::new(0.0001666109386329504, -0.0004589764419111396),
    c64::new(0.00016590668183927098, -0.0004592314797818693),
    c64::new(0.0001652020346511432, -0.000459485437036567),
    c64::new(0.00016449699872666995, -0.0004597383130776469),
    c64::new(0.00016379157572486897, -0.00045999010731006705),
    c64::new(0.0001630857673056687, -0.0004602408191413315),
    c64::new(0.00016237957512990438, -0.00046049044798149096),
    c64::new(0.00016167300085931466, -0.0004607389932431447),
    c64::new(0.0001609660461565369, -0.00046098645434144173),
    c64::new(0.00016025871268510384, -0.00046123283069408216),
    c64::new(0.00015955100210943932, -0.00046147812172131865),
    c64::new(0.00015884291609485497, -0.00046172232684595766),
    c64::new(0.0001581344563075454, -0.000461965445493361),
    c64::new(0.00015742562441458494, -0.00046220747709144704),
    c64::new(0.00015671642208392344, -0.000462448421070692),
    c64::new(0.0001560068509843827, -0.0004626882768641314),
    c64::new(0.00015529691278565188, -0.00046292704390736144),
    c64::new(0.0001545866091582841, -0.00046316472163854017),
    c64::new(0.0001538759417736926, -0.00046340130949838894),
    c64::new(0.00015316491230414625, -0.0004636368069301937),
    c64::new(0.00015245352242276614, -0.00046387121337980615),
    c64::new(0.00015174177380352123, -0.00046410452829564536),
    c64::new(0.00015102966812122496, -0.00046433675112869877),
    c64::new(0.00015031720705153075, -0.0004645678813325236),
    c64::new(0.00014960439227092826, -0.00046479791836324816),
    c64::new(0.00014889122545673944, -0.0004650268616795732),
    c64::new(0.00014817770828711493, -0.0004652547107427728),
    c64::new(0.0001474638424410294, -0.00046548146501669617),
    c64::new(0.00014674962959827803, -0.0004657071239677687),
    c64::new(0.0001460350714394729, -0.000465931687064993),
    c64::new(0.00014532016964603824, -0.00046615515377995036),
    c64::new(0.000144604925900207, -0.0004663775235868021),
    c64::new(0.00014388934188501676, -0.0004665987959622906),
    c64::new(0.000143173419284306, -0.00046681897038574043),
    c64::new(0.00014245715978270965, -0.0004670380463390601),
    c64::new(0.0001417405650656554, -0.00046725602330674264),
    c64::new(0.0001410236368193601, -0.0004674729007758671),
    c64::new(0.00014030637673082498, -0.00046768867823610005),
    c64::new(0.00013958878648783239, -0.00046790335517969613),
    c64::new(0.00013887086777894132, -0.0004681169311014998),
    c64::new(0.00013815262229348408, -0.00046832940549894624),
    c64::new(0.00013743405172156155, -0.00046854077787206275),
    c64::new(0.00013671515775403974, -0.00046875104772346965),
    c64::new(0.00013599594208254544, -0.00046896021455838173),
    c64::new(0.00013527640639946287, -0.0004691682778846092),
    c64::new(0.00013455655239792884, -0.0004693752372125589),
    c64::new(0.00013383638177182926, -0.0004695810920552356),
    c64::new(0.0001331158962157954, -0.00046978584192824296),
    c64::new(0.00013239509742519925, -0.00046998948634978475),
    c64::new(0.00013167398709615, -0.00047019202484066596),
    c64::new(0.0001309525669254898, -0.00047039345692429397),
    c64::new(0.00013023083861079024, -0.0004705937821266796),
    c64::new(0.0001295088038503476, -0.00047079299997643834),
    c64::new(0.00012878646434317945, -0.00047099111000479133),
    c64::new(0.0001280638217890203, -0.00047118811174556656),
    c64::new(0.00012734087788831815, -0.0004713840047351999),
    c64::new(0.00012661763434222966, -0.0004715787885127362),
    c64::new(0.00012589409285261667, -0.0004717724626198304),
    c64::new(0.00012517025512204244, -0.0004719650266007486),
    c64::new(0.0001244461228537669, -0.0004721564800023692),
    c64::new(0.0001237216977517433, -0.00047234682237418384),
    c64::new(0.00012299698152061373, -0.0004725360532682986),
    c64::new(0.0001222719758657057, -0.0004727241722394348),
    c64::new(0.00012154668249302747, -0.0004729111788449303),
    c64::new(0.00012082110310926437, -0.00047309707264474066),
    c64::new(0.0001200952394217747, -0.0004732818532014397),
    c64::new(0.00011936909313858604, -0.0004734655200802209),
    c64::new(0.00011864266596839062, -0.0004736480728488984),
    c64::new(0.00011791595962054164, -0.00047382951107790796),
    c64::new(0.00011718897580504956, -0.0004740098343403078),
    c64::new(0.00011646171623257739, -0.00047418904221177983),
    c64::new(0.00011573418261443718, -0.00047436713427063075),
    c64::new(0.00011500637666258567, -0.0004745441100977927),
    c64::new(0.00011427830008962083, -0.00047471996927682435),
    c64::new(0.00011354995460877702, -0.0004748947113939122),
    c64::new(0.00011282134193392151, -0.0004750683360378711),
    c64::new(0.00011209246377955023, -0.00047524084280014563),
    c64::new(0.00011136332186078408, -0.0004754122312748105),
    c64::new(0.00011063391789336431, -0.0004755825010585722),
    c64::new(0.00010990425359364882, -0.00047575165175076946),
    c64::new(0.00010917433067860839, -0.00047591968295337435),
    c64::new(0.00010844415086582207, -0.0004760865942709932),
    c64::new(0.00010771371587347346, -0.0004762523853108676),
    c64::new(0.00010698302742034657, -0.0004764170556828753),
    c64::new(0.0001062520872258221, -0.0004765806049995308),
    c64::new(0.00010552089700987285, -0.0004767430328759869),
    c64::new(0.00010478945849306002, -0.0004769043389300351),
    c64::new(0.00010405777339652899, -0.00047706452278210664),
    c64::new(0.00010332584344200567, -0.0004772235840552734),
    c64::new(0.00010259367035179186, -0.000477381522375249),
    c64::new(0.00010186125584876144, -0.00047753833737038923),
    c64::new(0.00010112860165635672, -0.0004776940286716932),
    c64::new(0.00010039570949858365, -0.00047784859591280425),
    c64::new(9.96625811000083e-05, -0.00047800203873001084),
    c64::new(9.892921818575249e-05, -0.0004781543567622471),
    c64::new(9.81956224814902e-05, -0.000478305549651094),
    c64::new(9.74617957134429e-05, -0.0004784556170407801),
    c64::new(9.672773960837584e-05, -0.0004786045585781823),
    c64::new(9.599345589359386e-05, -0.00047875237391282685),
    c64::new(9.525894629693765e-05, -0.00047889906269688986),
    c64::new(9.452421254677916e-05, -0.00047904462458519853),
    c64::new(9.378925637201776e-05, -0.0004791890592352317),
    c64::new(9.305407950207655e-05, -0.00047933236630712066),
    c64::new(9.23186836668976e-05, -0.00047947454546365),
    c64::new(9.158307059693835e-05, -0.0004796155963702585),
    c64::new(9.084724202316729e-05, -0.00047975551869503967),
    c64::new(9.011119967706037e-05, -0.0004798943121087428),
    c64::new(8.937494529059621e-05, -0.0004800319762847736),
    c64::new(8.86384805962525e-05, -0.0004801685108991949),
    c64::new(8.790180732700172e-05, -0.00048030391563072766),
    c64::new(8.71649272163074e-05, -0.00048043819016075143),
    c64::new(8.642784199811956e-05, -0.0004805713341733052),
    c64::new(8.569055340687079e-05, -0.00048070334735508835),
    c64::new(8.495306317747258e-05, -0.00048083422939546104),
    c64::new(8.421537304531056e-05, -0.00048096397998644523),
    c64::new(8.34774847462409e-05, -0.0004810925988227252),
    c64::new(8.27394000165859e-05, -0.00048122008560164847),
    c64::new(8.200112059313047e-05, -0.00048134644002322625),
    c64::new(8.126264821311729e-05, -0.0004814716617901345),
    c64::new(8.05239846142432e-05, -0.0004815957506077142),
    c64::new(7.97851315346549e-05, -0.00048171870618397237),
    c64::new(7.904609071294525e-05, -0.0004818405282295827),
    c64::new(7.830686388814863e-05, -0.0004819612164578862),
    c64::new(7.756745279973701e-05, -0.0004820807705848918),
    c64::new(7.682785918761635e-05, -0.00048219919032927697),
    c64::new(7.608808479212174e-05, -0.00048231647541238876),
    c64::new(7.534813135401379e-05, -0.0004824326255582439),
    c64::new(7.460800061447432e-05, -0.00048254764049353),
    c64::new(7.386769431510266e-05, -0.00048266151994760576),
    c64::new(7.312721419791095e-05, -0.00048277426365250185),
    c64::new(7.238656200532041e-05, -0.0004828858713429214),
    c64::new(7.16457394801571e-05, -0.00048299634275624073),
    c64::new(7.09047483656482e-05, -0.00048310567763250995),
    c64::new(7.016359040541728e-05, -0.0004832138757144535),
    c64::new(6.942226734348047e-05, -0.00048332093674747096),
    c64::new(6.868078092424279e-05, -0.00048342686047963727),
    c64::new(6.79391328924933e-05, -0.00048353164666170367),
    c64::new(6.719732499340145e-05, -0.00048363529504709814),
    c64::new(6.645535897251278e-05, -0.0004837378053919261),
    c64::new(6.571323657574523e-05, -0.0004838391774549707),
    c64::new(6.497095954938436e-05, -0.00048393941099769374),
    c64::new(6.422852964007971e-05, -0.00048403850578423606),
    c64::new(6.34859485948405e-05, -0.0004841364615814179),
    c64::new(6.274321816103185e-05, -0.00048423327815874),
    c64::new(6.20003400863701e-05, -0.00048432895528838346),
    c64::new(6.125731611891905e-05, -0.00048442349274521084),
    c64::new(6.051414800708603e-05, -0.0004845168903067663),
    c64::new(5.9770837499617324e-05, -0.00048460914775327635),
    c64::new(5.902738634559435e-05, -0.0004847002648676504),
    c64::new(5.8283796294429366e-05, -0.000484790241435481),
    c64::new(5.7540069095861826e-05, -0.00048487907724504463),
    c64::new(5.6796206499953547e-05, -0.000484966772087302),
    c64::new(5.605221025708508e-05, -0.00048505332575589876),
    c64::new(5.530808211795134e-05, -0.0004851387380471657),
    c64::new(5.456382383355794e-05, -0.00048522300876011943),
    c64::new(5.38194371552164e-05, -0.0004853061376964629),
    c64::new(5.307492383454037e-05, -0.00048538812466058563),
    c64::new(5.233028562344183e-05, -0.0004854689694595644),
    c64::new(5.158552427412632e-05, -0.0004855486719031636),
    c64::new(5.084064153908922e-05, -0.00048562723180383565),
    c64::new(5.009563917111146e-05, -0.00048570464897672153),
    c64::new(4.935051892325579e-05, -0.0004857809232396512),
    c64::new(4.8605282548862026e-05, -0.00048585605441314386),
    c64::new(4.7859931801543345e-05, -0.0004859300423204086),
    c64::new(4.711446843518193e-05, -0.00048600288678734486),
    c64::new(4.636889420392532e-05, -0.00048607458764254245),
    c64::new(4.56232108621816e-05, -0.0004861451447172824),
    c64::new(4.4877420164615574e-05, -0.000486214557845537),
    c64::new(4.4131523866145025e-05, -0.00048628282686397057),
    c64::new(4.338552372193588e-05, -0.00048634995161193944),
    c64::new(4.263942148739854e-05, -0.0004864159319314926),
    c64::new(4.1893218918183535e-05, -0.00048648076766737207),
    c64::new(4.114691777017779e-05, -0.000486544458667013),
    c64::new(4.0400519799499904e-05, -0.00048660700478054435),
    c64::new(3.965402676249628e-05, -0.0004866684058607891),
    c64::new(3.8907440415737366e-05, -0.0004867286617632645),
    c64::new(3.816076251601284e-05, -0.0004867877723461827),
    c64::new(3.741399482032794e-05, -0.00048684573747045067),
    c64::new(3.666713908589906e-05, -0.000486902556999671),
    c64::new(3.5920197070150124e-05, -0.0004869582308001417),
    c64::new(3.517317053070772e-05, -0.0004870127587408571),
    c64::new(3.4426061225397465e-05, -0.00048706614069350755),
    c64::new(3.3678870912239615e-05, -0.00048711837653248025),
    c64::new(3.2931601349445364e-05, -0.0004871694661348592),
    c64::new(3.218425429541207e-05, -0.0004872194093804256),
    c64::new(3.143683150871944e-05, -0.0004872682061516582),
    c64::new(3.0689334748125693e-05, -0.0004873158563337335),
    c64::new(2.994176577256282e-05, -0.00048736235981452597),
    c64::new(2.9194126341132816e-05, -0.0004874077164846085),
    c64::new(2.8446418213103385e-05, -0.0004874519262372524),
    c64::new(2.769864314790419e-05, -0.0004874949889684278),
    c64::new(2.6950802905122086e-05, -0.0004875369045768041),
    c64::new(2.6202899244497402e-05, -0.00048757767296374966),
    c64::new(2.5454933925919588e-05, -0.00048761729403333265),
    c64::new(2.4706908709423504e-05, -0.00048765576769232077),
    c64::new(2.3958825355184635e-05, -0.00048769309385018184),
    c64::new(2.3210685623515264e-05, -0.00048772927241908377),
    c64::new(2.2462491274860666e-05, -0.00048776430331389496),
    c64::new(2.171424406979431e-05, -0.00048779818645218427),
    c64::new(2.096594576901414e-05, -0.00048783092175422137),
    c64::new(2.021759813333824e-05, -0.000487862509142977),
    c64::new(1.9469202923701096e-05, -0.000487892948544123),
    c64::new(1.8720761901148824e-05, -0.00048792223988603243),
    c64::new(1.7972276826835445e-05, -0.00048795038309978005),
    c64::new(1.722374946201853e-05, -0.000487977378119142),
    c64::new(1.647518156805549e-05, -0.00048800322488059655),
    c64::new(1.572657490639878e-05, -0.0004880279233233237),
    c64::new(1.4977931238592082e-05, -0.00048805147338920564),
    c64::new(1.4229252326266476e-05, -0.00048807387502282677),
    c64::new(1.3480539931135654e-05, -0.00048809512817147394),
    c64::new(1.2731795814992207e-05, -0.00048811523278513634),
    c64::new(1.1983021739703256e-05, -0.000488134188816506),
    c64::new(1.1234219467206743e-05, -0.0004881519962209774),
    c64::new(1.0485390759506645e-05, -0.0004881686549566481),
    c64::new(9.736537378669239e-06, -0.0004881841649843185),
    c64::new(8.98766108681876e-06, -0.0004881985262674919),
    c64::new(8.238763646133678e-06, -0.00048821173877237493),
    c64::new(7.489846818841904e-06, -0.0004882238024678772),
    c64::new(6.740912367216967e-06, -0.00048823471732561177),
    c64::new(5.991962053574192e-06, -0.0004882444833198948),
    c64::new(5.2429976402659035e-06, -0.00048825310042774605),
    c64::new(4.494020889677715e-06, -0.0004882605686288886),
    c64::new(3.7450335642241587e-06, -0.0004882668879057491),
    c64::new(2.9960374263449785e-06, -0.00048827205824345757),
    c64::new(2.247034238500328e-06, -0.00048827607962984785),
    c64::new(1.4980257631670595e-06, -0.0004882789520554571),
    c64::new(7.490137628343584e-07, -0.0004882806755135263),
    c64::new(2.989860349480843e-20, -0.00048828125),
    c64::new(-7.490137628342987e-07, -0.0004882806755135263),
    c64::new(-1.4980257631669998e-06, -0.0004882789520554571),
    c64::new(-2.247034238500268e-06, -0.00048827607962984785),
    c64::new(-2.9960374263449184e-06, -0.00048827205824345757),
    c64::new(-3.7450335642240986e-06, -0.0004882668879057491),
    c64::new(-4.4940208896776555e-06, -0.0004882605686288886),
    c64::new(-5.242997640265844e-06, -0.00048825310042774605),
    c64::new(-5.991962053574132e-06, -0.0004882444833198948),
    c64::new(-6.740912367216908e-06, -0.00048823471732561177),
    c64::new(-7.489846818841845e-06, -0.0004882238024678772),
    c64::new(-8.238763646133617e-06, -0.00048821173877237493),
    c64::new(-8.987661086818699e-06, -0.0004881985262674919),
    c64::new(-9.736537378669178e-06, -0.0004881841649843185),
    c64::new(-1.0485390759506584e-05, -0.0004881686549566481),
    c64::new(-1.1234219467206682e-05, -0.0004881519962209774),
    c64::new(-1.1983021739703195e-05, -0.000488134188816506),
    c64::new(-1.2731795814992146e-05, -0.00048811523278513634),
    c64::new(-1.3480539931135593e-05, -0.00048809512817147394),
    c64::new(-1.4229252326266415e-05, -0.00048807387502282677),
    c64::new(-1.4977931238592021e-05, -0.00048805147338920564),
    c64::new(-1.572657490639872e-05, -0.0004880279233233237),
    c64::new(-1.647518156805543e-05, -0.00048800322488059655),
    c64::new(-1.7223749462018468e-05, -0.000487977378119142),
    c64::new(-1.7972276826835384e-05, -0.00048795038309978005),
    c64::new(-1.8720761901148763e-05, -0.00048792223988603243),
    c64::new(-1.9469202923701035e-05, -0.000487892948544123),
    c64::new(-2.021759813333818e-05, -0.000487862509142977),
    c64::new(-2.096594576901408e-05, -0.00048783092175422137),
    c64::new(-2.1714244069794248e-05, -0.00048779818645218427),
    c64::new(-2.2462491274860605e-05, -0.00048776430331389496),
    c64::new(-2.3210685623515203e-05, -0.00048772927241908377),
    c64::new(-2.3958825355184574e-05, -0.00048769309385018184),
    c64::new(-2.4706908709423443e-05, -0.00048765576769232077),
    c64::new(-2.5454933925919527e-05, -0.00048761729403333265),
    c64::new(-2.620289924449734e-05, -0.00048757767296374966),
    c64::new(-2.6950802905122028e-05, -0.0004875369045768041),
    c64::new(-2.769864314790413e-05, -0.0004874949889684278),
    c64::new(-2.8446418213103324e-05, -0.0004874519262372524),
    c64::new(-2.9194126341132755e-05, -0.0004874077164846085),
    c64::new(-2.994176577256276e-05, -0.00048736235981452597),
    c64::new(-3.068933474812564e-05, -0.0004873158563337335),
    c64::new(-3.1436831508719384e-05, -0.0004872682061516582),
    c64::new(-3.2184254295412014e-05, -0.0004872194093804256),
    c64::new(-3.293160134944531e-05, -0.0004871694661348592),
    c64::new(-3.367887091223956e-05, -0.00048711837653248025),
    c64::new(-3.442606122539741e-05, -0.00048706614069350755),
    c64::new(-3.517317053070767e-05, -0.0004870127587408571),
    c64::new(-3.592019707015007e-05, -0.0004869582308001417),
    c64::new(-3.666713908589901e-05, -0.000486902556999671),
    c64::new(-3.7413994820327884e-05, -0.00048684573747045067),
    c64::new(-3.816076251601279e-05, -0.0004867877723461827),
    c64::new(-3.890744041573731e-05, -0.0004867286617632645),
    c64::new(-3.965402676249623e-05, -0.0004866684058607891),
    c64::new(-4.040051979949985e-05, -0.00048660700478054435),
    c64::new(-4.1146917770177735e-05, -0.000486544458667013),
    c64::new(-4.189321891818348e-05, -0.00048648076766737207),
    c64::new(-4.263942148739849e-05, -0.0004864159319314926),
    c64::new(-4.3385523721935826e-05, -0.00048634995161193944),
    c64::new(-4.413152386614497e-05, -0.00048628282686397057),
    c64::new(-4.487742016461552e-05, -0.000486214557845537),
    c64::new(-4.562321086218154e-05, -0.0004861451447172824),
    c64::new(-4.636889420392527e-05, -0.00048607458764254245),
    c64::new(-4.7114468435181876e-05, -0.00048600288678734486),
    c64::new(-4.785993180154329e-05, -0.00048593004232040866),
    c64::new(-4.860528254886197e-05, -0.00048585605441314386),
    c64::new(-4.935051892325574e-05, -0.0004857809232396512),
    c64::new(-5.0095639171111404e-05, -0.00048570464897672153),
    c64::new(-5.084064153908917e-05, -0.00048562723180383565),
    c64::new(-5.1585524274126264e-05, -0.0004855486719031636),
    c64::new(-5.233028562344178e-05, -0.0004854689694595644),
    c64::new(-5.3074923834540314e-05, -0.0004853881246605857),
    c64::new(-5.3819437155216344e-05, -0.0004853061376964629),
    c64::new(-5.4563823833557883e-05, -0.00048522300876011943),
    c64::new(-5.530808211795129e-05, -0.0004851387380471657),
    c64::new(-5.605221025708503e-05, -0.00048505332575589876),
    c64::new(-5.679620649995349e-05, -0.000484966772087302),
    c64::new(-5.754006909586177e-05, -0.00048487907724504463),
    c64::new(-5.828379629442931e-05, -0.000484790241435481),
    c64::new(-5.902738634559429e-05, -0.0004847002648676504),
    c64::new(-5.977083749961727e-05, -0.00048460914775327635),
    c64::new(-6.0514148007085975e-05, -0.0004845168903067663),
    c64::new(-6.125731611891899e-05, -0.00048442349274521084),
    c64::new(-6.200034008637005e-05, -0.00048432895528838346),
    c64::new(-6.27432181610318e-05, -0.00048423327815874),
    c64::new(-6.348594859484045e-05, -0.00048413646158141797),
    c64::new(-6.422852964007967e-05, -0.00048403850578423606),
    c64::new(-6.497095954938431e-05, -0.0004839394109976938),
    c64::new(-6.571323657574517e-05, -0.0004838391774549707),
    c64::new(-6.645535897251273e-05, -0.0004837378053919261),
    c64::new(-6.71973249934014e-05, -0.00048363529504709814),
    c64::new(-6.793913289249325e-05, -0.00048353164666170367),
    c64::new(-6.868078092424274e-05, -0.00048342686047963727),
    c64::new(-6.942226734348042e-05, -0.00048332093674747096),
    c64::new(-7.016359040541722e-05, -0.00048321387571445356),
    c64::new(-7.090474836564815e-05, -0.00048310567763250995),
    c64::new(-7.164573948015705e-05, -0.00048299634275624073),
    c64::new(-7.238656200532035e-05, -0.0004828858713429214),
    c64::new(-7.312721419791089e-05, -0.00048277426365250185),
    c64::new(-7.386769431510261e-05, -0.0004826615199476058),
    c64::new(-7.460800061447427e-05, -0.00048254764049353),
    c64::new(-7.534813135401374e-05, -0.0004824326255582439),
    c64::new(-7.608808479212168e-05, -0.00048231647541238876),
    c64::new(-7.682785918761629e-05, -0.000482199190329277),
    c64::new(-7.756745279973695e-05, -0.0004820807705848918),
    c64::new(-7.830686388814858e-05, -0.0004819612164578862),
    c64::new(-7.90460907129452e-05, -0.0004818405282295828),
    c64::new(-7.978513153465484e-05, -0.0004817187061839724),
    c64::new(-8.052398461424314e-05, -0.0004815957506077142),
    c64::new(-8.126264821311724e-05, -0.0004814716617901345),
    c64::new(-8.200112059313041e-05, -0.0004813464400232263),
    c64::new(-8.273940001658585e-05, -0.00048122008560164847),
    c64::new(-8.347748474624084e-05, -0.0004810925988227252),
    c64::new(-8.421537304531051e-05, -0.00048096397998644523),
    c64::new(-8.495306317747253e-05, -0.00048083422939546104),
    c64::new(-8.569055340687073e-05, -0.00048070334735508835),
    c64::new(-8.64278419981195e-05, -0.0004805713341733052),
    c64::new(-8.716492721630735e-05, -0.00048043819016075143),
    c64::new(-8.790180732700166e-05, -0.00048030391563072766),
    c64::new(-8.863848059625245e-05, -0.0004801685108991949),
    c64::new(-8.937494529059617e-05, -0.0004800319762847736),
    c64::new(-9.011119967706032e-05, -0.00047989431210874285),
    c64::new(-9.084724202316725e-05, -0.00047975551869503967),
    c64::new(-9.15830705969383e-05, -0.0004796155963702585),
    c64::new(-9.231868366689754e-05, -0.00047947454546365),
    c64::new(-9.305407950207649e-05, -0.00047933236630712066),
    c64::new(-9.378925637201771e-05, -0.0004791890592352317),
    c64::new(-9.45242125467791e-05, -0.0004790446245851986),
    c64::new(-9.52589462969376e-05, -0.00047889906269688986),
    c64::new(-9.59934558935938e-05, -0.00047875237391282685),
    c64::new(-9.672773960837579e-05, -0.00047860455857818234),
    c64::new(-9.746179571344284e-05, -0.0004784556170407801),
    c64::new(-9.819562248149015e-05, -0.000478305549651094),
    c64::new(-9.892921818575244e-05, -0.00047815435676224713),
    c64::new(-9.966258110000825e-05, -0.00047800203873001084),
    c64::new(-0.0001003957094985836, -0.0004778485959128043),
    c64::new(-0.00010112860165635666, -0.0004776940286716932),
    c64::new(-0.00010186125584876139, -0.00047753833737038923),
    c64::new(-0.0001025936703517918, -0.000477381522375249),
    c64::new(-0.00010332584344200561, -0.00047722358405527347),
    c64::new(-0.00010405777339652893, -0.00047706452278210664),
    c64::new(-0.00010478945849305997, -0.0004769043389300351),
    c64::new(-0.0001055208970098728, -0.0004767430328759869),
    c64::new(-0.00010625208722582205, -0.00047658060499953083),
    c64::new(-0.00010698302742034651, -0.0004764170556828753),
    c64::new(-0.0001077137158734734, -0.00047625238531086766),
    c64::new(-0.00010844415086582201, -0.0004760865942709932),
    c64::new(-0.00010917433067860835, -0.00047591968295337435),
    c64::new(-0.00010990425359364876, -0.00047575165175076946),
    c64::new(-0.00011063391789336426, -0.0004755825010585722),
    c64::new(-0.00011136332186078402, -0.00047541223127481054),
    c64::new(-0.00011209246377955017, -0.00047524084280014563),
    c64::new(-0.00011282134193392147, -0.0004750683360378712),
    c64::new(-0.00011354995460877696, -0.0004748947113939122),
    c64::new(-0.00011427830008962077, -0.00047471996927682435),
    c64::new(-0.00011500637666258562, -0.0004745441100977927),
    c64::new(-0.00011573418261443712, -0.00047436713427063075),
    c64::new(-0.00011646171623257734, -0.00047418904221177983),
    c64::new(-0.0001171889758050495, -0.0004740098343403078),
    c64::new(-0.0001179159596205416, -0.00047382951107790796),
    c64::new(-0.00011864266596839056, -0.0004736480728488984),
    c64::new(-0.00011936909313858598, -0.0004734655200802209),
    c64::new(-0.00012009523942177465, -0.0004732818532014397),
    c64::new(-0.00012082110310926432, -0.00047309707264474066),
    c64::new(-0.00012154668249302742, -0.0004729111788449303),
    c64::new(-0.00012227197586570565, -0.0004727241722394348),
    c64::new(-0.00012299698152061367, -0.0004725360532682986),
    c64::new(-0.00012372169775174324, -0.00047234682237418384),
    c64::new(-0.00012444612285376685, -0.0004721564800023692),
    c64::new(-0.00012517025512204238, -0.0004719650266007486),
    c64::new(-0.0001258940928526166, -0.0004717724626198304),
    c64::new(-0.0001266176343422296, -0.0004715787885127362),
    c64::new(-0.0001273408778883181, -0.0004713840047351999),
    c64::new(-0.00012806382178902025, -0.0004711881117455666),
    c64::new(-0.0001287864643431794, -0.0004709911100047914),
    c64::new(-0.00012950880385034755, -0.0004707929999764384),
    c64::new(-0.0001302308386107902, -0.0004705937821266796),
    c64::new(-0.00013095256692548978, -0.00047039345692429397),
    c64::new(-0.00013167398709614995, -0.00047019202484066596),
    c64::new(-0.0001323950974251992, -0.00046998948634978475),
    c64::new(-0.00013311589621579534, -0.00046978584192824296),
    c64::new(-0.0001338363817718292, -0.0004695810920552356),
    c64::new(-0.00013455655239792878, -0.0004693752372125589),
    c64::new(-0.0001352764063994628, -0.0004691682778846092),
    c64::new(-0.00013599594208254539, -0.0004689602145583818),
    c64::new(-0.00013671515775403968, -0.00046875104772346965),
    c64::new(-0.0001374340517215615, -0.00046854077787206275),
    c64::new(-0.00013815262229348402, -0.0004683294054989463),
    c64::new(-0.00013887086777894126, -0.0004681169311014998),
    c64::new(-0.00013958878648783233, -0.00046790335517969613),
    c64::new(-0.00014030637673082493, -0.00046768867823610005),
    c64::new(-0.00014102363681936004, -0.00046747290077586717),
    c64::new(-0.00014174056506565538, -0.00046725602330674264),
    c64::new(-0.0001424571597827096, -0.0004670380463390601),
    c64::new(-0.00014317341928430594, -0.0004668189703857405),
    c64::new(-0.0001438893418850167, -0.0004665987959622906),
    c64::new(-0.00014460492590020696, -0.0004663775235868021),
    c64::new(-0.00014532016964603818, -0.00046615515377995036),
    c64::new(-0.00014603507143947283, -0.000465931687064993),
    c64::new(-0.00014674962959827797, -0.0004657071239677687),
    c64::new(-0.00014746384244102934, -0.0004654814650166962),
    c64::new(-0.00014817770828711487, -0.0004652547107427728),
    c64::new(-0.0001488912254567394, -0.0004650268616795732),
    c64::new(-0.0001496043922709282, -0.0004647979183632482),
    c64::new(-0.0001503172070515307, -0.0004645678813325236),
    c64::new(-0.00015102966812122494, -0.00046433675112869877),
    c64::new(-0.00015174177380352117, -0.0004641045282956454),
    c64::new(-0.00015245352242276608, -0.00046387121337980615),
    c64::new(-0.00015316491230414622, -0.0004636368069301937),
    c64::new(-0.00015387594177369255, -0.00046340130949838894),
    c64::new(-0.00015458660915828405, -0.0004631647216385402),
    c64::new(-0.00015529691278565183, -0.00046292704390736144),
    c64::new(-0.00015600685098438263, -0.00046268827686413145),
    c64::new(-0.0001567164220839234, -0.000462448421070692),
    c64::new(-0.00015742562441458492, -0.00046220747709144704),
    c64::new(-0.00015813445630754534, -0.00046196544549336106),
    c64::new(-0.00015884291609485492, -0.0004617223268459577),
    c64::new(-0.0001595510021094393, -0.00046147812172131865),
    c64::new(-0.00016025871268510379, -0.00046123283069408216),
    c64::new(-0.00016096604615653683, -0.00046098645434144173),
    c64::new(-0.0001616730008593146, -0.0004607389932431447),
    c64::new(-0.00016237957512990435, -0.00046049044798149096),
    c64::new(-0.00016308576730566864, -0.0004602408191413315),
    c64::new(-0.00016379157572486892, -0.0004599901073100671),
    c64::new(-0.0001644969987266699, -0.0004597383130776469),
    c64::new(-0.00016520203465114316, -0.000459485437036567),
    c64::new(-0.00016590668183927092, -0.0004592314797818693),
    c64::new(-0.00016661093863295034, -0.00045897644191113967),
    c64::new(-0.0001673148033749972, -0.0004587203240245068),
    c64::new(-0.00016801827440914984, -0.00045846312672464086),
    c64::new(-0.00016872135008007277, -0.00045820485061675204),
    c64::new(-0.00016942402873336112, -0.0004579454963085888),
    c64::new(-0.00017012630871554414, -0.000457685064410437),
    c64::new(-0.00017082818837408932, -0.00045742355553511795),
    c64::new(-0.00017152966605740578, -0.00045716097029798743),
    c64::new(-0.00017223074011484893, -0.0004568973093169338),
    c64::new(-0.00017293140889672383, -0.00045663257321237676),
    c64::new(-0.000173631670754289, -0.000456366762607266),
    c64::new(-0.0001743315240397607, -0.0004560998781270796),
    c64::new(-0.00017503096710631662, -0.0004558319203998224),
    c64::new(-0.0001757299983080997, -0.0004555628900560248),
    c64::new(-0.0001764286160002218, -0.00045529278772874144),
    c64::new(-0.00017712681853876812, -0.0004550216140535489),
    c64::new(-0.00017782460428080065, -0.0004547493696685452),
    c64::new(-0.00017852197158436223, -0.0004544760552143475),
    c64::new(-0.00017921891880848004, -0.00045420167133409134),
    c64::new(-0.00017991544431317002, -0.0004539262186734284),
    c64::new(-0.0001806115464594405, -0.00045364969788052527),
    c64::new(-0.00018130722360929568, -0.0004533721096060623),
    c64::new(-0.0001820024741257401, -0.0004530934545032314),
    c64::new(-0.00018269729637278217, -0.0004528137332277349),
    c64::new(-0.0001833916887154381, -0.00045253294643778406),
    c64::new(-0.00018408564951973545, -0.00045225109479409735),
    c64::new(-0.00018477917715271753, -0.00045196817895989884),
    c64::new(-0.00018547226998244677, -0.00045168419960091684),
    c64::new(-0.00018616492637800897, -0.00045139915738538213),
    c64::new(-0.00018685714470951647, -0.00045111305298402673),
    c64::new(-0.00018754892334811276, -0.00045082588707008185),
    c64::new(-0.00018824026066597603, -0.0004505376603192766),
    c64::new(-0.00018893115503632255, -0.00045024837340983646),
    c64::new(-0.00018962160483341123, -0.00044995802702248145),
    c64::new(-0.00019031160843254702, -0.00044966662184042457),
    c64::new(-0.0001910011642100849, -0.0004493741585493704),
    c64::new(-0.00019169027054343333, -0.00044908063783751336),
    c64::new(-0.0001923789258110586, -0.000448786060395536),
    c64::new(-0.00019306712839248837, -0.00044849042691660754),
    c64::new(-0.00019375487666831558, -0.0004481937380963821),
    c64::new(-0.00019444216902020193, -0.0004478959946329971),
    c64::new(-0.0001951290038308822, -0.00044759719722707163),
    c64::new(-0.0001958153794841678, -0.00044729734658170494),
    c64::new(-0.0001965012943649504, -0.00044699644340247453),
    c64::new(-0.00019718674685920606, -0.00044669448839743467),
    c64::new(-0.0001978717353539989, -0.0004463914822771146),
    c64::new(-0.00019855625823748488, -0.000446087425754517),
    c64::new(-0.00019924031389891535, -0.0004457823195451163),
    c64::new(-0.00019992390072864138, -0.00044547616436685686),
    c64::new(-0.0002006070171181171, -0.0004451689609401513),
    c64::new(-0.00020128966145990375, -0.0004448607099878791),
    c64::new(-0.0002019718321476731, -0.0004445514122353844),
    c64::new(-0.00020265352757621168, -0.00044424106841047477),
    c64::new(-0.0002033347461414243, -0.0004439296792434192),
    c64::new(-0.00020401548624033802, -0.00044361724546694635),
    c64::new(-0.0002046957462711055, -0.00044330376781624324),
    c64::new(-0.00020537552463300898, -0.0004429892470289531),
    c64::new(-0.0002060548197264647, -0.0004426736838451735),
    c64::new(-0.00020673362995302528, -0.00044235707900745536),
    c64::new(-0.00020741195371538513, -0.0004420394332608002),
    c64::new(-0.00020808978941738265, -0.0004417207473526593),
    c64::new(-0.00020876713546400482, -0.00044140102203293137),
    c64::new(-0.00020944399026139133, -0.00044108025805396085),
    c64::new(-0.00021012035221683727, -0.00044075845617053654),
    c64::new(-0.0002107962197387974, -0.0004404356171398893),
    c64::new(-0.0002114715912368906, -0.00044011174172169044),
    c64::new(-0.00021214646512190225, -0.0004397868306780502),
    c64::new(-0.00021282083980578875, -0.00043946088477351576),
    c64::new(-0.00021349471370168168, -0.00043913390477506913),
    c64::new(-0.00021416808522389032, -0.00043880589145212594),
    c64::new(-0.0002148409527879068, -0.00043847684557653314),
    c64::new(-0.00021551331481040856, -0.0004381467679225675),
    c64::new(-0.0002161851697092626, -0.0004378156592669337),
    c64::new(-0.00021685651590352986, -0.0004374835203887623),
    c64::new(-0.0002175273518134677, -0.00043715035206960814),
    c64::new(-0.00021819767586053418, -0.00043681615509344864),
    c64::new(-0.00021886748646739252, -0.0004364809302466813),
    c64::new(-0.00021953678205791343, -0.00043614467831812266),
    c64::new(-0.00022020556105717957, -0.0004358074000990062),
    c64::new(-0.0002208738218914897, -0.00043546909638297985),
    c64::new(-0.00022154156298836116, -0.0004351297679661052),
    c64::new(-0.00022220878277653502, -0.0004347894156468545),
    c64::new(-0.0002228754796859785, -0.00043444804022610975),
    c64::new(-0.00022354165214788916, -0.0004341056425071605),
    c64::new(-0.00022420729859469924, -0.00043376222329570135),
    c64::new(-0.00022487241746007818, -0.000433417783399831),
    c64::new(-0.0002255370071789368, -0.00043307232363004995),
    c64::new(-0.0002262010661874317, -0.0004327258447992581),
    c64::new(-0.00022686459292296774, -0.00043237834772275383),
    c64::new(-0.00022752758582420216, -0.0004320298332182314),
    c64::new(-0.00022819004333104906, -0.0004316803021057788),
    c64::new(-0.00022885196388468148, -0.0004313297552078768),
    c64::new(-0.00022951334592753687, -0.0004309781933493959),
    c64::new(-0.0002301741879033192, -0.00043062561735759524),
    c64::new(-0.0002308344882570032, -0.0004302720280621202),
    c64::new(-0.00023149424543483885, -0.0004299174262950004),
    c64::new(-0.0002321534578843535, -0.00042956181289064815),
    c64::new(-0.00023281212405435643, -0.0004292051886858562),
    c64::new(-0.00023347024239494288, -0.0004288475545197957),
    c64::new(-0.0002341278113574966, -0.0004284889112340143),
    c64::new(-0.00023478482939469423, -0.00042812925967243445),
    c64::new(-0.00023544129496050914, -0.0004277686006813509),
    c64::new(-0.00023609720651021407, -0.0004274069351094292),
    c64::new(-0.0002367525625003862, -0.00042704426380770323),
    c64::new(-0.00023740736138890938, -0.00042668058762957366),
    c64::new(-0.00023806160163497838, -0.00042631590743080576),
    c64::new(-0.00023871528169910306, -0.0004259502240695271),
    c64::new(-0.00023936840004311092, -0.00042558353840622606),
    c64::new(-0.00024002095513015124, -0.00042521585130374955),
    c64::new(-0.00024067294542469918, -0.0004248471636273005),
    c64::new(-0.0002413243693925585, -0.0004244774762444369),
    c64::new(-0.0002419752255008654, -0.00042410679002506885),
    c64::new(-0.000242625512218093, -0.0004237351058414567),
    c64::new(-0.00024327522801405348, -0.0004233624245682094),
    c64::new(-0.0002439243713599032, -0.0004229887470822818),
    c64::new(-0.0002445729407281449, -0.0004226140742629732),
    c64::new(-0.0002452209345926321, -0.00042223840699192504),
    c64::new(-0.000245868351428573, -0.00042186174615311856),
    c64::new(-0.000246515189712533, -0.0004214840926328731),
    c64::new(-0.000247161447922439, -0.0004211054473198441),
    c64::new(-0.0002478071245375834, -0.0004207258111050202),
    c64::new(-0.0002484522180386265, -0.00042034518488172226),
    c64::new(-0.00024909672690760087, -0.0004199635695456007),
    c64::new(-0.0002497406496279152, -0.00041958096599463304),
    c64::new(-0.0002503839846843568, -0.00041919737512912254),
    c64::new(-0.00025102673056309646, -0.00041881279785169537),
    c64::new(-0.0002516688857516909, -0.0004184272350672992),
    c64::new(-0.00025231044873908674, -0.0004180406876832005),
    c64::new(-0.0002529514180156246, -0.00041765315660898244),
    c64::new(-0.0002535917920730418, -0.00041726464275654323),
    c64::new(-0.00025423156940447604, -0.0004168751470400935),
    c64::new(-0.00025487074850446986, -0.00041648467037615417),
    c64::new(-0.00025550932786897287, -0.0004160932136835546),
    c64::new(-0.00025614730599534606, -0.00041570077788343027),
    c64::new(-0.00025678468138236554, -0.00041530736389922045),
    c64::new(-0.0002574214525302251, -0.00041491297265666634),
    c64::new(-0.00025805761794054114, -0.00041451760508380846),
    c64::new(-0.0002586931761163548, -0.000414121262110985),
    c64::new(-0.00025932812556213607, -0.00041372394467082936),
    c64::new(-0.000259962464783788, -0.00041332565369826764),
    c64::new(-0.0002605961922886489, -0.0004129263901305171),
    c64::new(-0.00026122930658549655, -0.0004125261549070836),
    c64::new(-0.0002618618061845523, -0.00041212494896975904),
    c64::new(-0.00026249368959748316, -0.00041172277326261984),
    c64::new(-0.00026312495533740635, -0.0004113196287320244),
    c64::new(-0.000263755601918893, -0.00041091551632661057),
    c64::new(-0.0002643856278579703, -0.00041051043699729403),
    c64::new(-0.00026501503167212684, -0.00041010439169726545),
    c64::new(-0.00026564381188031437, -0.00040969738138198864),
    c64::new(-0.0002662719670029522, -0.00040928940700919833),
    c64::new(-0.00026689949556193103, -0.0004088804695388975),
    c64::new(-0.0002675263960806153, -0.00040847056993335547),
    c64::new(-0.00026815266708384743, -0.0004080597091571058),
    c64::new(-0.0002687783070979515, -0.0004076478881769434),
    c64::new(-0.00026940331465073606, -0.00040723510796192286),
    c64::new(-0.00027002768827149773, -0.0004068213694833561),
    c64::new(-0.0002706514264910254, -0.0004064066737148094),
    c64::new(-0.0002712745278416025, -0.0004059910216321023),
    c64::new(-0.00027189699085701166, -0.0004055744142133039),
    c64::new(-0.00027251881407253716, -0.00040515685243873196),
    c64::new(-0.00027313999602496865, -0.00040473833729094975),
    c64::new(-0.00027376053525260543, -0.0004043188697547636),
    c64::new(-0.00027438043029525853, -0.00040389845081722126),
    c64::new(-0.0002749996796942548, -0.0004034770814676092),
    c64::new(-0.0002756182819924411, -0.00040305476269745),
    c64::new(-0.00027623623573418615, -0.00040263149550050063),
    c64::new(-0.000276853539465385, -0.0004022072808727498),
    c64::new(-0.00027747019173346264, -0.00040178211981241525),
    c64::new(-0.0002780861910873766, -0.0004013560133199421),
    c64::new(-0.0002787015360776207, -0.00040092896239800037),
    c64::new(-0.00027931622525622903, -0.0004005009680514818),
    c64::new(-0.0002799302571767783, -0.0004000720312874986),
    c64::new(-0.0002805436303943924, -0.00039964215311538026),
    c64::new(-0.0002811563434657448, -0.00039921133454667173),
    c64::new(-0.0002817683949490622, -0.00039877957659513084),
    c64::new(-0.00028237978340412865, -0.00039834688027672553),
    c64::new(-0.00028299050739228796, -0.0003979132466096321),
    c64::new(-0.0002836005654764474, -0.0003974786766142327),
    c64::new(-0.00028420995622108176, -0.0003970431713131121),
    c64::new(-0.00028481867819223555, -0.0003966067317310566),
    c64::new(-0.0002854267299575269, -0.00039616935889505076),
    c64::new(-0.0002860341100861518, -0.0003957310538342748),
    c64::new(-0.0002866408171488856, -0.00039529181758010326),
    c64::new(-0.0002872468497180885, -0.0003948516511661011),
    c64::new(-0.000287852206367707, -0.00039441055562802264),
    c64::new(-0.00028845688567327835, -0.0003939685320038083),
    c64::new(-0.000289060886211934, -0.0003935255813335822),
    c64::new(-0.0002896642065624023, -0.00039308170465965017),
    c64::new(-0.0002902668453050119, -0.00039263690302649703),
    c64::new(-0.0002908688010216959, -0.00039219117748078366),
    c64::new(-0.0002914700722959944, -0.0003917445290713455),
    c64::new(-0.0002920706577130577, -0.0003912969588491896),
    c64::new(-0.00029267055585965065, -0.0003908484678674916),
    c64::new(-0.0002932697653241546, -0.00039039905718159435),
    c64::new(-0.0002938682846965722, -0.00038994872784900436),
    c64::new(-0.0002944661125685294, -0.0003894974809293902),
    c64::new(-0.0002950632475332792, -0.00038904531748457956),
    c64::new(-0.0002956596881857058, -0.00038859223857855643),
    c64::new(-0.00029625543312232646, -0.0003881382452774595),
    c64::new(-0.00029685048094129566, -0.00038768333864957875),
    c64::new(-0.00029744483024240874, -0.0003872275197653534),
    c64::new(-0.00029803847962710424, -0.0003867707896973693),
    c64::new(-0.0002986314276984677, -0.0003863131495203566),
    c64::new(-0.0002992236730612352, -0.00038585460031118654),
    c64::new(-0.000299815214321796, -0.00038539514314887006),
    c64::new(-0.00030040605008819663, -0.0003849347791145539),
    c64::new(-0.0003009961789701434, -0.0003844735092915192),
    c64::new(-0.00030158559957900574, -0.00038401133476517865),
    c64::new(-0.0003021743105278204, -0.00038354825662307326),
    c64::new(-0.00030276231043129356, -0.0003830842759548707),
    c64::new(-0.0003033495979058044, -0.00038261939385236246),
    c64::new(-0.00030393617156940916, -0.0003821536114094608),
    c64::new(-0.0003045220300418431, -0.00038168692972219697),
    c64::new(-0.00030510717194452453, -0.00038121934988871807),
    c64::new(-0.00030569159590055836, -0.00038075087300928445),
    c64::new(-0.0003062753005347382, -0.0003802815001862677),
    c64::new(-0.0003068582844735511, -0.00037981123252414704),
    c64::new(-0.0003074405463451792, -0.00037934007112950803),
    c64::new(-0.0003080220847795041, -0.00037886801711103896),
    c64::new(-0.00030860289840810984, -0.0003783950715795283),
    c64::new(-0.0003091829858642858, -0.0003779212356478627),
    c64::new(-0.0003097623457830299, -0.000377446510431024),
    c64::new(-0.0003103409768010528, -0.0003769708970460862),
    c64::new(-0.00031091887755677945, -0.0003764943966122137),
    c64::new(-0.0003114960466903535, -0.00037601701025065826),
    c64::new(-0.0003120724828436405, -0.0003755387390847557),
    c64::new(-0.0003126481846602302, -0.00037505958423992463),
    c64::new(-0.00031322315078544094, -0.00037457954684366236),
    c64::new(-0.0003137973798663217, -0.0003740986280255434),
    c64::new(-0.0003143708705516559, -0.00037361682891721634),
    c64::new(-0.0003149436214919648, -0.0003731341506524008),
    c64::new(-0.00031551563133950996, -0.0003726505943668854),
    c64::new(-0.00031608689874829685, -0.00037216616119852496),
    c64::new(-0.00031665742237407834, -0.00037168085228723725),
    c64::new(-0.0003172272008743571, -0.00037119466877500113),
    c64::new(-0.00031779623290838905, -0.0003707076118058536),
    c64::new(-0.00031836451713718726, -0.00037021968252588637),
    c64::new(-0.0003189320522235237, -0.0003697308820832445),
    c64::new(-0.0003194988368319338, -0.00036924121162812233),
    c64::new(-0.00032006486962871843, -0.000368750672312762),
    c64::new(-0.00032063014928194767, -0.00036825926529144984),
    c64::new(-0.0003211946744614641, -0.0003677669917205139),
    c64::new(-0.0003217584438388854, -0.00036727385275832163),
    c64::new(-0.00032232145608760755, -0.00036677984956527666),
    c64::new(-0.00032288370988280867, -0.00036628498330381595),
    c64::new(-0.00032344520390145107, -0.00036578925513840784),
    c64::new(-0.000324005936822285, -0.0003652926662355487),
    c64::new(-0.00032456590732585195, -0.00036479521776375983),
    c64::new(-0.000325125114094487, -0.00036429691089358594),
    c64::new(-0.00032568355581232293, -0.00036379774679759085),
    c64::new(-0.0003262412311652922, -0.0003632977266503561),
    c64::new(-0.00032679813884113076, -0.00036279685162847746),
    c64::new(-0.00032735427752938134, -0.00036229512291056204),
    c64::new(-0.0003279096459213957, -0.00036179254167722607),
    c64::new(-0.00032846424271033826, -0.00036128910911109195),
    c64::new(-0.00032901806659118947, -0.0003607848263967848),
    c64::new(-0.00032957111626074806, -0.00036027969472093066),
    c64::new(-0.0003301233904176347, -0.00035977371527215333),
    c64::new(-0.0003306748877622951, -0.0003592668892410711),
    c64::new(-0.00033122560699700256, -0.0003587592178202948),
    c64::new(-0.0003317755468258618, -0.000358250702204424),
    c64::new(-0.00033232470595481105, -0.0003577413435900453),
    c64::new(-0.0003328730830916258, -0.0003572311431757289),
    c64::new(-0.00033342067694592186, -0.0003567201021620252),
    c64::new(-0.00033396748622915797, -0.0003562082217514634),
    c64::new(-0.0003345135096546388, -0.0003556955031485475),
    c64::new(-0.00033505874593751884, -0.00035518194755975385),
    c64::new(-0.0003356031937948043, -0.00035466755619352827),
    c64::new(-0.0003361468519453566, -0.00035415233026028363),
    c64::new(-0.0003366897191098959, -0.000353636270972396),
    c64::new(-0.0003372317940110031, -0.00035311937954420297),
    c64::new(-0.00033777307537312385, -0.00035260165719199973),
    c64::new(-0.0003383135619225706, -0.00035208310513403716),
    c64::new(-0.00033885325238752626, -0.0003515637245905185),
    c64::new(-0.00033939214549804725, -0.00035104351678359594),
    c64::new(-0.0003399302399860659, -0.000350522482937369),
    c64::new(-0.00034046753458539375, -0.0003500006242778807),
    c64::new(-0.000341004028031725, -0.0003494779420331146),
    c64::new(-0.0003415397190626386, -0.00034895443743299267),
    c64::new(-0.00034207460641760167, -0.00034843011170937195),
    c64::new(-0.0003426086888379729, -0.0003479049660960412),
    c64::new(-0.00034314196506700443, -0.0003473790018287191),
    c64::new(-0.0003436744338498461, -0.00034685222014505),
    c64::new(-0.0003442060939335473, -0.0003463246222846024),
    c64::new(-0.0003447369440670603, -0.000345796209488865),
    c64::new(-0.0003452669830012439, -0.00034526698300124393),
    c64::new(-0.00034579620948886494, -0.0003447369440670604),
    c64::new(-0.0003463246222846023, -0.0003442060939335474),
    c64::new(-0.00034685222014504995, -0.0003436744338498462),
    c64::new(-0.000347379001828719, -0.0003431419650670045),
    c64::new(-0.0003479049660960412, -0.000342608688837973),
    c64::new(-0.00034843011170937184, -0.0003420746064176017),
    c64::new(-0.0003489544374329926, -0.00034153971906263867),
    c64::new(-0.00034947794203311455, -0.00034100402803172506),
    c64::new(-0.00035000062427788063, -0.0003404675345853938),
    c64::new(-0.00035052248293736894, -0.00033993023998606597),
    c64::new(-0.0003510435167835959, -0.0003393921454980473),
    c64::new(-0.0003515637245905184, -0.0003388532523875263),
    c64::new(-0.0003520831051340371, -0.00033831356192257073),
    c64::new(-0.0003526016571919996, -0.00033777307537312395),
    c64::new(-0.00035311937954420286, -0.0003372317940110032),
    c64::new(-0.00035363627097239594, -0.000336689719109896),
    c64::new(-0.0003541523302602836, -0.00033614685194535666),
    c64::new(-0.0003546675561935282, -0.00033560319379480433),
    c64::new(-0.00035518194755975374, -0.0003350587459375189),
    c64::new(-0.00035569550314854746, -0.00033451350965463885),
    c64::new(-0.00035620822175146333, -0.000333967486229158),
    c64::new(-0.0003567201021620252, -0.0003334206769459219),
    c64::new(-0.0003572311431757288, -0.00033287308309162583),
    c64::new(-0.00035774134359004527, -0.00033232470595481115),
    c64::new(-0.00035825070220442396, -0.00033177554682586185),
    c64::new(-0.00035875921782029473, -0.00033122560699700267),
    c64::new(-0.000359266889241071, -0.0003306748877622952),
    c64::new(-0.0003597737152721532, -0.00033012339041763473),
    c64::new(-0.00036027969472093055, -0.0003295711162607482),
    c64::new(-0.0003607848263967847, -0.0003290180665911895),
    c64::new(-0.0003612891091110919, -0.0003284642427103383),
    c64::new(-0.000361792541677226, -0.0003279096459213958),
    c64::new(-0.000362295122910562, -0.00032735427752938144),
    c64::new(-0.0003627968516284774, -0.00032679813884113086),
    c64::new(-0.00036329772665035604, -0.0003262412311652923),
    c64::new(-0.0003637977467975908, -0.00032568355581232304),
    c64::new(-0.0003642969108935859, -0.0003251251140944871),
    c64::new(-0.0003647952177637598, -0.00032456590732585206),
    c64::new(-0.00036529266623554863, -0.00032400593682228504),
    c64::new(-0.00036578925513840773, -0.0003234452039014512),
    c64::new(-0.0003662849833038159, -0.0003228837098828087),
    c64::new(-0.00036677984956527655, -0.00032232145608760766),
    c64::new(-0.0003672738527583216, -0.0003217584438388855),
    c64::new(-0.00036776699172051387, -0.00032119467446146423),
    c64::new(-0.0003682592652914498, -0.0003206301492819478),
    c64::new(-0.00036875067231276193, -0.00032006486962871854),
    c64::new(-0.00036924121162812233, -0.00031949883683193386),
    c64::new(-0.00036973088208324447, -0.00031893205222352375),
    c64::new(-0.0003702196825258863, -0.00031836451713718736),
    c64::new(-0.0003707076118058535, -0.00031779623290838915),
    c64::new(-0.0003711946687750011, -0.00031722720087435714),
    c64::new(-0.0003716808522872372, -0.0003166574223740784),
    c64::new(-0.00037216616119852485, -0.0003160868987482969),
    c64::new(-0.00037265059436688533, -0.00031551563133951),
    c64::new(-0.0003731341506524007, -0.00031494362149196486),
    c64::new(-0.0003736168289172163, -0.00031437087055165595),
    c64::new(-0.00037409862802554336, -0.0003137973798663218),
    c64::new(-0.00037457954684366225, -0.000313223150785441),
    c64::new(-0.0003750595842399245, -0.00031264818466023026),
    c64::new(-0.00037553873908475566, -0.00031207248284364055),
    c64::new(-0.00037601701025065815, -0.00031149604669035354),
    c64::new(-0.00037649439661221367, -0.0003109188775567795),
    c64::new(-0.00037697089704608614, -0.00031034097680105284),
    c64::new(-0.0003774465104310239, -0.00030976234578303),
    c64::new(-0.0003779212356478626, -0.00030918298586428585),
    c64::new(-0.00037839507157952823, -0.0003086028984081099),
    c64::new(-0.0003788680171110389, -0.00030802208477950414),
    c64::new(-0.000379340071129508, -0.0003074405463451793),
    c64::new(-0.000379811232524147, -0.00030685828447355113),
    c64::new(-0.0003802815001862676, -0.0003062753005347383),
    c64::new(-0.00038075087300928434, -0.0003056915959005584),
    c64::new(-0.000381219349888718, -0.0003051071719445246),
    c64::new(-0.0003816869297221969, -0.0003045220300418432),
    c64::new(-0.00038215361140946075, -0.0003039361715694092),
    c64::new(-0.00038261939385236246, -0.00030334959790580447),
    c64::new(-0.00038308427595487067, -0.00030276231043129367),
    c64::new(-0.0003835482566230732, -0.0003021743105278205),
    c64::new(-0.0003840113347651786, -0.0003015855995790058),
    c64::new(-0.0003844735092915192, -0.00030099617897014344),
    c64::new(-0.0003849347791145538, -0.00030040605008819674),
    c64::new(-0.00038539514314887, -0.0002998152143217961),
    c64::new(-0.0003858546003111865, -0.00029922367306123526),
    c64::new(-0.0003863131495203565, -0.00029863142769846773),
    c64::new(-0.0003867707896973692, -0.00029803847962710434),
    c64::new(-0.00038722751976535334, -0.00029744483024240885),
    c64::new(-0.0003876833386495787, -0.0002968504809412957),
    c64::new(-0.00038813824527745937, -0.0002962554331223265),
    c64::new(-0.0003885922385785564, -0.00029565968818570583),
    c64::new(-0.0003890453174845795, -0.0002950632475332793),
    c64::new(-0.0003894974809293901, -0.0002944661125685295),
    c64::new(-0.0003899487278490043, -0.00029386828469657233),
    c64::new(-0.00039039905718159424, -0.0002932697653241547),
    c64::new(-0.00039084846786749156, -0.0002926705558596507),
    c64::new(-0.00039129695884918957, -0.0002920706577130578),
    c64::new(-0.0003917445290713455, -0.0002914700722959945),
    c64::new(-0.0003921911774807836, -0.00029086880102169603),
    c64::new(-0.0003926369030264969, -0.00029026684530501193),
    c64::new(-0.0003930817046596501, -0.00028966420656240235),
    c64::new(-0.00039352558133358217, -0.0002890608862119341),
    c64::new(-0.00039396853200380827, -0.0002884568856732784),
    c64::new(-0.0003944105556280226, -0.00028785220636770706),
    c64::new(-0.00039485165116610107, -0.00028724684971808854),
    c64::new(-0.0003952918175801032, -0.00028664081714888567),
    c64::new(-0.00039573105383427473, -0.00028603411008615184),
    c64::new(-0.00039616935889505065, -0.000285426729957527),
    c64::new(-0.0003966067317310565, -0.0002848186781922356),
    c64::new(-0.00039704317131311205, -0.00028420995622108186),
    c64::new(-0.0003974786766142326, -0.0002836005654764475),
    c64::new(-0.00039791324660963203, -0.00028299050739228807),
    c64::new(-0.0003983468802767255, -0.00028237978340412876),
    c64::new(-0.0003987795765951308, -0.00028176839494906223),
    c64::new(-0.0003992113345466717, -0.00028115634346574485),
    c64::new(-0.0003996421531153802, -0.0002805436303943925),
    c64::new(-0.00040007203128749854, -0.00027993025717677837),
    c64::new(-0.0004005009680514817, -0.00027931622525622914),
    c64::new(-0.0004009289623980003, -0.00027870153607762073),
    c64::new(-0.0004013560133199421, -0.0002780861910873767),
    c64::new(-0.00040178211981241514, -0.00027747019173346275),
    c64::new(-0.0004022072808727497, -0.0002768535394653851),
    c64::new(-0.0004026314955005005, -0.00027623623573418626),
    c64::new(-0.0004030547626974499, -0.00027561828199244116),
    c64::new(-0.00040347708146760915, -0.00027499967969425493),
    c64::new(-0.0004038984508172212, -0.0002743804302952586),
    c64::new(-0.0004043188697547635, -0.0002737605352526055),
    c64::new(-0.0004047383372909497, -0.00027313999602496876),
    c64::new(-0.0004051568524387319, -0.00027251881407253727),
    c64::new(-0.00040557441421330386, -0.00027189699085701177),
    c64::new(-0.0004059910216321022, -0.0002712745278416026),
    c64::new(-0.00040640667371480936, -0.0002706514264910255),
    c64::new(-0.00040682136948335605, -0.0002700276882714978),
    c64::new(-0.00040723510796192286, -0.00026940331465073617),
    c64::new(-0.00040764788817694336, -0.0002687783070979516),
    c64::new(-0.00040805970915710576, -0.0002681526670838475),
    c64::new(-0.0004084705699333554, -0.0002675263960806154),
    c64::new(-0.0004088804695388974, -0.00026689949556193114),
    c64::new(-0.0004092894070091983, -0.0002662719670029523),
    c64::new(-0.0004096973813819886, -0.0002656438118803145),
    c64::new(-0.0004101043916972654, -0.0002650150316721269),
    c64::new(-0.00041051043699729403, -0.0002643856278579704),
    c64::new(-0.0004109155163266105, -0.00026375560191889305),
    c64::new(-0.00041131962873202434, -0.00026312495533740646),
    c64::new(-0.0004117227732626198, -0.00026249368959748327),
    c64::new(-0.000412124948969759, -0.00026186180618455234),
    c64::new(-0.00041252615490708355, -0.00026122930658549666),
    c64::new(-0.00041292639013051707, -0.000260596192288649),
    c64::new(-0.0004133256536982676, -0.0002599624647837881),
    c64::new(-0.0004137239446708293, -0.0002593281255621362),
    c64::new(-0.0004141212621109849, -0.0002586931761163549),
    c64::new(-0.0004145176050838084, -0.00025805761794054125),
    c64::new(-0.0004149129726566663, -0.00025742145253022523),
    c64::new(-0.0004153073638992204, -0.00025678468138236565),
    c64::new(-0.00041570077788343027, -0.00025614730599534617),
    c64::new(-0.00041609321368355455, -0.000255509327868973),
    c64::new(-0.0004164846703761541, -0.0002548707485044699),
    c64::new(-0.00041687514704009346, -0.00025423156940447615),
    c64::new(-0.0004172646427565432, -0.00025359179207304187),
    c64::new(-0.00041765315660898244, -0.00025295141801562473),
    c64::new(-0.0004180406876832005, -0.0002523104487390868),
    c64::new(-0.00041842723506729915, -0.000251668885751691),
    c64::new(-0.00041881279785169537, -0.00025102673056309657),
    c64::new(-0.0004191973751291225, -0.0002503839846843569),
    c64::new(-0.00041958096599463294, -0.0002497406496279153),
    c64::new(-0.00041996356954560065, -0.000249096726907601),
    c64::new(-0.00042034518488172226, -0.0002484522180386266),
    c64::new(-0.00042072581110502017, -0.0002478071245375835),
    c64::new(-0.000421105447319844, -0.0002471614479224391),
    c64::new(-0.00042148409263287307, -0.00024651518971253306),
    c64::new(-0.0004218617461531185, -0.00024586835142857303),
    c64::new(-0.000422238406991925, -0.00024522093459263217),
    c64::new(-0.00042261407426297313, -0.000244572940728145),
    c64::new(-0.00042298874708228174, -0.00024392437135990327),
    c64::new(-0.0004233624245682093, -0.0002432752280140536),
    c64::new(-0.00042373510584145665, -0.0002426255122180931),
    c64::new(-0.0004241067900250688, -0.0002419752255008655),
    c64::new(-0.0004244774762444368, -0.00024132436939255857),
    c64::new(-0.00042484716362730046, -0.0002406729454246993),
    c64::new(-0.00042521585130374945, -0.00024002095513015132),
    c64::new(-0.00042558353840622606, -0.00023936840004311103),
    c64::new(-0.00042595022406952707, -0.00023871528169910314),
    c64::new(-0.00042631590743080576, -0.00023806160163497846),
    c64::new(-0.0004266805876295736, -0.00023740736138890947),
    c64::new(-0.0004270442638077032, -0.0002367525625003863),
    c64::new(-0.00042740693510942915, -0.00023609720651021417),
    c64::new(-0.0004277686006813508, -0.00023544129496050922),
    c64::new(-0.0004281292596724344, -0.00023478482939469432),
    c64::new(-0.0004284889112340142, -0.00023412781135749672),
    c64::new(-0.00042884755451979564, -0.00023347024239494296),
    c64::new(-0.0004292051886858562, -0.0002328121240543565),
    c64::new(-0.0004295618128906481, -0.00023215345788435356),
    c64::new(-0.00042991742629500037, -0.0002314942454348389),
    c64::new(-0.00043027202806212016, -0.0002308344882570033),
    c64::new(-0.0004306256173575952, -0.00023017418790331924),
    c64::new(-0.0004309781933493959, -0.00022951334592753695),
    c64::new(-0.0004313297552078768, -0.00022885196388468157),
    c64::new(-0.00043168030210577875, -0.0002281900433310491),
    c64::new(-0.00043202983321823134, -0.00022752758582420224),
    c64::new(-0.00043237834772275383, -0.00022686459292296782),
    c64::new(-0.00043272584479925807, -0.0002262010661874318),
    c64::new(-0.0004330723236300499, -0.00022553700717893688),
    c64::new(-0.00043341778339983097, -0.00022487241746007824),
    c64::new(-0.0004337622232957013, -0.00022420729859469932),
    c64::new(-0.00043410564250716044, -0.00022354165214788925),
    c64::new(-0.0004344480402261097, -0.00022287547968597858),
    c64::new(-0.00043478941564685443, -0.0002222087827765351),
    c64::new(-0.0004351297679661051, -0.0002215415629883612),
    c64::new(-0.0004354690963829798, -0.00022087382189148975),
    c64::new(-0.00043580740009900614, -0.00022020556105717962),
    c64::new(-0.00043614467831812266, -0.0002195367820579135),
    c64::new(-0.0004364809302466813, -0.0002188674864673926),
    c64::new(-0.0004368161550934486, -0.00021819767586053426),
    c64::new(-0.00043715035206960814, -0.00021752735181346776),
    c64::new(-0.00043748352038876226, -0.00021685651590352994),
    c64::new(-0.0004378156592669337, -0.00021618516970926268),
    c64::new(-0.0004381467679225675, -0.00021551331481040864),
    c64::new(-0.0004384768455765331, -0.0002148409527879069),
    c64::new(-0.00043880589145212594, -0.0002141680852238904),
    c64::new(-0.0004391339047750691, -0.00021349471370168176),
    c64::new(-0.00043946088477351576, -0.0002128208398057888),
    c64::new(-0.00043978683067805017, -0.00021214646512190234),
    c64::new(-0.0004401117417216904, -0.00021147159123689066),
    c64::new(-0.00044043561713988925, -0.0002107962197387975),
    c64::new(-0.0004407584561705365, -0.00021012035221683735),
    c64::new(-0.00044108025805396085, -0.0002094439902613914),
    c64::new(-0.0004414010220329313, -0.0002087671354640049),
    c64::new(-0.00044172074735265924, -0.00020808978941738273),
    c64::new(-0.00044203943326080015, -0.00020741195371538519),
    c64::new(-0.0004423570790074553, -0.00020673362995302536),
    c64::new(-0.0004426736838451735, -0.00020605481972646477),
    c64::new(-0.00044298924702895304, -0.00020537552463300904),
    c64::new(-0.0004433037678162432, -0.00020469574627110559),
    c64::new(-0.00044361724546694635, -0.0002040154862403381),
    c64::new(-0.00044392967924341913, -0.0002033347461414244),
    c64::new(-0.00044424106841047466, -0.00020265352757621184),
    c64::new(-0.00044455141223538434, -0.00020197183214767315),
    c64::new(-0.0004448607099878791, -0.00020128966145990373),
    c64::new(-0.00044516896094015125, -0.00020060701711811725),
    c64::new(-0.0004454761643668568, -0.00019992390072864143),
    c64::new(-0.0004457823195451163, -0.00019924031389891532),
    c64::new(-0.000446087425754517, -0.00019855625823748496),
    c64::new(-0.0004463914822771146, -0.00019787173535399896),
    c64::new(-0.00044669448839743456, -0.00019718674685920625),
    c64::new(-0.0004469964434024745, -0.00019650129436495046),
    c64::new(-0.00044729734658170494, -0.00019581537948416778),
    c64::new(-0.0004475971972270716, -0.00019512900383088236),
    c64::new(-0.000447895994632997, -0.000194442169020202),
    c64::new(-0.0004481937380963821, -0.00019375487666831555),
    c64::new(-0.0004484904269166075, -0.00019306712839248854),
    c64::new(-0.000448786060395536, -0.00019237892581105867),
    c64::new(-0.00044908063783751336, -0.0001916902705434333),
    c64::new(-0.00044937415854937035, -0.00019100116421008498),
    c64::new(-0.0004496666218404245, -0.0001903116084325471),
    c64::new(-0.00044995802702248134, -0.00018962160483341142),
    c64::new(-0.00045024837340983646, -0.0001889311550363226),
    c64::new(-0.0004505376603192767, -0.000188240260665976),
    c64::new(-0.0004508258870700818, -0.00018754892334811295),
    c64::new(-0.00045111305298402673, -0.00018685714470951652),
    c64::new(-0.0004513991573853822, -0.00018616492637800895),
    c64::new(-0.00045168419960091673, -0.00018547226998244696),
    c64::new(-0.0004519681789598988, -0.00018477917715271759),
    c64::new(-0.00045225109479409735, -0.00018408564951973542),
    c64::new(-0.00045253294643778406, -0.00018339168871543817),
    c64::new(-0.0004528137332277349, -0.00018269729637278222),
    c64::new(-0.00045309345450323134, -0.00018200247412574025),
    c64::new(-0.00045337210960606225, -0.00018130722360929573),
    c64::new(-0.00045364969788052527, -0.00018061154645944048),
    c64::new(-0.00045392621867342833, -0.00017991544431317021),
    c64::new(-0.0004542016713340913, -0.00017921891880848012),
    c64::new(-0.0004544760552143475, -0.0001785219715843622),
    c64::new(-0.0004547493696685451, -0.00017782460428080084),
    c64::new(-0.0004550216140535489, -0.0001771268185387682),
    c64::new(-0.00045529278772874144, -0.00017642861600022176),
    c64::new(-0.0004555628900560248, -0.00017572999830809977),
    c64::new(-0.00045583192039982235, -0.0001750309671063167),
    c64::new(-0.00045609987812707947, -0.0001743315240397609),
    c64::new(-0.00045636676260726595, -0.00017363167075428907),
    c64::new(-0.00045663257321237676, -0.0001729314088967238),
    c64::new(-0.0004568973093169337, -0.0001722307401148491),
    c64::new(-0.00045716097029798743, -0.00017152966605740586),
    c64::new(-0.000457423555535118, -0.0001708281883740893),
    c64::new(-0.00045768506441043694, -0.00017012630871554433),
    c64::new(-0.00045794549630858876, -0.0001694240287333612),
    c64::new(-0.00045820485061675204, -0.00016872135008007275),
    c64::new(-0.00045846312672464086, -0.00016801827440914992),
    c64::new(-0.0004587203240245068, -0.00016731480337499728),
    c64::new(-0.00045897644191113956, -0.00016661093863295053),
    c64::new(-0.0004592314797818693, -0.000165906681839271),
    c64::new(-0.000459485437036567, -0.00016520203465114313),
    c64::new(-0.0004597383130776468, -0.00016449699872667006),
    c64::new(-0.00045999010731006705, -0.000163791575724869),
    c64::new(-0.0004602408191413315, -0.00016308576730566862),
    c64::new(-0.0004604904479814909, -0.00016237957512990451),
    c64::new(-0.0004607389932431447, -0.0001616730008593147),
    c64::new(-0.00046098645434144173, -0.00016096604615653683),
    c64::new(-0.0004612328306940821, -0.00016025871268510387),
    c64::new(-0.00046147812172131865, -0.00015955100210943935),
    c64::new(-0.00046172232684595766, -0.00015884291609485508),
    c64::new(-0.000461965445493361, -0.00015813445630754542),
    c64::new(-0.00046220747709144704, -0.0001574256244145849),
    c64::new(-0.00046244842107069195, -0.00015671642208392358),
    c64::new(-0.0004626882768641314, -0.00015600685098438272),
    c64::new(-0.00046292704390736144, -0.0001552969127856518),
    c64::new(-0.00046316472163854017, -0.0001545866091582842),
    c64::new(-0.00046340130949838894, -0.0001538759417736926),
    c64::new(-0.0004636368069301937, -0.0001531649123041462),
    c64::new(-0.00046387121337980615, -0.00015245352242276617),
    c64::new(-0.00046410452829564536, -0.00015174177380352126),
    c64::new(-0.0004643367511286987, -0.0001510296681212251),
    c64::new(-0.0004645678813325236, -0.00015031720705153078),
    c64::new(-0.0004647979183632482, -0.00014960439227092818),
    c64::new(-0.00046502686167957313, -0.00014889122545673958),
    c64::new(-0.00046525471074277276, -0.00014817770828711495),
    c64::new(-0.0004654814650166962, -0.0001474638424410293),
    c64::new(-0.00046570712396776866, -0.00014674962959827816),
    c64::new(-0.00046593168706499295, -0.00014603507143947292),
    c64::new(-0.00046615515377995036, -0.00014532016964603815),
    c64::new(-0.0004663775235868021, -0.00014460492590020704),
    c64::new(-0.0004665987959622906, -0.0001438893418850168),
    c64::new(-0.00046681897038574043, -0.00014317341928430613),
    c64::new(-0.00046703804633906005, -0.00014245715978270968),
    c64::new(-0.0004672560233067426, -0.00014174056506565544),
    c64::new(-0.0004674729007758671, -0.00014102363681936023),
    c64::new(-0.00046768867823610005, -0.000140306376730825),
    c64::new(-0.00046790335517969613, -0.0001395887864878323),
    c64::new(-0.00046811693110149975, -0.00013887086777894143),
    c64::new(-0.00046832940549894624, -0.0001381526222934841),
    c64::new(-0.00046854077787206275, -0.00013743405172156147),
    c64::new(-0.00046875104772346965, -0.00013671515775403977),
    c64::new(-0.00046896021455838173, -0.00013599594208254547),
    c64::new(-0.00046916827788460915, -0.000135276406399463),
    c64::new(-0.0004693752372125589, -0.00013455655239792886),
    c64::new(-0.0004695810920552356, -0.0001338363817718293),
    c64::new(-0.00046978584192824296, -0.00013311589621579553),
    c64::new(-0.00046998948634978475, -0.00013239509742519928),
    c64::new(-0.00047019202484066596, -0.00013167398709614992),
    c64::new(-0.0004703934569242939, -0.00013095256692548994),
    c64::new(-0.0004705937821266796, -0.00013023083861079027),
    c64::new(-0.0004707929999764384, -0.00012950880385034753),
    c64::new(-0.00047099111000479133, -0.00012878646434317948),
    c64::new(-0.00047118811174556656, -0.00012806382178902033),
    c64::new(-0.00047138400473519985, -0.00012734087788831828),
    c64::new(-0.0004715787885127361, -0.0001266176343422297),
    c64::new(-0.0004717724626198303, -0.0001258940928526167),
    c64::new(-0.00047196502660074854, -0.00012517025512204255),
    c64::new(-0.0004721564800023692, -0.00012444612285376693),
    c64::new(-0.00047234682237418384, -0.00012372169775174321),
    c64::new(-0.00047253605326829854, -0.00012299698152061386),
    c64::new(-0.00047272417223943473, -0.00012227197586570573),
    c64::new(-0.00047291117884493036, -0.00012154668249302739),
    c64::new(-0.00047309707264474066, -0.0001208211031092644),
    c64::new(-0.0004732818532014397, -0.00012009523942177473),
    c64::new(-0.00047346552008022087, -0.00011936909313858617),
    c64::new(-0.0004736480728488984, -0.00011864266596839064),
    c64::new(-0.00047382951107790796, -0.00011791595962054167),
    c64::new(-0.00047400983434030773, -0.00011718897580504968),
    c64::new(-0.00047418904221177983, -0.00011646171623257742),
    c64::new(-0.00047436713427063075, -0.0001157341826144371),
    c64::new(-0.0004745441100977926, -0.00011500637666258581),
    c64::new(-0.00047471996927682435, -0.00011427830008962084),
    c64::new(-0.0004748947113939122, -0.00011354995460877693),
    c64::new(-0.0004750683360378711, -0.00011282134193392154),
    c64::new(-0.00047524084280014563, -0.00011209246377955026),
    c64::new(-0.0004754122312748105, -0.00011136332186078421),
    c64::new(-0.0004755825010585722, -0.00011063391789336434),
    c64::new(-0.00047575165175076946, -0.00010990425359364884),
    c64::new(-0.00047591968295337435, -0.00010917433067860853),
    c64::new(-0.0004760865942709932, -0.0001084441508658221),
    c64::new(-0.00047625238531086766, -0.00010771371587347338),
    c64::new(-0.0004764170556828752, -0.00010698302742034669),
    c64::new(-0.0004765806049995308, -0.00010625208722582213),
    c64::new(-0.0004767430328759869, -0.00010552089700987277),
    c64::new(-0.000476904338930035, -0.00010478945849306005),
    c64::new(-0.00047706452278210664, -0.00010405777339652901),
    c64::new(-0.0004772235840552734, -0.0001033258434420058),
    c64::new(-0.000477381522375249, -0.00010259367035179188),
    c64::new(-0.00047753833737038923, -0.00010186125584876147),
    c64::new(-0.00047769402867169314, -0.00010112860165635685),
    c64::new(-0.00047784859591280425, -0.00010039570949858368),
    c64::new(-0.00047800203873001084, -9.966258110000822e-05),
    c64::new(-0.0004781543567622471, -9.892921818575261e-05),
    c64::new(-0.000478305549651094, -9.819562248149023e-05),
    c64::new(-0.0004784556170407801, -9.746179571344282e-05),
    c64::new(-0.0004786045585781823, -9.672773960837587e-05),
    c64::new(-0.00047875237391282685, -9.599345589359389e-05),
    c64::new(-0.00047889906269688986, -9.525894629693778e-05),
    c64::new(-0.00047904462458519853, -9.452421254677919e-05),
    c64::new(-0.0004791890592352317, -9.378925637201779e-05),
    c64::new(-0.00047933236630712066, -9.305407950207668e-05),
    c64::new(-0.00047947454546365, -9.231868366689762e-05),
    c64::new(-0.0004796155963702585, -9.158307059693827e-05),
    c64::new(-0.00047975551869503967, -9.084724202316742e-05),
    c64::new(-0.0004798943121087428, -9.01111996770604e-05),
    c64::new(-0.00048003197628477363, -8.937494529059614e-05),
    c64::new(-0.0004801685108991949, -8.863848059625253e-05),
    c64::new(-0.00048030391563072766, -8.790180732700174e-05),
    c64::new(-0.00048043819016075143, -8.716492721630732e-05),
    c64::new(-0.0004805713341733052, -8.642784199811959e-05),
    c64::new(-0.00048070334735508835, -8.569055340687081e-05),
    c64::new(-0.00048083422939546104, -8.49530631774727e-05),
    c64::new(-0.00048096397998644523, -8.421537304531059e-05),
    c64::new(-0.0004810925988227252, -8.347748474624082e-05),
    c64::new(-0.00048122008560164847, -8.273940001658604e-05),
    c64::new(-0.00048134644002322625, -8.20011205931305e-05),
    c64::new(-0.0004814716617901345, -8.126264821311721e-05),
    c64::new(-0.00048159575060771414, -8.052398461424322e-05),
    c64::new(-0.00048171870618397237, -7.978513153465492e-05),
    c64::new(-0.0004818405282295828, -7.904609071294517e-05),
    c64::new(-0.0004819612164578862, -7.830686388814864e-05),
    c64::new(-0.0004820807705848918, -7.756745279973703e-05),
    c64::new(-0.00048219919032927697, -7.682785918761648e-05),
    c64::new(-0.00048231647541238876, -7.608808479212176e-05),
    c64::new(-0.0004824326255582439, -7.534813135401371e-05),
    c64::new(-0.00048254764049353, -7.460800061447446e-05),
    c64::new(-0.00048266151994760576, -7.386769431510269e-05),
    c64::new(-0.00048277426365250185, -7.312721419791087e-05),
    c64::new(-0.0004828858713429214, -7.238656200532044e-05),
    c64::new(-0.00048299634275624073, -7.164573948015713e-05),
    c64::new(-0.00048310567763250995, -7.090474836564812e-05),
    c64::new(-0.0004832138757144535, -7.01635904054173e-05),
    c64::new(-0.00048332093674747096, -6.94222673434805e-05),
    c64::new(-0.0004834268604796372, -6.868078092424293e-05),
    c64::new(-0.00048353164666170367, -6.793913289249333e-05),
    c64::new(-0.0004836352950470982, -6.719732499340137e-05),
    c64::new(-0.0004837378053919261, -6.645535897251292e-05),
    c64::new(-0.0004838391774549707, -6.571323657574525e-05),
    c64::new(-0.0004839394109976938, -6.497095954938428e-05),
    c64::new(-0.00048403850578423606, -6.422852964007974e-05),
    c64::new(-0.0004841364615814179, -6.348594859484053e-05),
    c64::new(-0.00048423327815874, -6.274321816103177e-05),
    c64::new(-0.00048432895528838346, -6.200034008637013e-05),
    c64::new(-0.00048442349274521084, -6.125731611891907e-05),
    c64::new(-0.00048451689030676624, -6.051414800708617e-05),
    c64::new(-0.00048460914775327635, -5.977083749961736e-05),
    c64::new(-0.0004847002648676504, -5.902738634559427e-05),
    c64::new(-0.000484790241435481, -5.828379629442951e-05),
    c64::new(-0.00048487907724504463, -5.754006909586186e-05),
    c64::new(-0.000484966772087302, -5.679620649995347e-05),
    c64::new(-0.00048505332575589876, -5.605221025708512e-05),
    c64::new(-0.0004851387380471657, -5.5308082117951375e-05),
    c64::new(-0.00048522300876011943, -5.456382383355786e-05),
    c64::new(-0.0004853061376964629, -5.381943715521643e-05),
    c64::new(-0.00048538812466058563, -5.30749238345404e-05),
    c64::new(-0.0004854689694595644, -5.2330285623441974e-05),
    c64::new(-0.0004855486719031636, -5.158552427412635e-05),
    c64::new(-0.00048562723180383565, -5.084064153908915e-05),
    c64::new(-0.00048570464897672153, -5.00956391711116e-05),
    c64::new(-0.0004857809232396512, -4.9350518923255827e-05),
    c64::new(-0.00048585605441314386, -4.860528254886195e-05),
    c64::new(-0.0004859300423204086, -4.785993180154338e-05),
    c64::new(-0.00048600288678734486, -4.7114468435181964e-05),
    c64::new(-0.00048607458764254245, -4.636889420392525e-05),
    c64::new(-0.0004861451447172824, -4.562321086218163e-05),
    c64::new(-0.000486214557845537, -4.487742016461561e-05),
    c64::new(-0.00048628282686397057, -4.413152386614517e-05),
    c64::new(-0.00048634995161193944, -4.3385523721935915e-05),
    c64::new(-0.00048641593193149267, -4.2639421487398466e-05),
    c64::new(-0.00048648076766737207, -4.189321891818367e-05),
    c64::new(-0.000486544458667013, -4.114691777017782e-05),
    c64::new(-0.00048660700478054435, -4.040051979949983e-05),
    c64::new(-0.0004866684058607891, -3.965402676249642e-05),
    c64::new(-0.0004867286617632645, -3.89074404157374e-05),
    c64::new(-0.0004867877723461827, -3.816076251601277e-05),
    c64::new(-0.00048684573747045067, -3.741399482032797e-05),
    c64::new(-0.00048690255699967094, -3.6667139085899095e-05),
    c64::new(-0.0004869582308001417, -3.5920197070150266e-05),
    c64::new(-0.0004870127587408571, -3.5173170530707756e-05),
    c64::new(-0.00048706614069350755, -3.442606122539739e-05),
    c64::new(-0.00048711837653248025, -3.367887091223976e-05),
    c64::new(-0.0004871694661348592, -3.29316013494454e-05),
    c64::new(-0.0004872194093804256, -3.2184254295411994e-05),
    c64::new(-0.0004872682061516582, -3.143683150871958e-05),
    c64::new(-0.0004873158563337335, -3.068933474812573e-05),
    c64::new(-0.00048736235981452597, -2.9941765772562742e-05),
    c64::new(-0.0004874077164846085, -2.9194126341132846e-05),
    c64::new(-0.0004874519262372524, -2.8446418213103415e-05),
    c64::new(-0.0004874949889684278, -2.769864314790433e-05),
    c64::new(-0.0004875369045768041, -2.6950802905122116e-05),
    c64::new(-0.00048757767296374966, -2.6202899244497324e-05),
    c64::new(-0.0004876172940333326, -2.5454933925919727e-05),
    c64::new(-0.00048765576769232077, -2.4706908709423535e-05),
    c64::new(-0.00048769309385018184, -2.3958825355184557e-05),
    c64::new(-0.00048772927241908377, -2.3210685623515403e-05),
    c64::new(-0.00048776430331389496, -2.2462491274860696e-05),
    c64::new(-0.00048779818645218427, -2.171424406979423e-05),
    c64::new(-0.00048783092175422137, -2.096594576901417e-05),
    c64::new(-0.000487862509142977, -2.0217598133338272e-05),
    c64::new(-0.000487892948544123, -1.9469202923701235e-05),
    c64::new(-0.00048792223988603243, -1.8720761901148854e-05),
    c64::new(-0.00048795038309978005, -1.7972276826835367e-05),
    c64::new(-0.000487977378119142, -1.7223749462018668e-05),
    c64::new(-0.00048800322488059655, -1.647518156805552e-05),
    c64::new(-0.0004880279233233237, -1.5726574906398703e-05),
    c64::new(-0.00048805147338920564, -1.497793123859222e-05),
    c64::new(-0.00048807387502282677, -1.4229252326266505e-05),
    c64::new(-0.00048809512817147394, -1.3480539931135574e-05),
    c64::new(-0.00048811523278513634, -1.2731795814992236e-05),
    c64::new(-0.000488134188816506, -1.1983021739703284e-05),
    c64::new(-0.0004881519962209774, -1.123421946720688e-05),
    c64::new(-0.0004881686549566481, -1.0485390759506674e-05),
    c64::new(-0.0004881841649843185, -9.736537378669159e-06),
    c64::new(-0.0004881985262674919, -8.987661086818897e-06),
    c64::new(-0.00048821173877237493, -8.238763646133707e-06),
    c64::new(-0.0004882238024678772, -7.489846818841826e-06),
    c64::new(-0.0004882347173256117, -6.740912367217106e-06),
    c64::new(-0.0004882444833198948, -5.991962053574222e-06),
    c64::new(-0.00048825310042774605, -5.2429976402658256e-06),
    c64::new(-0.0004882605686288886, -4.494020889677745e-06),
    c64::new(-0.0004882668879057491, -3.7450335642241884e-06),
    c64::new(-0.00048827205824345757, -2.9960374263451166e-06),
    c64::new(-0.00048827607962984785, -2.2470342385003578e-06),
    c64::new(-0.0004882789520554571, -1.498025763166981e-06),
    c64::new(-0.0004882806755135263, -7.490137628344968e-07),
];

static TWIDDLES_4096: [c64; 4096] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999997058628822, 0.0007669903187427045),
    c64::new(0.9999988234517019, 0.0015339801862847655),
    c64::new(0.9999973527669782, 0.002300969151425805),
    c64::new(0.9999952938095762, 0.003067956762965976),
    c64::new(0.9999926465807072, 0.0038349425697062275),
    c64::new(0.9999894110819284, 0.0046019261204485705),
    c64::new(0.9999855873151432, 0.0053689069639963425),
    c64::new(0.9999811752826011, 0.006135884649154475),
    c64::new(0.9999761749868976, 0.006902858724729756),
    c64::new(0.9999705864309741, 0.007669828739531097),
    c64::new(0.9999644096181183, 0.008436794242369799),
    c64::new(0.9999576445519639, 0.00920375478205982),
    c64::new(0.9999502912364905, 0.00997070990741803),
    c64::new(0.9999423496760239, 0.01073765916726449),
    c64::new(0.999933819875236, 0.011504602110422714),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9999149955731135, 0.013038467241987334),
    c64::new(0.9999047010828529, 0.01380538852806039),
    c64::new(0.9998938183744185, 0.014572301692779064),
    c64::new(0.9998823474542126, 0.0153392062849881),
    c64::new(0.999870288328983, 0.016106101853537287),
    c64::new(0.9998576410058239, 0.01687298794728171),
    c64::new(0.9998444054921752, 0.017639864115082053),
    c64::new(0.9998305817958234, 0.01840672990580482),
    c64::new(0.9998161699249004, 0.019173584868322623),
    c64::new(0.9998011698878843, 0.01994042855151444),
    c64::new(0.9997855816935992, 0.020707260504265895),
    c64::new(0.9997694053512153, 0.021474080275469508),
    c64::new(0.9997526408702488, 0.02224088741402496),
    c64::new(0.9997352882605617, 0.02300768146883937),
    c64::new(0.9997173475323622, 0.023774461988827555),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9996797017629879, 0.02530798062002457),
    c64::new(0.9996599967439592, 0.0260747178291039),
    c64::new(0.9996397036507102, 0.02684143969909853),
    c64::new(0.9996188224951786, 0.02760814577896574),
    c64::new(0.9995973532896484, 0.0283748356176721),
    c64::new(0.9995752960467492, 0.029141508764193722),
    c64::new(0.999552650779457, 0.029908164767516555),
    c64::new(0.9995294175010931, 0.030674803176636626),
    c64::new(0.9995055962253253, 0.0314414235405603),
    c64::new(0.999481186966167, 0.032208025408304586),
    c64::new(0.9994561897379773, 0.032974608328897335),
    c64::new(0.9994306045554617, 0.03374117185137758),
    c64::new(0.9994044314336713, 0.03450771552479575),
    c64::new(0.9993776703880028, 0.03527423889821395),
    c64::new(0.9993503214341994, 0.03604074152070623),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9992938598668878, 0.037573682709270494),
    c64::new(0.9992647472865944, 0.038340120373552694),
    c64::new(0.9992350468645959, 0.03910653548332989),
    c64::new(0.9992047586183639, 0.03987292758773981),
    c64::new(0.9991738825657164, 0.040639296235933736),
    c64::new(0.9991424187248169, 0.04140564097707674),
    c64::new(0.9991103671141749, 0.04217196136034795),
    c64::new(0.9990777277526454, 0.04293825693494082),
    c64::new(0.9990445006594293, 0.04370452725006342),
    c64::new(0.9990106858540734, 0.04447077185493867),
    c64::new(0.9989762833564698, 0.04523699029880459),
    c64::new(0.9989412931868569, 0.04600318213091462),
    c64::new(0.9989057153658183, 0.04676934690053786),
    c64::new(0.9988695499142836, 0.0475354841569593),
    c64::new(0.998832796853528, 0.048301593449480144),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9987575279911833, 0.04983372634010728),
    c64::new(0.9987190122338729, 0.05059974903689928),
    c64::new(0.9986799089558991, 0.05136574196716259),
    c64::new(0.9986402181802653, 0.052131704680283324),
    c64::new(0.9985999399303204, 0.052897636725665324),
    c64::new(0.9985590742297593, 0.05366353765273052),
    c64::new(0.9985176211026222, 0.05442940701091913),
    c64::new(0.9984755805732948, 0.05519524434968994),
    c64::new(0.9984329526665084, 0.05596104921852057),
    c64::new(0.9983897374073402, 0.05672682116690775),
    c64::new(0.9983459348212124, 0.057492559744367566),
    c64::new(0.9983015449338929, 0.05825826450043575),
    c64::new(0.9982565677714952, 0.05902393498466793),
    c64::new(0.9982110033604782, 0.05978957074663987),
    c64::new(0.9981648517276462, 0.06055517133594779),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.9980707869054823, 0.06208626519506009),
    c64::new(0.9980228737714862, 0.0628517575641614),
    c64::new(0.997974373526347, 0.0636172129591931),
    c64::new(0.997925286198596, 0.06438263092985747),
    c64::new(0.9978756118171102, 0.06514801102587883),
    c64::new(0.9978253504111116, 0.0659133527970038),
    c64::new(0.9977745020101678, 0.06667865579300156),
    c64::new(0.9977230666441916, 0.06744391956366405),
    c64::new(0.997671044343441, 0.06820914365880633),
    c64::new(0.9976184351385196, 0.06897432762826675),
    c64::new(0.9975652390603758, 0.0697394710219073),
    c64::new(0.9975114561403035, 0.07050457338961386),
    c64::new(0.9974570864099419, 0.0712696342812964),
    c64::new(0.9974021299012753, 0.07203465324688933),
    c64::new(0.9973465866466332, 0.07279962983635167),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.9972337400304663, 0.07432945408684576),
    c64::new(0.9971764367353262, 0.0750943008479213),
    c64::new(0.99711854682698, 0.07585910343295445),
    c64::new(0.997060070339483, 0.07662386139203149),
    c64::new(0.9970010073072353, 0.07738857427526505),
    c64::new(0.9969413577649822, 0.07815324163279423),
    c64::new(0.9968811217478138, 0.07891786301478494),
    c64::new(0.9968202992911657, 0.07968243797143013),
    c64::new(0.996758890430818, 0.08044696605295001),
    c64::new(0.9966968952028961, 0.08121144680959244),
    c64::new(0.9966343136438699, 0.08197587979163307),
    c64::new(0.9965711457905548, 0.08274026454937569),
    c64::new(0.9965073916801108, 0.08350460063315243),
    c64::new(0.9964430513500426, 0.08426888759332407),
    c64::new(0.9963781248382002, 0.08503312498028028),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.9962465134223155, 0.08656144923625117),
    c64::new(0.996179828595697, 0.08732553520619206),
    c64::new(0.9961125577421511, 0.0880895698047705),
    c64::new(0.996044700901252, 0.0888535525825246),
    c64::new(0.9959762581129178, 0.08961748309002296),
    c64::new(0.9959072294174117, 0.09038136087786498),
    c64::new(0.9958376148553416, 0.091145185496681),
    c64::new(0.9957674144676598, 0.09190895649713272),
    c64::new(0.9956966282956635, 0.09267267342991331),
    c64::new(0.9956252563809943, 0.09343633584574779),
    c64::new(0.9955532987656385, 0.0941999432953932),
    c64::new(0.9954807554919269, 0.09496349532963899),
    c64::new(0.9954076266025349, 0.09572699149930716),
    c64::new(0.9953339121404823, 0.09649043135525259),
    c64::new(0.9952596121491334, 0.09725381444836327),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9951092557537261, 0.09878040854979962),
    c64::new(0.9950331994381186, 0.09954361866006932),
    c64::new(0.9949565577701164, 0.10030677021139286),
    c64::new(0.9948793307948056, 0.10106986275482782),
    c64::new(0.9948015185576171, 0.10183289584146653),
    c64::new(0.9947231211043257, 0.10259586902243628),
    c64::new(0.9946441384810507, 0.10335878184889961),
    c64::new(0.9945645707342554, 0.10412163387205459),
    c64::new(0.9944844179107476, 0.10488442464313497),
    c64::new(0.9944036800576791, 0.10564715371341062),
    c64::new(0.9943223572225458, 0.10640982063418768),
    c64::new(0.9942404494531879, 0.10717242495680884),
    c64::new(0.9941579567977897, 0.10793496623265365),
    c64::new(0.9940748793048794, 0.10869744401313872),
    c64::new(0.9939912170233294, 0.10945985784971798),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.9938221382915197, 0.11098449189716339),
    c64::new(0.9937367219407246, 0.11174671121112659),
    c64::new(0.9936507210002191, 0.11250886478737869),
    c64::new(0.9935641355205953, 0.11327095217756435),
    c64::new(0.9934769655527892, 0.1140329729333672),
    c64::new(0.9933892111480807, 0.11479492660651008),
    c64::new(0.9933008723580933, 0.11555681274875526),
    c64::new(0.9932119492347945, 0.11631863091190475),
    c64::new(0.9931224418304956, 0.11708038064780059),
    c64::new(0.9930323501978514, 0.11784206150832498),
    c64::new(0.9929416743898605, 0.11860367304540072),
    c64::new(0.9928504144598651, 0.11936521481099135),
    c64::new(0.9927585704615511, 0.12012668635710151),
    c64::new(0.992666142448948, 0.12088808723577708),
    c64::new(0.9925731304764288, 0.12164941699910553),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.9923853548708517, 0.12317186138828048),
    c64::new(0.9922905913482574, 0.12393297511851216),
    c64::new(0.9921952440866739, 0.12469401594216764),
    c64::new(0.9920993131421918, 0.12545498341154623),
    c64::new(0.9920027985712445, 0.12621587707899035),
    c64::new(0.9919057004306093, 0.12697669649688587),
    c64::new(0.9918080187774064, 0.1277374412176623),
    c64::new(0.9917097536690995, 0.12849811079379317),
    c64::new(0.9916109051634954, 0.12925870477779614),
    c64::new(0.9915114733187439, 0.13001922272223335),
    c64::new(0.9914114581933385, 0.1307796641797117),
    c64::new(0.9913108598461154, 0.13154002870288312),
    c64::new(0.9912096783362541, 0.13230031584444465),
    c64::new(0.9911079137232769, 0.13306052515713906),
    c64::new(0.9910055660670494, 0.13382065619375472),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.9907991218660204, 0.13534068165013421),
    c64::new(0.9906950254426646, 0.1361005751757062),
    c64::new(0.9905903462189501, 0.13686038863681638),
    c64::new(0.9904850842564571, 0.13762012158648604),
    c64::new(0.9903792396171082, 0.1383797735777839),
    c64::new(0.9902728123631691, 0.1391393441638262),
    c64::new(0.9901658025572484, 0.13989883289777721),
    c64::new(0.9900582102622971, 0.1406582393328492),
    c64::new(0.989950035541609, 0.14141756302230302),
    c64::new(0.9898412784588205, 0.14217680351944803),
    c64::new(0.9897319390779106, 0.14293596037764267),
    c64::new(0.9896220174632009, 0.14369503315029447),
    c64::new(0.9895115136793552, 0.14445402139086047),
    c64::new(0.9894004277913804, 0.14521292465284746),
    c64::new(0.9892887598646252, 0.1459717424898122),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9890636781578815, 0.14748912010315357),
    c64::new(0.988950264510303, 0.14824767898689603),
    c64::new(0.9888362690887635, 0.14900615066034845),
    c64::new(0.9887216919603238, 0.1497645346773215),
    c64::new(0.9886065331923864, 0.1505228305916774),
    c64::new(0.9884907928526966, 0.15128103795733022),
    c64::new(0.9883744710093413, 0.15203915632824605),
    c64::new(0.9882575677307495, 0.15279718525844344),
    c64::new(0.9881400830856926, 0.15355512430199345),
    c64::new(0.9880220171432835, 0.1543129730130201),
    c64::new(0.9879033699729778, 0.1550707309457005),
    c64::new(0.9877841416445722, 0.15582839765426523),
    c64::new(0.9876643322282057, 0.15658597269299843),
    c64::new(0.9875439417943592, 0.15734345561623825),
    c64::new(0.9874229704138554, 0.15810084597837698),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.9871792850978743, 0.15961534723719306),
    c64::new(0.987056571305751, 0.16037245724292828),
    c64::new(0.9869332768536777, 0.1611294729056788),
    c64::new(0.9868094018141855, 0.16188639378011183),
    c64::new(0.9866849462601467, 0.1626432194209503),
    c64::new(0.9865599102647754, 0.16339994938297323),
    c64::new(0.9864342939016272, 0.1641565832210158),
    c64::new(0.9863080972445987, 0.16491312048996992),
    c64::new(0.9861813203679283, 0.16566956074478412),
    c64::new(0.9860539633461954, 0.1664259035404641),
    c64::new(0.9859260262543211, 0.16718214843207294),
    c64::new(0.9857975091675675, 0.16793829497473117),
    c64::new(0.9856684121615376, 0.16869434272361733),
    c64::new(0.9855387353121761, 0.16945029123396796),
    c64::new(0.9854084786957684, 0.17020614006107807),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9851462264686622, 0.17171753688704997),
    c64::new(0.9850142310122398, 0.17247308399679595),
    c64::new(0.9848816560973237, 0.17322852964507032),
    c64::new(0.9847485018019042, 0.17398387338746382),
    c64::new(0.9846147682043126, 0.1747391147796272),
    c64::new(0.9844804553832209, 0.17549425337727143),
    c64::new(0.9843455634176419, 0.1762492887361679),
    c64::new(0.984210092386929, 0.17700422041214875),
    c64::new(0.9840740423707764, 0.17775904796110717),
    c64::new(0.9839374134492189, 0.1785137709389975),
    c64::new(0.9838002057026316, 0.17926838890183575),
    c64::new(0.9836624192117303, 0.18002290140569951),
    c64::new(0.9835240540575713, 0.1807773080067286),
    c64::new(0.9833851103215512, 0.18153160826112497),
    c64::new(0.9832455880854071, 0.1822858017251533),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9829648084413964, 0.18379386650747845),
    c64::new(0.9828235511987052, 0.18454773693861962),
    c64::new(0.9826817157862409, 0.1853014988050819),
    c64::new(0.9825393022874412, 0.18605515166344663),
    c64::new(0.9823963107860847, 0.18680869507035927),
    c64::new(0.9822527413662894, 0.1875621285825296),
    c64::new(0.9821085941125136, 0.18831545175673212),
    c64::new(0.9819638691095552, 0.1890686641498062),
    c64::new(0.9818185664425525, 0.1898217653186564),
    c64::new(0.9816726861969831, 0.19057475482025274),
    c64::new(0.9815262284586648, 0.1913276322116309),
    c64::new(0.9813791933137546, 0.19208039704989244),
    c64::new(0.9812315808487497, 0.19283304889220523),
    c64::new(0.9810833911504867, 0.1935855872958036),
    c64::new(0.9809346243061416, 0.1943380118179886),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.9806353595296081, 0.19584251744765785),
    c64::new(0.9804848617734694, 0.19659459767008022),
    c64::new(0.980333787223348, 0.19734656224096592),
    c64::new(0.9801821359681174, 0.19809841071795356),
    c64::new(0.9800299080969901, 0.1988501426587501),
    c64::new(0.9798771036995176, 0.19960175762113097),
    c64::new(0.9797237228655912, 0.20035325516294045),
    c64::new(0.9795697656854405, 0.2011046348420919),
    c64::new(0.9794152322496348, 0.20185589621656805),
    c64::new(0.979260122649082, 0.20260703884442113),
    c64::new(0.9791044369750292, 0.20335806228377332),
    c64::new(0.9789481753190622, 0.20410896609281687),
    c64::new(0.9787913377731057, 0.20485974982981442),
    c64::new(0.9786339244294232, 0.20561041305309924),
    c64::new(0.9784759353806168, 0.2063609553210755),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.978158230539735, 0.20786167522507507),
    c64::new(0.9779985149345571, 0.20861185197826349),
    c64::new(0.9778382239980504, 0.20936190601047416),
    c64::new(0.9776773578245099, 0.2101118368804696),
    c64::new(0.9775159165085693, 0.21086164414708486),
    c64::new(0.9773539001452001, 0.21161132736922755),
    c64::new(0.9771913088297123, 0.21236088610587842),
    c64::new(0.9770281426577544, 0.21311031991609136),
    c64::new(0.9768644017253126, 0.21385962835899375),
    c64::new(0.9767000861287118, 0.21460881099378676),
    c64::new(0.9765351959646145, 0.21535786737974555),
    c64::new(0.9763697313300211, 0.21610679707621952),
    c64::new(0.9762036923222706, 0.21685559964263262),
    c64::new(0.976037079039039, 0.21760427463848364),
    c64::new(0.975869891578341, 0.21835282162334632),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.9755337945182914, 0.2198495297987787),
    c64::new(0.975364885116657, 0.2205976901088735),
    c64::new(0.9751954019329904, 0.2213457206470308),
    c64::new(0.9750253450669941, 0.2220936209732035),
    c64::new(0.9748547146187084, 0.22284139064742112),
    c64::new(0.9746835106885107, 0.22358902922979),
    c64::new(0.9745117333771157, 0.2243365362804936),
    c64::new(0.9743393827855759, 0.22508391135979283),
    c64::new(0.9741664590152803, 0.22583115402802617),
    c64::new(0.9739929621679558, 0.22657826384561),
    c64::new(0.9738188923456661, 0.22732524037303886),
    c64::new(0.973644249650812, 0.22807208317088573),
    c64::new(0.9734690341861311, 0.22881879179980222),
    c64::new(0.9732932460546982, 0.22956536582051887),
    c64::new(0.9731168853599251, 0.23031180479384544),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.9727624466956886, 0.23180427584196478),
    c64::new(0.9725843689347322, 0.23255030703877524),
    c64::new(0.9724057190274498, 0.23329620143223162),
    c64::new(0.9722264970789363, 0.23404195858354343),
    c64::new(0.9720467031946235, 0.23478757805400097),
    c64::new(0.9718663374802794, 0.2355330594049755),
    c64::new(0.9716854000420085, 0.23627840219791957),
    c64::new(0.9715038909862518, 0.2370236059943672),
    c64::new(0.9713218104197862, 0.2377686703559342),
    c64::new(0.9711391584497251, 0.23851359484431842),
    c64::new(0.970955935183518, 0.23925837902129998),
    c64::new(0.9707721407289504, 0.2400030224487415),
    c64::new(0.9705877751941436, 0.24074752468858843),
    c64::new(0.9704028386875555, 0.24149188530286933),
    c64::new(0.9702173313179792, 0.242236103853696),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.9698446044267148, 0.24372411301385216),
    c64::new(0.9696573851242924, 0.24446790274782415),
    c64::new(0.9694695953974131, 0.24521154866762754),
    c64::new(0.9692812353565485, 0.2459550503357946),
    c64::new(0.9690923051125062, 0.24669840731494241),
    c64::new(0.9689028047764289, 0.24744161916777327),
    c64::new(0.9687127344597948, 0.24818468545707478),
    c64::new(0.9685220942744173, 0.24892760574572015),
    c64::new(0.9683308843324452, 0.24967037959666855),
    c64::new(0.9681391047463624, 0.2504130065729652),
    c64::new(0.9679467556289878, 0.2511554862377419),
    c64::new(0.9677538370934755, 0.25189781815421697),
    c64::new(0.9675603492533145, 0.2526400018856955),
    c64::new(0.9673662922223285, 0.25338203699557016),
    c64::new(0.9671716661146766, 0.2541239230473206),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9667807071276833, 0.2556072462308074),
    c64::new(0.9665843744783331, 0.2563486824899429),
    c64::new(0.9663874732122989, 0.2570899679457531),
    c64::new(0.9661900034454125, 0.257831102162159),
    c64::new(0.9659919652938406, 0.25857208470317034),
    c64::new(0.9657933588740837, 0.25931291513288623),
    c64::new(0.9655941843029768, 0.2600535930154952),
    c64::new(0.9653944416976894, 0.2607941179152755),
    c64::new(0.9651941311757247, 0.2615344893965955),
    c64::new(0.9649932528549203, 0.2622747070239136),
    c64::new(0.9647918068534479, 0.263014770361779),
    c64::new(0.9645897932898128, 0.26375467897483135),
    c64::new(0.9643872122828543, 0.26449443242780163),
    c64::new(0.9641840639517458, 0.2652340302855118),
    c64::new(0.9639803484159941, 0.2659734721128756),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9635712162102573, 0.2674518859366776),
    c64::new(0.963365799780954, 0.2681908570634032),
    c64::new(0.9631598166283714, 0.26892967042035726),
    c64::new(0.9629532668736839, 0.2696683255729151),
    c64::new(0.9627461506383994, 0.2704068220865448),
    c64::new(0.9625384680443592, 0.271145159526808),
    c64::new(0.9623302192137374, 0.2718833374593597),
    c64::new(0.9621214042690416, 0.272621355449949),
    c64::new(0.9619120233331122, 0.2733592130644187),
    c64::new(0.9617020765291225, 0.2740969098687064),
    c64::new(0.961491563980579, 0.27483444542884394),
    c64::new(0.9612804858113206, 0.27557181931095814),
    c64::new(0.9610688421455194, 0.2763090310812711),
    c64::new(0.9608566331076797, 0.2770460803060999),
    c64::new(0.9606438588226386, 0.2777829665518577),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.9602166150119634, 0.2792562483722912),
    c64::new(0.960002145737666, 0.2799926430802732),
    c64::new(0.9597871117188399, 0.2807288730757972),
    c64::new(0.9595715130819845, 0.28146493792575794),
    c64::new(0.9593553499539308, 0.28220083719714756),
    c64::new(0.9591386224618419, 0.2829365704570554),
    c64::new(0.9589213307332132, 0.28367213727266843),
    c64::new(0.9587034748958716, 0.2844075372112719),
    c64::new(0.9584850550779762, 0.28514276984024867),
    c64::new(0.9582660714080177, 0.2858778347270806),
    c64::new(0.9580465240148186, 0.2866127314393478),
    c64::new(0.9578264130275329, 0.2873474595447295),
    c64::new(0.9576057385756463, 0.28808201861100413),
    c64::new(0.9573845007889759, 0.2888164082060495),
    c64::new(0.9571626997976701, 0.289550627897843),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.956717408723403, 0.2910185558440851),
    c64::new(0.9564939189023951, 0.29175226323498926),
    c64::new(0.956269866400658, 0.2924857989955539),
    c64::new(0.9560452513499964, 0.29321916269425863),
    c64::new(0.9558200738825454, 0.29395235389968466),
    c64::new(0.9555943341307711, 0.2946853721805143),
    c64::new(0.9553680322274704, 0.295418217105532),
    c64::new(0.9551411683057707, 0.2961508882436238),
    c64::new(0.9549137424991305, 0.29688338516377827),
    c64::new(0.9546857549413383, 0.2976157074350862),
    c64::new(0.9544572057665135, 0.2983478546267414),
    c64::new(0.9542280951091057, 0.2990798263080405),
    c64::new(0.9539984231038945, 0.29981162204838335),
    c64::new(0.9537681898859903, 0.30054324141727345),
    c64::new(0.9535373955908333, 0.30127468398431795),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9530741243121722, 0.30273703699181914),
    c64::new(0.9528416476011987, 0.3034679465720113),
    c64::new(0.9526086103580333, 0.3041986776298291),
    c64::new(0.9523750127197659, 0.3049292297354024),
    c64::new(0.9521408548238158, 0.3056596024589661),
    c64::new(0.9519061368079323, 0.3063897953708609),
    c64::new(0.9516708588101939, 0.3071198080415331),
    c64::new(0.9514350209690083, 0.30784964004153487),
    c64::new(0.9511986234231132, 0.30857929094152503),
    c64::new(0.9509616663115751, 0.3093087603122687),
    c64::new(0.9507241497737896, 0.3100380477246379),
    c64::new(0.9504860739494817, 0.3107671527496115),
    c64::new(0.9502474389787052, 0.3114960749582759),
    c64::new(0.950008245001843, 0.3122248139218249),
    c64::new(0.9497684921596067, 0.3129533692115602),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9492873104435021, 0.31440992705533666),
    c64::new(0.9490458818527006, 0.31513792875252244),
    c64::new(0.9488038949626585, 0.31586574506218396),
    c64::new(0.9485613499157303, 0.31659337555616585),
    c64::new(0.9483182468545992, 0.31732081980642174),
    c64::new(0.9480745859222762, 0.31804807738501495),
    c64::new(0.947830367262101, 0.3187751478641185),
    c64::new(0.9475855910177411, 0.3195020308160157),
    c64::new(0.947340257333192, 0.32022872581309986),
    c64::new(0.9470943663527772, 0.3209552324278752),
    c64::new(0.946847918221148, 0.3216815502329566),
    c64::new(0.9466009130832835, 0.32240767880106985),
    c64::new(0.9463533510844906, 0.32313361770505233),
    c64::new(0.9461052323704034, 0.32385936651785285),
    c64::new(0.9458565570869839, 0.32458492481253215),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9453575373976323, 0.32603546814033024),
    c64::new(0.9451071932852606, 0.32676045232013173),
    c64::new(0.9448562931906772, 0.327485244275178),
    c64::new(0.9446048372614803, 0.3282098435790925),
    c64::new(0.9443528256455948, 0.3289342498056122),
    c64::new(0.9441002584912727, 0.3296584625285875),
    c64::new(0.9438471359470927, 0.3303824813219828),
    c64::new(0.9435934581619604, 0.33110630575987643),
    c64::new(0.9433392252851077, 0.3318299354164611),
    c64::new(0.9430844374660935, 0.3325533698660442),
    c64::new(0.9428290948548027, 0.3332766086830479),
    c64::new(0.9425731976014469, 0.3339996514420094),
    c64::new(0.9423167458565638, 0.3347224977175812),
    c64::new(0.9420597397710173, 0.3354451470845316),
    c64::new(0.9418021794959976, 0.3361675991177445),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.9412853969839287, 0.3376119094830746),
    c64::new(0.9410261750508893, 0.3383337669655411),
    c64::new(0.9407663995363961, 0.33905542541496964),
    c64::new(0.9405060705932683, 0.33977688440682685),
    c64::new(0.9402451883746509, 0.34049814351669716),
    c64::new(0.939983753034014, 0.34121920232028236),
    c64::new(0.9397217647251533, 0.3419400603934022),
    c64::new(0.9394592236021899, 0.3426607173119944),
    c64::new(0.9391961298195699, 0.34338117265211504),
    c64::new(0.9389324835320645, 0.3441014259899388),
    c64::new(0.9386682848947702, 0.3448214769017593),
    c64::new(0.9384035340631081, 0.3455413249639891),
    c64::new(0.9381382311928244, 0.34626096975316),
    c64::new(0.9378723764399899, 0.3469804108459237),
    c64::new(0.937605969961, 0.3476996478190514),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.9370715024517592, 0.34913750771408497),
    c64::new(0.9368034417359216, 0.3498561297901349),
    c64::new(0.9365348299227555, 0.3505745460548375),
    c64::new(0.9362656671702783, 0.3512927560855671),
    c64::new(0.9359959536368314, 0.3520107594598191),
    c64::new(0.9357256894810804, 0.3527285557552107),
    c64::new(0.9354548748620146, 0.3534461445494808),
    c64::new(0.9351835099389476, 0.35416352542049034),
    c64::new(0.9349115948715161, 0.3548806979462228),
    c64::new(0.9346391298196808, 0.35559766170478385),
    c64::new(0.9343661149437259, 0.3563144162744024),
    c64::new(0.934092550404259, 0.35703096123343),
    c64::new(0.933818436362211, 0.3577472961603419),
    c64::new(0.9335437729788362, 0.35846342063373654),
    c64::new(0.933268560415712, 0.3591793342323365),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9327164883981403, 0.36061052712066227),
    c64::new(0.9324396292684624, 0.3613258055684543),
    c64::new(0.9321622216085744, 0.3620408714575842),
    c64::new(0.9318842655816681, 0.3627557243673972),
    c64::new(0.9316057613512578, 0.36347036387736376),
    c64::new(0.9313267090811804, 0.3641847895670799),
    c64::new(0.9310471089355953, 0.36489900101626727),
    c64::new(0.9307669610789837, 0.36561299780477385),
    c64::new(0.9304862656761497, 0.3663267795125736),
    c64::new(0.9302050228922191, 0.3670403457197672),
    c64::new(0.9299232328926397, 0.367753696006582),
    c64::new(0.9296408958431812, 0.3684668299533723),
    c64::new(0.9293580119099355, 0.36917974714062),
    c64::new(0.9290745812593157, 0.3698924471489341),
    c64::new(0.928790604058057, 0.37060492955905167),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9282210106721694, 0.372029239908285),
    c64::new(0.9279353948226179, 0.37274106700951576),
    c64::new(0.9276492330925812, 0.3734526748367803),
    c64::new(0.9273625256504011, 0.37416406297145793),
    c64::new(0.9270752726647401, 0.37487523099505754),
    c64::new(0.9267874743045817, 0.3755861784892172),
    c64::new(0.9264991307392305, 0.3762969050357048),
    c64::new(0.9262102421383114, 0.37700741021641826),
    c64::new(0.9259208086717701, 0.37771769361338564),
    c64::new(0.9256308305098727, 0.37842775480876556),
    c64::new(0.9253403078232063, 0.3791375933848473),
    c64::new(0.9250492407826776, 0.37984720892405116),
    c64::new(0.9247576295595139, 0.3805566010089285),
    c64::new(0.9244654743252626, 0.3812657692221624),
    c64::new(0.9241727752517912, 0.3819747131465672),
    c64::new(0.9238795325112867, 0.3826834323650898),
    c64::new(0.9235857462762567, 0.38339192646080866),
    c64::new(0.9232914167195276, 0.38410019501693504),
    c64::new(0.9229965440142462, 0.3848082376168129),
    c64::new(0.9227011283338786, 0.38551605384391885),
    c64::new(0.92240516985221, 0.386223643281863),
    c64::new(0.9221086687433451, 0.3869310055143886),
    c64::new(0.9218116251817081, 0.38763814012537273),
    c64::new(0.9215140393420419, 0.38834504669882625),
    c64::new(0.9212159113994087, 0.3890517248188944),
    c64::new(0.9209172415291895, 0.3897581740698564),
    c64::new(0.9206180299070839, 0.3904643940361266),
    c64::new(0.9203182767091106, 0.39117038430225387),
    c64::new(0.9200179821116066, 0.39187614445292235),
    c64::new(0.9197171462912274, 0.39258167407295147),
    c64::new(0.9194157694249471, 0.3932869727472964),
    c64::new(0.9191138516900578, 0.3939920400610481),
    c64::new(0.9188113932641699, 0.39469687559943356),
    c64::new(0.9185083943252123, 0.39540147894781635),
    c64::new(0.918204855051431, 0.39610584969169627),
    c64::new(0.9179007756213905, 0.3968099874167103),
    c64::new(0.917596156213973, 0.3975138917086323),
    c64::new(0.9172909970083779, 0.39821756215337356),
    c64::new(0.916985298184123, 0.3989209983369829),
    c64::new(0.9166790599210427, 0.3996241998456468),
    c64::new(0.9163722823992891, 0.4003271662656901),
    c64::new(0.9160649657993317, 0.4010298971835756),
    c64::new(0.9157571103019567, 0.401732392185905),
    c64::new(0.9154487160882678, 0.40243465085941843),
    c64::new(0.9151397833396853, 0.4031366727909953),
    c64::new(0.9148303122379462, 0.4038384575676541),
    c64::new(0.9145203029651044, 0.404540004776553),
    c64::new(0.9142097557035307, 0.40524131400498986),
    c64::new(0.9138986706359117, 0.4059423848404025),
    c64::new(0.9135870479452508, 0.40664321687036903),
    c64::new(0.9132748878148678, 0.40734380968260797),
    c64::new(0.9129621904283982, 0.4080441628649787),
    c64::new(0.9126489559697939, 0.40874427600548136),
    c64::new(0.9123351846233227, 0.4094441486922576),
    c64::new(0.9120208765735683, 0.41014378051359024),
    c64::new(0.9117060320054299, 0.4108431710579039),
    c64::new(0.9113906511041224, 0.4115423199137652),
    c64::new(0.9110747340551764, 0.4122412266698829),
    c64::new(0.9107582810444376, 0.4129398909151081),
    c64::new(0.9104412922580672, 0.41363831223843456),
    c64::new(0.9101237678825417, 0.4143364902289991),
    c64::new(0.9098057081046522, 0.41503442447608163),
    c64::new(0.9094871131115054, 0.41573211456910536),
    c64::new(0.9091679830905224, 0.41642956009763715),
    c64::new(0.9088483182294391, 0.4171267606513879),
    c64::new(0.9085281187163061, 0.41782371582021227),
    c64::new(0.9082073847394887, 0.4185204251941097),
    c64::new(0.9078861164876663, 0.4192168883632239),
    c64::new(0.9075643141498326, 0.4199131049178436),
    c64::new(0.9072419779152958, 0.4206090744484025),
    c64::new(0.9069191079736781, 0.42130479654547964),
    c64::new(0.9065957045149153, 0.4220002707997997),
    c64::new(0.9062717677292577, 0.42269549680223295),
    c64::new(0.9059472978072685, 0.42339047414379605),
    c64::new(0.9056222949398253, 0.42408520241565156),
    c64::new(0.9052967593181188, 0.4247796812091088),
    c64::new(0.9049706911336532, 0.4254739101156238),
    c64::new(0.9046440905782462, 0.4261678887267996),
    c64::new(0.9043169578440283, 0.42686161663438643),
    c64::new(0.9039892931234433, 0.4275550934302821),
    c64::new(0.903661096609248, 0.42824831870653196),
    c64::new(0.9033323684945118, 0.4289412920553295),
    c64::new(0.9030031089726172, 0.4296340130690164),
    c64::new(0.9026733182372588, 0.4303264813400826),
    c64::new(0.9023429964824442, 0.43101869646116703),
    c64::new(0.9020121439024932, 0.43171065802505726),
    c64::new(0.9016807606920377, 0.43240236562469014),
    c64::new(0.901348847046022, 0.43309381885315196),
    c64::new(0.9010164031597023, 0.4337850173036786),
    c64::new(0.900683429228647, 0.43447596056965565),
    c64::new(0.9003499254487356, 0.43516664824461926),
    c64::new(0.9000158920161603, 0.4358570799222555),
    c64::new(0.8996813291274239, 0.4365472551964012),
    c64::new(0.8993462369793416, 0.4372371736610441),
    c64::new(0.8990106157690391, 0.43792683491032286),
    c64::new(0.8986744656939538, 0.43861623853852766),
    c64::new(0.8983377869518343, 0.43930538414009995),
    c64::new(0.8980005797407399, 0.43999427130963326),
    c64::new(0.8976628442590409, 0.4406828996418729),
    c64::new(0.8973245807054183, 0.44137126873171667),
    c64::new(0.896985789278864, 0.4420593781742147),
    c64::new(0.8966464701786803, 0.44274722756457),
    c64::new(0.8963066236044795, 0.4434348164981385),
    c64::new(0.8959662497561852, 0.4441221445704292),
    c64::new(0.89562534883403, 0.4448092113771049),
    c64::new(0.8952839210385576, 0.44549601651398174),
    c64::new(0.8949419665706208, 0.44618255957703007),
    c64::new(0.8945994856313827, 0.44686884016237416),
    c64::new(0.8942564784223161, 0.447554857866293),
    c64::new(0.8939129451452033, 0.4482406122852199),
    c64::new(0.8935688860021359, 0.44892610301574326),
    c64::new(0.8932243011955153, 0.44961132965460654),
    c64::new(0.8928791909280518, 0.4502962917987086),
    c64::new(0.8925335554027646, 0.45098098904510386),
    c64::new(0.8921873948229825, 0.4516654209910025),
    c64::new(0.8918407093923427, 0.4523495872337709),
    c64::new(0.8914934993147914, 0.4530334873709316),
    c64::new(0.8911457647945832, 0.45371712100016387),
    c64::new(0.8907975060362815, 0.4544004877193036),
    c64::new(0.8904487232447579, 0.45508358712634384),
    c64::new(0.8900994166251923, 0.45576641881943464),
    c64::new(0.8897495863830729, 0.4564489823968839),
    c64::new(0.8893992327241955, 0.457131277457157),
    c64::new(0.8890483558546646, 0.45781330359887723),
    c64::new(0.8886969559808916, 0.4584950604208263),
    c64::new(0.8883450333095964, 0.4591765475219441),
    c64::new(0.8879925880478056, 0.45985776450132954),
    c64::new(0.8876396204028539, 0.46053871095824),
    c64::new(0.8872861305823831, 0.4612193864920924),
    c64::new(0.8869321187943422, 0.46189979070246273),
    c64::new(0.886577585246987, 0.4625799231890868),
    c64::new(0.8862225301488806, 0.46325978355186015),
    c64::new(0.8858669537088928, 0.4639393713908385),
    c64::new(0.8855108561362, 0.4646186863062378),
    c64::new(0.8851542376402851, 0.4652977278984346),
    c64::new(0.8847970984309378, 0.4659764957679662),
    c64::new(0.8844394387182537, 0.4666549895155309),
    c64::new(0.884081258712635, 0.46733320874198847),
    c64::new(0.8837225586247897, 0.46801115304835983),
    c64::new(0.8833633386657316, 0.46868882203582796),
    c64::new(0.8830035990467808, 0.4693662153057375),
    c64::new(0.8826433399795628, 0.4700433324595956),
    c64::new(0.8822825616760087, 0.4707201730990716),
    c64::new(0.881921264348355, 0.47139673682599764),
    c64::new(0.8815594482091438, 0.47207302324236866),
    c64::new(0.881197113471222, 0.4727490319503428),
    c64::new(0.8808342603477419, 0.47342476255224153),
    c64::new(0.8804708890521608, 0.47410021465054997),
    c64::new(0.8801069997982404, 0.4747753878479171),
    c64::new(0.8797425928000474, 0.47545028174715587),
    c64::new(0.8793776682719533, 0.47612489595124363),
    c64::new(0.8790122264286335, 0.4767992300633221),
    c64::new(0.8786462674850681, 0.47747328368669806),
    c64::new(0.8782797916565416, 0.478147056424843),
    c64::new(0.8779127991586418, 0.47882054788139394),
    c64::new(0.8775452902072614, 0.479493757660153),
    c64::new(0.877177265018596, 0.4801666853650884),
    c64::new(0.8768087238091457, 0.48083933060033396),
    c64::new(0.8764396667957136, 0.48151169297018986),
    c64::new(0.8760700941954066, 0.4821837720791227),
    c64::new(0.8757000062256346, 0.4828555675317657),
    c64::new(0.8753294031041109, 0.48352707893291874),
    c64::new(0.8749582850488516, 0.48419830588754903),
    c64::new(0.8745866522781762, 0.48486924800079106),
    c64::new(0.8742145050107063, 0.48553990487794696),
    c64::new(0.8738418434653669, 0.4862102761244864),
    c64::new(0.8734686678613849, 0.48688036134604734),
    c64::new(0.8730949784182901, 0.487550160148436),
    c64::new(0.8727207753559143, 0.4882196721376268),
    c64::new(0.8723460588943915, 0.48888889691976317),
    c64::new(0.8719708292541578, 0.48955783410115744),
    c64::new(0.871595086655951, 0.49022648328829116),
    c64::new(0.871218831320811, 0.49089484408781514),
    c64::new(0.8708420634700789, 0.4915629161065499),
    c64::new(0.8704647833253977, 0.492230698951486),
    c64::new(0.8700869911087115, 0.49289819222978404),
    c64::new(0.8697086870422657, 0.49356539554877477),
    c64::new(0.8693298713486068, 0.4942323085159597),
    c64::new(0.8689505442505824, 0.49489893073901126),
    c64::new(0.8685707059713409, 0.49556526182577254),
    c64::new(0.8681903567343313, 0.49623130138425825),
    c64::new(0.8678094967633033, 0.49689704902265447),
    c64::new(0.8674281262823069, 0.49756250434931915),
    c64::new(0.8670462455156926, 0.49822766697278187),
    c64::new(0.8666638546881111, 0.4988925365017446),
    c64::new(0.866280954024513, 0.49955711254508184),
    c64::new(0.8658975437501488, 0.5002213947118407),
    c64::new(0.8655136240905691, 0.5008853826112407),
    c64::new(0.8651291952716238, 0.5015490758526754),
    c64::new(0.8647442575194624, 0.5022124740457108),
    c64::new(0.864358811060534, 0.502875576800087),
    c64::new(0.8639728561215867, 0.5035383837257176),
    c64::new(0.8635863929296681, 0.5042008944326903),
    c64::new(0.8631994217121242, 0.5048631085312676),
    c64::new(0.8628119426966003, 0.5055250256318854),
    c64::new(0.8624239561110405, 0.5061866453451552),
    c64::new(0.8620354621836872, 0.5068479672818632),
    c64::new(0.8616464611430813, 0.5075089910529709),
    c64::new(0.8612569532180622, 0.5081697162696146),
    c64::new(0.8608669386377673, 0.508830142543107),
    c64::new(0.8604764176316321, 0.5094902694849364),
    c64::new(0.8600853904293901, 0.5101500967067668),
    c64::new(0.8596938572610726, 0.510809623820439),
    c64::new(0.8593018183570085, 0.5114688504379703),
    c64::new(0.8589092739478239, 0.5121277761715547),
    c64::new(0.8585162242644429, 0.512786400633563),
    c64::new(0.8581226695380861, 0.5134447234365435),
    c64::new(0.8577286100002721, 0.5141027441932217),
    c64::new(0.8573340458828156, 0.5147604625165012),
    c64::new(0.8569389774178288, 0.5154178780194629),
    c64::new(0.85654340483772, 0.5160749903153666),
    c64::new(0.8561473283751945, 0.5167317990176499),
    c64::new(0.8557507482632538, 0.5173883037399291),
    c64::new(0.855353664735196, 0.5180445040959993),
    c64::new(0.8549560780246149, 0.518700399699835),
    c64::new(0.8545579883654005, 0.5193559901655896),
    c64::new(0.8541593959917388, 0.520011275107596),
    c64::new(0.8537603011381114, 0.5206662541403672),
    c64::new(0.8533607040392954, 0.5213209268785957),
    c64::new(0.8529606049303636, 0.5219752929371544),
    c64::new(0.8525600040466841, 0.5226293519310966),
    c64::new(0.8521589016239198, 0.5232831034756564),
    c64::new(0.8517572978980291, 0.5239365471862486),
    c64::new(0.8513551931052652, 0.524589682678469),
    c64::new(0.8509525874821757, 0.5252425095680947),
    c64::new(0.8505494812656035, 0.5258950274710846),
    c64::new(0.8501458746926852, 0.5265472360035794),
    c64::new(0.8497417680008525, 0.5271991347819014),
    c64::new(0.8493371614278308, 0.5278507234225552),
    c64::new(0.8489320552116396, 0.5285020015422285),
    c64::new(0.8485264495905926, 0.5291529687577906),
    c64::new(0.8481203448032972, 0.5298036246862946),
    c64::new(0.8477137410886544, 0.5304539689449763),
    c64::new(0.8473066386858583, 0.531104001151255),
    c64::new(0.8468990378343972, 0.5317537209227333),
    c64::new(0.8464909387740521, 0.5324031278771979),
    c64::new(0.846082341744897, 0.5330522216326194),
    c64::new(0.8456732469872991, 0.533701001807153),
    c64::new(0.8452636547419182, 0.5343494680191375),
    c64::new(0.8448535652497071, 0.5349976198870972),
    c64::new(0.8444429787519107, 0.5356454570297411),
    c64::new(0.8440318954900664, 0.5362929790659632),
    c64::new(0.8436203157060042, 0.5369401856148429),
    c64::new(0.8432082396418454, 0.5375870762956454),
    c64::new(0.8427956675400041, 0.5382336507278217),
    c64::new(0.8423825996431858, 0.5388799085310084),
    c64::new(0.8419690361943877, 0.5395258493250289),
    c64::new(0.8415549774368984, 0.5401714727298929),
    c64::new(0.8411404236142981, 0.5408167783657967),
    c64::new(0.8407253749704581, 0.5414617658531234),
    c64::new(0.8403098317495408, 0.5421064348124439),
    c64::new(0.8398937941959995, 0.5427507848645159),
    c64::new(0.8394772625545786, 0.5433948156302848),
    c64::new(0.8390602370703127, 0.5440385267308838),
    c64::new(0.8386427179885273, 0.5446819177876345),
    c64::new(0.838224705554838, 0.5453249884220465),
    c64::new(0.8378062000151509, 0.5459677382558176),
    c64::new(0.8373872016156619, 0.5466101669108349),
    c64::new(0.836967710602857, 0.5472522740091741),
    c64::new(0.836547727223512, 0.5478940591731002),
    c64::new(0.8361272517246923, 0.5485355220250674),
    c64::new(0.8357062843537526, 0.5491766621877197),
    c64::new(0.8352848253583374, 0.5498174792838909),
    c64::new(0.83486287498638, 0.5504579729366048),
    c64::new(0.8344404334861032, 0.5510981427690754),
    c64::new(0.8340175011060181, 0.5517379884047073),
    c64::new(0.8335940780949251, 0.5523775094670961),
    c64::new(0.8331701647019132, 0.5530167055800275),
    c64::new(0.8327457611763595, 0.5536555763674793),
    c64::new(0.8323208677679297, 0.55429412145362),
    c64::new(0.8318954847265776, 0.5549323404628104),
    c64::new(0.8314696123025452, 0.5555702330196022),
    c64::new(0.8310432507463623, 0.5562077987487399),
    c64::new(0.8306164003088463, 0.5568450372751601),
    c64::new(0.8301890612411024, 0.5574819482239916),
    c64::new(0.829761233794523, 0.5581185312205561),
    c64::new(0.8293329182207883, 0.5587547858903683),
    c64::new(0.8289041147718649, 0.5593907118591361),
    c64::new(0.8284748237000071, 0.5600263087527604),
    c64::new(0.8280450452577558, 0.560661576197336),
    c64::new(0.8276147796979384, 0.5612965138191515),
    c64::new(0.8271840272736691, 0.5619311212446895),
    c64::new(0.8267527882383485, 0.5625653981006264),
    c64::new(0.8263210628456635, 0.5631993440138341),
    c64::new(0.8258888513495868, 0.5638329586113782),
    c64::new(0.8254561540043776, 0.5644662415205195),
    c64::new(0.8250229710645802, 0.565099192368714),
    c64::new(0.8245893027850253, 0.5657318107836131),
    c64::new(0.8241551494208286, 0.5663640963930638),
    c64::new(0.8237205112273914, 0.5669960488251087),
    c64::new(0.8232853884604001, 0.5676276677079862),
    c64::new(0.8228497813758264, 0.5682589526701315),
    c64::new(0.8224136902299264, 0.5688899033401759),
    c64::new(0.8219771152792416, 0.5695205193469471),
    c64::new(0.8215400567805976, 0.5701508003194703),
    c64::new(0.8211025149911046, 0.5707807458869673),
    c64::new(0.8206644901681575, 0.5714103556788572),
    c64::new(0.8202259825694347, 0.572039629324757),
    c64::new(0.819786992452899, 0.5726685664544812),
    c64::new(0.819347520076797, 0.5732971666980422),
    c64::new(0.818907565699659, 0.5739254296856507),
    c64::new(0.8184671295802987, 0.5745533550477158),
    c64::new(0.8180262119778134, 0.5751809424148451),
    c64::new(0.8175848131515837, 0.5758081914178453),
    c64::new(0.817142933361273, 0.5764351016877218),
    c64::new(0.8167005728668278, 0.5770616728556794),
    c64::new(0.8162577319284774, 0.5776879045531228),
    c64::new(0.8158144108067338, 0.5783137964116556),
    c64::new(0.8153706097623913, 0.5789393480630818),
    c64::new(0.8149263290565266, 0.5795645591394056),
    c64::new(0.8144815689504986, 0.5801894292728317),
    c64::new(0.8140363297059484, 0.5808139580957645),
    c64::new(0.8135906115847985, 0.5814381452408102),
    c64::new(0.8131444148492536, 0.5820619903407754),
    c64::new(0.8126977397617996, 0.5826854930286685),
    c64::new(0.8122505865852039, 0.5833086529376983),
    c64::new(0.8118029555825155, 0.5839314697012762),
    c64::new(0.8113548470170637, 0.5845539429530153),
    c64::new(0.8109062611524597, 0.5851760723267304),
    c64::new(0.8104571982525948, 0.5857978574564389),
    c64::new(0.8100076585816411, 0.5864192979763605),
    c64::new(0.8095576424040513, 0.587040393520918),
    c64::new(0.8091071499845582, 0.5876611437247367),
    c64::new(0.808656181588175, 0.5882815482226452),
    c64::new(0.8082047374801948, 0.5889016066496757),
    c64::new(0.8077528179261904, 0.5895213186410639),
    c64::new(0.8073004231920144, 0.5901406838322488),
    c64::new(0.8068475535437993, 0.5907597018588742),
    c64::new(0.8063942092479564, 0.5913783723567876),
    c64::new(0.8059403905711763, 0.591996694962041),
    c64::new(0.8054860977804292, 0.5926146693108911),
    c64::new(0.8050313311429637, 0.5932322950397998),
    c64::new(0.8045760909263071, 0.5938495717854336),
    c64::new(0.8041203773982657, 0.5944664991846644),
    c64::new(0.8036641908269241, 0.59508307687457),
    c64::new(0.8032075314806449, 0.5956993044924334),
    c64::new(0.8027503996280692, 0.5963151816757437),
    c64::new(0.8022927955381157, 0.5969307080621965),
    c64::new(0.8018347194799813, 0.5975458832896932),
    c64::new(0.8013761717231402, 0.5981607069963424),
    c64::new(0.8009171525373443, 0.5987751788204587),
    c64::new(0.8004576621926228, 0.5993892984005645),
    c64::new(0.7999977009592819, 0.6000030653753889),
    c64::new(0.799537269107905, 0.600616479383869),
    c64::new(0.7990763669093524, 0.6012295400651485),
    c64::new(0.7986149946347609, 0.60184224705858),
    c64::new(0.7981531525555438, 0.6024546000037238),
    c64::new(0.7976908409433912, 0.6030665985403482),
    c64::new(0.7972280600702688, 0.6036782423084304),
    c64::new(0.7967648102084188, 0.604289530948156),
    c64::new(0.7963010916303591, 0.6049004640999198),
    c64::new(0.7958369046088836, 0.6055110414043255),
    c64::new(0.7953722494170613, 0.6061212625021861),
    c64::new(0.794907126328237, 0.6067311270345245),
    c64::new(0.7944415356160306, 0.6073406346425729),
    c64::new(0.7939754775543372, 0.6079497849677736),
    c64::new(0.7935089524173267, 0.6085585776517795),
    c64::new(0.7930419604794436, 0.6091670123364532),
    c64::new(0.7925745020154077, 0.6097750886638684),
    c64::new(0.7921065773002124, 0.6103828062763095),
    c64::new(0.7916381866091259, 0.6109901648162717),
    c64::new(0.7911693302176901, 0.6115971639264619),
    c64::new(0.7907000084017216, 0.612203803249798),
    c64::new(0.79023022143731, 0.6128100824294097),
    c64::new(0.7897599696008191, 0.6134160011086386),
    c64::new(0.7892892531688857, 0.6140215589310385),
    c64::new(0.7888180724184203, 0.614626755540375),
    c64::new(0.7883464276266063, 0.6152315905806268),
    c64::new(0.7878743190709002, 0.6158360636959851),
    c64::new(0.7874017470290313, 0.6164401745308536),
    c64::new(0.7869287117790018, 0.6170439227298498),
    c64::new(0.7864552135990858, 0.6176473079378039),
    c64::new(0.7859812527678302, 0.6182503297997602),
    c64::new(0.7855068295640539, 0.6188529879609763),
    c64::new(0.7850319442668481, 0.619455282066924),
    c64::new(0.7845565971555752, 0.6200572117632891),
    c64::new(0.78408078850987, 0.6206587766959721),
    c64::new(0.7836045186096383, 0.6212599765110876),
    c64::new(0.7831277877350573, 0.6218608108549654),
    c64::new(0.7826505961665757, 0.62246127937415),
    c64::new(0.782172944184913, 0.6230613817154013),
    c64::new(0.7816948320710595, 0.6236611175256945),
    c64::new(0.7812162601062761, 0.6242604864522207),
    c64::new(0.7807372285720945, 0.6248594881423863),
    c64::new(0.7802577377503166, 0.6254581222438144),
    c64::new(0.7797777879230146, 0.6260563884043435),
    c64::new(0.7792973793725304, 0.6266542862720293),
    c64::new(0.778816512381476, 0.6272518154951441),
    c64::new(0.7783351872327332, 0.6278489757221766),
    c64::new(0.7778534042094531, 0.6284457666018327),
    c64::new(0.7773711635950563, 0.629042187783036),
    c64::new(0.7768884656732324, 0.629638238914927),
    c64::new(0.7764053107279404, 0.6302339196468644),
    c64::new(0.7759216990434077, 0.6308292296284245),
    c64::new(0.7754376309041305, 0.6314241685094019),
    c64::new(0.7749531065948738, 0.6320187359398091),
    c64::new(0.7744681264006709, 0.6326129315698775),
    c64::new(0.7739826906068229, 0.6332067550500572),
    c64::new(0.773496799498899, 0.6338002060310173),
    c64::new(0.773010453362737, 0.6343932841636455),
    c64::new(0.7725236524844413, 0.6349859890990495),
    c64::new(0.7720363971503845, 0.6355783204885561),
    c64::new(0.7715486876472063, 0.6361702779837122),
    c64::new(0.7710605242618138, 0.6367618612362842),
    c64::new(0.7705719072813808, 0.6373530698982591),
    c64::new(0.7700828369933479, 0.637943903621844),
    c64::new(0.7695933136854229, 0.6385343620594668),
    c64::new(0.7691033376455797, 0.6391244448637757),
    c64::new(0.7686129091620584, 0.6397141516876405),
    c64::new(0.7681220285233654, 0.6403034821841517),
    c64::new(0.7676306960182734, 0.6408924360066214),
    c64::new(0.7671389119358204, 0.6414810128085832),
    c64::new(0.7666466765653104, 0.6420692122437925),
    c64::new(0.7661539901963129, 0.6426570339662269),
    c64::new(0.7656608531186625, 0.6432444776300859),
    c64::new(0.765167265622459, 0.6438315428897914),
    c64::new(0.7646732279980671, 0.6444182293999884),
    c64::new(0.7641787405361167, 0.6450045368155439),
    c64::new(0.7636838035275019, 0.6455904647915487),
    c64::new(0.7631884172633814, 0.6461760129833163),
    c64::new(0.762692582035178, 0.6467611810463839),
    c64::new(0.7621962981345789, 0.6473459686365121),
    c64::new(0.7616995658535353, 0.6479303754096853),
    c64::new(0.7612023854842618, 0.6485144010221124),
    c64::new(0.7607047573192369, 0.649098045130226),
    c64::new(0.7602066816512024, 0.6496813073906832),
    c64::new(0.7597081587731634, 0.6502641874603659),
    c64::new(0.759209188978388, 0.650846684996381),
    c64::new(0.7587097725604074, 0.6514287996560598),
    c64::new(0.7582099098130153, 0.6520105310969595),
    c64::new(0.7577096010302682, 0.6525918789768624),
    c64::new(0.7572088465064846, 0.6531728429537768),
    c64::new(0.7567076465362457, 0.6537534226859361),
    c64::new(0.7562060014143945, 0.6543336178318004),
    c64::new(0.755703911436036, 0.6549134280500561),
    c64::new(0.7552013768965365, 0.6554928529996153),
    c64::new(0.7546983980915245, 0.6560718923396176),
    c64::new(0.7541949753168892, 0.656650545729429),
    c64::new(0.7536911088687813, 0.6572288128286425),
    c64::new(0.7531867990436125, 0.6578066932970786),
    c64::new(0.7526820461380553, 0.658384186794785),
    c64::new(0.7521768504490427, 0.6589612929820373),
    c64::new(0.7516712122737684, 0.6595380115193387),
    c64::new(0.7511651319096864, 0.6601143420674205),
    c64::new(0.7506586096545107, 0.6606902842872423),
    c64::new(0.7501516458062151, 0.6612658378399923),
    c64::new(0.7496442406630335, 0.6618410023870869),
    c64::new(0.7491363945234594, 0.6624157775901718),
    c64::new(0.7486281076862453, 0.6629901631111215),
    c64::new(0.7481193804504036, 0.6635641586120398),
    c64::new(0.7476102131152051, 0.66413776375526),
    c64::new(0.7471006059801801, 0.6647109782033448),
    c64::new(0.7465905593451173, 0.6652838016190872),
    c64::new(0.7460800735100638, 0.6658562336655097),
    c64::new(0.7455691487753254, 0.6664282740058654),
    c64::new(0.745057785441466, 0.6669999223036375),
    c64::new(0.7445459838093074, 0.6675711782225403),
    c64::new(0.7440337441799293, 0.6681420414265185),
    c64::new(0.7435210668546692, 0.6687125115797481),
    c64::new(0.7430079521351217, 0.669282588346636),
    c64::new(0.7424944003231393, 0.669852271391821),
    c64::new(0.7419804117208311, 0.6704215603801731),
    c64::new(0.7414659866305633, 0.6709904549767942),
    c64::new(0.7409511253549592, 0.6715589548470183),
    c64::new(0.740435828196898, 0.6721270596564117),
    c64::new(0.7399200954595162, 0.6726947690707729),
    c64::new(0.7394039274462058, 0.673262082756133),
    c64::new(0.7388873244606151, 0.673829000378756),
    c64::new(0.7383702868066486, 0.674395521605139),
    c64::new(0.737852814788466, 0.674961646102012),
    c64::new(0.7373349087104829, 0.6755273735363385),
    c64::new(0.7368165688773698, 0.6760927035753159),
    c64::new(0.7362977955940531, 0.676657635886375),
    c64::new(0.7357785891657136, 0.6772221701371803),
    c64::new(0.7352589498977868, 0.6777863059956315),
    c64::new(0.7347388780959634, 0.6783500431298615),
    c64::new(0.7342183740661883, 0.6789133812082384),
    c64::new(0.7336974381146604, 0.679476319899365),
    c64::new(0.7331760705478327, 0.6800388588720789),
    c64::new(0.7326542716724128, 0.680600997795453),
    c64::new(0.7321320417953614, 0.6811627363387954),
    c64::new(0.7316093812238926, 0.6817240741716497),
    c64::new(0.7310862902654742, 0.6822850109637956),
    c64::new(0.7305627692278276, 0.6828455463852481),
    c64::new(0.7300388184189261, 0.6834056801062587),
    c64::new(0.729514438146997, 0.6839654117973155),
    c64::new(0.7289896287205193, 0.6845247411291423),
    c64::new(0.7284643904482252, 0.6850836677727004),
    c64::new(0.7279387236390986, 0.6856421913991875),
    c64::new(0.7274126286023758, 0.6862003116800386),
    c64::new(0.7268861056475449, 0.6867580282869259),
    c64::new(0.726359155084346, 0.687315340891759),
    c64::new(0.7258317772227704, 0.6878722491666855),
    c64::new(0.7253039723730608, 0.6884287527840904),
    c64::new(0.7247757408457113, 0.688984851416597),
    c64::new(0.724247082951467, 0.6895405447370668),
    c64::new(0.7237179990013234, 0.6900958324186),
    c64::new(0.7231884893065275, 0.6906507141345346),
    c64::new(0.7226585541785757, 0.6912051895584485),
    c64::new(0.7221281939292153, 0.6917592583641577),
    c64::new(0.7215974088704438, 0.6923129202257182),
    c64::new(0.7210661993145081, 0.6928661748174246),
    c64::new(0.7205345655739053, 0.6934190218138118),
    c64::new(0.7200025079613817, 0.6939714608896539),
    c64::new(0.719470026789933, 0.6945234917199655),
    c64::new(0.7189371223728045, 0.6950751139800009),
    c64::new(0.7184037950234898, 0.6956263273452549),
    c64::new(0.7178700450557317, 0.696177131491463),
    c64::new(0.7173358727835217, 0.6967275260946011),
    c64::new(0.7168012785210995, 0.6972775108308865),
    c64::new(0.7162662625829532, 0.6978270853767773),
    c64::new(0.7157308252838186, 0.6983762494089729),
    c64::new(0.7151949669386801, 0.6989250026044141),
    c64::new(0.7146586878627691, 0.6994733446402838),
    c64::new(0.7141219883715648, 0.7000212751940063),
    c64::new(0.7135848687807935, 0.7005687939432483),
    c64::new(0.7130473294064292, 0.7011159005659187),
    c64::new(0.7125093705646923, 0.7016625947401685),
    c64::new(0.7119709925720501, 0.7022088761443919),
    c64::new(0.7114321957452164, 0.7027547444572253),
    c64::new(0.7108929804011517, 0.7033001993575487),
    c64::new(0.7103533468570624, 0.7038452405244849),
    c64::new(0.7098132954304008, 0.7043898676374004),
    c64::new(0.7092728264388657, 0.7049340803759049),
    c64::new(0.7087319402004006, 0.7054778784198521),
    c64::new(0.7081906370331954, 0.7060212614493397),
    c64::new(0.7076489172556844, 0.7065642291447095),
    c64::new(0.7071067811865476, 0.7071067811865475),
    c64::new(0.7065642291447096, 0.7076489172556844),
    c64::new(0.7060212614493399, 0.7081906370331953),
    c64::new(0.7054778784198522, 0.7087319402004006),
    c64::new(0.704934080375905, 0.7092728264388656),
    c64::new(0.7043898676374005, 0.7098132954304008),
    c64::new(0.703845240524485, 0.7103533468570623),
    c64::new(0.7033001993575487, 0.7108929804011517),
    c64::new(0.7027547444572254, 0.7114321957452163),
    c64::new(0.7022088761443919, 0.71197099257205),
    c64::new(0.7016625947401686, 0.7125093705646923),
    c64::new(0.7011159005659188, 0.7130473294064292),
    c64::new(0.7005687939432484, 0.7135848687807935),
    c64::new(0.7000212751940064, 0.7141219883715647),
    c64::new(0.6994733446402839, 0.714658687862769),
    c64::new(0.6989250026044143, 0.7151949669386801),
    c64::new(0.6983762494089729, 0.7157308252838186),
    c64::new(0.6978270853767774, 0.716266262582953),
    c64::new(0.6972775108308866, 0.7168012785210994),
    c64::new(0.6967275260946012, 0.7173358727835216),
    c64::new(0.6961771314914631, 0.7178700450557316),
    c64::new(0.695626327345255, 0.7184037950234897),
    c64::new(0.6950751139800009, 0.7189371223728044),
    c64::new(0.6945234917199656, 0.719470026789933),
    c64::new(0.693971460889654, 0.7200025079613817),
    c64::new(0.6934190218138119, 0.7205345655739053),
    c64::new(0.6928661748174247, 0.721066199314508),
    c64::new(0.6923129202257183, 0.7215974088704437),
    c64::new(0.6917592583641579, 0.7221281939292152),
    c64::new(0.6912051895584486, 0.7226585541785756),
    c64::new(0.6906507141345347, 0.7231884893065272),
    c64::new(0.6900958324186001, 0.7237179990013234),
    c64::new(0.6895405447370669, 0.7242470829514669),
    c64::new(0.6889848514165972, 0.7247757408457112),
    c64::new(0.6884287527840905, 0.7253039723730607),
    c64::new(0.6878722491666855, 0.7258317772227703),
    c64::new(0.6873153408917592, 0.7263591550843459),
    c64::new(0.686758028286926, 0.7268861056475449),
    c64::new(0.6862003116800387, 0.7274126286023757),
    c64::new(0.6856421913991876, 0.7279387236390986),
    c64::new(0.6850836677727005, 0.7284643904482252),
    c64::new(0.6845247411291424, 0.7289896287205193),
    c64::new(0.6839654117973155, 0.7295144381469969),
    c64::new(0.6834056801062588, 0.7300388184189261),
    c64::new(0.6828455463852481, 0.7305627692278276),
    c64::new(0.6822850109637957, 0.7310862902654742),
    c64::new(0.6817240741716498, 0.7316093812238925),
    c64::new(0.6811627363387954, 0.7321320417953613),
    c64::new(0.6806009977954531, 0.7326542716724127),
    c64::new(0.680038858872079, 0.7331760705478327),
    c64::new(0.6794763198993651, 0.7336974381146603),
    c64::new(0.6789133812082385, 0.7342183740661882),
    c64::new(0.6783500431298616, 0.7347388780959634),
    c64::new(0.6777863059956316, 0.7352589498977868),
    c64::new(0.6772221701371804, 0.7357785891657135),
    c64::new(0.676657635886375, 0.7362977955940531),
    c64::new(0.676092703575316, 0.7368165688773698),
    c64::new(0.6755273735363386, 0.7373349087104828),
    c64::new(0.674961646102012, 0.737852814788466),
    c64::new(0.674395521605139, 0.7383702868066485),
    c64::new(0.6738290003787561, 0.7388873244606151),
    c64::new(0.6732620827561331, 0.7394039274462056),
    c64::new(0.672694769070773, 0.7399200954595161),
    c64::new(0.6721270596564118, 0.740435828196898),
    c64::new(0.6715589548470184, 0.740951125354959),
    c64::new(0.6709904549767943, 0.7414659866305633),
    c64::new(0.6704215603801732, 0.741980411720831),
    c64::new(0.6698522713918211, 0.7424944003231391),
    c64::new(0.6692825883466361, 0.7430079521351216),
    c64::new(0.6687125115797481, 0.743521066854669),
    c64::new(0.6681420414265186, 0.7440337441799292),
    c64::new(0.6675711782225404, 0.7445459838093073),
    c64::new(0.6669999223036376, 0.745057785441466),
    c64::new(0.6664282740058654, 0.7455691487753253),
    c64::new(0.6658562336655097, 0.7460800735100637),
    c64::new(0.6652838016190872, 0.7465905593451172),
    c64::new(0.6647109782033449, 0.7471006059801801),
    c64::new(0.66413776375526, 0.7476102131152051),
    c64::new(0.6635641586120399, 0.7481193804504035),
    c64::new(0.6629901631111215, 0.7486281076862453),
    c64::new(0.6624157775901718, 0.7491363945234593),
    c64::new(0.6618410023870869, 0.7496442406630335),
    c64::new(0.6612658378399923, 0.750151645806215),
    c64::new(0.6606902842872423, 0.7506586096545106),
    c64::new(0.6601143420674206, 0.7511651319096864),
    c64::new(0.6595380115193388, 0.7516712122737684),
    c64::new(0.6589612929820373, 0.7521768504490427),
    c64::new(0.6583841867947852, 0.7526820461380552),
    c64::new(0.6578066932970787, 0.7531867990436124),
    c64::new(0.6572288128286427, 0.7536911088687812),
    c64::new(0.656650545729429, 0.7541949753168892),
    c64::new(0.6560718923396177, 0.7546983980915244),
    c64::new(0.6554928529996155, 0.7552013768965364),
    c64::new(0.6549134280500561, 0.7557039114360358),
    c64::new(0.6543336178318006, 0.7562060014143945),
    c64::new(0.6537534226859362, 0.7567076465362457),
    c64::new(0.6531728429537769, 0.7572088465064845),
    c64::new(0.6525918789768625, 0.7577096010302681),
    c64::new(0.6520105310969596, 0.7582099098130153),
    c64::new(0.6514287996560599, 0.7587097725604073),
    c64::new(0.650846684996381, 0.759209188978388),
    c64::new(0.650264187460366, 0.7597081587731633),
    c64::new(0.6496813073906833, 0.7602066816512023),
    c64::new(0.6490980451302261, 0.7607047573192368),
    c64::new(0.6485144010221126, 0.7612023854842618),
    c64::new(0.6479303754096855, 0.7616995658535353),
    c64::new(0.6473459686365122, 0.7621962981345789),
    c64::new(0.646761181046384, 0.7626925820351779),
    c64::new(0.6461760129833164, 0.7631884172633813),
    c64::new(0.6455904647915488, 0.7636838035275018),
    c64::new(0.645004536815544, 0.7641787405361167),
    c64::new(0.6444182293999885, 0.764673227998067),
    c64::new(0.6438315428897915, 0.7651672656224588),
    c64::new(0.6432444776300859, 0.7656608531186624),
    c64::new(0.642657033966227, 0.7661539901963128),
    c64::new(0.6420692122437927, 0.7666466765653104),
    c64::new(0.6414810128085832, 0.7671389119358203),
    c64::new(0.6408924360066214, 0.7676306960182733),
    c64::new(0.6403034821841518, 0.7681220285233653),
    c64::new(0.6397141516876406, 0.7686129091620583),
    c64::new(0.6391244448637758, 0.7691033376455796),
    c64::new(0.6385343620594668, 0.7695933136854229),
    c64::new(0.6379439036218442, 0.7700828369933479),
    c64::new(0.6373530698982591, 0.7705719072813807),
    c64::new(0.6367618612362843, 0.7710605242618137),
    c64::new(0.6361702779837123, 0.7715486876472062),
    c64::new(0.6355783204885562, 0.7720363971503844),
    c64::new(0.6349859890990496, 0.7725236524844412),
    c64::new(0.6343932841636456, 0.7730104533627369),
    c64::new(0.6338002060310173, 0.773496799498899),
    c64::new(0.6332067550500573, 0.7739826906068228),
    c64::new(0.6326129315698775, 0.7744681264006708),
    c64::new(0.6320187359398091, 0.7749531065948738),
    c64::new(0.6314241685094019, 0.7754376309041304),
    c64::new(0.6308292296284246, 0.7759216990434076),
    c64::new(0.6302339196468645, 0.7764053107279403),
    c64::new(0.6296382389149271, 0.7768884656732324),
    c64::new(0.6290421877830361, 0.7773711635950562),
    c64::new(0.6284457666018327, 0.777853404209453),
    c64::new(0.6278489757221766, 0.7783351872327331),
    c64::new(0.6272518154951442, 0.7788165123814759),
    c64::new(0.6266542862720295, 0.7792973793725303),
    c64::new(0.6260563884043436, 0.7797777879230144),
    c64::new(0.6254581222438144, 0.7802577377503166),
    c64::new(0.6248594881423865, 0.7807372285720944),
    c64::new(0.6242604864522208, 0.781216260106276),
    c64::new(0.6236611175256946, 0.7816948320710594),
    c64::new(0.6230613817154014, 0.7821729441849129),
    c64::new(0.6224612793741501, 0.7826505961665756),
    c64::new(0.6218608108549654, 0.7831277877350573),
    c64::new(0.6212599765110877, 0.7836045186096382),
    c64::new(0.6206587766959722, 0.78408078850987),
    c64::new(0.6200572117632892, 0.7845565971555752),
    c64::new(0.619455282066924, 0.785031944266848),
    c64::new(0.6188529879609764, 0.7855068295640539),
    c64::new(0.6182503297997602, 0.78598125276783),
    c64::new(0.617647307937804, 0.7864552135990858),
    c64::new(0.6170439227298499, 0.7869287117790017),
    c64::new(0.6164401745308536, 0.7874017470290313),
    c64::new(0.6158360636959851, 0.7878743190709001),
    c64::new(0.6152315905806269, 0.7883464276266062),
    c64::new(0.614626755540375, 0.7888180724184202),
    c64::new(0.6140215589310385, 0.7892892531688855),
    c64::new(0.6134160011086387, 0.7897599696008191),
    c64::new(0.6128100824294098, 0.79023022143731),
    c64::new(0.6122038032497981, 0.7907000084017216),
    c64::new(0.611597163926462, 0.7911693302176901),
    c64::new(0.6109901648162718, 0.7916381866091258),
    c64::new(0.6103828062763095, 0.7921065773002123),
    c64::new(0.6097750886638684, 0.7925745020154076),
    c64::new(0.6091670123364533, 0.7930419604794436),
    c64::new(0.6085585776517795, 0.7935089524173266),
    c64::new(0.6079497849677737, 0.7939754775543371),
    c64::new(0.6073406346425729, 0.7944415356160306),
    c64::new(0.6067311270345246, 0.794907126328237),
    c64::new(0.6061212625021862, 0.7953722494170612),
    c64::new(0.6055110414043255, 0.7958369046088835),
    c64::new(0.6049004640999199, 0.796301091630359),
    c64::new(0.6042895309481561, 0.7967648102084187),
    c64::new(0.6036782423084305, 0.7972280600702687),
    c64::new(0.6030665985403483, 0.797690840943391),
    c64::new(0.6024546000037239, 0.7981531525555438),
    c64::new(0.6018422470585801, 0.7986149946347608),
    c64::new(0.6012295400651486, 0.7990763669093524),
    c64::new(0.600616479383869, 0.799537269107905),
    c64::new(0.600003065375389, 0.7999977009592819),
    c64::new(0.5993892984005647, 0.8004576621926227),
    c64::new(0.5987751788204588, 0.8009171525373443),
    c64::new(0.5981607069963424, 0.8013761717231401),
    c64::new(0.5975458832896933, 0.8018347194799813),
    c64::new(0.5969307080621965, 0.8022927955381156),
    c64::new(0.5963151816757438, 0.802750399628069),
    c64::new(0.5956993044924335, 0.8032075314806448),
    c64::new(0.5950830768745701, 0.8036641908269241),
    c64::new(0.5944664991846645, 0.8041203773982657),
    c64::new(0.5938495717854336, 0.804576090926307),
    c64::new(0.5932322950397999, 0.8050313311429635),
    c64::new(0.5926146693108912, 0.8054860977804291),
    c64::new(0.591996694962041, 0.8059403905711763),
    c64::new(0.5913783723567877, 0.8063942092479562),
    c64::new(0.5907597018588743, 0.8068475535437992),
    c64::new(0.5901406838322489, 0.8073004231920144),
    c64::new(0.5895213186410639, 0.8077528179261902),
    c64::new(0.5889016066496758, 0.8082047374801947),
    c64::new(0.5882815482226453, 0.808656181588175),
    c64::new(0.5876611437247368, 0.8091071499845581),
    c64::new(0.5870403935209181, 0.8095576424040511),
    c64::new(0.5864192979763606, 0.8100076585816411),
    c64::new(0.5857978574564389, 0.8104571982525948),
    c64::new(0.5851760723267305, 0.8109062611524597),
    c64::new(0.5845539429530154, 0.8113548470170637),
    c64::new(0.5839314697012763, 0.8118029555825154),
    c64::new(0.5833086529376984, 0.8122505865852039),
    c64::new(0.5826854930286685, 0.8126977397617995),
    c64::new(0.5820619903407755, 0.8131444148492535),
    c64::new(0.5814381452408103, 0.8135906115847984),
    c64::new(0.5808139580957646, 0.8140363297059483),
    c64::new(0.5801894292728318, 0.8144815689504986),
    c64::new(0.5795645591394057, 0.8149263290565265),
    c64::new(0.5789393480630819, 0.8153706097623912),
    c64::new(0.5783137964116557, 0.8158144108067338),
    c64::new(0.5776879045531229, 0.8162577319284774),
    c64::new(0.5770616728556796, 0.8167005728668277),
    c64::new(0.5764351016877218, 0.817142933361273),
    c64::new(0.5758081914178454, 0.8175848131515837),
    c64::new(0.5751809424148452, 0.8180262119778134),
    c64::new(0.5745533550477159, 0.8184671295802987),
    c64::new(0.5739254296856509, 0.818907565699659),
    c64::new(0.5732971666980423, 0.8193475200767969),
    c64::new(0.5726685664544813, 0.8197869924528989),
    c64::new(0.5720396293247572, 0.8202259825694347),
    c64::new(0.5714103556788573, 0.8206644901681575),
    c64::new(0.5707807458869674, 0.8211025149911046),
    c64::new(0.5701508003194704, 0.8215400567805975),
    c64::new(0.5695205193469473, 0.8219771152792414),
    c64::new(0.568889903340176, 0.8224136902299264),
    c64::new(0.5682589526701316, 0.8228497813758263),
    c64::new(0.5676276677079863, 0.8232853884604001),
    c64::new(0.5669960488251087, 0.8237205112273913),
    c64::new(0.566364096393064, 0.8241551494208286),
    c64::new(0.5657318107836132, 0.8245893027850253),
    c64::new(0.5650991923687141, 0.8250229710645802),
    c64::new(0.5644662415205195, 0.8254561540043774),
    c64::new(0.5638329586113783, 0.8258888513495868),
    c64::new(0.5631993440138342, 0.8263210628456634),
    c64::new(0.5625653981006266, 0.8267527882383485),
    c64::new(0.5619311212446895, 0.827184027273669),
    c64::new(0.5612965138191516, 0.8276147796979383),
    c64::new(0.560661576197336, 0.8280450452577557),
    c64::new(0.5600263087527605, 0.828474823700007),
    c64::new(0.5593907118591361, 0.8289041147718648),
    c64::new(0.5587547858903684, 0.8293329182207883),
    c64::new(0.5581185312205562, 0.829761233794523),
    c64::new(0.5574819482239917, 0.8301890612411023),
    c64::new(0.5568450372751602, 0.8306164003088462),
    c64::new(0.55620779874874, 0.8310432507463623),
    c64::new(0.5555702330196023, 0.8314696123025451),
    c64::new(0.5549323404628105, 0.8318954847265776),
    c64::new(0.5542941214536201, 0.8323208677679297),
    c64::new(0.5536555763674793, 0.8327457611763595),
    c64::new(0.5530167055800276, 0.8331701647019132),
    c64::new(0.5523775094670962, 0.8335940780949251),
    c64::new(0.5517379884047074, 0.834017501106018),
    c64::new(0.5510981427690755, 0.8344404334861031),
    c64::new(0.5504579729366049, 0.83486287498638),
    c64::new(0.549817479283891, 0.8352848253583373),
    c64::new(0.5491766621877198, 0.8357062843537525),
    c64::new(0.5485355220250675, 0.8361272517246922),
    c64::new(0.5478940591731003, 0.8365477272235119),
    c64::new(0.5472522740091741, 0.836967710602857),
    c64::new(0.546610166910835, 0.8373872016156618),
    c64::new(0.5459677382558177, 0.8378062000151508),
    c64::new(0.5453249884220466, 0.838224705554838),
    c64::new(0.5446819177876346, 0.8386427179885273),
    c64::new(0.5440385267308839, 0.8390602370703126),
    c64::new(0.5433948156302849, 0.8394772625545784),
    c64::new(0.542750784864516, 0.8398937941959994),
    c64::new(0.542106434812444, 0.8403098317495407),
    c64::new(0.5414617658531236, 0.840725374970458),
    c64::new(0.5408167783657968, 0.8411404236142981),
    c64::new(0.540171472729893, 0.8415549774368983),
    c64::new(0.539525849325029, 0.8419690361943877),
    c64::new(0.5388799085310084, 0.8423825996431858),
    c64::new(0.5382336507278218, 0.8427956675400041),
    c64::new(0.5375870762956455, 0.8432082396418454),
    c64::new(0.5369401856148431, 0.843620315706004),
    c64::new(0.5362929790659632, 0.8440318954900664),
    c64::new(0.5356454570297411, 0.8444429787519107),
    c64::new(0.5349976198870974, 0.844853565249707),
    c64::new(0.5343494680191376, 0.8452636547419182),
    c64::new(0.533701001807153, 0.8456732469872991),
    c64::new(0.5330522216326197, 0.8460823417448968),
    c64::new(0.532403127877198, 0.846490938774052),
    c64::new(0.5317537209227333, 0.8468990378343972),
    c64::new(0.5311040011512551, 0.8473066386858583),
    c64::new(0.5304539689449764, 0.8477137410886543),
    c64::new(0.5298036246862948, 0.8481203448032971),
    c64::new(0.5291529687577907, 0.8485264495905926),
    c64::new(0.5285020015422285, 0.8489320552116396),
    c64::new(0.5278507234225555, 0.8493371614278307),
    c64::new(0.5271991347819014, 0.8497417680008524),
    c64::new(0.5265472360035794, 0.8501458746926852),
    c64::new(0.5258950274710849, 0.8505494812656034),
    c64::new(0.5252425095680947, 0.8509525874821757),
    c64::new(0.524589682678469, 0.8513551931052652),
    c64::new(0.5239365471862487, 0.851757297898029),
    c64::new(0.5232831034756564, 0.8521589016239198),
    c64::new(0.5226293519310968, 0.852560004046684),
    c64::new(0.5219752929371544, 0.8529606049303636),
    c64::new(0.5213209268785957, 0.8533607040392954),
    c64::new(0.5206662541403673, 0.8537603011381113),
    c64::new(0.5200112751075961, 0.8541593959917387),
    c64::new(0.5193559901655896, 0.8545579883654005),
    c64::new(0.5187003996998352, 0.8549560780246148),
    c64::new(0.5180445040959994, 0.855353664735196),
    c64::new(0.5173883037399291, 0.8557507482632538),
    c64::new(0.51673179901765, 0.8561473283751944),
    c64::new(0.5160749903153667, 0.85654340483772),
    c64::new(0.5154178780194631, 0.8569389774178287),
    c64::new(0.5147604625165012, 0.8573340458828156),
    c64::new(0.5141027441932218, 0.8577286100002721),
    c64::new(0.5134447234365436, 0.858122669538086),
    c64::new(0.5127864006335631, 0.8585162242644427),
    c64::new(0.5121277761715547, 0.8589092739478239),
    c64::new(0.5114688504379705, 0.8593018183570083),
    c64::new(0.5108096238204392, 0.8596938572610726),
    c64::new(0.5101500967067668, 0.8600853904293901),
    c64::new(0.5094902694849365, 0.8604764176316321),
    c64::new(0.5088301425431071, 0.8608669386377672),
    c64::new(0.5081697162696148, 0.8612569532180621),
    c64::new(0.507508991052971, 0.8616464611430813),
    c64::new(0.5068479672818633, 0.8620354621836872),
    c64::new(0.5061866453451555, 0.8624239561110405),
    c64::new(0.5055250256318855, 0.8628119426966003),
    c64::new(0.5048631085312676, 0.8631994217121242),
    c64::new(0.5042008944326906, 0.863586392929668),
    c64::new(0.5035383837257176, 0.8639728561215867),
    c64::new(0.502875576800087, 0.864358811060534),
    c64::new(0.5022124740457109, 0.8647442575194624),
    c64::new(0.5015490758526754, 0.8651291952716237),
    c64::new(0.5008853826112409, 0.865513624090569),
    c64::new(0.5002213947118407, 0.8658975437501488),
    c64::new(0.4995571125450819, 0.866280954024513),
    c64::new(0.4988925365017448, 0.866663854688111),
    c64::new(0.4982276669727819, 0.8670462455156926),
    c64::new(0.49756250434931915, 0.8674281262823069),
    c64::new(0.4968970490226547, 0.8678094967633032),
    c64::new(0.49623130138425836, 0.8681903567343312),
    c64::new(0.49556526182577254, 0.8685707059713409),
    c64::new(0.49489893073901137, 0.8689505442505824),
    c64::new(0.4942323085159598, 0.8693298713486067),
    c64::new(0.49356539554877493, 0.8697086870422656),
    c64::new(0.49289819222978415, 0.8700869911087113),
    c64::new(0.4922306989514861, 0.8704647833253976),
    c64::new(0.4915629161065501, 0.8708420634700789),
    c64::new(0.4908948440878152, 0.8712188313208109),
    c64::new(0.49022648328829116, 0.871595086655951),
    c64::new(0.4895578341011576, 0.8719708292541577),
    c64::new(0.4888888969197632, 0.8723460588943914),
    c64::new(0.4882196721376268, 0.8727207753559143),
    c64::new(0.4875501601484361, 0.87309497841829),
    c64::new(0.48688036134604745, 0.8734686678613849),
    c64::new(0.4862102761244866, 0.8738418434653668),
    c64::new(0.48553990487794707, 0.8742145050107062),
    c64::new(0.4848692480007911, 0.8745866522781761),
    c64::new(0.4841983058875492, 0.8749582850488515),
    c64::new(0.4835270789329188, 0.8753294031041108),
    c64::new(0.4828555675317657, 0.8757000062256346),
    c64::new(0.4821837720791229, 0.8760700941954065),
    c64::new(0.48151169297019, 0.8764396667957136),
    c64::new(0.48083933060033396, 0.8768087238091457),
    c64::new(0.4801666853650885, 0.8771772650185959),
    c64::new(0.47949375766015306, 0.8775452902072612),
    c64::new(0.47882054788139405, 0.8779127991586417),
    c64::new(0.4781470564248431, 0.8782797916565415),
    c64::new(0.47747328368669806, 0.8786462674850681),
    c64::new(0.47679923006332225, 0.8790122264286334),
    c64::new(0.47612489595124363, 0.8793776682719532),
    c64::new(0.47545028174715587, 0.8797425928000474),
    c64::new(0.47477538784791723, 0.8801069997982404),
    c64::new(0.47410021465055, 0.8804708890521608),
    c64::new(0.47342476255224153, 0.880834260347742),
    c64::new(0.4727490319503429, 0.881197113471222),
    c64::new(0.47207302324236866, 0.8815594482091438),
    c64::new(0.4713967368259978, 0.8819212643483549),
    c64::new(0.4707201730990717, 0.8822825616760086),
    c64::new(0.4700433324595956, 0.8826433399795628),
    c64::new(0.46936621530573763, 0.8830035990467807),
    c64::new(0.46868882203582796, 0.8833633386657316),
    c64::new(0.46801115304835983, 0.8837225586247897),
    c64::new(0.4673332087419885, 0.884081258712635),
    c64::new(0.46665498951553097, 0.8844394387182537),
    c64::new(0.4659764957679661, 0.8847970984309378),
    c64::new(0.46529772789843465, 0.8851542376402851),
    c64::new(0.4646186863062378, 0.8855108561362),
    c64::new(0.46393937139083846, 0.8858669537088928),
    c64::new(0.46325978355186026, 0.8862225301488806),
    c64::new(0.4625799231890868, 0.886577585246987),
    c64::new(0.46189979070246284, 0.8869321187943421),
    c64::new(0.46121938649209243, 0.8872861305823831),
    c64::new(0.46053871095824, 0.8876396204028539),
    c64::new(0.4598577645013296, 0.8879925880478056),
    c64::new(0.45917654752194415, 0.8883450333095964),
    c64::new(0.4584950604208262, 0.8886969559808917),
    c64::new(0.4578133035988773, 0.8890483558546645),
    c64::new(0.457131277457157, 0.8893992327241955),
    c64::new(0.45644898239688386, 0.8897495863830729),
    c64::new(0.45576641881943475, 0.8900994166251922),
    c64::new(0.45508358712634384, 0.8904487232447579),
    c64::new(0.45440048771930375, 0.8907975060362815),
    c64::new(0.4537171210001639, 0.8911457647945832),
    c64::new(0.4530334873709316, 0.8914934993147914),
    c64::new(0.452349587233771, 0.8918407093923427),
    c64::new(0.45166542099100254, 0.8921873948229825),
    c64::new(0.4509809890451038, 0.8925335554027647),
    c64::new(0.4502962917987087, 0.8928791909280517),
    c64::new(0.4496113296546066, 0.8932243011955153),
    c64::new(0.4489261030157432, 0.893568886002136),
    c64::new(0.44824061228522, 0.8939129451452033),
    c64::new(0.447554857866293, 0.8942564784223161),
    c64::new(0.4468688401623743, 0.8945994856313826),
    c64::new(0.4461825595770301, 0.8949419665706206),
    c64::new(0.44549601651398174, 0.8952839210385576),
    c64::new(0.444809211377105, 0.89562534883403),
    c64::new(0.44412214457042926, 0.8959662497561851),
    c64::new(0.44343481649813843, 0.8963066236044797),
    c64::new(0.44274722756457013, 0.8966464701786802),
    c64::new(0.44205937817421476, 0.896985789278864),
    c64::new(0.4413712687317166, 0.8973245807054183),
    c64::new(0.440682899641873, 0.8976628442590407),
    c64::new(0.43999427130963326, 0.8980005797407399),
    c64::new(0.43930538414010006, 0.8983377869518342),
    c64::new(0.4386162385385277, 0.8986744656939538),
    c64::new(0.43792683491032286, 0.8990106157690391),
    c64::new(0.4372371736610442, 0.8993462369793415),
    c64::new(0.43654725519640125, 0.8996813291274239),
    c64::new(0.4358570799222555, 0.9000158920161603),
    c64::new(0.43516664824461937, 0.9003499254487356),
    c64::new(0.4344759605696557, 0.9006834292286469),
    c64::new(0.4337850173036785, 0.9010164031597023),
    c64::new(0.433093818853152, 0.901348847046022),
    c64::new(0.43240236562469014, 0.9016807606920377),
    c64::new(0.43171065802505737, 0.9020121439024931),
    c64::new(0.4310186964611671, 0.9023429964824442),
    c64::new(0.4303264813400826, 0.9026733182372588),
    c64::new(0.4296340130690165, 0.9030031089726172),
    c64::new(0.42894129205532955, 0.9033323684945118),
    c64::new(0.4282483187065319, 0.903661096609248),
    c64::new(0.4275550934302822, 0.9039892931234433),
    c64::new(0.4268616166343865, 0.9043169578440283),
    c64::new(0.42616788872679956, 0.9046440905782462),
    c64::new(0.4254739101156239, 0.9049706911336532),
    c64::new(0.4247796812091088, 0.9052967593181188),
    c64::new(0.4240852024156517, 0.9056222949398252),
    c64::new(0.4233904741437961, 0.9059472978072685),
    c64::new(0.42269549680223295, 0.9062717677292577),
    c64::new(0.4220002707997998, 0.9065957045149153),
    c64::new(0.4213047965454797, 0.906919107973678),
    c64::new(0.4206090744484025, 0.9072419779152958),
    c64::new(0.4199131049178437, 0.9075643141498325),
    c64::new(0.41921688836322396, 0.9078861164876662),
    c64::new(0.4185204251941097, 0.9082073847394887),
    c64::new(0.4178237158202124, 0.9085281187163061),
    c64::new(0.4171267606513879, 0.9088483182294391),
    c64::new(0.4164295600976373, 0.9091679830905224),
    c64::new(0.4157321145691054, 0.9094871131115054),
    c64::new(0.41503442447608163, 0.9098057081046522),
    c64::new(0.4143364902289992, 0.9101237678825416),
    c64::new(0.41363831223843456, 0.9104412922580671),
    c64::new(0.412939890915108, 0.9107582810444376),
    c64::new(0.412241226669883, 0.9110747340551762),
    c64::new(0.4115423199137653, 0.9113906511041223),
    c64::new(0.4108431710579039, 0.9117060320054299),
    c64::new(0.41014378051359035, 0.9120208765735682),
    c64::new(0.4094441486922576, 0.9123351846233227),
    c64::new(0.4087442760054815, 0.9126489559697939),
    c64::new(0.40804416286497874, 0.9129621904283981),
    c64::new(0.40734380968260797, 0.9132748878148678),
    c64::new(0.40664321687036914, 0.9135870479452508),
    c64::new(0.4059423848404025, 0.9138986706359117),
    c64::new(0.4052413140049898, 0.9142097557035307),
    c64::new(0.4045400047765531, 0.9145203029651044),
    c64::new(0.40383845756765413, 0.9148303122379462),
    c64::new(0.40313667279099524, 0.9151397833396853),
    c64::new(0.40243465085941854, 0.9154487160882678),
    c64::new(0.401732392185905, 0.9157571103019567),
    c64::new(0.4010298971835758, 0.9160649657993316),
    c64::new(0.40032716626569015, 0.9163722823992891),
    c64::new(0.3996241998456468, 0.9166790599210427),
    c64::new(0.398920998336983, 0.9169852981841229),
    c64::new(0.3982175621533736, 0.9172909970083779),
    c64::new(0.3975138917086323, 0.917596156213973),
    c64::new(0.3968099874167104, 0.9179007756213904),
    c64::new(0.3961058496916963, 0.9182048550514309),
    c64::new(0.3954014789478163, 0.9185083943252123),
    c64::new(0.39469687559943367, 0.9188113932641699),
    c64::new(0.3939920400610481, 0.9191138516900578),
    c64::new(0.39328697274729657, 0.919415769424947),
    c64::new(0.3925816740729515, 0.9197171462912274),
    c64::new(0.39187614445292235, 0.9200179821116066),
    c64::new(0.391170384302254, 0.9203182767091105),
    c64::new(0.39046439403612665, 0.9206180299070839),
    c64::new(0.3897581740698564, 0.9209172415291895),
    c64::new(0.3890517248188945, 0.9212159113994087),
    c64::new(0.3883450466988263, 0.9215140393420419),
    c64::new(0.3876381401253727, 0.9218116251817081),
    c64::new(0.3869310055143887, 0.9221086687433451),
    c64::new(0.386223643281863, 0.92240516985221),
    c64::new(0.385516053843919, 0.9227011283338785),
    c64::new(0.38480823761681293, 0.9229965440142462),
    c64::new(0.38410019501693504, 0.9232914167195276),
    c64::new(0.3833919264608088, 0.9235857462762566),
    c64::new(0.38268343236508984, 0.9238795325112867),
    c64::new(0.3819747131465672, 0.9241727752517912),
    c64::new(0.3812657692221625, 0.9244654743252626),
    c64::new(0.3805566010089286, 0.9247576295595139),
    c64::new(0.3798472089240511, 0.9250492407826776),
    c64::new(0.3791375933848474, 0.9253403078232063),
    c64::new(0.37842775480876556, 0.9256308305098727),
    c64::new(0.3777176936133858, 0.92592080867177),
    c64::new(0.3770074102164183, 0.9262102421383114),
    c64::new(0.3762969050357048, 0.9264991307392305),
    c64::new(0.3755861784892173, 0.9267874743045817),
    c64::new(0.3748752309950576, 0.9270752726647401),
    c64::new(0.37416406297145793, 0.9273625256504011),
    c64::new(0.3734526748367804, 0.9276492330925812),
    c64::new(0.3727410670095158, 0.9279353948226179),
    c64::new(0.37202923990828496, 0.9282210106721694),
    c64::new(0.3713171939518376, 0.9285060804732155),
    c64::new(0.37060492955905167, 0.928790604058057),
    c64::new(0.3698924471489342, 0.9290745812593157),
    c64::new(0.36917974714062, 0.9293580119099355),
    c64::new(0.3684668299533723, 0.9296408958431812),
    c64::new(0.3677536960065821, 0.9299232328926396),
    c64::new(0.36704034571976724, 0.9302050228922191),
    c64::new(0.3663267795125736, 0.9304862656761497),
    c64::new(0.36561299780477396, 0.9307669610789837),
    c64::new(0.3648990010162673, 0.9310471089355953),
    c64::new(0.36418478956707984, 0.9313267090811805),
    c64::new(0.36347036387736387, 0.9316057613512578),
    c64::new(0.3627557243673972, 0.9318842655816681),
    c64::new(0.36204087145758435, 0.9321622216085743),
    c64::new(0.36132580556845434, 0.9324396292684624),
    c64::new(0.36061052712066227, 0.9327164883981403),
    c64::new(0.3598950365349883, 0.9329927988347388),
    c64::new(0.35917933423233656, 0.933268560415712),
    c64::new(0.35846342063373654, 0.9335437729788362),
    c64::new(0.35774729616034195, 0.933818436362211),
    c64::new(0.35703096123343003, 0.9340925504042589),
    c64::new(0.35631441627440236, 0.9343661149437259),
    c64::new(0.35559766170478396, 0.9346391298196808),
    c64::new(0.3548806979462228, 0.9349115948715161),
    c64::new(0.3541635254204905, 0.9351835099389475),
    c64::new(0.35344614454948087, 0.9354548748620146),
    c64::new(0.3527285557552107, 0.9357256894810804),
    c64::new(0.35201075945981924, 0.9359959536368313),
    c64::new(0.35129275608556715, 0.9362656671702783),
    c64::new(0.3505745460548375, 0.9365348299227555),
    c64::new(0.34985612979013503, 0.9368034417359216),
    c64::new(0.349137507714085, 0.9370715024517592),
    c64::new(0.3484186802494345, 0.937339011912575),
    c64::new(0.34769964781905144, 0.937605969961),
    c64::new(0.3469804108459237, 0.9378723764399899),
    c64::new(0.3462609697531601, 0.9381382311928244),
    c64::new(0.34554132496398915, 0.9384035340631081),
    c64::new(0.3448214769017593, 0.9386682848947702),
    c64::new(0.344101425989939, 0.9389324835320645),
    c64::new(0.3433811726521151, 0.9391961298195699),
    c64::new(0.3426607173119944, 0.9394592236021899),
    c64::new(0.3419400603934023, 0.9397217647251533),
    c64::new(0.3412192023202824, 0.9399837530340139),
    c64::new(0.3404981435166971, 0.9402451883746509),
    c64::new(0.33977688440682696, 0.9405060705932683),
    c64::new(0.33905542541496964, 0.9407663995363961),
    c64::new(0.3383337669655413, 0.9410261750508893),
    c64::new(0.3376119094830747, 0.9412853969839287),
    c64::new(0.33688985339222005, 0.9415440651830208),
    c64::new(0.33616759911774463, 0.9418021794959976),
    c64::new(0.33544514708453166, 0.9420597397710173),
    c64::new(0.3347224977175812, 0.9423167458565638),
    c64::new(0.3339996514420095, 0.9425731976014469),
    c64::new(0.3332766086830479, 0.9428290948548027),
    c64::new(0.33255336986604417, 0.9430844374660935),
    c64::new(0.3318299354164612, 0.9433392252851077),
    c64::new(0.33110630575987643, 0.9435934581619604),
    c64::new(0.3303824813219829, 0.9438471359470927),
    c64::new(0.32965846252858755, 0.9441002584912727),
    c64::new(0.3289342498056122, 0.9443528256455948),
    c64::new(0.32820984357909266, 0.9446048372614803),
    c64::new(0.32748524427517806, 0.9448562931906772),
    c64::new(0.32676045232013173, 0.9451071932852606),
    c64::new(0.32603546814033035, 0.9453575373976323),
    c64::new(0.325310292162263, 0.9456073253805213),
    c64::new(0.3245849248125321, 0.9458565570869839),
    c64::new(0.32385936651785296, 0.9461052323704033),
    c64::new(0.32313361770505233, 0.9463533510844906),
    c64::new(0.32240767880106996, 0.9466009130832835),
    c64::new(0.32168155023295664, 0.946847918221148),
    c64::new(0.3209552324278752, 0.9470943663527772),
    c64::new(0.3202287258131, 0.9473402573331919),
    c64::new(0.31950203081601575, 0.9475855910177411),
    c64::new(0.3187751478641185, 0.947830367262101),
    c64::new(0.31804807738501506, 0.9480745859222762),
    c64::new(0.3173208198064218, 0.9483182468545991),
    c64::new(0.31659337555616585, 0.9485613499157303),
    c64::new(0.31586574506218407, 0.9488038949626584),
    c64::new(0.31513792875252244, 0.9490458818527006),
    c64::new(0.3144099270553368, 0.949287310443502),
    c64::new(0.3136817403988915, 0.9495281805930367),
    c64::new(0.3129533692115602, 0.9497684921596067),
    c64::new(0.31222481392182505, 0.950008245001843),
    c64::new(0.31149607495827597, 0.9502474389787052),
    c64::new(0.3107671527496115, 0.9504860739494817),
    c64::new(0.310038047724638, 0.9507241497737896),
    c64::new(0.3093087603122687, 0.9509616663115751),
    c64::new(0.30857929094152503, 0.9511986234231132),
    c64::new(0.307849640041535, 0.9514350209690083),
    c64::new(0.3071198080415331, 0.9516708588101939),
    c64::new(0.3063897953708611, 0.9519061368079322),
    c64::new(0.30565960245896623, 0.9521408548238158),
    c64::new(0.3049292297354024, 0.9523750127197659),
    c64::new(0.3041986776298293, 0.9526086103580332),
    c64::new(0.30346794657201137, 0.9528416476011987),
    c64::new(0.30273703699181914, 0.9530741243121722),
    c64::new(0.3020059493192282, 0.9533060403541938),
    c64::new(0.301274683984318, 0.9535373955908333),
    c64::new(0.3005432414172734, 0.9537681898859903),
    c64::new(0.29981162204838346, 0.9539984231038945),
    c64::new(0.2990798263080405, 0.9542280951091057),
    c64::new(0.29834785462674157, 0.9544572057665135),
    c64::new(0.2976157074350863, 0.9546857549413383),
    c64::new(0.29688338516377827, 0.9549137424991305),
    c64::new(0.29615088824362396, 0.9551411683057707),
    c64::new(0.29541821710553207, 0.9553680322274702),
    c64::new(0.2946853721805143, 0.9555943341307711),
    c64::new(0.29395235389968477, 0.9558200738825454),
    c64::new(0.2932191626942587, 0.9560452513499964),
    c64::new(0.2924857989955538, 0.9562698664006581),
    c64::new(0.2917522632349894, 0.9564939189023951),
    c64::new(0.2910185558440851, 0.956717408723403),
    c64::new(0.2902846772544623, 0.9569403357322089),
    c64::new(0.28955062789784314, 0.9571626997976701),
    c64::new(0.2888164082060495, 0.9573845007889759),
    c64::new(0.2880820186110043, 0.9576057385756462),
    c64::new(0.28734745954472957, 0.9578264130275329),
    c64::new(0.2866127314393478, 0.9580465240148186),
    c64::new(0.2858778347270807, 0.9582660714080177),
    c64::new(0.2851427698402487, 0.9584850550779761),
    c64::new(0.2844075372112718, 0.9587034748958716),
    c64::new(0.28367213727266855, 0.9589213307332132),
    c64::new(0.2829365704570554, 0.9591386224618419),
    c64::new(0.2822008371971475, 0.9593553499539308),
    c64::new(0.28146493792575805, 0.9595715130819845),
    c64::new(0.2807288730757972, 0.9597871117188399),
    c64::new(0.2799926430802734, 0.9600021457376658),
    c64::new(0.27925624837229124, 0.9602166150119634),
    c64::new(0.27851968938505306, 0.9604305194155658),
    c64::new(0.27778296655185775, 0.9606438588226385),
    c64::new(0.27704608030609995, 0.9608566331076797),
    c64::new(0.27630903108127103, 0.9610688421455194),
    c64::new(0.27557181931095825, 0.9612804858113206),
    c64::new(0.27483444542884394, 0.961491563980579),
    c64::new(0.27409690986870633, 0.9617020765291225),
    c64::new(0.2733592130644188, 0.9619120233331121),
    c64::new(0.272621355449949, 0.9621214042690416),
    c64::new(0.2718833374593599, 0.9623302192137374),
    c64::new(0.27114515952680807, 0.9625384680443592),
    c64::new(0.2704068220865448, 0.9627461506383994),
    c64::new(0.2696683255729152, 0.9629532668736839),
    c64::new(0.2689296704203573, 0.9631598166283714),
    c64::new(0.2681908570634031, 0.963365799780954),
    c64::new(0.26745188593667774, 0.9635712162102572),
    c64::new(0.2667127574748984, 0.9637760657954398),
    c64::new(0.26597347211287553, 0.9639803484159941),
    c64::new(0.2652340302855119, 0.9641840639517457),
    c64::new(0.26449443242780163, 0.9643872122828543),
    c64::new(0.2637546789748315, 0.9645897932898126),
    c64::new(0.26301477036177906, 0.9647918068534479),
    c64::new(0.2622747070239136, 0.9649932528549203),
    c64::new(0.2615344893965956, 0.9651941311757247),
    c64::new(0.26079411791527557, 0.9653944416976894),
    c64::new(0.26005359301549513, 0.9655941843029768),
    c64::new(0.25931291513288635, 0.9657933588740837),
    c64::new(0.2585720847031704, 0.9659919652938406),
    c64::new(0.25783110216215893, 0.9661900034454126),
    c64::new(0.25708996794575323, 0.9663874732122988),
    c64::new(0.2563486824899429, 0.9665843744783331),
    c64::new(0.25560724623080755, 0.9667807071276833),
    c64::new(0.2548656596045146, 0.9669764710448521),
    c64::new(0.2541239230473206, 0.9671716661146766),
    c64::new(0.25338203699557027, 0.9673662922223285),
    c64::new(0.2526400018856956, 0.9675603492533144),
    c64::new(0.2518978181542169, 0.9677538370934755),
    c64::new(0.25115548623774203, 0.9679467556289878),
    c64::new(0.2504130065729653, 0.9681391047463624),
    c64::new(0.24967037959666852, 0.9683308843324452),
    c64::new(0.24892760574572026, 0.9685220942744173),
    c64::new(0.24818468545707478, 0.9687127344597948),
    c64::new(0.24744161916777344, 0.9689028047764289),
    c64::new(0.2466984073149425, 0.9690923051125061),
    c64::new(0.2459550503357946, 0.9692812353565485),
    c64::new(0.24521154866762768, 0.969469595397413),
    c64::new(0.2444679027478242, 0.9696573851242924),
    c64::new(0.24372411301385213, 0.9698446044267148),
    c64::new(0.24298017990326398, 0.970031253194544),
    c64::new(0.24223610385369604, 0.9702173313179792),
    c64::new(0.24149188530286927, 0.9704028386875555),
    c64::new(0.2407475246885885, 0.9705877751941436),
    c64::new(0.2400030224487415, 0.9707721407289504),
    c64::new(0.23925837902130012, 0.970955935183518),
    c64::new(0.2385135948443185, 0.9711391584497251),
    c64::new(0.2377686703559342, 0.9713218104197862),
    c64::new(0.23702360599436734, 0.9715038909862518),
    c64::new(0.23627840219791962, 0.9716854000420085),
    c64::new(0.23553305940497546, 0.9718663374802794),
    c64::new(0.23478757805400105, 0.9720467031946234),
    c64::new(0.23404195858354346, 0.9722264970789363),
    c64::new(0.23329620143223156, 0.9724057190274498),
    c64::new(0.23255030703877533, 0.9725843689347322),
    c64::new(0.23180427584196478, 0.9727624466956886),
    c64::new(0.23105810828067125, 0.9729399522055601),
    c64::new(0.23031180479384553, 0.9731168853599251),
    c64::new(0.22956536582051887, 0.9732932460546982),
    c64::new(0.22881879179980236, 0.973469034186131),
    c64::new(0.2280720831708858, 0.9736442496508119),
    c64::new(0.22732524037303883, 0.9738188923456661),
    c64::new(0.2265782638456101, 0.9739929621679558),
    c64::new(0.2258311540280262, 0.9741664590152803),
    c64::new(0.22508391135979278, 0.9743393827855759),
    c64::new(0.2243365362804937, 0.9745117333771157),
    c64::new(0.22358902922979, 0.9746835106885107),
    c64::new(0.22284139064742128, 0.9748547146187084),
    c64::new(0.2220936209732036, 0.9750253450669941),
    c64::new(0.2213457206470308, 0.9751954019329904),
    c64::new(0.22059769010887365, 0.9753648851166569),
    c64::new(0.21984952979877875, 0.9755337945182914),
    c64::new(0.21910124015686977, 0.9757021300385286),
    c64::new(0.21835282162334643, 0.975869891578341),
    c64::new(0.21760427463848367, 0.976037079039039),
    c64::new(0.21685559964263257, 0.9762036923222706),
    c64::new(0.2161067970762196, 0.9763697313300211),
    c64::new(0.21535786737974555, 0.9765351959646145),
    c64::new(0.21460881099378692, 0.9767000861287118),
    c64::new(0.21385962835899383, 0.9768644017253126),
    c64::new(0.21311031991609136, 0.9770281426577544),
    c64::new(0.21236088610587855, 0.9771913088297123),
    c64::new(0.2116113273692276, 0.9773539001452),
    c64::new(0.21086164414708483, 0.9775159165085693),
    c64::new(0.21011183688046972, 0.9776773578245099),
    c64::new(0.2093619060104742, 0.9778382239980504),
    c64::new(0.20861185197826343, 0.9779985149345571),
    c64::new(0.20786167522507515, 0.978158230539735),
    c64::new(0.20711137619221856, 0.9783173707196277),
    c64::new(0.20636095532107568, 0.9784759353806168),
    c64::new(0.20561041305309932, 0.9786339244294231),
    c64::new(0.20485974982981442, 0.9787913377731057),
    c64::new(0.204108966092817, 0.9789481753190622),
    c64::new(0.20335806228377337, 0.9791044369750292),
    c64::new(0.2026070388444211, 0.979260122649082),
    c64::new(0.20185589621656813, 0.9794152322496348),
    c64::new(0.20110463484209193, 0.9795697656854405),
    c64::new(0.2003532551629404, 0.9797237228655912),
    c64::new(0.19960175762113105, 0.9798771036995176),
    c64::new(0.19885014265875012, 0.9800299080969901),
    c64::new(0.19809841071795373, 0.9801821359681173),
    c64::new(0.197346562240966, 0.980333787223348),
    c64::new(0.19659459767008022, 0.9804848617734694),
    c64::new(0.195842517447658, 0.9806353595296081),
    c64::new(0.1950903220161283, 0.9807852804032304),
    c64::new(0.1943380118179886, 0.9809346243061416),
    c64::new(0.19358558729580372, 0.9810833911504866),
    c64::new(0.19283304889220526, 0.9812315808487497),
    c64::new(0.19208039704989238, 0.9813791933137546),
    c64::new(0.191327632211631, 0.9815262284586647),
    c64::new(0.19057475482025277, 0.9816726861969831),
    c64::new(0.18982176531865658, 0.9818185664425525),
    c64::new(0.18906866414980628, 0.9819638691095552),
    c64::new(0.18831545175673212, 0.9821085941125136),
    c64::new(0.18756212858252974, 0.9822527413662894),
    c64::new(0.18680869507035933, 0.9823963107860847),
    c64::new(0.1860551516634466, 0.9825393022874412),
    c64::new(0.185301498805082, 0.9826817157862409),
    c64::new(0.18454773693861964, 0.9828235511987052),
    c64::new(0.1837938665074784, 0.9829648084413964),
    c64::new(0.18303988795514103, 0.9831054874312163),
    c64::new(0.18228580172515332, 0.9832455880854071),
    c64::new(0.18153160826112513, 0.9833851103215512),
    c64::new(0.18077730800672867, 0.9835240540575713),
    c64::new(0.18002290140569951, 0.9836624192117303),
    c64::new(0.17926838890183588, 0.9838002057026315),
    c64::new(0.17851377093899756, 0.9839374134492189),
    c64::new(0.17775904796110714, 0.9840740423707764),
    c64::new(0.17700422041214886, 0.984210092386929),
    c64::new(0.17624928873616794, 0.9843455634176419),
    c64::new(0.17549425337727137, 0.9844804553832209),
    c64::new(0.1747391147796273, 0.9846147682043126),
    c64::new(0.17398387338746385, 0.9847485018019042),
    c64::new(0.1732285296450705, 0.9848816560973237),
    c64::new(0.17247308399679603, 0.9850142310122398),
    c64::new(0.17171753688704997, 0.9851462264686622),
    c64::new(0.17096188876030136, 0.9852776423889412),
    c64::new(0.17020614006107812, 0.9854084786957684),
    c64::new(0.16945029123396793, 0.9855387353121761),
    c64::new(0.16869434272361744, 0.9856684121615376),
    c64::new(0.1679382949747312, 0.9857975091675674),
    c64::new(0.16718214843207288, 0.9859260262543211),
    c64::new(0.16642590354046422, 0.9860539633461954),
    c64::new(0.16566956074478414, 0.9861813203679283),
    c64::new(0.16491312048997006, 0.9863080972445987),
    c64::new(0.1641565832210159, 0.9864342939016271),
    c64::new(0.16339994938297323, 0.9865599102647754),
    c64::new(0.16264321942095045, 0.9866849462601467),
    c64::new(0.16188639378011188, 0.9868094018141854),
    c64::new(0.16112947290567878, 0.9869332768536777),
    c64::new(0.1603724572429284, 0.987056571305751),
    c64::new(0.1596153472371931, 0.9871792850978743),
    c64::new(0.1588581433338614, 0.9873014181578584),
    c64::new(0.1581008459783771, 0.9874229704138554),
    c64::new(0.15734345561623828, 0.9875439417943592),
    c64::new(0.1565859726929986, 0.9876643322282057),
    c64::new(0.15582839765426532, 0.9877841416445722),
    c64::new(0.1550707309457005, 0.9879033699729778),
    c64::new(0.15431297301302024, 0.9880220171432835),
    c64::new(0.1535551243019935, 0.9881400830856926),
    c64::new(0.1527971852584434, 0.9882575677307495),
    c64::new(0.15203915632824616, 0.9883744710093413),
    c64::new(0.15128103795733025, 0.9884907928526966),
    c64::new(0.15052283059167737, 0.9886065331923864),
    c64::new(0.14976453467732162, 0.9887216919603238),
    c64::new(0.14900615066034847, 0.9888362690887635),
    c64::new(0.1482476789868962, 0.988950264510303),
    c64::new(0.14748912010315365, 0.9890636781578815),
    c64::new(0.14673047445536175, 0.989176509964781),
    c64::new(0.14597174248981235, 0.9892887598646252),
    c64::new(0.14521292465284752, 0.9894004277913804),
    c64::new(0.14445402139086044, 0.9895115136793552),
    c64::new(0.14369503315029458, 0.9896220174632008),
    c64::new(0.1429359603776427, 0.9897319390779106),
    c64::new(0.142176803519448, 0.9898412784588205),
    c64::new(0.14141756302230313, 0.989950035541609),
    c64::new(0.14065823933284924, 0.9900582102622971),
    c64::new(0.13989883289777735, 0.9901658025572484),
    c64::new(0.13913934416382628, 0.9902728123631691),
    c64::new(0.1383797735777839, 0.9903792396171082),
    c64::new(0.13762012158648618, 0.990485084256457),
    c64::new(0.13686038863681643, 0.9905903462189501),
    c64::new(0.13610057517570617, 0.9906950254426646),
    c64::new(0.13534068165013433, 0.9907991218660204),
    c64::new(0.13458070850712622, 0.99090263542778),
    c64::new(0.1338206561937547, 0.9910055660670494),
    c64::new(0.13306052515713918, 0.9911079137232768),
    c64::new(0.13230031584444468, 0.9912096783362541),
    c64::new(0.13154002870288325, 0.9913108598461154),
    c64::new(0.1307796641797118, 0.9914114581933385),
    c64::new(0.13001922272223335, 0.9915114733187439),
    c64::new(0.12925870477779627, 0.9916109051634954),
    c64::new(0.12849811079379322, 0.9917097536690995),
    c64::new(0.12773744121766228, 0.9918080187774064),
    c64::new(0.12697669649688598, 0.9919057004306093),
    c64::new(0.1262158770789904, 0.9920027985712445),
    c64::new(0.1254549834115462, 0.9920993131421918),
    c64::new(0.12469401594216775, 0.9921952440866739),
    c64::new(0.12393297511851219, 0.9922905913482574),
    c64::new(0.12317186138828064, 0.9923853548708517),
    c64::new(0.12241067519921628, 0.99247953459871),
    c64::new(0.12164941699910553, 0.9925731304764288),
    c64::new(0.12088808723577722, 0.992666142448948),
    c64::new(0.12012668635710157, 0.9927585704615511),
    c64::new(0.11936521481099134, 0.9928504144598651),
    c64::new(0.11860367304540084, 0.9929416743898605),
    c64::new(0.11784206150832502, 0.9930323501978514),
    c64::new(0.11708038064780055, 0.9931224418304956),
    c64::new(0.11631863091190486, 0.9932119492347945),
    c64::new(0.11555681274875529, 0.9933008723580933),
    c64::new(0.11479492660651025, 0.9933892111480807),
    c64::new(0.11403297293336728, 0.9934769655527892),
    c64::new(0.11327095217756435, 0.9935641355205953),
    c64::new(0.11250886478737883, 0.9936507210002191),
    c64::new(0.11174671121112666, 0.9937367219407246),
    c64::new(0.11098449189716338, 0.9938221382915197),
    c64::new(0.11022220729388318, 0.9939069700023561),
    c64::new(0.10945985784971803, 0.9939912170233294),
    c64::new(0.10869744401313867, 0.9940748793048794),
    c64::new(0.10793496623265376, 0.9941579567977897),
    c64::new(0.10717242495680887, 0.9942404494531879),
    c64::new(0.10640982063418784, 0.9943223572225458),
    c64::new(0.1056471537134107, 0.9944036800576791),
    c64::new(0.10488442464313497, 0.9944844179107476),
    c64::new(0.10412163387205473, 0.9945645707342554),
    c64::new(0.10335878184889968, 0.9946441384810507),
    c64::new(0.10259586902243627, 0.9947231211043257),
    c64::new(0.10183289584146665, 0.9948015185576171),
    c64::new(0.10106986275482786, 0.9948793307948056),
    c64::new(0.10030677021139282, 0.9949565577701164),
    c64::new(0.09954361866006943, 0.9950331994381186),
    c64::new(0.09878040854979965, 0.9951092557537261),
    c64::new(0.09801714032956077, 0.9951847266721968),
    c64::new(0.09725381444836335, 0.9952596121491334),
    c64::new(0.09649043135525259, 0.9953339121404823),
    c64::new(0.09572699149930732, 0.9954076266025349),
    c64::new(0.09496349532963906, 0.9954807554919269),
    c64::new(0.09419994329539319, 0.9955532987656385),
    c64::new(0.09343633584574791, 0.9956252563809943),
    c64::new(0.09267267342991335, 0.9956966282956635),
    c64::new(0.0919089564971327, 0.9957674144676598),
    c64::new(0.09114518549668112, 0.9958376148553416),
    c64::new(0.09038136087786501, 0.9959072294174117),
    c64::new(0.08961748309002292, 0.9959762581129178),
    c64::new(0.08885355258252468, 0.996044700901252),
    c64::new(0.0880895698047705, 0.9961125577421511),
    c64::new(0.08732553520619221, 0.9961798285956969),
    c64::new(0.08656144923625124, 0.9962465134223155),
    c64::new(0.08579731234443988, 0.996312612182778),
    c64::new(0.0850331249802804, 0.9963781248382002),
    c64::new(0.08426888759332411, 0.9964430513500426),
    c64::new(0.0835046006331524, 0.9965073916801108),
    c64::new(0.0827402645493758, 0.9965711457905548),
    c64::new(0.0819758797916331, 0.9966343136438699),
    c64::new(0.08121144680959239, 0.9966968952028961),
    c64::new(0.0804469660529501, 0.996758890430818),
    c64::new(0.07968243797143013, 0.9968202992911657),
    c64::new(0.0789178630147851, 0.9968811217478138),
    c64::new(0.0781532416327943, 0.9969413577649822),
    c64::new(0.07738857427526503, 0.9970010073072353),
    c64::new(0.07662386139203162, 0.997060070339483),
    c64::new(0.07585910343295449, 0.99711854682698),
    c64::new(0.07509430084792128, 0.9971764367353262),
    c64::new(0.07432945408684587, 0.9972337400304662),
    c64::new(0.07356456359966745, 0.9972904566786902),
    c64::new(0.07279962983635162, 0.9973465866466332),
    c64::new(0.07203465324688942, 0.9974021299012753),
    c64::new(0.07126963428129642, 0.9974570864099419),
    c64::new(0.07050457338961401, 0.9975114561403035),
    c64::new(0.06973947102190738, 0.9975652390603758),
    c64::new(0.06897432762826673, 0.9976184351385196),
    c64::new(0.06820914365880645, 0.997671044343441),
    c64::new(0.0674439195636641, 0.9977230666441916),
    c64::new(0.06667865579300153, 0.9977745020101678),
    c64::new(0.06591335279700392, 0.9978253504111116),
    c64::new(0.06514801102587886, 0.9978756118171102),
    c64::new(0.06438263092985741, 0.997925286198596),
    c64::new(0.06361721295919319, 0.997974373526347),
    c64::new(0.06285175756416142, 0.9980228737714862),
    c64::new(0.06208626519506025, 0.9980707869054823),
    c64::new(0.061320736302208655, 0.9981181129001492),
    c64::new(0.06055517133594778, 0.9981648517276462),
    c64::new(0.05978957074664001, 0.9982110033604782),
    c64::new(0.059023934984667986, 0.9982565677714952),
    c64::new(0.05825826450043573, 0.9983015449338929),
    c64::new(0.057492559744367684, 0.9983459348212124),
    c64::new(0.05672682116690778, 0.9983897374073402),
    c64::new(0.05596104921852053, 0.9984329526665084),
    c64::new(0.05519524434969003, 0.9984755805732948),
    c64::new(0.05442940701091915, 0.9985176211026222),
    c64::new(0.05366353765273068, 0.9985590742297593),
    c64::new(0.0528976367256654, 0.9985999399303204),
    c64::new(0.05213170468028332, 0.9986402181802653),
    c64::new(0.05136574196716273, 0.9986799089558991),
    c64::new(0.05059974903689934, 0.9987190122338729),
    c64::new(0.04983372634010726, 0.9987575279911833),
    c64::new(0.04906767432741813, 0.9987954562051724),
    c64::new(0.04830159344948018, 0.998832796853528),
    c64::new(0.04753548415695926, 0.9988695499142836),
    c64::new(0.04676934690053796, 0.9989057153658183),
    c64::new(0.046003182130914644, 0.9989412931868569),
    c64::new(0.04523699029880475, 0.9989762833564698),
    c64::new(0.044470771854938744, 0.9990106858540734),
    c64::new(0.04370452725006342, 0.9990445006594293),
    c64::new(0.04293825693494096, 0.9990777277526454),
    c64::new(0.042171961360348, 0.9991103671141749),
    c64::new(0.04140564097707672, 0.9991424187248169),
    c64::new(0.040639296235933854, 0.9991738825657164),
    c64::new(0.039872927587739845, 0.9992047586183639),
    c64::new(0.039106535483329846, 0.9992350468645959),
    c64::new(0.03834012037355279, 0.9992647472865944),
    c64::new(0.037573682709270514, 0.9992938598668878),
    c64::new(0.03680722294135899, 0.9993223845883495),
    c64::new(0.036040741520706306, 0.9993503214341994),
    c64::new(0.03527423889821395, 0.9993776703880028),
    c64::new(0.03450771552479589, 0.9994044314336713),
    c64::new(0.03374117185137764, 0.9994306045554617),
    c64::new(0.032974608328897315, 0.9994561897379773),
    c64::new(0.032208025408304704, 0.999481186966167),
    c64::new(0.03144142354056034, 0.9995055962253253),
    c64::new(0.030674803176636584, 0.9995294175010931),
    c64::new(0.029908164767516655, 0.999552650779457),
    c64::new(0.029141508764193743, 0.9995752960467492),
    c64::new(0.028374835617672258, 0.9995973532896484),
    c64::new(0.02760814577896582, 0.9996188224951786),
    c64::new(0.02684143969909853, 0.9996397036507102),
    c64::new(0.02607471782910404, 0.9996599967439592),
    c64::new(0.02530798062002463, 0.9996797017629879),
    c64::new(0.024541228522912267, 0.9996988186962042),
    c64::new(0.023774461988827676, 0.9997173475323622),
    c64::new(0.02300768146883941, 0.9997352882605617),
    c64::new(0.02224088741402492, 0.9997526408702488),
    c64::new(0.02147408027546961, 0.9997694053512153),
    c64::new(0.020707260504265915, 0.9997855816935992),
    c64::new(0.0199404285515146, 0.9998011698878843),
    c64::new(0.019173584868322702, 0.9998161699249004),
    c64::new(0.01840672990580482, 0.9998305817958234),
    c64::new(0.017639864115082195, 0.9998444054921752),
    c64::new(0.016872987947281773, 0.9998576410058239),
    c64::new(0.016106101853537266, 0.999870288328983),
    c64::new(0.01533920628498822, 0.9998823474542126),
    c64::new(0.014572301692779104, 0.9998938183744185),
    c64::new(0.013805388528060349, 0.9999047010828529),
    c64::new(0.013038467241987433, 0.9999149955731135),
    c64::new(0.012271538285719944, 0.9999247018391445),
    c64::new(0.011504602110422875, 0.999933819875236),
    c64::new(0.01073765916726457, 0.9999423496760239),
    c64::new(0.009970709907418029, 0.9999502912364905),
    c64::new(0.00920375478205996, 0.9999576445519639),
    c64::new(0.00843679424236986, 0.9999644096181183),
    c64::new(0.007669828739531077, 0.9999705864309741),
    c64::new(0.006902858724729877, 0.9999761749868976),
    c64::new(0.006135884649154516, 0.9999811752826011),
    c64::new(0.005368906963996303, 0.9999855873151432),
    c64::new(0.004601926120448672, 0.9999894110819284),
    c64::new(0.003834942569706248, 0.9999926465807072),
    c64::new(0.003067956762966138, 0.9999952938095762),
    c64::new(0.002300969151425886, 0.9999973527669782),
    c64::new(0.001533980186284766, 0.9999988234517019),
    c64::new(0.0007669903187428464, 0.9999997058628822),
    c64::new(6.123233995736766e-17, 1.0),
    c64::new(-0.0007669903187427239, 0.9999997058628822),
    c64::new(-0.0015339801862846436, 0.9999988234517019),
    c64::new(-0.002300969151425764, 0.9999973527669782),
    c64::new(-0.0030679567629660156, 0.9999952938095762),
    c64::new(-0.0038349425697061256, 0.9999926465807072),
    c64::new(-0.004601926120448549, 0.9999894110819284),
    c64::new(-0.0053689069639961794, 0.9999855873151432),
    c64::new(-0.006135884649154393, 0.9999811752826011),
    c64::new(-0.006902858724729754, 0.9999761749868976),
    c64::new(-0.007669828739530954, 0.9999705864309741),
    c64::new(-0.008436794242369738, 0.9999644096181183),
    c64::new(-0.009203754782059839, 0.9999576445519639),
    c64::new(-0.009970709907417908, 0.9999502912364905),
    c64::new(-0.010737659167264449, 0.9999423496760239),
    c64::new(-0.011504602110422753, 0.999933819875236),
    c64::new(-0.012271538285719823, 0.9999247018391445),
    c64::new(-0.013038467241987312, 0.9999149955731135),
    c64::new(-0.013805388528060228, 0.9999047010828529),
    c64::new(-0.014572301692778983, 0.9998938183744185),
    c64::new(-0.015339206284988098, 0.9998823474542126),
    c64::new(-0.01610610185353714, 0.999870288328983),
    c64::new(-0.016872987947281648, 0.9998576410058239),
    c64::new(-0.01763986411508207, 0.9998444054921752),
    c64::new(-0.018406729905804695, 0.9998305817958234),
    c64::new(-0.019173584868322577, 0.9998161699249004),
    c64::new(-0.019940428551514476, 0.9998011698878843),
    c64::new(-0.02070726050426579, 0.9997855816935992),
    c64::new(-0.021474080275469484, 0.9997694053512153),
    c64::new(-0.022240887414024794, 0.9997526408702488),
    c64::new(-0.023007681468839285, 0.9997352882605617),
    c64::new(-0.02377446198882755, 0.9997173475323622),
    c64::new(-0.024541228522912142, 0.9996988186962042),
    c64::new(-0.025307980620024505, 0.9996797017629879),
    c64::new(-0.026074717829103915, 0.9996599967439592),
    c64::new(-0.026841439699098406, 0.9996397036507102),
    c64::new(-0.027608145778965695, 0.9996188224951786),
    c64::new(-0.028374835617672133, 0.9995973532896484),
    c64::new(-0.029141508764193618, 0.9995752960467492),
    c64::new(-0.02990816476751653, 0.999552650779457),
    c64::new(-0.03067480317663646, 0.9995294175010931),
    c64::new(-0.03144142354056022, 0.9995055962253253),
    c64::new(-0.03220802540830458, 0.999481186966167),
    c64::new(-0.03297460832889719, 0.9994561897379773),
    c64::new(-0.03374117185137752, 0.9994306045554617),
    c64::new(-0.034507715524795764, 0.9994044314336713),
    c64::new(-0.03527423889821382, 0.9993776703880028),
    c64::new(-0.03604074152070618, 0.9993503214341994),
    c64::new(-0.036807222941358866, 0.9993223845883495),
    c64::new(-0.03757368270927039, 0.9992938598668878),
    c64::new(-0.038340120373552666, 0.9992647472865944),
    c64::new(-0.03910653548332972, 0.9992350468645959),
    c64::new(-0.03987292758773972, 0.9992047586183639),
    c64::new(-0.04063929623593373, 0.9991738825657164),
    c64::new(-0.041405640977076594, 0.9991424187248169),
    c64::new(-0.04217196136034788, 0.9991103671141749),
    c64::new(-0.042938256934940834, 0.9990777277526454),
    c64::new(-0.043704527250063296, 0.9990445006594293),
    c64::new(-0.04447077185493862, 0.9990106858540734),
    c64::new(-0.045236990298804625, 0.9989762833564698),
    c64::new(-0.04600318213091452, 0.9989412931868569),
    c64::new(-0.046769346900537835, 0.9989057153658183),
    c64::new(-0.047535484156959136, 0.9988695499142836),
    c64::new(-0.048301593449480054, 0.998832796853528),
    c64::new(-0.04906767432741801, 0.9987954562051724),
    c64::new(-0.04983372634010713, 0.9987575279911833),
    c64::new(-0.05059974903689921, 0.9987190122338729),
    c64::new(-0.051365741967162606, 0.9986799089558991),
    c64::new(-0.05213170468028319, 0.9986402181802653),
    c64::new(-0.052897636725665276, 0.9985999399303204),
    c64::new(-0.053663537652730554, 0.9985590742297593),
    c64::new(-0.05442940701091902, 0.9985176211026222),
    c64::new(-0.05519524434968991, 0.9984755805732948),
    c64::new(-0.0559610492185204, 0.9984329526665084),
    c64::new(-0.05672682116690766, 0.9983897374073402),
    c64::new(-0.05749255974436756, 0.9983459348212124),
    c64::new(-0.05825826450043561, 0.9983015449338929),
    c64::new(-0.05902393498466786, 0.9982565677714952),
    c64::new(-0.05978957074663988, 0.9982110033604782),
    c64::new(-0.060555171335947656, 0.9981648517276462),
    c64::new(-0.06132073630220853, 0.9981181129001492),
    c64::new(-0.06208626519506012, 0.9980707869054823),
    c64::new(-0.06285175756416131, 0.9980228737714862),
    c64::new(-0.06361721295919308, 0.997974373526347),
    c64::new(-0.0643826309298573, 0.997925286198596),
    c64::new(-0.06514801102587875, 0.9978756118171102),
    c64::new(-0.0659133527970038, 0.9978253504111116),
    c64::new(-0.06667865579300142, 0.9977745020101678),
    c64::new(-0.067443919563664, 0.9977230666441916),
    c64::new(-0.06820914365880634, 0.997671044343441),
    c64::new(-0.06897432762826662, 0.9976184351385196),
    c64::new(-0.06973947102190726, 0.9975652390603758),
    c64::new(-0.0705045733896139, 0.9975114561403035),
    c64::new(-0.0712696342812963, 0.9974570864099419),
    c64::new(-0.0720346532468893, 0.9974021299012753),
    c64::new(-0.0727996298363515, 0.9973465866466332),
    c64::new(-0.07356456359966734, 0.9972904566786902),
    c64::new(-0.07432945408684576, 0.9972337400304663),
    c64::new(-0.07509430084792117, 0.9971764367353262),
    c64::new(-0.07585910343295438, 0.99711854682698),
    c64::new(-0.0766238613920315, 0.997060070339483),
    c64::new(-0.07738857427526492, 0.9970010073072353),
    c64::new(-0.07815324163279419, 0.9969413577649822),
    c64::new(-0.07891786301478498, 0.9968811217478138),
    c64::new(-0.07968243797143001, 0.9968202992911657),
    c64::new(-0.08044696605294999, 0.996758890430818),
    c64::new(-0.08121144680959227, 0.9966968952028961),
    c64::new(-0.08197587979163298, 0.9966343136438699),
    c64::new(-0.08274026454937569, 0.9965711457905548),
    c64::new(-0.08350460063315229, 0.9965073916801108),
    c64::new(-0.084268887593324, 0.9964430513500426),
    c64::new(-0.08503312498028029, 0.9963781248382002),
    c64::new(-0.08579731234443977, 0.996312612182778),
    c64::new(-0.08656144923625113, 0.9962465134223155),
    c64::new(-0.0873255352061921, 0.9961798285956969),
    c64::new(-0.0880895698047704, 0.9961125577421512),
    c64::new(-0.08885355258252457, 0.996044700901252),
    c64::new(-0.0896174830900228, 0.9959762581129178),
    c64::new(-0.0903813608778649, 0.9959072294174117),
    c64::new(-0.091145185496681, 0.9958376148553416),
    c64::new(-0.09190895649713259, 0.9957674144676598),
    c64::new(-0.09267267342991324, 0.9956966282956635),
    c64::new(-0.0934363358457478, 0.9956252563809943),
    c64::new(-0.09419994329539308, 0.9955532987656386),
    c64::new(-0.09496349532963895, 0.9954807554919269),
    c64::new(-0.0957269914993072, 0.9954076266025349),
    c64::new(-0.09649043135525248, 0.9953339121404823),
    c64::new(-0.09725381444836324, 0.9952596121491334),
    c64::new(-0.09801714032956066, 0.9951847266721969),
    c64::new(-0.09878040854979954, 0.9951092557537261),
    c64::new(-0.09954361866006932, 0.9950331994381186),
    c64::new(-0.10030677021139271, 0.9949565577701164),
    c64::new(-0.10106986275482775, 0.9948793307948056),
    c64::new(-0.10183289584146654, 0.9948015185576171),
    c64::new(-0.10259586902243616, 0.9947231211043257),
    c64::new(-0.10335878184889957, 0.9946441384810507),
    c64::new(-0.10412163387205461, 0.9945645707342554),
    c64::new(-0.10488442464313485, 0.9944844179107476),
    c64::new(-0.10564715371341059, 0.9944036800576791),
    c64::new(-0.10640982063418773, 0.9943223572225458),
    c64::new(-0.10717242495680876, 0.9942404494531879),
    c64::new(-0.10793496623265365, 0.9941579567977897),
    c64::new(-0.10869744401313856, 0.9940748793048795),
    c64::new(-0.10945985784971791, 0.9939912170233294),
    c64::new(-0.11022220729388307, 0.9939069700023561),
    c64::new(-0.11098449189716326, 0.9938221382915197),
    c64::new(-0.11174671121112655, 0.9937367219407246),
    c64::new(-0.11250886478737872, 0.9936507210002191),
    c64::new(-0.11327095217756424, 0.9935641355205953),
    c64::new(-0.11403297293336717, 0.9934769655527892),
    c64::new(-0.11479492660651014, 0.9933892111480807),
    c64::new(-0.11555681274875518, 0.9933008723580933),
    c64::new(-0.11631863091190475, 0.9932119492347945),
    c64::new(-0.11708038064780044, 0.9931224418304956),
    c64::new(-0.11784206150832491, 0.9930323501978514),
    c64::new(-0.11860367304540073, 0.9929416743898605),
    c64::new(-0.11936521481099123, 0.9928504144598651),
    c64::new(-0.12012668635710147, 0.9927585704615511),
    c64::new(-0.12088808723577711, 0.992666142448948),
    c64::new(-0.12164941699910542, 0.9925731304764288),
    c64::new(-0.12241067519921617, 0.99247953459871),
    c64::new(-0.12317186138828054, 0.9923853548708517),
    c64::new(-0.12393297511851208, 0.9922905913482574),
    c64::new(-0.12469401594216764, 0.9921952440866739),
    c64::new(-0.1254549834115461, 0.9920993131421918),
    c64::new(-0.1262158770789903, 0.9920027985712445),
    c64::new(-0.12697669649688587, 0.9919057004306093),
    c64::new(-0.12773744121766217, 0.9918080187774065),
    c64::new(-0.1284981107937931, 0.9917097536690995),
    c64::new(-0.12925870477779616, 0.9916109051634954),
    c64::new(-0.13001922272223324, 0.991511473318744),
    c64::new(-0.13077966417971168, 0.9914114581933385),
    c64::new(-0.13154002870288317, 0.9913108598461154),
    c64::new(-0.13230031584444457, 0.9912096783362541),
    c64::new(-0.13306052515713906, 0.9911079137232769),
    c64::new(-0.13382065619375458, 0.9910055660670494),
    c64::new(-0.1345807085071261, 0.99090263542778),
    c64::new(-0.13534068165013421, 0.9907991218660204),
    c64::new(-0.13610057517570606, 0.9906950254426646),
    c64::new(-0.13686038863681632, 0.9905903462189501),
    c64::new(-0.13762012158648607, 0.990485084256457),
    c64::new(-0.13837977357778378, 0.9903792396171082),
    c64::new(-0.13913934416382617, 0.9902728123631691),
    c64::new(-0.13989883289777727, 0.9901658025572484),
    c64::new(-0.14065823933284913, 0.9900582102622971),
    c64::new(-0.14141756302230302, 0.989950035541609),
    c64::new(-0.1421768035194479, 0.9898412784588205),
    c64::new(-0.14293596037764258, 0.9897319390779106),
    c64::new(-0.14369503315029447, 0.9896220174632009),
    c64::new(-0.14445402139086033, 0.9895115136793553),
    c64::new(-0.1452129246528474, 0.9894004277913804),
    c64::new(-0.14597174248981223, 0.9892887598646252),
    c64::new(-0.14673047445536164, 0.989176509964781),
    c64::new(-0.14748912010315354, 0.9890636781578815),
    c64::new(-0.1482476789868961, 0.988950264510303),
    c64::new(-0.14900615066034836, 0.9888362690887635),
    c64::new(-0.1497645346773215, 0.9887216919603238),
    c64::new(-0.15052283059167726, 0.9886065331923866),
    c64::new(-0.15128103795733014, 0.9884907928526967),
    c64::new(-0.15203915632824605, 0.9883744710093413),
    c64::new(-0.1527971852584433, 0.9882575677307495),
    c64::new(-0.1535551243019934, 0.9881400830856926),
    c64::new(-0.15431297301302013, 0.9880220171432835),
    c64::new(-0.1550707309457004, 0.9879033699729778),
    c64::new(-0.1558283976542652, 0.9877841416445722),
    c64::new(-0.15658597269299848, 0.9876643322282057),
    c64::new(-0.15734345561623816, 0.9875439417943593),
    c64::new(-0.15810084597837698, 0.9874229704138554),
    c64::new(-0.15885814333386128, 0.9873014181578584),
    c64::new(-0.15961534723719298, 0.9871792850978743),
    c64::new(-0.16037245724292828, 0.987056571305751),
    c64::new(-0.16112947290567867, 0.9869332768536778),
    c64::new(-0.16188639378011177, 0.9868094018141855),
    c64::new(-0.16264321942095034, 0.9866849462601467),
    c64::new(-0.16339994938297311, 0.9865599102647755),
    c64::new(-0.16415658322101578, 0.9864342939016272),
    c64::new(-0.16491312048996995, 0.9863080972445987),
    c64::new(-0.16566956074478403, 0.9861813203679283),
    c64::new(-0.1664259035404641, 0.9860539633461954),
    c64::new(-0.16718214843207277, 0.9859260262543211),
    c64::new(-0.1679382949747311, 0.9857975091675675),
    c64::new(-0.16869434272361733, 0.9856684121615376),
    c64::new(-0.16945029123396782, 0.9855387353121761),
    c64::new(-0.170206140061078, 0.9854084786957684),
    c64::new(-0.17096188876030124, 0.9852776423889412),
    c64::new(-0.17171753688704985, 0.9851462264686622),
    c64::new(-0.17247308399679592, 0.9850142310122398),
    c64::new(-0.17322852964507038, 0.9848816560973237),
    c64::new(-0.17398387338746374, 0.9847485018019042),
    c64::new(-0.1747391147796272, 0.9846147682043126),
    c64::new(-0.17549425337727126, 0.9844804553832209),
    c64::new(-0.17624928873616783, 0.9843455634176419),
    c64::new(-0.17700422041214875, 0.984210092386929),
    c64::new(-0.17775904796110703, 0.9840740423707766),
    c64::new(-0.17851377093899745, 0.9839374134492189),
    c64::new(-0.17926838890183577, 0.9838002057026316),
    c64::new(-0.1800229014056994, 0.9836624192117303),
    c64::new(-0.18077730800672856, 0.9835240540575713),
    c64::new(-0.18153160826112502, 0.9833851103215512),
    c64::new(-0.1822858017251532, 0.9832455880854072),
    c64::new(-0.18303988795514095, 0.9831054874312163),
    c64::new(-0.18379386650747828, 0.9829648084413964),
    c64::new(-0.18454773693861953, 0.9828235511987053),
    c64::new(-0.1853014988050819, 0.9826817157862409),
    c64::new(-0.18605515166344652, 0.9825393022874412),
    c64::new(-0.18680869507035922, 0.9823963107860847),
    c64::new(-0.18756212858252963, 0.9822527413662894),
    c64::new(-0.188315451756732, 0.9821085941125136),
    c64::new(-0.18906866414980616, 0.9819638691095552),
    c64::new(-0.18982176531865647, 0.9818185664425525),
    c64::new(-0.19057475482025266, 0.9816726861969831),
    c64::new(-0.19132763221163088, 0.9815262284586648),
    c64::new(-0.19208039704989227, 0.9813791933137546),
    c64::new(-0.19283304889220518, 0.9812315808487497),
    c64::new(-0.1935855872958036, 0.9810833911504867),
    c64::new(-0.1943380118179885, 0.9809346243061418),
    c64::new(-0.1950903220161282, 0.9807852804032304),
    c64::new(-0.19584251744765788, 0.9806353595296081),
    c64::new(-0.1965945976700801, 0.9804848617734694),
    c64::new(-0.1973465622409659, 0.980333787223348),
    c64::new(-0.19809841071795362, 0.9801821359681174),
    c64::new(-0.19885014265875, 0.9800299080969901),
    c64::new(-0.19960175762113094, 0.9798771036995176),
    c64::new(-0.20035325516294028, 0.9797237228655912),
    c64::new(-0.20110463484209182, 0.9795697656854405),
    c64::new(-0.20185589621656805, 0.9794152322496348),
    c64::new(-0.202607038844421, 0.9792601226490821),
    c64::new(-0.20335806228377326, 0.9791044369750292),
    c64::new(-0.2041089660928169, 0.9789481753190622),
    c64::new(-0.2048597498298143, 0.9787913377731057),
    c64::new(-0.2056104130530992, 0.9786339244294232),
    c64::new(-0.20636095532107557, 0.9784759353806168),
    c64::new(-0.20711137619221845, 0.9783173707196277),
    c64::new(-0.20786167522507504, 0.978158230539735),
    c64::new(-0.20861185197826332, 0.9779985149345571),
    c64::new(-0.20936190601047408, 0.9778382239980504),
    c64::new(-0.2101118368804696, 0.9776773578245099),
    c64::new(-0.21086164414708472, 0.9775159165085693),
    c64::new(-0.2116113273692275, 0.9773539001452001),
    c64::new(-0.21236088610587844, 0.9771913088297123),
    c64::new(-0.21311031991609125, 0.9770281426577544),
    c64::new(-0.21385962835899372, 0.9768644017253126),
    c64::new(-0.2146088109937868, 0.9767000861287118),
    c64::new(-0.21535786737974544, 0.9765351959646145),
    c64::new(-0.2161067970762195, 0.9763697313300211),
    c64::new(-0.21685559964263246, 0.9762036923222706),
    c64::new(-0.21760427463848356, 0.9760370790390391),
    c64::new(-0.21835282162334632, 0.975869891578341),
    c64::new(-0.21910124015686966, 0.9757021300385286),
    c64::new(-0.21984952979877864, 0.9755337945182914),
    c64::new(-0.22059769010887353, 0.975364885116657),
    c64::new(-0.2213457206470307, 0.9751954019329904),
    c64::new(-0.22209362097320348, 0.9750253450669941),
    c64::new(-0.22284139064742117, 0.9748547146187084),
    c64::new(-0.2235890292297899, 0.9746835106885107),
    c64::new(-0.22433653628049358, 0.9745117333771158),
    c64::new(-0.22508391135979267, 0.9743393827855759),
    c64::new(-0.2258311540280261, 0.9741664590152804),
    c64::new(-0.22657826384561, 0.9739929621679558),
    c64::new(-0.22732524037303872, 0.9738188923456662),
    c64::new(-0.22807208317088568, 0.973644249650812),
    c64::new(-0.22881879179980225, 0.9734690341861311),
    c64::new(-0.22956536582051876, 0.9732932460546982),
    c64::new(-0.23031180479384541, 0.9731168853599251),
    c64::new(-0.23105810828067117, 0.9729399522055602),
    c64::new(-0.23180427584196467, 0.9727624466956886),
    c64::new(-0.23255030703877522, 0.9725843689347322),
    c64::new(-0.23329620143223145, 0.9724057190274498),
    c64::new(-0.23404195858354335, 0.9722264970789363),
    c64::new(-0.23478757805400097, 0.9720467031946235),
    c64::new(-0.23553305940497535, 0.9718663374802794),
    c64::new(-0.2362784021979195, 0.9716854000420085),
    c64::new(-0.23702360599436723, 0.9715038909862518),
    c64::new(-0.2377686703559341, 0.9713218104197863),
    c64::new(-0.2385135948443184, 0.9711391584497251),
    c64::new(-0.2392583790213, 0.970955935183518),
    c64::new(-0.2400030224487414, 0.9707721407289504),
    c64::new(-0.2407475246885884, 0.9705877751941436),
    c64::new(-0.2414918853028692, 0.9704028386875555),
    c64::new(-0.24223610385369593, 0.9702173313179792),
    c64::new(-0.24298017990326387, 0.970031253194544),
    c64::new(-0.24372411301385202, 0.9698446044267149),
    c64::new(-0.2444679027478241, 0.9696573851242924),
    c64::new(-0.24521154866762757, 0.9694695953974131),
    c64::new(-0.24595505033579448, 0.9692812353565485),
    c64::new(-0.2466984073149424, 0.9690923051125062),
    c64::new(-0.24744161916777332, 0.9689028047764289),
    c64::new(-0.24818468545707467, 0.9687127344597948),
    c64::new(-0.24892760574572015, 0.9685220942744173),
    c64::new(-0.2496703795966684, 0.9683308843324453),
    c64::new(-0.25041300657296517, 0.9681391047463624),
    c64::new(-0.2511554862377419, 0.9679467556289878),
    c64::new(-0.2518978181542168, 0.9677538370934755),
    c64::new(-0.25264000188569546, 0.9675603492533145),
    c64::new(-0.25338203699557016, 0.9673662922223285),
    c64::new(-0.2541239230473205, 0.9671716661146766),
    c64::new(-0.2548656596045145, 0.9669764710448521),
    c64::new(-0.25560724623080744, 0.9667807071276833),
    c64::new(-0.2563486824899428, 0.9665843744783331),
    c64::new(-0.2570899679457531, 0.9663874732122989),
    c64::new(-0.2578311021621588, 0.9661900034454126),
    c64::new(-0.2585720847031703, 0.9659919652938406),
    c64::new(-0.25931291513288623, 0.9657933588740837),
    c64::new(-0.260053593015495, 0.9655941843029768),
    c64::new(-0.26079411791527546, 0.9653944416976894),
    c64::new(-0.2615344893965955, 0.9651941311757247),
    c64::new(-0.2622747070239135, 0.9649932528549204),
    c64::new(-0.26301477036177895, 0.9647918068534479),
    c64::new(-0.2637546789748314, 0.9645897932898128),
    c64::new(-0.2644944324278015, 0.9643872122828543),
    c64::new(-0.2652340302855118, 0.9641840639517458),
    c64::new(-0.2659734721128754, 0.9639803484159941),
    c64::new(-0.2667127574748983, 0.9637760657954398),
    c64::new(-0.2674518859366776, 0.9635712162102573),
    c64::new(-0.26819085706340307, 0.963365799780954),
    c64::new(-0.2689296704203572, 0.9631598166283715),
    c64::new(-0.2696683255729151, 0.9629532668736839),
    c64::new(-0.2704068220865447, 0.9627461506383994),
    c64::new(-0.27114515952680796, 0.9625384680443592),
    c64::new(-0.2718833374593598, 0.9623302192137374),
    c64::new(-0.27262135544994887, 0.9621214042690416),
    c64::new(-0.2733592130644187, 0.9619120233331122),
    c64::new(-0.2740969098687062, 0.9617020765291225),
    c64::new(-0.27483444542884383, 0.961491563980579),
    c64::new(-0.27557181931095814, 0.9612804858113206),
    c64::new(-0.2763090310812709, 0.9610688421455194),
    c64::new(-0.27704608030609984, 0.9608566331076797),
    c64::new(-0.2777829665518577, 0.9606438588226386),
    c64::new(-0.27851968938505295, 0.9604305194155659),
    c64::new(-0.2792562483722911, 0.9602166150119634),
    c64::new(-0.2799926430802733, 0.9600021457376658),
    c64::new(-0.2807288730757971, 0.95978711171884),
    c64::new(-0.28146493792575794, 0.9595715130819845),
    c64::new(-0.2822008371971474, 0.9593553499539308),
    c64::new(-0.2829365704570553, 0.959138622461842),
    c64::new(-0.28367213727266843, 0.9589213307332132),
    c64::new(-0.2844075372112717, 0.9587034748958716),
    c64::new(-0.2851427698402486, 0.9584850550779762),
    c64::new(-0.2858778347270806, 0.9582660714080177),
    c64::new(-0.2866127314393477, 0.9580465240148186),
    c64::new(-0.28734745954472946, 0.9578264130275329),
    c64::new(-0.2880820186110042, 0.9576057385756463),
    c64::new(-0.28881640820604937, 0.957384500788976),
    c64::new(-0.289550627897843, 0.9571626997976701),
    c64::new(-0.2902846772544622, 0.9569403357322089),
    c64::new(-0.291018555844085, 0.9567174087234032),
    c64::new(-0.29175226323498926, 0.9564939189023951),
    c64::new(-0.2924857989955537, 0.9562698664006581),
    c64::new(-0.2932191626942586, 0.9560452513499965),
    c64::new(-0.29395235389968466, 0.9558200738825454),
    c64::new(-0.2946853721805142, 0.9555943341307711),
    c64::new(-0.29541821710553196, 0.9553680322274704),
    c64::new(-0.29615088824362384, 0.9551411683057707),
    c64::new(-0.29688338516377816, 0.9549137424991305),
    c64::new(-0.2976157074350862, 0.9546857549413383),
    c64::new(-0.29834785462674146, 0.9544572057665135),
    c64::new(-0.29907982630804036, 0.9542280951091057),
    c64::new(-0.29981162204838335, 0.9539984231038945),
    c64::new(-0.3005432414172733, 0.9537681898859903),
    c64::new(-0.3012746839843179, 0.9535373955908333),
    c64::new(-0.3020059493192281, 0.9533060403541939),
    c64::new(-0.30273703699181903, 0.9530741243121723),
    c64::new(-0.30346794657201126, 0.9528416476011987),
    c64::new(-0.30419867762982916, 0.9526086103580332),
    c64::new(-0.3049292297354023, 0.9523750127197659),
    c64::new(-0.3056596024589661, 0.9521408548238158),
    c64::new(-0.306389795370861, 0.9519061368079323),
    c64::new(-0.307119808041533, 0.9516708588101939),
    c64::new(-0.30784964004153487, 0.9514350209690083),
    c64::new(-0.3085792909415249, 0.9511986234231133),
    c64::new(-0.30930876031226867, 0.9509616663115751),
    c64::new(-0.3100380477246379, 0.9507241497737896),
    c64::new(-0.31076715274961136, 0.9504860739494818),
    c64::new(-0.31149607495827586, 0.9502474389787052),
    c64::new(-0.31222481392182494, 0.950008245001843),
    c64::new(-0.3129533692115601, 0.9497684921596067),
    c64::new(-0.31368174039889146, 0.9495281805930367),
    c64::new(-0.3144099270553367, 0.949287310443502),
    c64::new(-0.31513792875252233, 0.9490458818527006),
    c64::new(-0.31586574506218396, 0.9488038949626585),
    c64::new(-0.31659337555616573, 0.9485613499157304),
    c64::new(-0.3173208198064217, 0.9483182468545992),
    c64::new(-0.31804807738501495, 0.9480745859222762),
    c64::new(-0.31877514786411837, 0.9478303672621011),
    c64::new(-0.31950203081601564, 0.9475855910177412),
    c64::new(-0.3202287258130999, 0.947340257333192),
    c64::new(-0.3209552324278751, 0.9470943663527772),
    c64::new(-0.3216815502329565, 0.946847918221148),
    c64::new(-0.3224076788010699, 0.9466009130832835),
    c64::new(-0.3231336177050522, 0.9463533510844906),
    c64::new(-0.32385936651785285, 0.9461052323704034),
    c64::new(-0.32458492481253204, 0.9458565570869839),
    c64::new(-0.32531029216226287, 0.9456073253805214),
    c64::new(-0.32603546814033024, 0.9453575373976323),
    c64::new(-0.3267604523201317, 0.9451071932852606),
    c64::new(-0.32748524427517794, 0.9448562931906772),
    c64::new(-0.32820984357909255, 0.9446048372614803),
    c64::new(-0.3289342498056121, 0.9443528256455949),
    c64::new(-0.32965846252858744, 0.9441002584912727),
    c64::new(-0.33038248132198283, 0.9438471359470927),
    c64::new(-0.3311063057598763, 0.9435934581619604),
    c64::new(-0.3318299354164611, 0.9433392252851077),
    c64::new(-0.3325533698660441, 0.9430844374660935),
    c64::new(-0.33327660868304787, 0.9428290948548027),
    c64::new(-0.3339996514420094, 0.9425731976014469),
    c64::new(-0.3347224977175811, 0.9423167458565638),
    c64::new(-0.33544514708453155, 0.9420597397710174),
    c64::new(-0.3361675991177446, 0.9418021794959976),
    c64::new(-0.33688985339221994, 0.9415440651830208),
    c64::new(-0.33761190948307457, 0.9412853969839288),
    c64::new(-0.3383337669655412, 0.9410261750508893),
    c64::new(-0.3390554254149695, 0.9407663995363961),
    c64::new(-0.33977688440682685, 0.9405060705932683),
    c64::new(-0.340498143516697, 0.9402451883746509),
    c64::new(-0.3412192023202823, 0.939983753034014),
    c64::new(-0.3419400603934022, 0.9397217647251533),
    c64::new(-0.34266071731199427, 0.9394592236021899),
    c64::new(-0.343381172652115, 0.9391961298195699),
    c64::new(-0.34410142598993887, 0.9389324835320645),
    c64::new(-0.3448214769017592, 0.9386682848947703),
    c64::new(-0.34554132496398904, 0.9384035340631082),
    c64::new(-0.34626096975316006, 0.9381382311928244),
    c64::new(-0.34698041084592357, 0.9378723764399899),
    c64::new(-0.3476996478190514, 0.937605969961),
    c64::new(-0.3484186802494344, 0.937339011912575),
    c64::new(-0.3491375077140849, 0.9370715024517592),
    c64::new(-0.3498561297901349, 0.9368034417359216),
    c64::new(-0.35057454605483745, 0.9365348299227556),
    c64::new(-0.35129275608556704, 0.9362656671702783),
    c64::new(-0.35201075945981913, 0.9359959536368314),
    c64::new(-0.3527285557552106, 0.9357256894810804),
    c64::new(-0.35344614454948076, 0.9354548748620146),
    c64::new(-0.3541635254204904, 0.9351835099389476),
    c64::new(-0.3548806979462227, 0.9349115948715161),
    c64::new(-0.35559766170478385, 0.9346391298196808),
    c64::new(-0.35631441627440225, 0.9343661149437259),
    c64::new(-0.3570309612334299, 0.934092550404259),
    c64::new(-0.3577472961603419, 0.933818436362211),
    c64::new(-0.35846342063373643, 0.9335437729788363),
    c64::new(-0.35917933423233644, 0.933268560415712),
    c64::new(-0.35989503653498817, 0.9329927988347388),
    c64::new(-0.36061052712066216, 0.9327164883981403),
    c64::new(-0.36132580556845423, 0.9324396292684625),
    c64::new(-0.36204087145758423, 0.9321622216085744),
    c64::new(-0.3627557243673971, 0.9318842655816681),
    c64::new(-0.36347036387736376, 0.9316057613512578),
    c64::new(-0.3641847895670797, 0.9313267090811805),
    c64::new(-0.36489900101626727, 0.9310471089355953),
    c64::new(-0.36561299780477385, 0.9307669610789837),
    c64::new(-0.3663267795125735, 0.9304862656761498),
    c64::new(-0.3670403457197671, 0.9302050228922191),
    c64::new(-0.367753696006582, 0.9299232328926397),
    c64::new(-0.3684668299533722, 0.9296408958431813),
    c64::new(-0.36917974714061996, 0.9293580119099356),
    c64::new(-0.36989244714893416, 0.9290745812593157),
    c64::new(-0.37060492955905155, 0.928790604058057),
    c64::new(-0.37131719395183754, 0.9285060804732156),
    c64::new(-0.37202923990828485, 0.9282210106721696),
    c64::new(-0.3727410670095157, 0.9279353948226179),
    c64::new(-0.3734526748367803, 0.9276492330925812),
    c64::new(-0.3741640629714579, 0.9273625256504011),
    c64::new(-0.37487523099505754, 0.9270752726647401),
    c64::new(-0.3755861784892172, 0.9267874743045817),
    c64::new(-0.3762969050357047, 0.9264991307392305),
    c64::new(-0.3770074102164182, 0.9262102421383114),
    c64::new(-0.3777176936133857, 0.92592080867177),
    c64::new(-0.3784277548087655, 0.9256308305098728),
    c64::new(-0.3791375933848473, 0.9253403078232063),
    c64::new(-0.379847208924051, 0.9250492407826777),
    c64::new(-0.38055660100892846, 0.9247576295595139),
    c64::new(-0.3812657692221624, 0.9244654743252626),
    c64::new(-0.3819747131465671, 0.9241727752517912),
    c64::new(-0.3826834323650897, 0.9238795325112867),
    c64::new(-0.38339192646080866, 0.9235857462762567),
    c64::new(-0.38410019501693493, 0.9232914167195276),
    c64::new(-0.3848082376168128, 0.9229965440142462),
    c64::new(-0.3855160538439189, 0.9227011283338785),
    c64::new(-0.38622364328186287, 0.92240516985221),
    c64::new(-0.3869310055143886, 0.9221086687433451),
    c64::new(-0.38763814012537257, 0.9218116251817081),
    c64::new(-0.3883450466988262, 0.921514039342042),
    c64::new(-0.3890517248188944, 0.9212159113994087),
    c64::new(-0.3897581740698563, 0.9209172415291895),
    c64::new(-0.39046439403612654, 0.920618029907084),
    c64::new(-0.39117038430225387, 0.9203182767091106),
    c64::new(-0.39187614445292224, 0.9200179821116066),
    c64::new(-0.39258167407295147, 0.9197171462912274),
    c64::new(-0.39328697274729646, 0.919415769424947),
    c64::new(-0.393992040061048, 0.9191138516900578),
    c64::new(-0.39469687559943356, 0.9188113932641699),
    c64::new(-0.3954014789478162, 0.9185083943252123),
    c64::new(-0.3961058496916962, 0.918204855051431),
    c64::new(-0.3968099874167103, 0.9179007756213905),
    c64::new(-0.3975138917086322, 0.917596156213973),
    c64::new(-0.39821756215337356, 0.917290997008378),
    c64::new(-0.3989209983369829, 0.916985298184123),
    c64::new(-0.39962419984564673, 0.9166790599210427),
    c64::new(-0.40032716626569004, 0.9163722823992891),
    c64::new(-0.4010298971835757, 0.9160649657993317),
    c64::new(-0.4017323921859049, 0.9157571103019568),
    c64::new(-0.40243465085941843, 0.9154487160882678),
    c64::new(-0.40313667279099513, 0.9151397833396853),
    c64::new(-0.403838457567654, 0.9148303122379462),
    c64::new(-0.404540004776553, 0.9145203029651044),
    c64::new(-0.40524131400498975, 0.9142097557035307),
    c64::new(-0.40594238484040246, 0.9138986706359117),
    c64::new(-0.40664321687036903, 0.9135870479452508),
    c64::new(-0.40734380968260786, 0.9132748878148678),
    c64::new(-0.40804416286497863, 0.9129621904283982),
    c64::new(-0.4087442760054814, 0.9126489559697939),
    c64::new(-0.40944414869225754, 0.9123351846233229),
    c64::new(-0.41014378051359024, 0.9120208765735683),
    c64::new(-0.4108431710579038, 0.9117060320054299),
    c64::new(-0.41154231991376516, 0.9113906511041224),
    c64::new(-0.4122412266698829, 0.9110747340551764),
    c64::new(-0.4129398909151079, 0.9107582810444376),
    c64::new(-0.4136383122384345, 0.9104412922580672),
    c64::new(-0.4143364902289991, 0.9101237678825417),
    c64::new(-0.4150344244760815, 0.9098057081046523),
    c64::new(-0.4157321145691053, 0.9094871131115054),
    c64::new(-0.416429560097637, 0.9091679830905225),
    c64::new(-0.41712676065138776, 0.9088483182294391),
    c64::new(-0.41782371582021227, 0.9085281187163061),
    c64::new(-0.41852042519410976, 0.9082073847394887),
    c64::new(-0.41921688836322407, 0.9078861164876662),
    c64::new(-0.41991310491784345, 0.9075643141498327),
    c64::new(-0.4206090744484024, 0.9072419779152959),
    c64::new(-0.4213047965454796, 0.9069191079736781),
    c64::new(-0.4220002707997997, 0.9065957045149153),
    c64::new(-0.42269549680223306, 0.9062717677292575),
    c64::new(-0.42339047414379577, 0.9059472978072686),
    c64::new(-0.4240852024156514, 0.9056222949398254),
    c64::new(-0.42477968120910875, 0.9052967593181188),
    c64::new(-0.4254739101156238, 0.9049706911336532),
    c64::new(-0.42616788872679967, 0.9046440905782462),
    c64::new(-0.4268616166343866, 0.9043169578440282),
    c64::new(-0.42755509343028186, 0.9039892931234434),
    c64::new(-0.4282483187065318, 0.9036610966092481),
    c64::new(-0.42894129205532944, 0.9033323684945118),
    c64::new(-0.4296340130690164, 0.9030031089726172),
    c64::new(-0.4303264813400827, 0.9026733182372588),
    c64::new(-0.4310186964611668, 0.9023429964824443),
    c64::new(-0.4317106580250571, 0.9020121439024933),
    c64::new(-0.43240236562469003, 0.9016807606920377),
    c64::new(-0.43309381885315196, 0.901348847046022),
    c64::new(-0.4337850173036786, 0.9010164031597023),
    c64::new(-0.4344759605696558, 0.9006834292286469),
    c64::new(-0.43516664824461904, 0.9003499254487357),
    c64::new(-0.43585707992225536, 0.9000158920161603),
    c64::new(-0.43654725519640114, 0.8996813291274239),
    c64::new(-0.4372371736610441, 0.8993462369793416),
    c64::new(-0.43792683491032297, 0.8990106157690391),
    c64::new(-0.4386162385385274, 0.8986744656939539),
    c64::new(-0.4393053841400998, 0.8983377869518344),
    c64::new(-0.43999427130963314, 0.8980005797407399),
    c64::new(-0.4406828996418729, 0.8976628442590409),
    c64::new(-0.4413712687317167, 0.8973245807054182),
    c64::new(-0.44205937817421487, 0.896985789278864),
    c64::new(-0.4427472275645698, 0.8966464701786803),
    c64::new(-0.4434348164981383, 0.8963066236044797),
    c64::new(-0.44412214457042914, 0.8959662497561852),
    c64::new(-0.4448092113771049, 0.89562534883403),
    c64::new(-0.44549601651398185, 0.8952839210385575),
    c64::new(-0.4461825595770298, 0.8949419665706209),
    c64::new(-0.446868840162374, 0.8945994856313828),
    c64::new(-0.4475548578662929, 0.8942564784223161),
    c64::new(-0.4482406122852199, 0.8939129451452033),
    c64::new(-0.4489261030157433, 0.8935688860021359),
    c64::new(-0.4496113296546067, 0.8932243011955152),
    c64::new(-0.45029629179870845, 0.8928791909280518),
    c64::new(-0.45098098904510375, 0.8925335554027647),
    c64::new(-0.45166542099100243, 0.8921873948229826),
    c64::new(-0.4523495872337709, 0.8918407093923427),
    c64::new(-0.4530334873709317, 0.8914934993147914),
    c64::new(-0.45371712100016365, 0.8911457647945834),
    c64::new(-0.45440048771930347, 0.8907975060362816),
    c64::new(-0.4550835871263437, 0.890448723244758),
    c64::new(-0.45576641881943464, 0.8900994166251923),
    c64::new(-0.456448982396884, 0.8897495863830728),
    c64::new(-0.4571312774571571, 0.8893992327241955),
    c64::new(-0.457813303598877, 0.8890483558546647),
    c64::new(-0.4584950604208261, 0.8886969559808917),
    c64::new(-0.45917654752194403, 0.8883450333095964),
    c64::new(-0.45985776450132954, 0.8879925880478056),
    c64::new(-0.4605387109582401, 0.8876396204028539),
    c64::new(-0.46121938649209215, 0.8872861305823833),
    c64::new(-0.46189979070246256, 0.8869321187943423),
    c64::new(-0.4625799231890867, 0.886577585246987),
    c64::new(-0.46325978355186015, 0.8862225301488806),
    c64::new(-0.4639393713908386, 0.8858669537088928),
    c64::new(-0.4646186863062379, 0.8855108561361998),
    c64::new(-0.4652977278984344, 0.8851542376402852),
    c64::new(-0.465976495767966, 0.8847970984309379),
    c64::new(-0.46665498951553086, 0.8844394387182538),
    c64::new(-0.46733320874198847, 0.884081258712635),
    c64::new(-0.4680111530483599, 0.8837225586247895),
    c64::new(-0.4686888220358277, 0.8833633386657317),
    c64::new(-0.46936621530573736, 0.8830035990467809),
    c64::new(-0.4700433324595955, 0.8826433399795628),
    c64::new(-0.4707201730990716, 0.8822825616760087),
    c64::new(-0.4713967368259977, 0.881921264348355),
    c64::new(-0.47207302324236877, 0.8815594482091438),
    c64::new(-0.47274903195034257, 0.8811971134712222),
    c64::new(-0.4734247625522414, 0.880834260347742),
    c64::new(-0.47410021465054997, 0.8804708890521609),
    c64::new(-0.4747753878479171, 0.8801069997982404),
    c64::new(-0.475450281747156, 0.8797425928000474),
    c64::new(-0.47612489595124335, 0.8793776682719534),
    c64::new(-0.476799230063322, 0.8790122264286335),
    c64::new(-0.47747328368669795, 0.8786462674850682),
    c64::new(-0.478147056424843, 0.8782797916565416),
    c64::new(-0.47882054788139394, 0.8779127991586417),
    c64::new(-0.47949375766015306, 0.8775452902072612),
    c64::new(-0.4801666853650881, 0.8771772650185962),
    c64::new(-0.4808393306003338, 0.8768087238091458),
    c64::new(-0.48151169297018975, 0.8764396667957137),
    c64::new(-0.4821837720791227, 0.8760700941954066),
    c64::new(-0.4828555675317657, 0.8757000062256346),
    c64::new(-0.4835270789329184, 0.875329403104111),
    c64::new(-0.4841983058875488, 0.8749582850488518),
    c64::new(-0.48486924800079095, 0.8745866522781762),
    c64::new(-0.48553990487794685, 0.8742145050107063),
    c64::new(-0.4862102761244864, 0.8738418434653669),
    c64::new(-0.48688036134604745, 0.8734686678613849),
    c64::new(-0.4875501601484357, 0.8730949784182902),
    c64::new(-0.48821967213762657, 0.8727207753559144),
    c64::new(-0.48888889691976306, 0.8723460588943915),
    c64::new(-0.48955783410115744, 0.8719708292541578),
    c64::new(-0.49022648328829116, 0.871595086655951),
    c64::new(-0.4908948440878148, 0.8712188313208111),
    c64::new(-0.49156291610654973, 0.8708420634700791),
    c64::new(-0.4922306989514859, 0.8704647833253978),
    c64::new(-0.4928981922297839, 0.8700869911087115),
    c64::new(-0.49356539554877477, 0.8697086870422657),
    c64::new(-0.4942323085159598, 0.8693298713486067),
    c64::new(-0.4948989307390109, 0.8689505442505825),
    c64::new(-0.4955652618257723, 0.868570705971341),
    c64::new(-0.49623130138425814, 0.8681903567343314),
    c64::new(-0.49689704902265447, 0.8678094967633033),
    c64::new(-0.49756250434931915, 0.8674281262823069),
    c64::new(-0.49822766697278154, 0.8670462455156929),
    c64::new(-0.4988925365017444, 0.8666638546881112),
    c64::new(-0.4995571125450817, 0.8662809540245131),
    c64::new(-0.5002213947118406, 0.8658975437501489),
    c64::new(-0.5008853826112407, 0.8655136240905691),
    c64::new(-0.5015490758526754, 0.8651291952716237),
    c64::new(-0.5022124740457106, 0.8647442575194625),
    c64::new(-0.5028755768000868, 0.864358811060534),
    c64::new(-0.5035383837257175, 0.8639728561215868),
    c64::new(-0.5042008944326903, 0.8635863929296681),
    c64::new(-0.5048631085312676, 0.8631994217121242),
    c64::new(-0.5055250256318852, 0.8628119426966006),
    c64::new(-0.5061866453451551, 0.8624239561110407),
    c64::new(-0.5068479672818632, 0.8620354621836873),
    c64::new(-0.5075089910529708, 0.8616464611430814),
    c64::new(-0.5081697162696146, 0.8612569532180622),
    c64::new(-0.5088301425431071, 0.8608669386377672),
    c64::new(-0.509490269484936, 0.8604764176316323),
    c64::new(-0.5101500967067666, 0.8600853904293903),
    c64::new(-0.5108096238204389, 0.8596938572610727),
    c64::new(-0.5114688504379703, 0.8593018183570085),
    c64::new(-0.5121277761715547, 0.8589092739478239),
    c64::new(-0.5127864006335627, 0.858516224264443),
    c64::new(-0.5134447234365432, 0.8581226695380862),
    c64::new(-0.5141027441932216, 0.8577286100002721),
    c64::new(-0.5147604625165011, 0.8573340458828157),
    c64::new(-0.5154178780194629, 0.8569389774178288),
    c64::new(-0.5160749903153667, 0.85654340483772),
    c64::new(-0.5167317990176497, 0.8561473283751946),
    c64::new(-0.5173883037399288, 0.855750748263254),
    c64::new(-0.5180445040959992, 0.855353664735196),
    c64::new(-0.518700399699835, 0.8549560780246149),
    c64::new(-0.5193559901655896, 0.8545579883654005),
    c64::new(-0.5200112751075957, 0.854159395991739),
    c64::new(-0.5206662541403669, 0.8537603011381115),
    c64::new(-0.5213209268785954, 0.8533607040392955),
    c64::new(-0.5219752929371543, 0.8529606049303637),
    c64::new(-0.5226293519310966, 0.8525600040466841),
    c64::new(-0.5232831034756564, 0.8521589016239198),
    c64::new(-0.5239365471862483, 0.8517572978980292),
    c64::new(-0.5245896826784687, 0.8513551931052652),
    c64::new(-0.5252425095680946, 0.8509525874821758),
    c64::new(-0.5258950274710846, 0.8505494812656035),
    c64::new(-0.5265472360035794, 0.8501458746926852),
    c64::new(-0.5271991347819011, 0.8497417680008527),
    c64::new(-0.5278507234225551, 0.8493371614278309),
    c64::new(-0.5285020015422283, 0.8489320552116397),
    c64::new(-0.5291529687577906, 0.8485264495905928),
    c64::new(-0.5298036246862946, 0.8481203448032972),
    c64::new(-0.5304539689449764, 0.8477137410886543),
    c64::new(-0.5311040011512547, 0.8473066386858585),
    c64::new(-0.5317537209227331, 0.8468990378343974),
    c64::new(-0.5324031278771978, 0.8464909387740521),
    c64::new(-0.5330522216326194, 0.846082341744897),
    c64::new(-0.533701001807153, 0.8456732469872991),
    c64::new(-0.5343494680191372, 0.8452636547419184),
    c64::new(-0.5349976198870969, 0.8448535652497072),
    c64::new(-0.5356454570297409, 0.8444429787519108),
    c64::new(-0.5362929790659631, 0.8440318954900665),
    c64::new(-0.5369401856148429, 0.8436203157060042),
    c64::new(-0.5375870762956455, 0.8432082396418454),
    c64::new(-0.5382336507278214, 0.8427956675400043),
    c64::new(-0.5388799085310082, 0.842382599643186),
    c64::new(-0.5395258493250288, 0.8419690361943878),
    c64::new(-0.5401714727298929, 0.8415549774368984),
    c64::new(-0.5408167783657967, 0.8411404236142981),
    c64::new(-0.5414617658531232, 0.8407253749704582),
    c64::new(-0.5421064348124437, 0.8403098317495409),
    c64::new(-0.5427507848645158, 0.8398937941959996),
    c64::new(-0.5433948156302847, 0.8394772625545786),
    c64::new(-0.5440385267308838, 0.8390602370703127),
    c64::new(-0.5446819177876346, 0.8386427179885273),
    c64::new(-0.5453249884220461, 0.8382247055548382),
    c64::new(-0.5459677382558173, 0.837806200015151),
    c64::new(-0.5466101669108347, 0.837387201615662),
    c64::new(-0.5472522740091741, 0.836967710602857),
    c64::new(-0.5478940591731002, 0.836547727223512),
    c64::new(-0.5485355220250671, 0.8361272517246924),
    c64::new(-0.5491766621877195, 0.8357062843537527),
    c64::new(-0.5498174792838908, 0.8352848253583375),
    c64::new(-0.5504579729366047, 0.8348628749863801),
    c64::new(-0.5510981427690754, 0.8344404334861032),
    c64::new(-0.5517379884047074, 0.834017501106018),
    c64::new(-0.5523775094670957, 0.8335940780949254),
    c64::new(-0.5530167055800274, 0.8331701647019133),
    c64::new(-0.5536555763674792, 0.8327457611763596),
    c64::new(-0.55429412145362, 0.8323208677679297),
    c64::new(-0.5549323404628104, 0.8318954847265776),
    c64::new(-0.555570233019602, 0.8314696123025455),
    c64::new(-0.5562077987487397, 0.8310432507463624),
    c64::new(-0.5568450372751599, 0.8306164003088464),
    c64::new(-0.5574819482239914, 0.8301890612411025),
    c64::new(-0.5581185312205561, 0.829761233794523),
    c64::new(-0.5587547858903684, 0.8293329182207883),
    c64::new(-0.5593907118591358, 0.8289041147718651),
    c64::new(-0.5600263087527602, 0.8284748237000072),
    c64::new(-0.5606615761973359, 0.8280450452577559),
    c64::new(-0.5612965138191515, 0.8276147796979384),
    c64::new(-0.5619311212446895, 0.8271840272736691),
    c64::new(-0.5625653981006266, 0.8267527882383485),
    c64::new(-0.5631993440138339, 0.8263210628456636),
    c64::new(-0.5638329586113779, 0.825888851349587),
    c64::new(-0.5644662415205194, 0.8254561540043776),
    c64::new(-0.565099192368714, 0.8250229710645802),
    c64::new(-0.5657318107836132, 0.8245893027850253),
    c64::new(-0.5663640963930636, 0.8241551494208288),
    c64::new(-0.5669960488251085, 0.8237205112273915),
    c64::new(-0.5676276677079861, 0.8232853884604002),
    c64::new(-0.5682589526701315, 0.8228497813758264),
    c64::new(-0.5688899033401759, 0.8224136902299264),
    c64::new(-0.5695205193469473, 0.8219771152792414),
    c64::new(-0.5701508003194701, 0.8215400567805977),
    c64::new(-0.5707807458869671, 0.8211025149911048),
    c64::new(-0.5714103556788572, 0.8206644901681576),
    c64::new(-0.572039629324757, 0.8202259825694347),
    c64::new(-0.5726685664544813, 0.8197869924528989),
    c64::new(-0.573297166698042, 0.8193475200767971),
    c64::new(-0.5739254296856505, 0.8189075656996591),
    c64::new(-0.5745533550477157, 0.8184671295802988),
    c64::new(-0.5751809424148451, 0.8180262119778134),
    c64::new(-0.5758081914178453, 0.8175848131515837),
    c64::new(-0.5764351016877218, 0.817142933361273),
    c64::new(-0.5770616728556793, 0.816700572866828),
    c64::new(-0.5776879045531226, 0.8162577319284776),
    c64::new(-0.5783137964116555, 0.8158144108067339),
    c64::new(-0.5789393480630818, 0.8153706097623913),
    c64::new(-0.5795645591394057, 0.8149263290565265),
    c64::new(-0.5801894292728313, 0.8144815689504988),
    c64::new(-0.5808139580957643, 0.8140363297059485),
    c64::new(-0.5814381452408102, 0.8135906115847986),
    c64::new(-0.5820619903407754, 0.8131444148492536),
    c64::new(-0.5826854930286685, 0.8126977397617996),
    c64::new(-0.5833086529376984, 0.8122505865852039),
    c64::new(-0.5839314697012761, 0.8118029555825156),
    c64::new(-0.5845539429530151, 0.811354847017064),
    c64::new(-0.5851760723267303, 0.8109062611524598),
    c64::new(-0.5857978574564389, 0.8104571982525948),
    c64::new(-0.5864192979763606, 0.8100076585816411),
    c64::new(-0.5870403935209177, 0.8095576424040515),
    c64::new(-0.5876611437247365, 0.8091071499845583),
    c64::new(-0.5882815482226452, 0.8086561815881751),
    c64::new(-0.5889016066496757, 0.8082047374801948),
    c64::new(-0.5895213186410639, 0.8077528179261904),
    c64::new(-0.5901406838322489, 0.8073004231920144),
    c64::new(-0.590759701858874, 0.8068475535437994),
    c64::new(-0.5913783723567874, 0.8063942092479564),
    c64::new(-0.5919966949620409, 0.8059403905711764),
    c64::new(-0.5926146693108911, 0.8054860977804292),
    c64::new(-0.5932322950397999, 0.8050313311429635),
    c64::new(-0.5938495717854333, 0.8045760909263073),
    c64::new(-0.5944664991846643, 0.8041203773982659),
    c64::new(-0.5950830768745698, 0.8036641908269242),
    c64::new(-0.5956993044924332, 0.8032075314806449),
    c64::new(-0.5963151816757437, 0.8027503996280692),
    c64::new(-0.5969307080621965, 0.8022927955381156),
    c64::new(-0.597545883289693, 0.8018347194799815),
    c64::new(-0.5981607069963422, 0.8013761717231404),
    c64::new(-0.5987751788204586, 0.8009171525373444),
    c64::new(-0.5993892984005645, 0.8004576621926228),
    c64::new(-0.600003065375389, 0.7999977009592819),
    c64::new(-0.6006164793838686, 0.7995372691079052),
    c64::new(-0.6012295400651484, 0.7990763669093526),
    c64::new(-0.6018422470585799, 0.7986149946347609),
    c64::new(-0.6024546000037238, 0.7981531525555439),
    c64::new(-0.6030665985403482, 0.7976908409433912),
    c64::new(-0.6036782423084305, 0.7972280600702687),
    c64::new(-0.6042895309481559, 0.7967648102084189),
    c64::new(-0.6049004640999197, 0.7963010916303592),
    c64::new(-0.6055110414043254, 0.7958369046088836),
    c64::new(-0.6061212625021861, 0.7953722494170613),
    c64::new(-0.6067311270345246, 0.794907126328237),
    c64::new(-0.6073406346425726, 0.7944415356160308),
    c64::new(-0.6079497849677735, 0.7939754775543373),
    c64::new(-0.6085585776517793, 0.7935089524173267),
    c64::new(-0.6091670123364531, 0.7930419604794438),
    c64::new(-0.6097750886638684, 0.7925745020154077),
    c64::new(-0.6103828062763095, 0.7921065773002123),
    c64::new(-0.6109901648162716, 0.791638186609126),
    c64::new(-0.6115971639264618, 0.7911693302176903),
    c64::new(-0.612203803249798, 0.7907000084017217),
    c64::new(-0.6128100824294097, 0.79023022143731),
    c64::new(-0.6134160011086387, 0.7897599696008191),
    c64::new(-0.6140215589310382, 0.7892892531688859),
    c64::new(-0.6146267555403748, 0.7888180724184204),
    c64::new(-0.6152315905806267, 0.7883464276266063),
    c64::new(-0.615836063695985, 0.7878743190709002),
    c64::new(-0.6164401745308536, 0.7874017470290313),
    c64::new(-0.6170439227298499, 0.7869287117790017),
    c64::new(-0.6176473079378038, 0.7864552135990859),
    c64::new(-0.61825032979976, 0.7859812527678303),
    c64::new(-0.6188529879609762, 0.785506829564054),
    c64::new(-0.619455282066924, 0.7850319442668481),
    c64::new(-0.6200572117632892, 0.7845565971555752),
    c64::new(-0.6206587766959718, 0.7840807885098702),
    c64::new(-0.6212599765110874, 0.7836045186096383),
    c64::new(-0.6218608108549653, 0.7831277877350574),
    c64::new(-0.62246127937415, 0.7826505961665757),
    c64::new(-0.6230613817154013, 0.782172944184913),
    c64::new(-0.6236611175256946, 0.7816948320710594),
    c64::new(-0.6242604864522204, 0.7812162601062762),
    c64::new(-0.6248594881423862, 0.7807372285720946),
    c64::new(-0.6254581222438143, 0.7802577377503167),
    c64::new(-0.6260563884043435, 0.7797777879230146),
    c64::new(-0.6266542862720295, 0.7792973793725303),
    c64::new(-0.6272518154951439, 0.7788165123814762),
    c64::new(-0.6278489757221764, 0.7783351872327333),
    c64::new(-0.6284457666018326, 0.7778534042094531),
    c64::new(-0.6290421877830359, 0.7773711635950563),
    c64::new(-0.629638238914927, 0.7768884656732324),
    c64::new(-0.6302339196468645, 0.7764053107279403),
    c64::new(-0.6308292296284244, 0.7759216990434078),
    c64::new(-0.6314241685094016, 0.7754376309041306),
    c64::new(-0.632018735939809, 0.7749531065948739),
    c64::new(-0.6326129315698775, 0.7744681264006709),
    c64::new(-0.6332067550500573, 0.7739826906068228),
    c64::new(-0.6338002060310169, 0.7734967994988993),
    c64::new(-0.6343932841636453, 0.7730104533627371),
    c64::new(-0.6349859890990494, 0.7725236524844414),
    c64::new(-0.6355783204885561, 0.7720363971503845),
    c64::new(-0.6361702779837122, 0.7715486876472063),
    c64::new(-0.6367618612362843, 0.7710605242618137),
    c64::new(-0.6373530698982589, 0.7705719072813809),
    c64::new(-0.6379439036218439, 0.7700828369933481),
    c64::new(-0.6385343620594667, 0.769593313685423),
    c64::new(-0.6391244448637757, 0.7691033376455797),
    c64::new(-0.6397141516876406, 0.7686129091620583),
    c64::new(-0.6403034821841515, 0.7681220285233656),
    c64::new(-0.6408924360066212, 0.7676306960182735),
    c64::new(-0.641481012808583, 0.7671389119358205),
    c64::new(-0.6420692122437924, 0.7666466765653105),
    c64::new(-0.6426570339662269, 0.7661539901963129),
    c64::new(-0.6432444776300859, 0.7656608531186624),
    c64::new(-0.6438315428897913, 0.7651672656224591),
    c64::new(-0.6444182293999882, 0.7646732279980673),
    c64::new(-0.6450045368155439, 0.7641787405361168),
    c64::new(-0.6455904647915487, 0.7636838035275019),
    c64::new(-0.6461760129833164, 0.7631884172633813),
    c64::new(-0.6467611810463837, 0.7626925820351782),
    c64::new(-0.647345968636512, 0.7621962981345791),
    c64::new(-0.6479303754096852, 0.7616995658535354),
    c64::new(-0.6485144010221124, 0.7612023854842619),
    c64::new(-0.649098045130226, 0.7607047573192369),
    c64::new(-0.6496813073906833, 0.7602066816512023),
    c64::new(-0.6502641874603657, 0.7597081587731636),
    c64::new(-0.6508466849963808, 0.7592091889783882),
    c64::new(-0.6514287996560597, 0.7587097725604075),
    c64::new(-0.6520105310969595, 0.7582099098130153),
    c64::new(-0.6525918789768625, 0.7577096010302681),
    c64::new(-0.6531728429537765, 0.7572088465064848),
    c64::new(-0.653753422685936, 0.7567076465362459),
    c64::new(-0.6543336178318004, 0.7562060014143945),
    c64::new(-0.654913428050056, 0.755703911436036),
    c64::new(-0.6554928529996153, 0.7552013768965365),
    c64::new(-0.6560718923396177, 0.7546983980915244),
    c64::new(-0.6566505457294288, 0.7541949753168893),
    c64::new(-0.6572288128286424, 0.7536911088687814),
    c64::new(-0.6578066932970785, 0.7531867990436125),
    c64::new(-0.658384186794785, 0.7526820461380553),
    c64::new(-0.6589612929820373, 0.7521768504490427),
    c64::new(-0.6595380115193384, 0.7516712122737687),
    c64::new(-0.6601143420674203, 0.7511651319096866),
    c64::new(-0.6606902842872422, 0.7506586096545107),
    c64::new(-0.6612658378399922, 0.7501516458062151),
    c64::new(-0.6618410023870869, 0.7496442406630335),
    c64::new(-0.6624157775901718, 0.7491363945234593),
    c64::new(-0.6629901631111212, 0.7486281076862455),
    c64::new(-0.6635641586120397, 0.7481193804504037),
    c64::new(-0.6641377637552599, 0.7476102131152051),
    c64::new(-0.6647109782033448, 0.7471006059801801),
    c64::new(-0.6652838016190872, 0.7465905593451172),
    c64::new(-0.6658562336655094, 0.746080073510064),
    c64::new(-0.6664282740058651, 0.7455691487753255),
    c64::new(-0.6669999223036374, 0.7450577854414661),
    c64::new(-0.6675711782225402, 0.7445459838093074),
    c64::new(-0.6681420414265185, 0.7440337441799293),
    c64::new(-0.6687125115797481, 0.743521066854669),
    c64::new(-0.6692825883466358, 0.7430079521351218),
    c64::new(-0.6698522713918209, 0.7424944003231394),
    c64::new(-0.670421560380173, 0.7419804117208311),
    c64::new(-0.6709904549767942, 0.7414659866305633),
    c64::new(-0.6715589548470184, 0.740951125354959),
    c64::new(-0.6721270596564115, 0.7404358281968982),
    c64::new(-0.6726947690707727, 0.7399200954595163),
    c64::new(-0.6732620827561329, 0.7394039274462059),
    c64::new(-0.673829000378756, 0.7388873244606152),
    c64::new(-0.674395521605139, 0.7383702868066486),
    c64::new(-0.674961646102012, 0.737852814788466),
    c64::new(-0.6755273735363384, 0.737334908710483),
    c64::new(-0.6760927035753158, 0.73681656887737),
    c64::new(-0.6766576358863748, 0.7362977955940532),
    c64::new(-0.6772221701371803, 0.7357785891657136),
    c64::new(-0.6777863059956315, 0.7352589498977868),
    c64::new(-0.6783500431298612, 0.7347388780959637),
    c64::new(-0.6789133812082383, 0.7342183740661885),
    c64::new(-0.679476319899365, 0.7336974381146604),
    c64::new(-0.6800388588720789, 0.7331760705478328),
    c64::new(-0.680600997795453, 0.7326542716724128),
    c64::new(-0.6811627363387954, 0.7321320417953613),
    c64::new(-0.6817240741716496, 0.7316093812238927),
    c64::new(-0.6822850109637955, 0.7310862902654744),
    c64::new(-0.682845546385248, 0.7305627692278276),
    c64::new(-0.6834056801062587, 0.7300388184189261),
    c64::new(-0.6839654117973155, 0.729514438146997),
    c64::new(-0.6845247411291421, 0.7289896287205196),
    c64::new(-0.6850836677727002, 0.7284643904482253),
    c64::new(-0.6856421913991874, 0.7279387236390987),
    c64::new(-0.6862003116800386, 0.7274126286023759),
    c64::new(-0.6867580282869259, 0.7268861056475449),
    c64::new(-0.6873153408917592, 0.7263591550843459),
    c64::new(-0.6878722491666853, 0.7258317772227705),
    c64::new(-0.6884287527840903, 0.7253039723730609),
    c64::new(-0.688984851416597, 0.7247757408457114),
    c64::new(-0.6895405447370668, 0.724247082951467),
    c64::new(-0.6900958324186, 0.7237179990013234),
    c64::new(-0.6906507141345344, 0.7231884893065277),
    c64::new(-0.6912051895584482, 0.7226585541785758),
    c64::new(-0.6917592583641576, 0.7221281939292155),
    c64::new(-0.6923129202257181, 0.7215974088704438),
    c64::new(-0.6928661748174246, 0.7210661993145081),
    c64::new(-0.6934190218138119, 0.7205345655739053),
    c64::new(-0.6939714608896538, 0.7200025079613819),
    c64::new(-0.6945234917199654, 0.7194700267899332),
    c64::new(-0.6950751139800008, 0.7189371223728045),
    c64::new(-0.6956263273452549, 0.7184037950234898),
    c64::new(-0.696177131491463, 0.7178700450557317),
    c64::new(-0.6967275260946009, 0.717335872783522),
    c64::new(-0.6972775108308864, 0.7168012785210996),
    c64::new(-0.6978270853767772, 0.7162662625829532),
    c64::new(-0.6983762494089728, 0.7157308252838187),
    c64::new(-0.6989250026044141, 0.7151949669386801),
    c64::new(-0.6994733446402839, 0.714658687862769),
    c64::new(-0.7000212751940061, 0.7141219883715649),
    c64::new(-0.7005687939432482, 0.7135848687807937),
    c64::new(-0.7011159005659185, 0.7130473294064295),
    c64::new(-0.7016625947401685, 0.7125093705646923),
    c64::new(-0.7022088761443919, 0.7119709925720501),
    c64::new(-0.7027547444572251, 0.7114321957452168),
    c64::new(-0.7033001993575485, 0.7108929804011519),
    c64::new(-0.7038452405244848, 0.7103533468570624),
    c64::new(-0.7043898676374003, 0.709813295430401),
    c64::new(-0.7049340803759049, 0.7092728264388657),
    c64::new(-0.7054778784198522, 0.7087319402004006),
    c64::new(-0.7060212614493395, 0.7081906370331955),
    c64::new(-0.7065642291447094, 0.7076489172556846),
    c64::new(-0.7071067811865475, 0.7071067811865476),
    c64::new(-0.7076489172556844, 0.7065642291447095),
    c64::new(-0.7081906370331954, 0.7060212614493397),
    c64::new(-0.7087319402004004, 0.7054778784198523),
    c64::new(-0.7092728264388655, 0.7049340803759051),
    c64::new(-0.7098132954304008, 0.7043898676374005),
    c64::new(-0.7103533468570623, 0.703845240524485),
    c64::new(-0.7108929804011517, 0.7033001993575487),
    c64::new(-0.7114321957452165, 0.7027547444572252),
    c64::new(-0.7119709925720499, 0.702208876144392),
    c64::new(-0.7125093705646923, 0.7016625947401687),
    c64::new(-0.7130473294064292, 0.7011159005659188),
    c64::new(-0.7135848687807935, 0.7005687939432483),
    c64::new(-0.7141219883715648, 0.7000212751940063),
    c64::new(-0.7146586878627689, 0.699473344640284),
    c64::new(-0.7151949669386799, 0.6989250026044143),
    c64::new(-0.7157308252838186, 0.6983762494089729),
    c64::new(-0.716266262582953, 0.6978270853767774),
    c64::new(-0.7168012785210995, 0.6972775108308865),
    c64::new(-0.7173358727835217, 0.6967275260946011),
    c64::new(-0.7178700450557316, 0.6961771314914631),
    c64::new(-0.7184037950234896, 0.695626327345255),
    c64::new(-0.7189371223728044, 0.6950751139800009),
    c64::new(-0.719470026789933, 0.6945234917199655),
    c64::new(-0.7200025079613817, 0.6939714608896539),
    c64::new(-0.720534565573905, 0.6934190218138121),
    c64::new(-0.721066199314508, 0.6928661748174249),
    c64::new(-0.7215974088704437, 0.6923129202257183),
    c64::new(-0.7221281939292152, 0.6917592583641579),
    c64::new(-0.7226585541785757, 0.6912051895584485),
    c64::new(-0.7231884893065275, 0.6906507141345346),
    c64::new(-0.7237179990013234, 0.6900958324186001),
    c64::new(-0.7242470829514669, 0.689540544737067),
    c64::new(-0.7247757408457112, 0.6889848514165972),
    c64::new(-0.7253039723730608, 0.6884287527840904),
    c64::new(-0.7258317772227704, 0.6878722491666855),
    c64::new(-0.7263591550843458, 0.6873153408917593),
    c64::new(-0.7268861056475449, 0.686758028286926),
    c64::new(-0.7274126286023757, 0.6862003116800387),
    c64::new(-0.7279387236390986, 0.6856421913991876),
    c64::new(-0.7284643904482252, 0.6850836677727004),
    c64::new(-0.7289896287205195, 0.6845247411291422),
    c64::new(-0.7295144381469969, 0.6839654117973156),
    c64::new(-0.7300388184189261, 0.6834056801062589),
    c64::new(-0.7305627692278276, 0.6828455463852481),
    c64::new(-0.7310862902654742, 0.6822850109637956),
    c64::new(-0.7316093812238926, 0.6817240741716497),
    c64::new(-0.7321320417953612, 0.6811627363387956),
    c64::new(-0.7326542716724127, 0.6806009977954532),
    c64::new(-0.7331760705478327, 0.680038858872079),
    c64::new(-0.7336974381146603, 0.6794763198993651),
    c64::new(-0.7342183740661883, 0.6789133812082384),
    c64::new(-0.7347388780959636, 0.6783500431298615),
    c64::new(-0.7352589498977866, 0.6777863059956316),
    c64::new(-0.7357785891657134, 0.6772221701371806),
    c64::new(-0.7362977955940531, 0.676657635886375),
    c64::new(-0.7368165688773698, 0.6760927035753159),
    c64::new(-0.7373349087104829, 0.6755273735363385),
    c64::new(-0.7378528147884658, 0.6749616461020123),
    c64::new(-0.7383702868066484, 0.6743955216051392),
    c64::new(-0.7388873244606151, 0.6738290003787561),
    c64::new(-0.7394039274462056, 0.6732620827561331),
    c64::new(-0.7399200954595162, 0.6726947690707729),
    c64::new(-0.7404358281968981, 0.6721270596564117),
    c64::new(-0.7409511253549589, 0.6715589548470186),
    c64::new(-0.7414659866305632, 0.6709904549767943),
    c64::new(-0.741980411720831, 0.6704215603801732),
    c64::new(-0.7424944003231393, 0.669852271391821),
    c64::new(-0.7430079521351217, 0.669282588346636),
    c64::new(-0.7435210668546689, 0.6687125115797483),
    c64::new(-0.7440337441799292, 0.6681420414265187),
    c64::new(-0.7445459838093073, 0.6675711782225404),
    c64::new(-0.745057785441466, 0.6669999223036376),
    c64::new(-0.7455691487753254, 0.6664282740058654),
    c64::new(-0.7460800735100639, 0.6658562336655096),
    c64::new(-0.7465905593451171, 0.6652838016190874),
    c64::new(-0.74710060598018, 0.664710978203345),
    c64::new(-0.7476102131152051, 0.66413776375526),
    c64::new(-0.7481193804504036, 0.6635641586120398),
    c64::new(-0.7486281076862453, 0.6629901631111215),
    c64::new(-0.749136394523459, 0.662415777590172),
    c64::new(-0.7496442406630334, 0.661841002387087),
    c64::new(-0.750151645806215, 0.6612658378399923),
    c64::new(-0.7506586096545106, 0.6606902842872423),
    c64::new(-0.7511651319096864, 0.6601143420674205),
    c64::new(-0.7516712122737685, 0.6595380115193386),
    c64::new(-0.7521768504490426, 0.6589612929820375),
    c64::new(-0.7526820461380552, 0.6583841867947853),
    c64::new(-0.7531867990436124, 0.6578066932970787),
    c64::new(-0.7536911088687813, 0.6572288128286425),
    c64::new(-0.7541949753168892, 0.656650545729429),
    c64::new(-0.7546983980915243, 0.6560718923396179),
    c64::new(-0.7552013768965364, 0.6554928529996156),
    c64::new(-0.7557039114360358, 0.6549134280500561),
    c64::new(-0.7562060014143945, 0.6543336178318006),
    c64::new(-0.7567076465362457, 0.6537534226859361),
    c64::new(-0.7572088465064847, 0.6531728429537766),
    c64::new(-0.757709601030268, 0.6525918789768628),
    c64::new(-0.7582099098130152, 0.6520105310969597),
    c64::new(-0.7587097725604073, 0.6514287996560599),
    c64::new(-0.759209188978388, 0.650846684996381),
    c64::new(-0.7597081587731634, 0.6502641874603659),
    c64::new(-0.7602066816512022, 0.6496813073906834),
    c64::new(-0.7607047573192368, 0.6490980451302262),
    c64::new(-0.7612023854842618, 0.6485144010221126),
    c64::new(-0.7616995658535353, 0.6479303754096855),
    c64::new(-0.7621962981345789, 0.6473459686365121),
    c64::new(-0.762692582035178, 0.6467611810463838),
    c64::new(-0.7631884172633812, 0.6461760129833165),
    c64::new(-0.7636838035275018, 0.6455904647915489),
    c64::new(-0.7641787405361167, 0.645004536815544),
    c64::new(-0.7646732279980671, 0.6444182293999884),
    c64::new(-0.765167265622459, 0.6438315428897914),
    c64::new(-0.7656608531186623, 0.6432444776300861),
    c64::new(-0.7661539901963128, 0.6426570339662271),
    c64::new(-0.7666466765653104, 0.6420692122437927),
    c64::new(-0.7671389119358203, 0.6414810128085832),
    c64::new(-0.7676306960182734, 0.6408924360066214),
    c64::new(-0.7681220285233654, 0.6403034821841516),
    c64::new(-0.7686129091620582, 0.6397141516876407),
    c64::new(-0.7691033376455796, 0.6391244448637758),
    c64::new(-0.7695933136854229, 0.6385343620594668),
    c64::new(-0.7700828369933479, 0.637943903621844),
    c64::new(-0.7705719072813808, 0.6373530698982591),
    c64::new(-0.7710605242618136, 0.6367618612362844),
    c64::new(-0.7715486876472062, 0.6361702779837123),
    c64::new(-0.7720363971503844, 0.6355783204885562),
    c64::new(-0.7725236524844412, 0.6349859890990496),
    c64::new(-0.773010453362737, 0.6343932841636455),
    c64::new(-0.7734967994988992, 0.6338002060310172),
    c64::new(-0.7739826906068226, 0.6332067550500574),
    c64::new(-0.7744681264006708, 0.6326129315698776),
    c64::new(-0.7749531065948738, 0.6320187359398091),
    c64::new(-0.7754376309041305, 0.6314241685094019),
    c64::new(-0.7759216990434077, 0.6308292296284245),
    c64::new(-0.7764053107279402, 0.6302339196468647),
    c64::new(-0.7768884656732323, 0.6296382389149272),
    c64::new(-0.7773711635950562, 0.6290421877830361),
    c64::new(-0.777853404209453, 0.6284457666018327),
    c64::new(-0.7783351872327332, 0.6278489757221766),
    c64::new(-0.7788165123814761, 0.6272518154951441),
    c64::new(-0.7792973793725301, 0.6266542862720297),
    c64::new(-0.7797777879230143, 0.6260563884043436),
    c64::new(-0.7802577377503166, 0.6254581222438144),
    c64::new(-0.7807372285720945, 0.6248594881423863),
    c64::new(-0.7812162601062761, 0.6242604864522207),
    c64::new(-0.7816948320710593, 0.6236611175256949),
    c64::new(-0.7821729441849128, 0.6230613817154015),
    c64::new(-0.7826505961665756, 0.6224612793741501),
    c64::new(-0.7831277877350573, 0.6218608108549654),
    c64::new(-0.7836045186096383, 0.6212599765110876),
    c64::new(-0.7840807885098701, 0.620658776695972),
    c64::new(-0.7845565971555751, 0.6200572117632894),
    c64::new(-0.785031944266848, 0.6194552820669241),
    c64::new(-0.7855068295640539, 0.6188529879609764),
    c64::new(-0.7859812527678302, 0.6182503297997602),
    c64::new(-0.7864552135990858, 0.6176473079378039),
    c64::new(-0.7869287117790016, 0.61704392272985),
    c64::new(-0.7874017470290313, 0.6164401745308538),
    c64::new(-0.7878743190709001, 0.6158360636959851),
    c64::new(-0.7883464276266062, 0.6152315905806269),
    c64::new(-0.7888180724184203, 0.614626755540375),
    c64::new(-0.7892892531688858, 0.6140215589310384),
    c64::new(-0.7897599696008188, 0.6134160011086388),
    c64::new(-0.7902302214373099, 0.6128100824294098),
    c64::new(-0.7907000084017216, 0.6122038032497981),
    c64::new(-0.7911693302176901, 0.6115971639264619),
    c64::new(-0.7916381866091259, 0.6109901648162717),
    c64::new(-0.7921065773002122, 0.6103828062763097),
    c64::new(-0.7925745020154076, 0.6097750886638685),
    c64::new(-0.7930419604794436, 0.6091670123364533),
    c64::new(-0.7935089524173266, 0.6085585776517795),
    c64::new(-0.7939754775543372, 0.6079497849677736),
    c64::new(-0.7944415356160306, 0.6073406346425728),
    c64::new(-0.7949071263282368, 0.6067311270345247),
    c64::new(-0.7953722494170612, 0.6061212625021863),
    c64::new(-0.7958369046088835, 0.6055110414043255),
    c64::new(-0.7963010916303591, 0.6049004640999198),
    c64::new(-0.7967648102084188, 0.604289530948156),
    c64::new(-0.7972280600702686, 0.6036782423084306),
    c64::new(-0.7976908409433909, 0.6030665985403484),
    c64::new(-0.7981531525555438, 0.6024546000037239),
    c64::new(-0.7986149946347608, 0.6018422470585801),
    c64::new(-0.7990763669093524, 0.6012295400651485),
    c64::new(-0.799537269107905, 0.6006164793838689),
    c64::new(-0.7999977009592818, 0.6000030653753893),
    c64::new(-0.8004576621926227, 0.5993892984005647),
    c64::new(-0.8009171525373443, 0.5987751788204588),
    c64::new(-0.8013761717231402, 0.5981607069963424),
    c64::new(-0.8018347194799813, 0.5975458832896932),
    c64::new(-0.8022927955381156, 0.5969307080621967),
    c64::new(-0.802750399628069, 0.5963151816757439),
    c64::new(-0.8032075314806448, 0.5956993044924335),
    c64::new(-0.8036641908269241, 0.5950830768745701),
    c64::new(-0.8041203773982657, 0.5944664991846644),
    c64::new(-0.8045760909263071, 0.5938495717854335),
    c64::new(-0.8050313311429634, 0.5932322950398),
    c64::new(-0.805486097780429, 0.5926146693108914),
    c64::new(-0.8059403905711763, 0.591996694962041),
    c64::new(-0.8063942092479564, 0.5913783723567876),
    c64::new(-0.8068475535437993, 0.5907597018588742),
    c64::new(-0.8073004231920142, 0.5901406838322492),
    c64::new(-0.8077528179261902, 0.589521318641064),
    c64::new(-0.8082047374801946, 0.588901606649676),
    c64::new(-0.808656181588175, 0.5882815482226453),
    c64::new(-0.8091071499845582, 0.5876611437247367),
    c64::new(-0.8095576424040514, 0.5870403935209179),
    c64::new(-0.8100076585816409, 0.5864192979763607),
    c64::new(-0.8104571982525947, 0.585797857456439),
    c64::new(-0.8109062611524597, 0.5851760723267305),
    c64::new(-0.8113548470170637, 0.5845539429530153),
    c64::new(-0.8118029555825155, 0.5839314697012762),
    c64::new(-0.8122505865852037, 0.5833086529376985),
    c64::new(-0.8126977397617994, 0.5826854930286686),
    c64::new(-0.8131444148492535, 0.5820619903407757),
    c64::new(-0.8135906115847984, 0.5814381452408103),
    c64::new(-0.8140363297059484, 0.5808139580957645),
    c64::new(-0.8144815689504987, 0.5801894292728316),
    c64::new(-0.8149263290565264, 0.579564559139406),
    c64::new(-0.8153706097623912, 0.578939348063082),
    c64::new(-0.8158144108067338, 0.5783137964116557),
    c64::new(-0.8162577319284774, 0.5776879045531228),
    c64::new(-0.8167005728668278, 0.5770616728556794),
    c64::new(-0.8171429333612727, 0.576435101687722),
    c64::new(-0.8175848131515836, 0.5758081914178454),
    c64::new(-0.8180262119778134, 0.5751809424148453),
    c64::new(-0.8184671295802987, 0.5745533550477159),
    c64::new(-0.818907565699659, 0.5739254296856507),
    c64::new(-0.819347520076797, 0.5732971666980421),
    c64::new(-0.8197869924528989, 0.5726685664544815),
    c64::new(-0.8202259825694346, 0.5720396293247573),
    c64::new(-0.8206644901681575, 0.5714103556788573),
    c64::new(-0.8211025149911046, 0.5707807458869673),
    c64::new(-0.8215400567805976, 0.5701508003194703),
    c64::new(-0.8219771152792414, 0.5695205193469475),
    c64::new(-0.8224136902299263, 0.5688899033401761),
    c64::new(-0.8228497813758262, 0.5682589526701317),
    c64::new(-0.8232853884604001, 0.5676276677079863),
    c64::new(-0.8237205112273914, 0.5669960488251087),
    c64::new(-0.8241551494208286, 0.5663640963930637),
    c64::new(-0.8245893027850251, 0.5657318107836135),
    c64::new(-0.8250229710645801, 0.5650991923687142),
    c64::new(-0.8254561540043774, 0.5644662415205195),
    c64::new(-0.8258888513495868, 0.5638329586113782),
    c64::new(-0.8263210628456635, 0.5631993440138341),
    c64::new(-0.8267527882383484, 0.5625653981006268),
    c64::new(-0.827184027273669, 0.5619311212446896),
    c64::new(-0.8276147796979383, 0.5612965138191517),
    c64::new(-0.8280450452577557, 0.560661576197336),
    c64::new(-0.8284748237000071, 0.5600263087527604),
    c64::new(-0.828904114771865, 0.559390711859136),
    c64::new(-0.829332918220788, 0.5587547858903685),
    c64::new(-0.8297612337945229, 0.5581185312205563),
    c64::new(-0.8301890612411023, 0.5574819482239917),
    c64::new(-0.8306164003088463, 0.5568450372751601),
    c64::new(-0.8310432507463623, 0.5562077987487399),
    c64::new(-0.8314696123025453, 0.5555702330196022),
    c64::new(-0.8318954847265775, 0.5549323404628106),
    c64::new(-0.8323208677679296, 0.5542941214536202),
    c64::new(-0.8327457611763595, 0.5536555763674793),
    c64::new(-0.8331701647019132, 0.5530167055800275),
    c64::new(-0.8335940780949253, 0.552377509467096),
    c64::new(-0.834017501106018, 0.5517379884047077),
    c64::new(-0.8344404334861031, 0.5510981427690755),
    c64::new(-0.83486287498638, 0.5504579729366049),
    c64::new(-0.8352848253583374, 0.5498174792838909),
    c64::new(-0.8357062843537526, 0.5491766621877197),
    c64::new(-0.8361272517246923, 0.5485355220250673),
    c64::new(-0.8365477272235119, 0.5478940591731003),
    c64::new(-0.8369677106028569, 0.5472522740091742),
    c64::new(-0.8373872016156618, 0.546610166910835),
    c64::new(-0.8378062000151509, 0.5459677382558176),
    c64::new(-0.8382247055548381, 0.5453249884220464),
    c64::new(-0.8386427179885272, 0.5446819177876348),
    c64::new(-0.8390602370703126, 0.544038526730884),
    c64::new(-0.8394772625545784, 0.5433948156302849),
    c64::new(-0.8398937941959995, 0.5427507848645159),
    c64::new(-0.8403098317495408, 0.5421064348124439),
    c64::new(-0.8407253749704582, 0.5414617658531233),
    c64::new(-0.841140423614298, 0.5408167783657968),
    c64::new(-0.8415549774368983, 0.540171472729893),
    c64::new(-0.8419690361943877, 0.539525849325029),
    c64::new(-0.8423825996431858, 0.5388799085310084),
    c64::new(-0.8427956675400042, 0.5382336507278216),
    c64::new(-0.8432082396418453, 0.5375870762956457),
    c64::new(-0.843620315706004, 0.5369401856148431),
    c64::new(-0.8440318954900664, 0.5362929790659632),
    c64::new(-0.8444429787519107, 0.5356454570297411),
    c64::new(-0.8448535652497071, 0.5349976198870972),
    c64::new(-0.8452636547419183, 0.5343494680191374),
    c64::new(-0.845673246987299, 0.5337010018071532),
    c64::new(-0.8460823417448968, 0.5330522216326197),
    c64::new(-0.846490938774052, 0.532403127877198),
    c64::new(-0.8468990378343972, 0.5317537209227333),
    c64::new(-0.8473066386858584, 0.5311040011512549),
    c64::new(-0.8477137410886542, 0.5304539689449767),
    c64::new(-0.8481203448032971, 0.5298036246862948),
    c64::new(-0.8485264495905926, 0.5291529687577907),
    c64::new(-0.8489320552116396, 0.5285020015422285),
    c64::new(-0.8493371614278308, 0.5278507234225552),
    c64::new(-0.8497417680008525, 0.5271991347819013),
    c64::new(-0.8501458746926851, 0.5265472360035796),
    c64::new(-0.8505494812656034, 0.5258950274710849),
    c64::new(-0.8509525874821757, 0.5252425095680947),
    c64::new(-0.8513551931052652, 0.524589682678469),
    c64::new(-0.8517572978980292, 0.5239365471862485),
    c64::new(-0.8521589016239197, 0.5232831034756567),
    c64::new(-0.852560004046684, 0.5226293519310968),
    c64::new(-0.8529606049303636, 0.5219752929371544),
    c64::new(-0.8533607040392954, 0.5213209268785957),
    c64::new(-0.8537603011381114, 0.5206662541403672),
    c64::new(-0.8541593959917388, 0.5200112751075959),
    c64::new(-0.8545579883654004, 0.5193559901655898),
    c64::new(-0.8549560780246148, 0.5187003996998352),
    c64::new(-0.855353664735196, 0.5180445040959994),
    c64::new(-0.8557507482632538, 0.5173883037399291),
    c64::new(-0.8561473283751946, 0.5167317990176498),
    c64::new(-0.8565434048377198, 0.5160749903153669),
    c64::new(-0.8569389774178287, 0.5154178780194631),
    c64::new(-0.8573340458828156, 0.5147604625165012),
    c64::new(-0.8577286100002721, 0.5141027441932218),
    c64::new(-0.8581226695380861, 0.5134447234365435),
    c64::new(-0.8585162242644429, 0.512786400633563),
    c64::new(-0.8589092739478237, 0.5121277761715549),
    c64::new(-0.8593018183570083, 0.5114688504379705),
    c64::new(-0.8596938572610726, 0.5108096238204392),
    c64::new(-0.8600853904293901, 0.5101500967067668),
    c64::new(-0.8604764176316322, 0.5094902694849363),
    c64::new(-0.8608669386377672, 0.5088301425431073),
    c64::new(-0.8612569532180621, 0.5081697162696148),
    c64::new(-0.8616464611430813, 0.507508991052971),
    c64::new(-0.8620354621836872, 0.5068479672818633),
    c64::new(-0.8624239561110405, 0.5061866453451552),
    c64::new(-0.8628119426966004, 0.5055250256318853),
    c64::new(-0.863199421712124, 0.5048631085312677),
    c64::new(-0.863586392929668, 0.5042008944326906),
    c64::new(-0.8639728561215867, 0.5035383837257176),
    c64::new(-0.864358811060534, 0.502875576800087),
    c64::new(-0.8647442575194624, 0.5022124740457107),
    c64::new(-0.8651291952716236, 0.5015490758526756),
    c64::new(-0.865513624090569, 0.5008853826112409),
    c64::new(-0.8658975437501488, 0.5002213947118407),
    c64::new(-0.866280954024513, 0.4995571125450819),
    c64::new(-0.8666638546881111, 0.4988925365017446),
    c64::new(-0.8670462455156926, 0.49822766697278176),
    c64::new(-0.8674281262823068, 0.49756250434931937),
    c64::new(-0.8678094967633032, 0.4968970490226547),
    c64::new(-0.8681903567343312, 0.49623130138425836),
    c64::new(-0.8685707059713409, 0.49556526182577254),
    c64::new(-0.8689505442505825, 0.49489893073901114),
    c64::new(-0.8693298713486066, 0.49423230851595995),
    c64::new(-0.8697086870422656, 0.49356539554877493),
    c64::new(-0.8700869911087113, 0.49289819222978415),
    c64::new(-0.8704647833253976, 0.4922306989514861),
    c64::new(-0.8708420634700789, 0.4915629161065499),
    c64::new(-0.871218831320811, 0.49089484408781503),
    c64::new(-0.871595086655951, 0.4902264832882914),
    c64::new(-0.8719708292541577, 0.4895578341011576),
    c64::new(-0.8723460588943914, 0.4888888969197632),
    c64::new(-0.8727207753559143, 0.4882196721376268),
    c64::new(-0.8730949784182902, 0.4875501601484359),
    c64::new(-0.8734686678613848, 0.4868803613460476),
    c64::new(-0.8738418434653668, 0.4862102761244866),
    c64::new(-0.8742145050107062, 0.48553990487794707),
    c64::new(-0.8745866522781761, 0.4848692480007911),
    c64::new(-0.8749582850488516, 0.48419830588754903),
    c64::new(-0.8753294031041109, 0.48352707893291863),
    c64::new(-0.8757000062256345, 0.4828555675317659),
    c64::new(-0.8760700941954065, 0.4821837720791229),
    c64::new(-0.8764396667957136, 0.48151169297019),
    c64::new(-0.8768087238091457, 0.48083933060033396),
    c64::new(-0.877177265018596, 0.48016668536508833),
    c64::new(-0.8775452902072611, 0.4794937576601533),
    c64::new(-0.8779127991586417, 0.4788205478813941),
    c64::new(-0.8782797916565415, 0.4781470564248432),
    c64::new(-0.8786462674850681, 0.4774732836866981),
    c64::new(-0.8790122264286335, 0.4767992300633221),
    c64::new(-0.8793776682719533, 0.4761248959512435),
    c64::new(-0.8797425928000473, 0.4754502817471561),
    c64::new(-0.8801069997982403, 0.4747753878479173),
    c64::new(-0.8804708890521608, 0.4741002146505501),
    c64::new(-0.8808342603477419, 0.47342476255224153),
    c64::new(-0.8811971134712221, 0.47274903195034274),
    c64::new(-0.8815594482091437, 0.47207302324236894),
    c64::new(-0.8819212643483549, 0.4713967368259978),
    c64::new(-0.8822825616760086, 0.47072017309907177),
    c64::new(-0.8826433399795628, 0.4700433324595957),
    c64::new(-0.8830035990467808, 0.4693662153057375),
    c64::new(-0.8833633386657317, 0.46868882203582785),
    c64::new(-0.8837225586247894, 0.46801115304836005),
    c64::new(-0.8840812587126349, 0.4673332087419886),
    c64::new(-0.8844394387182537, 0.466654989515531),
    c64::new(-0.8847970984309378, 0.4659764957679662),
    c64::new(-0.8851542376402851, 0.46529772789843454),
    c64::new(-0.8855108561361998, 0.4646186863062381),
    c64::new(-0.8858669537088927, 0.46393937139083874),
    c64::new(-0.8862225301488805, 0.4632597835518603),
    c64::new(-0.886577585246987, 0.46257992318908686),
    c64::new(-0.8869321187943422, 0.46189979070246273),
    c64::new(-0.8872861305823833, 0.46121938649209226),
    c64::new(-0.8876396204028538, 0.4605387109582402),
    c64::new(-0.8879925880478055, 0.45985776450132965),
    c64::new(-0.8883450333095962, 0.4591765475219442),
    c64::new(-0.8886969559808916, 0.4584950604208263),
    c64::new(-0.8890483558546646, 0.4578133035988772),
    c64::new(-0.8893992327241954, 0.4571312774571572),
    c64::new(-0.8897495863830727, 0.45644898239688414),
    c64::new(-0.8900994166251922, 0.4557664188194348),
    c64::new(-0.8904487232447579, 0.4550835871263439),
    c64::new(-0.8907975060362815, 0.4544004877193036),
    c64::new(-0.8911457647945833, 0.45371712100016376),
    c64::new(-0.8914934993147913, 0.45303348737093185),
    c64::new(-0.8918407093923426, 0.452349587233771),
    c64::new(-0.8921873948229825, 0.4516654209910026),
    c64::new(-0.8925335554027646, 0.45098098904510386),
    c64::new(-0.8928791909280518, 0.45029629179870856),
    c64::new(-0.8932243011955152, 0.4496113296546069),
    c64::new(-0.8935688860021359, 0.4489261030157435),
    c64::new(-0.8939129451452033, 0.44824061228522005),
    c64::new(-0.894256478422316, 0.44755485786629307),
    c64::new(-0.8945994856313827, 0.44686884016237416),
    c64::new(-0.8949419665706208, 0.44618255957702996),
    c64::new(-0.8952839210385575, 0.44549601651398196),
    c64::new(-0.89562534883403, 0.44480921137710505),
    c64::new(-0.8959662497561851, 0.4441221445704293),
    c64::new(-0.8963066236044795, 0.4434348164981385),
    c64::new(-0.8966464701786803, 0.44274722756456997),
    c64::new(-0.8969857892788639, 0.44205937817421503),
    c64::new(-0.8973245807054182, 0.4413712687317169),
    c64::new(-0.8976628442590407, 0.440682899641873),
    c64::new(-0.8980005797407398, 0.4399942713096333),
    c64::new(-0.8983377869518343, 0.43930538414009995),
    c64::new(-0.8986744656939539, 0.43861623853852755),
    c64::new(-0.899010615769039, 0.43792683491032314),
    c64::new(-0.8993462369793415, 0.43723717366104425),
    c64::new(-0.8996813291274239, 0.4365472551964013),
    c64::new(-0.9000158920161603, 0.4358570799222555),
    c64::new(-0.9003499254487357, 0.4351666482446192),
    c64::new(-0.9006834292286467, 0.434475960569656),
    c64::new(-0.9010164031597022, 0.43378501730367874),
    c64::new(-0.9013488470460219, 0.43309381885315207),
    c64::new(-0.9016807606920377, 0.4324023656246902),
    c64::new(-0.9020121439024932, 0.43171065802505726),
    c64::new(-0.9023429964824443, 0.4310186964611669),
    c64::new(-0.9026733182372587, 0.4303264813400829),
    c64::new(-0.903003108972617, 0.42963401306901655),
    c64::new(-0.9033323684945118, 0.4289412920553296),
    c64::new(-0.903661096609248, 0.42824831870653196),
    c64::new(-0.9039892931234433, 0.42755509343028203),
    c64::new(-0.9043169578440282, 0.42686161663438676),
    c64::new(-0.9046440905782461, 0.42616788872679984),
    c64::new(-0.9049706911336532, 0.42547391011562397),
    c64::new(-0.9052967593181187, 0.42477968120910886),
    c64::new(-0.9056222949398253, 0.42408520241565156),
    c64::new(-0.9059472978072685, 0.42339047414379594),
    c64::new(-0.9062717677292575, 0.42269549680223323),
    c64::new(-0.9065957045149153, 0.42200027079979985),
    c64::new(-0.906919107973678, 0.42130479654547975),
    c64::new(-0.9072419779152958, 0.4206090744484025),
    c64::new(-0.9075643141498326, 0.41991310491784356),
    c64::new(-0.907886116487666, 0.41921688836322424),
    c64::new(-0.9082073847394886, 0.4185204251941099),
    c64::new(-0.9085281187163061, 0.41782371582021244),
    c64::new(-0.908848318229439, 0.41712676065138793),
    c64::new(-0.9091679830905224, 0.41642956009763715),
    c64::new(-0.9094871131115054, 0.41573211456910525),
    c64::new(-0.9098057081046521, 0.41503442447608185),
    c64::new(-0.9101237678825416, 0.41433649022899927),
    c64::new(-0.9104412922580671, 0.4136383122384346),
    c64::new(-0.9107582810444376, 0.4129398909151081),
    c64::new(-0.9110747340551764, 0.41224122666988283),
    c64::new(-0.9113906511041222, 0.4115423199137655),
    c64::new(-0.9117060320054298, 0.41084317105790413),
    c64::new(-0.9120208765735682, 0.41014378051359035),
    c64::new(-0.9123351846233227, 0.40944414869225765),
    c64::new(-0.9126489559697939, 0.40874427600548136),
    c64::new(-0.9129621904283982, 0.4080441628649786),
    c64::new(-0.9132748878148677, 0.4073438096826082),
    c64::new(-0.9135870479452508, 0.4066432168703692),
    c64::new(-0.9138986706359116, 0.40594238484040257),
    c64::new(-0.9142097557035307, 0.40524131400498986),
    c64::new(-0.9145203029651046, 0.40454000477655294),
    c64::new(-0.914830312237946, 0.4038384575676544),
    c64::new(-0.9151397833396852, 0.40313667279099547),
    c64::new(-0.9154487160882677, 0.40243465085941854),
    c64::new(-0.9157571103019567, 0.40173239218590506),
    c64::new(-0.9160649657993317, 0.4010298971835756),
    c64::new(-0.9163722823992891, 0.40032716626569),
    c64::new(-0.9166790599210426, 0.39962419984564707),
    c64::new(-0.9169852981841229, 0.398920998336983),
    c64::new(-0.9172909970083779, 0.39821756215337367),
    c64::new(-0.9175961562139728, 0.3975138917086324),
    c64::new(-0.9179007756213905, 0.39680998741671025),
    c64::new(-0.9182048550514308, 0.39610584969169654),
    c64::new(-0.9185083943252121, 0.3954014789478165),
    c64::new(-0.9188113932641699, 0.3946968755994337),
    c64::new(-0.9191138516900578, 0.39399204006104815),
    c64::new(-0.9194157694249471, 0.3932869727472964),
    c64::new(-0.9197171462912274, 0.3925816740729514),
    c64::new(-0.9200179821116065, 0.39187614445292257),
    c64::new(-0.9203182767091105, 0.39117038430225404),
    c64::new(-0.9206180299070839, 0.3904643940361267),
    c64::new(-0.9209172415291894, 0.38975817406985647),
    c64::new(-0.9212159113994087, 0.38905172481889433),
    c64::new(-0.9215140393420418, 0.3883450466988266),
    c64::new(-0.921811625181708, 0.3876381401253729),
    c64::new(-0.9221086687433451, 0.3869310055143887),
    c64::new(-0.9224051698522099, 0.38622364328186304),
    c64::new(-0.9227011283338786, 0.38551605384391885),
    c64::new(-0.9229965440142464, 0.38480823761681276),
    c64::new(-0.9232914167195275, 0.3841001950169353),
    c64::new(-0.9235857462762566, 0.38339192646080883),
    c64::new(-0.9238795325112867, 0.38268343236508984),
    c64::new(-0.9241727752517911, 0.3819747131465673),
    c64::new(-0.9244654743252627, 0.3812657692221623),
    c64::new(-0.9247576295595138, 0.3805566010089288),
    c64::new(-0.9250492407826775, 0.3798472089240514),
    c64::new(-0.9253403078232062, 0.3791375933848475),
    c64::new(-0.9256308305098727, 0.3784277548087656),
    c64::new(-0.9259208086717701, 0.37771769361338564),
    c64::new(-0.9262102421383114, 0.37700741021641815),
    c64::new(-0.9264991307392304, 0.37629690503570506),
    c64::new(-0.9267874743045817, 0.3755861784892174),
    c64::new(-0.9270752726647401, 0.37487523099505765),
    c64::new(-0.9273625256504011, 0.374164062971458),
    c64::new(-0.9276492330925812, 0.37345267483678024),
    c64::new(-0.9279353948226178, 0.37274106700951604),
    c64::new(-0.9282210106721693, 0.37202923990828524),
    c64::new(-0.9285060804732155, 0.37131719395183765),
    c64::new(-0.9287906040580569, 0.37060492955905167),
    c64::new(-0.9290745812593157, 0.3698924471489341),
    c64::new(-0.9293580119099356, 0.3691797471406199),
    c64::new(-0.9296408958431812, 0.3684668299533726),
    c64::new(-0.9299232328926396, 0.36775369600658214),
    c64::new(-0.930205022892219, 0.3670403457197673),
    c64::new(-0.9304862656761497, 0.36632677951257364),
    c64::new(-0.9307669610789837, 0.3656129978047738),
    c64::new(-0.9310471089355952, 0.3648990010162676),
    c64::new(-0.9313267090811803, 0.3641847895670801),
    c64::new(-0.9316057613512578, 0.3634703638773639),
    c64::new(-0.9318842655816681, 0.3627557243673973),
    c64::new(-0.9321622216085744, 0.3620408714575842),
    c64::new(-0.9324396292684625, 0.3613258055684542),
    c64::new(-0.9327164883981401, 0.36061052712066255),
    c64::new(-0.9329927988347388, 0.35989503653498833),
    c64::new(-0.933268560415712, 0.3591793342323366),
    c64::new(-0.9335437729788362, 0.3584634206337366),
    c64::new(-0.933818436362211, 0.35774729616034184),
    c64::new(-0.9340925504042588, 0.3570309612334303),
    c64::new(-0.9343661149437258, 0.3563144162744026),
    c64::new(-0.9346391298196807, 0.355597661704784),
    c64::new(-0.9349115948715161, 0.35488069794622285),
    c64::new(-0.9351835099389476, 0.35416352542049034),
    c64::new(-0.9354548748620147, 0.3534461445494807),
    c64::new(-0.9357256894810803, 0.35272855575521095),
    c64::new(-0.9359959536368313, 0.3520107594598193),
    c64::new(-0.9362656671702783, 0.3512927560855672),
    c64::new(-0.9365348299227555, 0.35057454605483757),
    c64::new(-0.9368034417359217, 0.34985612979013486),
    c64::new(-0.9370715024517591, 0.34913750771408525),
    c64::new(-0.9373390119125748, 0.3484186802494348),
    c64::new(-0.937605969961, 0.3476996478190515),
    c64::new(-0.9378723764399898, 0.34698041084592374),
    c64::new(-0.9381382311928244, 0.34626096975316),
    c64::new(-0.9384035340631082, 0.345541324963989),
    c64::new(-0.93866828489477, 0.34482147690175957),
    c64::new(-0.9389324835320645, 0.34410142598993904),
    c64::new(-0.9391961298195699, 0.34338117265211515),
    c64::new(-0.9394592236021899, 0.34266071731199443),
    c64::new(-0.9397217647251533, 0.34194006039340213),
    c64::new(-0.9399837530340138, 0.3412192023202827),
    c64::new(-0.9402451883746508, 0.3404981435166974),
    c64::new(-0.9405060705932683, 0.339776884406827),
    c64::new(-0.9407663995363961, 0.3390554254149697),
    c64::new(-0.9410261750508893, 0.3383337669655411),
    c64::new(-0.9412853969839288, 0.3376119094830745),
    c64::new(-0.9415440651830207, 0.3368898533922203),
    c64::new(-0.9418021794959976, 0.3361675991177447),
    c64::new(-0.9420597397710173, 0.3354451470845317),
    c64::new(-0.9423167458565638, 0.3347224977175813),
    c64::new(-0.9425731976014469, 0.3339996514420093),
    c64::new(-0.9428290948548026, 0.3332766086830482),
    c64::new(-0.9430844374660934, 0.33255336986604445),
    c64::new(-0.9433392252851077, 0.3318299354164613),
    c64::new(-0.9435934581619604, 0.3311063057598765),
    c64::new(-0.9438471359470927, 0.3303824813219828),
    c64::new(-0.9441002584912727, 0.32965846252858744),
    c64::new(-0.9443528256455946, 0.3289342498056124),
    c64::new(-0.9446048372614801, 0.3282098435790927),
    c64::new(-0.9448562931906772, 0.3274852442751781),
    c64::new(-0.9451071932852606, 0.3267604523201318),
    c64::new(-0.9453575373976323, 0.3260354681403302),
    c64::new(-0.9456073253805213, 0.3253102921622632),
    c64::new(-0.9458565570869838, 0.32458492481253237),
    c64::new(-0.9461052323704033, 0.323859366517853),
    c64::new(-0.9463533510844906, 0.3231336177050524),
    c64::new(-0.9466009130832835, 0.32240767880106985),
    c64::new(-0.9468479182211481, 0.32168155023295647),
    c64::new(-0.9470943663527771, 0.3209552324278755),
    c64::new(-0.9473402573331919, 0.3202287258131001),
    c64::new(-0.9475855910177411, 0.3195020308160158),
    c64::new(-0.947830367262101, 0.31877514786411854),
    c64::new(-0.9480745859222762, 0.3180480773850149),
    c64::new(-0.948318246854599, 0.317320819806422),
    c64::new(-0.9485613499157303, 0.31659337555616607),
    c64::new(-0.9488038949626584, 0.3158657450621841),
    c64::new(-0.9490458818527006, 0.31513792875252244),
    c64::new(-0.9492873104435021, 0.31440992705533666),
    c64::new(-0.9495281805930367, 0.3136817403988914),
    c64::new(-0.9497684921596066, 0.3129533692115605),
    c64::new(-0.950008245001843, 0.3122248139218251),
    c64::new(-0.9502474389787052, 0.311496074958276),
    c64::new(-0.9504860739494817, 0.31076715274961153),
    c64::new(-0.9507241497737896, 0.31003804772463783),
    c64::new(-0.950961666311575, 0.309308760312269),
    c64::new(-0.9511986234231132, 0.30857929094152525),
    c64::new(-0.9514350209690083, 0.30784964004153503),
    c64::new(-0.9516708588101939, 0.30711980804153316),
    c64::new(-0.9519061368079323, 0.3063897953708609),
    c64::new(-0.9521408548238158, 0.30565960245896606),
    c64::new(-0.9523750127197658, 0.30492922973540265),
    c64::new(-0.9526086103580332, 0.30419867762982933),
    c64::new(-0.9528416476011986, 0.3034679465720114),
    c64::new(-0.9530741243121722, 0.3027370369918192),
    c64::new(-0.9533060403541939, 0.30200594931922803),
    c64::new(-0.9535373955908332, 0.3012746839843183),
    c64::new(-0.9537681898859902, 0.3005432414172737),
    c64::new(-0.9539984231038945, 0.2998116220483835),
    c64::new(-0.9542280951091056, 0.29907982630804053),
    c64::new(-0.9544572057665135, 0.2983478546267414),
    c64::new(-0.9546857549413383, 0.29761570743508614),
    c64::new(-0.9549137424991304, 0.2968833851637785),
    c64::new(-0.9551411683057707, 0.296150888243624),
    c64::new(-0.9553680322274702, 0.2954182171055321),
    c64::new(-0.9555943341307711, 0.2946853721805144),
    c64::new(-0.9558200738825454, 0.2939523538996846),
    c64::new(-0.9560452513499964, 0.29321916269425896),
    c64::new(-0.956269866400658, 0.2924857989955541),
    c64::new(-0.956493918902395, 0.29175226323498943),
    c64::new(-0.956717408723403, 0.29101855584408515),
    c64::new(-0.9569403357322088, 0.29028467725446233),
    c64::new(-0.9571626997976702, 0.28955062789784297),
    c64::new(-0.9573845007889759, 0.28881640820604976),
    c64::new(-0.9576057385756462, 0.2880820186110043),
    c64::new(-0.9578264130275329, 0.2873474595447296),
    c64::new(-0.9580465240148185, 0.28661273143934785),
    c64::new(-0.9582660714080177, 0.28587783472708056),
    c64::new(-0.9584850550779761, 0.285142769840249),
    c64::new(-0.9587034748958715, 0.28440753721127204),
    c64::new(-0.9589213307332131, 0.2836721372726686),
    c64::new(-0.9591386224618419, 0.28293657045705545),
    c64::new(-0.9593553499539308, 0.28220083719714756),
    c64::new(-0.9595715130819845, 0.2814649379257579),
    c64::new(-0.9597871117188399, 0.28072887307579747),
    c64::new(-0.9600021457376658, 0.27999264308027344),
    c64::new(-0.9602166150119634, 0.2792562483722913),
    c64::new(-0.9604305194155658, 0.2785196893850531),
    c64::new(-0.9606438588226386, 0.27778296655185764),
    c64::new(-0.9608566331076795, 0.27704608030610023),
    c64::new(-0.9610688421455192, 0.2763090310812713),
    c64::new(-0.9612804858113206, 0.2755718193109583),
    c64::new(-0.961491563980579, 0.274834445428844),
    c64::new(-0.9617020765291225, 0.2740969098687064),
    c64::new(-0.9619120233331122, 0.2733592130644186),
    c64::new(-0.9621214042690416, 0.27262135544994925),
    c64::new(-0.9623302192137374, 0.27188333745935994),
    c64::new(-0.9625384680443592, 0.2711451595268081),
    c64::new(-0.9627461506383994, 0.2704068220865449),
    c64::new(-0.9629532668736839, 0.26966832557291504),
    c64::new(-0.9631598166283714, 0.2689296704203576),
    c64::new(-0.9633657997809539, 0.2681908570634034),
    c64::new(-0.9635712162102572, 0.2674518859366778),
    c64::new(-0.9637760657954398, 0.2667127574748985),
    c64::new(-0.9639803484159941, 0.2659734721128756),
    c64::new(-0.9641840639517458, 0.26523403028551173),
    c64::new(-0.9643872122828542, 0.2644944324278019),
    c64::new(-0.9645897932898126, 0.26375467897483157),
    c64::new(-0.9647918068534478, 0.2630147703617791),
    c64::new(-0.9649932528549203, 0.26227470702391364),
    c64::new(-0.9651941311757247, 0.26153448939659546),
    c64::new(-0.9653944416976893, 0.26079411791527585),
    c64::new(-0.9655941843029767, 0.2600535930154954),
    c64::new(-0.9657933588740836, 0.2593129151328864),
    c64::new(-0.9659919652938406, 0.25857208470317045),
    c64::new(-0.9661900034454125, 0.257831102162159),
    c64::new(-0.9663874732122989, 0.25708996794575306),
    c64::new(-0.966584374478333, 0.25634868248994314),
    c64::new(-0.9667807071276833, 0.2556072462308076),
    c64::new(-0.9669764710448521, 0.2548656596045147),
    c64::new(-0.9671716661146765, 0.2541239230473207),
    c64::new(-0.9673662922223285, 0.2533820369955701),
    c64::new(-0.9675603492533144, 0.25264000188569585),
    c64::new(-0.9677538370934754, 0.2518978181542172),
    c64::new(-0.9679467556289878, 0.2511554862377421),
    c64::new(-0.9681391047463623, 0.25041300657296534),
    c64::new(-0.9683308843324452, 0.24967037959666855),
    c64::new(-0.9685220942744174, 0.2489276057457201),
    c64::new(-0.9687127344597947, 0.24818468545707506),
    c64::new(-0.9689028047764289, 0.2474416191677735),
    c64::new(-0.9690923051125061, 0.24669840731494255),
    c64::new(-0.9692812353565485, 0.24595505033579465),
    c64::new(-0.9694695953974131, 0.2452115486676275),
    c64::new(-0.9696573851242923, 0.24446790274782448),
    c64::new(-0.9698446044267148, 0.24372411301385238),
    c64::new(-0.970031253194544, 0.24298017990326404),
    c64::new(-0.9702173313179792, 0.2422361038536961),
    c64::new(-0.9704028386875555, 0.24149188530286933),
    c64::new(-0.9705877751941436, 0.24074752468858837),
    c64::new(-0.9707721407289502, 0.24000302244874175),
    c64::new(-0.970955935183518, 0.23925837902130015),
    c64::new(-0.9711391584497251, 0.23851359484431855),
    c64::new(-0.9713218104197862, 0.23776867035593424),
    c64::new(-0.9715038909862518, 0.23702360599436717),
    c64::new(-0.9716854000420084, 0.23627840219791987),
    c64::new(-0.9718663374802793, 0.23553305940497574),
    c64::new(-0.9720467031946234, 0.2347875780540011),
    c64::new(-0.9722264970789363, 0.23404195858354349),
    c64::new(-0.9724057190274498, 0.23329620143223162),
    c64::new(-0.9725843689347322, 0.23255030703877516),
    c64::new(-0.9727624466956885, 0.23180427584196503),
    c64::new(-0.9729399522055601, 0.2310581082806713),
    c64::new(-0.9731168853599251, 0.23031180479384558),
    c64::new(-0.9732932460546982, 0.22956536582051892),
    c64::new(-0.9734690341861311, 0.2288187917998022),
    c64::new(-0.9736442496508119, 0.22807208317088606),
    c64::new(-0.9738188923456661, 0.2273252403730391),
    c64::new(-0.9739929621679558, 0.22657826384561017),
    c64::new(-0.9741664590152803, 0.22583115402802625),
    c64::new(-0.9743393827855759, 0.22508391135979283),
    c64::new(-0.9745117333771158, 0.22433653628049352),
    c64::new(-0.9746835106885107, 0.22358902922979026),
    c64::new(-0.9748547146187084, 0.22284139064742134),
    c64::new(-0.9750253450669941, 0.22209362097320365),
    c64::new(-0.9751954019329904, 0.22134572064703087),
    c64::new(-0.975364885116657, 0.22059769010887348),
    c64::new(-0.9755337945182913, 0.219849529798779),
    c64::new(-0.9757021300385285, 0.21910124015687002),
    c64::new(-0.975869891578341, 0.2183528216233465),
    c64::new(-0.976037079039039, 0.21760427463848372),
    c64::new(-0.9762036923222706, 0.21685559964263262),
    c64::new(-0.9763697313300211, 0.21610679707621944),
    c64::new(-0.9765351959646144, 0.21535786737974583),
    c64::new(-0.9767000861287117, 0.21460881099378698),
    c64::new(-0.9768644017253126, 0.2138596283589939),
    c64::new(-0.9770281426577544, 0.21311031991609142),
    c64::new(-0.9771913088297123, 0.2123608861058784),
    c64::new(-0.9773539001452, 0.2116113273692279),
    c64::new(-0.9775159165085692, 0.21086164414708508),
    c64::new(-0.9776773578245099, 0.21011183688046978),
    c64::new(-0.9778382239980504, 0.20936190601047425),
    c64::new(-0.9779985149345571, 0.20861185197826349),
    c64::new(-0.978158230539735, 0.20786167522507498),
    c64::new(-0.9783173707196275, 0.20711137619221884),
    c64::new(-0.9784759353806168, 0.20636095532107573),
    c64::new(-0.9786339244294231, 0.20561041305309938),
    c64::new(-0.9787913377731057, 0.20485974982981447),
    c64::new(-0.9789481753190622, 0.20410896609281684),
    c64::new(-0.9791044369750291, 0.20335806228377362),
    c64::new(-0.979260122649082, 0.20260703884442136),
    c64::new(-0.9794152322496348, 0.2018558962165682),
    c64::new(-0.9795697656854405, 0.20110463484209198),
    c64::new(-0.9797237228655912, 0.20035325516294045),
    c64::new(-0.9798771036995176, 0.1996017576211309),
    c64::new(-0.98002990809699, 0.1988501426587504),
    c64::new(-0.9801821359681173, 0.19809841071795378),
    c64::new(-0.980333787223348, 0.19734656224096606),
    c64::new(-0.9804848617734694, 0.19659459767008028),
    c64::new(-0.9806353595296081, 0.19584251744765782),
    c64::new(-0.9807852804032304, 0.19509032201612858),
    c64::new(-0.9809346243061416, 0.19433801181798885),
    c64::new(-0.9810833911504866, 0.19358558729580377),
    c64::new(-0.9812315808487496, 0.19283304889220532),
    c64::new(-0.9813791933137546, 0.19208039704989244),
    c64::new(-0.9815262284586648, 0.19132763221163082),
    c64::new(-0.9816726861969831, 0.19057475482025305),
    c64::new(-0.9818185664425525, 0.18982176531865663),
    c64::new(-0.9819638691095552, 0.18906866414980633),
    c64::new(-0.9821085941125135, 0.18831545175673217),
    c64::new(-0.9822527413662894, 0.18756212858252957),
    c64::new(-0.9823963107860847, 0.18680869507035916),
    c64::new(-0.9825393022874412, 0.18605515166344688),
    c64::new(-0.9826817157862409, 0.18530149880508207),
    c64::new(-0.9828235511987052, 0.1845477369386197),
    c64::new(-0.9829648084413964, 0.18379386650747845),
    c64::new(-0.9831054874312164, 0.1830398879551409),
    c64::new(-0.9832455880854071, 0.1822858017251536),
    c64::new(-0.9833851103215512, 0.1815316082611252),
    c64::new(-0.9835240540575713, 0.18077730800672873),
    c64::new(-0.9836624192117303, 0.18002290140569957),
    c64::new(-0.9838002057026316, 0.17926838890183572),
    c64::new(-0.9839374134492189, 0.1785137709389974),
    c64::new(-0.9840740423707764, 0.17775904796110742),
    c64::new(-0.984210092386929, 0.17700422041214892),
    c64::new(-0.9843455634176419, 0.17624928873616796),
    c64::new(-0.9844804553832209, 0.17549425337727143),
    c64::new(-0.9846147682043126, 0.17473911477962714),
    c64::new(-0.9847485018019042, 0.1739838733874641),
    c64::new(-0.9848816560973237, 0.17322852964507054),
    c64::new(-0.9850142310122398, 0.1724730839967961),
    c64::new(-0.9851462264686621, 0.17171753688705002),
    c64::new(-0.9852776423889412, 0.1709618887603012),
    c64::new(-0.9854084786957684, 0.17020614006107795),
    c64::new(-0.9855387353121761, 0.1694502912339682),
    c64::new(-0.9856684121615376, 0.1686943427236175),
    c64::new(-0.9857975091675674, 0.16793829497473126),
    c64::new(-0.9859260262543211, 0.16718214843207294),
    c64::new(-0.9860539633461954, 0.16642590354046405),
    c64::new(-0.9861813203679282, 0.1656695607447844),
    c64::new(-0.9863080972445986, 0.16491312048997012),
    c64::new(-0.9864342939016271, 0.16415658322101595),
    c64::new(-0.9865599102647754, 0.16339994938297328),
    c64::new(-0.9866849462601467, 0.16264321942095028),
    c64::new(-0.9868094018141855, 0.16188639378011171),
    c64::new(-0.9869332768536777, 0.16112947290567906),
    c64::new(-0.987056571305751, 0.16037245724292842),
    c64::new(-0.9871792850978743, 0.15961534723719314),
    c64::new(-0.9873014181578584, 0.15885814333386145),
    c64::new(-0.9874229704138554, 0.15810084597837692),
    c64::new(-0.9875439417943592, 0.15734345561623855),
    c64::new(-0.9876643322282057, 0.15658597269299865),
    c64::new(-0.9877841416445722, 0.15582839765426537),
    c64::new(-0.9879033699729778, 0.15507073094570056),
    c64::new(-0.9880220171432835, 0.15431297301302008),
    c64::new(-0.9881400830856926, 0.15355512430199333),
    c64::new(-0.9882575677307495, 0.15279718525844369),
    c64::new(-0.9883744710093413, 0.15203915632824622),
    c64::new(-0.9884907928526966, 0.1512810379573303),
    c64::new(-0.9886065331923864, 0.15052283059167743),
    c64::new(-0.9887216919603238, 0.14976453467732145),
    c64::new(-0.9888362690887634, 0.14900615066034875),
    c64::new(-0.988950264510303, 0.14824767898689625),
    c64::new(-0.9890636781578815, 0.1474891201031537),
    c64::new(-0.989176509964781, 0.1467304744553618),
    c64::new(-0.9892887598646252, 0.14597174248981218),
    c64::new(-0.9894004277913804, 0.14521292465284735),
    c64::new(-0.9895115136793552, 0.14445402139086072),
    c64::new(-0.9896220174632008, 0.14369503315029464),
    c64::new(-0.9897319390779106, 0.14293596037764275),
    c64::new(-0.9898412784588205, 0.14217680351944806),
    c64::new(-0.989950035541609, 0.14141756302230296),
    c64::new(-0.990058210262297, 0.14065823933284952),
    c64::new(-0.9901658025572484, 0.1398988328977774),
    c64::new(-0.9902728123631691, 0.13913934416382634),
    c64::new(-0.9903792396171082, 0.13837977357778394),
    c64::new(-0.9904850842564571, 0.137620121586486),
    c64::new(-0.9905903462189501, 0.13686038863681627),
    c64::new(-0.9906950254426646, 0.13610057517570645),
    c64::new(-0.9907991218660204, 0.13534068165013438),
    c64::new(-0.99090263542778, 0.13458070850712628),
    c64::new(-0.9910055660670494, 0.13382065619375474),
    c64::new(-0.9911079137232769, 0.133060525157139),
    c64::new(-0.991209678336254, 0.13230031584444496),
    c64::new(-0.9913108598461154, 0.1315400287028833),
    c64::new(-0.9914114581933385, 0.13077966417971185),
    c64::new(-0.9915114733187439, 0.1300192227222334),
    c64::new(-0.9916109051634954, 0.1292587047777961),
    c64::new(-0.9917097536690995, 0.12849811079379306),
    c64::new(-0.9918080187774064, 0.12773744121766256),
    c64::new(-0.9919057004306093, 0.12697669649688603),
    c64::new(-0.9920027985712445, 0.12621587707899046),
    c64::new(-0.9920993131421918, 0.12545498341154626),
    c64::new(-0.9921952440866739, 0.12469401594216759),
    c64::new(-0.9922905913482573, 0.12393297511851248),
    c64::new(-0.9923853548708517, 0.1231718613882807),
    c64::new(-0.99247953459871, 0.12241067519921635),
    c64::new(-0.9925731304764288, 0.1216494169991056),
    c64::new(-0.992666142448948, 0.12088808723577707),
    c64::new(-0.9927585704615511, 0.12012668635710143),
    c64::new(-0.9928504144598651, 0.11936521481099163),
    c64::new(-0.9929416743898605, 0.11860367304540091),
    c64::new(-0.9930323501978514, 0.11784206150832509),
    c64::new(-0.9931224418304956, 0.11708038064780062),
    c64::new(-0.9932119492347945, 0.11631863091190471),
    c64::new(-0.9933008723580933, 0.11555681274875558),
    c64::new(-0.9933892111480807, 0.11479492660651032),
    c64::new(-0.9934769655527892, 0.11403297293336735),
    c64::new(-0.9935641355205953, 0.11327095217756442),
    c64::new(-0.9936507210002191, 0.11250886478737868),
    c64::new(-0.9937367219407246, 0.1117467112111265),
    c64::new(-0.9938221382915197, 0.11098449189716367),
    c64::new(-0.9939069700023561, 0.11022220729388325),
    c64::new(-0.9939912170233294, 0.1094598578497181),
    c64::new(-0.9940748793048794, 0.10869744401313874),
    c64::new(-0.9941579567977897, 0.10793496623265361),
    c64::new(-0.9942404494531879, 0.10717242495680916),
    c64::new(-0.9943223572225458, 0.10640982063418791),
    c64::new(-0.9944036800576791, 0.10564715371341077),
    c64::new(-0.9944844179107476, 0.10488442464313504),
    c64::new(-0.9945645707342554, 0.10412163387205457),
    c64::new(-0.9946441384810507, 0.10335878184889953),
    c64::new(-0.9947231211043257, 0.10259586902243656),
    c64::new(-0.9948015185576171, 0.10183289584146672),
    c64::new(-0.9948793307948056, 0.10106986275482793),
    c64::new(-0.9949565577701164, 0.10030677021139289),
    c64::new(-0.9950331994381186, 0.09954361866006928),
    c64::new(-0.9951092557537261, 0.09878040854979994),
    c64::new(-0.9951847266721968, 0.09801714032956084),
    c64::new(-0.9952596121491333, 0.09725381444836342),
    c64::new(-0.9953339121404823, 0.09649043135525266),
    c64::new(-0.9954076266025349, 0.09572699149930716),
    c64::new(-0.9954807554919269, 0.09496349532963891),
    c64::new(-0.9955532987656385, 0.09419994329539348),
    c64::new(-0.9956252563809943, 0.09343633584574798),
    c64::new(-0.9956966282956635, 0.09267267342991342),
    c64::new(-0.9957674144676598, 0.09190895649713277),
    c64::new(-0.9958376148553416, 0.09114518549668096),
    c64::new(-0.9959072294174117, 0.0903813608778653),
    c64::new(-0.9959762581129178, 0.0896174830900232),
    c64::new(-0.996044700901252, 0.08885355258252475),
    c64::new(-0.9961125577421511, 0.08808956980477058),
    c64::new(-0.996179828595697, 0.08732553520619206),
    c64::new(-0.9962465134223155, 0.08656144923625109),
    c64::new(-0.996312612182778, 0.08579731234444016),
    c64::new(-0.9963781248382002, 0.08503312498028047),
    c64::new(-0.9964430513500426, 0.08426888759332418),
    c64::new(-0.9965073916801108, 0.08350460063315247),
    c64::new(-0.9965711457905548, 0.08274026454937565),
    c64::new(-0.9966343136438699, 0.08197587979163339),
    c64::new(-0.9966968952028961, 0.08121144680959266),
    c64::new(-0.996758890430818, 0.08044696605295017),
    c64::new(-0.9968202992911657, 0.0796824379714302),
    c64::new(-0.9968811217478138, 0.07891786301478494),
    c64::new(-0.9969413577649822, 0.07815324163279415),
    c64::new(-0.9970010073072353, 0.07738857427526533),
    c64::new(-0.997060070339483, 0.07662386139203169),
    c64::new(-0.99711854682698, 0.07585910343295456),
    c64::new(-0.9971764367353261, 0.07509430084792135),
    c64::new(-0.9972337400304663, 0.07432945408684571),
    c64::new(-0.9972904566786902, 0.07356456359966775),
    c64::new(-0.9973465866466331, 0.07279962983635191),
    c64::new(-0.9974021299012753, 0.07203465324688949),
    c64::new(-0.9974570864099419, 0.07126963428129648),
    c64::new(-0.9975114561403035, 0.07050457338961386),
    c64::new(-0.9975652390603758, 0.06973947102190722),
    c64::new(-0.9976184351385196, 0.06897432762826702),
    c64::new(-0.997671044343441, 0.06820914365880652),
    c64::new(-0.9977230666441916, 0.06744391956366418),
    c64::new(-0.9977745020101678, 0.0666786557930016),
    c64::new(-0.9978253504111116, 0.06591335279700376),
    c64::new(-0.9978756118171102, 0.06514801102587915),
    c64::new(-0.997925286198596, 0.0643826309298577),
    c64::new(-0.997974373526347, 0.06361721295919326),
    c64::new(-0.9980228737714862, 0.06285175756416149),
    c64::new(-0.9980707869054823, 0.06208626519506009),
    c64::new(-0.9981181129001492, 0.061320736302208495),
    c64::new(-0.9981648517276462, 0.060555171335948066),
    c64::new(-0.9982110033604782, 0.05978957074664007),
    c64::new(-0.9982565677714952, 0.05902393498466805),
    c64::new(-0.9983015449338929, 0.058258264500435794),
    c64::new(-0.9983459348212124, 0.057492559744367525),
    c64::new(-0.9983897374073402, 0.05672682116690807),
    c64::new(-0.9984329526665084, 0.055961049218520804),
    c64::new(-0.9984755805732948, 0.055195244349690094),
    c64::new(-0.9985176211026221, 0.05442940701091921),
    c64::new(-0.9985590742297593, 0.05366353765273052),
    c64::new(-0.9985999399303204, 0.05289763672566524),
    c64::new(-0.9986402181802652, 0.0521317046802836),
    c64::new(-0.9986799089558991, 0.051365741967162794),
    c64::new(-0.9987190122338729, 0.0505997490368994),
    c64::new(-0.9987575279911833, 0.04983372634010732),
    c64::new(-0.9987954562051724, 0.04906767432741797),
    c64::new(-0.998832796853528, 0.048301593449480464),
    c64::new(-0.9988695499142836, 0.047535484156959545),
    c64::new(-0.9989057153658183, 0.04676934690053802),
    c64::new(-0.9989412931868569, 0.046003182130914706),
    c64::new(-0.9989762833564698, 0.04523699029880459),
    c64::new(-0.9990106858540734, 0.044470771854938584),
    c64::new(-0.9990445006594293, 0.043704527250063706),
    c64::new(-0.9990777277526454, 0.04293825693494102),
    c64::new(-0.9991103671141749, 0.042171961360348065),
    c64::new(-0.9991424187248169, 0.04140564097707678),
    c64::new(-0.9991738825657164, 0.040639296235933695),
    c64::new(-0.9992047586183639, 0.03987292758774013),
    c64::new(-0.9992350468645959, 0.03910653548333013),
    c64::new(-0.9992647472865944, 0.038340120373552854),
    c64::new(-0.9992938598668877, 0.03757368270927058),
    c64::new(-0.9993223845883495, 0.03680722294135883),
    c64::new(-0.9993503214341994, 0.036040741520706146),
    c64::new(-0.9993776703880028, 0.03527423889821423),
    c64::new(-0.9994044314336713, 0.03450771552479595),
    c64::new(-0.9994306045554617, 0.033741171851377705),
    c64::new(-0.9994561897379773, 0.03297460832889738),
    c64::new(-0.999481186966167, 0.032208025408304544),
    c64::new(-0.9995055962253253, 0.03144142354056063),
    c64::new(-0.9995294175010931, 0.030674803176636865),
    c64::new(-0.999552650779457, 0.029908164767516714),
    c64::new(-0.9995752960467492, 0.029141508764193802),
    c64::new(-0.9995973532896484, 0.028374835617672095),
    c64::new(-0.9996188224951786, 0.027608145778965656),
    c64::new(-0.9996397036507102, 0.02684143969909881),
    c64::new(-0.9996599967439592, 0.0260747178291041),
    c64::new(-0.9996797017629879, 0.02530798062002469),
    c64::new(-0.9996988186962042, 0.024541228522912326),
    c64::new(-0.9997173475323622, 0.023774461988827513),
    c64::new(-0.9997352882605617, 0.02300768146883969),
    c64::new(-0.9997526408702488, 0.0222408874140252),
    c64::new(-0.9997694053512153, 0.021474080275469667),
    c64::new(-0.9997855816935992, 0.020707260504265974),
    c64::new(-0.9998011698878843, 0.019940428551514438),
    c64::new(-0.9998161699249004, 0.01917358486832254),
    c64::new(-0.9998305817958234, 0.0184067299058051),
    c64::new(-0.9998444054921752, 0.017639864115082254),
    c64::new(-0.9998576410058239, 0.016872987947281832),
    c64::new(-0.999870288328983, 0.016106101853537325),
    c64::new(-0.9998823474542126, 0.01533920628498806),
    c64::new(-0.9998938183744185, 0.014572301692779389),
    c64::new(-0.9999047010828528, 0.013805388528060633),
    c64::new(-0.9999149955731135, 0.013038467241987496),
    c64::new(-0.9999247018391445, 0.012271538285720007),
    c64::new(-0.999933819875236, 0.011504602110422715),
    c64::new(-0.9999423496760239, 0.01073765916726441),
    c64::new(-0.9999502912364905, 0.009970709907418314),
    c64::new(-0.9999576445519639, 0.009203754782060022),
    c64::new(-0.9999644096181183, 0.008436794242369922),
    c64::new(-0.9999705864309741, 0.007669828739531138),
    c64::new(-0.9999761749868976, 0.006902858724729716),
    c64::new(-0.9999811752826011, 0.006135884649154799),
    c64::new(-0.9999855873151432, 0.005368906963996585),
    c64::new(-0.9999894110819284, 0.004601926120448733),
    c64::new(-0.9999926465807072, 0.003834942569706309),
    c64::new(-0.9999952938095762, 0.003067956762965977),
    c64::new(-0.9999973527669782, 0.002300969151425725),
    c64::new(-0.9999988234517019, 0.0015339801862850493),
    c64::new(-0.9999997058628822, 0.0007669903187429077),
];

static INVERSE_TWIDDLES_4096: [c64; 4096] = [
    c64::new(0.000244140625, -0.0),
    c64::new(0.00024414055318918023, -1.872534957867931e-07),
    c64::new(0.00024414033775676315, -3.745068814171791e-07),
    c64::new(0.00024413997870287554, -5.617600467348157e-07),
    c64::new(0.00024413947602772856, -7.490128815834903e-07),
    c64::new(0.00024413882973161797, -9.362652758071845e-07),
    c64::new(0.00024413803981492393, -1.1235171192501393e-06),
    c64::new(0.00024413710627811113, -1.3107683017569196e-06),
    c64::new(0.00024413602912172879, -1.4980187131724793e-06),
    c64::new(0.00024413480834641055, -1.6852682433422255e-06),
    c64::new(0.00024413344395287455, -1.8725167821120842e-06),
    c64::new(0.0002441319359419234, -2.059764219328564e-06),
    c64::new(0.0002441302843144443, -2.247010444838823e-06),
    c64::new(0.0002441284890714088, -2.43425534849073e-06),
    c64::new(0.00024412655021387303, -2.6214988201329323e-06),
    c64::new(0.00024412446774297754, -2.8087407496149203e-06),
    c64::new(0.0002441222416599474, -2.995981026787091e-06),
    c64::new(0.00024411987196609216, -3.183219541500814e-06),
    c64::new(0.00024411735866280588, -3.370456183608494e-06),
    c64::new(0.000244114701751567, -3.5576908429636388e-06),
    c64::new(0.0002441119012339386, -3.744923409420923e-06),
    c64::new(0.0002441089571115681, -3.932153772836252e-06),
    c64::new(0.00024410586938618747, -4.119381823066824e-06),
    c64::new(0.0002441026380596131, -4.306607449971204e-06),
    c64::new(0.00024409926313374595, -4.49383054340938e-06),
    c64::new(0.0002440957446105714, -4.681050993242828e-06),
    c64::new(0.00024409208249215924, -4.86826868933458e-06),
    c64::new(0.00024408827678066387, -5.055483521549291e-06),
    c64::new(0.00024408432747832404, -5.242695379753298e-06),
    c64::new(0.0002440802345874631, -5.429904153814688e-06),
    c64::new(0.0002440759981104887, -5.6171097336033615e-06),
    c64::new(0.0002440716180498931, -5.804312008991102e-06),
    c64::new(0.000244067094408253, -5.991510869851633e-06),
    c64::new(0.00024406242718822947, -6.178706206060686e-06),
    c64::new(0.00024405761639256817, -6.3658979074960695e-06),
    c64::new(0.00024405266202409917, -6.553085864037727e-06),
    c64::new(0.00024404756408573697, -6.7402699655678076e-06),
    c64::new(0.00024404232258048056, -6.927450101970727e-06),
    c64::new(0.00024403693751141338, -7.114626163133233e-06),
    c64::new(0.00024403140888170337, -7.301798038944471e-06),
    c64::new(0.00024402573669460282, -7.488965619296051e-06),
    c64::new(0.00024401992095344856, -7.676128794082105e-06),
    c64::new(0.00024401396166166185, -7.863287453199362e-06),
    c64::new(0.00024400785882274837, -8.050441486547201e-06),
    c64::new(0.00024400161244029827, -8.23759078402773e-06),
    c64::new(0.00024399522251798616, -8.424735235545837e-06),
    c64::new(0.000243988689059571, -8.611874731009264e-06),
    c64::new(0.00024398201206889635, -8.79900916032867e-06),
    c64::new(0.00024397519154989003, -8.986138413417684e-06),
    c64::new(0.0002439682275065644, -9.173262380192992e-06),
    c64::new(0.00024396111994301622, -9.360380950574388e-06),
    c64::new(0.00024395386886342672, -9.547494014484836e-06),
    c64::new(0.0002439464742720615, -9.73460146185054e-06),
    c64::new(0.0002439389361732706, -9.92170318260101e-06),
    c64::new(0.0002439312545714885, -1.0108799066669126e-05),
    c64::new(0.0002439234294712341, -1.0295889003991198e-05),
    c64::new(0.00024391546087711068, -1.0482972884507036e-05),
    c64::new(0.00024390734879380598, -1.0670050598160015e-05),
    c64::new(0.00024389909322609213, -1.0857122034897136e-05),
    c64::new(0.00024389069417882564, -1.104418708466909e-05),
    c64::new(0.00024388215165694748, -1.1231245637430328e-05),
    c64::new(0.00024387346566548298, -1.1418297583139127e-05),
    c64::new(0.00024386463620954189, -1.1605342811757642e-05),
    c64::new(0.00024385566329431836, -1.1792381213251988e-05),
    c64::new(0.00024384654692509092, -1.1979412677592289e-05),
    c64::new(0.0002438372871072225, -1.2166437094752753e-05),
    c64::new(0.00024382788384616039, -1.2353454354711739e-05),
    c64::new(0.0002438183371474363, -1.2540464347451805e-05),
    c64::new(0.00024380864701666633, -1.2727466962959796e-05),
    c64::new(0.00024379881345955087, -1.2914462091226886e-05),
    c64::new(0.00024378883648187483, -1.3101449622248662e-05),
    c64::new(0.00024377871608950738, -1.3288429446025179e-05),
    c64::new(0.00024376845228840204, -1.347540145256102e-05),
    c64::new(0.0002437580450845968, -1.3662365531865373e-05),
    c64::new(0.0002437474944842139, -1.3849321573952087e-05),
    c64::new(0.00024373680049346005, -1.4036269468839738e-05),
    c64::new(0.0002437259631186262, -1.4223209106551697e-05),
    c64::new(0.0002437149823660877, -1.4410140377116194e-05),
    c64::new(0.00024370385824230425, -1.4597063170566374e-05),
    c64::new(0.00024369259075381988, -1.4783977376940378e-05),
    c64::new(0.00024368117990726298, -1.4970882886281391e-05),
    c64::new(0.00024366962570934628, -1.5157779588637717e-05),
    c64::new(0.00024365792816686676, -1.5344667374062843e-05),
    c64::new(0.0002436460872867058, -1.5531546132615504e-05),
    c64::new(0.0002436341030758291, -1.5718415754359733e-05),
    c64::new(0.00024362197554128666, -1.590527612936495e-05),
    c64::new(0.0002436097046902128, -1.6092127147706007e-05),
    c64::new(0.00024359729052982613, -1.627896869946327e-05),
    c64::new(0.0002435847330674296, -1.646580067472267e-05),
    c64::new(0.0002435720323104104, -1.6652622963575764e-05),
    c64::new(0.00024355918826624013, -1.683943545611981e-05),
    c64::new(0.00024354620094247455, -1.702623804245784e-05),
    c64::new(0.00024353307034675377, -1.7213030612698695e-05),
    c64::new(0.00024351979648680222, -1.739981305695713e-05),
    c64::new(0.00024350637937042854, -1.758658526535384e-05),
    c64::new(0.0002434928190055257, -1.7773347128015545e-05),
    c64::new(0.00024347911540007085, -1.7960098535075055e-05),
    c64::new(0.00024346526856212556, -1.8146839376671327e-05),
    c64::new(0.0002434512784998355, -1.8333569542949537e-05),
    c64::new(0.00024343714522143066, -1.8520288924061144e-05),
    c64::new(0.00024342286873522533, -1.870699741016394e-05),
    c64::new(0.000243408449049618, -1.889369489142213e-05),
    c64::new(0.00024339388617309135, -1.9080381258006404e-05),
    c64::new(0.00024337918011421237, -1.926705640009398e-05),
    c64::new(0.00024336433088163224, -1.9453720207868683e-05),
    c64::new(0.00024334933848408643, -1.9640372571521e-05),
    c64::new(0.00024333420293039455, -1.9827013381248155e-05),
    c64::new(0.00024331892422946042, -2.0013642527254167e-05),
    c64::new(0.00024330350239027218, -2.0200259899749925e-05),
    c64::new(0.00024328793742190206, -2.038686538895323e-05),
    c64::new(0.0002432722293335065, -2.0573458885088885e-05),
    c64::new(0.00024325637813432622, -2.076004027838874e-05),
    c64::new(0.00024324038383368604, -2.094660945909177e-05),
    c64::new(0.000243224246440995, -2.1133166317444133e-05),
    c64::new(0.00024320796596574633, -2.1319710743699233e-05),
    c64::new(0.00024319154241751737, -2.15062426281178e-05),
    c64::new(0.00024317497580596972, -2.169276186096792e-05),
    c64::new(0.00024315826614084907, -2.1879268332525136e-05),
    c64::new(0.00024314141343198528, -2.2065761933072506e-05),
    c64::new(0.00024312441768929239, -2.2252242552900636e-05),
    c64::new(0.0002431072789227685, -2.2438710082307794e-05),
    c64::new(0.00024308999714249598, -2.262516441159993e-05),
    c64::new(0.0002430725723586412, -2.2811605431090768e-05),
    c64::new(0.0002430550045814547, -2.2998033031101856e-05),
    c64::new(0.00024303729382127123, -2.3184447101962644e-05),
    c64::new(0.0002430194400885095, -2.3370847534010538e-05),
    c64::new(0.00024300144339367243, -2.3557234217590965e-05),
    c64::new(0.00024298330374734702, -2.374360704305744e-05),
    c64::new(0.00024296502116020433, -2.3929965900771632e-05),
    c64::new(0.00024294659564299954, -2.4116310681103423e-05),
    c64::new(0.00024292802720657193, -2.4302641274430986e-05),
    c64::new(0.00024290931586184482, -2.4488957571140836e-05),
    c64::new(0.0002428904616198256, -2.4675259461627886e-05),
    c64::new(0.00024287146449160574, -2.486154683629554e-05),
    c64::new(0.00024285232448836077, -2.5047819585555733e-05),
    c64::new(0.00024283304162135027, -2.5234077599829007e-05),
    c64::new(0.00024281361590191783, -2.5420320769544577e-05),
    c64::new(0.0002427940473414911, -2.5606548985140373e-05),
    c64::new(0.0002427743359515818, -2.579276213706314e-05),
    c64::new(0.0002427544817437856, -2.5978960115768476e-05),
    c64::new(0.0002427344847297822, -2.616514281172091e-05),
    c64::new(0.00024271434492133538, -2.635131011539396e-05),
    c64::new(0.00024269406233029281, -2.6537461917270194e-05),
    c64::new(0.00024267363696858627, -2.6723598107841305e-05),
    c64::new(0.00024265306884823146, -2.690971857760817e-05),
    c64::new(0.00024263235798132804, -2.7095823217080906e-05),
    c64::new(0.00024261150438005972, -2.7281911916778952e-05),
    c64::new(0.00024259050805669412, -2.7467984567231126e-05),
    c64::new(0.00024256936902358284, -2.765404105897567e-05),
    c64::new(0.00024254808729316142, -2.784008128256035e-05),
    c64::new(0.00024252666287794938, -2.80261051285425e-05),
    c64::new(0.00024250509579055012, -2.8212112487489077e-05),
    c64::new(0.000242483386043651, -2.8398103249976746e-05),
    c64::new(0.00024246153365002334, -2.858407730659194e-05),
    c64::new(0.00024243953862252232, -2.8770034547930903e-05),
    c64::new(0.00024241740097408703, -2.8955974864599785e-05),
    c64::new(0.0002423951207177405, -2.9141898147214686e-05),
    c64::new(0.00024237269786658963, -2.9327804286401736e-05),
    c64::new(0.0002423501324338252, -2.951369317279714e-05),
    c64::new(0.00024232742443272188, -2.969956469704725e-05),
    c64::new(0.00024230457387663818, -2.988541874980864e-05),
    c64::new(0.00024228158077901652, -3.0071255221748165e-05),
    c64::new(0.00024225844515338315, -3.0257074003543008e-05),
    c64::new(0.00024223516701334813, -3.044287498588077e-05),
    c64::new(0.00024221174637260542, -3.062865805945953e-05),
    c64::new(0.00024218818324493274, -3.081442311498788e-05),
    c64::new(0.00024216447764419173, -3.1000170043185026e-05),
    c64::new(0.00024214062958432774, -3.118589873478084e-05),
    c64::new(0.00024211663907937, -3.137160908051591e-05),
    c64::new(0.0002420925061434315, -3.1557300971141635e-05),
    c64::new(0.00024206823079070896, -3.174297429742025e-05),
    c64::new(0.00024204381303548304, -3.192862895012493e-05),
    c64::new(0.00024201925289211803, -3.211426482003982e-05),
    c64::new(0.00024199455037506203, -3.229988179796012e-05),
    c64::new(0.0002419697054988469, -3.2485479774692154e-05),
    c64::new(0.00024194471827808823, -3.26710586410534e-05),
    c64::new(0.00024191958872748535, -3.28566182878726e-05),
    c64::new(0.00024189431686182138, -3.30421586059898e-05),
    c64::new(0.00024186890269596305, -3.32276794862564e-05),
    c64::new(0.00024184334624486088, -3.341318081953525e-05),
    c64::new(0.0002418176475235491, -3.359866249670069e-05),
    c64::new(0.00024179180654714555, -3.3784124408638644e-05),
    c64::new(0.00024176582333085183, -3.396956644624663e-05),
    c64::new(0.00024173969788995322, -3.415498850043389e-05),
    c64::new(0.00024171343023981863, -3.434039046212139e-05),
    c64::new(0.00024168702039590063, -3.452577222224195e-05),
    c64::new(0.00024166046837373548, -3.471113367174024e-05),
    c64::new(0.000241633774188943, -3.489647470157292e-05),
    c64::new(0.00024160693785722678, -3.508179520270861e-05),
    c64::new(0.00024157995939437383, -3.5267095066128045e-05),
    c64::new(0.00024155283881625497, -3.545237418282409e-05),
    c64::new(0.0002415255761388245, -3.563763244380181e-05),
    c64::new(0.00024149817137812036, -3.582286974007855e-05),
    c64::new(0.00024147062455026405, -3.600808596268398e-05),
    c64::new(0.0002414429356714607, -3.6193281002660164e-05),
    c64::new(0.0002414151047579989, -3.637845475106163e-05),
    c64::new(0.00024138713182625092, -3.6563607098955447e-05),
    c64::new(0.00024135901689267247, -3.674873793742124e-05),
    c64::new(0.00024133075997380288, -3.6933847157551324e-05),
    c64::new(0.00024130236108626496, -3.7118934650450696e-05),
    c64::new(0.000241273820246765, -3.730400030723717e-05),
    c64::new(0.0002412451374720929, -3.748904401904137e-05),
    c64::new(0.00024121631277912195, -3.767406567700686e-05),
    c64::new(0.00024118734618480903, -3.785906517229016e-05),
    c64::new(0.00024115823770619438, -3.804404239606085e-05),
    c64::new(0.00024112898736040178, -3.822899723950157e-05),
    c64::new(0.00024109959516463848, -3.8413929593808166e-05),
    c64::new(0.00024107006113619517, -3.859883935018969e-05),
    c64::new(0.0002410403852924459, -3.878372639986852e-05),
    c64::new(0.00024101056765084823, -3.896859063408034e-05),
    c64::new(0.0002409806082289431, -3.915343194407429e-05),
    c64::new(0.0002409505070443549, -3.933825022111299e-05),
    c64::new(0.0002409202641147914, -3.9523045356472614e-05),
    c64::new(0.00024088987945804363, -3.9707817241442946e-05),
    c64::new(0.00024085935309198618, -3.989256576732745e-05),
    c64::new(0.00024082868503457695, -4.007729082544331e-05),
    c64::new(0.0002407978753038571, -4.026199230712156e-05),
    c64::new(0.00024076692391795124, -4.044667010370706e-05),
    c64::new(0.00024073583089506725, -4.063132410655862e-05),
    c64::new(0.00024070459625349637, -4.0815954207049057e-05),
    c64::new(0.00024067322001161315, -4.100056029656523e-05),
    c64::new(0.00024064170218787538, -4.118514226650814e-05),
    c64::new(0.00024061004280082423, -4.136970000829296e-05),
    c64::new(0.0002405782418690841, -4.155423341334914e-05),
    c64::new(0.0002405462994113626, -4.1738742373120414e-05),
    c64::new(0.00024051421544645074, -4.192322677906493e-05),
    c64::new(0.00024048198999322262, -4.210768652265526e-05),
    c64::new(0.00024044962307063567, -4.22921214953785e-05),
    c64::new(0.00024041711469773052, -4.2476531588736285e-05),
    c64::new(0.000240384464893631, -4.266091669424492e-05),
    c64::new(0.00024035167367754417, -4.2845276703435407e-05),
    c64::new(0.00024031874106876023, -4.302961150785349e-05),
    c64::new(0.0002402856670866526, -4.321392099905975e-05),
    c64::new(0.00024025245175067784, -4.339820506862968e-05),
    c64::new(0.00024021909508037571, -4.358246360815369e-05),
    c64::new(0.00024018559709536904, -4.376669650923724e-05),
    c64::new(0.00024015195781536383, -4.395090366350086e-05),
    c64::new(0.00024011817726014923, -4.413508496258022e-05),
    c64::new(0.00024008425544959746, -4.431924029812621e-05),
    c64::new(0.00024005019240366384, -4.4503369561805004e-05),
    c64::new(0.0002400159881423868, -4.4687472645298084e-05),
    c64::new(0.0002399816426858878, -4.4871549440302356e-05),
    c64::new(0.0002399471560543714, -4.505559983853018e-05),
    c64::new(0.0002399125282681252, -4.523962373170945e-05),
    c64::new(0.00023987775934751983, -4.542362101158365e-05),
    c64::new(0.00023984284931300896, -4.560759156991193e-05),
    c64::new(0.00023980779818512924, -4.579153529846914e-05),
    c64::new(0.0002397726059845004, -4.597545208904593e-05),
    c64::new(0.000239737272731825, -4.615934183344878e-05),
    c64::new(0.0002397017984478888, -4.63432044235001e-05),
    c64::new(0.00023966618315356033, -4.6527039751038267e-05),
    c64::new(0.0002396304268697912, -4.67108477079177e-05),
    c64::new(0.00023959452961761586, -4.6894628186008896e-05),
    c64::new(0.0002395584914181518, -4.707838107719854e-05),
    c64::new(0.0002395223122925993, -4.726210627338955e-05),
    c64::new(0.0002394859922622416, -4.744580366650112e-05),
    c64::new(0.00023944953134844493, -4.762947314846881e-05),
    c64::new(0.00023941292957265823, -4.781311461124459e-05),
    c64::new(0.00023937618695641342, -4.799672794679693e-05),
    c64::new(0.00023933930352132519, -4.818031304711082e-05),
    c64::new(0.00023930227928909117, -4.836386980418788e-05),
    c64::new(0.00023926511428149172, -4.854739811004641e-05),
    c64::new(0.00023922780852039005, -4.873089785672143e-05),
    c64::new(0.00023919036202773222, -4.891436893626476e-05),
    c64::new(0.000239152774825547, -4.909781124074509e-05),
    c64::new(0.000239115046935946, -4.928122466224806e-05),
    c64::new(0.00023907717838112354, -4.946460909287625e-05),
    c64::new(0.00023903916918335675, -4.9647964424749345e-05),
    c64::new(0.00023900101936500542, -4.983129055000412e-05),
    c64::new(0.00023896272894851213, -5.0014587360794536e-05),
    c64::new(0.00023892429795640215, -5.0197854749291806e-05),
    c64::new(0.0002388857264112834, -5.038109260768445e-05),
    c64::new(0.0002388470143358466, -5.056430082817836e-05),
    c64::new(0.000238808161752865, -5.074747930299684e-05),
    c64::new(0.00023876916868519461, -5.0930627924380734e-05),
    c64::new(0.00023873003515577403, -5.111374658458842e-05),
    c64::new(0.0002386907611876245, -5.12968351758959e-05),
    c64::new(0.00023865134680384992, -5.147989359059689e-05),
    c64::new(0.00023861179202763674, -5.166292172100282e-05),
    c64::new(0.00023857209688225397, -5.184591945944297e-05),
    c64::new(0.00023853226139105332, -5.202888669826449e-05),
    c64::new(0.0002384922855774689, -5.221182332983246e-05),
    c64::new(0.00023845216946501754, -5.239472924652997e-05),
    c64::new(0.00023841191307729845, -5.257760434075819e-05),
    c64::new(0.00023837151643799344, -5.2760448504936406e-05),
    c64::new(0.00023833097957086683, -5.2943261631502105e-05),
    c64::new(0.0002382903024997654, -5.3126043612911045e-05),
    c64::new(0.00023824948524861842, -5.3308794341637286e-05),
    c64::new(0.00023820852784143764, -5.349151371017329e-05),
    c64::new(0.00023816743030231723, -5.3674201611029956e-05),
    c64::new(0.00023812619265543383, -5.3856857936736696e-05),
    c64::new(0.00023808481492504648, -5.403948257984151e-05),
    c64::new(0.0002380432971354966, -5.422207543291101e-05),
    c64::new(0.0002380016393112081, -5.4404636388530546e-05),
    c64::new(0.00023795984147668717, -5.4587165339304196e-05),
    c64::new(0.0002379179036565224, -5.4769662177854884e-05),
    c64::new(0.00023787582587538473, -5.495212679682442e-05),
    c64::new(0.00023783360815802742, -5.513455908887358e-05),
    c64::new(0.0002377912505292861, -5.531695894668213e-05),
    c64::new(0.00023774875301407864, -5.549932626294894e-05),
    c64::new(0.00023770611563740527, -5.5681660930392024e-05),
    c64::new(0.0002376633384243484, -5.586396284174859e-05),
    c64::new(0.00023762042140007281, -5.6046231889775114e-05),
    c64::new(0.00023757736458982547, -5.622846796724742e-05),
    c64::new(0.0002375341680189356, -5.641067096696072e-05),
    c64::new(0.0002374908317128146, -5.659284078172968e-05),
    c64::new(0.0002374473556969561, -5.677497730438849e-05),
    c64::new(0.00023740373999693598, -5.695708042779092e-05),
    c64::new(0.00023735998463841218, -5.713915004481041e-05),
    c64::new(0.00023731608964712488, -5.732118604834008e-05),
    c64::new(0.00023727205504889634, -5.7503188331292843e-05),
    c64::new(0.000237227880869631, -5.768515678660146e-05),
    c64::new(0.00023718356713531538, -5.7867091307218554e-05),
    c64::new(0.0002371391138720181, -5.8048991786116745e-05),
    c64::new(0.00023709452110588992, -5.8230858116288676e-05),
    c64::new(0.00023704978886316357, -5.8412690190747066e-05),
    c64::new(0.0002370049171701539, -5.859448790252478e-05),
    c64::new(0.00023695990605325772, -5.877625114467491e-05),
    c64::new(0.00023691475553895398, -5.895797981027083e-05),
    c64::new(0.0002368694656538035, -5.913967379240625e-05),
    c64::new(0.0002368240364244492, -5.932133298419528e-05),
    c64::new(0.00023677846787761593, -5.95029572787725e-05),
    c64::new(0.00023673276004011046, -5.9684546569293006e-05),
    c64::new(0.00023668691293882155, -5.9866100748932504e-05),
    c64::new(0.00023664092660071986, -6.004761971088735e-05),
    c64::new(0.00023659480105285796, -6.022910334837461e-05),
    c64::new(0.00023654853632237033, -6.0410551554632146e-05),
    c64::new(0.00023650213243647333, -6.059196422291865e-05),
    c64::new(0.00023645558942246515, -6.077334124651371e-05),
    c64::new(0.00023640890730772588, -6.095468251871791e-05),
    c64::new(0.0002363620861197174, -6.113598793285284e-05),
    c64::new(0.00023631512588598335, -6.131725738226121e-05),
    c64::new(0.0002362680266341493, -6.149849076030688e-05),
    c64::new(0.00023622078839192248, -6.167968796037488e-05),
    c64::new(0.00023617341118709192, -6.186084887587162e-05),
    c64::new(0.00023612589504752848, -6.204197340022476e-05),
    c64::new(0.0002360782400011846, -6.222306142688344e-05),
    c64::new(0.00023603044607609455, -6.240411284931821e-05),
    c64::new(0.0002359825133003743, -6.258512756102122e-05),
    c64::new(0.0002359344417022214, -6.276610545550613e-05),
    c64::new(0.00023588623130991516, -6.294704642630835e-05),
    c64::new(0.00023583788215181655, -6.312795036698495e-05),
    c64::new(0.0002357893942563681, -6.33088171711148e-05),
    c64::new(0.00023574076765209395, -6.348964673229863e-05),
    c64::new(0.00023569200236759995, -6.367043894415906e-05),
    c64::new(0.00023564309843157342, -6.38511937003407e-05),
    c64::new(0.00023559405587278328, -6.403191089451015e-05),
    c64::new(0.00023554487472008005, -6.42125904203562e-05),
    c64::new(0.0002354955550023957, -6.439323217158968e-05),
    c64::new(0.00023544609674874372, -6.457383604194376e-05),
    c64::new(0.0002353964999882192, -6.475440192517378e-05),
    c64::new(0.00023534676474999856, -6.493492971505752e-05),
    c64::new(0.0002352968910633398, -6.511541930539511e-05),
    c64::new(0.00023524687895758235, -6.529587059000919e-05),
    c64::new(0.00023519672846214698, -6.547628346274492e-05),
    c64::new(0.00023514643960653598, -6.565665781747003e-05),
    c64::new(0.00023509601242033298, -6.583699354807497e-05),
    c64::new(0.00023504544693320298, -6.601729054847286e-05),
    c64::new(0.00023499474317489237, -6.619754871259961e-05),
    c64::new(0.00023494390117522886, -6.6377767934414e-05),
    c64::new(0.00023489292096412148, -6.65579481078977e-05),
    c64::new(0.0002348418025715606, -6.673808912705534e-05),
    c64::new(0.0002347905460276178, -6.691819088591464e-05),
    c64::new(0.00023473915136244605, -6.709825327852635e-05),
    c64::new(0.00023468761860627945, -6.727827619896439e-05),
    c64::new(0.00023463594778943344, -6.745825954132595e-05),
    c64::new(0.0002345841389423046, -6.763820319973142e-05),
    c64::new(0.00023453219209537075, -6.781810706832463e-05),
    c64::new(0.00023448010727919087, -6.799797104127272e-05),
    c64::new(0.00023442788452440513, -6.81777950127664e-05),
    c64::new(0.00023437552386173485, -6.835757887701983e-05),
    c64::new(0.0002343230253219824, -6.85373225282708e-05),
    c64::new(0.00023427038893603138, -6.871702586078075e-05),
    c64::new(0.00023421761473484638, -6.889668876883485e-05),
    c64::new(0.00023416470274947312, -6.907631114674204e-05),
    c64::new(0.00023411165301103837, -6.925589288883507e-05),
    c64::new(0.0002340584655507499, -6.943543388947067e-05),
    c64::new(0.00023400514039989654, -6.961493404302946e-05),
    c64::new(0.00023395167758984806, -6.979439324391617e-05),
    c64::new(0.00023389807715205532, -6.997381138655952e-05),
    c64::new(0.00023384433911805003, -7.015318836541248e-05),
    c64::new(0.0002337904635194449, -7.033252407495218e-05),
    c64::new(0.00023373645038793356, -7.051181840968005e-05),
    c64::new(0.00023368229975529055, -7.069107126412183e-05),
    c64::new(0.0002336280116533713, -7.087028253282772e-05),
    c64::new(0.00023357358611411207, -7.104945211037234e-05),
    c64::new(0.00023351902316953005, -7.12285798913548e-05),
    c64::new(0.00023346432285172315, -7.14076657703989e-05),
    c64::new(0.00023340948519287022, -7.158670964215299e-05),
    c64::new(0.00023335451022523081, -7.17657114012902e-05),
    c64::new(0.0002332993979811453, -7.194467094250838e-05),
    c64::new(0.00023324414849303475, -7.212358816053028e-05),
    c64::new(0.00023318876179340104, -7.230246295010346e-05),
    c64::new(0.0002331332379148268, -7.248129520600055e-05),
    c64::new(0.00023307757688997518, -7.266008482301909e-05),
    c64::new(0.0002330217787515902, -7.283883169598179e-05),
    c64::new(0.0002329658435324965, -7.301753571973644e-05),
    c64::new(0.00023290977126559924, -7.319619678915609e-05),
    c64::new(0.00023285356198388436, -7.337481479913903e-05),
    c64::new(0.00023279721572041828, -7.355338964460887e-05),
    c64::new(0.0002327407325083481, -7.373192122051467e-05),
    c64::new(0.00023268411238090142, -7.391040942183085e-05),
    c64::new(0.0002326273553713864, -7.408885414355745e-05),
    c64::new(0.00023257046151319174, -7.426725528072e-05),
    c64::new(0.0002325134308397866, -7.444561272836972e-05),
    c64::new(0.00023245626338472066, -7.462392638158352e-05),
    c64::new(0.0002323989591816241, -7.480219613546409e-05),
    c64::new(0.0002323415182642075, -7.498042188513992e-05),
    c64::new(0.0002322839406662618, -7.515860352576535e-05),
    c64::new(0.0002322262264216585, -7.533674095252076e-05),
    c64::new(0.00023216837556434939, -7.551483406061248e-05),
    c64::new(0.0002321103881283666, -7.569288274527292e-05),
    c64::new(0.00023205226414782268, -7.587088690176061e-05),
    c64::new(0.00023199400365691046, -7.604884642536033e-05),
    c64::new(0.00023193560668990308, -7.622676121138303e-05),
    c64::new(0.00023187707328115397, -7.640463115516606e-05),
    c64::new(0.00023181840346509684, -7.658245615207312e-05),
    c64::new(0.00023175959727624563, -7.67602360974943e-05),
    c64::new(0.00023170065474919447, -7.69379708868463e-05),
    c64::new(0.0002316415759186178, -7.711566041557225e-05),
    c64::new(0.00023158236081927008, -7.729330457914205e-05),
    c64::new(0.00023152300948598613, -7.747090327305218e-05),
    c64::new(0.00023146352195368072, -7.764845639282592e-05),
    c64::new(0.00023140389825734888, -7.78259638340133e-05),
    c64::new(0.0002313441384320657, -7.800342549219133e-05),
    c64::new(0.00023128424251298634, -7.818084126296383e-05),
    c64::new(0.000231224210535346, -7.835821104196172e-05),
    c64::new(0.00023116404253445996, -7.853553472484292e-05),
    c64::new(0.00023110373854572352, -7.871281220729244e-05),
    c64::new(0.00023104329860461196, -7.889004338502254e-05),
    c64::new(0.00023098272274668053, -7.906722815377267e-05),
    c64::new(0.00023092201100756443, -7.924436640930961e-05),
    c64::new(0.00023086116342297883, -7.942145804742747e-05),
    c64::new(0.00023080018002871882, -7.959850296394781e-05),
    c64::new(0.00023073906086065933, -7.977550105471966e-05),
    c64::new(0.00023067780595475518, -7.995245221561963e-05),
    c64::new(0.00023061641534704108, -8.012935634255188e-05),
    c64::new(0.00023055488907363153, -8.030621333144829e-05),
    c64::new(0.00023049322717072086, -8.048302307826843e-05),
    c64::new(0.00023043142967458318, -8.06597854789997e-05),
    c64::new(0.00023036949662157236, -8.083650042965733e-05),
    c64::new(0.000230307428048122, -8.101316782628445e-05),
    c64::new(0.00023024522399074548, -8.11897875649522e-05),
    c64::new(0.00023018288448603582, -8.136635954175975e-05),
    c64::new(0.00023012040957066574, -8.154288365283432e-05),
    c64::new(0.00023005779928138764, -8.171935979433135e-05),
    c64::new(0.00022999505365503352, -8.189578786243447e-05),
    c64::new(0.00022993217272851505, -8.20721677533556e-05),
    c64::new(0.00022986915653882344, -8.224849936333497e-05),
    c64::new(0.00022980600512302946, -8.242478258864127e-05),
    c64::new(0.0002297427185182835, -8.260101732557156e-05),
    c64::new(0.00022967929676181545, -8.277720347045157e-05),
    c64::new(0.00022961573989093464, -8.295334091963546e-05),
    c64::new(0.00022955204794303, -8.312942956950614e-05),
    c64::new(0.00022948822095556984, -8.330546931647518e-05),
    c64::new(0.0002294242589661019, -8.348146005698296e-05),
    c64::new(0.0002293601620122534, -8.365740168749863e-05),
    c64::new(0.00022929593013173093, -8.383329410452027e-05),
    c64::new(0.00022923156336232043, -8.40091372045749e-05),
    c64::new(0.00022916706174188725, -8.418493088421858e-05),
    c64::new(0.000229102425308376, -8.43606750400364e-05),
    c64::new(0.00022903765409981063, -8.453636956864258e-05),
    c64::new(0.0002289727481542944, -8.471201436668059e-05),
    c64::new(0.00022890770751000976, -8.488760933082309e-05),
    c64::new(0.0002288425322052185, -8.506315435777211e-05),
    c64::new(0.00022877722227826152, -8.523864934425903e-05),
    c64::new(0.00022871177776755897, -8.541409418704466e-05),
    c64::new(0.00022864619871161023, -8.558948878291931e-05),
    c64::new(0.00022858048514899372, -8.57648330287029e-05),
    c64::new(0.00022851463711836704, -8.594012682124489e-05),
    c64::new(0.0002284486546584669, -8.61153700574245e-05),
    c64::new(0.00022838253780810904, -8.629056263415059e-05),
    c64::new(0.00022831628660618838, -8.64657044483619e-05),
    c64::new(0.00022824990109167873, -8.664079539702705e-05),
    c64::new(0.000228183381303633, -8.68158353771445e-05),
    c64::new(0.00022811672728118308, -8.699082428574278e-05),
    c64::new(0.0002280499390635398, -8.716576201988037e-05),
    c64::new(0.0002279830166899929, -8.734064847664597e-05),
    c64::new(0.00022791596019991117, -8.751548355315834e-05),
    c64::new(0.0002278487696327422, -8.769026714656653e-05),
    c64::new(0.00022778144502801244, -8.786499915404983e-05),
    c64::new(0.0002277139864253272, -8.803967947281794e-05),
    c64::new(0.0002276463938643707, -8.821430800011091e-05),
    c64::new(0.00022757866738490587, -8.838888463319926e-05),
    c64::new(0.00022751080702677445, -8.856340926938409e-05),
    c64::new(0.00022744281282989693, -8.873788180599701e-05),
    c64::new(0.00022737468483427256, -8.891230214040036e-05),
    c64::new(0.00022730642307997932, -8.908667016998713e-05),
    c64::new(0.00022723802760717376, -8.926098579218112e-05),
    c64::new(0.00022716949845609123, -8.943524890443691e-05),
    c64::new(0.00022710083566704567, -8.960945940424003e-05),
    c64::new(0.0002270320392804296, -8.978361718910693e-05),
    c64::new(0.00022696310933671416, -8.995772215658504e-05),
    c64::new(0.0002268940458764491, -9.013177420425293e-05),
    c64::new(0.00022682484894026263, -9.030577322972024e-05),
    c64::new(0.00022675551856886158, -9.047971913062785e-05),
    c64::new(0.00022668605480303115, -9.065361180464784e-05),
    c64::new(0.00022661645768363512, -9.082745114948365e-05),
    c64::new(0.0002265467272516157, -9.100123706287006e-05),
    c64::new(0.00022647686354799345, -9.117496944257331e-05),
    c64::new(0.00022640686661386746, -9.13486481863911e-05),
    c64::new(0.00022633673649041506, -9.152227319215272e-05),
    c64::new(0.00022626647321889203, -9.169584435771905e-05),
    c64::new(0.00022619607684063245, -9.186936158098261e-05),
    c64::new(0.00022612554739704868, -9.204282475986774e-05),
    c64::new(0.00022605488492963136, -9.221623379233048e-05),
    c64::new(0.0002259840894799494, -9.238958857635878e-05),
    c64::new(0.00022591316108964998, -9.256288900997249e-05),
    c64::new(0.0002258420998004584, -9.273613499122343e-05),
    c64::new(0.0002257709056541782, -9.290932641819544e-05),
    c64::new(0.00022569957869269107, -9.308246318900449e-05),
    c64::new(0.00022562811895795684, -9.325554520179864e-05),
    c64::new(0.00022555652649201336, -9.342857235475825e-05),
    c64::new(0.00022548480133697673, -9.360154454609586e-05),
    c64::new(0.00022541294353504093, -9.37744616740564e-05),
    c64::new(0.0002253409531284781, -9.39473236369172e-05),
    c64::new(0.00022526883015963834, -9.4120130332988e-05),
    c64::new(0.0002251965746709497, -9.429288166061108e-05),
    c64::new(0.00022512418670491823, -9.446557751816127e-05),
    c64::new(0.00022505166630412796, -9.463821780404608e-05),
    c64::new(0.0002249790135112407, -9.481080241670563e-05),
    c64::new(0.00022490622836899627, -9.498333125461289e-05),
    c64::new(0.00022483331092021229, -9.515580421627354e-05),
    c64::new(0.00022476026120778414, -9.532822120022622e-05),
    c64::new(0.0002246870792746852, -9.550058210504245e-05),
    c64::new(0.00022461376516396645, -9.567288682932675e-05),
    c64::new(0.00022454031891875668, -9.584513527171667e-05),
    c64::new(0.00022446674058226247, -9.601732733088291e-05),
    c64::new(0.000224393030197768, -9.618946290552932e-05),
    c64::new(0.00022431918780863524, -9.636154189439296e-05),
    c64::new(0.00022424521345830377, -9.653356419624423e-05),
    c64::new(0.00022417110719029077, -9.670552970988678e-05),
    c64::new(0.00022409686904819104, -9.687743833415779e-05),
    c64::new(0.000224022499075677, -9.704928996792781e-05),
    c64::new(0.0002239479973164985, -9.722108451010097e-05),
    c64::new(0.00022387336381448315, -9.739282185961497e-05),
    c64::new(0.00022379859861353582, -9.756450191544111e-05),
    c64::new(0.00022372370175763895, -9.77361245765845e-05),
    c64::new(0.00022364867329085247, -9.790768974208389e-05),
    c64::new(0.00022357351325731365, -9.807919731101196e-05),
    c64::new(0.00022349822170123726, -9.82506471824752e-05),
    c64::new(0.00022342279866691535, -9.842203925561409e-05),
    c64::new(0.00022334724419871733, -9.859337342960305e-05),
    c64::new(0.00022327155834108995, -9.876464960365063e-05),
    c64::new(0.0002231957411385573, -9.893586767699948e-05),
    c64::new(0.00022311979263572063, -9.91070275489264e-05),
    c64::new(0.0002230437128772585, -9.927812911874244e-05),
    c64::new(0.0002229675019079267, -9.944917228579296e-05),
    c64::new(0.00022289115977255816, -9.962015694945769e-05),
    c64::new(0.00022281468651606296, -9.979108300915072e-05),
    c64::new(0.0002227380821834284, -9.99619503643207e-05),
    c64::new(0.00022266134681971883, -0.00010013275891445074),
    c64::new(0.00022258448047007565, -0.00010030350855905857),
    c64::new(0.0002225074831797174, -0.00010047419919769659),
    c64::new(0.00022243035499393954, -0.00010064483072995188),
    c64::new(0.00022235309595811464, -0.00010081540305544631),
    c64::new(0.0002222757061176922, -0.00010098591607383656),
    c64::new(0.00022219818551819865, -0.00010115636968481423),
    c64::new(0.00022212053420523736, -0.00010132676378810587),
    c64::new(0.00022204275222448863, -0.00010149709828347299),
    c64::new(0.00022196483962170957, -0.0001016673730707122),
    c64::new(0.00022188679644273416, -0.00010183758804965524),
    c64::new(0.00022180862273347317, -0.00010200774312016901),
    c64::new(0.00022173031853991423, -0.00010217783818215569),
    c64::new(0.00022165188390812165, -0.00010234787313555271),
    c64::new(0.00022157331888423648, -0.00010251784788033291),
    c64::new(0.00022149462351447652, -0.00010268776231650452),
    c64::new(0.00022141579784513626, -0.00010285761634411124),
    c64::new(0.00022133684192258675, -0.00010302740986323234),
    c64::new(0.0002212577557932758, -0.00010319714277398265),
    c64::new(0.00022117853950372765, -0.00010336681497651271),
    c64::new(0.00022109919310054328, -0.00010353642637100868),
    c64::new(0.0002210197166304001, -0.00010370597685769258),
    c64::new(0.00022094011014005206, -0.00010387546633682222),
    c64::new(0.00022086037367632965, -0.00010404489470869131),
    c64::new(0.00022078050728613973, -0.0001042142618736295),
    c64::new(0.00022070051101646566, -0.00010438356773200246),
    c64::new(0.00022062038491436718, -0.0001045528121842119),
    c64::new(0.00022054012902698042, -0.00010472199513069568),
    c64::new(0.00022045974340151786, -0.00010489111647192783),
    c64::new(0.00022037922808526827, -0.0001050601761084186),
    c64::new(0.00022029858312559673, -0.0001052291739407146),
    c64::new(0.00022021780856994462, -0.00010539810986939874),
    c64::new(0.00022013690446582952, -0.00010556698379509037),
    c64::new(0.00022005587086084522, -0.0001057357956184453),
    c64::new(0.0002199747078026617, -0.0001059045452401559),
    c64::new(0.00021989341533902514, -0.00010607323256095109),
    c64::new(0.00021981199351775771, -0.0001062418574815965),
    c64::new(0.00021973044238675788, -0.0001064104199028944),
    c64::new(0.00021964876199399998, -0.00010657891972568389),
    c64::new(0.00021956695238753456, -0.00010674735685084084),
    c64::new(0.00021948501361548805, -0.00010691573117927804),
    c64::new(0.00021940294572606294, -0.00010708404261194523),
    c64::new(0.00021932074876753767, -0.00010725229104982909),
    c64::new(0.00021923842278826657, -0.00010742047639395343),
    c64::new(0.0002191559678366799, -0.00010758859854537913),
    c64::new(0.00021907338396128377, -0.00010775665740520427),
    c64::new(0.00021899067121066015, -0.00010792465287456414),
    c64::new(0.00021890782963346686, -0.00010809258485463135),
    c64::new(0.0002188248592784374, -0.00010826045324661584),
    c64::new(0.00021874176019438116, -0.00010842825795176494),
    c64::new(0.0002186585324301831, -0.0001085959988713635),
    c64::new(0.0002185751760348041, -0.00010876367590673382),
    c64::new(0.00021849169105728046, -0.00010893128895923586),
    c64::new(0.0002184080775467243, -0.00010909883793026713),
    c64::new(0.00021832433555232328, -0.00010926632272126294),
    c64::new(0.00021824046512334064, -0.00010943374323369626),
    c64::new(0.0002181564663091152, -0.00010960109936907794),
    c64::new(0.00021807233915906136, -0.00010976839102895667),
    c64::new(0.0002179880837226689, -0.0001099356181149191),
    c64::new(0.00021790370004950307, -0.00011010278052858981),
    c64::new(0.0002178191881892047, -0.00011026987817163147),
    c64::new(0.00021773454819148992, -0.00011043691094574485),
    c64::new(0.00021764978010615024, -0.00011060387875266884),
    c64::new(0.00021756488398305254, -0.00011077078149418063),
    c64::new(0.00021747985987213904, -0.0001109376190720956),
    c64::new(0.00021739470782342722, -0.00011110439138826754),
    c64::new(0.00021730942788700984, -0.00011127109834458853),
    c64::new(0.0002172240201130549, -0.00011143773984298924),
    c64::new(0.00021713848455180555, -0.00011160431578543872),
    c64::new(0.00021705282125358022, -0.00011177082607394464),
    c64::new(0.00021696703026877236, -0.00011193727061055329),
    c64::new(0.00021688111164785067, -0.00011210364929734963),
    c64::new(0.00021679506544135878, -0.0001122699620364574),
    c64::new(0.0002167088916999155, -0.00011243620873003906),
    c64::new(0.00021662259047421464, -0.00011260238928029599),
    c64::new(0.00021653616181502495, -0.00011276850358946844),
    c64::new(0.0002164496057731902, -0.00011293455155983565),
    c64::new(0.00021636292239962906, -0.00011310053309371586),
    c64::new(0.00021627611174533515, -0.00011326644809346643),
    c64::new(0.00021618917386137694, -0.00011343229646148384),
    c64::new(0.00021610210879889773, -0.00011359807810020376),
    c64::new(0.00021601491660911567, -0.00011376379291210112),
    c64::new(0.00021592759734332366, -0.00011392944079969016),
    c64::new(0.0002158401510528894, -0.00011409502166552453),
    c64::new(0.0002157525777892553, -0.00011426053541219722),
    c64::new(0.00021566487760393837, -0.00011442598194234081),
    c64::new(0.00021557705054853048, -0.00011459136115862732),
    c64::new(0.00021548909667469795, -0.00011475667296376846),
    c64::new(0.00021540101603418181, -0.00011492191726051553),
    c64::new(0.00021531280867879762, -0.00011508709395165958),
    c64::new(0.0002152244746604355, -0.00011525220294003141),
    c64::new(0.00021513601403106005, -0.00011541724412850166),
    c64::new(0.00021504742684271043, -0.00011558221741998084),
    c64::new(0.00021495871314750018, -0.00011574712271741943),
    c64::new(0.00021486987299761728, -0.00011591195992380789),
    c64::new(0.00021478090644532407, -0.00011607672894217673),
    c64::new(0.00021469181354295735, -0.00011624142967559659),
    c64::new(0.0002146025943429281, -0.00011640606202717824),
    c64::new(0.0002145132488977217, -0.00011657062590007277),
    c64::new(0.00021442377725989785, -0.00011673512119747144),
    c64::new(0.0002143341794820903, -0.00011689954782260594),
    c64::new(0.00021424445561700717, -0.00011706390567874829),
    c64::new(0.00021415460571743068, -0.00011722819466921103),
    c64::new(0.0002140646298362172, -0.00011739241469734716),
    c64::new(0.00021397452802629727, -0.00011755656566655026),
    c64::new(0.00021388430034067544, -0.00011772064748025457),
    c64::new(0.00021379394683243032, -0.00011788466004193498),
    c64::new(0.00021370346755471457, -0.00011804860325510711),
    c64::new(0.0002136128625607548, -0.0001182124770233274),
    c64::new(0.00021352213190385162, -0.00011837628125019313),
    c64::new(0.00021343127563737947, -0.00011854001583934252),
    c64::new(0.00021334029381478683, -0.00011870368069445469),
    c64::new(0.00021324918648959592, -0.00011886727571924984),
    c64::new(0.00021315795371540285, -0.00011903080081748926),
    c64::new(0.00021306659554587752, -0.00011919425589297529),
    c64::new(0.00021297511203476356, -0.00011935764084955156),
    c64::new(0.00021288350323587837, -0.00011952095559110289),
    c64::new(0.00021279176920311303, -0.00011968420002155546),
    c64::new(0.00021269990999043238, -0.00011984737404487674),
    c64::new(0.00021260792565187472, -0.00012001047756507566),
    c64::new(0.00021251581624155217, -0.00012017351048620264),
    c64::new(0.00021242358181365026, -0.00012033647271234962),
    c64::new(0.00021233122242242814, -0.00012049936414765009),
    c64::new(0.00021223873812221847, -0.00012066218469627922),
    c64::new(0.00021214612896742734, -0.00012082493426245392),
    c64::new(0.0002120533950125344, -0.00012098761275043275),
    c64::new(0.0002119605363120926, -0.00012115022006451617),
    c64::new(0.00021186755292072835, -0.0001213127561090465),
    c64::new(0.00021177444489314134, -0.000121475220788408),
    c64::new(0.00021168121228410465, -0.00012163761400702682),
    c64::new(0.00021158785514846463, -0.00012179993566937124),
    c64::new(0.00021149437354114087, -0.00012196218567995162),
    c64::new(0.00021140076751712618, -0.00012212436394332048),
    c64::new(0.0002113070371314866, -0.00012228647036407244),
    c64::new(0.00021121318243936128, -0.00012244850484684458),
    c64::new(0.0002111192034959625, -0.0001226104672963161),
    c64::new(0.0002110251003565757, -0.00012277235761720874),
    c64::new(0.00021093087307655925, -0.00012293417571428652),
    c64::new(0.00021083652171134475, -0.00012309592149235604),
    c64::new(0.00021074204631643656, -0.0001232575948562665),
    c64::new(0.0002106474469474122, -0.00012341919571090952),
    c64::new(0.000210552723659922, -0.00012358072396121954),
    c64::new(0.00021045787650968926, -0.00012374217951217364),
    c64::new(0.00021036290555251008, -0.00012390356226879172),
    c64::new(0.00021026781084425346, -0.00012406487213613638),
    c64::new(0.00021017259244086116, -0.00012422610901931323),
    c64::new(0.00021007725039834767, -0.0001243872728234708),
    c64::new(0.00020998178477280033, -0.0001245483634538005),
    c64::new(0.00020988619562037906, -0.00012470938081553688),
    c64::new(0.00020979048299731652, -0.0001248703248139576),
    c64::new(0.00020969464695991795, -0.00012503119535438347),
    c64::new(0.00020959868756456124, -0.00012519199234217846),
    c64::new(0.0002095026048676968, -0.00012535271568274987),
    c64::new(0.00020940639892584769, -0.00012551336528154826),
    c64::new(0.00020931006979560927, -0.00012567394104406768),
    c64::new(0.0002092136175336496, -0.00012583444287584544),
    c64::new(0.00020911704219670897, -0.00012599487068246256),
    c64::new(0.0002090203438416002, -0.00012615522436954343),
    c64::new(0.00020892352252520845, -0.00012631550384275612),
    c64::new(0.00020882657830449122, -0.00012647570900781234),
    c64::new(0.00020872951123647825, -0.00012663583977046752),
    c64::new(0.00020863232137827161, -0.0001267958960365209),
    c64::new(0.00020853500878704562, -0.00012695587771181544),
    c64::new(0.00020843757352004673, -0.00012711578470223808),
    c64::new(0.0002083400156345936, -0.00012727561691371964),
    c64::new(0.00020824233518807706, -0.00012743537425223496),
    c64::new(0.00020814453223795998, -0.00012759505662380288),
    c64::new(0.0002080466068417773, -0.00012775466393448643),
    c64::new(0.00020794855905713602, -0.00012791419609039273),
    c64::new(0.00020785038894171514, -0.00012807365299767308),
    c64::new(0.00020775209655326556, -0.00012823303456252312),
    c64::new(0.00020765368194961022, -0.00012839234069118277),
    c64::new(0.00020755514518864385, -0.0001285515712899364),
    c64::new(0.00020745648632833314, -0.00012871072626511264),
    c64::new(0.0002073577054267165, -0.00012886980552308477),
    c64::new(0.0002072588025419042, -0.00012902880897027063),
    c64::new(0.00020715977773207828, -0.00012918773651313247),
    c64::new(0.0002070606310554925, -0.0001293465880581774),
    c64::new(0.00020696136257047226, -0.0001295053635119571),
    c64::new(0.00020686197233541463, -0.00012966406278106812),
    c64::new(0.0002067624604087884, -0.0001298226857721517),
    c64::new(0.00020666282684913382, -0.00012998123239189402),
    c64::new(0.00020656307171506276, -0.00013013970254702623),
    c64::new(0.00020646319506525856, -0.00013029809614432445),
    c64::new(0.00020636319695847613, -0.00013045641309060975),
    c64::new(0.00020626307745354178, -0.00013061465329274833),
    c64::new(0.0002061628366093532, -0.00013077281665765163),
    c64::new(0.0002060624744848795, -0.00013093090309227617),
    c64::new(0.00020596199113916117, -0.00013108891250362376),
    c64::new(0.00020586138663130992, -0.00013124684479874155),
    c64::new(0.00020576066102050882, -0.0001314046998847221),
    c64::new(0.00020565981436601217, -0.00013156247766870323),
    c64::new(0.00020555884672714543, -0.00013172017805786838),
    c64::new(0.00020545775816330528, -0.0001318778009594465),
    c64::new(0.0002053565487339595, -0.00013203534628071208),
    c64::new(0.000205255218498647, -0.00013219281392898522),
    c64::new(0.00020515376751697773, -0.00013235020381163182),
    c64::new(0.0002050521958486327, -0.00013250751583606345),
    c64::new(0.0002049505035533639, -0.0001326647499097375),
    c64::new(0.00020484869069099432, -0.00013282190594015718),
    c64::new(0.0002047467573214178, -0.0001329789838348717),
    c64::new(0.0002046447035045991, -0.0001331359835014762),
    c64::new(0.00020454252930057396, -0.0001332929048476117),
    c64::new(0.00020444023476944871, -0.00013344974778096554),
    c64::new(0.00020433781997140064, -0.00013360651220927102),
    c64::new(0.00020423528496667774, -0.00013376319804030766),
    c64::new(0.0002041326298155987, -0.00013391980518190122),
    c64::new(0.00020402985457855288, -0.00013407633354192374),
    c64::new(0.00020392695931600033, -0.00013423278302829368),
    c64::new(0.00020382394408847168, -0.00013438915354897578),
    c64::new(0.00020372080895656816, -0.0001345454450119813),
    c64::new(0.00020361755398096146, -0.000134701657325368),
    c64::new(0.00020351417922239383, -0.00013485779039724025),
    c64::new(0.00020341068474167802, -0.0001350138441357489),
    c64::new(0.00020330707059969713, -0.00013516981844909163),
    c64::new(0.0002032033368574047, -0.0001353257132455127),
    c64::new(0.0002030994835758246, -0.0001354815284333033),
    c64::new(0.00020299551081605108, -0.0001356372639208013),
    c64::new(0.0002028914186392486, -0.00013579291961639159),
    c64::new(0.00020278720710665193, -0.00013594849542850588),
    c64::new(0.000202682876279566, -0.00013610399126562294),
    c64::new(0.00020257842621936598, -0.00013625940703626858),
    c64::new(0.00020247385698749713, -0.0001364147426490157),
    c64::new(0.00020236916864547482, -0.0001365699980124844),
    c64::new(0.00020226436125488455, -0.0001367251730353419),
    c64::new(0.00020215943487738179, -0.00013688026762630274),
    c64::new(0.000202054389574692, -0.00013703528169412878),
    c64::new(0.00020194922540861063, -0.00013719021514762926),
    c64::new(0.00020184394244100306, -0.00013734506789566075),
    c64::new(0.00020173854073380457, -0.00013749983984712746),
    c64::new(0.0002016330203490202, -0.000137654530910981),
    c64::new(0.000201527381348725, -0.00013780914099622058),
    c64::new(0.00020142162379506353, -0.00013796367001189306),
    c64::new(0.00020131574775025032, -0.00013811811786709305),
    c64::new(0.00020120975327656948, -0.00013827248447096285),
    c64::new(0.00020110364043637486, -0.00013842676973269255),
    c64::new(0.00020099740929208987, -0.00013858097356152008),
    c64::new(0.00020089105990620762, -0.00013873509586673132),
    c64::new(0.00020078459234129062, -0.00013888913655766012),
    c64::new(0.00020067800665997108, -0.00013904309554368827),
    c64::new(0.0002005713029249506, -0.00013919697273424568),
    c64::new(0.00020046448119900016, -0.00013935076803881037),
    c64::new(0.00020035754154496032, -0.0001395044813669085),
    c64::new(0.0002002504840257409, -0.00013965811262811451),
    c64::new(0.00020014330870432104, -0.00013981166173205107),
    c64::new(0.00020003601564374927, -0.0001399651285883892),
    c64::new(0.0001999286049071433, -0.00014011851310684833),
    c64::new(0.0001998210765576901, -0.00014027181519719623),
    c64::new(0.00019971343065864586, -0.00014042503476924929),
    c64::new(0.00019960566727333587, -0.0001405781717328724),
    c64::new(0.00019949778646515453, -0.00014073122599797896),
    c64::new(0.0001993897882975654, -0.00014088419747453111),
    c64::new(0.00019928167283410093, -0.00014103708607253975),
    c64::new(0.00019917344013836274, -0.00014118989170206435),
    c64::new(0.0001990650902740213, -0.00014134261427321333),
    c64::new(0.00019895662330481607, -0.00014149525369614395),
    c64::new(0.00019884803929455533, -0.00014164780988106242),
    c64::new(0.0001987393383071163, -0.00014180028273822376),
    c64::new(0.00019863052040644495, -0.00014195267217793217),
    c64::new(0.00019852158565655605, -0.00014210497811054088),
    c64::new(0.0001984125341215331, -0.00014225720044645226),
    c64::new(0.0001983033658655283, -0.00014240933909611775),
    c64::new(0.00019819408095276257, -0.00014256139397003813),
    c64::new(0.00019808467944752532, -0.0001427133649787635),
    c64::new(0.00019797516141417472, -0.00014286525203289317),
    c64::new(0.0001978655269171374, -0.0001430170550430759),
    c64::new(0.00019775577602090848, -0.0001431687739200099),
    c64::new(0.00019764590879005158, -0.00014332040857444286),
    c64::new(0.0001975359252891988, -0.00014347195891717204),
    c64::new(0.00019742582558305053, -0.00014362342485904424),
    c64::new(0.0001973156097363757, -0.000143774806310956),
    c64::new(0.00019720527781401132, -0.0001439261031838535),
    c64::new(0.0001970948298808629, -0.00014407731538873262),
    c64::new(0.00019698426600190413, -0.0001442284428366392),
    c64::new(0.00019687358624217684, -0.00014437948543866884),
    c64::new(0.00019676279066679108, -0.00014453044310596704),
    c64::new(0.0001966518793409251, -0.00014468131574972928),
    c64::new(0.0001965408523298251, -0.00014483210328120112),
    c64::new(0.00019642970969880545, -0.00014498280561167813),
    c64::new(0.00019631845151324846, -0.00014513342265250596),
    c64::new(0.0001962070778386045, -0.00014528395431508056),
    c64::new(0.00019609558874039183, -0.000145434400510848),
    c64::new(0.00019598398428419657, -0.00014558476115130462),
    c64::new(0.00019587226453567278, -0.0001457350361479972),
    c64::new(0.0001957604295605423, -0.00014588522541252274),
    c64::new(0.00019564847942459479, -0.0001460353288565289),
    c64::new(0.00019553641419368757, -0.00014618534639171355),
    c64::new(0.0001954242339337458, -0.00014633527792982533),
    c64::new(0.00019531193871076219, -0.00014648512338266332),
    c64::new(0.00019519952859079712, -0.00014663488266207739),
    c64::new(0.0001950870036399786, -0.0001467845556799679),
    c64::new(0.00019497436392450218, -0.00014693414234828614),
    c64::new(0.0001948616095106308, -0.00014708364257903412),
    c64::new(0.0001947487404646951, -0.0001472330562842647),
    c64::new(0.00019463575685309297, -0.00014738238337608163),
    c64::new(0.00019452265874228975, -0.00014753162376663964),
    c64::new(0.00019440944619881814, -0.0001476807773681445),
    c64::new(0.00019429611928927821, -0.00014782984409285292),
    c64::new(0.00019418267808033723, -0.00014797882385307278),
    c64::new(0.00019406912263872974, -0.0001481277165611632),
    c64::new(0.00019395545303125747, -0.0001482765221295344),
    c64::new(0.00019384166932478935, -0.00014842524047064786),
    c64::new(0.0001937277715862614, -0.00014857387149701647),
    c64::new(0.00019361375988267667, -0.0001487224151212044),
    c64::new(0.0001934996342811054, -0.00014887087125582725),
    c64::new(0.00019338539484868467, -0.00014901923981355212),
    c64::new(0.00019327104165261862, -0.00014916752070709757),
    c64::new(0.00019315657476017825, -0.00014931571384923386),
    c64::new(0.00019304199423870156, -0.0001494638191527827),
    c64::new(0.00019292730015559327, -0.0001496118365306176),
    c64::new(0.00019281249257832497, -0.00014975976589566372),
    c64::new(0.00019269757157443497, -0.00014990760716089807),
    c64::new(0.0001925825372115284, -0.00015005536023934938),
    c64::new(0.00019246738955727694, -0.00015020302504409834),
    c64::new(0.000192352128679419, -0.0001503506014882776),
    c64::new(0.0001922367546457596, -0.0001504980894850717),
    c64::new(0.00019212126752417036, -0.00015064548894771723),
    c64::new(0.0001920056673825893, -0.0001507927997895029),
    c64::new(0.00019188995428902103, -0.0001509400219237696),
    c64::new(0.0001917741283115366, -0.00015108715526391023),
    c64::new(0.00019165818951827346, -0.00015123419972337012),
    c64::new(0.00019154213797743536, -0.00015138115521564675),
    c64::new(0.00019142597375729247, -0.00015152802165429007),
    c64::new(0.00019130969692618123, -0.00015167479895290223),
    c64::new(0.00019119330755250423, -0.00015182148702513803),
    c64::new(0.0001910768057047304, -0.00015196808578470458),
    c64::new(0.00019096019145139478, -0.00015211459514536164),
    c64::new(0.0001908434648610985, -0.00015226101502092152),
    c64::new(0.0001907266260025088, -0.00015240734532524918),
    c64::new(0.000190609674944359, -0.0001525535859722623),
    c64::new(0.0001904926117554484, -0.00015269973687593124),
    c64::new(0.00019037543650464223, -0.00015284579795027918),
    c64::new(0.00019025814926087168, -0.00015299176910938217),
    c64::new(0.00019014075009313378, -0.00015313765026736916),
    c64::new(0.0001900232390704915, -0.00015328344133842201),
    c64::new(0.00018990561626207352, -0.00015342914223677556),
    c64::new(0.0001897878817370743, -0.00015357475287671777),
    c64::new(0.00018967003556475401, -0.0001537202731725896),
    c64::new(0.00018955207781443857, -0.00015386570303878525),
    c64::new(0.00018943400855551945, -0.00015401104238975207),
    c64::new(0.00018931582785745374, -0.0001541562911399907),
    c64::new(0.00018919753578976412, -0.00015430144920405495),
    c64::new(0.00018907913242203879, -0.00015444651649655213),
    c64::new(0.00018896061782393137, -0.00015459149293214287),
    c64::new(0.0001888419920651609, -0.00015473637842554133),
    c64::new(0.00018872325521551196, -0.000154881172891515),
    c64::new(0.0001886044073448343, -0.00015502587624488512),
    c64::new(0.0001884854485230431, -0.0001551704884005264),
    c64::new(0.00018836637882011873, -0.00015531500927336723),
    c64::new(0.0001882471983061069, -0.0001554594387783897),
    c64::new(0.00018812790705111836, -0.00015560377683062967),
    c64::new(0.00018800850512532908, -0.00015574802334517677),
    c64::new(0.0001878889925989802, -0.0001558921782371745),
    c64::new(0.00018776936954237786, -0.00015603624142182025),
    c64::new(0.00018764963602589316, -0.00015618021281436535),
    c64::new(0.00018752979211996226, -0.00015632409233011515),
    c64::new(0.00018740983789508628, -0.00015646787988442905),
    c64::new(0.00018728977342183115, -0.0001566115753927205),
    c64::new(0.00018716959877082773, -0.00015675517877045716),
    c64::new(0.0001870493140127717, -0.00015689868993316086),
    c64::new(0.00018692891921842346, -0.00015704210879640768),
    c64::new(0.00018680841445860814, -0.00015718543527582798),
    c64::new(0.0001866877998042156, -0.00015732866928710654),
    c64::new(0.00018656707532620036, -0.0001574718107459824),
    c64::new(0.0001864462410955815, -0.0001576148595682492),
    c64::new(0.00018632529718344272, -0.00015775781566975495),
    c64::new(0.00018620424366093212, -0.00015790067896640232),
    c64::new(0.00018608308059926243, -0.00015804344937414845),
    c64::new(0.00018596180806971076, -0.0001581861268090052),
    c64::new(0.0001858404261436186, -0.00015832871118703917),
    c64::new(0.00018571893489239183, -0.00015847120242437157),
    c64::new(0.0001855973343875006, -0.00015861360043717851),
    c64::new(0.00018547562470047936, -0.00015875590514169088),
    c64::new(0.00018535380590292675, -0.00015889811645419458),
    c64::new(0.0001852318780665057, -0.00015904023429103023),
    c64::new(0.00018510984126294318, -0.00015918225856859363),
    c64::new(0.00018498769556403032, -0.00015932418920333556),
    c64::new(0.0001848654410416222, -0.0001594660261117619),
    c64::new(0.0001847430777676381, -0.0001596077692104336),
    c64::new(0.00018462060581406117, -0.0001597494184159669),
    c64::new(0.00018449802525293847, -0.00015989097364503324),
    c64::new(0.000184375336156381, -0.00016003243481435922),
    c64::new(0.0001842525385965636, -0.00016017380184072695),
    c64::new(0.0001841296326457249, -0.0001603150746409739),
    c64::new(0.00018400661837616732, -0.0001604562531319928),
    c64::new(0.00018388349586025696, -0.0001605973372307321),
    c64::new(0.00018376026517042367, -0.00016073832685419557),
    c64::new(0.00018363692637916082, -0.0001608792219194427),
    c64::new(0.0001835134795590255, -0.00016102002234358854),
    c64::new(0.00018338992478263827, -0.00016116072804380383),
    c64::new(0.00018326626212268328, -0.00016130133893731502),
    c64::new(0.00018314249165190798, -0.00016144185494140436),
    c64::new(0.0001830186134431234, -0.00016158227597340988),
    c64::new(0.00018289462756920395, -0.00016172260195072553),
    c64::new(0.00018277053410308724, -0.00016186283279080114),
    c64::new(0.00018264633311777432, -0.00016200296841114252),
    c64::new(0.00018252202468632938, -0.00016214300872931153),
    c64::new(0.00018239760888187992, -0.00016228295366292597),
    c64::new(0.00018227308577761653, -0.00016242280312965996),
    c64::new(0.00018214845544679291, -0.00016256255704724358),
    c64::new(0.00018202371796272594, -0.00016270221533346322),
    c64::new(0.0001818988733987954, -0.0001628417779061615),
    c64::new(0.00018177392182844418, -0.00016298124468323738),
    c64::new(0.00018164886332517805, -0.0001631206155826461),
    c64::new(0.00018152369796256573, -0.00016325989052239944),
    c64::new(0.0001813984258142387, -0.00016339906942056543),
    c64::new(0.00018127304695389143, -0.0001635381521952688),
    c64::new(0.00018114756145528102, -0.0001636771387646907),
    c64::new(0.00018102196939222737, -0.0001638160290470689),
    c64::new(0.0001808962708386131, -0.00016395482296069783),
    c64::new(0.0001807704658683833, -0.00016409352042392864),
    c64::new(0.00018064455455554595, -0.00016423212135516915),
    c64::new(0.00018051853697417133, -0.00016437062567288403),
    c64::new(0.00018039241319839236, -0.00016450903329559474),
    c64::new(0.00018026618330240445, -0.00016464734414187965),
    c64::new(0.00018013984736046533, -0.00016478555813037403),
    c64::new(0.00018001340544689524, -0.00016492367517977015),
    c64::new(0.0001798868576360766, -0.00016506169520881736),
    c64::new(0.00017976020400245436, -0.000165199618136322),
    c64::new(0.00017963344462053555, -0.00016533744388114754),
    c64::new(0.00017950657956488937, -0.00016547517236221472),
    c64::new(0.0001793796089101473, -0.00016561280349850134),
    c64::new(0.000179252532731003, -0.00016575033720904258),
    c64::new(0.000179125351102212, -0.0001658877734129309),
    c64::new(0.00017899806409859198, -0.00016602511202931615),
    c64::new(0.00017887067179502266, -0.00016616235297740552),
    c64::new(0.00017874317426644565, -0.00016629949617646373),
    c64::new(0.0001786155715878644, -0.00016643654154581292),
    c64::new(0.0001784878638343443, -0.0001665734890048329),
    c64::new(0.0001783600510810126, -0.00016671033847296096),
    c64::new(0.00017823213340305814, -0.00016684708986969206),
    c64::new(0.0001781041108757317, -0.00016698374311457898),
    c64::new(0.00017797598357434554, -0.000167120298127232),
    c64::new(0.00017784775157427373, -0.00016725675482731942),
    c64::new(0.0001777194149509518, -0.00016739311313456725),
    c64::new(0.0001775909737798769, -0.00016752937296875942),
    c64::new(0.00017746242813660763, -0.00016766553424973777),
    c64::new(0.00017733377809676416, -0.0001678015968974021),
    c64::new(0.00017720502373602792, -0.00016793756083171034),
    c64::new(0.0001770761651301418, -0.00016807342597267833),
    c64::new(0.00017694720235490998, -0.00016820919224038014),
    c64::new(0.000176818135486198, -0.00016834485955494796),
    c64::new(0.00017668896459993247, -0.00016848042783657225),
    c64::new(0.00017655968977210143, -0.0001686158970055016),
    c64::new(0.00017643031107875384, -0.00016875126698204308),
    c64::new(0.00017630082859599984, -0.00016888653768656195),
    c64::new(0.00017617124240001069, -0.000169021709039482),
    c64::new(0.00017604155256701858, -0.0001691567809612853),
    c64::new(0.00017591175917331672, -0.00016929175337251264),
    c64::new(0.0001757818622952592, -0.00016942662619376316),
    c64::new(0.00017565186200926098, -0.0001695613993456947),
    c64::new(0.00017552175839179797, -0.00016969607274902365),
    c64::new(0.0001753915515194067, -0.00016983064632452511),
    c64::new(0.0001752612414686845, -0.00016996511999303296),
    c64::new(0.00017513082831628949, -0.00017009949367543972),
    c64::new(0.00017500031213894032, -0.0001702337672926969),
    c64::new(0.00017486969301341632, -0.00017036794076581477),
    c64::new(0.00017473897101655727, -0.00017050201401586253),
    c64::new(0.0001746081462252637, -0.0001706359869639683),
    c64::new(0.00017447721871649636, -0.00017076985953131928),
    c64::new(0.00017434618856727657, -0.00017090363163916168),
    c64::new(0.00017421505585468592, -0.00017103730320880086),
    c64::new(0.0001740838206558665, -0.00017117087416160124),
    c64::new(0.0001739524830480206, -0.00017130434441898644),
    c64::new(0.00017382104310841067, -0.00017143771390243942),
    c64::new(0.00017368950091435948, -0.00017157098253350227),
    c64::new(0.00017355785654324992, -0.00017170415023377654),
    c64::new(0.000173426110072525, -0.00017183721692492308),
    c64::new(0.0001732942615796877, -0.0001719701825286622),
    c64::new(0.0001731623111423012, -0.00017210304696677365),
    c64::new(0.00017303025883798844, -0.0001722358101610967),
    c64::new(0.00017289810474443247, -0.0001723684720335302),
    c64::new(0.00017276584893937606, -0.0001725010325060326),
    c64::new(0.00017263349150062197, -0.00017263349150062194),
    c64::new(0.00017250103250603262, -0.00017276584893937606),
    c64::new(0.00017236847203353024, -0.00017289810474443244),
    c64::new(0.00017223581016109673, -0.00017303025883798844),
    c64::new(0.00017210304696677368, -0.00017316231114230117),
    c64::new(0.00017197018252866224, -0.0001732942615796877),
    c64::new(0.0001718372169249231, -0.00017342611007252498),
    c64::new(0.00017170415023377654, -0.00017355785654324992),
    c64::new(0.0001715709825335023, -0.00017368950091435945),
    c64::new(0.00017143771390243942, -0.00017382104310841064),
    c64::new(0.00017130434441898647, -0.0001739524830480206),
    c64::new(0.00017117087416160126, -0.0001740838206558665),
    c64::new(0.0001710373032088009, -0.00017421505585468592),
    c64::new(0.0001709036316391617, -0.00017434618856727654),
    c64::new(0.0001707698595313193, -0.00017447721871649633),
    c64::new(0.00017063598696396832, -0.0001746081462252637),
    c64::new(0.00017050201401586253, -0.00017473897101655727),
    c64::new(0.0001703679407658148, -0.00017486969301341626),
    c64::new(0.00017023376729269693, -0.0001750003121389403),
    c64::new(0.00017009949367543975, -0.00017513082831628946),
    c64::new(0.00016996511999303298, -0.00017526124146868447),
    c64::new(0.00016983064632452514, -0.00017539155151940667),
    c64::new(0.00016969607274902365, -0.00017552175839179794),
    c64::new(0.00016956139934569474, -0.00017565186200926098),
    c64::new(0.00016942662619376318, -0.0001757818622952592),
    c64::new(0.00016929175337251266, -0.00017591175917331672),
    c64::new(0.00016915678096128534, -0.00017604155256701855),
    c64::new(0.00016902170903948201, -0.00017617124240001066),
    c64::new(0.00016888653768656198, -0.0001763008285959998),
    c64::new(0.0001687512669820431, -0.0001764303110787538),
    c64::new(0.00016861589700550164, -0.00017655968977210138),
    c64::new(0.00016848042783657228, -0.00017668896459993247),
    c64::new(0.00016834485955494798, -0.00017681813548619797),
    c64::new(0.00016820919224038016, -0.00017694720235490995),
    c64::new(0.00016807342597267836, -0.00017707616513014176),
    c64::new(0.00016793756083171034, -0.0001772050237360279),
    c64::new(0.00016780159689740214, -0.00017733377809676414),
    c64::new(0.0001676655342497378, -0.00017746242813660763),
    c64::new(0.00016752937296875945, -0.00017759097377987687),
    c64::new(0.00016739311313456728, -0.0001777194149509518),
    c64::new(0.00016725675482731945, -0.00017784775157427373),
    c64::new(0.00016712029812723203, -0.00017797598357434554),
    c64::new(0.00016698374311457898, -0.00017810411087573167),
    c64::new(0.0001668470898696921, -0.00017823213340305814),
    c64::new(0.00016671033847296096, -0.0001783600510810126),
    c64::new(0.00016657348900483293, -0.0001784878638343443),
    c64::new(0.00016643654154581294, -0.00017861557158786438),
    c64::new(0.00016629949617646373, -0.00017874317426644563),
    c64::new(0.00016616235297740555, -0.00017887067179502263),
    c64::new(0.00016602511202931617, -0.00017899806409859198),
    c64::new(0.00016588777341293093, -0.00017912535110221198),
    c64::new(0.0001657503372090426, -0.00017925253273100297),
    c64::new(0.00016561280349850136, -0.0001793796089101473),
    c64::new(0.00016547517236221475, -0.00017950657956488937),
    c64::new(0.00016533744388114757, -0.00017963344462053552),
    c64::new(0.000165199618136322, -0.00017976020400245436),
    c64::new(0.0001650616952088174, -0.0001798868576360766),
    c64::new(0.00016492367517977017, -0.00018001340544689521),
    c64::new(0.00016478555813037403, -0.00018013984736046533),
    c64::new(0.00016464734414187965, -0.00018026618330240442),
    c64::new(0.00016450903329559476, -0.00018039241319839236),
    c64::new(0.00016437062567288405, -0.0001805185369741713),
    c64::new(0.00016423212135516918, -0.00018064455455554592),
    c64::new(0.00016409352042392867, -0.0001807704658683833),
    c64::new(0.00016395482296069786, -0.00018089627083861304),
    c64::new(0.00016381602904706893, -0.00018102196939222737),
    c64::new(0.00016367713876469072, -0.000181147561455281),
    c64::new(0.00016353815219526883, -0.00018127304695389137),
    c64::new(0.00016339906942056546, -0.00018139842581423867),
    c64::new(0.00016325989052239944, -0.00018152369796256568),
    c64::new(0.00016312061558264613, -0.00018164886332517802),
    c64::new(0.0001629812446832374, -0.00018177392182844415),
    c64::new(0.00016284177790616152, -0.0001818988733987954),
    c64::new(0.00016270221533346322, -0.0001820237179627259),
    c64::new(0.00016256255704724358, -0.0001821484554467929),
    c64::new(0.00016242280312965996, -0.0001822730857776165),
    c64::new(0.000162282953662926, -0.00018239760888187992),
    c64::new(0.00016214300872931153, -0.00018252202468632938),
    c64::new(0.00016200296841114255, -0.0001826463331177743),
    c64::new(0.00016186283279080114, -0.00018277053410308724),
    c64::new(0.00016172260195072553, -0.00018289462756920392),
    c64::new(0.00016158227597340988, -0.0001830186134431234),
    c64::new(0.00016144185494140436, -0.00018314249165190795),
    c64::new(0.00016130133893731502, -0.00018326626212268325),
    c64::new(0.00016116072804380385, -0.00018338992478263827),
    c64::new(0.00016102002234358857, -0.0001835134795590255),
    c64::new(0.0001608792219194427, -0.00018363692637916082),
    c64::new(0.0001607383268541956, -0.00018376026517042364),
    c64::new(0.00016059733723073212, -0.00018388349586025694),
    c64::new(0.00016045625313199283, -0.0001840066183761673),
    c64::new(0.0001603150746409739, -0.0001841296326457249),
    c64::new(0.00016017380184072698, -0.00018425253859656357),
    c64::new(0.00016003243481435924, -0.00018437533615638097),
    c64::new(0.00015989097364503324, -0.00018449802525293842),
    c64::new(0.00015974941841596693, -0.00018462060581406117),
    c64::new(0.00015960776921043364, -0.0001847430777676381),
    c64::new(0.00015946602611176193, -0.00018486544104162218),
    c64::new(0.00015932418920333558, -0.0001849876955640303),
    c64::new(0.00015918225856859366, -0.00018510984126294318),
    c64::new(0.00015904023429103026, -0.00018523187806650568),
    c64::new(0.00015889811645419458, -0.00018535380590292675),
    c64::new(0.0001587559051416909, -0.00018547562470047933),
    c64::new(0.00015861360043717854, -0.00018559733438750056),
    c64::new(0.0001584712024243716, -0.0001857189348923918),
    c64::new(0.0001583287111870392, -0.0001858404261436186),
    c64::new(0.00015818612680900524, -0.00018596180806971076),
    c64::new(0.00015804344937414848, -0.00018608308059926243),
    c64::new(0.00015790067896640235, -0.0001862042436609321),
    c64::new(0.00015775781566975498, -0.0001863252971834427),
    c64::new(0.00015761485956824922, -0.00018644624109558148),
    c64::new(0.00015747181074598243, -0.00018656707532620036),
    c64::new(0.00015732866928710657, -0.00018668779980421558),
    c64::new(0.000157185435275828, -0.00018680841445860812),
    c64::new(0.00015704210879640768, -0.00018692891921842343),
    c64::new(0.00015689868993316088, -0.00018704931401277168),
    c64::new(0.0001567551787704572, -0.00018716959877082773),
    c64::new(0.0001566115753927205, -0.00018728977342183113),
    c64::new(0.00015646787988442905, -0.00018740983789508625),
    c64::new(0.00015632409233011518, -0.00018752979211996223),
    c64::new(0.00015618021281436537, -0.00018764963602589313),
    c64::new(0.00015603624142182027, -0.00018776936954237783),
    c64::new(0.0001558921782371745, -0.0001878889925989802),
    c64::new(0.0001557480233451768, -0.00018800850512532908),
    c64::new(0.00015560377683062967, -0.00018812790705111833),
    c64::new(0.00015545943877838972, -0.00018824719830610686),
    c64::new(0.00015531500927336726, -0.0001883663788201187),
    c64::new(0.00015517048840052642, -0.00018848544852304307),
    c64::new(0.00015502587624488515, -0.00018860440734483428),
    c64::new(0.00015488117289151504, -0.00018872325521551193),
    c64::new(0.00015473637842554133, -0.0001888419920651609),
    c64::new(0.0001545914929321429, -0.00018896061782393134),
    c64::new(0.00015444651649655213, -0.00018907913242203876),
    c64::new(0.00015430144920405495, -0.00018919753578976412),
    c64::new(0.0001541562911399907, -0.00018931582785745372),
    c64::new(0.0001540110423897521, -0.00018943400855551943),
    c64::new(0.00015386570303878527, -0.00018955207781443854),
    c64::new(0.00015372027317258962, -0.00018967003556475401),
    c64::new(0.0001535747528767178, -0.00018978788173707427),
    c64::new(0.00015342914223677556, -0.0001899056162620735),
    c64::new(0.00015328344133842201, -0.00019002323907049148),
    c64::new(0.0001531376502673692, -0.00019014075009313376),
    c64::new(0.0001529917691093822, -0.00019025814926087165),
    c64::new(0.0001528457979502792, -0.0001903754365046422),
    c64::new(0.00015269973687593124, -0.0001904926117554484),
    c64::new(0.00015255358597226232, -0.00019060967494435898),
    c64::new(0.0001524073453252492, -0.00019072662600250879),
    c64::new(0.00015226101502092154, -0.00019084346486109848),
    c64::new(0.00015211459514536166, -0.00019096019145139475),
    c64::new(0.0001519680857847046, -0.00019107680570473038),
    c64::new(0.00015182148702513803, -0.00019119330755250423),
    c64::new(0.00015167479895290226, -0.0001913096969261812),
    c64::new(0.0001515280216542901, -0.00019142597375729247),
    c64::new(0.00015138115521564678, -0.00019154213797743536),
    c64::new(0.00015123419972337012, -0.00019165818951827343),
    c64::new(0.00015108715526391026, -0.0001917741283115366),
    c64::new(0.0001509400219237696, -0.000191889954289021),
    c64::new(0.00015079279978950292, -0.0001920056673825893),
    c64::new(0.00015064548894771725, -0.00019212126752417034),
    c64::new(0.0001504980894850717, -0.0001922367546457596),
    c64::new(0.0001503506014882776, -0.00019235212867941897),
    c64::new(0.00015020302504409837, -0.0001924673895572769),
    c64::new(0.00015005536023934938, -0.00019258253721152836),
    c64::new(0.00014990760716089807, -0.00019269757157443495),
    c64::new(0.00014975976589566374, -0.00019281249257832497),
    c64::new(0.00014961183653061763, -0.00019292730015559327),
    c64::new(0.00014946381915278273, -0.00019304199423870156),
    c64::new(0.0001493157138492339, -0.00019315657476017825),
    c64::new(0.0001491675207070976, -0.0001932710416526186),
    c64::new(0.00014901923981355212, -0.00019338539484868464),
    c64::new(0.00014887087125582725, -0.00019349963428110537),
    c64::new(0.00014872241512120442, -0.00019361375988267667),
    c64::new(0.00014857387149701647, -0.00019372777158626137),
    c64::new(0.0001484252404706479, -0.00019384166932478932),
    c64::new(0.0001482765221295344, -0.00019395545303125747),
    c64::new(0.00014812771656116323, -0.00019406912263872974),
    c64::new(0.0001479788238530728, -0.0001941826780803372),
    c64::new(0.00014782984409285292, -0.0001942961192892782),
    c64::new(0.00014768077736814451, -0.00019440944619881812),
    c64::new(0.00014753162376663967, -0.00019452265874228973),
    c64::new(0.00014738238337608166, -0.00019463575685309294),
    c64::new(0.00014723305628426472, -0.00019474874046469508),
    c64::new(0.00014708364257903415, -0.0001948616095106308),
    c64::new(0.00014693414234828617, -0.00019497436392450215),
    c64::new(0.00014678455567996792, -0.0001950870036399786),
    c64::new(0.00014663488266207739, -0.00019519952859079712),
    c64::new(0.00014648512338266334, -0.00019531193871076219),
    c64::new(0.00014633527792982535, -0.00019542423393374578),
    c64::new(0.00014618534639171358, -0.00019553641419368757),
    c64::new(0.0001460353288565289, -0.00019564847942459476),
    c64::new(0.00014588522541252277, -0.0001957604295605423),
    c64::new(0.0001457350361479972, -0.00019587226453567276),
    c64::new(0.00014558476115130464, -0.00019598398428419654),
    c64::new(0.00014543440051084801, -0.0001960955887403918),
    c64::new(0.00014528395431508058, -0.0001962070778386045),
    c64::new(0.000145133422652506, -0.00019631845151324846),
    c64::new(0.00014498280561167813, -0.00019642970969880542),
    c64::new(0.00014483210328120115, -0.00019654085232982508),
    c64::new(0.0001446813157497293, -0.00019665187934092508),
    c64::new(0.00014453044310596704, -0.00019676279066679108),
    c64::new(0.00014437948543866887, -0.00019687358624217682),
    c64::new(0.00014422844283663923, -0.0001969842660019041),
    c64::new(0.00014407731538873265, -0.0001970948298808629),
    c64::new(0.0001439261031838535, -0.0001972052778140113),
    c64::new(0.00014377480631095601, -0.00019731560973637566),
    c64::new(0.00014362342485904427, -0.00019742582558305053),
    c64::new(0.00014347195891717206, -0.00019753592528919876),
    c64::new(0.0001433204085744429, -0.00019764590879005155),
    c64::new(0.00014316877392000991, -0.00019775577602090848),
    c64::new(0.0001430170550430759, -0.0001978655269171374),
    c64::new(0.0001428652520328932, -0.00019797516141417472),
    c64::new(0.00014271336497876353, -0.00019808467944752532),
    c64::new(0.00014256139397003816, -0.00019819408095276254),
    c64::new(0.00014240933909611777, -0.0001983033658655283),
    c64::new(0.00014225720044645226, -0.00019841253412153308),
    c64::new(0.0001421049781105409, -0.00019852158565655603),
    c64::new(0.0001419526721779322, -0.00019863052040644492),
    c64::new(0.0001418002827382238, -0.00019873933830711628),
    c64::new(0.00014164780988106245, -0.00019884803929455533),
    c64::new(0.00014149525369614398, -0.00019895662330481604),
    c64::new(0.00014134261427321335, -0.00019906509027402128),
    c64::new(0.00014118989170206438, -0.00019917344013836274),
    c64::new(0.00014103708607253977, -0.00019928167283410093),
    c64::new(0.00014088419747453114, -0.00019938978829756536),
    c64::new(0.00014073122599797896, -0.00019949778646515453),
    c64::new(0.00014057817173287242, -0.00019960566727333587),
    c64::new(0.0001404250347692493, -0.00019971343065864586),
    c64::new(0.00014027181519719626, -0.0001998210765576901),
    c64::new(0.00014011851310684835, -0.0001999286049071433),
    c64::new(0.00013996512858838924, -0.00020003601564374924),
    c64::new(0.0001398116617320511, -0.00020014330870432101),
    c64::new(0.00013965811262811454, -0.0002002504840257409),
    c64::new(0.00013950448136690853, -0.00020035754154496032),
    c64::new(0.0001393507680388104, -0.00020046448119900016),
    c64::new(0.0001391969727342457, -0.00020057130292495056),
    c64::new(0.0001390430955436883, -0.00020067800665997106),
    c64::new(0.00013888913655766015, -0.00020078459234129062),
    c64::new(0.00013873509586673135, -0.0002008910599062076),
    c64::new(0.0001385809735615201, -0.00020099740929208987),
    c64::new(0.00013842676973269255, -0.00020110364043637483),
    c64::new(0.00013827248447096288, -0.00020120975327656948),
    c64::new(0.00013811811786709307, -0.00020131574775025032),
    c64::new(0.0001379636700118931, -0.00020142162379506353),
    c64::new(0.00013780914099622058, -0.00020152738134872496),
    c64::new(0.00013765453091098102, -0.0002016330203490202),
    c64::new(0.0001374998398471275, -0.00020173854073380455),
    c64::new(0.00013734506789566078, -0.00020184394244100306),
    c64::new(0.00013719021514762926, -0.0002019492254086106),
    c64::new(0.0001370352816941288, -0.00020205438957469196),
    c64::new(0.00013688026762630274, -0.00020215943487738176),
    c64::new(0.00013672517303534192, -0.00020226436125488453),
    c64::new(0.0001365699980124844, -0.0002023691686454748),
    c64::new(0.00013641474264901573, -0.00020247385698749713),
    c64::new(0.0001362594070362686, -0.00020257842621936598),
    c64::new(0.00013610399126562296, -0.00020268287627956598),
    c64::new(0.0001359484954285059, -0.0002027872071066519),
    c64::new(0.0001357929196163916, -0.0002028914186392486),
    c64::new(0.00013563726392080134, -0.00020299551081605106),
    c64::new(0.00013548152843330334, -0.0002030994835758246),
    c64::new(0.00013532571324551272, -0.0002032033368574047),
    c64::new(0.00013516981844909163, -0.00020330707059969713),
    c64::new(0.00013501384413574892, -0.00020341068474167802),
    c64::new(0.00013485779039724028, -0.00020351417922239383),
    c64::new(0.00013470165732536803, -0.00020361755398096143),
    c64::new(0.00013454544501198133, -0.00020372080895656814),
    c64::new(0.0001343891535489758, -0.00020382394408847168),
    c64::new(0.0001342327830282937, -0.0002039269593160003),
    c64::new(0.00013407633354192377, -0.00020402985457855285),
    c64::new(0.00013391980518190125, -0.00020413262981559867),
    c64::new(0.0001337631980403077, -0.0002042352849666777),
    c64::new(0.00013360651220927102, -0.00020433781997140064),
    c64::new(0.00013344974778096557, -0.0002044402347694487),
    c64::new(0.00013329290484761174, -0.00020454252930057393),
    c64::new(0.00013313598350147621, -0.0002046447035045991),
    c64::new(0.00013297898383487174, -0.0002047467573214178),
    c64::new(0.0001328219059401572, -0.0002048486906909943),
    c64::new(0.00013266474990973753, -0.00020495050355336388),
    c64::new(0.00013250751583606348, -0.00020505219584863267),
    c64::new(0.00013235020381163184, -0.0002051537675169777),
    c64::new(0.00013219281392898524, -0.00020525521849864696),
    c64::new(0.0001320353462807121, -0.0002053565487339595),
    c64::new(0.00013187780095944652, -0.00020545775816330526),
    c64::new(0.0001317201780578684, -0.00020555884672714543),
    c64::new(0.00013156247766870323, -0.00020565981436601217),
    c64::new(0.00013140469988472212, -0.00020576066102050882),
    c64::new(0.00013124684479874158, -0.00020586138663130992),
    c64::new(0.0001310889125036238, -0.00020596199113916114),
    c64::new(0.00013093090309227617, -0.0002060624744848795),
    c64::new(0.00013077281665765163, -0.0002061628366093532),
    c64::new(0.00013061465329274838, -0.00020626307745354175),
    c64::new(0.00013045641309060977, -0.00020636319695847613),
    c64::new(0.00013029809614432445, -0.00020646319506525856),
    c64::new(0.0001301397025470263, -0.0002065630717150627),
    c64::new(0.00012998123239189405, -0.0002066628268491338),
    c64::new(0.0001298226857721517, -0.0002067624604087884),
    c64::new(0.00012966406278106814, -0.00020686197233541463),
    c64::new(0.00012950536351195714, -0.00020696136257047223),
    c64::new(0.00012934658805817745, -0.00020706063105549246),
    c64::new(0.0001291877365131325, -0.00020715977773207828),
    c64::new(0.00012902880897027063, -0.0002072588025419042),
    c64::new(0.00012886980552308483, -0.00020735770542671647),
    c64::new(0.00012871072626511264, -0.00020745648632833311),
    c64::new(0.0001285515712899364, -0.00020755514518864385),
    c64::new(0.00012839234069118283, -0.0002076536819496102),
    c64::new(0.00012823303456252312, -0.00020775209655326556),
    c64::new(0.00012807365299767308, -0.00020785038894171514),
    c64::new(0.00012791419609039275, -0.000207948559057136),
    c64::new(0.00012775466393448643, -0.0002080466068417773),
    c64::new(0.00012759505662380294, -0.00020814453223795995),
    c64::new(0.00012743537425223496, -0.00020824233518807706),
    c64::new(0.00012727561691371964, -0.0002083400156345936),
    c64::new(0.0001271157847022381, -0.0002084375735200467),
    c64::new(0.00012695587771181547, -0.0002085350087870456),
    c64::new(0.0001267958960365209, -0.00020863232137827161),
    c64::new(0.00012663583977046757, -0.00020872951123647823),
    c64::new(0.00012647570900781237, -0.00020882657830449122),
    c64::new(0.00012631550384275612, -0.00020892352252520845),
    c64::new(0.00012615522436954345, -0.00020902034384160019),
    c64::new(0.00012599487068246258, -0.00020911704219670897),
    c64::new(0.0001258344428758455, -0.00020921361753364957),
    c64::new(0.00012567394104406768, -0.00020931006979560927),
    c64::new(0.00012551336528154828, -0.00020940639892584769),
    c64::new(0.0001253527156827499, -0.00020950260486769678),
    c64::new(0.00012519199234217848, -0.00020959868756456122),
    c64::new(0.00012503119535438347, -0.00020969464695991795),
    c64::new(0.00012487032481395765, -0.00020979048299731647),
    c64::new(0.0001247093808155369, -0.00020988619562037906),
    c64::new(0.0001245483634538005, -0.00020998178477280033),
    c64::new(0.00012438727282347082, -0.00021007725039834767),
    c64::new(0.00012422610901931326, -0.00021017259244086113),
    c64::new(0.00012406487213613643, -0.00021026781084425343),
    c64::new(0.00012390356226879174, -0.00021036290555251008),
    c64::new(0.00012374217951217366, -0.00021045787650968926),
    c64::new(0.0001235807239612196, -0.000210552723659922),
    c64::new(0.00012341919571090955, -0.0002106474469474122),
    c64::new(0.0001232575948562665, -0.00021074204631643656),
    c64::new(0.0001230959214923561, -0.00021083652171134472),
    c64::new(0.00012293417571428652, -0.00021093087307655925),
    c64::new(0.00012277235761720874, -0.0002110251003565757),
    c64::new(0.00012261046729631614, -0.0002111192034959625),
    c64::new(0.00012244850484684458, -0.00021121318243936125),
    c64::new(0.0001222864703640725, -0.00021130703713148657),
    c64::new(0.00012212436394332048, -0.00021140076751712618),
    c64::new(0.00012196218567995163, -0.00021149437354114087),
    c64::new(0.00012179993566937129, -0.0002115878551484646),
    c64::new(0.00012163761400702684, -0.00021168121228410465),
    c64::new(0.000121475220788408, -0.00021177444489314134),
    c64::new(0.00012131275610904656, -0.00021186755292072832),
    c64::new(0.0001211502200645162, -0.00021196053631209258),
    c64::new(0.00012098761275043275, -0.0002120533950125344),
    c64::new(0.00012082493426245395, -0.00021214612896742734),
    c64::new(0.00012066218469627924, -0.00021223873812221844),
    c64::new(0.00012049936414765013, -0.00021233122242242812),
    c64::new(0.00012033647271234965, -0.00021242358181365023),
    c64::new(0.00012017351048620266, -0.00021251581624155214),
    c64::new(0.00012001047756507571, -0.00021260792565187472),
    c64::new(0.00011984737404487676, -0.00021269990999043235),
    c64::new(0.00011968420002155546, -0.00021279176920311303),
    c64::new(0.00011952095559110293, -0.00021288350323587834),
    c64::new(0.00011935764084955157, -0.00021297511203476353),
    c64::new(0.00011919425589297529, -0.00021306659554587752),
    c64::new(0.00011903080081748928, -0.00021315795371540283),
    c64::new(0.00011886727571924987, -0.00021324918648959592),
    c64::new(0.00011870368069445473, -0.0002133402938147868),
    c64::new(0.00011854001583934255, -0.00021343127563737944),
    c64::new(0.00011837628125019314, -0.0002135221319038516),
    c64::new(0.00011821247702332744, -0.00021361286256075477),
    c64::new(0.00011804860325510713, -0.00021370346755471455),
    c64::new(0.00011788466004193498, -0.00021379394683243032),
    c64::new(0.00011772064748025461, -0.0002138843003406754),
    c64::new(0.00011755656566655029, -0.00021397452802629727),
    c64::new(0.00011739241469734716, -0.0002140646298362172),
    c64::new(0.00011722819466921106, -0.00021415460571743065),
    c64::new(0.0001170639056787483, -0.00021424445561700714),
    c64::new(0.00011689954782260597, -0.00021433417948209027),
    c64::new(0.00011673512119747146, -0.00021442377725989782),
    c64::new(0.00011657062590007277, -0.0002145132488977217),
    c64::new(0.00011640606202717828, -0.00021460259434292808),
    c64::new(0.00011624142967559659, -0.00021469181354295732),
    c64::new(0.00011607672894217673, -0.00021478090644532407),
    c64::new(0.00011591195992380792, -0.00021486987299761728),
    c64::new(0.00011574712271741944, -0.00021495871314750018),
    c64::new(0.00011558221741998084, -0.00021504742684271046),
    c64::new(0.00011541724412850168, -0.00021513601403106005),
    c64::new(0.00011525220294003141, -0.0002152244746604355),
    c64::new(0.00011508709395165962, -0.0002153128086787976),
    c64::new(0.00011492191726051556, -0.0002154010160341818),
    c64::new(0.00011475667296376846, -0.00021548909667469795),
    c64::new(0.00011459136115862735, -0.00021557705054853045),
    c64::new(0.00011442598194234081, -0.00021566487760393837),
    c64::new(0.00011426053541219722, -0.0002157525777892553),
    c64::new(0.00011409502166552454, -0.0002158401510528894),
    c64::new(0.00011392944079969018, -0.00021592759734332366),
    c64::new(0.0001137637929121011, -0.00021601491660911567),
    c64::new(0.00011359807810020377, -0.00021610210879889773),
    c64::new(0.00011343229646148384, -0.00021618917386137694),
    c64::new(0.00011326644809346642, -0.00021627611174533515),
    c64::new(0.00011310053309371588, -0.00021636292239962906),
    c64::new(0.00011293455155983565, -0.0002164496057731902),
    c64::new(0.00011276850358946847, -0.00021653616181502492),
    c64::new(0.000112602389280296, -0.00021662259047421464),
    c64::new(0.00011243620873003906, -0.0002167088916999155),
    c64::new(0.00011226996203645742, -0.00021679506544135878),
    c64::new(0.00011210364929734964, -0.00021688111164785067),
    c64::new(0.00011193727061055328, -0.0002169670302687724),
    c64::new(0.00011177082607394465, -0.0002170528212535802),
    c64::new(0.00011160431578543872, -0.00021713848455180555),
    c64::new(0.00011143773984298922, -0.0002172240201130549),
    c64::new(0.00011127109834458856, -0.00021730942788700982),
    c64::new(0.00011110439138826754, -0.00021739470782342722),
    c64::new(0.00011093761907209564, -0.00021747985987213904),
    c64::new(0.00011077078149418065, -0.00021756488398305254),
    c64::new(0.00011060387875266884, -0.00021764978010615024),
    c64::new(0.00011043691094574487, -0.00021773454819148992),
    c64::new(0.00011026987817163148, -0.0002178191881892047),
    c64::new(0.0001101027805285898, -0.0002179037000495031),
    c64::new(0.00010993561811491912, -0.00021798808372266887),
    c64::new(0.00010976839102895669, -0.00021807233915906136),
    c64::new(0.00010960109936907793, -0.00021815646630911524),
    c64::new(0.00010943374323369629, -0.00021824046512334064),
    c64::new(0.00010926632272126294, -0.00021832433555232328),
    c64::new(0.00010909883793026717, -0.00021840807754672426),
    c64::new(0.00010893128895923587, -0.00021849169105728043),
    c64::new(0.00010876367590673382, -0.0002185751760348041),
    c64::new(0.00010859599887136352, -0.0002186585324301831),
    c64::new(0.00010842825795176496, -0.00021874176019438113),
    c64::new(0.00010826045324661583, -0.00021882485927843742),
    c64::new(0.00010809258485463138, -0.00021890782963346683),
    c64::new(0.00010792465287456415, -0.00021899067121066015),
    c64::new(0.00010775665740520425, -0.00021907338396128377),
    c64::new(0.00010758859854537915, -0.00021915596783667987),
    c64::new(0.00010742047639395343, -0.00021923842278826657),
    c64::new(0.00010725229104982912, -0.00021932074876753765),
    c64::new(0.00010708404261194524, -0.00021940294572606294),
    c64::new(0.00010691573117927804, -0.00021948501361548805),
    c64::new(0.00010674735685084087, -0.00021956695238753454),
    c64::new(0.0001065789197256839, -0.00021964876199399998),
    c64::new(0.0001064104199028944, -0.00021973044238675788),
    c64::new(0.00010624185748159653, -0.00021981199351775771),
    c64::new(0.0001060732325609511, -0.0002198934153390251),
    c64::new(0.00010590454524015589, -0.0002199747078026617),
    c64::new(0.00010573579561844532, -0.00022005587086084522),
    c64::new(0.00010556698379509037, -0.00022013690446582952),
    c64::new(0.00010539810986939877, -0.0002202178085699446),
    c64::new(0.00010522917394071462, -0.00022029858312559673),
    c64::new(0.0001050601761084186, -0.00022037922808526827),
    c64::new(0.00010489111647192786, -0.00022045974340151786),
    c64::new(0.00010472199513069569, -0.00022054012902698042),
    c64::new(0.00010455281218421189, -0.00022062038491436718),
    c64::new(0.00010438356773200249, -0.00022070051101646566),
    c64::new(0.00010421426187362951, -0.00022078050728613973),
    c64::new(0.0001040448947086913, -0.00022086037367632965),
    c64::new(0.00010387546633682224, -0.00022094011014005206),
    c64::new(0.00010370597685769258, -0.0002210197166304001),
    c64::new(0.00010353642637100871, -0.00022109919310054325),
    c64::new(0.00010336681497651272, -0.00022117853950372765),
    c64::new(0.00010319714277398265, -0.0002212577557932758),
    c64::new(0.00010302740986323237, -0.00022133684192258675),
    c64::new(0.00010285761634411125, -0.00022141579784513624),
    c64::new(0.00010268776231650452, -0.00022149462351447652),
    c64::new(0.00010251784788033294, -0.00022157331888423645),
    c64::new(0.00010234787313555272, -0.00022165188390812162),
    c64::new(0.00010217783818215569, -0.00022173031853991423),
    c64::new(0.00010200774312016904, -0.00022180862273347317),
    c64::new(0.00010183758804965524, -0.00022188679644273416),
    c64::new(0.00010166737307071224, -0.00022196483962170957),
    c64::new(0.000101497098283473, -0.00022204275222448863),
    c64::new(0.00010132676378810587, -0.00022212053420523736),
    c64::new(0.00010115636968481426, -0.00022219818551819862),
    c64::new(0.00010098591607383656, -0.00022227570611769217),
    c64::new(0.0001008154030554463, -0.00022235309595811464),
    c64::new(0.0001006448307299519, -0.00022243035499393951),
    c64::new(0.0001004741991976966, -0.00022250748317971736),
    c64::new(0.00010030350855905857, -0.00022258448047007565),
    c64::new(0.00010013275891445077, -0.0002226613468197188),
    c64::new(9.99619503643207e-05, -0.0002227380821834284),
    c64::new(9.979108300915076e-05, -0.00022281468651606296),
    c64::new(9.96201569494577e-05, -0.00022289115977255813),
    c64::new(9.944917228579296e-05, -0.0002229675019079267),
    c64::new(9.927812911874247e-05, -0.0002230437128772585),
    c64::new(9.91070275489264e-05, -0.00022311979263572063),
    c64::new(9.893586767699946e-05, -0.0002231957411385573),
    c64::new(9.876464960365066e-05, -0.00022327155834108995),
    c64::new(9.859337342960306e-05, -0.00022334724419871733),
    c64::new(9.842203925561407e-05, -0.00022342279866691535),
    c64::new(9.825064718247523e-05, -0.00022349822170123726),
    c64::new(9.807919731101196e-05, -0.00022357351325731365),
    c64::new(9.790768974208393e-05, -0.00022364867329085244),
    c64::new(9.773612457658451e-05, -0.00022372370175763895),
    c64::new(9.756450191544111e-05, -0.00022379859861353582),
    c64::new(9.7392821859615e-05, -0.00022387336381448313),
    c64::new(9.722108451010098e-05, -0.0002239479973164985),
    c64::new(9.704928996792781e-05, -0.000224022499075677),
    c64::new(9.687743833415782e-05, -0.000224096869048191),
    c64::new(9.67055297098868e-05, -0.00022417110719029075),
    c64::new(9.653356419624421e-05, -0.00022424521345830377),
    c64::new(9.636154189439299e-05, -0.00022431918780863524),
    c64::new(9.618946290552932e-05, -0.000224393030197768),
    c64::new(9.601732733088295e-05, -0.00022446674058226244),
    c64::new(9.584513527171668e-05, -0.00022454031891875668),
    c64::new(9.567288682932675e-05, -0.00022461376516396645),
    c64::new(9.550058210504248e-05, -0.00022468707927468518),
    c64::new(9.532822120022623e-05, -0.00022476026120778414),
    c64::new(9.515580421627354e-05, -0.00022483331092021229),
    c64::new(9.498333125461291e-05, -0.00022490622836899627),
    c64::new(9.481080241670564e-05, -0.0002249790135112407),
    c64::new(9.463821780404606e-05, -0.00022505166630412796),
    c64::new(9.44655775181613e-05, -0.00022512418670491823),
    c64::new(9.429288166061108e-05, -0.0002251965746709497),
    c64::new(9.412013033298804e-05, -0.0002252688301596383),
    c64::new(9.394732363691722e-05, -0.0002253409531284781),
    c64::new(9.37744616740564e-05, -0.00022541294353504093),
    c64::new(9.360154454609589e-05, -0.0002254848013369767),
    c64::new(9.342857235475826e-05, -0.00022555652649201336),
    c64::new(9.325554520179864e-05, -0.00022562811895795684),
    c64::new(9.308246318900451e-05, -0.00022569957869269107),
    c64::new(9.290932641819545e-05, -0.0002257709056541782),
    c64::new(9.273613499122341e-05, -0.0002258420998004584),
    c64::new(9.256288900997252e-05, -0.00022591316108964998),
    c64::new(9.238958857635878e-05, -0.0002259840894799494),
    c64::new(9.221623379233052e-05, -0.00022605488492963134),
    c64::new(9.204282475986775e-05, -0.00022612554739704868),
    c64::new(9.186936158098261e-05, -0.00022619607684063245),
    c64::new(9.169584435771907e-05, -0.00022626647321889203),
    c64::new(9.152227319215273e-05, -0.00022633673649041506),
    c64::new(9.13486481863911e-05, -0.00022640686661386746),
    c64::new(9.117496944257334e-05, -0.00022647686354799345),
    c64::new(9.100123706287007e-05, -0.0002265467272516157),
    c64::new(9.082745114948363e-05, -0.00022661645768363512),
    c64::new(9.065361180464785e-05, -0.00022668605480303112),
    c64::new(9.047971913062785e-05, -0.00022675551856886158),
    c64::new(9.030577322972027e-05, -0.00022682484894026263),
    c64::new(9.013177420425293e-05, -0.0002268940458764491),
    c64::new(8.995772215658504e-05, -0.00022696310933671416),
    c64::new(8.978361718910695e-05, -0.00022703203928042958),
    c64::new(8.960945940424005e-05, -0.00022710083566704567),
    c64::new(8.943524890443691e-05, -0.00022716949845609123),
    c64::new(8.926098579218114e-05, -0.00022723802760717376),
    c64::new(8.908667016998714e-05, -0.00022730642307997932),
    c64::new(8.891230214040035e-05, -0.0002273746848342726),
    c64::new(8.873788180599704e-05, -0.00022744281282989693),
    c64::new(8.856340926938409e-05, -0.00022751080702677445),
    c64::new(8.83888846331993e-05, -0.00022757866738490584),
    c64::new(8.821430800011092e-05, -0.0002276463938643707),
    c64::new(8.803967947281794e-05, -0.0002277139864253272),
    c64::new(8.786499915404987e-05, -0.0002277814450280124),
    c64::new(8.769026714656654e-05, -0.0002278487696327422),
    c64::new(8.751548355315834e-05, -0.00022791596019991117),
    c64::new(8.734064847664598e-05, -0.0002279830166899929),
    c64::new(8.716576201988038e-05, -0.00022804993906353976),
    c64::new(8.699082428574276e-05, -0.00022811672728118308),
    c64::new(8.681583537714452e-05, -0.000228183381303633),
    c64::new(8.664079539702705e-05, -0.00022824990109167873),
    c64::new(8.646570444836194e-05, -0.00022831628660618835),
    c64::new(8.62905626341506e-05, -0.00022838253780810904),
    c64::new(8.61153700574245e-05, -0.0002284486546584669),
    c64::new(8.594012682124493e-05, -0.00022851463711836702),
    c64::new(8.576483302870292e-05, -0.00022858048514899372),
    c64::new(8.558948878291931e-05, -0.00022864619871161023),
    c64::new(8.541409418704468e-05, -0.00022871177776755897),
    c64::new(8.523864934425904e-05, -0.00022877722227826152),
    c64::new(8.50631543577721e-05, -0.0002288425322052185),
    c64::new(8.48876093308231e-05, -0.00022890770751000976),
    c64::new(8.471201436668059e-05, -0.0002289727481542944),
    c64::new(8.453636956864261e-05, -0.00022903765409981063),
    c64::new(8.436067504003641e-05, -0.000229102425308376),
    c64::new(8.418493088421858e-05, -0.00022916706174188725),
    c64::new(8.400913720457495e-05, -0.00022923156336232043),
    c64::new(8.383329410452029e-05, -0.00022929593013173093),
    c64::new(8.365740168749863e-05, -0.0002293601620122534),
    c64::new(8.348146005698298e-05, -0.0002294242589661019),
    c64::new(8.33054693164752e-05, -0.0002294882209555698),
    c64::new(8.312942956950613e-05, -0.00022955204794303),
    c64::new(8.295334091963549e-05, -0.00022961573989093464),
    c64::new(8.277720347045157e-05, -0.00022967929676181545),
    c64::new(8.26010173255716e-05, -0.0002297427185182835),
    c64::new(8.242478258864128e-05, -0.00022980600512302946),
    c64::new(8.224849936333497e-05, -0.00022986915653882344),
    c64::new(8.207216775335562e-05, -0.00022993217272851505),
    c64::new(8.189578786243449e-05, -0.00022999505365503352),
    c64::new(8.171935979433135e-05, -0.00023005779928138764),
    c64::new(8.154288365283435e-05, -0.00023012040957066574),
    c64::new(8.136635954175975e-05, -0.00023018288448603582),
    c64::new(8.118978756495219e-05, -0.00023024522399074548),
    c64::new(8.101316782628448e-05, -0.000230307428048122),
    c64::new(8.083650042965733e-05, -0.00023036949662157236),
    c64::new(8.065978547899973e-05, -0.00023043142967458318),
    c64::new(8.048302307826844e-05, -0.00023049322717072086),
    c64::new(8.030621333144829e-05, -0.00023055488907363153),
    c64::new(8.012935634255192e-05, -0.00023061641534704108),
    c64::new(7.995245221561964e-05, -0.00023067780595475518),
    c64::new(7.977550105471966e-05, -0.00023073906086065933),
    c64::new(7.959850296394784e-05, -0.00023080018002871882),
    c64::new(7.942145804742749e-05, -0.00023086116342297883),
    c64::new(7.92443664093096e-05, -0.00023092201100756443),
    c64::new(7.90672281537727e-05, -0.0002309827227466805),
    c64::new(7.889004338502254e-05, -0.00023104329860461196),
    c64::new(7.871281220729247e-05, -0.00023110373854572352),
    c64::new(7.853553472484293e-05, -0.00023116404253445996),
    c64::new(7.835821104196172e-05, -0.000231224210535346),
    c64::new(7.818084126296387e-05, -0.0002312842425129863),
    c64::new(7.800342549219134e-05, -0.0002313441384320657),
    c64::new(7.78259638340133e-05, -0.00023140389825734888),
    c64::new(7.764845639282594e-05, -0.00023146352195368072),
    c64::new(7.74709032730522e-05, -0.0002315230094859861),
    c64::new(7.729330457914205e-05, -0.00023158236081927008),
    c64::new(7.711566041557228e-05, -0.00023164157591861777),
    c64::new(7.69379708868463e-05, -0.00023170065474919447),
    c64::new(7.676023609749434e-05, -0.0002317595972762456),
    c64::new(7.658245615207312e-05, -0.00023181840346509684),
    c64::new(7.640463115516606e-05, -0.00023187707328115397),
    c64::new(7.622676121138307e-05, -0.00023193560668990308),
    c64::new(7.604884642536034e-05, -0.00023199400365691046),
    c64::new(7.587088690176061e-05, -0.00023205226414782268),
    c64::new(7.569288274527295e-05, -0.0002321103881283666),
    c64::new(7.551483406061248e-05, -0.00023216837556434939),
    c64::new(7.533674095252076e-05, -0.0002322262264216585),
    c64::new(7.515860352576538e-05, -0.0002322839406662618),
    c64::new(7.498042188513992e-05, -0.0002323415182642075),
    c64::new(7.480219613546413e-05, -0.00023239895918162408),
    c64::new(7.462392638158355e-05, -0.00023245626338472066),
    c64::new(7.444561272836972e-05, -0.0002325134308397866),
    c64::new(7.426725528072004e-05, -0.0002325704615131917),
    c64::new(7.408885414355746e-05, -0.0002326273553713864),
    c64::new(7.391040942183085e-05, -0.00023268411238090142),
    c64::new(7.37319212205147e-05, -0.00023274073250834808),
    c64::new(7.355338964460889e-05, -0.00023279721572041828),
    c64::new(7.337481479913901e-05, -0.00023285356198388436),
    c64::new(7.319619678915612e-05, -0.00023290977126559924),
    c64::new(7.301753571973644e-05, -0.0002329658435324965),
    c64::new(7.283883169598183e-05, -0.0002330217787515902),
    c64::new(7.266008482301912e-05, -0.00023307757688997518),
    c64::new(7.248129520600055e-05, -0.0002331332379148268),
    c64::new(7.23024629501035e-05, -0.00023318876179340104),
    c64::new(7.212358816053029e-05, -0.00023324414849303473),
    c64::new(7.194467094250838e-05, -0.0002332993979811453),
    c64::new(7.176571140129023e-05, -0.00023335451022523081),
    c64::new(7.1586709642153e-05, -0.00023340948519287022),
    c64::new(7.140766577039888e-05, -0.00023346432285172318),
    c64::new(7.122857989135483e-05, -0.00023351902316953005),
    c64::new(7.104945211037234e-05, -0.00023357358611411207),
    c64::new(7.08702825328277e-05, -0.00023362801165337132),
    c64::new(7.069107126412186e-05, -0.00023368229975529055),
    c64::new(7.051181840968005e-05, -0.00023373645038793356),
    c64::new(7.033252407495222e-05, -0.00023379046351944488),
    c64::new(7.015318836541249e-05, -0.00023384433911805003),
    c64::new(6.997381138655952e-05, -0.00023389807715205532),
    c64::new(6.979439324391619e-05, -0.00023395167758984806),
    c64::new(6.961493404302947e-05, -0.0002340051403998965),
    c64::new(6.943543388947066e-05, -0.0002340584655507499),
    c64::new(6.92558928888351e-05, -0.00023411165301103837),
    c64::new(6.907631114674204e-05, -0.00023416470274947312),
    c64::new(6.889668876883484e-05, -0.00023421761473484638),
    c64::new(6.871702586078077e-05, -0.00023427038893603138),
    c64::new(6.85373225282708e-05, -0.0002343230253219824),
    c64::new(6.835757887701987e-05, -0.00023437552386173483),
    c64::new(6.817779501276642e-05, -0.00023442788452440513),
    c64::new(6.799797104127272e-05, -0.00023448010727919087),
    c64::new(6.781810706832465e-05, -0.00023453219209537072),
    c64::new(6.763820319973143e-05, -0.0002345841389423046),
    c64::new(6.745825954132593e-05, -0.00023463594778943344),
    c64::new(6.727827619896442e-05, -0.00023468761860627945),
    c64::new(6.709825327852635e-05, -0.00023473915136244605),
    c64::new(6.691819088591463e-05, -0.0002347905460276178),
    c64::new(6.673808912705537e-05, -0.00023484180257156057),
    c64::new(6.65579481078977e-05, -0.00023489292096412148),
    c64::new(6.637776793441403e-05, -0.00023494390117522886),
    c64::new(6.619754871259963e-05, -0.00023499474317489237),
    c64::new(6.601729054847286e-05, -0.00023504544693320298),
    c64::new(6.5836993548075e-05, -0.00023509601242033298),
    c64::new(6.565665781747005e-05, -0.00023514643960653598),
    c64::new(6.54762834627449e-05, -0.00023519672846214698),
    c64::new(6.529587059000921e-05, -0.00023524687895758233),
    c64::new(6.511541930539512e-05, -0.0002352968910633398),
    c64::new(6.49349297150575e-05, -0.00023534676474999856),
    c64::new(6.47544019251738e-05, -0.00023539649998821917),
    c64::new(6.457383604194376e-05, -0.00023544609674874372),
    c64::new(6.439323217158972e-05, -0.00023549555500239567),
    c64::new(6.421259042035622e-05, -0.00023554487472008005),
    c64::new(6.403191089451015e-05, -0.00023559405587278328),
    c64::new(6.385119370034073e-05, -0.00023564309843157342),
    c64::new(6.367043894415907e-05, -0.00023569200236759995),
    c64::new(6.348964673229862e-05, -0.00023574076765209395),
    c64::new(6.330881717111483e-05, -0.0002357893942563681),
    c64::new(6.312795036698496e-05, -0.00023583788215181655),
    c64::new(6.294704642630833e-05, -0.0002358862313099152),
    c64::new(6.276610545550616e-05, -0.00023593444170222138),
    c64::new(6.258512756102122e-05, -0.0002359825133003743),
    c64::new(6.240411284931825e-05, -0.00023603044607609455),
    c64::new(6.222306142688345e-05, -0.0002360782400011846),
    c64::new(6.204197340022476e-05, -0.00023612589504752848),
    c64::new(6.186084887587165e-05, -0.00023617341118709192),
    c64::new(6.16796879603749e-05, -0.00023622078839192245),
    c64::new(6.149849076030686e-05, -0.0002362680266341493),
    c64::new(6.131725738226124e-05, -0.00023631512588598335),
    c64::new(6.113598793285285e-05, -0.0002363620861197174),
    c64::new(6.09546825187179e-05, -0.00023640890730772588),
    c64::new(6.0773341246513735e-05, -0.00023645558942246515),
    c64::new(6.059196422291865e-05, -0.00023650213243647333),
    c64::new(6.0410551554632186e-05, -0.00023654853632237033),
    c64::new(6.022910334837463e-05, -0.00023659480105285794),
    c64::new(6.004761971088735e-05, -0.00023664092660071986),
    c64::new(5.986610074893254e-05, -0.00023668691293882152),
    c64::new(5.968454656929302e-05, -0.00023673276004011046),
    c64::new(5.9502957278772494e-05, -0.00023677846787761593),
    c64::new(5.932133298419531e-05, -0.0002368240364244492),
    c64::new(5.913967379240626e-05, -0.0002368694656538035),
    c64::new(5.895797981027082e-05, -0.00023691475553895398),
    c64::new(5.877625114467493e-05, -0.00023695990605325772),
    c64::new(5.859448790252478e-05, -0.0002370049171701539),
    c64::new(5.84126901907471e-05, -0.00023704978886316357),
    c64::new(5.8230858116288696e-05, -0.00023709452110588992),
    c64::new(5.8048991786116745e-05, -0.0002371391138720181),
    c64::new(5.786709130721859e-05, -0.00023718356713531538),
    c64::new(5.768515678660147e-05, -0.000237227880869631),
    c64::new(5.750318833129284e-05, -0.00023727205504889634),
    c64::new(5.73211860483401e-05, -0.00023731608964712485),
    c64::new(5.7139150044810415e-05, -0.00023735998463841218),
    c64::new(5.695708042779091e-05, -0.00023740373999693598),
    c64::new(5.677497730438851e-05, -0.0002374473556969561),
    c64::new(5.659284078172968e-05, -0.0002374908317128146),
    c64::new(5.6410670966960754e-05, -0.00023753416801893556),
    c64::new(5.622846796724744e-05, -0.00023757736458982547),
    c64::new(5.6046231889775114e-05, -0.00023762042140007281),
    c64::new(5.586396284174862e-05, -0.00023766333842434838),
    c64::new(5.568166093039204e-05, -0.00023770611563740524),
    c64::new(5.5499326262948934e-05, -0.00023774875301407864),
    c64::new(5.5316958946682156e-05, -0.0002377912505292861),
    c64::new(5.5134559088873584e-05, -0.00023783360815802742),
    c64::new(5.495212679682441e-05, -0.00023787582587538473),
    c64::new(5.4769662177854904e-05, -0.0002379179036565224),
    c64::new(5.4587165339304196e-05, -0.00023795984147668717),
    c64::new(5.440463638853059e-05, -0.0002380016393112081),
    c64::new(5.422207543291103e-05, -0.0002380432971354966),
    c64::new(5.403948257984151e-05, -0.00023808481492504648),
    c64::new(5.385685793673673e-05, -0.0002381261926554338),
    c64::new(5.367420161102997e-05, -0.00023816743030231723),
    c64::new(5.3491513710173284e-05, -0.00023820852784143764),
    c64::new(5.330879434163731e-05, -0.00023824948524861842),
    c64::new(5.312604361291105e-05, -0.0002382903024997654),
    c64::new(5.294326163150209e-05, -0.00023833097957086683),
    c64::new(5.2760448504936427e-05, -0.00023837151643799344),
    c64::new(5.257760434075819e-05, -0.00023841191307729845),
    c64::new(5.239472924653001e-05, -0.00023845216946501754),
    c64::new(5.221182332983248e-05, -0.0002384922855774689),
    c64::new(5.202888669826449e-05, -0.00023853226139105332),
    c64::new(5.1845919459443007e-05, -0.00023857209688225397),
    c64::new(5.1662921721002834e-05, -0.0002386117920276367),
    c64::new(5.147989359059688e-05, -0.00023865134680384992),
    c64::new(5.129683517589593e-05, -0.0002386907611876245),
    c64::new(5.1113746584588426e-05, -0.00023873003515577403),
    c64::new(5.093062792438072e-05, -0.00023876916868519461),
    c64::new(5.074747930299686e-05, -0.000238808161752865),
    c64::new(5.056430082817836e-05, -0.0002388470143358466),
    c64::new(5.038109260768449e-05, -0.0002388857264112834),
    c64::new(5.019785474929183e-05, -0.00023892429795640212),
    c64::new(5.0014587360794536e-05, -0.00023896272894851213),
    c64::new(4.983129055000415e-05, -0.00023900101936500542),
    c64::new(4.964796442474936e-05, -0.00023903916918335675),
    c64::new(4.9464609092876247e-05, -0.00023907717838112354),
    c64::new(4.928122466224808e-05, -0.000239115046935946),
    c64::new(4.90978112407451e-05, -0.000239152774825547),
    c64::new(4.8914368936264744e-05, -0.00023919036202773222),
    c64::new(4.873089785672145e-05, -0.00023922780852039005),
    c64::new(4.8547398110046415e-05, -0.00023926511428149172),
    c64::new(4.836386980418792e-05, -0.00023930227928909114),
    c64::new(4.818031304711084e-05, -0.00023933930352132519),
    c64::new(4.799672794679693e-05, -0.00023937618695641342),
    c64::new(4.7813114611244626e-05, -0.00023941292957265823),
    c64::new(4.7629473148468824e-05, -0.00023944953134844493),
    c64::new(4.744580366650112e-05, -0.0002394859922622416),
    c64::new(4.726210627338958e-05, -0.00023952231229259927),
    c64::new(4.707838107719855e-05, -0.0002395584914181518),
    c64::new(4.689462818600888e-05, -0.00023959452961761586),
    c64::new(4.671084770791772e-05, -0.00023963042686979118),
    c64::new(4.652703975103827e-05, -0.00023966618315356033),
    c64::new(4.634320442350014e-05, -0.0002397017984478888),
    c64::new(4.61593418334488e-05, -0.000239737272731825),
    c64::new(4.597545208904593e-05, -0.0002397726059845004),
    c64::new(4.5791535298469175e-05, -0.00023980779818512924),
    c64::new(4.5607591569911945e-05, -0.00023984284931300896),
    c64::new(4.5423621011583644e-05, -0.00023987775934751983),
    c64::new(4.5239623731709475e-05, -0.0002399125282681252),
    c64::new(4.5055599838530187e-05, -0.0002399471560543714),
    c64::new(4.487154944030234e-05, -0.0002399816426858878),
    c64::new(4.4687472645298104e-05, -0.0002400159881423868),
    c64::new(4.450336956180501e-05, -0.00024005019240366384),
    c64::new(4.431924029812625e-05, -0.00024008425544959746),
    c64::new(4.413508496258024e-05, -0.00024011817726014923),
    c64::new(4.395090366350086e-05, -0.00024015195781536383),
    c64::new(4.3766696509237276e-05, -0.00024018559709536902),
    c64::new(4.35824636081537e-05, -0.00024021909508037571),
    c64::new(4.339820506862967e-05, -0.00024025245175067784),
    c64::new(4.321392099905978e-05, -0.0002402856670866526),
    c64::new(4.30296115078535e-05, -0.00024031874106876023),
    c64::new(4.284527670343539e-05, -0.00024035167367754417),
    c64::new(4.266091669424495e-05, -0.000240384464893631),
    c64::new(4.247653158873629e-05, -0.00024041711469773052),
    c64::new(4.229212149537854e-05, -0.00024044962307063567),
    c64::new(4.210768652265528e-05, -0.00024048198999322262),
    c64::new(4.192322677906493e-05, -0.00024051421544645074),
    c64::new(4.173874237312045e-05, -0.0002405462994113626),
    c64::new(4.155423341334915e-05, -0.0002405782418690841),
    c64::new(4.136970000829295e-05, -0.00024061004280082423),
    c64::new(4.1185142266508164e-05, -0.00024064170218787538),
    c64::new(4.1000560296565234e-05, -0.00024067322001161313),
    c64::new(4.081595420704904e-05, -0.00024070459625349637),
    c64::new(4.0631324106558646e-05, -0.00024073583089506725),
    c64::new(4.0446670103707067e-05, -0.00024076692391795124),
    c64::new(4.02619923071216e-05, -0.0002407978753038571),
    c64::new(4.0077290825443334e-05, -0.00024082868503457692),
    c64::new(3.989256576732745e-05, -0.00024085935309198618),
    c64::new(3.970781724144298e-05, -0.00024088987945804363),
    c64::new(3.952304535647263e-05, -0.00024092026411479136),
    c64::new(3.933825022111298e-05, -0.0002409505070443549),
    c64::new(3.9153431944074315e-05, -0.0002409806082289431),
    c64::new(3.8968590634080344e-05, -0.00024101056765084823),
    c64::new(3.8783726399868503e-05, -0.0002410403852924459),
    c64::new(3.859883935018972e-05, -0.00024107006113619517),
    c64::new(3.841392959380817e-05, -0.00024109959516463848),
    c64::new(3.822899723950161e-05, -0.00024112898736040178),
    c64::new(3.804404239606087e-05, -0.00024115823770619438),
    c64::new(3.785906517229016e-05, -0.00024118734618480903),
    c64::new(3.7674065677006895e-05, -0.00024121631277912195),
    c64::new(3.748904401904138e-05, -0.0002412451374720929),
    c64::new(3.730400030723716e-05, -0.000241273820246765),
    c64::new(3.711893465045072e-05, -0.00024130236108626496),
    c64::new(3.693384715755133e-05, -0.00024133075997380288),
    c64::new(3.674873793742123e-05, -0.00024135901689267247),
    c64::new(3.6563607098955474e-05, -0.00024138713182625092),
    c64::new(3.637845475106164e-05, -0.0002414151047579989),
    c64::new(3.6193281002660204e-05, -0.0002414429356714607),
    c64::new(3.6008085962684e-05, -0.00024147062455026405),
    c64::new(3.582286974007855e-05, -0.00024149817137812036),
    c64::new(3.563763244380184e-05, -0.0002415255761388245),
    c64::new(3.54523741828241e-05, -0.00024155283881625497),
    c64::new(3.526709506612804e-05, -0.00024157995939437383),
    c64::new(3.508179520270864e-05, -0.00024160693785722675),
    c64::new(3.4896474701572924e-05, -0.000241633774188943),
    c64::new(3.4711133671740235e-05, -0.00024166046837373548),
    c64::new(3.4525772222241975e-05, -0.00024168702039590063),
    c64::new(3.4340390462121396e-05, -0.00024171343023981863),
    c64::new(3.4154988500433924e-05, -0.00024173969788995322),
    c64::new(3.396956644624665e-05, -0.00024176582333085183),
    c64::new(3.3784124408638644e-05, -0.00024179180654714555),
    c64::new(3.359866249670073e-05, -0.00024181764752354907),
    c64::new(3.341318081953526e-05, -0.00024184334624486088),
    c64::new(3.322767948625639e-05, -0.00024186890269596305),
    c64::new(3.3042158605989826e-05, -0.00024189431686182138),
    c64::new(3.285661828787261e-05, -0.00024191958872748535),
    c64::new(3.267105864105339e-05, -0.00024194471827808823),
    c64::new(3.248547977469218e-05, -0.00024196970549884687),
    c64::new(3.229988179796013e-05, -0.00024199455037506203),
    c64::new(3.211426482003986e-05, -0.00024201925289211803),
    c64::new(3.192862895012495e-05, -0.00024204381303548304),
    c64::new(3.174297429742025e-05, -0.00024206823079070896),
    c64::new(3.155730097114167e-05, -0.0002420925061434315),
    c64::new(3.1371609080515924e-05, -0.00024211663907937),
    c64::new(3.118589873478083e-05, -0.00024214062958432774),
    c64::new(3.100017004318505e-05, -0.00024216447764419173),
    c64::new(3.081442311498789e-05, -0.00024218818324493274),
    c64::new(3.062865805945952e-05, -0.00024221174637260542),
    c64::new(3.04428749858808e-05, -0.00024223516701334813),
    c64::new(3.0257074003543014e-05, -0.00024225844515338315),
    c64::new(3.0071255221748202e-05, -0.00024228158077901652),
    c64::new(2.9885418749808662e-05, -0.00024230457387663818),
    c64::new(2.969956469704725e-05, -0.00024232742443272188),
    c64::new(2.9513693172797173e-05, -0.0002423501324338252),
    c64::new(2.932780428640175e-05, -0.00024237269786658963),
    c64::new(2.9141898147214683e-05, -0.0002423951207177405),
    c64::new(2.8955974864599815e-05, -0.00024241740097408703),
    c64::new(2.8770034547930913e-05, -0.00024243953862252232),
    c64::new(2.858407730659193e-05, -0.00024246153365002334),
    c64::new(2.8398103249976773e-05, -0.000242483386043651),
    c64::new(2.8212112487489084e-05, -0.00024250509579055012),
    c64::new(2.802610512854254e-05, -0.00024252666287794938),
    c64::new(2.7840081282560372e-05, -0.00024254808729316142),
    c64::new(2.765404105897567e-05, -0.00024256936902358284),
    c64::new(2.746798456723116e-05, -0.00024259050805669412),
    c64::new(2.728191191677897e-05, -0.00024261150438005972),
    c64::new(2.7095823217080902e-05, -0.00024263235798132804),
    c64::new(2.69097185776082e-05, -0.00024265306884823146),
    c64::new(2.6723598107841315e-05, -0.00024267363696858627),
    c64::new(2.6537461917270184e-05, -0.00024269406233029281),
    c64::new(2.6351310115393985e-05, -0.00024271434492133538),
    c64::new(2.6165142811720916e-05, -0.0002427344847297822),
    c64::new(2.5978960115768516e-05, -0.0002427544817437856),
    c64::new(2.579276213706316e-05, -0.0002427743359515818),
    c64::new(2.5606548985140373e-05, -0.0002427940473414911),
    c64::new(2.542032076954461e-05, -0.00024281361590191783),
    c64::new(2.5234077599829024e-05, -0.00024283304162135027),
    c64::new(2.504781958555573e-05, -0.00024285232448836077),
    c64::new(2.486154683629557e-05, -0.00024287146449160574),
    c64::new(2.4675259461627896e-05, -0.0002428904616198256),
    c64::new(2.4488957571140826e-05, -0.00024290931586184482),
    c64::new(2.4302641274431013e-05, -0.00024292802720657193),
    c64::new(2.411631068110343e-05, -0.00024294659564299954),
    c64::new(2.3929965900771672e-05, -0.0002429650211602043),
    c64::new(2.374360704305746e-05, -0.00024298330374734702),
    c64::new(2.3557234217590965e-05, -0.00024300144339367243),
    c64::new(2.3370847534010575e-05, -0.0002430194400885095),
    c64::new(2.318444710196266e-05, -0.00024303729382127123),
    c64::new(2.2998033031101853e-05, -0.0002430550045814547),
    c64::new(2.28116054310908e-05, -0.0002430725723586412),
    c64::new(2.262516441159994e-05, -0.00024308999714249598),
    c64::new(2.2438710082307787e-05, -0.0002431072789227685),
    c64::new(2.2252242552900663e-05, -0.00024312441768929239),
    c64::new(2.2065761933072512e-05, -0.00024314141343198528),
    c64::new(2.1879268332525126e-05, -0.00024315826614084907),
    c64::new(2.169276186096794e-05, -0.00024317497580596972),
    c64::new(2.15062426281178e-05, -0.00024319154241751737),
    c64::new(2.131971074369927e-05, -0.0002432079659657463),
    c64::new(2.113316631744415e-05, -0.000243224246440995),
    c64::new(2.0946609459091768e-05, -0.00024324038383368604),
    c64::new(2.076004027838877e-05, -0.00024325637813432622),
    c64::new(2.0573458885088895e-05, -0.0002432722293335065),
    c64::new(2.0386865388953224e-05, -0.00024328793742190206),
    c64::new(2.0200259899749952e-05, -0.00024330350239027218),
    c64::new(2.0013642527254173e-05, -0.00024331892422946042),
    c64::new(1.982701338124814e-05, -0.00024333420293039455),
    c64::new(1.964037257152102e-05, -0.00024334933848408643),
    c64::new(1.9453720207868683e-05, -0.00024336433088163224),
    c64::new(1.9267056400094017e-05, -0.00024337918011421237),
    c64::new(1.908038125800642e-05, -0.00024339388617309135),
    c64::new(1.8893694891422128e-05, -0.000243408449049618),
    c64::new(1.870699741016397e-05, -0.00024342286873522533),
    c64::new(1.8520288924061155e-05, -0.00024343714522143066),
    c64::new(1.833356954294953e-05, -0.0002434512784998355),
    c64::new(1.8146839376671354e-05, -0.00024346526856212553),
    c64::new(1.7960098535075062e-05, -0.00024347911540007085),
    c64::new(1.777334712801553e-05, -0.0002434928190055257),
    c64::new(1.758658526535386e-05, -0.00024350637937042854),
    c64::new(1.7399813056957133e-05, -0.00024351979648680222),
    c64::new(1.7213030612698733e-05, -0.00024353307034675377),
    c64::new(1.7026238042457855e-05, -0.00024354620094247455),
    c64::new(1.6839435456119808e-05, -0.00024355918826624013),
    c64::new(1.6652622963575794e-05, -0.0002435720323104104),
    c64::new(1.6465800674722682e-05, -0.0002435847330674296),
    c64::new(1.6278968699463264e-05, -0.00024359729052982613),
    c64::new(1.6092127147706034e-05, -0.0002436097046902128),
    c64::new(1.5905276129364956e-05, -0.00024362197554128666),
    c64::new(1.571841575435972e-05, -0.0002436341030758291),
    c64::new(1.5531546132615525e-05, -0.0002436460872867058),
    c64::new(1.5344667374062847e-05, -0.00024365792816686676),
    c64::new(1.5157779588637756e-05, -0.00024366962570934628),
    c64::new(1.497088288628141e-05, -0.00024368117990726298),
    c64::new(1.4783977376940376e-05, -0.00024369259075381988),
    c64::new(1.4597063170566408e-05, -0.00024370385824230425),
    c64::new(1.4410140377116208e-05, -0.0002437149823660877),
    c64::new(1.4223209106551692e-05, -0.0002437259631186262),
    c64::new(1.4036269468839767e-05, -0.00024373680049346005),
    c64::new(1.3849321573952095e-05, -0.0002437474944842139),
    c64::new(1.3662365531865363e-05, -0.0002437580450845968),
    c64::new(1.3475401452561043e-05, -0.00024376845228840204),
    c64::new(1.3288429446025182e-05, -0.00024377871608950738),
    c64::new(1.3101449622248701e-05, -0.00024378883648187483),
    c64::new(1.2914462091226904e-05, -0.00024379881345955087),
    c64::new(1.2727466962959794e-05, -0.00024380864701666633),
    c64::new(1.2540464347451839e-05, -0.0002438183371474363),
    c64::new(1.2353454354711752e-05, -0.00024382788384616039),
    c64::new(1.2166437094752748e-05, -0.0002438372871072225),
    c64::new(1.1979412677592318e-05, -0.00024384654692509092),
    c64::new(1.1792381213251997e-05, -0.00024385566329431836),
    c64::new(1.1605342811757632e-05, -0.00024386463620954189),
    c64::new(1.141829758313915e-05, -0.00024387346566548298),
    c64::new(1.1231245637430333e-05, -0.00024388215165694748),
    c64::new(1.1044187084669128e-05, -0.00024389069417882564),
    c64::new(1.0857122034897154e-05, -0.00024389909322609213),
    c64::new(1.0670050598160015e-05, -0.00024390734879380598),
    c64::new(1.048297288450707e-05, -0.00024391546087711068),
    c64::new(1.0295889003991211e-05, -0.0002439234294712341),
    c64::new(1.010879906666912e-05, -0.0002439312545714885),
    c64::new(9.921703182601039e-06, -0.0002439389361732706),
    c64::new(9.734601461850548e-06, -0.0002439464742720615),
    c64::new(9.547494014484826e-06, -0.00024395386886342672),
    c64::new(9.360380950574412e-06, -0.00024396111994301622),
    c64::new(9.173262380192997e-06, -0.0002439682275065644),
    c64::new(8.986138413417722e-06, -0.00024397519154989003),
    c64::new(8.799009160328688e-06, -0.00024398201206889635),
    c64::new(8.611874731009264e-06, -0.000243988689059571),
    c64::new(8.424735235545871e-06, -0.00024399522251798616),
    c64::new(8.237590784027745e-06, -0.00024400161244029827),
    c64::new(8.050441486547196e-06, -0.00024400785882274837),
    c64::new(7.86328745319939e-06, -0.00024401396166166185),
    c64::new(7.676128794082115e-06, -0.00024401992095344856),
    c64::new(7.488965619296041e-06, -0.00024402573669460282),
    c64::new(7.301798038944496e-06, -0.00024403140888170337),
    c64::new(7.114626163133238e-06, -0.00024403693751141338),
    c64::new(6.927450101970766e-06, -0.00024404232258048056),
    c64::new(6.740269965567827e-06, -0.00024404756408573697),
    c64::new(6.553085864037727e-06, -0.00024405266202409917),
    c64::new(6.3658979074961034e-06, -0.00024405761639256817),
    c64::new(6.178706206060701e-06, -0.00024406242718822947),
    c64::new(5.991510869851628e-06, -0.000244067094408253),
    c64::new(5.804312008991132e-06, -0.0002440716180498931),
    c64::new(5.617109733603372e-06, -0.0002440759981104887),
    c64::new(5.429904153814678e-06, -0.0002440802345874631),
    c64::new(5.242695379753322e-06, -0.00024408432747832404),
    c64::new(5.055483521549296e-06, -0.00024408827678066387),
    c64::new(4.868268689334619e-06, -0.00024409208249215924),
    c64::new(4.681050993242847e-06, -0.0002440957446105714),
    c64::new(4.49383054340938e-06, -0.00024409926313374595),
    c64::new(4.306607449971239e-06, -0.0002441026380596131),
    c64::new(4.119381823066839e-06, -0.00024410586938618747),
    c64::new(3.932153772836247e-06, -0.0002441089571115681),
    c64::new(3.744923409420952e-06, -0.0002441119012339386),
    c64::new(3.5576908429636485e-06, -0.000244114701751567),
    c64::new(3.3704561836084836e-06, -0.00024411735866280588),
    c64::new(3.1832195415008382e-06, -0.00024411987196609216),
    c64::new(2.995981026787096e-06, -0.0002441222416599474),
    c64::new(2.8087407496149597e-06, -0.00024412446774297754),
    c64::new(2.6214988201329517e-06, -0.00024412655021387303),
    c64::new(2.4342553484907298e-06, -0.0002441284890714088),
    c64::new(2.2470104448388574e-06, -0.0002441302843144443),
    c64::new(2.059764219328579e-06, -0.0002441319359419234),
    c64::new(1.8725167821120794e-06, -0.00024413344395287455),
    c64::new(1.6852682433422552e-06, -0.00024413480834641055),
    c64::new(1.4980187131724893e-06, -0.00024413602912172879),
    c64::new(1.3107683017569098e-06, -0.00024413710627811113),
    c64::new(1.123517119250164e-06, -0.00024413803981492393),
    c64::new(9.362652758071894e-07, -0.00024413882973161797),
    c64::new(7.490128815835298e-07, -0.00024413947602772856),
    c64::new(5.617600467348355e-07, -0.00024413997870287554),
    c64::new(3.745068814171792e-07, -0.00024414033775676315),
    c64::new(1.8725349578682774e-07, -0.00024414055318918023),
    c64::new(1.4949301747404214e-20, -0.000244140625),
    c64::new(-1.8725349578679783e-07, -0.00024414055318918023),
    c64::new(-3.745068814171493e-07, -0.00024414033775676315),
    c64::new(-5.617600467348056e-07, -0.00024413997870287554),
    c64::new(-7.490128815834999e-07, -0.00024413947602772856),
    c64::new(-9.362652758071596e-07, -0.00024413882973161797),
    c64::new(-1.123517119250134e-06, -0.00024413803981492393),
    c64::new(-1.3107683017568797e-06, -0.00024413710627811113),
    c64::new(-1.4980187131724592e-06, -0.00024413602912172879),
    c64::new(-1.6852682433422251e-06, -0.00024413480834641055),
    c64::new(-1.8725167821120493e-06, -0.00024413344395287455),
    c64::new(-2.0597642193285493e-06, -0.0002441319359419234),
    c64::new(-2.2470104448388278e-06, -0.0002441302843144443),
    c64::new(-2.4342553484907e-06, -0.0002441284890714088),
    c64::new(-2.621498820132922e-06, -0.00024412655021387303),
    c64::new(-2.80874074961493e-06, -0.00024412446774297754),
    c64::new(-2.995981026787066e-06, -0.0002441222416599474),
    c64::new(-3.1832195415008086e-06, -0.00024411987196609216),
    c64::new(-3.370456183608454e-06, -0.00024411735866280588),
    c64::new(-3.557690842963619e-06, -0.000244114701751567),
    c64::new(-3.7449234094209225e-06, -0.0002441119012339386),
    c64::new(-3.932153772836216e-06, -0.0002441089571115681),
    c64::new(-4.119381823066809e-06, -0.00024410586938618747),
    c64::new(-4.306607449971209e-06, -0.0002441026380596131),
    c64::new(-4.4938305434093494e-06, -0.00024409926313374595),
    c64::new(-4.681050993242817e-06, -0.0002440957446105714),
    c64::new(-4.868268689334589e-06, -0.00024409208249215924),
    c64::new(-5.055483521549265e-06, -0.00024408827678066387),
    c64::new(-5.242695379753292e-06, -0.00024408432747832404),
    c64::new(-5.429904153814647e-06, -0.0002440802345874631),
    c64::new(-5.617109733603341e-06, -0.0002440759981104887),
    c64::new(-5.804312008991101e-06, -0.0002440716180498931),
    c64::new(-5.991510869851597e-06, -0.000244067094408253),
    c64::new(-6.17870620606067e-06, -0.00024406242718822947),
    c64::new(-6.365897907496073e-06, -0.00024405761639256817),
    c64::new(-6.553085864037697e-06, -0.00024405266202409917),
    c64::new(-6.7402699655677965e-06, -0.00024404756408573697),
    c64::new(-6.927450101970736e-06, -0.00024404232258048056),
    c64::new(-7.1146261631332075e-06, -0.00024403693751141338),
    c64::new(-7.3017980389444654e-06, -0.00024403140888170337),
    c64::new(-7.488965619296011e-06, -0.00024402573669460282),
    c64::new(-7.676128794082084e-06, -0.00024401992095344856),
    c64::new(-7.86328745319936e-06, -0.00024401396166166185),
    c64::new(-8.050441486547165e-06, -0.00024400785882274837),
    c64::new(-8.237590784027714e-06, -0.00024400161244029827),
    c64::new(-8.42473523554584e-06, -0.00024399522251798616),
    c64::new(-8.611874731009234e-06, -0.000243988689059571),
    c64::new(-8.799009160328657e-06, -0.00024398201206889635),
    c64::new(-8.986138413417692e-06, -0.00024397519154989003),
    c64::new(-9.173262380192966e-06, -0.0002439682275065644),
    c64::new(-9.360380950574381e-06, -0.00024396111994301622),
    c64::new(-9.547494014484795e-06, -0.00024395386886342672),
    c64::new(-9.734601461850518e-06, -0.0002439464742720615),
    c64::new(-9.921703182601008e-06, -0.0002439389361732706),
    c64::new(-1.010879906666909e-05, -0.0002439312545714885),
    c64::new(-1.0295889003991181e-05, -0.0002439234294712341),
    c64::new(-1.048297288450704e-05, -0.00024391546087711068),
    c64::new(-1.0670050598159984e-05, -0.00024390734879380598),
    c64::new(-1.0857122034897124e-05, -0.00024389909322609213),
    c64::new(-1.1044187084669098e-05, -0.00024389069417882564),
    c64::new(-1.1231245637430302e-05, -0.00024388215165694748),
    c64::new(-1.141829758313912e-05, -0.00024387346566548298),
    c64::new(-1.1605342811757602e-05, -0.00024386463620954189),
    c64::new(-1.1792381213251966e-05, -0.00024385566329431836),
    c64::new(-1.1979412677592287e-05, -0.00024384654692509092),
    c64::new(-1.2166437094752718e-05, -0.0002438372871072225),
    c64::new(-1.2353454354711722e-05, -0.00024382788384616039),
    c64::new(-1.2540464347451808e-05, -0.0002438183371474363),
    c64::new(-1.2727466962959764e-05, -0.00024380864701666633),
    c64::new(-1.2914462091226874e-05, -0.00024379881345955087),
    c64::new(-1.310144962224867e-05, -0.00024378883648187483),
    c64::new(-1.3288429446025152e-05, -0.00024377871608950738),
    c64::new(-1.3475401452561014e-05, -0.00024376845228840204),
    c64::new(-1.3662365531865333e-05, -0.0002437580450845968),
    c64::new(-1.3849321573952065e-05, -0.0002437474944842139),
    c64::new(-1.4036269468839736e-05, -0.00024373680049346005),
    c64::new(-1.4223209106551662e-05, -0.0002437259631186262),
    c64::new(-1.4410140377116177e-05, -0.0002437149823660877),
    c64::new(-1.4597063170566377e-05, -0.00024370385824230425),
    c64::new(-1.4783977376940346e-05, -0.00024369259075381988),
    c64::new(-1.497088288628138e-05, -0.00024368117990726298),
    c64::new(-1.5157779588637725e-05, -0.00024366962570934628),
    c64::new(-1.534466737406282e-05, -0.00024365792816686676),
    c64::new(-1.5531546132615498e-05, -0.0002436460872867058),
    c64::new(-1.5718415754359692e-05, -0.0002436341030758291),
    c64::new(-1.590527612936493e-05, -0.00024362197554128666),
    c64::new(-1.6092127147706007e-05, -0.0002436097046902128),
    c64::new(-1.6278968699463237e-05, -0.00024359729052982613),
    c64::new(-1.6465800674722655e-05, -0.0002435847330674296),
    c64::new(-1.6652622963575767e-05, -0.0002435720323104104),
    c64::new(-1.683943545611978e-05, -0.00024355918826624013),
    c64::new(-1.702623804245783e-05, -0.00024354620094247455),
    c64::new(-1.7213030612698705e-05, -0.00024353307034675377),
    c64::new(-1.7399813056957105e-05, -0.00024351979648680222),
    c64::new(-1.7586585265353834e-05, -0.00024350637937042854),
    c64::new(-1.7773347128015505e-05, -0.0002434928190055257),
    c64::new(-1.7960098535075035e-05, -0.00024347911540007085),
    c64::new(-1.8146839376671327e-05, -0.00024346526856212556),
    c64::new(-1.8333569542949504e-05, -0.0002434512784998355),
    c64::new(-1.8520288924061127e-05, -0.00024343714522143066),
    c64::new(-1.8706997410163942e-05, -0.00024342286873522533),
    c64::new(-1.88936948914221e-05, -0.000243408449049618),
    c64::new(-1.9080381258006394e-05, -0.00024339388617309135),
    c64::new(-1.926705640009399e-05, -0.00024337918011421237),
    c64::new(-1.9453720207868656e-05, -0.00024336433088163224),
    c64::new(-1.9640372571520993e-05, -0.00024334933848408643),
    c64::new(-1.9827013381248114e-05, -0.00024333420293039455),
    c64::new(-2.0013642527254146e-05, -0.00024331892422946042),
    c64::new(-2.0200259899749925e-05, -0.00024330350239027218),
    c64::new(-2.0386865388953196e-05, -0.00024328793742190206),
    c64::new(-2.0573458885088868e-05, -0.0002432722293335065),
    c64::new(-2.0760040278388742e-05, -0.00024325637813432622),
    c64::new(-2.094660945909174e-05, -0.00024324038383368604),
    c64::new(-2.1133166317444123e-05, -0.000243224246440995),
    c64::new(-2.1319710743699243e-05, -0.0002432079659657463),
    c64::new(-2.1506242628117772e-05, -0.0002431915424175174),
    c64::new(-2.1692761860967913e-05, -0.00024317497580596972),
    c64::new(-2.18792683325251e-05, -0.00024315826614084907),
    c64::new(-2.2065761933072485e-05, -0.00024314141343198528),
    c64::new(-2.2252242552900636e-05, -0.00024312441768929239),
    c64::new(-2.243871008230776e-05, -0.0002431072789227685),
    c64::new(-2.2625164411599912e-05, -0.00024308999714249598),
    c64::new(-2.281160543109077e-05, -0.0002430725723586412),
    c64::new(-2.2998033031101826e-05, -0.00024305500458145473),
    c64::new(-2.3184447101962634e-05, -0.00024303729382127123),
    c64::new(-2.3370847534010548e-05, -0.0002430194400885095),
    c64::new(-2.3557234217590938e-05, -0.00024300144339367243),
    c64::new(-2.3743607043057432e-05, -0.00024298330374734702),
    c64::new(-2.3929965900771645e-05, -0.00024296502116020433),
    c64::new(-2.4116310681103403e-05, -0.00024294659564299954),
    c64::new(-2.4302641274430986e-05, -0.00024292802720657193),
    c64::new(-2.44889575711408e-05, -0.00024290931586184482),
    c64::new(-2.467525946162787e-05, -0.0002428904616198256),
    c64::new(-2.4861546836295542e-05, -0.00024287146449160574),
    c64::new(-2.5047819585555702e-05, -0.00024285232448836077),
    c64::new(-2.5234077599828997e-05, -0.00024283304162135027),
    c64::new(-2.5420320769544584e-05, -0.00024281361590191783),
    c64::new(-2.5606548985140345e-05, -0.0002427940473414911),
    c64::new(-2.5792762137063132e-05, -0.0002427743359515818),
    c64::new(-2.597896011576849e-05, -0.0002427544817437856),
    c64::new(-2.616514281172089e-05, -0.0002427344847297822),
    c64::new(-2.635131011539396e-05, -0.00024271434492133538),
    c64::new(-2.6537461917270157e-05, -0.00024269406233029284),
    c64::new(-2.6723598107841288e-05, -0.00024267363696858627),
    c64::new(-2.6909718577608172e-05, -0.00024265306884823146),
    c64::new(-2.7095823217080875e-05, -0.00024263235798132804),
    c64::new(-2.7281911916778942e-05, -0.00024261150438005972),
    c64::new(-2.7467984567231132e-05, -0.00024259050805669412),
    c64::new(-2.7654041058975643e-05, -0.00024256936902358284),
    c64::new(-2.7840081282560345e-05, -0.00024254808729316142),
    c64::new(-2.8026105128542514e-05, -0.00024252666287794938),
    c64::new(-2.8212112487489057e-05, -0.00024250509579055012),
    c64::new(-2.8398103249976746e-05, -0.000242483386043651),
    c64::new(-2.8584077306591903e-05, -0.00024246153365002334),
    c64::new(-2.8770034547930886e-05, -0.00024243953862252232),
    c64::new(-2.8955974864599788e-05, -0.00024241740097408703),
    c64::new(-2.9141898147214656e-05, -0.0002423951207177405),
    c64::new(-2.9327804286401726e-05, -0.00024237269786658963),
    c64::new(-2.9513693172797146e-05, -0.0002423501324338252),
    c64::new(-2.969956469704722e-05, -0.00024232742443272188),
    c64::new(-2.9885418749808635e-05, -0.00024230457387663818),
    c64::new(-3.007125522174818e-05, -0.00024228158077901652),
    c64::new(-3.0257074003542987e-05, -0.00024225844515338315),
    c64::new(-3.044287498588077e-05, -0.00024223516701334813),
    c64::new(-3.0628658059459496e-05, -0.00024221174637260542),
    c64::new(-3.0814423114987864e-05, -0.00024218818324493274),
    c64::new(-3.1000170043185026e-05, -0.00024216447764419173),
    c64::new(-3.1185898734780804e-05, -0.00024214062958432777),
    c64::new(-3.13716090805159e-05, -0.00024211663907937),
    c64::new(-3.155730097114164e-05, -0.0002420925061434315),
    c64::new(-3.1742974297420223e-05, -0.00024206823079070899),
    c64::new(-3.192862895012492e-05, -0.00024204381303548304),
    c64::new(-3.211426482003984e-05, -0.00024201925289211803),
    c64::new(-3.22998817979601e-05, -0.00024199455037506203),
    c64::new(-3.2485479774692154e-05, -0.0002419697054988469),
    c64::new(-3.2671058641053364e-05, -0.00024194471827808823),
    c64::new(-3.2856618287872586e-05, -0.00024191958872748535),
    c64::new(-3.30421586059898e-05, -0.00024189431686182138),
    c64::new(-3.322767948625636e-05, -0.00024186890269596305),
    c64::new(-3.3413180819535235e-05, -0.00024184334624486088),
    c64::new(-3.35986624967007e-05, -0.00024181764752354907),
    c64::new(-3.378412440863862e-05, -0.00024179180654714555),
    c64::new(-3.3969566446246624e-05, -0.00024176582333085183),
    c64::new(-3.4154988500433904e-05, -0.00024173969788995322),
    c64::new(-3.434039046212137e-05, -0.00024171343023981863),
    c64::new(-3.452577222224195e-05, -0.00024168702039590063),
    c64::new(-3.471113367174021e-05, -0.00024166046837373548),
    c64::new(-3.48964747015729e-05, -0.000241633774188943),
    c64::new(-3.508179520270861e-05, -0.00024160693785722678),
    c64::new(-3.526709506612801e-05, -0.00024157995939437385),
    c64::new(-3.5452374182824074e-05, -0.00024155283881625497),
    c64::new(-3.5637632443801815e-05, -0.0002415255761388245),
    c64::new(-3.5822869740078525e-05, -0.00024149817137812036),
    c64::new(-3.600808596268397e-05, -0.00024147062455026405),
    c64::new(-3.619328100266018e-05, -0.0002414429356714607),
    c64::new(-3.637845475106161e-05, -0.0002414151047579989),
    c64::new(-3.6563607098955447e-05, -0.00024138713182625092),
    c64::new(-3.6748737937421206e-05, -0.0002413590168926725),
    c64::new(-3.6933847157551303e-05, -0.0002413307599738029),
    c64::new(-3.7118934650450696e-05, -0.00024130236108626496),
    c64::new(-3.730400030723713e-05, -0.000241273820246765),
    c64::new(-3.7489044019041355e-05, -0.0002412451374720929),
    c64::new(-3.767406567700687e-05, -0.00024121631277912195),
    c64::new(-3.7859065172290136e-05, -0.00024118734618480903),
    c64::new(-3.804404239606084e-05, -0.00024115823770619438),
    c64::new(-3.822899723950158e-05, -0.00024112898736040178),
    c64::new(-3.8413929593808146e-05, -0.0002410995951646385),
    c64::new(-3.859883935018969e-05, -0.00024107006113619517),
    c64::new(-3.8783726399868476e-05, -0.0002410403852924459),
    c64::new(-3.896859063408032e-05, -0.00024101056765084823),
    c64::new(-3.915343194407429e-05, -0.0002409806082289431),
    c64::new(-3.9338250221112956e-05, -0.00024095050704435494),
    c64::new(-3.95230453564726e-05, -0.0002409202641147914),
    c64::new(-3.970781724144295e-05, -0.00024088987945804363),
    c64::new(-3.989256576732742e-05, -0.0002408593530919862),
    c64::new(-4.0077290825443307e-05, -0.00024082868503457695),
    c64::new(-4.026199230712157e-05, -0.0002407978753038571),
    c64::new(-4.044667010370704e-05, -0.00024076692391795124),
    c64::new(-4.063132410655862e-05, -0.00024073583089506725),
    c64::new(-4.0815954207049016e-05, -0.00024070459625349637),
    c64::new(-4.100056029656521e-05, -0.00024067322001161315),
    c64::new(-4.118514226650814e-05, -0.00024064170218787538),
    c64::new(-4.1369700008292925e-05, -0.00024061004280082423),
    c64::new(-4.1554233413349123e-05, -0.0002405782418690841),
    c64::new(-4.173874237312042e-05, -0.0002405462994113626),
    c64::new(-4.1923226779064906e-05, -0.00024051421544645074),
    c64::new(-4.2107686522655255e-05, -0.00024048198999322262),
    c64::new(-4.229212149537851e-05, -0.00024044962307063567),
    c64::new(-4.2476531588736264e-05, -0.00024041711469773052),
    c64::new(-4.266091669424492e-05, -0.000240384464893631),
    c64::new(-4.2845276703435366e-05, -0.00024035167367754417),
    c64::new(-4.302961150785347e-05, -0.00024031874106876023),
    c64::new(-4.321392099905975e-05, -0.0002402856670866526),
    c64::new(-4.3398205068629646e-05, -0.00024025245175067787),
    c64::new(-4.3582463608153675e-05, -0.00024021909508037571),
    c64::new(-4.376669650923725e-05, -0.00024018559709536904),
    c64::new(-4.395090366350083e-05, -0.00024015195781536383),
    c64::new(-4.4135084962580215e-05, -0.00024011817726014923),
    c64::new(-4.4319240298126226e-05, -0.00024008425544959746),
    c64::new(-4.4503369561804984e-05, -0.00024005019240366386),
    c64::new(-4.4687472645298084e-05, -0.0002400159881423868),
    c64::new(-4.4871549440302315e-05, -0.0002399816426858878),
    c64::new(-4.505559983853016e-05, -0.00023994715605437142),
    c64::new(-4.523962373170945e-05, -0.0002399125282681252),
    c64::new(-4.5423621011583624e-05, -0.00023987775934751983),
    c64::new(-4.560759156991192e-05, -0.00023984284931300896),
    c64::new(-4.579153529846915e-05, -0.00023980779818512924),
    c64::new(-4.59754520890459e-05, -0.0002397726059845004),
    c64::new(-4.615934183344877e-05, -0.000239737272731825),
    c64::new(-4.6343204423500114e-05, -0.0002397017984478888),
    c64::new(-4.6527039751038246e-05, -0.00023966618315356033),
    c64::new(-4.6710847707917695e-05, -0.0002396304268697912),
    c64::new(-4.6894628186008855e-05, -0.00023959452961761586),
    c64::new(-4.707838107719853e-05, -0.0002395584914181518),
    c64::new(-4.726210627338955e-05, -0.0002395223122925993),
    c64::new(-4.7445803666501096e-05, -0.00023948599226224164),
    c64::new(-4.76294731484688e-05, -0.00023944953134844493),
    c64::new(-4.78131146112446e-05, -0.00023941292957265823),
    c64::new(-4.79967279467969e-05, -0.00023937618695641342),
    c64::new(-4.818031304711081e-05, -0.00023933930352132519),
    c64::new(-4.8363869804187894e-05, -0.00023930227928909117),
    c64::new(-4.854739811004639e-05, -0.00023926511428149172),
    c64::new(-4.873089785672142e-05, -0.00023922780852039005),
    c64::new(-4.891436893626472e-05, -0.00023919036202773222),
    c64::new(-4.909781124074507e-05, -0.000239152774825547),
    c64::new(-4.928122466224806e-05, -0.000239115046935946),
    c64::new(-4.946460909287622e-05, -0.00023907717838112357),
    c64::new(-4.964796442474933e-05, -0.00023903916918335675),
    c64::new(-4.9831290550004125e-05, -0.00023900101936500542),
    c64::new(-5.001458736079451e-05, -0.00023896272894851213),
    c64::new(-5.01978547492918e-05, -0.00023892429795640215),
    c64::new(-5.0381092607684465e-05, -0.0002388857264112834),
    c64::new(-5.056430082817833e-05, -0.0002388470143358466),
    c64::new(-5.0747479302996835e-05, -0.000238808161752865),
    c64::new(-5.093062792438069e-05, -0.00023876916868519461),
    c64::new(-5.11137465845884e-05, -0.00023873003515577403),
    c64::new(-5.12968351758959e-05, -0.0002386907611876245),
    c64::new(-5.1479893590596856e-05, -0.00023865134680384992),
    c64::new(-5.166292172100281e-05, -0.00023861179202763674),
    c64::new(-5.184591945944298e-05, -0.00023857209688225397),
    c64::new(-5.2028886698264466e-05, -0.00023853226139105332),
    c64::new(-5.221182332983245e-05, -0.0002384922855774689),
    c64::new(-5.2394729246529984e-05, -0.00023845216946501754),
    c64::new(-5.2577604340758164e-05, -0.00023841191307729845),
    c64::new(-5.27604485049364e-05, -0.00023837151643799344),
    c64::new(-5.2943261631502065e-05, -0.00023833097957086683),
    c64::new(-5.3126043612911025e-05, -0.00023829030249976541),
    c64::new(-5.3308794341637286e-05, -0.00023824948524861842),
    c64::new(-5.3491513710173257e-05, -0.00023820852784143764),
    c64::new(-5.367420161102994e-05, -0.00023816743030231723),
    c64::new(-5.38568579367367e-05, -0.00023812619265543383),
    c64::new(-5.403948257984148e-05, -0.00023808481492504648),
    c64::new(-5.4222075432911006e-05, -0.0002380432971354966),
    c64::new(-5.440463638853056e-05, -0.0002380016393112081),
    c64::new(-5.4587165339304176e-05, -0.00023795984147668717),
    c64::new(-5.476966217785488e-05, -0.00023791790365652242),
    c64::new(-5.495212679682438e-05, -0.00023787582587538473),
    c64::new(-5.5134559088873557e-05, -0.00023783360815802745),
    c64::new(-5.531695894668213e-05, -0.0002377912505292861),
    c64::new(-5.549932626294891e-05, -0.00023774875301407866),
    c64::new(-5.568166093039201e-05, -0.00023770611563740527),
    c64::new(-5.5863962841748595e-05, -0.0002376633384243484),
    c64::new(-5.604623188977509e-05, -0.00023762042140007281),
    c64::new(-5.6228467967247416e-05, -0.00023757736458982547),
    c64::new(-5.6410670966960734e-05, -0.0002375341680189356),
    c64::new(-5.6592840781729655e-05, -0.0002374908317128146),
    c64::new(-5.677497730438848e-05, -0.0002374473556969561),
    c64::new(-5.695708042779088e-05, -0.00023740373999693598),
    c64::new(-5.713915004481039e-05, -0.00023735998463841218),
    c64::new(-5.732118604834008e-05, -0.00023731608964712488),
    c64::new(-5.750318833129281e-05, -0.00023727205504889634),
    c64::new(-5.7685156786601444e-05, -0.000237227880869631),
    c64::new(-5.786709130721856e-05, -0.00023718356713531538),
    c64::new(-5.8048991786116724e-05, -0.00023713911387201813),
    c64::new(-5.823085811628867e-05, -0.00023709452110588992),
    c64::new(-5.841269019074707e-05, -0.00023704978886316357),
    c64::new(-5.859448790252475e-05, -0.0002370049171701539),
    c64::new(-5.87762511446749e-05, -0.00023695990605325772),
    c64::new(-5.89579798102708e-05, -0.00023691475553895398),
    c64::new(-5.913967379240623e-05, -0.0002368694656538035),
    c64::new(-5.932133298419528e-05, -0.0002368240364244492),
    c64::new(-5.9502957278772467e-05, -0.00023677846787761595),
    c64::new(-5.968454656929299e-05, -0.00023673276004011046),
    c64::new(-5.986610074893251e-05, -0.00023668691293882155),
    c64::new(-6.0047619710887325e-05, -0.00023664092660071986),
    c64::new(-6.0229103348374606e-05, -0.00023659480105285796),
    c64::new(-6.041055155463216e-05, -0.00023654853632237033),
    c64::new(-6.059196422291862e-05, -0.00023650213243647333),
    c64::new(-6.077334124651371e-05, -0.00023645558942246515),
    c64::new(-6.095468251871787e-05, -0.0002364089073077259),
    c64::new(-6.113598793285282e-05, -0.0002363620861197174),
    c64::new(-6.131725738226121e-05, -0.00023631512588598335),
    c64::new(-6.149849076030684e-05, -0.0002362680266341493),
    c64::new(-6.167968796037487e-05, -0.00023622078839192248),
    c64::new(-6.186084887587162e-05, -0.00023617341118709192),
    c64::new(-6.204197340022473e-05, -0.00023612589504752848),
    c64::new(-6.222306142688343e-05, -0.0002360782400011846),
    c64::new(-6.240411284931822e-05, -0.00023603044607609455),
    c64::new(-6.258512756102119e-05, -0.0002359825133003743),
    c64::new(-6.276610545550613e-05, -0.0002359344417022214),
    c64::new(-6.29470464263083e-05, -0.0002358862313099152),
    c64::new(-6.312795036698493e-05, -0.00023583788215181655),
    c64::new(-6.33088171711148e-05, -0.0002357893942563681),
    c64::new(-6.348964673229859e-05, -0.00023574076765209395),
    c64::new(-6.367043894415905e-05, -0.00023569200236759995),
    c64::new(-6.38511937003407e-05, -0.00023564309843157342),
    c64::new(-6.403191089451013e-05, -0.0002355940558727833),
    c64::new(-6.421259042035619e-05, -0.00023554487472008005),
    c64::new(-6.43932321715897e-05, -0.0002354955550023957),
    c64::new(-6.457383604194373e-05, -0.00023544609674874372),
    c64::new(-6.475440192517378e-05, -0.0002353964999882192),
    c64::new(-6.493492971505748e-05, -0.00023534676474999856),
    c64::new(-6.51154193053951e-05, -0.0002352968910633398),
    c64::new(-6.529587059000919e-05, -0.00023524687895758235),
    c64::new(-6.547628346274489e-05, -0.00023519672846214698),
    c64::new(-6.565665781747002e-05, -0.000235146439606536),
    c64::new(-6.583699354807497e-05, -0.00023509601242033298),
    c64::new(-6.601729054847283e-05, -0.00023504544693320298),
    c64::new(-6.61975487125996e-05, -0.00023499474317489237),
    c64::new(-6.637776793441401e-05, -0.00023494390117522886),
    c64::new(-6.655794810789767e-05, -0.00023489292096412148),
    c64::new(-6.673808912705534e-05, -0.0002348418025715606),
    c64::new(-6.69181908859146e-05, -0.0002347905460276178),
    c64::new(-6.709825327852633e-05, -0.00023473915136244605),
    c64::new(-6.727827619896439e-05, -0.00023468761860627945),
    c64::new(-6.745825954132591e-05, -0.00023463594778943344),
    c64::new(-6.76382031997314e-05, -0.0002345841389423046),
    c64::new(-6.781810706832463e-05, -0.00023453219209537075),
    c64::new(-6.799797104127269e-05, -0.0002344801072791909),
    c64::new(-6.817779501276639e-05, -0.00023442788452440513),
    c64::new(-6.835757887701984e-05, -0.00023437552386173483),
    c64::new(-6.853732252827077e-05, -0.00023432302532198243),
    c64::new(-6.871702586078075e-05, -0.00023427038893603138),
    c64::new(-6.889668876883481e-05, -0.00023421761473484638),
    c64::new(-6.907631114674201e-05, -0.00023416470274947315),
    c64::new(-6.925589288883507e-05, -0.00023411165301103837),
    c64::new(-6.943543388947063e-05, -0.0002340584655507499),
    c64::new(-6.961493404302945e-05, -0.00023400514039989654),
    c64::new(-6.979439324391617e-05, -0.00023395167758984806),
    c64::new(-6.997381138655949e-05, -0.00023389807715205532),
    c64::new(-7.015318836541246e-05, -0.00023384433911805003),
    c64::new(-7.03325240749522e-05, -0.0002337904635194449),
    c64::new(-7.051181840968002e-05, -0.00023373645038793359),
    c64::new(-7.069107126412183e-05, -0.00023368229975529055),
    c64::new(-7.087028253282769e-05, -0.00023362801165337132),
    c64::new(-7.104945211037231e-05, -0.0002335735861141121),
    c64::new(-7.12285798913548e-05, -0.00023351902316953005),
    c64::new(-7.140766577039886e-05, -0.00023346432285172318),
    c64::new(-7.158670964215297e-05, -0.00023340948519287024),
    c64::new(-7.17657114012902e-05, -0.00023335451022523081),
    c64::new(-7.194467094250835e-05, -0.0002332993979811453),
    c64::new(-7.212358816053026e-05, -0.00023324414849303475),
    c64::new(-7.230246295010348e-05, -0.00023318876179340104),
    c64::new(-7.248129520600053e-05, -0.0002331332379148268),
    c64::new(-7.266008482301909e-05, -0.00023307757688997518),
    c64::new(-7.28388316959818e-05, -0.0002330217787515902),
    c64::new(-7.301753571973642e-05, -0.0002329658435324965),
    c64::new(-7.319619678915609e-05, -0.00023290977126559924),
    c64::new(-7.337481479913899e-05, -0.00023285356198388436),
    c64::new(-7.355338964460886e-05, -0.00023279721572041828),
    c64::new(-7.373192122051467e-05, -0.0002327407325083481),
    c64::new(-7.391040942183082e-05, -0.00023268411238090144),
    c64::new(-7.408885414355744e-05, -0.0002326273553713864),
    c64::new(-7.426725528072001e-05, -0.0002325704615131917),
    c64::new(-7.44456127283697e-05, -0.0002325134308397866),
    c64::new(-7.462392638158352e-05, -0.00023245626338472066),
    c64::new(-7.48021961354641e-05, -0.0002323989591816241),
    c64::new(-7.498042188513989e-05, -0.0002323415182642075),
    c64::new(-7.515860352576535e-05, -0.0002322839406662618),
    c64::new(-7.533674095252073e-05, -0.00023222622642165853),
    c64::new(-7.551483406061247e-05, -0.00023216837556434939),
    c64::new(-7.569288274527292e-05, -0.0002321103881283666),
    c64::new(-7.587088690176059e-05, -0.0002320522641478227),
    c64::new(-7.604884642536032e-05, -0.00023199400365691046),
    c64::new(-7.622676121138304e-05, -0.00023193560668990308),
    c64::new(-7.640463115516604e-05, -0.00023187707328115397),
    c64::new(-7.658245615207311e-05, -0.00023181840346509684),
    c64::new(-7.676023609749431e-05, -0.0002317595972762456),
    c64::new(-7.693797088684627e-05, -0.00023170065474919447),
    c64::new(-7.711566041557225e-05, -0.0002316415759186178),
    c64::new(-7.729330457914203e-05, -0.0002315823608192701),
    c64::new(-7.747090327305217e-05, -0.00023152300948598613),
    c64::new(-7.764845639282592e-05, -0.00023146352195368072),
    c64::new(-7.782596383401327e-05, -0.0002314038982573489),
    c64::new(-7.800342549219132e-05, -0.00023134413843206572),
    c64::new(-7.818084126296385e-05, -0.00023128424251298634),
    c64::new(-7.83582110419617e-05, -0.000231224210535346),
    c64::new(-7.85355347248429e-05, -0.00023116404253445996),
    c64::new(-7.871281220729246e-05, -0.00023110373854572352),
    c64::new(-7.889004338502252e-05, -0.00023104329860461196),
    c64::new(-7.906722815377267e-05, -0.00023098272274668053),
    c64::new(-7.924436640930958e-05, -0.00023092201100756443),
    c64::new(-7.942145804742746e-05, -0.00023086116342297886),
    c64::new(-7.959850296394781e-05, -0.00023080018002871882),
    c64::new(-7.977550105471965e-05, -0.00023073906086065933),
    c64::new(-7.995245221561962e-05, -0.00023067780595475518),
    c64::new(-8.012935634255189e-05, -0.00023061641534704108),
    c64::new(-8.030621333144826e-05, -0.00023055488907363156),
    c64::new(-8.048302307826842e-05, -0.00023049322717072086),
    c64::new(-8.065978547899972e-05, -0.00023043142967458318),
    c64::new(-8.08365004296573e-05, -0.00023036949662157236),
    c64::new(-8.101316782628445e-05, -0.000230307428048122),
    c64::new(-8.118978756495218e-05, -0.00023024522399074548),
    c64::new(-8.136635954175973e-05, -0.00023018288448603582),
    c64::new(-8.154288365283432e-05, -0.00023012040957066574),
    c64::new(-8.171935979433133e-05, -0.00023005779928138764),
    c64::new(-8.189578786243446e-05, -0.00022999505365503355),
    c64::new(-8.207216775335561e-05, -0.00022993217272851505),
    c64::new(-8.224849936333495e-05, -0.00022986915653882344),
    c64::new(-8.242478258864125e-05, -0.00022980600512302948),
    c64::new(-8.260101732557158e-05, -0.0002297427185182835),
    c64::new(-8.277720347045154e-05, -0.00022967929676181545),
    c64::new(-8.295334091963546e-05, -0.00022961573989093464),
    c64::new(-8.31294295695061e-05, -0.00022955204794303),
    c64::new(-8.330546931647517e-05, -0.00022948822095556984),
    c64::new(-8.348146005698296e-05, -0.0002294242589661019),
    c64::new(-8.36574016874986e-05, -0.0002293601620122534),
    c64::new(-8.383329410452026e-05, -0.00022929593013173093),
    c64::new(-8.400913720457492e-05, -0.00022923156336232043),
    c64::new(-8.418493088421855e-05, -0.00022916706174188727),
    c64::new(-8.436067504003639e-05, -0.00022910242530837602),
    c64::new(-8.45363695686426e-05, -0.00022903765409981063),
    c64::new(-8.471201436668056e-05, -0.0002289727481542944),
    c64::new(-8.488760933082309e-05, -0.00022890770751000976),
    c64::new(-8.506315435777207e-05, -0.0002288425322052185),
    c64::new(-8.523864934425901e-05, -0.00022877722227826152),
    c64::new(-8.541409418704466e-05, -0.00022871177776755897),
    c64::new(-8.55894887829193e-05, -0.00022864619871161026),
    c64::new(-8.576483302870289e-05, -0.00022858048514899372),
    c64::new(-8.59401268212449e-05, -0.00022851463711836704),
    c64::new(-8.611537005742447e-05, -0.0002284486546584669),
    c64::new(-8.629056263415058e-05, -0.00022838253780810904),
    c64::new(-8.646570444836191e-05, -0.00022831628660618838),
    c64::new(-8.664079539702702e-05, -0.00022824990109167873),
    c64::new(-8.68158353771445e-05, -0.000228183381303633),
    c64::new(-8.699082428574274e-05, -0.00022811672728118308),
    c64::new(-8.716576201988035e-05, -0.0002280499390635398),
    c64::new(-8.734064847664597e-05, -0.0002279830166899929),
    c64::new(-8.751548355315831e-05, -0.0002279159601999112),
    c64::new(-8.769026714656651e-05, -0.0002278487696327422),
    c64::new(-8.786499915404985e-05, -0.0002277814450280124),
    c64::new(-8.803967947281791e-05, -0.0002277139864253272),
    c64::new(-8.82143080001109e-05, -0.00022764639386437072),
    c64::new(-8.838888463319928e-05, -0.00022757866738490587),
    c64::new(-8.856340926938406e-05, -0.00022751080702677445),
    c64::new(-8.873788180599701e-05, -0.00022744281282989693),
    c64::new(-8.891230214040032e-05, -0.0002273746848342726),
    c64::new(-8.908667016998713e-05, -0.00022730642307997932),
    c64::new(-8.926098579218112e-05, -0.00022723802760717376),
    c64::new(-8.943524890443688e-05, -0.00022716949845609126),
    c64::new(-8.960945940424002e-05, -0.00022710083566704567),
    c64::new(-8.978361718910693e-05, -0.0002270320392804296),
    c64::new(-8.995772215658501e-05, -0.0002269631093367142),
    c64::new(-9.013177420425292e-05, -0.00022689404587644912),
    c64::new(-9.030577322972025e-05, -0.00022682484894026263),
    c64::new(-9.047971913062782e-05, -0.00022675551856886158),
    c64::new(-9.065361180464784e-05, -0.00022668605480303115),
    c64::new(-9.08274511494836e-05, -0.00022661645768363515),
    c64::new(-9.100123706287004e-05, -0.0002265467272516157),
    c64::new(-9.117496944257331e-05, -0.00022647686354799345),
    c64::new(-9.134864818639108e-05, -0.00022640686661386746),
    c64::new(-9.152227319215272e-05, -0.00022633673649041506),
    c64::new(-9.169584435771905e-05, -0.00022626647321889203),
    c64::new(-9.186936158098259e-05, -0.00022619607684063245),
    c64::new(-9.204282475986773e-05, -0.00022612554739704868),
    c64::new(-9.221623379233049e-05, -0.00022605488492963134),
    c64::new(-9.238958857635877e-05, -0.00022598408947994942),
    c64::new(-9.256288900997249e-05, -0.00022591316108964998),
    c64::new(-9.273613499122339e-05, -0.00022584209980045842),
    c64::new(-9.290932641819543e-05, -0.0002257709056541782),
    c64::new(-9.308246318900449e-05, -0.00022569957869269107),
    c64::new(-9.325554520179861e-05, -0.00022562811895795684),
    c64::new(-9.342857235475823e-05, -0.00022555652649201336),
    c64::new(-9.360154454609586e-05, -0.00022548480133697673),
    c64::new(-9.377446167405638e-05, -0.00022541294353504093),
    c64::new(-9.394732363691719e-05, -0.0002253409531284781),
    c64::new(-9.412013033298801e-05, -0.0002252688301596383),
    c64::new(-9.429288166061105e-05, -0.0002251965746709497),
    c64::new(-9.446557751816127e-05, -0.00022512418670491823),
    c64::new(-9.463821780404604e-05, -0.00022505166630412796),
    c64::new(-9.481080241670561e-05, -0.00022497901351124073),
    c64::new(-9.498333125461289e-05, -0.00022490622836899627),
    c64::new(-9.515580421627351e-05, -0.00022483331092021229),
    c64::new(-9.53282212002262e-05, -0.00022476026120778417),
    c64::new(-9.550058210504245e-05, -0.0002246870792746852),
    c64::new(-9.567288682932672e-05, -0.00022461376516396645),
    c64::new(-9.584513527171667e-05, -0.00022454031891875668),
    c64::new(-9.601732733088292e-05, -0.00022446674058226244),
    c64::new(-9.61894629055293e-05, -0.000224393030197768),
    c64::new(-9.636154189439296e-05, -0.00022431918780863524),
    c64::new(-9.653356419624419e-05, -0.00022424521345830377),
    c64::new(-9.670552970988677e-05, -0.00022417110719029077),
    c64::new(-9.687743833415779e-05, -0.00022409686904819104),
    c64::new(-9.704928996792779e-05, -0.000224022499075677),
    c64::new(-9.722108451010097e-05, -0.00022394799731649854),
    c64::new(-9.739282185961497e-05, -0.00022387336381448315),
    c64::new(-9.75645019154411e-05, -0.00022379859861353582),
    c64::new(-9.773612457658448e-05, -0.00022372370175763895),
    c64::new(-9.79076897420839e-05, -0.00022364867329085247),
    c64::new(-9.807919731101194e-05, -0.00022357351325731368),
    c64::new(-9.82506471824752e-05, -0.00022349822170123726),
    c64::new(-9.842203925561405e-05, -0.00022342279866691535),
    c64::new(-9.859337342960303e-05, -0.00022334724419871733),
    c64::new(-9.876464960365063e-05, -0.00022327155834108995),
    c64::new(-9.893586767699945e-05, -0.0002231957411385573),
    c64::new(-9.910702754892638e-05, -0.00022311979263572063),
    c64::new(-9.927812911874244e-05, -0.0002230437128772585),
    c64::new(-9.944917228579293e-05, -0.0002229675019079267),
    c64::new(-9.962015694945767e-05, -0.00022289115977255816),
    c64::new(-9.979108300915074e-05, -0.00022281468651606296),
    c64::new(-9.996195036432069e-05, -0.00022273808218342843),
    c64::new(-0.00010013275891445074, -0.00022266134681971883),
    c64::new(-0.00010030350855905854, -0.00022258448047007565),
    c64::new(-0.00010047419919769657, -0.0002225074831797174),
    c64::new(-0.00010064483072995188, -0.00022243035499393954),
    c64::new(-0.00010081540305544627, -0.00022235309595811464),
    c64::new(-0.00010098591607383655, -0.0002222757061176922),
    c64::new(-0.00010115636968481423, -0.00022219818551819865),
    c64::new(-0.00010132676378810584, -0.00022212053420523739),
    c64::new(-0.00010149709828347297, -0.00022204275222448863),
    c64::new(-0.00010166737307071215, -0.0002219648396217096),
    c64::new(-0.00010183758804965522, -0.00022188679644273416),
    c64::new(-0.00010200774312016901, -0.00022180862273347317),
    c64::new(-0.0001021778381821557, -0.00022173031853991423),
    c64::new(-0.00010234787313555275, -0.00022165188390812162),
    c64::new(-0.00010251784788033287, -0.0002215733188842365),
    c64::new(-0.00010268776231650449, -0.00022149462351447654),
    c64::new(-0.00010285761634411123, -0.00022141579784513626),
    c64::new(-0.00010302740986323234, -0.00022133684192258675),
    c64::new(-0.00010319714277398268, -0.00022125775579327577),
    c64::new(-0.00010336681497651264, -0.00022117853950372768),
    c64::new(-0.00010353642637100864, -0.0002210991931005433),
    c64::new(-0.00010370597685769257, -0.0002210197166304001),
    c64::new(-0.00010387546633682222, -0.00022094011014005206),
    c64::new(-0.00010404489470869133, -0.00022086037367632965),
    c64::new(-0.00010421426187362954, -0.0002207805072861397),
    c64::new(-0.00010438356773200241, -0.00022070051101646569),
    c64::new(-0.00010455281218421186, -0.0002206203849143672),
    c64::new(-0.00010472199513069566, -0.00022054012902698042),
    c64::new(-0.00010489111647192783, -0.00022045974340151786),
    c64::new(-0.00010506017610841863, -0.00022037922808526827),
    c64::new(-0.00010522917394071455, -0.00022029858312559676),
    c64::new(-0.0001053981098693987, -0.00022021780856994465),
    c64::new(-0.00010556698379509034, -0.00022013690446582952),
    c64::new(-0.0001057357956184453, -0.00022005587086084522),
    c64::new(-0.0001059045452401559, -0.0002199747078026617),
    c64::new(-0.00010607323256095113, -0.0002198934153390251),
    c64::new(-0.00010624185748159644, -0.00021981199351775774),
    c64::new(-0.00010641041990289438, -0.00021973044238675788),
    c64::new(-0.00010657891972568387, -0.00021964876199399998),
    c64::new(-0.00010674735685084084, -0.00021956695238753456),
    c64::new(-0.00010691573117927807, -0.00021948501361548805),
    c64::new(-0.00010708404261194516, -0.00021940294572606297),
    c64::new(-0.00010725229104982905, -0.0002193207487675377),
    c64::new(-0.0001074204763939534, -0.00021923842278826657),
    c64::new(-0.00010758859854537913, -0.0002191559678366799),
    c64::new(-0.00010775665740520428, -0.00021907338396128374),
    c64::new(-0.00010792465287456418, -0.00021899067121066015),
    c64::new(-0.0001080925848546313, -0.00021890782963346686),
    c64::new(-0.0001082604532466158, -0.00021882485927843742),
    c64::new(-0.00010842825795176493, -0.00021874176019438116),
    c64::new(-0.0001085959988713635, -0.0002186585324301831),
    c64::new(-0.00010876367590673385, -0.00021857517603480407),
    c64::new(-0.00010893128895923579, -0.00021849169105728048),
    c64::new(-0.00010909883793026709, -0.00021840807754672432),
    c64::new(-0.00010926632272126291, -0.00021832433555232328),
    c64::new(-0.00010943374323369626, -0.00021824046512334064),
    c64::new(-0.00010960109936907796, -0.0002181564663091152),
    c64::new(-0.00010976839102895672, -0.00021807233915906133),
    c64::new(-0.00010993561811491905, -0.0002179880837226689),
    c64::new(-0.00011010278052858978, -0.0002179037000495031),
    c64::new(-0.00011026987817163145, -0.00021781918818920473),
    c64::new(-0.00011043691094574485, -0.00021773454819148992),
    c64::new(-0.00011060387875266887, -0.00021764978010615024),
    c64::new(-0.00011077078149418058, -0.0002175648839830526),
    c64::new(-0.00011093761907209557, -0.00021747985987213906),
    c64::new(-0.00011110439138826751, -0.00021739470782342724),
    c64::new(-0.00011127109834458853, -0.00021730942788700984),
    c64::new(-0.00011143773984298925, -0.00021722402011305488),
    c64::new(-0.00011160431578543874, -0.00021713848455180555),
    c64::new(-0.00011177082607394458, -0.00021705282125358025),
    c64::new(-0.00011193727061055325, -0.0002169670302687724),
    c64::new(-0.00011210364929734962, -0.00021688111164785067),
    c64::new(-0.0001122699620364574, -0.00021679506544135878),
    c64::new(-0.00011243620873003909, -0.0002167088916999155),
    c64::new(-0.00011260238928029594, -0.00021662259047421466),
    c64::new(-0.0001127685035894684, -0.00021653616181502498),
    c64::new(-0.00011293455155983562, -0.0002164496057731902),
    c64::new(-0.00011310053309371586, -0.00021636292239962906),
    c64::new(-0.00011326644809346645, -0.00021627611174533515),
    c64::new(-0.00011343229646148387, -0.00021618917386137691),
    c64::new(-0.0001135980781002037, -0.00021610210879889776),
    c64::new(-0.00011376379291210108, -0.0002160149166091157),
    c64::new(-0.00011392944079969015, -0.00021592759734332368),
    c64::new(-0.00011409502166552453, -0.0002158401510528894),
    c64::new(-0.00011426053541219724, -0.00021575257778925526),
    c64::new(-0.00011442598194234074, -0.0002156648776039384),
    c64::new(-0.00011459136115862728, -0.0002155770505485305),
    c64::new(-0.00011475667296376843, -0.00021548909667469795),
    c64::new(-0.00011492191726051553, -0.00021540101603418181),
    c64::new(-0.0001150870939516596, -0.00021531280867879762),
    c64::new(-0.00011525220294003144, -0.0002152244746604355),
    c64::new(-0.0001154172441285016, -0.0002151360140310601),
    c64::new(-0.00011558221741998081, -0.00021504742684271046),
    c64::new(-0.00011574712271741943, -0.0002149587131475002),
    c64::new(-0.00011591195992380789, -0.00021486987299761728),
    c64::new(-0.00011607672894217675, -0.00021478090644532407),
    c64::new(-0.00011624142967559652, -0.00021469181354295737),
    c64::new(-0.00011640606202717822, -0.0002146025943429281),
    c64::new(-0.00011657062590007274, -0.00021451324889772174),
    c64::new(-0.00011673512119747144, -0.00021442377725989785),
    c64::new(-0.00011689954782260594, -0.00021433417948209027),
    c64::new(-0.0001170639056787483, -0.00021424445561700714),
    c64::new(-0.00011722819466921096, -0.0002141546057174307),
    c64::new(-0.00011739241469734712, -0.00021406462983621723),
    c64::new(-0.00011755656566655023, -0.0002139745280262973),
    c64::new(-0.00011772064748025457, -0.00021388430034067544),
    c64::new(-0.00011788466004193498, -0.00021379394683243032),
    c64::new(-0.00011804860325510703, -0.0002137034675547146),
    c64::new(-0.00011821247702332735, -0.00021361286256075482),
    c64::new(-0.0001183762812501931, -0.00021352213190385162),
    c64::new(-0.00011854001583934249, -0.00021343127563737947),
    c64::new(-0.00011870368069445469, -0.00021334029381478683),
    c64::new(-0.00011886727571924987, -0.00021324918648959592),
    c64::new(-0.00011903080081748919, -0.00021315795371540288),
    c64::new(-0.00011919425589297524, -0.00021306659554587754),
    c64::new(-0.00011935764084955153, -0.00021297511203476356),
    c64::new(-0.00011952095559110289, -0.00021288350323587837),
    c64::new(-0.00011968420002155546, -0.00021279176920311303),
    c64::new(-0.00011984737404487666, -0.0002126999099904324),
    c64::new(-0.00012001047756507562, -0.00021260792565187478),
    c64::new(-0.00012017351048620262, -0.0002125158162415522),
    c64::new(-0.00012033647271234959, -0.00021242358181365026),
    c64::new(-0.00012049936414765009, -0.00021233122242242814),
    c64::new(-0.00012066218469627924, -0.00021223873812221844),
    c64::new(-0.00012082493426245384, -0.00021214612896742737),
    c64::new(-0.0001209876127504327, -0.00021205339501253443),
    c64::new(-0.00012115022006451615, -0.00021196053631209263),
    c64::new(-0.0001213127561090465, -0.00021186755292072835),
    c64::new(-0.000121475220788408, -0.00021177444489314134),
    c64::new(-0.00012163761400702674, -0.0002116812122841047),
    c64::new(-0.0001217999356693712, -0.00021158785514846466),
    c64::new(-0.0001219621856799516, -0.0002114943735411409),
    c64::new(-0.00012212436394332045, -0.0002114007675171262),
    c64::new(-0.00012228647036407244, -0.0002113070371314866),
    c64::new(-0.00012244850484684458, -0.00021121318243936125),
    c64::new(-0.00012261046729631606, -0.00021111920349596252),
    c64::new(-0.00012277235761720868, -0.0002110251003565757),
    c64::new(-0.0001229341757142865, -0.00021093087307655928),
    c64::new(-0.00012309592149235604, -0.00021083652171134475),
    c64::new(-0.0001232575948562665, -0.00021074204631643656),
    c64::new(-0.00012341919571090947, -0.00021064744694741224),
    c64::new(-0.0001235807239612195, -0.00021055272365992205),
    c64::new(-0.00012374217951217364, -0.0002104578765096893),
    c64::new(-0.0001239035622687917, -0.0002103629055525101),
    c64::new(-0.00012406487213613638, -0.00021026781084425346),
    c64::new(-0.00012422610901931326, -0.00021017259244086113),
    c64::new(-0.0001243872728234707, -0.00021007725039834773),
    c64::new(-0.00012454836345380044, -0.00020998178477280035),
    c64::new(-0.00012470938081553685, -0.00020988619562037908),
    c64::new(-0.0001248703248139576, -0.00020979048299731652),
    c64::new(-0.00012503119535438347, -0.00020969464695991795),
    c64::new(-0.0001251919923421784, -0.00020959868756456127),
    c64::new(-0.00012535271568274981, -0.00020950260486769684),
    c64::new(-0.00012551336528154823, -0.00020940639892584769),
    c64::new(-0.00012567394104406765, -0.0002093100697956093),
    c64::new(-0.00012583444287584544, -0.0002092136175336496),
    c64::new(-0.00012599487068246258, -0.00020911704219670897),
    c64::new(-0.00012615522436954337, -0.00020902034384160024),
    c64::new(-0.00012631550384275606, -0.0002089235225252085),
    c64::new(-0.0001264757090078123, -0.00020882657830449122),
    c64::new(-0.00012663583977046752, -0.00020872951123647825),
    c64::new(-0.0001267958960365209, -0.00020863232137827161),
    c64::new(-0.00012695587771181536, -0.00020853500878704564),
    c64::new(-0.00012711578470223802, -0.00020843757352004676),
    c64::new(-0.0001272756169137196, -0.00020834001563459364),
    c64::new(-0.00012743537425223493, -0.00020824233518807709),
    c64::new(-0.00012759505662380288, -0.00020814453223795998),
    c64::new(-0.00012775466393448643, -0.0002080466068417773),
    c64::new(-0.00012791419609039264, -0.00020794855905713604),
    c64::new(-0.00012807365299767303, -0.00020785038894171514),
    c64::new(-0.0001282330345625231, -0.00020775209655326559),
    c64::new(-0.00012839234069118277, -0.00020765368194961022),
    c64::new(-0.0001285515712899364, -0.00020755514518864385),
    c64::new(-0.00012871072626511256, -0.00020745648632833317),
    c64::new(-0.00012886980552308475, -0.00020735770542671653),
    c64::new(-0.00012902880897027057, -0.00020725880254190423),
    c64::new(-0.00012918773651313247, -0.0002071597777320783),
    c64::new(-0.0001293465880581774, -0.0002070606310554925),
    c64::new(-0.00012950536351195714, -0.00020696136257047223),
    c64::new(-0.00012966406278106803, -0.00020686197233541468),
    c64::new(-0.00012982268577215164, -0.00020676246040878842),
    c64::new(-0.000129981232391894, -0.00020666282684913382),
    c64::new(-0.00013013970254702623, -0.00020656307171506276),
    c64::new(-0.00013029809614432445, -0.00020646319506525856),
    c64::new(-0.00013045641309060966, -0.00020636319695847618),
    c64::new(-0.00013061465329274827, -0.0002062630774535418),
    c64::new(-0.00013077281665765158, -0.00020616283660935321),
    c64::new(-0.00013093090309227614, -0.00020606247448487952),
    c64::new(-0.00013108891250362376, -0.00020596199113916117),
    c64::new(-0.00013124684479874158, -0.00020586138663130992),
    c64::new(-0.000131404699884722, -0.00020576066102050887),
    c64::new(-0.00013156247766870317, -0.0002056598143660122),
    c64::new(-0.00013172017805786836, -0.00020555884672714546),
    c64::new(-0.0001318778009594465, -0.00020545775816330528),
    c64::new(-0.00013203534628071208, -0.0002053565487339595),
    c64::new(-0.00013219281392898516, -0.00020525521849864702),
    c64::new(-0.00013235020381163176, -0.00020515376751697775),
    c64::new(-0.00013250751583606342, -0.00020505219584863272),
    c64::new(-0.00013266474990973747, -0.0002049505035533639),
    c64::new(-0.00013282190594015718, -0.00020484869069099432),
    c64::new(-0.00013297898383487174, -0.0002047467573214178),
    c64::new(-0.0001331359835014761, -0.00020464470350459917),
    c64::new(-0.00013329290484761166, -0.00020454252930057399),
    c64::new(-0.00013344974778096551, -0.00020444023476944874),
    c64::new(-0.00013360651220927102, -0.00020433781997140064),
    c64::new(-0.00013376319804030766, -0.00020423528496667774),
    c64::new(-0.00013391980518190114, -0.00020413262981559872),
    c64::new(-0.00013407633354192372, -0.0002040298545785529),
    c64::new(-0.00013423278302829365, -0.00020392695931600036),
    c64::new(-0.00013438915354897576, -0.0002038239440884717),
    c64::new(-0.0001345454450119813, -0.00020372080895656816),
    c64::new(-0.00013470165732536803, -0.00020361755398096143),
    c64::new(-0.00013485779039724017, -0.0002035141792223939),
    c64::new(-0.00013501384413574887, -0.00020341068474167805),
    c64::new(-0.0001351698184490916, -0.00020330707059969716),
    c64::new(-0.0001353257132455127, -0.0002032033368574047),
    c64::new(-0.0001354815284333033, -0.0002030994835758246),
    c64::new(-0.00013563726392080126, -0.00020299551081605114),
    c64::new(-0.00013579291961639153, -0.00020289141863924864),
    c64::new(-0.00013594849542850583, -0.00020278720710665196),
    c64::new(-0.0001361039912656229, -0.00020268287627956603),
    c64::new(-0.00013625940703626858, -0.00020257842621936598),
    c64::new(-0.00013641474264901573, -0.00020247385698749713),
    c64::new(-0.00013656999801248433, -0.00020236916864547488),
    c64::new(-0.00013672517303534184, -0.00020226436125488458),
    c64::new(-0.00013688026762630271, -0.0002021594348773818),
    c64::new(-0.00013703528169412878, -0.000202054389574692),
    c64::new(-0.00013719021514762926, -0.00020194922540861063),
    c64::new(-0.00013734506789566078, -0.00020184394244100306),
    c64::new(-0.0001374998398471274, -0.0002017385407338046),
    c64::new(-0.00013765453091098094, -0.00020163302034902026),
    c64::new(-0.00013780914099622055, -0.000201527381348725),
    c64::new(-0.00013796367001189306, -0.00020142162379506353),
    c64::new(-0.00013811811786709307, -0.00020131574775025032),
    c64::new(-0.0001382724844709628, -0.00020120975327656953),
    c64::new(-0.0001384267697326925, -0.0002011036404363749),
    c64::new(-0.00013858097356152005, -0.0002009974092920899),
    c64::new(-0.00013873509586673132, -0.00020089105990620762),
    c64::new(-0.00013888913655766012, -0.00020078459234129062),
    c64::new(-0.0001390430955436883, -0.00020067800665997106),
    c64::new(-0.00013919697273424562, -0.00020057130292495061),
    c64::new(-0.00013935076803881034, -0.00020046448119900019),
    c64::new(-0.0001395044813669085, -0.00020035754154496034),
    c64::new(-0.00013965811262811451, -0.0002002504840257409),
    c64::new(-0.0001398116617320511, -0.00020014330870432101),
    c64::new(-0.00013996512858838916, -0.0002000360156437493),
    c64::new(-0.00014011851310684827, -0.00019992860490714332),
    c64::new(-0.0001402718151971962, -0.00019982107655769013),
    c64::new(-0.00014042503476924929, -0.00019971343065864586),
    c64::new(-0.0001405781717328724, -0.00019960566727333587),
    c64::new(-0.00014073122599797896, -0.00019949778646515453),
    c64::new(-0.0001408841974745311, -0.00019938978829756542),
    c64::new(-0.0001410370860725397, -0.00019928167283410098),
    c64::new(-0.00014118989170206433, -0.00019917344013836277),
    c64::new(-0.00014134261427321333, -0.0001990650902740213),
    c64::new(-0.00014149525369614398, -0.00019895662330481604),
    c64::new(-0.00014164780988106234, -0.00019884803929455538),
    c64::new(-0.0001418002827382237, -0.00019873933830711634),
    c64::new(-0.00014195267217793217, -0.00019863052040644497),
    c64::new(-0.00014210497811054088, -0.00019852158565655605),
    c64::new(-0.00014225720044645226, -0.0001984125341215331),
    c64::new(-0.00014240933909611777, -0.0001983033658655283),
    c64::new(-0.0001425613939700381, -0.0001981940809527626),
    c64::new(-0.00014271336497876345, -0.00019808467944752538),
    c64::new(-0.00014286525203289314, -0.00019797516141417475),
    c64::new(-0.0001430170550430759, -0.0001978655269171374),
    c64::new(-0.00014316877392000991, -0.00019775577602090848),
    c64::new(-0.0001433204085744428, -0.00019764590879005163),
    c64::new(-0.000143471958917172, -0.00019753592528919882),
    c64::new(-0.00014362342485904424, -0.00019742582558305056),
    c64::new(-0.000143774806310956, -0.0001973156097363757),
    c64::new(-0.0001439261031838535, -0.00019720527781401132),
    c64::new(-0.00014407731538873265, -0.0001970948298808629),
    c64::new(-0.00014422844283663917, -0.00019698426600190416),
    c64::new(-0.0001443794854386688, -0.00019687358624217684),
    c64::new(-0.000144530443105967, -0.0001967627906667911),
    c64::new(-0.00014468131574972928, -0.0001966518793409251),
    c64::new(-0.00014483210328120115, -0.00019654085232982508),
    c64::new(-0.00014498280561167805, -0.0001964297096988055),
    c64::new(-0.00014513342265250594, -0.00019631845151324852),
    c64::new(-0.00014528395431508053, -0.00019620707783860454),
    c64::new(-0.00014543440051084796, -0.00019609558874039183),
    c64::new(-0.00014558476115130462, -0.00019598398428419657),
    c64::new(-0.0001457350361479972, -0.00019587226453567276),
    c64::new(-0.00014588522541252272, -0.00019576042956054237),
    c64::new(-0.00014603532885652885, -0.0001956484794245948),
    c64::new(-0.00014618534639171353, -0.0001955364141936876),
    c64::new(-0.00014633527792982533, -0.0001954242339337458),
    c64::new(-0.00014648512338266334, -0.00019531193871076219),
    c64::new(-0.0001466348826620773, -0.00019519952859079718),
    c64::new(-0.00014678455567996787, -0.00019508700363997866),
    c64::new(-0.0001469341423482861, -0.00019497436392450218),
    c64::new(-0.00014708364257903412, -0.00019486160951063083),
    c64::new(-0.0001472330562842647, -0.0001947487404646951),
    c64::new(-0.00014738238337608166, -0.00019463575685309294),
    c64::new(-0.0001475316237666396, -0.00019452265874228978),
    c64::new(-0.00014768077736814446, -0.00019440944619881817),
    c64::new(-0.0001478298440928529, -0.00019429611928927821),
    c64::new(-0.00014797882385307278, -0.00019418267808033723),
    c64::new(-0.00014812771656116323, -0.00019406912263872974),
    c64::new(-0.00014827652212953433, -0.00019395545303125752),
    c64::new(-0.00014842524047064783, -0.00019384166932478938),
    c64::new(-0.00014857387149701644, -0.0001937277715862614),
    c64::new(-0.00014872241512120437, -0.0001936137598826767),
    c64::new(-0.00014887087125582725, -0.0001934996342811054),
    c64::new(-0.00014901923981355212, -0.00019338539484868464),
    c64::new(-0.00014916752070709755, -0.00019327104165261865),
    c64::new(-0.00014931571384923384, -0.0001931565747601783),
    c64::new(-0.0001494638191527827, -0.0001930419942387016),
    c64::new(-0.0001496118365306176, -0.00019292730015559327),
    c64::new(-0.00014975976589566374, -0.00019281249257832497),
    c64::new(-0.000149907607160898, -0.00019269757157443503),
    c64::new(-0.00015005536023934932, -0.00019258253721152842),
    c64::new(-0.00015020302504409832, -0.00019246738955727694),
    c64::new(-0.00015035060148827758, -0.000192352128679419),
    c64::new(-0.0001504980894850717, -0.0001922367546457596),
    c64::new(-0.00015064548894771725, -0.00019212126752417034),
    c64::new(-0.00015079279978950287, -0.00019200566738258933),
    c64::new(-0.00015094002192376954, -0.00019188995428902106),
    c64::new(-0.0001510871552639102, -0.00019177412831153663),
    c64::new(-0.00015123419972337012, -0.00019165818951827346),
    c64::new(-0.00015138115521564678, -0.00019154213797743536),
    c64::new(-0.00015152802165429, -0.00019142597375729252),
    c64::new(-0.0001516747989529022, -0.00019130969692618123),
    c64::new(-0.000151821487025138, -0.00019119330755250425),
    c64::new(-0.00015196808578470458, -0.0001910768057047304),
    c64::new(-0.00015211459514536164, -0.00019096019145139478),
    c64::new(-0.00015226101502092154, -0.00019084346486109848),
    c64::new(-0.00015240734532524913, -0.00019072662600250884),
    c64::new(-0.00015255358597226226, -0.00019060967494435903),
    c64::new(-0.00015269973687593121, -0.00019049261175544841),
    c64::new(-0.00015284579795027918, -0.00019037543650464223),
    c64::new(-0.0001529917691093822, -0.00019025814926087165),
    c64::new(-0.0001531376502673691, -0.00019014075009313384),
    c64::new(-0.00015328344133842196, -0.00019002323907049153),
    c64::new(-0.00015342914223677554, -0.00018990561626207352),
    c64::new(-0.00015357475287671775, -0.0001897878817370743),
    c64::new(-0.0001537202731725896, -0.00018967003556475401),
    c64::new(-0.00015386570303878527, -0.00018955207781443854),
    c64::new(-0.00015401104238975204, -0.00018943400855551948),
    c64::new(-0.00015415629113999063, -0.00018931582785745377),
    c64::new(-0.00015430144920405492, -0.00018919753578976414),
    c64::new(-0.00015444651649655213, -0.00018907913242203879),
    c64::new(-0.0001545914929321429, -0.00018896061782393134),
    c64::new(-0.00015473637842554125, -0.00018884199206516095),
    c64::new(-0.00015488117289151496, -0.000188723255215512),
    c64::new(-0.0001550258762448851, -0.00018860440734483434),
    c64::new(-0.0001551704884005264, -0.0001884854485230431),
    c64::new(-0.00015531500927336723, -0.00018836637882011873),
    c64::new(-0.00015545943877838972, -0.00018824719830610686),
    c64::new(-0.00015560377683062962, -0.0001881279070511184),
    c64::new(-0.00015574802334517674, -0.00018800850512532913),
    c64::new(-0.00015589217823717448, -0.00018788899259898024),
    c64::new(-0.00015603624142182025, -0.00018776936954237786),
    c64::new(-0.00015618021281436537, -0.00018764963602589313),
    c64::new(-0.0001563240923301151, -0.00018752979211996231),
    c64::new(-0.000156467879884429, -0.0001874098378950863),
    c64::new(-0.00015661157539272047, -0.00018728977342183118),
    c64::new(-0.00015675517877045714, -0.00018716959877082776),
    c64::new(-0.00015689868993316086, -0.0001870493140127717),
    c64::new(-0.00015704210879640768, -0.00018692891921842343),
    c64::new(-0.00015718543527582795, -0.00018680841445860817),
    c64::new(-0.00015732866928710649, -0.00018668779980421564),
    c64::new(-0.0001574718107459824, -0.0001865670753262004),
    c64::new(-0.0001576148595682492, -0.0001864462410955815),
    c64::new(-0.00015775781566975498, -0.0001863252971834427),
    c64::new(-0.00015790067896640227, -0.00018620424366093218),
    c64::new(-0.00015804344937414842, -0.00018608308059926248),
    c64::new(-0.00015818612680900518, -0.00018596180806971079),
    c64::new(-0.00015832871118703917, -0.00018584042614361863),
    c64::new(-0.00015847120242437157, -0.00018571893489239183),
    c64::new(-0.00015861360043717854, -0.00018559733438750056),
    c64::new(-0.00015875590514169085, -0.00018547562470047938),
    c64::new(-0.00015889811645419452, -0.0001853538059029268),
    c64::new(-0.0001590402342910302, -0.00018523187806650574),
    c64::new(-0.00015918225856859363, -0.00018510984126294318),
    c64::new(-0.00015932418920333558, -0.0001849876955640303),
    c64::new(-0.00015946602611176185, -0.00018486544104162226),
    c64::new(-0.00015960776921043358, -0.00018474307776763816),
    c64::new(-0.0001597494184159669, -0.00018462060581406117),
    c64::new(-0.0001598909736450332, -0.00018449802525293847),
    c64::new(-0.00016003243481435922, -0.000184375336156381),
    c64::new(-0.00016017380184072698, -0.00018425253859656357),
    c64::new(-0.00016031507464097383, -0.00018412963264572492),
    c64::new(-0.00016045625313199278, -0.00018400661837616734),
    c64::new(-0.00016059733723073206, -0.00018388349586025696),
    c64::new(-0.00016073832685419557, -0.00018376026517042367),
    c64::new(-0.0001608792219194427, -0.00018363692637916082),
    c64::new(-0.0001610200223435885, -0.00018351347955902555),
    c64::new(-0.00016116072804380377, -0.00018338992478263833),
    c64::new(-0.000161301338937315, -0.00018326626212268328),
    c64::new(-0.00016144185494140433, -0.00018314249165190798),
    c64::new(-0.00016158227597340988, -0.0001830186134431234),
    c64::new(-0.00016172260195072553, -0.00018289462756920392),
    c64::new(-0.00016186283279080109, -0.0001827705341030873),
    c64::new(-0.0001620029684111425, -0.00018264633311777434),
    c64::new(-0.0001621430087293115, -0.00018252202468632938),
    c64::new(-0.00016228295366292597, -0.00018239760888187992),
    c64::new(-0.00016242280312965996, -0.0001822730857776165),
    c64::new(-0.0001625625570472435, -0.00018214845544679297),
    c64::new(-0.00016270221533346317, -0.00018202371796272596),
    c64::new(-0.00016284177790616146, -0.00018189887339879542),
    c64::new(-0.00016298124468323735, -0.00018177392182844418),
    c64::new(-0.0001631206155826461, -0.00018164886332517805),
    c64::new(-0.00016325989052239944, -0.00018152369796256568),
    c64::new(-0.00016339906942056538, -0.00018139842581423873),
    c64::new(-0.00016353815219526878, -0.00018127304695389146),
    c64::new(-0.00016367713876469067, -0.00018114756145528102),
    c64::new(-0.0001638160290470689, -0.00018102196939222737),
    c64::new(-0.00016395482296069786, -0.00018089627083861304),
    c64::new(-0.0001640935204239286, -0.00018077046586838336),
    c64::new(-0.00016423212135516913, -0.00018064455455554598),
    c64::new(-0.000164370625672884, -0.00018051853697417135),
    c64::new(-0.00016450903329559474, -0.0001803924131983924),
    c64::new(-0.00016464734414187965, -0.00018026618330240445),
    c64::new(-0.00016478555813037403, -0.00018013984736046533),
    c64::new(-0.00016492367517977012, -0.00018001340544689527),
    c64::new(-0.00016506169520881734, -0.00017988685763607666),
    c64::new(-0.00016519961813632198, -0.0001797602040024544),
    c64::new(-0.00016533744388114754, -0.00017963344462053555),
    c64::new(-0.00016547517236221472, -0.00017950657956488937),
    c64::new(-0.00016561280349850128, -0.0001793796089101474),
    c64::new(-0.00016575033720904255, -0.00017925253273100305),
    c64::new(-0.0001658877734129309, -0.000179125351102212),
    c64::new(-0.00016602511202931615, -0.000178998064098592),
    c64::new(-0.00016616235297740552, -0.00017887067179502266),
    c64::new(-0.00016629949617646373, -0.00017874317426644563),
    c64::new(-0.0001664365415458129, -0.00017861557158786444),
    c64::new(-0.00016657348900483288, -0.00017848786383434435),
    c64::new(-0.00016671033847296093, -0.0001783600510810126),
    c64::new(-0.00016684708986969206, -0.00017823213340305814),
    c64::new(-0.00016698374311457898, -0.0001781041108757317),
    c64::new(-0.00016712029812723195, -0.00017797598357434562),
    c64::new(-0.0001672567548273194, -0.00017784775157427376),
    c64::new(-0.00016739311313456723, -0.00017771941495095184),
    c64::new(-0.00016752937296875942, -0.00017759097377987692),
    c64::new(-0.00016766553424973777, -0.00017746242813660763),
    c64::new(-0.00016780159689740214, -0.00017733377809676414),
    c64::new(-0.00016793756083171028, -0.00017720502373602795),
    c64::new(-0.0001680734259726783, -0.00017707616513014182),
    c64::new(-0.00016820919224038014, -0.00017694720235491),
    c64::new(-0.00016834485955494796, -0.000176818135486198),
    c64::new(-0.00016848042783657225, -0.00017668896459993247),
    c64::new(-0.00016861589700550156, -0.00017655968977210148),
    c64::new(-0.00016875126698204302, -0.00017643031107875386),
    c64::new(-0.00016888653768656192, -0.00017630082859599987),
    c64::new(-0.00016902170903948196, -0.00017617124240001069),
    c64::new(-0.0001691567809612853, -0.00017604155256701858),
    c64::new(-0.00016929175337251266, -0.00017591175917331672),
    c64::new(-0.00016942662619376313, -0.00017578186229525925),
    c64::new(-0.00016956139934569468, -0.00017565186200926104),
    c64::new(-0.00016969607274902363, -0.00017552175839179797),
    c64::new(-0.00016983064632452511, -0.0001753915515194067),
    c64::new(-0.00016996511999303296, -0.0001752612414686845),
    c64::new(-0.00017009949367543967, -0.00017513082831628954),
    c64::new(-0.00017023376729269688, -0.00017500031213894034),
    c64::new(-0.00017036794076581474, -0.00017486969301341632),
    c64::new(-0.0001705020140158625, -0.0001747389710165573),
    c64::new(-0.0001706359869639683, -0.0001746081462252637),
    c64::new(-0.0001707698595313193, -0.00017447721871649633),
    c64::new(-0.00017090363163916166, -0.0001743461885672766),
    c64::new(-0.00017103730320880084, -0.00017421505585468597),
    c64::new(-0.0001711708741616012, -0.00017408382065586657),
    c64::new(-0.00017130434441898644, -0.0001739524830480206),
    c64::new(-0.00017143771390243942, -0.00017382104310841067),
    c64::new(-0.00017157098253350222, -0.00017368950091435956),
    c64::new(-0.0001717041502337765, -0.00017355785654324997),
    c64::new(-0.00017183721692492305, -0.000173426110072525),
    c64::new(-0.00017197018252866218, -0.00017329426157968773),
    c64::new(-0.00017210304696677365, -0.0001731623111423012),
    c64::new(-0.00017223581016109673, -0.00017303025883798844),
    c64::new(-0.00017236847203353016, -0.0001728981047444325),
    c64::new(-0.00017250103250603257, -0.00017276584893937612),
    c64::new(-0.00017263349150062194, -0.00017263349150062197),
    c64::new(-0.00017276584893937606, -0.0001725010325060326),
    c64::new(-0.00017289810474443247, -0.0001723684720335302),
    c64::new(-0.00017303025883798838, -0.00017223581016109676),
    c64::new(-0.00017316231114230114, -0.0001721030469667737),
    c64::new(-0.0001732942615796877, -0.00017197018252866224),
    c64::new(-0.00017342611007252498, -0.0001718372169249231),
    c64::new(-0.00017355785654324992, -0.00017170415023377654),
    c64::new(-0.0001736895009143595, -0.00017157098253350224),
    c64::new(-0.00017382104310841061, -0.00017143771390243945),
    c64::new(-0.0001739524830480206, -0.0001713043444189865),
    c64::new(-0.0001740838206558665, -0.00017117087416160126),
    c64::new(-0.00017421505585468592, -0.00017103730320880086),
    c64::new(-0.00017434618856727657, -0.00017090363163916168),
    c64::new(-0.0001744772187164963, -0.00017076985953131933),
    c64::new(-0.00017460814622526365, -0.00017063598696396832),
    c64::new(-0.00017473897101655727, -0.00017050201401586253),
    c64::new(-0.00017486969301341626, -0.0001703679407658148),
    c64::new(-0.00017500031213894032, -0.0001702337672926969),
    c64::new(-0.00017513082831628949, -0.00017009949367543972),
    c64::new(-0.00017526124146868447, -0.00016996511999303298),
    c64::new(-0.00017539155151940664, -0.00016983064632452514),
    c64::new(-0.00017552175839179794, -0.00016969607274902365),
    c64::new(-0.00017565186200926098, -0.0001695613993456947),
    c64::new(-0.0001757818622952592, -0.00016942662619376316),
    c64::new(-0.00017591175917331666, -0.00016929175337251272),
    c64::new(-0.00017604155256701855, -0.00016915678096128536),
    c64::new(-0.00017617124240001066, -0.00016902170903948201),
    c64::new(-0.0001763008285959998, -0.00016888653768656198),
    c64::new(-0.00017643031107875384, -0.00016875126698204308),
    c64::new(-0.00017655968977210143, -0.0001686158970055016),
    c64::new(-0.00017668896459993247, -0.00016848042783657228),
    c64::new(-0.00017681813548619797, -0.000168344859554948),
    c64::new(-0.00017694720235490995, -0.00016820919224038016),
    c64::new(-0.0001770761651301418, -0.00016807342597267833),
    c64::new(-0.00017720502373602792, -0.00016793756083171034),
    c64::new(-0.0001773337780967641, -0.00016780159689740217),
    c64::new(-0.00017746242813660763, -0.0001676655342497378),
    c64::new(-0.00017759097377987687, -0.00016752937296875945),
    c64::new(-0.0001777194149509518, -0.00016739311313456728),
    c64::new(-0.00017784775157427373, -0.00016725675482731942),
    c64::new(-0.0001779759835743456, -0.00016712029812723198),
    c64::new(-0.00017810411087573167, -0.000166983743114579),
    c64::new(-0.00017823213340305814, -0.00016684708986969212),
    c64::new(-0.0001783600510810126, -0.00016671033847296096),
    c64::new(-0.0001784878638343443, -0.0001665734890048329),
    c64::new(-0.0001786155715878644, -0.00016643654154581292),
    c64::new(-0.0001787431742664456, -0.00016629949617646378),
    c64::new(-0.00017887067179502263, -0.00016616235297740558),
    c64::new(-0.00017899806409859198, -0.00016602511202931617),
    c64::new(-0.00017912535110221198, -0.00016588777341293093),
    c64::new(-0.000179252532731003, -0.00016575033720904258),
    c64::new(-0.00017937960891014737, -0.00016561280349850134),
    c64::new(-0.0001795065795648893, -0.00016547517236221475),
    c64::new(-0.0001796334446205355, -0.0001653374438811476),
    c64::new(-0.00017976020400245436, -0.000165199618136322),
    c64::new(-0.0001798868576360766, -0.00016506169520881736),
    c64::new(-0.00018001340544689524, -0.00016492367517977015),
    c64::new(-0.00018013984736046527, -0.0001647855581303741),
    c64::new(-0.0001802661833024044, -0.00016464734414187968),
    c64::new(-0.00018039241319839236, -0.00016450903329559476),
    c64::new(-0.0001805185369741713, -0.00016437062567288405),
    c64::new(-0.00018064455455554595, -0.00016423212135516915),
    c64::new(-0.00018077046586838333, -0.00016409352042392864),
    c64::new(-0.000180896270838613, -0.0001639548229606979),
    c64::new(-0.00018102196939222734, -0.00016381602904706893),
    c64::new(-0.000181147561455281, -0.00016367713876469072),
    c64::new(-0.00018127304695389143, -0.0001635381521952688),
    c64::new(-0.0001813984258142387, -0.00016339906942056543),
    c64::new(-0.00018152369796256565, -0.0001632598905223995),
    c64::new(-0.00018164886332517802, -0.00016312061558264616),
    c64::new(-0.00018177392182844415, -0.0001629812446832374),
    c64::new(-0.0001818988733987954, -0.00016284177790616152),
    c64::new(-0.00018202371796272594, -0.00016270221533346322),
    c64::new(-0.00018214845544679294, -0.00016256255704724356),
    c64::new(-0.00018227308577761648, -0.00016242280312966),
    c64::new(-0.0001823976088818799, -0.00016228295366292603),
    c64::new(-0.00018252202468632938, -0.00016214300872931153),
    c64::new(-0.00018264633311777432, -0.00016200296841114252),
    c64::new(-0.00018277053410308724, -0.00016186283279080114),
    c64::new(-0.00018289462756920387, -0.0001617226019507256),
    c64::new(-0.00018301861344312338, -0.0001615822759734099),
    c64::new(-0.00018314249165190795, -0.00016144185494140436),
    c64::new(-0.00018326626212268325, -0.00016130133893731502),
    c64::new(-0.00018338992478263827, -0.00016116072804380383),
    c64::new(-0.00018351347955902552, -0.00016102002234358851),
    c64::new(-0.0001836369263791608, -0.00016087922191944276),
    c64::new(-0.00018376026517042364, -0.00016073832685419562),
    c64::new(-0.00018388349586025694, -0.00016059733723073212),
    c64::new(-0.00018400661837616732, -0.0001604562531319928),
    c64::new(-0.0001841296326457249, -0.0001603150746409739),
    c64::new(-0.00018425253859656355, -0.00016017380184072703),
    c64::new(-0.00018437533615638097, -0.00016003243481435927),
    c64::new(-0.00018449802525293842, -0.00015989097364503324),
    c64::new(-0.00018462060581406117, -0.00015974941841596693),
    c64::new(-0.0001847430777676381, -0.0001596077692104336),
    c64::new(-0.00018486544104162224, -0.00015946602611176188),
    c64::new(-0.00018498769556403026, -0.00015932418920333564),
    c64::new(-0.00018510984126294316, -0.00015918225856859368),
    c64::new(-0.00018523187806650568, -0.00015904023429103026),
    c64::new(-0.00018535380590292675, -0.00015889811645419458),
    c64::new(-0.00018547562470047936, -0.00015875590514169088),
    c64::new(-0.00018559733438750054, -0.00015861360043717857),
    c64::new(-0.0001857189348923918, -0.00015847120242437162),
    c64::new(-0.0001858404261436186, -0.0001583287111870392),
    c64::new(-0.00018596180806971076, -0.00015818612680900524),
    c64::new(-0.00018608308059926243, -0.00015804344937414845),
    c64::new(-0.00018620424366093212, -0.0001579006789664023),
    c64::new(-0.00018632529718344267, -0.000157757815669755),
    c64::new(-0.00018644624109558148, -0.00015761485956824925),
    c64::new(-0.00018656707532620036, -0.00015747181074598243),
    c64::new(-0.0001866877998042156, -0.00015732866928710654),
    c64::new(-0.00018680841445860814, -0.00015718543527582798),
    c64::new(-0.0001869289192184234, -0.00015704210879640773),
    c64::new(-0.00018704931401277168, -0.0001568986899331609),
    c64::new(-0.00018716959877082773, -0.0001567551787704572),
    c64::new(-0.00018728977342183113, -0.0001566115753927205),
    c64::new(-0.00018740983789508628, -0.00015646787988442905),
    c64::new(-0.00018752979211996226, -0.00015632409233011513),
    c64::new(-0.0001876496360258931, -0.0001561802128143654),
    c64::new(-0.00018776936954237783, -0.00015603624142182027),
    c64::new(-0.0001878889925989802, -0.0001558921782371745),
    c64::new(-0.00018800850512532908, -0.00015574802334517677),
    c64::new(-0.00018812790705111836, -0.00015560377683062967),
    c64::new(-0.00018824719830610683, -0.00015545943877838975),
    c64::new(-0.0001883663788201187, -0.00015531500927336726),
    c64::new(-0.00018848544852304307, -0.00015517048840052642),
    c64::new(-0.00018860440734483428, -0.00015502587624488515),
    c64::new(-0.00018872325521551196, -0.000154881172891515),
    c64::new(-0.00018884199206516093, -0.0001547363784255413),
    c64::new(-0.0001889606178239313, -0.00015459149293214292),
    c64::new(-0.00018907913242203876, -0.00015444651649655216),
    c64::new(-0.00018919753578976412, -0.00015430144920405495),
    c64::new(-0.00018931582785745374, -0.0001541562911399907),
    c64::new(-0.00018943400855551945, -0.00015401104238975207),
    c64::new(-0.00018955207781443852, -0.00015386570303878533),
    c64::new(-0.000189670035564754, -0.00015372027317258965),
    c64::new(-0.00018978788173707427, -0.0001535747528767178),
    c64::new(-0.0001899056162620735, -0.00015342914223677556),
    c64::new(-0.0001900232390704915, -0.00015328344133842201),
    c64::new(-0.0001901407500931338, -0.00015313765026736916),
    c64::new(-0.0001902581492608716, -0.00015299176910938225),
    c64::new(-0.00019037543650464217, -0.0001528457979502792),
    c64::new(-0.0001904926117554484, -0.00015269973687593124),
    c64::new(-0.000190609674944359, -0.0001525535859722623),
    c64::new(-0.0001907266260025088, -0.00015240734532524918),
    c64::new(-0.00019084346486109846, -0.0001522610150209216),
    c64::new(-0.00019096019145139472, -0.0001521145951453617),
    c64::new(-0.00019107680570473038, -0.0001519680857847046),
    c64::new(-0.00019119330755250423, -0.00015182148702513803),
    c64::new(-0.00019130969692618123, -0.00015167479895290223),
    c64::new(-0.0001914259737572925, -0.00015152802165429005),
    c64::new(-0.00019154213797743533, -0.00015138115521564683),
    c64::new(-0.00019165818951827343, -0.00015123419972337015),
    c64::new(-0.0001917741283115366, -0.00015108715526391026),
    c64::new(-0.00019188995428902103, -0.0001509400219237696),
    c64::new(-0.0001920056673825893, -0.0001507927997895029),
    c64::new(-0.0001921212675241703, -0.00015064548894771728),
    c64::new(-0.0001922367546457596, -0.00015049808948507172),
    c64::new(-0.00019235212867941897, -0.0001503506014882776),
    c64::new(-0.0001924673895572769, -0.00015020302504409837),
    c64::new(-0.0001925825372115284, -0.00015005536023934938),
    c64::new(-0.000192697571574435, -0.00014990760716089804),
    c64::new(-0.0001928124925783249, -0.00014975976589566377),
    c64::new(-0.00019292730015559324, -0.00014961183653061763),
    c64::new(-0.00019304199423870156, -0.00014946381915278273),
    c64::new(-0.00019315657476017825, -0.00014931571384923386),
    c64::new(-0.00019327104165261862, -0.00014916752070709757),
    c64::new(-0.0001933853948486846, -0.00014901923981355217),
    c64::new(-0.00019349963428110537, -0.00014887087125582728),
    c64::new(-0.00019361375988267667, -0.00014872241512120442),
    c64::new(-0.00019372777158626137, -0.00014857387149701647),
    c64::new(-0.00019384166932478935, -0.00014842524047064786),
    c64::new(-0.00019395545303125747, -0.00014827652212953438),
    c64::new(-0.00019406912263872969, -0.00014812771656116326),
    c64::new(-0.0001941826780803372, -0.00014797882385307284),
    c64::new(-0.0001942961192892782, -0.00014782984409285292),
    c64::new(-0.00019440944619881814, -0.0001476807773681445),
    c64::new(-0.00019452265874228975, -0.00014753162376663964),
    c64::new(-0.00019463575685309292, -0.0001473823833760817),
    c64::new(-0.00019474874046469505, -0.00014723305628426474),
    c64::new(-0.0001948616095106308, -0.00014708364257903415),
    c64::new(-0.00019497436392450215, -0.00014693414234828617),
    c64::new(-0.0001950870036399786, -0.0001467845556799679),
    c64::new(-0.00019519952859079712, -0.00014663488266207736),
    c64::new(-0.00019531193871076216, -0.0001464851233826634),
    c64::new(-0.00019542423393374578, -0.00014633527792982535),
    c64::new(-0.00019553641419368757, -0.00014618534639171358),
    c64::new(-0.00019564847942459479, -0.0001460353288565289),
    c64::new(-0.0001957604295605423, -0.00014588522541252274),
    c64::new(-0.00019587226453567276, -0.00014573503614799725),
    c64::new(-0.00019598398428419654, -0.00014558476115130467),
    c64::new(-0.0001960955887403918, -0.00014543440051084801),
    c64::new(-0.0001962070778386045, -0.00014528395431508058),
    c64::new(-0.00019631845151324846, -0.00014513342265250596),
    c64::new(-0.00019642970969880545, -0.0001449828056116781),
    c64::new(-0.00019654085232982506, -0.00014483210328120118),
    c64::new(-0.00019665187934092505, -0.00014468131574972933),
    c64::new(-0.00019676279066679108, -0.00014453044310596704),
    c64::new(-0.00019687358624217684, -0.00014437948543866884),
    c64::new(-0.00019698426600190413, -0.0001442284428366392),
    c64::new(-0.00019709482988086285, -0.0001440773153887327),
    c64::new(-0.0001972052778140113, -0.00014392610318385353),
    c64::new(-0.00019731560973637564, -0.00014377480631095604),
    c64::new(-0.00019742582558305053, -0.00014362342485904427),
    c64::new(-0.0001975359252891988, -0.00014347195891717204),
    c64::new(-0.0001976459087900516, -0.00014332040857444284),
    c64::new(-0.00019775577602090843, -0.00014316877392000994),
    c64::new(-0.00019786552691713737, -0.00014301705504307592),
    c64::new(-0.00019797516141417472, -0.0001428652520328932),
    c64::new(-0.00019808467944752532, -0.0001427133649787635),
    c64::new(-0.00019819408095276257, -0.00014256139397003813),
    c64::new(-0.00019830336586552824, -0.0001424093390961178),
    c64::new(-0.00019841253412153305, -0.0001422572004464523),
    c64::new(-0.00019852158565655603, -0.00014210497811054093),
    c64::new(-0.00019863052040644492, -0.0001419526721779322),
    c64::new(-0.0001987393383071163, -0.00014180028273822376),
    c64::new(-0.00019884803929455535, -0.0001416478098810624),
    c64::new(-0.00019895662330481602, -0.00014149525369614403),
    c64::new(-0.00019906509027402128, -0.00014134261427321338),
    c64::new(-0.00019917344013836274, -0.00014118989170206438),
    c64::new(-0.00019928167283410093, -0.00014103708607253975),
    c64::new(-0.0001993897882975654, -0.00014088419747453111),
    c64::new(-0.00019949778646515448, -0.00014073122599797902),
    c64::new(-0.00019960566727333584, -0.00014057817173287242),
    c64::new(-0.00019971343065864586, -0.00014042503476924934),
    c64::new(-0.0001998210765576901, -0.00014027181519719626),
    c64::new(-0.0001999286049071433, -0.00014011851310684833),
    c64::new(-0.00020003601564374927, -0.00013996512858838918),
    c64::new(-0.00020014330870432101, -0.00013981166173205115),
    c64::new(-0.00020025048402574086, -0.00013965811262811457),
    c64::new(-0.00020035754154496032, -0.00013950448136690853),
    c64::new(-0.00020046448119900016, -0.00013935076803881037),
    c64::new(-0.0002005713029249506, -0.00013919697273424568),
    c64::new(-0.00020067800665997106, -0.00013904309554368835),
    c64::new(-0.0002007845923412906, -0.00013888913655766018),
    c64::new(-0.00020089105990620757, -0.00013873509586673137),
    c64::new(-0.00020099740929208987, -0.0001385809735615201),
    c64::new(-0.00020110364043637486, -0.00013842676973269255),
    c64::new(-0.00020120975327656948, -0.00013827248447096283),
    c64::new(-0.00020131574775025026, -0.00013811811786709313),
    c64::new(-0.0002014216237950635, -0.00013796367001189311),
    c64::new(-0.00020152738134872496, -0.00013780914099622058),
    c64::new(-0.0002016330203490202, -0.000137654530910981),
    c64::new(-0.00020173854073380457, -0.00013749983984712746),
    c64::new(-0.00020184394244100303, -0.00013734506789566083),
    c64::new(-0.0002019492254086106, -0.0001371902151476293),
    c64::new(-0.00020205438957469196, -0.00013703528169412883),
    c64::new(-0.00020215943487738176, -0.00013688026762630274),
    c64::new(-0.00020226436125488455, -0.0001367251730353419),
    c64::new(-0.00020236916864547485, -0.00013656999801248438),
    c64::new(-0.00020247385698749708, -0.00013641474264901576),
    c64::new(-0.00020257842621936595, -0.00013625940703626863),
    c64::new(-0.00020268287627956598, -0.00013610399126562296),
    c64::new(-0.00020278720710665193, -0.00013594849542850588),
    c64::new(-0.0002028914186392486, -0.00013579291961639159),
    c64::new(-0.0002029955108160511, -0.0001356372639208013),
    c64::new(-0.00020309948357582458, -0.00013548152843330337),
    c64::new(-0.00020320333685740468, -0.00013532571324551275),
    c64::new(-0.00020330707059969713, -0.00013516981844909163),
    c64::new(-0.00020341068474167802, -0.0001350138441357489),
    c64::new(-0.00020351417922239386, -0.00013485779039724023),
    c64::new(-0.00020361755398096143, -0.00013470165732536808),
    c64::new(-0.00020372080895656814, -0.00013454544501198133),
    c64::new(-0.00020382394408847168, -0.0001343891535489758),
    c64::new(-0.00020392695931600033, -0.00013423278302829368),
    c64::new(-0.00020402985457855288, -0.00013407633354192374),
    c64::new(-0.0002041326298155987, -0.0001339198051819012),
    c64::new(-0.0002042352849666777, -0.0001337631980403077),
    c64::new(-0.0002043378199714006, -0.00013360651220927104),
    c64::new(-0.0002044402347694487, -0.00013344974778096557),
    c64::new(-0.00020454252930057396, -0.0001332929048476117),
    c64::new(-0.00020464470350459914, -0.00013313598350147616),
    c64::new(-0.00020474675732141777, -0.00013297898383487177),
    c64::new(-0.0002048486906909943, -0.00013282190594015724),
    c64::new(-0.00020495050355336388, -0.00013266474990973753),
    c64::new(-0.0002050521958486327, -0.00013250751583606345),
    c64::new(-0.00020515376751697773, -0.00013235020381163182),
    c64::new(-0.00020525521849864702, -0.0001321928139289852),
    c64::new(-0.00020535654873395947, -0.0001320353462807121),
    c64::new(-0.00020545775816330526, -0.00013187780095944652),
    c64::new(-0.00020555884672714543, -0.0001317201780578684),
    c64::new(-0.00020565981436601217, -0.00013156247766870323),
    c64::new(-0.00020576066102050885, -0.00013140469988472207),
    c64::new(-0.0002058613866313099, -0.00013124684479874163),
    c64::new(-0.00020596199113916114, -0.0001310889125036238),
    c64::new(-0.0002060624744848795, -0.00013093090309227617),
    c64::new(-0.0002061628366093532, -0.00013077281665765163),
    c64::new(-0.00020626307745354178, -0.00013061465329274833),
    c64::new(-0.00020636319695847616, -0.00013045641309060972),
    c64::new(-0.00020646319506525853, -0.0001302980961443245),
    c64::new(-0.0002065630717150627, -0.0001301397025470263),
    c64::new(-0.0002066628268491338, -0.00012998123239189405),
    c64::new(-0.0002067624604087884, -0.0001298226857721517),
    c64::new(-0.00020686197233541465, -0.0001296640627810681),
    c64::new(-0.0002069613625704722, -0.0001295053635119572),
    c64::new(-0.00020706063105549246, -0.00012934658805817745),
    c64::new(-0.00020715977773207828, -0.0001291877365131325),
    c64::new(-0.0002072588025419042, -0.00012902880897027063),
    c64::new(-0.0002073577054267165, -0.00012886980552308477),
    c64::new(-0.00020745648632833314, -0.00012871072626511262),
    c64::new(-0.00020755514518864382, -0.00012855157128993641),
    c64::new(-0.0002076536819496102, -0.00012839234069118283),
    c64::new(-0.00020775209655326556, -0.00012823303456252312),
    c64::new(-0.00020785038894171514, -0.00012807365299767308),
    c64::new(-0.00020794855905713604, -0.0001279141960903927),
    c64::new(-0.00020804660684177727, -0.0001277546639344865),
    c64::new(-0.00020814453223795995, -0.00012759505662380294),
    c64::new(-0.00020824233518807706, -0.00012743537425223496),
    c64::new(-0.0002083400156345936, -0.00012727561691371964),
    c64::new(-0.00020843757352004673, -0.00012711578470223808),
    c64::new(-0.00020853500878704562, -0.0001269558777118154),
    c64::new(-0.0002086323213782716, -0.00012679589603652094),
    c64::new(-0.00020872951123647823, -0.00012663583977046757),
    c64::new(-0.00020882657830449122, -0.00012647570900781237),
    c64::new(-0.00020892352252520845, -0.00012631550384275612),
    c64::new(-0.00020902034384160024, -0.0001261552243695434),
    c64::new(-0.00020911704219670895, -0.0001259948706824626),
    c64::new(-0.00020921361753364957, -0.0001258344428758455),
    c64::new(-0.00020931006979560927, -0.00012567394104406768),
    c64::new(-0.00020940639892584769, -0.00012551336528154828),
    c64::new(-0.0002095026048676968, -0.00012535271568274987),
    c64::new(-0.00020959868756456124, -0.00012519199234217846),
    c64::new(-0.0002096946469599179, -0.00012503119535438352),
    c64::new(-0.00020979048299731647, -0.00012487032481395765),
    c64::new(-0.00020988619562037906, -0.0001247093808155369),
    c64::new(-0.00020998178477280033, -0.0001245483634538005),
    c64::new(-0.0002100772503983477, -0.00012438727282347077),
    c64::new(-0.00021017259244086113, -0.0001242261090193133),
    c64::new(-0.00021026781084425343, -0.00012406487213613643),
    c64::new(-0.00021036290555251008, -0.00012390356226879174),
    c64::new(-0.00021045787650968926, -0.00012374217951217366),
    c64::new(-0.000210552723659922, -0.00012358072396121954),
    c64::new(-0.00021064744694741222, -0.0001234191957109095),
    c64::new(-0.00021074204631643654, -0.00012325759485626653),
    c64::new(-0.00021083652171134472, -0.0001230959214923561),
    c64::new(-0.00021093087307655925, -0.00012293417571428652),
    c64::new(-0.0002110251003565757, -0.00012277235761720874),
    c64::new(-0.0002111192034959625, -0.00012261046729631608),
    c64::new(-0.00021121318243936122, -0.00012244850484684463),
    c64::new(-0.00021130703713148657, -0.0001222864703640725),
    c64::new(-0.00021140076751712618, -0.00012212436394332048),
    c64::new(-0.00021149437354114087, -0.00012196218567995163),
    c64::new(-0.00021158785514846463, -0.00012179993566937124),
    c64::new(-0.00021168121228410465, -0.0001216376140070268),
    c64::new(-0.0002117744448931413, -0.00012147522078840805),
    c64::new(-0.00021186755292072832, -0.00012131275610904656),
    c64::new(-0.00021196053631209258, -0.0001211502200645162),
    c64::new(-0.0002120533950125344, -0.00012098761275043275),
    c64::new(-0.00021214612896742737, -0.00012082493426245389),
    c64::new(-0.0002122387381222184, -0.00012066218469627928),
    c64::new(-0.00021233122242242812, -0.00012049936414765013),
    c64::new(-0.00021242358181365023, -0.00012033647271234965),
    c64::new(-0.00021251581624155214, -0.00012017351048620266),
    c64::new(-0.00021260792565187472, -0.00012001047756507566),
    c64::new(-0.00021269990999043238, -0.00011984737404487672),
    c64::new(-0.00021279176920311303, -0.00011968420002155551),
    c64::new(-0.00021288350323587834, -0.00011952095559110293),
    c64::new(-0.00021297511203476353, -0.00011935764084955157),
    c64::new(-0.00021306659554587752, -0.00011919425589297529),
    c64::new(-0.00021315795371540288, -0.00011903080081748923),
    c64::new(-0.0002132491864895959, -0.00011886727571924991),
    c64::new(-0.0002133402938147868, -0.00011870368069445473),
    c64::new(-0.00021343127563737944, -0.00011854001583934255),
    c64::new(-0.0002135221319038516, -0.00011837628125019314),
    c64::new(-0.0002136128625607548, -0.0001182124770233274),
    c64::new(-0.00021370346755471457, -0.00011804860325510709),
    c64::new(-0.0002137939468324303, -0.00011788466004193503),
    c64::new(-0.0002138843003406754, -0.00011772064748025461),
    c64::new(-0.00021397452802629727, -0.00011755656566655029),
    c64::new(-0.0002140646298362172, -0.00011739241469734716),
    c64::new(-0.00021415460571743068, -0.00011722819466921102),
    c64::new(-0.0002142444556170071, -0.00011706390567874836),
    c64::new(-0.00021433417948209027, -0.00011689954782260598),
    c64::new(-0.00021442377725989782, -0.00011673512119747148),
    c64::new(-0.0002145132488977217, -0.00011657062590007278),
    c64::new(-0.0002146025943429281, -0.00011640606202717824),
    c64::new(-0.00021469181354295735, -0.00011624142967559656),
    c64::new(-0.00021478090644532405, -0.00011607672894217678),
    c64::new(-0.00021486987299761725, -0.00011591195992380793),
    c64::new(-0.00021495871314750018, -0.00011574712271741945),
    c64::new(-0.00021504742684271043, -0.00011558221741998084),
    c64::new(-0.00021513601403106008, -0.00011541724412850164),
    c64::new(-0.00021522447466043547, -0.00011525220294003148),
    c64::new(-0.0002153128086787976, -0.00011508709395165962),
    c64::new(-0.0002154010160341818, -0.00011492191726051557),
    c64::new(-0.00021548909667469795, -0.00011475667296376848),
    c64::new(-0.00021557705054853048, -0.00011459136115862732),
    c64::new(-0.0002156648776039384, -0.00011442598194234078),
    c64::new(-0.00021575257778925523, -0.00011426053541219728),
    c64::new(-0.00021584015105288937, -0.00011409502166552456),
    c64::new(-0.00021592759734332366, -0.00011392944079969019),
    c64::new(-0.00021601491660911567, -0.00011376379291210112),
    c64::new(-0.00021610210879889773, -0.00011359807810020375),
    c64::new(-0.00021618917386137691, -0.00011343229646148391),
    c64::new(-0.00021627611174533513, -0.00011326644809346649),
    c64::new(-0.00021636292239962903, -0.0001131005330937159),
    c64::new(-0.0002164496057731902, -0.00011293455155983566),
    c64::new(-0.00021653616181502495, -0.00011276850358946844),
    c64::new(-0.00021662259047421466, -0.00011260238928029596),
    c64::new(-0.00021670889169991548, -0.00011243620873003912),
    c64::new(-0.00021679506544135875, -0.00011226996203645743),
    c64::new(-0.00021688111164785065, -0.00011210364929734966),
    c64::new(-0.00021696703026877236, -0.00011193727061055329),
    c64::new(-0.00021705282125358022, -0.00011177082607394462),
    c64::new(-0.00021713848455180552, -0.00011160431578543877),
    c64::new(-0.00021722402011305485, -0.00011143773984298929),
    c64::new(-0.00021730942788700982, -0.00011127109834458858),
    c64::new(-0.00021739470782342722, -0.00011110439138826755),
    c64::new(-0.00021747985987213904, -0.0001109376190720956),
    c64::new(-0.00021756488398305256, -0.0001107707814941806),
    c64::new(-0.0002176497801061502, -0.00011060387875266891),
    c64::new(-0.0002177345481914899, -0.00011043691094574487),
    c64::new(-0.0002178191881892047, -0.0001102698781716315),
    c64::new(-0.00021790370004950307, -0.00011010278052858981),
    c64::new(-0.0002179880837226689, -0.00010993561811491908),
    c64::new(-0.00021807233915906133, -0.00010976839102895676),
    c64::new(-0.0002181564663091152, -0.000109601099369078),
    c64::new(-0.00021824046512334064, -0.0001094337432336963),
    c64::new(-0.00021832433555232325, -0.00010926632272126296),
    c64::new(-0.0002184080775467243, -0.00010909883793026713),
    c64::new(-0.00021849169105728046, -0.00010893128895923583),
    c64::new(-0.00021857517603480407, -0.00010876367590673388),
    c64::new(-0.0002186585324301831, -0.00010859599887136354),
    c64::new(-0.00021874176019438113, -0.00010842825795176497),
    c64::new(-0.0002188248592784374, -0.00010826045324661584),
    c64::new(-0.00021890782963346686, -0.00010809258485463134),
    c64::new(-0.00021899067121066012, -0.00010792465287456422),
    c64::new(-0.00021907338396128374, -0.00010775665740520432),
    c64::new(-0.00021915596783667987, -0.00010758859854537915),
    c64::new(-0.00021923842278826654, -0.00010742047639395345),
    c64::new(-0.00021932074876753767, -0.00010725229104982909),
    c64::new(-0.00021940294572606297, -0.0001070840426119452),
    c64::new(-0.00021948501361548803, -0.00010691573117927811),
    c64::new(-0.00021956695238753454, -0.00010674735685084088),
    c64::new(-0.00021964876199399998, -0.00010657891972568391),
    c64::new(-0.00021973044238675788, -0.0001064104199028944),
    c64::new(-0.00021981199351775774, -0.00010624185748159649),
    c64::new(-0.00021989341533902508, -0.00010607323256095117),
    c64::new(-0.00021997470780266167, -0.00010590454524015594),
    c64::new(-0.0002200558708608452, -0.00010573579561844533),
    c64::new(-0.00022013690446582952, -0.00010556698379509038),
    c64::new(-0.00022021780856994462, -0.00010539810986939874),
    c64::new(-0.00022029858312559676, -0.00010522917394071458),
    c64::new(-0.00022037922808526824, -0.00010506017610841867),
    c64::new(-0.00022045974340151783, -0.00010489111647192787),
    c64::new(-0.00022054012902698042, -0.0001047219951306957),
    c64::new(-0.00022062038491436718, -0.0001045528121842119),
    c64::new(-0.00022070051101646566, -0.00010438356773200245),
    c64::new(-0.0002207805072861397, -0.00010421426187362958),
    c64::new(-0.00022086037367632962, -0.00010404489470869137),
    c64::new(-0.00022094011014005206, -0.00010387546633682226),
    c64::new(-0.00022101971663040007, -0.00010370597685769259),
    c64::new(-0.00022109919310054328, -0.00010353642637100868),
    c64::new(-0.00022117853950372765, -0.00010336681497651268),
    c64::new(-0.00022125775579327577, -0.00010319714277398272),
    c64::new(-0.00022133684192258675, -0.00010302740986323238),
    c64::new(-0.00022141579784513624, -0.00010285761634411127),
    c64::new(-0.00022149462351447652, -0.00010268776231650452),
    c64::new(-0.00022157331888423648, -0.0001025178478803329),
    c64::new(-0.0002216518839081216, -0.00010234787313555279),
    c64::new(-0.0002217303185399142, -0.00010217783818215574),
    c64::new(-0.00022180862273347317, -0.00010200774312016905),
    c64::new(-0.00022188679644273413, -0.00010183758804965526),
    c64::new(-0.00022196483962170957, -0.0001016673730707122),
    c64::new(-0.00022204275222448863, -0.00010149709828347296),
    c64::new(-0.00022212053420523733, -0.00010132676378810592),
    c64::new(-0.00022219818551819862, -0.00010115636968481427),
    c64::new(-0.00022227570611769217, -0.00010098591607383658),
    c64::new(-0.00022235309595811464, -0.00010081540305544631),
    c64::new(-0.00022243035499393954, -0.00010064483072995186),
    c64::new(-0.00022250748317971734, -0.00010047419919769665),
    c64::new(-0.00022258448047007563, -0.00010030350855905863),
    c64::new(-0.0002226613468197188, -0.00010013275891445077),
    c64::new(-0.0002227380821834284, -9.996195036432071e-05),
    c64::new(-0.00022281468651606296, -9.979108300915072e-05),
    c64::new(-0.00022289115977255816, -9.962015694945766e-05),
    c64::new(-0.00022296750190792667, -9.944917228579302e-05),
    c64::new(-0.0002230437128772585, -9.927812911874248e-05),
    c64::new(-0.0002231197926357206, -9.910702754892641e-05),
    c64::new(-0.0002231957411385573, -9.893586767699948e-05),
    c64::new(-0.00022327155834108998, -9.876464960365062e-05),
    c64::new(-0.00022334724419871728, -9.859337342960313e-05),
    c64::new(-0.00022342279866691532, -9.842203925561413e-05),
    c64::new(-0.00022349822170123724, -9.825064718247523e-05),
    c64::new(-0.00022357351325731365, -9.807919731101198e-05),
    c64::new(-0.00022364867329085247, -9.790768974208389e-05),
    c64::new(-0.00022372370175763895, -9.773612457658447e-05),
    c64::new(-0.0002237985986135358, -9.756450191544118e-05),
    c64::new(-0.00022387336381448313, -9.7392821859615e-05),
    c64::new(-0.0002239479973164985, -9.7221084510101e-05),
    c64::new(-0.00022402249907567696, -9.704928996792783e-05),
    c64::new(-0.00022409686904819104, -9.687743833415778e-05),
    c64::new(-0.00022417110719029072, -9.670552970988685e-05),
    c64::new(-0.00022424521345830375, -9.653356419624427e-05),
    c64::new(-0.00022431918780863524, -9.6361541894393e-05),
    c64::new(-0.000224393030197768, -9.618946290552933e-05),
    c64::new(-0.00022446674058226247, -9.601732733088291e-05),
    c64::new(-0.00022454031891875668, -9.584513527171665e-05),
    c64::new(-0.00022461376516396642, -9.56728868293268e-05),
    c64::new(-0.00022468707927468518, -9.550058210504249e-05),
    c64::new(-0.00022476026120778414, -9.532822120022625e-05),
    c64::new(-0.00022483331092021226, -9.515580421627355e-05),
    c64::new(-0.00022490622836899627, -9.498333125461287e-05),
    c64::new(-0.00022497901351124067, -9.481080241670571e-05),
    c64::new(-0.00022505166630412793, -9.463821780404612e-05),
    c64::new(-0.00022512418670491823, -9.44655775181613e-05),
    c64::new(-0.00022519657467094968, -9.42928816606111e-05),
    c64::new(-0.00022526883015963834, -9.4120130332988e-05),
    c64::new(-0.00022534095312847811, -9.394732363691718e-05),
    c64::new(-0.0002254129435350409, -9.377446167405647e-05),
    c64::new(-0.0002254848013369767, -9.36015445460959e-05),
    c64::new(-0.00022555652649201336, -9.342857235475826e-05),
    c64::new(-0.0002256281189579568, -9.325554520179865e-05),
    c64::new(-0.0002256995786926911, -9.308246318900447e-05),
    c64::new(-0.00022577090565417817, -9.290932641819551e-05),
    c64::new(-0.00022584209980045837, -9.273613499122348e-05),
    c64::new(-0.00022591316108964995, -9.256288900997253e-05),
    c64::new(-0.0002259840894799494, -9.238958857635879e-05),
    c64::new(-0.00022605488492963136, -9.221623379233048e-05),
    c64::new(-0.00022612554739704868, -9.204282475986771e-05),
    c64::new(-0.00022619607684063242, -9.186936158098268e-05),
    c64::new(-0.00022626647321889203, -9.169584435771909e-05),
    c64::new(-0.00022633673649041506, -9.152227319215275e-05),
    c64::new(-0.00022640686661386746, -9.134864818639111e-05),
    c64::new(-0.00022647686354799345, -9.11749694425733e-05),
    c64::new(-0.00022654672725161567, -9.100123706287013e-05),
    c64::new(-0.0002266164576836351, -9.08274511494837e-05),
    c64::new(-0.00022668605480303112, -9.065361180464786e-05),
    c64::new(-0.00022675551856886155, -9.047971913062785e-05),
    c64::new(-0.00022682484894026263, -9.030577322972024e-05),
    c64::new(-0.00022689404587644912, -9.013177420425291e-05),
    c64::new(-0.00022696310933671416, -8.995772215658511e-05),
    c64::new(-0.00022703203928042958, -8.978361718910697e-05),
    c64::new(-0.00022710083566704564, -8.960945940424006e-05),
    c64::new(-0.00022716949845609123, -8.943524890443692e-05),
    c64::new(-0.00022723802760717376, -8.92609857921811e-05),
    c64::new(-0.0002273064230799793, -8.908667016998721e-05),
    c64::new(-0.00022737468483427254, -8.891230214040042e-05),
    c64::new(-0.00022744281282989693, -8.873788180599705e-05),
    c64::new(-0.00022751080702677445, -8.85634092693841e-05),
    c64::new(-0.00022757866738490587, -8.838888463319926e-05),
    c64::new(-0.00022764639386437072, -8.821430800011088e-05),
    c64::new(-0.00022771398642532718, -8.8039679472818e-05),
    c64::new(-0.0002277814450280124, -8.786499915404989e-05),
    c64::new(-0.0002278487696327422, -8.769026714656656e-05),
    c64::new(-0.00022791596019991117, -8.751548355315835e-05),
    c64::new(-0.0002279830166899929, -8.734064847664596e-05),
    c64::new(-0.00022804993906353974, -8.716576201988045e-05),
    c64::new(-0.00022811672728118305, -8.699082428574282e-05),
    c64::new(-0.00022818338130363298, -8.681583537714454e-05),
    c64::new(-0.00022824990109167873, -8.664079539702706e-05),
    c64::new(-0.00022831628660618838, -8.64657044483619e-05),
    c64::new(-0.00022838253780810906, -8.629056263415056e-05),
    c64::new(-0.00022844865465846686, -8.611537005742455e-05),
    c64::new(-0.00022851463711836702, -8.594012682124495e-05),
    c64::new(-0.00022858048514899372, -8.576483302870293e-05),
    c64::new(-0.00022864619871161023, -8.558948878291933e-05),
    c64::new(-0.000228711777767559, -8.541409418704464e-05),
    c64::new(-0.0002287772222782615, -8.52386493442591e-05),
    c64::new(-0.00022884253220521847, -8.506315435777216e-05),
    c64::new(-0.00022890770751000976, -8.488760933082312e-05),
    c64::new(-0.00022897274815429438, -8.47120143666806e-05),
    c64::new(-0.00022903765409981063, -8.453636956864258e-05),
    c64::new(-0.00022910242530837602, -8.436067504003637e-05),
    c64::new(-0.00022916706174188722, -8.418493088421865e-05),
    c64::new(-0.00022923156336232043, -8.400913720457496e-05),
    c64::new(-0.00022929593013173093, -8.38332941045203e-05),
    c64::new(-0.0002293601620122534, -8.365740168749864e-05),
    c64::new(-0.0002294242589661019, -8.348146005698294e-05),
    c64::new(-0.00022948822095556978, -8.330546931647527e-05),
    c64::new(-0.00022955204794302997, -8.31294295695062e-05),
    c64::new(-0.00022961573989093464, -8.29533409196355e-05),
    c64::new(-0.00022967929676181545, -8.277720347045158e-05),
    c64::new(-0.0002297427185182835, -8.260101732557156e-05),
    c64::new(-0.00022980600512302948, -8.242478258864124e-05),
    c64::new(-0.0002298691565388234, -8.224849936333503e-05),
    c64::new(-0.00022993217272851505, -8.207216775335564e-05),
    c64::new(-0.00022999505365503352, -8.18957878624345e-05),
    c64::new(-0.00023005779928138764, -8.171935979433137e-05),
    c64::new(-0.00023012040957066574, -8.154288365283431e-05),
    c64::new(-0.0002301828844860358, -8.136635954175982e-05),
    c64::new(-0.00023024522399074546, -8.118978756495226e-05),
    c64::new(-0.000230307428048122, -8.101316782628449e-05),
    c64::new(-0.00023036949662157236, -8.083650042965734e-05),
    c64::new(-0.00023043142967458318, -8.06597854789997e-05),
    c64::new(-0.00023049322717072086, -8.048302307826842e-05),
    c64::new(-0.0002305548890736315, -8.030621333144835e-05),
    c64::new(-0.00023061641534704105, -8.012935634255193e-05),
    c64::new(-0.00023067780595475518, -7.995245221561966e-05),
    c64::new(-0.00023073906086065933, -7.977550105471968e-05),
    c64::new(-0.00023080018002871882, -7.95985029639478e-05),
    c64::new(-0.00023086116342297883, -7.942145804742754e-05),
    c64::new(-0.0002309220110075644, -7.924436640930966e-05),
    c64::new(-0.0002309827227466805, -7.906722815377271e-05),
    c64::new(-0.00023104329860461196, -7.889004338502256e-05),
    c64::new(-0.00023110373854572352, -7.871281220729244e-05),
    c64::new(-0.00023116404253446, -7.853553472484289e-05),
    c64::new(-0.00023122421053534597, -7.835821104196179e-05),
    c64::new(-0.0002312842425129863, -7.818084126296389e-05),
    c64::new(-0.0002313441384320657, -7.800342549219136e-05),
    c64::new(-0.00023140389825734888, -7.782596383401331e-05),
    c64::new(-0.00023146352195368072, -7.76484563928259e-05),
    c64::new(-0.00023152300948598608, -7.747090327305225e-05),
    c64::new(-0.00023158236081927008, -7.72933045791421e-05),
    c64::new(-0.00023164157591861777, -7.71156604155723e-05),
    c64::new(-0.00023170065474919447, -7.69379708868463e-05),
    c64::new(-0.00023175959727624563, -7.67602360974943e-05),
    c64::new(-0.00023181840346509684, -7.65824561520731e-05),
    c64::new(-0.00023187707328115395, -7.640463115516613e-05),
    c64::new(-0.00023193560668990308, -7.622676121138308e-05),
    c64::new(-0.00023199400365691046, -7.604884642536036e-05),
    c64::new(-0.00023205226414782268, -7.587088690176063e-05),
    c64::new(-0.0002321103881283666, -7.569288274527291e-05),
    c64::new(-0.00023216837556434936, -7.551483406061255e-05),
    c64::new(-0.0002322262264216585, -7.533674095252081e-05),
    c64::new(-0.0002322839406662618, -7.515860352576539e-05),
    c64::new(-0.0002323415182642075, -7.498042188513993e-05),
    c64::new(-0.0002323989591816241, -7.480219613546409e-05),
    c64::new(-0.00023245626338472066, -7.462392638158351e-05),
    c64::new(-0.00023251343083978656, -7.444561272836979e-05),
    c64::new(-0.0002325704615131917, -7.426725528072005e-05),
    c64::new(-0.00023262735537138638, -7.408885414355748e-05),
    c64::new(-0.00023268411238090142, -7.391040942183086e-05),
    c64::new(-0.0002327407325083481, -7.373192122051466e-05),
    c64::new(-0.00023279721572041825, -7.355338964460896e-05),
    c64::new(-0.00023285356198388433, -7.337481479913908e-05),
    c64::new(-0.00023290977126559924, -7.319619678915613e-05),
    c64::new(-0.00023296584353249647, -7.301753571973646e-05),
    c64::new(-0.0002330217787515902, -7.283883169598179e-05),
    c64::new(-0.00023307757688997518, -7.266008482301908e-05),
    c64::new(-0.00023313323791482676, -7.248129520600061e-05),
    c64::new(-0.00023318876179340104, -7.230246295010352e-05),
    c64::new(-0.00023324414849303473, -7.21235881605303e-05),
    c64::new(-0.0002332993979811453, -7.19446709425084e-05),
    c64::new(-0.00023335451022523081, -7.176571140129019e-05),
    c64::new(-0.00023340948519287022, -7.158670964215307e-05),
    c64::new(-0.00023346432285172315, -7.140766577039895e-05),
    c64::new(-0.00023351902316953003, -7.122857989135484e-05),
    c64::new(-0.00023357358611411207, -7.104945211037235e-05),
    c64::new(-0.0002336280116533713, -7.087028253282772e-05),
    c64::new(-0.00023368229975529058, -7.069107126412182e-05),
    c64::new(-0.00023373645038793356, -7.051181840968012e-05),
    c64::new(-0.00023379046351944488, -7.033252407495222e-05),
    c64::new(-0.00023384433911805003, -7.01531883654125e-05),
    c64::new(-0.0002338980771520553, -6.997381138655953e-05),
    c64::new(-0.00023395167758984806, -6.979439324391615e-05),
    c64::new(-0.0002340051403998965, -6.961493404302954e-05),
    c64::new(-0.00023405846555074988, -6.943543388947071e-05),
    c64::new(-0.00023411165301103835, -6.925589288883511e-05),
    c64::new(-0.00023416470274947312, -6.907631114674205e-05),
    c64::new(-0.00023421761473484638, -6.889668876883485e-05),
    c64::new(-0.00023427038893603138, -6.871702586078073e-05),
    c64::new(-0.0002343230253219824, -6.853732252827087e-05),
    c64::new(-0.00023437552386173483, -6.835757887701988e-05),
    c64::new(-0.00023442788452440513, -6.817779501276643e-05),
    c64::new(-0.00023448010727919087, -6.799797104127273e-05),
    c64::new(-0.00023453219209537075, -6.781810706832462e-05),
    c64::new(-0.00023458413894230458, -6.76382031997315e-05),
    c64::new(-0.0002346359477894334, -6.7458259541326e-05),
    c64::new(-0.00023468761860627945, -6.727827619896443e-05),
    c64::new(-0.00023473915136244605, -6.709825327852637e-05),
    c64::new(-0.0002347905460276178, -6.691819088591464e-05),
    c64::new(-0.0002348418025715606, -6.673808912705533e-05),
    c64::new(-0.00023489292096412148, -6.655794810789777e-05),
    c64::new(-0.00023494390117522886, -6.637776793441405e-05),
    c64::new(-0.00023499474317489237, -6.619754871259964e-05),
    c64::new(-0.00023504544693320298, -6.601729054847287e-05),
    c64::new(-0.00023509601242033298, -6.583699354807496e-05),
    c64::new(-0.00023514643960653598, -6.565665781747012e-05),
    c64::new(-0.00023519672846214696, -6.547628346274497e-05),
    c64::new(-0.00023524687895758233, -6.529587059000923e-05),
    c64::new(-0.0002352968910633398, -6.511541930539514e-05),
    c64::new(-0.00023534676474999856, -6.493492971505752e-05),
    c64::new(-0.0002353964999882192, -6.475440192517376e-05),
    c64::new(-0.0002354460967487437, -6.457383604194382e-05),
    c64::new(-0.00023549555500239567, -6.439323217158974e-05),
    c64::new(-0.00023554487472008003, -6.421259042035623e-05),
    c64::new(-0.00023559405587278328, -6.403191089451017e-05),
    c64::new(-0.00023564309843157342, -6.385119370034069e-05),
    c64::new(-0.00023569200236759992, -6.367043894415914e-05),
    c64::new(-0.00023574076765209393, -6.348964673229868e-05),
    c64::new(-0.00023578939425636806, -6.330881717111484e-05),
    c64::new(-0.00023583788215181655, -6.312795036698497e-05),
    c64::new(-0.00023588623130991516, -6.294704642630835e-05),
    c64::new(-0.0002359344417022214, -6.276610545550612e-05),
    c64::new(-0.00023598251330037427, -6.258512756102127e-05),
    c64::new(-0.00023603044607609455, -6.240411284931826e-05),
    c64::new(-0.0002360782400011846, -6.222306142688347e-05),
    c64::new(-0.00023612589504752845, -6.204197340022477e-05),
    c64::new(-0.00023617341118709192, -6.186084887587161e-05),
    c64::new(-0.00023622078839192245, -6.167968796037496e-05),
    c64::new(-0.00023626802663414927, -6.149849076030693e-05),
    c64::new(-0.00023631512588598335, -6.131725738226125e-05),
    c64::new(-0.00023636208611971737, -6.113598793285287e-05),
    c64::new(-0.00023640890730772588, -6.095468251871791e-05),
    c64::new(-0.00023645558942246518, -6.0773341246513695e-05),
    c64::new(-0.0002365021324364733, -6.0591964222918716e-05),
    c64::new(-0.00023654853632237033, -6.04105515546322e-05),
    c64::new(-0.00023659480105285794, -6.022910334837465e-05),
    c64::new(-0.00023664092660071986, -6.0047619710887366e-05),
    c64::new(-0.00023668691293882155, -5.98661007489325e-05),
    c64::new(-0.00023673276004011043, -5.968454656929309e-05),
    c64::new(-0.00023677846787761593, -5.9502957278772555e-05),
    c64::new(-0.0002368240364244492, -5.932133298419532e-05),
    c64::new(-0.0002368694656538035, -5.913967379240627e-05),
    c64::new(-0.00023691475553895398, -5.895797981027083e-05),
    c64::new(-0.00023695990605325772, -5.8776251144674895e-05),
    c64::new(-0.00023700491717015387, -5.859448790252484e-05),
    c64::new(-0.00023704978886316357, -5.841269019074711e-05),
    c64::new(-0.00023709452110588992, -5.823085811628871e-05),
    c64::new(-0.0002371391138720181, -5.804899178611676e-05),
    c64::new(-0.00023718356713531538, -5.786709130721855e-05),
    c64::new(-0.00023722788086963096, -5.768515678660153e-05),
    c64::new(-0.0002372720550488963, -5.7503188331292904e-05),
    c64::new(-0.00023731608964712485, -5.7321186048340114e-05),
    c64::new(-0.00023735998463841218, -5.713915004481042e-05),
    c64::new(-0.00023740373999693598, -5.695708042779092e-05),
    c64::new(-0.0002374473556969561, -5.677497730438847e-05),
    c64::new(-0.00023749083171281456, -5.6592840781729744e-05),
    c64::new(-0.00023753416801893556, -5.641067096696077e-05),
    c64::new(-0.00023757736458982547, -5.6228467967247456e-05),
    c64::new(-0.00023762042140007281, -5.604623188977513e-05),
    c64::new(-0.0002376633384243484, -5.586396284174858e-05),
    c64::new(-0.00023770611563740524, -5.5681660930392105e-05),
    c64::new(-0.00023774875301407864, -5.5499326262949e-05),
    c64::new(-0.0002377912505292861, -5.531695894668217e-05),
    c64::new(-0.00023783360815802742, -5.51345590888736e-05),
    c64::new(-0.00023787582587538473, -5.495212679682442e-05),
    c64::new(-0.00023791790365652242, -5.476966217785486e-05),
    c64::new(-0.00023795984147668717, -5.4587165339304264e-05),
    c64::new(-0.0002380016393112081, -5.44046363885306e-05),
    c64::new(-0.0002380432971354966, -5.422207543291105e-05),
    c64::new(-0.00023808481492504648, -5.403948257984152e-05),
    c64::new(-0.00023812619265543383, -5.385685793673669e-05),
    c64::new(-0.0002381674303023172, -5.367420161103003e-05),
    c64::new(-0.0002382085278414376, -5.3491513710173345e-05),
    c64::new(-0.00023824948524861842, -5.3308794341637326e-05),
    c64::new(-0.0002382903024997654, -5.3126043612911066e-05),
    c64::new(-0.00023833097957086683, -5.2943261631502105e-05),
    c64::new(-0.00023837151643799344, -5.2760448504936386e-05),
    c64::new(-0.00023841191307729843, -5.257760434075826e-05),
    c64::new(-0.0002384521694650175, -5.2394729246530024e-05),
    c64::new(-0.0002384922855774689, -5.221182332983249e-05),
    c64::new(-0.00023853226139105332, -5.2028886698264506e-05),
    c64::new(-0.00023857209688225397, -5.1845919459442966e-05),
    c64::new(-0.0002386117920276367, -5.16629217210029e-05),
    c64::new(-0.0002386513468038499, -5.1479893590596944e-05),
    c64::new(-0.0002386907611876245, -5.129683517589594e-05),
    c64::new(-0.00023873003515577403, -5.111374658458844e-05),
    c64::new(-0.00023876916868519461, -5.0930627924380734e-05),
    c64::new(-0.000238808161752865, -5.074747930299682e-05),
    c64::new(-0.00023884701433584657, -5.056430082817843e-05),
    c64::new(-0.0002388857264112834, -5.0381092607684505e-05),
    c64::new(-0.00023892429795640212, -5.019785474929184e-05),
    c64::new(-0.00023896272894851213, -5.001458736079455e-05),
    c64::new(-0.00023900101936500542, -4.983129055000411e-05),
    c64::new(-0.00023903916918335672, -4.964796442474942e-05),
    c64::new(-0.00023907717838112354, -4.946460909287631e-05),
    c64::new(-0.000239115046935946, -4.928122466224809e-05),
    c64::new(-0.000239152774825547, -4.909781124074511e-05),
    c64::new(-0.00023919036202773222, -4.891436893626476e-05),
    c64::new(-0.00023922780852039005, -4.873089785672141e-05),
    c64::new(-0.0002392651142814917, -4.854739811004648e-05),
    c64::new(-0.00023930227928909114, -4.8363869804187935e-05),
    c64::new(-0.00023933930352132519, -4.8180313047110853e-05),
    c64::new(-0.00023937618695641342, -4.799672794679694e-05),
    c64::new(-0.00023941292957265823, -4.7813114611244585e-05),
    c64::new(-0.00023944953134844493, -4.762947314846889e-05),
    c64::new(-0.0002394859922622416, -4.7445803666501184e-05),
    c64::new(-0.00023952231229259927, -4.726210627338959e-05),
    c64::new(-0.00023955849141815176, -4.7078381077198564e-05),
    c64::new(-0.00023959452961761586, -4.6894628186008896e-05),
    c64::new(-0.0002396304268697912, -4.671084770791768e-05),
    c64::new(-0.00023966618315356033, -4.652703975103834e-05),
    c64::new(-0.0002397017984478888, -4.6343204423500154e-05),
    c64::new(-0.000239737272731825, -4.615934183344881e-05),
    c64::new(-0.00023977260598450037, -4.597545208904594e-05),
    c64::new(-0.00023980779818512924, -4.5791535298469134e-05),
    c64::new(-0.00023984284931300896, -4.5607591569911904e-05),
    c64::new(-0.00023987775934751983, -4.542362101158371e-05),
    c64::new(-0.0002399125282681252, -4.523962373170949e-05),
    c64::new(-0.0002399471560543714, -4.50555998385302e-05),
    c64::new(-0.0002399816426858878, -4.4871549440302356e-05),
    c64::new(-0.00024001598814238682, -4.468747264529807e-05),
    c64::new(-0.00024005019240366384, -4.450336956180508e-05),
    c64::new(-0.00024008425544959746, -4.4319240298126267e-05),
    c64::new(-0.00024011817726014923, -4.4135084962580255e-05),
    c64::new(-0.00024015195781536383, -4.395090366350087e-05),
    c64::new(-0.00024018559709536904, -4.3766696509237236e-05),
    c64::new(-0.00024021909508037571, -4.358246360815366e-05),
    c64::new(-0.00024025245175067784, -4.339820506862974e-05),
    c64::new(-0.0002402856670866526, -4.3213920999059794e-05),
    c64::new(-0.00024031874106876023, -4.302961150785351e-05),
    c64::new(-0.00024035167367754417, -4.2845276703435407e-05),
    c64::new(-0.000240384464893631, -4.266091669424491e-05),
    c64::new(-0.00024041711469773052, -4.247653158873635e-05),
    c64::new(-0.00024044962307063567, -4.229212149537855e-05),
    c64::new(-0.00024048198999322262, -4.2107686522655295e-05),
    c64::new(-0.00024051421544645071, -4.1923226779064946e-05),
    c64::new(-0.0002405462994113626, -4.173874237312041e-05),
    c64::new(-0.0002405782418690841, -4.155423341334911e-05),
    c64::new(-0.00024061004280082423, -4.136970000829302e-05),
    c64::new(-0.00024064170218787538, -4.118514226650818e-05),
    c64::new(-0.00024067322001161313, -4.100056029656525e-05),
    c64::new(-0.00024070459625349637, -4.0815954207049057e-05),
    c64::new(-0.00024073583089506725, -4.0631324106558606e-05),
    c64::new(-0.0002407669239179512, -4.044667010370713e-05),
    c64::new(-0.00024079787530385707, -4.026199230712161e-05),
    c64::new(-0.00024082868503457692, -4.007729082544335e-05),
    c64::new(-0.00024085935309198618, -3.989256576732746e-05),
    c64::new(-0.00024088987945804363, -3.970781724144294e-05),
    c64::new(-0.0002409202641147914, -3.952304535647259e-05),
    c64::new(-0.0002409505070443549, -3.933825022111305e-05),
    c64::new(-0.0002409806082289431, -3.915343194407432e-05),
    c64::new(-0.00024101056765084823, -3.896859063408036e-05),
    c64::new(-0.0002410403852924459, -3.878372639986852e-05),
    c64::new(-0.00024107006113619517, -3.859883935018968e-05),
    c64::new(-0.00024109959516463848, -3.841392959380824e-05),
    c64::new(-0.00024112898736040178, -3.822899723950162e-05),
    c64::new(-0.00024115823770619438, -3.804404239606088e-05),
    c64::new(-0.00024118734618480903, -3.7859065172290177e-05),
    c64::new(-0.00024121631277912195, -3.7674065677006855e-05),
    c64::new(-0.0002412451374720929, -3.748904401904134e-05),
    c64::new(-0.000241273820246765, -3.730400030723723e-05),
    c64::new(-0.00024130236108626496, -3.7118934650450736e-05),
    c64::new(-0.00024133075997380288, -3.6933847157551344e-05),
    c64::new(-0.00024135901689267247, -3.6748737937421246e-05),
    c64::new(-0.00024138713182625092, -3.656360709895543e-05),
    c64::new(-0.00024141510475799888, -3.637845475106171e-05),
    c64::new(-0.0002414429356714607, -3.619328100266022e-05),
    c64::new(-0.00024147062455026405, -3.600808596268401e-05),
    c64::new(-0.00024149817137812036, -3.5822869740078565e-05),
    c64::new(-0.0002415255761388245, -3.56376324438018e-05),
    c64::new(-0.00024155283881625497, -3.545237418282406e-05),
    c64::new(-0.00024157995939437383, -3.5267095066128106e-05),
    c64::new(-0.00024160693785722675, -3.508179520270865e-05),
    c64::new(-0.000241633774188943, -3.489647470157294e-05),
    c64::new(-0.00024166046837373548, -3.471113367174025e-05),
    c64::new(-0.00024168702039590063, -3.4525772222241934e-05),
    c64::new(-0.0002417134302398186, -3.4340390462121464e-05),
    c64::new(-0.00024173969788995322, -3.415498850043394e-05),
    c64::new(-0.00024176582333085183, -3.3969566446246665e-05),
    c64::new(-0.00024179180654714555, -3.378412440863866e-05),
    c64::new(-0.0002418176475235491, -3.3598662496700686e-05),
    c64::new(-0.00024184334624486088, -3.341318081953522e-05),
    c64::new(-0.00024186890269596305, -3.322767948625646e-05),
    c64::new(-0.00024189431686182138, -3.304215860598984e-05),
    c64::new(-0.00024191958872748535, -3.285661828787263e-05),
    c64::new(-0.00024194471827808823, -3.2671058641053404e-05),
    c64::new(-0.0002419697054988469, -3.248547977469214e-05),
    c64::new(-0.000241994550375062, -3.2299881797960195e-05),
    c64::new(-0.00024201925289211803, -3.211426482003987e-05),
    c64::new(-0.00024204381303548304, -3.192862895012496e-05),
    c64::new(-0.00024206823079070896, -3.1742974297420264e-05),
    c64::new(-0.0002420925061434315, -3.155730097114163e-05),
    c64::new(-0.00024211663907937, -3.137160908051588e-05),
    c64::new(-0.00024214062958432774, -3.11858987347809e-05),
    c64::new(-0.00024216447764419173, -3.1000170043185066e-05),
    c64::new(-0.00024218818324493274, -3.0814423114987905e-05),
    c64::new(-0.00024221174637260542, -3.0628658059459536e-05),
    c64::new(-0.00024223516701334813, -3.0442874985880758e-05),
    c64::new(-0.00024225844515338312, -3.0257074003543086e-05),
    c64::new(-0.00024228158077901652, -3.007125522174822e-05),
    c64::new(-0.00024230457387663818, -2.988541874980868e-05),
    c64::new(-0.00024232742443272188, -2.9699564697047266e-05),
    c64::new(-0.0002423501324338252, -2.9513693172797136e-05),
    c64::new(-0.00024237269786658963, -2.9327804286401716e-05),
    c64::new(-0.0002423951207177405, -2.9141898147214754e-05),
    c64::new(-0.00024241740097408703, -2.8955974864599832e-05),
    c64::new(-0.00024243953862252232, -2.877003454793093e-05),
    c64::new(-0.00024246153365002334, -2.8584077306591947e-05),
    c64::new(-0.000242483386043651, -2.8398103249976736e-05),
    c64::new(-0.00024250509579055012, -2.8212112487489155e-05),
    c64::new(-0.00024252666287794938, -2.802610512854256e-05),
    c64::new(-0.00024254808729316142, -2.784008128256039e-05),
    c64::new(-0.00024256936902358284, -2.7654041058975687e-05),
    c64::new(-0.00024259050805669412, -2.7467984567231122e-05),
    c64::new(-0.00024261150438005972, -2.728191191677893e-05),
    c64::new(-0.00024263235798132804, -2.7095823217080973e-05),
    c64::new(-0.00024265306884823146, -2.6909718577608216e-05),
    c64::new(-0.00024267363696858627, -2.6723598107841332e-05),
    c64::new(-0.00024269406233029281, -2.65374619172702e-05),
    c64::new(-0.00024271434492133538, -2.6351310115393948e-05),
    c64::new(-0.0002427344847297822, -2.6165142811720987e-05),
    c64::new(-0.0002427544817437856, -2.5978960115768533e-05),
    c64::new(-0.0002427743359515818, -2.5792762137063176e-05),
    c64::new(-0.0002427940473414911, -2.560654898514039e-05),
    c64::new(-0.00024281361590191783, -2.5420320769544573e-05),
    c64::new(-0.00024283304162135027, -2.5234077599828987e-05),
    c64::new(-0.00024285232448836077, -2.50478195855558e-05),
    c64::new(-0.00024287146449160574, -2.4861546836295587e-05),
    c64::new(-0.0002428904616198256, -2.4675259461627913e-05),
    c64::new(-0.00024290931586184482, -2.4488957571140843e-05),
    c64::new(-0.00024292802720657193, -2.4302641274430976e-05),
    c64::new(-0.00024294659564299954, -2.41163106811035e-05),
    c64::new(-0.0002429650211602043, -2.392996590077169e-05),
    c64::new(-0.000242983303747347, -2.3743607043057476e-05),
    c64::new(-0.00024300144339367243, -2.3557234217590982e-05),
    c64::new(-0.0002430194400885095, -2.3370847534010538e-05),
    c64::new(-0.00024303729382127123, -2.3184447101962624e-05),
    c64::new(-0.0002430550045814547, -2.2998033031101924e-05),
    c64::new(-0.0002430725723586412, -2.2811605431090816e-05),
    c64::new(-0.00024308999714249598, -2.2625164411599956e-05),
    c64::new(-0.0002431072789227685, -2.2438710082307804e-05),
    c64::new(-0.00024312441768929239, -2.2252242552900626e-05),
    c64::new(-0.00024314141343198528, -2.2065761933072584e-05),
    c64::new(-0.00024315826614084907, -2.1879268332525194e-05),
    c64::new(-0.00024317497580596972, -2.1692761860967957e-05),
    c64::new(-0.00024319154241751737, -2.1506242628117816e-05),
    c64::new(-0.00024320796596574633, -2.1319710743699233e-05),
    c64::new(-0.000243224246440995, -2.1133166317444113e-05),
    c64::new(-0.00024324038383368604, -2.0946609459091835e-05),
    c64::new(-0.00024325637813432622, -2.0760040278388787e-05),
    c64::new(-0.0002432722293335065, -2.057345888508891e-05),
    c64::new(-0.00024328793742190206, -2.038686538895324e-05),
    c64::new(-0.00024330350239027218, -2.0200259899749915e-05),
    c64::new(-0.00024331892422946042, -2.0013642527254244e-05),
    c64::new(-0.00024333420293039455, -1.982701338124821e-05),
    c64::new(-0.00024334933848408643, -1.9640372571521037e-05),
    c64::new(-0.00024336433088163224, -1.94537202078687e-05),
    c64::new(-0.00024337918011421237, -1.926705640009398e-05),
    c64::new(-0.00024339388617309135, -1.9080381258006384e-05),
    c64::new(-0.000243408449049618, -1.88936948914222e-05),
    c64::new(-0.00024342286873522533, -1.8706997410163986e-05),
    c64::new(-0.00024343714522143066, -1.852028892406117e-05),
    c64::new(-0.00024345127849983547, -1.8333569542949548e-05),
    c64::new(-0.00024346526856212556, -1.8146839376671317e-05),
    c64::new(-0.00024347911540007085, -1.7960098535075133e-05),
    c64::new(-0.00024349281900552566, -1.7773347128015603e-05),
    c64::new(-0.00024350637937042854, -1.7586585265353878e-05),
    c64::new(-0.00024351979648680222, -1.739981305695715e-05),
    c64::new(-0.00024353307034675377, -1.7213030612698695e-05),
    c64::new(-0.00024354620094247455, -1.7026238042457818e-05),
    c64::new(-0.00024355918826624013, -1.683943545611988e-05),
    c64::new(-0.0002435720323104104, -1.665262296357581e-05),
    c64::new(-0.0002435847330674296, -1.64658006747227e-05),
    c64::new(-0.00024359729052982613, -1.627896869946328e-05),
    c64::new(-0.0002436097046902128, -1.6092127147705997e-05),
    c64::new(-0.00024362197554128666, -1.5905276129365027e-05),
    c64::new(-0.0002436341030758291, -1.571841575435979e-05),
    c64::new(-0.0002436460872867058, -1.5531546132615542e-05),
    c64::new(-0.00024365792816686676, -1.5344667374062864e-05),
    c64::new(-0.00024366962570934628, -1.5157779588637717e-05),
    c64::new(-0.00024368117990726298, -1.4970882886281371e-05),
    c64::new(-0.00024369259075381988, -1.4783977376940446e-05),
    c64::new(-0.00024370385824230425, -1.4597063170566423e-05),
    c64::new(-0.0002437149823660877, -1.4410140377116223e-05),
    c64::new(-0.0002437259631186262, -1.4223209106551708e-05),
    c64::new(-0.00024373680049346005, -1.4036269468839728e-05),
    c64::new(-0.0002437474944842139, -1.3849321573952165e-05),
    c64::new(-0.0002437580450845968, -1.366236553186543e-05),
    c64::new(-0.00024376845228840204, -1.3475401452561058e-05),
    c64::new(-0.00024377871608950735, -1.3288429446025198e-05),
    c64::new(-0.00024378883648187483, -1.3101449622248662e-05),
    c64::new(-0.00024379881345955087, -1.2914462091226865e-05),
    c64::new(-0.0002438086470166663, -1.2727466962959864e-05),
    c64::new(-0.0002438183371474363, -1.2540464347451854e-05),
    c64::new(-0.00024382788384616039, -1.2353454354711767e-05),
    c64::new(-0.0002438372871072225, -1.2166437094752763e-05),
    c64::new(-0.00024384654692509092, -1.1979412677592279e-05),
    c64::new(-0.00024385566329431836, -1.1792381213252066e-05),
    c64::new(-0.00024386463620954189, -1.1605342811757702e-05),
    c64::new(-0.00024387346566548298, -1.1418297583139166e-05),
    c64::new(-0.00024388215165694748, -1.1231245637430348e-05),
    c64::new(-0.00024389069417882564, -1.104418708466909e-05),
    c64::new(-0.00024389909322609213, -1.0857122034897115e-05),
    c64::new(-0.00024390734879380598, -1.0670050598160084e-05),
    c64::new(-0.00024391546087711068, -1.0482972884507085e-05),
    c64::new(-0.0002439234294712341, -1.0295889003991227e-05),
    c64::new(-0.0002439312545714885, -1.0108799066669136e-05),
    c64::new(-0.0002439389361732706, -9.921703182601e-06),
    c64::new(-0.0002439464742720615, -9.734601461850618e-06),
    c64::new(-0.00024395386886342672, -9.547494014484895e-06),
    c64::new(-0.00024396111994301622, -9.360380950574427e-06),
    c64::new(-0.00024396822750656437, -9.173262380193012e-06),
    c64::new(-0.00024397519154989003, -8.986138413417684e-06),
    c64::new(-0.00024398201206889635, -8.799009160328649e-06),
    c64::new(-0.000243988689059571, -8.611874731009334e-06),
    c64::new(-0.00024399522251798616, -8.424735235545886e-06),
    c64::new(-0.00024400161244029827, -8.23759078402776e-06),
    c64::new(-0.00024400785882274837, -8.050441486547211e-06),
    c64::new(-0.00024401396166166185, -7.863287453199352e-06),
    c64::new(-0.00024401992095344856, -7.676128794082184e-06),
    c64::new(-0.00024402573669460282, -7.48896561929611e-06),
    c64::new(-0.00024403140888170337, -7.30179803894451e-06),
    c64::new(-0.00024403693751141338, -7.1146261631332524e-06),
    c64::new(-0.00024404232258048056, -6.927450101970726e-06),
    c64::new(-0.00024404756408573697, -6.740269965567787e-06),
    c64::new(-0.00024405266202409917, -6.553085864037795e-06),
    c64::new(-0.00024405761639256817, -6.365897907496118e-06),
    c64::new(-0.00024406242718822947, -6.178706206060715e-06),
    c64::new(-0.000244067094408253, -5.991510869851642e-06),
    c64::new(-0.0002440716180498931, -5.804312008991092e-06),
    c64::new(-0.0002440759981104887, -5.61710973360344e-06),
    c64::new(-0.0002440802345874631, -5.429904153814746e-06),
    c64::new(-0.00024408432747832404, -5.242695379753337e-06),
    c64::new(-0.00024408827678066387, -5.05548352154931e-06),
    c64::new(-0.00024409208249215924, -4.8682686893345796e-06),
    c64::new(-0.0002440957446105714, -4.6810509932428075e-06),
    c64::new(-0.00024409926313374595, -4.4938305434094485e-06),
    c64::new(-0.0002441026380596131, -4.3066074499712535e-06),
    c64::new(-0.00024410586938618747, -4.1193818230668535e-06),
    c64::new(-0.0002441089571115681, -3.932153772836261e-06),
    c64::new(-0.0002441119012339386, -3.744923409420913e-06),
    c64::new(-0.000244114701751567, -3.557690842963718e-06),
    c64::new(-0.00024411735866280586, -3.370456183608553e-06),
    c64::new(-0.00024411987196609216, -3.1832195415008535e-06),
    c64::new(-0.0002441222416599474, -2.995981026787111e-06),
    c64::new(-0.00024412446774297754, -2.8087407496149207e-06),
    c64::new(-0.00024412655021387303, -2.6214988201329128e-06),
    c64::new(-0.0002441284890714088, -2.4342553484907992e-06),
    c64::new(-0.0002441302843144443, -2.2470104448388727e-06),
    c64::new(-0.0002441319359419234, -2.0597642193285942e-06),
    c64::new(-0.00024413344395287455, -1.8725167821120942e-06),
    c64::new(-0.00024413480834641055, -1.6852682433422158e-06),
    c64::new(-0.00024413602912172879, -1.4980187131725583e-06),
    c64::new(-0.00024413710627811113, -1.3107683017569788e-06),
    c64::new(-0.00024413803981492393, -1.1235171192501789e-06),
    c64::new(-0.00024413882973161797, -9.362652758072044e-07),
    c64::new(-0.00024413947602772856, -7.490128815834905e-07),
    c64::new(-0.00024413997870287554, -5.617600467347962e-07),
    c64::new(-0.00024414033775676315, -3.745068814172484e-07),
    c64::new(-0.00024414055318918023, -1.872534957868427e-07),
];

static TWIDDLES_8192: [c64; 8192] = [
    c64::new(1.0, 0.0),
    c64::new(0.9999999264657179, 0.00038349518757139556),
    c64::new(0.9999997058628822, 0.0007669903187427045),
    c64::new(0.9999993381915255, 0.0011504853371138485),
    c64::new(0.9999988234517019, 0.0015339801862847655),
    c64::new(0.999998161643487, 0.001917474809855419),
    c64::new(0.9999973527669782, 0.002300969151425805),
    c64::new(0.9999963968222944, 0.0026844631545959617),
    c64::new(0.9999952938095762, 0.003067956762965976),
    c64::new(0.9999940437289858, 0.003451449920135994),
    c64::new(0.9999926465807072, 0.0038349425697062275),
    c64::new(0.9999911023649456, 0.004218434655276963),
    c64::new(0.9999894110819284, 0.0046019261204485705),
    c64::new(0.9999875727319041, 0.0049854169088215106),
    c64::new(0.9999855873151432, 0.0053689069639963425),
    c64::new(0.9999834548319377, 0.005752396229573736),
    c64::new(0.9999811752826011, 0.006135884649154475),
    c64::new(0.9999787486674688, 0.006519372166339468),
    c64::new(0.9999761749868976, 0.006902858724729756),
    c64::new(0.9999734542412659, 0.007286344267926521),
    c64::new(0.9999705864309741, 0.007669828739531097),
    c64::new(0.9999675715564438, 0.008053312083144972),
    c64::new(0.9999644096181183, 0.008436794242369799),
    c64::new(0.9999611006164628, 0.008820275160807411),
    c64::new(0.9999576445519639, 0.00920375478205982),
    c64::new(0.9999540414251298, 0.009587233049729225),
    c64::new(0.9999502912364905, 0.00997070990741803),
    c64::new(0.9999463939865975, 0.010354185298728842),
    c64::new(0.9999423496760239, 0.01073765916726449),
    c64::new(0.9999381583053646, 0.011121131456628021),
    c64::new(0.999933819875236, 0.011504602110422714),
    c64::new(0.9999293343862761, 0.011888071072252092),
    c64::new(0.9999247018391445, 0.012271538285719925),
    c64::new(0.9999199222345228, 0.012655003694430242),
    c64::new(0.9999149955731135, 0.013038467241987334),
    c64::new(0.9999099218556415, 0.013421928871995765),
    c64::new(0.9999047010828529, 0.01380538852806039),
    c64::new(0.9998993332555154, 0.014188846153786345),
    c64::new(0.9998938183744185, 0.014572301692779064),
    c64::new(0.9998881564403733, 0.014955755088644296),
    c64::new(0.9998823474542126, 0.0153392062849881),
    c64::new(0.9998763914167904, 0.015722655225416857),
    c64::new(0.999870288328983, 0.016106101853537287),
    c64::new(0.9998640381916877, 0.016489546112956437),
    c64::new(0.9998576410058239, 0.01687298794728171),
    c64::new(0.9998510967723322, 0.017256427300120877),
    c64::new(0.9998444054921752, 0.017639864115082053),
    c64::new(0.9998375671663371, 0.018023298335773746),
    c64::new(0.9998305817958234, 0.01840672990580482),
    c64::new(0.9998234493816616, 0.018790158768784555),
    c64::new(0.9998161699249004, 0.019173584868322623),
    c64::new(0.9998087434266105, 0.019557008148029083),
    c64::new(0.9998011698878843, 0.01994042855151444),
    c64::new(0.9997934493098353, 0.020323846022389593),
    c64::new(0.9997855816935992, 0.020707260504265895),
    c64::new(0.9997775670403329, 0.02109067194075512),
    c64::new(0.9997694053512153, 0.021474080275469508),
    c64::new(0.9997610966274466, 0.021857485452021735),
    c64::new(0.9997526408702488, 0.02224088741402496),
    c64::new(0.9997440380808654, 0.022624286105092803),
    c64::new(0.9997352882605617, 0.02300768146883937),
    c64::new(0.9997263914106245, 0.02339107344887926),
    c64::new(0.9997173475323622, 0.023774461988827555),
    c64::new(0.9997081566271049, 0.024157847032299864),
    c64::new(0.9996988186962042, 0.024541228522912288),
    c64::new(0.9996893337410336, 0.024924606404281468),
    c64::new(0.9996797017629879, 0.02530798062002457),
    c64::new(0.9996699227634838, 0.025691351113759295),
    c64::new(0.9996599967439592, 0.0260747178291039),
    c64::new(0.9996499237058742, 0.026458080709677187),
    c64::new(0.9996397036507102, 0.02684143969909853),
    c64::new(0.9996293365799701, 0.027224794740987875),
    c64::new(0.9996188224951786, 0.02760814577896574),
    c64::new(0.9996081613978821, 0.027991492756653243),
    c64::new(0.9995973532896484, 0.0283748356176721),
    c64::new(0.9995863981720671, 0.028758174305644615),
    c64::new(0.9995752960467492, 0.029141508764193722),
    c64::new(0.9995640469153277, 0.029524838936942976),
    c64::new(0.999552650779457, 0.029908164767516555),
    c64::new(0.9995411076408129, 0.030291486199539284),
    c64::new(0.9995294175010931, 0.030674803176636626),
    c64::new(0.999517580362017, 0.0310581156424347),
    c64::new(0.9995055962253253, 0.0314414235405603),
    c64::new(0.9994934650927806, 0.03182472681464089),
    c64::new(0.999481186966167, 0.032208025408304586),
    c64::new(0.99946876184729, 0.032591319265180226),
    c64::new(0.9994561897379773, 0.032974608328897335),
    c64::new(0.9994434706400778, 0.03335789254308614),
    c64::new(0.9994306045554617, 0.03374117185137758),
    c64::new(0.9994175914860217, 0.034124446197403326),
    c64::new(0.9994044314336713, 0.03450771552479575),
    c64::new(0.999391124400346, 0.034890979777188004),
    c64::new(0.9993776703880028, 0.03527423889821395),
    c64::new(0.9993640693986205, 0.03565749283150822),
    c64::new(0.9993503214341994, 0.03604074152070623),
    c64::new(0.9993364264967612, 0.03642398490944411),
    c64::new(0.9993223845883495, 0.03680722294135883),
    c64::new(0.9993081957110295, 0.03719045556008812),
    c64::new(0.9992938598668878, 0.037573682709270494),
    c64::new(0.9992793770580327, 0.03795690433254531),
    c64::new(0.9992647472865944, 0.038340120373552694),
    c64::new(0.9992499705547244, 0.03872333077593362),
    c64::new(0.9992350468645959, 0.03910653548332989),
    c64::new(0.9992199762184035, 0.03948973443938412),
    c64::new(0.9992047586183639, 0.03987292758773981),
    c64::new(0.9991893940667149, 0.04025611487204128),
    c64::new(0.9991738825657164, 0.040639296235933736),
    c64::new(0.9991582241176494, 0.04102247162306324),
    c64::new(0.9991424187248169, 0.04140564097707674),
    c64::new(0.9991264663895434, 0.04178880424162206),
    c64::new(0.9991103671141749, 0.04217196136034795),
    c64::new(0.9990941209010791, 0.04255511227690402),
    c64::new(0.9990777277526454, 0.04293825693494082),
    c64::new(0.9990611876712846, 0.043321395278109825),
    c64::new(0.9990445006594293, 0.04370452725006342),
    c64::new(0.9990276667195337, 0.044087652794454944),
    c64::new(0.9990106858540734, 0.04447077185493867),
    c64::new(0.9989935580655457, 0.044853884375169815),
    c64::new(0.9989762833564698, 0.04523699029880459),
    c64::new(0.9989588617293861, 0.045620089569500144),
    c64::new(0.9989412931868569, 0.04600318213091462),
    c64::new(0.9989235777314658, 0.046386267926707164),
    c64::new(0.9989057153658183, 0.04676934690053786),
    c64::new(0.9988877060925413, 0.04715241899606787),
    c64::new(0.9988695499142836, 0.0475354841569593),
    c64::new(0.9988512468337152, 0.04791854232687533),
    c64::new(0.998832796853528, 0.048301593449480144),
    c64::new(0.9988141999764354, 0.04868463746843894),
    c64::new(0.9987954562051724, 0.049067674327418015),
    c64::new(0.9987765655424956, 0.049450703970084664),
    c64::new(0.9987575279911833, 0.04983372634010728),
    c64::new(0.9987383435540352, 0.05021674138115531),
    c64::new(0.9987190122338729, 0.05059974903689928),
    c64::new(0.9986995340335393, 0.0509827492510108),
    c64::new(0.9986799089558991, 0.05136574196716259),
    c64::new(0.9986601370038385, 0.051748727129028456),
    c64::new(0.9986402181802653, 0.052131704680283324),
    c64::new(0.9986201524881089, 0.05251467456460322),
    c64::new(0.9985999399303204, 0.052897636725665324),
    c64::new(0.9985795805098725, 0.053280591107147945),
    c64::new(0.9985590742297593, 0.05366353765273052),
    c64::new(0.9985384210929967, 0.05404647630609366),
    c64::new(0.9985176211026222, 0.05442940701091913),
    c64::new(0.9984966742616946, 0.054812329710889854),
    c64::new(0.9984755805732948, 0.05519524434968994),
    c64::new(0.9984543400405248, 0.05557815087100468),
    c64::new(0.9984329526665084, 0.05596104921852057),
    c64::new(0.9984114184543913, 0.05634393933592529),
    c64::new(0.9983897374073402, 0.05672682116690775),
    c64::new(0.9983679095285438, 0.05710969465515806),
    c64::new(0.9983459348212124, 0.057492559744367566),
    c64::new(0.9983238132885776, 0.05787541637822886),
    c64::new(0.9983015449338929, 0.05825826450043575),
    c64::new(0.9982791297604332, 0.05864110405468334),
    c64::new(0.9982565677714952, 0.05902393498466793),
    c64::new(0.9982338589703968, 0.05940675723408715),
    c64::new(0.9982110033604782, 0.05978957074663987),
    c64::new(0.9981880009451003, 0.06017237546602626),
    c64::new(0.9981648517276462, 0.06055517133594779),
    c64::new(0.9981415557115205, 0.0609379583001072),
    c64::new(0.9981181129001492, 0.06132073630220858),
    c64::new(0.99809452329698, 0.0617035052859573),
    c64::new(0.9980707869054823, 0.06208626519506009),
    c64::new(0.9980469037291468, 0.062469015973224996),
    c64::new(0.9980228737714862, 0.0628517575641614),
    c64::new(0.9979986970360344, 0.06323448991158007),
    c64::new(0.997974373526347, 0.0636172129591931),
    c64::new(0.9979499032460012, 0.06399992665071394),
    c64::new(0.997925286198596, 0.06438263092985747),
    c64::new(0.9979005223877516, 0.06476532574033989),
    c64::new(0.9978756118171102, 0.06514801102587883),
    c64::new(0.9978505544903351, 0.06553068673019333),
    c64::new(0.9978253504111116, 0.0659133527970038),
    c64::new(0.9977999995831465, 0.06629600917003213),
    c64::new(0.9977745020101678, 0.06667865579300156),
    c64::new(0.9977488576959257, 0.06706129260963682),
    c64::new(0.9977230666441916, 0.06744391956366405),
    c64::new(0.9976971288587585, 0.06782653659881087),
    c64::new(0.997671044343441, 0.06820914365880633),
    c64::new(0.9976448131020754, 0.06859174068738094),
    c64::new(0.9976184351385196, 0.06897432762826675),
    c64::new(0.9975919104566526, 0.06935690442519721),
    c64::new(0.9975652390603758, 0.0697394710219073),
    c64::new(0.9975384209536113, 0.07012202736213352),
    c64::new(0.9975114561403035, 0.07050457338961386),
    c64::new(0.9974843446244179, 0.0708871090480878),
    c64::new(0.9974570864099419, 0.0712696342812964),
    c64::new(0.9974296815008842, 0.07165214903298221),
    c64::new(0.9974021299012753, 0.07203465324688933),
    c64::new(0.9973744316151671, 0.07241714686676341),
    c64::new(0.9973465866466332, 0.07279962983635167),
    c64::new(0.9973185949997686, 0.07318210209940289),
    c64::new(0.9972904566786902, 0.07356456359966743),
    c64::new(0.9972621716875362, 0.0739470142808972),
    c64::new(0.9972337400304663, 0.07432945408684576),
    c64::new(0.9972051617116618, 0.07471188296126821),
    c64::new(0.9971764367353262, 0.0750943008479213),
    c64::new(0.9971475651056835, 0.07547670769056339),
    c64::new(0.99711854682698, 0.07585910343295445),
    c64::new(0.9970893819034834, 0.07624148801885607),
    c64::new(0.997060070339483, 0.07662386139203149),
    c64::new(0.9970306121392895, 0.07700622349624564),
    c64::new(0.9970010073072353, 0.07738857427526505),
    c64::new(0.9969712558476743, 0.07777091367285795),
    c64::new(0.9969413577649822, 0.07815324163279423),
    c64::new(0.9969113130635557, 0.07853555809884548),
    c64::new(0.9968811217478138, 0.07891786301478494),
    c64::new(0.9968507838221966, 0.0793001563243876),
    c64::new(0.9968202992911657, 0.07968243797143013),
    c64::new(0.9967896681592046, 0.08006470789969089),
    c64::new(0.996758890430818, 0.08044696605295001),
    c64::new(0.9967279661105325, 0.08082921237498933),
    c64::new(0.9966968952028961, 0.08121144680959244),
    c64::new(0.9966656777124782, 0.08159366930054465),
    c64::new(0.9966343136438699, 0.08197587979163307),
    c64::new(0.9966028030016841, 0.08235807822664654),
    c64::new(0.9965711457905548, 0.08274026454937569),
    c64::new(0.9965393420151379, 0.08312243870361291),
    c64::new(0.9965073916801108, 0.08350460063315243),
    c64::new(0.9964752947901722, 0.08388675028179023),
    c64::new(0.9964430513500426, 0.08426888759332407),
    c64::new(0.9964106613644641, 0.08465101251155362),
    c64::new(0.9963781248382002, 0.08503312498028028),
    c64::new(0.9963454417760359, 0.08541522494330733),
    c64::new(0.996312612182778, 0.0857973123444399),
    c64::new(0.9962796360632546, 0.0861793871274849),
    c64::new(0.9962465134223155, 0.08656144923625117),
    c64::new(0.996213244264832, 0.08694349861454938),
    c64::new(0.996179828595697, 0.08732553520619206),
    c64::new(0.9961462664198246, 0.08770755895499366),
    c64::new(0.9961125577421511, 0.0880895698047705),
    c64::new(0.996078702567634, 0.08847156769934077),
    c64::new(0.996044700901252, 0.0888535525825246),
    c64::new(0.9960105527480059, 0.08923552439814401),
    c64::new(0.9959762581129178, 0.08961748309002296),
    c64::new(0.9959418170010313, 0.08999942860198734),
    c64::new(0.9959072294174117, 0.09038136087786498),
    c64::new(0.9958724953671457, 0.09076327986148562),
    c64::new(0.9958376148553416, 0.091145185496681),
    c64::new(0.9958025878871292, 0.09152707772728483),
    c64::new(0.9957674144676598, 0.09190895649713272),
    c64::new(0.9957320946021064, 0.09229082175006235),
    c64::new(0.9956966282956635, 0.09267267342991331),
    c64::new(0.9956610155535469, 0.09305451148052725),
    c64::new(0.9956252563809943, 0.09343633584574779),
    c64::new(0.9955893507832646, 0.09381814646942055),
    c64::new(0.9955532987656385, 0.0941999432953932),
    c64::new(0.9955171003334181, 0.09458172626751545),
    c64::new(0.9954807554919269, 0.09496349532963899),
    c64::new(0.9954442642465103, 0.09534525042561762),
    c64::new(0.9954076266025349, 0.09572699149930716),
    c64::new(0.995370842565389, 0.09610871849456551),
    c64::new(0.9953339121404823, 0.09649043135525259),
    c64::new(0.9952968353332461, 0.09687213002523047),
    c64::new(0.9952596121491334, 0.09725381444836327),
    c64::new(0.9952222425936184, 0.0976354845685172),
    c64::new(0.9951847266721969, 0.0980171403295606),
    c64::new(0.9951470643903865, 0.09839878167536388),
    c64::new(0.9951092557537261, 0.09878040854979962),
    c64::new(0.9950713007677762, 0.0991620208967425),
    c64::new(0.9950331994381186, 0.09954361866006932),
    c64::new(0.994994951770357, 0.09992520178365907),
    c64::new(0.9949565577701164, 0.10030677021139286),
    c64::new(0.9949180174430432, 0.10068832388715396),
    c64::new(0.9948793307948056, 0.10106986275482782),
    c64::new(0.9948404978310932, 0.10145138675830208),
    c64::new(0.9948015185576171, 0.10183289584146653),
    c64::new(0.9947623929801099, 0.1022143899482132),
    c64::new(0.9947231211043257, 0.10259586902243628),
    c64::new(0.9946837029360402, 0.10297733300803222),
    c64::new(0.9946441384810507, 0.10335878184889961),
    c64::new(0.9946044277451757, 0.10374021548893937),
    c64::new(0.9945645707342554, 0.10412163387205459),
    c64::new(0.9945245674541517, 0.10450303694215057),
    c64::new(0.9944844179107476, 0.10488442464313497),
    c64::new(0.994444122109948, 0.1052657969189176),
    c64::new(0.9944036800576791, 0.10564715371341062),
    c64::new(0.9943630917598886, 0.10602849497052841),
    c64::new(0.9943223572225458, 0.10640982063418768),
    c64::new(0.9942814764516416, 0.10679113064830739),
    c64::new(0.9942404494531879, 0.10717242495680884),
    c64::new(0.9941992762332189, 0.10755370350361562),
    c64::new(0.9941579567977897, 0.10793496623265365),
    c64::new(0.9941164911529771, 0.10831621308785117),
    c64::new(0.9940748793048794, 0.10869744401313872),
    c64::new(0.9940331212596164, 0.10907865895244924),
    c64::new(0.9939912170233294, 0.10945985784971798),
    c64::new(0.9939491666021811, 0.1098410406488826),
    c64::new(0.9939069700023561, 0.11022220729388306),
    c64::new(0.9938646272300597, 0.11060335772866173),
    c64::new(0.9938221382915197, 0.11098449189716339),
    c64::new(0.9937795031929846, 0.11136560974333516),
    c64::new(0.9937367219407246, 0.11174671121112659),
    c64::new(0.9936937945410318, 0.11212779624448964),
    c64::new(0.9936507210002191, 0.11250886478737869),
    c64::new(0.9936075013246216, 0.11288991678375052),
    c64::new(0.9935641355205953, 0.11327095217756435),
    c64::new(0.9935206235945181, 0.11365197091278187),
    c64::new(0.9934769655527892, 0.1140329729333672),
    c64::new(0.9934331614018294, 0.11441395818328692),
    c64::new(0.9933892111480807, 0.11479492660651008),
    c64::new(0.9933451147980069, 0.11517587814700819),
    c64::new(0.9933008723580933, 0.11555681274875526),
    c64::new(0.9932564838348464, 0.11593773035572778),
    c64::new(0.9932119492347945, 0.11631863091190475),
    c64::new(0.9931672685644872, 0.11669951436126769),
    c64::new(0.9931224418304956, 0.11708038064780059),
    c64::new(0.9930774690394123, 0.11746122971548999),
    c64::new(0.9930323501978514, 0.11784206150832498),
    c64::new(0.9929870853124484, 0.11822287597029717),
    c64::new(0.9929416743898605, 0.11860367304540072),
    c64::new(0.992896117436766, 0.11898445267763234),
    c64::new(0.9928504144598651, 0.11936521481099135),
    c64::new(0.9928045654658791, 0.1197459593894796),
    c64::new(0.9927585704615511, 0.12012668635710151),
    c64::new(0.9927124294536455, 0.12050739565786413),
    c64::new(0.992666142448948, 0.12088808723577708),
    c64::new(0.9926197094542661, 0.1212687610348526),
    c64::new(0.9925731304764288, 0.12164941699910553),
    c64::new(0.9925264055222861, 0.12203005507255336),
    c64::new(0.99247953459871, 0.1224106751992162),
    c64::new(0.9924325177125937, 0.12279127732311677),
    c64::new(0.9923853548708517, 0.12317186138828048),
    c64::new(0.9923380460804204, 0.12355242733873537),
    c64::new(0.9922905913482574, 0.12393297511851216),
    c64::new(0.9922429906813417, 0.12431350467164423),
    c64::new(0.9921952440866739, 0.12469401594216764),
    c64::new(0.9921473515712761, 0.12507450887412117),
    c64::new(0.9920993131421918, 0.12545498341154623),
    c64::new(0.9920511288064857, 0.125835439498487),
    c64::new(0.9920027985712445, 0.12621587707899035),
    c64::new(0.991954322443576, 0.12659629609710585),
    c64::new(0.9919057004306093, 0.12697669649688587),
    c64::new(0.9918569325394955, 0.1273570782223854),
    c64::new(0.9918080187774064, 0.1277374412176623),
    c64::new(0.9917589591515361, 0.12811778542677713),
    c64::new(0.9917097536690995, 0.12849811079379317),
    c64::new(0.9916604023373332, 0.12887841726277655),
    c64::new(0.9916109051634954, 0.12925870477779614),
    c64::new(0.9915612621548653, 0.12963897328292356),
    c64::new(0.9915114733187439, 0.13001922272223335),
    c64::new(0.9914615386624538, 0.1303994530398027),
    c64::new(0.9914114581933385, 0.1307796641797117),
    c64::new(0.9913612319187635, 0.13115985608604327),
    c64::new(0.9913108598461154, 0.13154002870288312),
    c64::new(0.9912603419828024, 0.1319201819743198),
    c64::new(0.9912096783362541, 0.13230031584444465),
    c64::new(0.9911588689139214, 0.13268043025735207),
    c64::new(0.9911079137232769, 0.13306052515713906),
    c64::new(0.9910568127718143, 0.13344060048790568),
    c64::new(0.9910055660670494, 0.13382065619375472),
    c64::new(0.9909541736165185, 0.13420069221879202),
    c64::new(0.99090263542778, 0.13458070850712617),
    c64::new(0.9908509515084136, 0.13496070500286875),
    c64::new(0.9907991218660204, 0.13534068165013421),
    c64::new(0.9907471465082227, 0.1357206383930399),
    c64::new(0.9906950254426646, 0.1361005751757062),
    c64::new(0.9906427586770116, 0.13648049194225628),
    c64::new(0.9905903462189501, 0.13686038863681638),
    c64::new(0.9905377880761888, 0.1372402652035156),
    c64::new(0.9904850842564571, 0.13762012158648604),
    c64::new(0.990432234767506, 0.1379999577298628),
    c64::new(0.9903792396171082, 0.1383797735777839),
    c64::new(0.9903260988130573, 0.13875956907439035),
    c64::new(0.9902728123631691, 0.1391393441638262),
    c64::new(0.99021938027528, 0.1395190987902385),
    c64::new(0.9901658025572484, 0.13989883289777721),
    c64::new(0.9901120792169538, 0.14027854643059542),
    c64::new(0.9900582102622971, 0.1406582393328492),
    c64::new(0.9900041957012009, 0.14103791154869771),
    c64::new(0.989950035541609, 0.14141756302230302),
    c64::new(0.9898957297914867, 0.1417971936978304),
    c64::new(0.9898412784588205, 0.14217680351944803),
    c64::new(0.9897866815516186, 0.14255639243132734),
    c64::new(0.9897319390779106, 0.14293596037764267),
    c64::new(0.9896770510457472, 0.1433155073025715),
    c64::new(0.9896220174632009, 0.14369503315029447),
    c64::new(0.9895668383383651, 0.14407453786499516),
    c64::new(0.9895115136793552, 0.14445402139086047),
    c64::new(0.9894560434943077, 0.1448334836720802),
    c64::new(0.9894004277913804, 0.14521292465284746),
    c64::new(0.9893446665787526, 0.14559234427735834),
    c64::new(0.9892887598646252, 0.1459717424898122),
    c64::new(0.98923270765722, 0.14635111923441146),
    c64::new(0.989176509964781, 0.14673047445536175),
    c64::new(0.9891201667955727, 0.14710980809687182),
    c64::new(0.9890636781578815, 0.14748912010315357),
    c64::new(0.9890070440600153, 0.14786841041842222),
    c64::new(0.988950264510303, 0.14824767898689603),
    c64::new(0.9888933395170951, 0.14862692575279654),
    c64::new(0.9888362690887635, 0.14900615066034845),
    c64::new(0.9887790532337015, 0.14938535365377972),
    c64::new(0.9887216919603238, 0.1497645346773215),
    c64::new(0.9886641852770662, 0.1501436936752082),
    c64::new(0.9886065331923864, 0.1505228305916774),
    c64::new(0.9885487357147632, 0.15090194537097004),
    c64::new(0.9884907928526966, 0.15128103795733022),
    c64::new(0.9884327046147083, 0.1516601082950053),
    c64::new(0.9883744710093413, 0.15203915632824605),
    c64::new(0.9883160920451597, 0.15241818200130633),
    c64::new(0.9882575677307495, 0.15279718525844344),
    c64::new(0.9881988980747176, 0.15317616604391784),
    c64::new(0.9881400830856926, 0.15355512430199345),
    c64::new(0.9880811227723241, 0.15393405997693735),
    c64::new(0.9880220171432835, 0.1543129730130201),
    c64::new(0.9879627662072634, 0.15469186335451543),
    c64::new(0.9879033699729778, 0.1550707309457005),
    c64::new(0.9878438284491617, 0.15544957573085585),
    c64::new(0.9877841416445722, 0.15582839765426523),
    c64::new(0.987724309567987, 0.1562071966602159),
    c64::new(0.9876643322282057, 0.15658597269299843),
    c64::new(0.9876042096340492, 0.15696472569690678),
    c64::new(0.9875439417943592, 0.15734345561623825),
    c64::new(0.9874835287179997, 0.15772216239529363),
    c64::new(0.9874229704138554, 0.15810084597837698),
    c64::new(0.9873622668908324, 0.15847950630979596),
    c64::new(0.9873014181578584, 0.15885814333386145),
    c64::new(0.9872404242238823, 0.15923675699488785),
    c64::new(0.9871792850978743, 0.15961534723719306),
    c64::new(0.9871180007888263, 0.15999391400509827),
    c64::new(0.987056571305751, 0.16037245724292828),
    c64::new(0.986994996657683, 0.16075097689501122),
    c64::new(0.9869332768536777, 0.1611294729056788),
    c64::new(0.9868714119028125, 0.16150794521926612),
    c64::new(0.9868094018141855, 0.16188639378011183),
    c64::new(0.9867472465969165, 0.162264818532558),
    c64::new(0.9866849462601467, 0.1626432194209503),
    c64::new(0.9866225008130385, 0.16302159638963784),
    c64::new(0.9865599102647754, 0.16339994938297323),
    c64::new(0.9864971746245629, 0.16377827834531267),
    c64::new(0.9864342939016272, 0.1641565832210158),
    c64::new(0.986371268105216, 0.164534863954446),
    c64::new(0.9863080972445987, 0.16491312048996992),
    c64::new(0.9862447813290655, 0.165291352771958),
    c64::new(0.9861813203679283, 0.16566956074478412),
    c64::new(0.9861177143705201, 0.1660477443528258),
    c64::new(0.9860539633461954, 0.1664259035404641),
    c64::new(0.9859900673043301, 0.16680403825208373),
    c64::new(0.9859260262543211, 0.16718214843207294),
    c64::new(0.985861840205587, 0.16756023402482356),
    c64::new(0.9857975091675675, 0.16793829497473117),
    c64::new(0.9857330331497235, 0.16831633122619483),
    c64::new(0.9856684121615376, 0.16869434272361733),
    c64::new(0.9856036462125134, 0.16907232941140501),
    c64::new(0.9855387353121761, 0.16945029123396796),
    c64::new(0.9854736794700718, 0.16982822813571985),
    c64::new(0.9854084786957684, 0.17020614006107807),
    c64::new(0.9853431329988548, 0.1705840269544636),
    c64::new(0.9852776423889412, 0.17096188876030122),
    c64::new(0.9852120068756594, 0.1713397254230193),
    c64::new(0.9851462264686622, 0.17171753688704997),
    c64::new(0.9850803011776238, 0.172095323096829),
    c64::new(0.9850142310122398, 0.17247308399679595),
    c64::new(0.984948015982227, 0.17285081953139408),
    c64::new(0.9848816560973237, 0.17322852964507032),
    c64::new(0.9848151513672891, 0.1736062142822754),
    c64::new(0.9847485018019042, 0.17398387338746382),
    c64::new(0.9846817074109709, 0.17436150690509375),
    c64::new(0.9846147682043126, 0.1747391147796272),
    c64::new(0.984547684191774, 0.17511669695552992),
    c64::new(0.9844804553832209, 0.17549425337727143),
    c64::new(0.9844130817885407, 0.17587178398932504),
    c64::new(0.9843455634176419, 0.1762492887361679),
    c64::new(0.9842779002804544, 0.17662676756228088),
    c64::new(0.984210092386929, 0.17700422041214875),
    c64::new(0.9841421397470386, 0.17738164723026004),
    c64::new(0.9840740423707764, 0.17775904796110717),
    c64::new(0.9840058002681579, 0.1781364225491863),
    c64::new(0.9839374134492189, 0.1785137709389975),
    c64::new(0.9838688819240172, 0.17889109307504472),
    c64::new(0.9838002057026316, 0.17926838890183575),
    c64::new(0.9837313847951621, 0.17964565836388216),
    c64::new(0.9836624192117303, 0.18002290140569951),
    c64::new(0.9835933089624787, 0.18040011797180724),
    c64::new(0.9835240540575713, 0.1807773080067286),
    c64::new(0.9834546545071933, 0.1811544714549908),
    c64::new(0.9833851103215512, 0.18153160826112497),
    c64::new(0.9833154215108728, 0.18190871836966616),
    c64::new(0.9832455880854071, 0.1822858017251533),
    c64::new(0.9831756100554244, 0.18266285827212927),
    c64::new(0.9831054874312163, 0.18303988795514095),
    c64::new(0.9830352202230956, 0.1834168907187391),
    c64::new(0.9829648084413964, 0.18379386650747845),
    c64::new(0.9828942520964741, 0.18417081526591772),
    c64::new(0.9828235511987052, 0.18454773693861962),
    c64::new(0.9827527057584878, 0.18492463147015079),
    c64::new(0.9826817157862409, 0.1853014988050819),
    c64::new(0.9826105812924048, 0.18567833888798763),
    c64::new(0.9825393022874412, 0.18605515166344663),
    c64::new(0.9824678787818332, 0.1864319370760416),
    c64::new(0.9823963107860847, 0.18680869507035927),
    c64::new(0.9823245983107213, 0.18718542559099033),
    c64::new(0.9822527413662894, 0.1875621285825296),
    c64::new(0.9821807399633571, 0.1879388039895759),
    c64::new(0.9821085941125136, 0.18831545175673212),
    c64::new(0.982036303824369, 0.18869207182860523),
    c64::new(0.9819638691095552, 0.1890686641498062),
    c64::new(0.9818912899787251, 0.18944522866495023),
    c64::new(0.9818185664425525, 0.1898217653186564),
    c64::new(0.981745698511733, 0.19019827405554815),
    c64::new(0.9816726861969831, 0.19057475482025274),
    c64::new(0.9815995295090407, 0.1909512075574018),
    c64::new(0.9815262284586648, 0.1913276322116309),
    c64::new(0.9814527830566355, 0.1917040287275798),
    c64::new(0.9813791933137546, 0.19208039704989244),
    c64::new(0.9813054592408447, 0.19245673712321684),
    c64::new(0.9812315808487497, 0.19283304889220523),
    c64::new(0.9811575581483349, 0.19320933230151396),
    c64::new(0.9810833911504867, 0.1935855872958036),
    c64::new(0.9810090798661126, 0.19396181381973884),
    c64::new(0.9809346243061416, 0.1943380118179886),
    c64::new(0.9808600244815239, 0.19471418123522596),
    c64::new(0.9807852804032304, 0.19509032201612825),
    c64::new(0.980710392082254, 0.19546643410537698),
    c64::new(0.9806353595296081, 0.19584251744765785),
    c64::new(0.9805601827563278, 0.19621857198766088),
    c64::new(0.9804848617734694, 0.19659459767008022),
    c64::new(0.9804093965921099, 0.19697059443961434),
    c64::new(0.980333787223348, 0.19734656224096592),
    c64::new(0.9802580336783036, 0.19772250101884192),
    c64::new(0.9801821359681174, 0.19809841071795356),
    c64::new(0.9801060941039518, 0.19847429128301639),
    c64::new(0.9800299080969901, 0.1988501426587501),
    c64::new(0.9799535779584367, 0.19922596478987883),
    c64::new(0.9798771036995176, 0.19960175762113097),
    c64::new(0.9798004853314798, 0.19997752109723915),
    c64::new(0.9797237228655912, 0.20035325516294045),
    c64::new(0.9796468163131412, 0.20072895976297614),
    c64::new(0.9795697656854405, 0.2011046348420919),
    c64::new(0.9794925709938208, 0.20148028034503773),
    c64::new(0.9794152322496348, 0.20185589621656805),
    c64::new(0.9793377494642568, 0.20223148240144145),
    c64::new(0.979260122649082, 0.20260703884442113),
    c64::new(0.9791823518155269, 0.20298256549027444),
    c64::new(0.9791044369750292, 0.20335806228377332),
    c64::new(0.9790263781390476, 0.20373352916969392),
    c64::new(0.9789481753190622, 0.20410896609281687),
    c64::new(0.9788698285265741, 0.20448437299792724),
    c64::new(0.9787913377731057, 0.20485974982981442),
    c64::new(0.9787127030702004, 0.20523509653327235),
    c64::new(0.9786339244294232, 0.20561041305309924),
    c64::new(0.9785550018623596, 0.2059856993340979),
    c64::new(0.9784759353806168, 0.2063609553210755),
    c64::new(0.9783967249958231, 0.2067361809588437),
    c64::new(0.9783173707196277, 0.20711137619221856),
    c64::new(0.9782378725637011, 0.20748654096602065),
    c64::new(0.978158230539735, 0.20786167522507507),
    c64::new(0.9780784446594424, 0.20823677891421133),
    c64::new(0.9779985149345571, 0.20861185197826349),
    c64::new(0.9779184413768344, 0.20898689436207007),
    c64::new(0.9778382239980504, 0.20936190601047416),
    c64::new(0.9777578628100028, 0.20973688686832329),
    c64::new(0.9776773578245099, 0.2101118368804696),
    c64::new(0.9775967090534119, 0.21048675599176972),
    c64::new(0.9775159165085693, 0.21086164414708486),
    c64::new(0.9774349802018643, 0.21123650129128071),
    c64::new(0.9773539001452001, 0.21161132736922755),
    c64::new(0.9772726763505009, 0.21198612232580033),
    c64::new(0.9771913088297123, 0.21236088610587842),
    c64::new(0.9771097975948009, 0.21273561865434593),
    c64::new(0.9770281426577544, 0.21311031991609136),
    c64::new(0.9769463440305817, 0.21348498983600805),
    c64::new(0.9768644017253126, 0.21385962835899375),
    c64::new(0.9767823157539987, 0.214234235429951),
    c64::new(0.9767000861287118, 0.21460881099378676),
    c64::new(0.9766177128615456, 0.21498335499541282),
    c64::new(0.9765351959646145, 0.21535786737974555),
    c64::new(0.9764525354500541, 0.21573234809170588),
    c64::new(0.9763697313300211, 0.21610679707621952),
    c64::new(0.9762867836166936, 0.21648121427821676),
    c64::new(0.9762036923222706, 0.21685559964263262),
    c64::new(0.9761204574589719, 0.2172299531144068),
    c64::new(0.976037079039039, 0.21760427463848364),
    c64::new(0.9759535570747343, 0.2179785641598122),
    c64::new(0.975869891578341, 0.21835282162334632),
    c64::new(0.9757860825621639, 0.21872704697404444),
    c64::new(0.9757021300385286, 0.2191012401568698),
    c64::new(0.9756180340197818, 0.21947540111679031),
    c64::new(0.9755337945182914, 0.2198495297987787),
    c64::new(0.9754494115464464, 0.22022362614781238),
    c64::new(0.975364885116657, 0.2205976901088735),
    c64::new(0.9752802152413542, 0.2209717216269491),
    c64::new(0.9751954019329904, 0.2213457206470308),
    c64::new(0.9751104452040389, 0.22171968711411522),
    c64::new(0.9750253450669941, 0.2220936209732035),
    c64::new(0.9749401015343718, 0.22246752216930188),
    c64::new(0.9748547146187084, 0.22284139064742112),
    c64::new(0.9747691843325618, 0.22321522635257698),
    c64::new(0.9746835106885107, 0.22358902922979),
    c64::new(0.974597693699155, 0.22396279922408546),
    c64::new(0.9745117333771157, 0.2243365362804936),
    c64::new(0.974425629735035, 0.22471024034404943),
    c64::new(0.9743393827855759, 0.22508391135979283),
    c64::new(0.9742529925414225, 0.22545754927276854),
    c64::new(0.9741664590152803, 0.22583115402802617),
    c64::new(0.9740797822198757, 0.22620472557062019),
    c64::new(0.9739929621679558, 0.22657826384561),
    c64::new(0.9739059988722896, 0.2269517687980598),
    c64::new(0.9738188923456661, 0.22732524037303886),
    c64::new(0.9737316426008964, 0.22769867851562117),
    c64::new(0.973644249650812, 0.22807208317088573),
    c64::new(0.9735567135082656, 0.22844545428391647),
    c64::new(0.9734690341861311, 0.22881879179980222),
    c64::new(0.9733812116973033, 0.2291920956636368),
    c64::new(0.9732932460546982, 0.22956536582051887),
    c64::new(0.9732051372712528, 0.2299386022155522),
    c64::new(0.9731168853599251, 0.23031180479384544),
    c64::new(0.9730284903336942, 0.2306849735005122),
    c64::new(0.9729399522055602, 0.2310581082806711),
    c64::new(0.9728512709885442, 0.23143120907944575),
    c64::new(0.9727624466956886, 0.23180427584196478),
    c64::new(0.9726734793400564, 0.2321773085133617),
    c64::new(0.9725843689347322, 0.23255030703877524),
    c64::new(0.9724951154928212, 0.23292327136334898),
    c64::new(0.9724057190274498, 0.23329620143223162),
    c64::new(0.9723161795517653, 0.23366909719057682),
    c64::new(0.9722264970789363, 0.23404195858354343),
    c64::new(0.9721366716221522, 0.23441478555629516),
    c64::new(0.9720467031946235, 0.23478757805400097),
    c64::new(0.9719565918095817, 0.23516033602183473),
    c64::new(0.9718663374802794, 0.2355330594049755),
    c64::new(0.9717759402199901, 0.23590574814860737),
    c64::new(0.9716854000420085, 0.23627840219791957),
    c64::new(0.9715947169596502, 0.23665102149810638),
    c64::new(0.9715038909862518, 0.2370236059943672),
    c64::new(0.9714129221351709, 0.2373961556319066),
    c64::new(0.9713218104197862, 0.2377686703559342),
    c64::new(0.9712305558534974, 0.23814115011166484),
    c64::new(0.9711391584497251, 0.23851359484431842),
    c64::new(0.9710476182219111, 0.23888600449912004),
    c64::new(0.970955935183518, 0.23925837902129998),
    c64::new(0.9708641093480295, 0.23963071835609356),
    c64::new(0.9707721407289504, 0.2400030224487415),
    c64::new(0.9706800293398061, 0.24037529124448945),
    c64::new(0.9705877751941436, 0.24074752468858843),
    c64::new(0.9704953783055306, 0.2411197227262946),
    c64::new(0.9704028386875555, 0.24149188530286933),
    c64::new(0.9703101563538281, 0.24186401236357918),
    c64::new(0.9702173313179792, 0.242236103853696),
    c64::new(0.9701243635936603, 0.2426081597184968),
    c64::new(0.970031253194544, 0.24298017990326387),
    c64::new(0.969938000134324, 0.24335216435328474),
    c64::new(0.9698446044267148, 0.24372411301385216),
    c64::new(0.9697510660854521, 0.2440960258302642),
    c64::new(0.9696573851242924, 0.24446790274782415),
    c64::new(0.9695635615570132, 0.24483974371184067),
    c64::new(0.9694695953974131, 0.24521154866762754),
    c64::new(0.9693754866593113, 0.24558331756050406),
    c64::new(0.9692812353565485, 0.2459550503357946),
    c64::new(0.969186841502986, 0.24632674693882903),
    c64::new(0.9690923051125062, 0.24669840731494241),
    c64::new(0.9689976261990124, 0.24707003140947525),
    c64::new(0.9689028047764289, 0.24744161916777327),
    c64::new(0.968807840858701, 0.24781317053518767),
    c64::new(0.9687127344597948, 0.24818468545707478),
    c64::new(0.9686174855936975, 0.24855616387879656),
    c64::new(0.9685220942744173, 0.24892760574572015),
    c64::new(0.9684265605159832, 0.2492990110032182),
    c64::new(0.9683308843324452, 0.24967037959666855),
    c64::new(0.9682350657378743, 0.25004171147145465),
    c64::new(0.9681391047463624, 0.2504130065729652),
    c64::new(0.9680430013720223, 0.2507842648465945),
    c64::new(0.9679467556289878, 0.2511554862377419),
    c64::new(0.9678503675314136, 0.2515266706918126),
    c64::new(0.9677538370934755, 0.25189781815421697),
    c64::new(0.9676571643293699, 0.2522689285703708),
    c64::new(0.9675603492533145, 0.2526400018856955),
    c64::new(0.9674633918795476, 0.25301103804561786),
    c64::new(0.9673662922223285, 0.25338203699557016),
    c64::new(0.9672690502959378, 0.25375299868099),
    c64::new(0.9671716661146766, 0.2541239230473206),
    c64::new(0.967074139692867, 0.25449481004001073),
    c64::new(0.9669764710448521, 0.25486565960451457),
    c64::new(0.9668786601849959, 0.2552364716862917),
    c64::new(0.9667807071276833, 0.2556072462308074),
    c64::new(0.9666826118873202, 0.25597798318353243),
    c64::new(0.9665843744783331, 0.2563486824899429),
    c64::new(0.9664859949151698, 0.25671934409552066),
    c64::new(0.9663874732122989, 0.2570899679457531),
    c64::new(0.9662888093842097, 0.2574605539861331),
    c64::new(0.9661900034454125, 0.257831102162159),
    c64::new(0.9660910554104388, 0.25820161241933487),
    c64::new(0.9659919652938406, 0.25857208470317034),
    c64::new(0.9658927331101909, 0.2589425189591805),
    c64::new(0.9657933588740837, 0.25931291513288623),
    c64::new(0.9656938426001337, 0.25968327316981377),
    c64::new(0.9655941843029768, 0.2600535930154952),
    c64::new(0.9654943839972695, 0.260423874615468),
    c64::new(0.9653944416976894, 0.2607941179152755),
    c64::new(0.9652943574189347, 0.2611643228604665),
    c64::new(0.9651941311757247, 0.2615344893965955),
    c64::new(0.9650937629827996, 0.2619046174692226),
    c64::new(0.9649932528549203, 0.2622747070239136),
    c64::new(0.9648926008068689, 0.26264475800624004),
    c64::new(0.9647918068534479, 0.263014770361779),
    c64::new(0.964690871009481, 0.2633847440361133),
    c64::new(0.9645897932898128, 0.26375467897483135),
    c64::new(0.9644885737093084, 0.26412457512352755),
    c64::new(0.9643872122828543, 0.26449443242780163),
    c64::new(0.9642857090253575, 0.26486425083325926),
    c64::new(0.9641840639517458, 0.2652340302855118),
    c64::new(0.9640822770769681, 0.2656037707301763),
    c64::new(0.9639803484159941, 0.2659734721128756),
    c64::new(0.9638782779838142, 0.2663431343792382),
    c64::new(0.9637760657954398, 0.26671275747489837),
    c64::new(0.9636737118659032, 0.26708234134549624),
    c64::new(0.9635712162102573, 0.2674518859366776),
    c64::new(0.963468578843576, 0.26782139119409415),
    c64::new(0.963365799780954, 0.2681908570634032),
    c64::new(0.9632628790375071, 0.2685602834902679),
    c64::new(0.9631598166283714, 0.26892967042035726),
    c64::new(0.9630566125687043, 0.2692990177993461),
    c64::new(0.9629532668736839, 0.2696683255729151),
    c64::new(0.962849779558509, 0.27003759368675057),
    c64::new(0.9627461506383994, 0.2704068220865448),
    c64::new(0.9626423801285957, 0.270776010717996),
    c64::new(0.9625384680443592, 0.271145159526808),
    c64::new(0.9624344144009721, 0.2715142684586907),
    c64::new(0.9623302192137374, 0.2718833374593597),
    c64::new(0.962225882497979, 0.2722523664745367),
    c64::new(0.9621214042690416, 0.272621355449949),
    c64::new(0.9620167845422906, 0.2729903043313299),
    c64::new(0.9619120233331122, 0.2733592130644187),
    c64::new(0.9618071206569135, 0.27372808159496054),
    c64::new(0.9617020765291225, 0.2740969098687064),
    c64::new(0.9615968909651879, 0.2744656978314132),
    c64::new(0.961491563980579, 0.27483444542884394),
    c64::new(0.9613860955907862, 0.2752031526067673),
    c64::new(0.9612804858113206, 0.27557181931095814),
    c64::new(0.9611747346577142, 0.27594044548719715),
    c64::new(0.9610688421455194, 0.2763090310812711),
    c64::new(0.9609628082903098, 0.2766775760389724),
    c64::new(0.9608566331076797, 0.2770460803060999),
    c64::new(0.960750316613244, 0.2774145438284581),
    c64::new(0.9606438588226386, 0.2777829665518577),
    c64::new(0.96053725975152, 0.2781513484221151),
    c64::new(0.9604305194155658, 0.27851968938505306),
    c64::new(0.9603236378304739, 0.2788879893865003),
    c64::new(0.9602166150119634, 0.2792562483722912),
    c64::new(0.9601094509757739, 0.2796244662882666),
    c64::new(0.960002145737666, 0.2799926430802732),
    c64::new(0.9598946993134205, 0.2803607786941638),
    c64::new(0.9597871117188399, 0.2807288730757972),
    c64::new(0.9596793829697468, 0.28109692617103826),
    c64::new(0.9595715130819845, 0.28146493792575794),
    c64::new(0.9594635020714175, 0.28183290828583335),
    c64::new(0.9593553499539308, 0.28220083719714756),
    c64::new(0.9592470567454301, 0.28256872460558974),
    c64::new(0.9591386224618419, 0.2829365704570554),
    c64::new(0.9590300471191137, 0.28330437469744574),
    c64::new(0.9589213307332132, 0.28367213727266843),
    c64::new(0.9588124733201293, 0.2840398581286372),
    c64::new(0.9587034748958716, 0.2844075372112719),
    c64::new(0.9585943354764702, 0.2847751744664983),
    c64::new(0.9584850550779762, 0.28514276984024867),
    c64::new(0.9583756337164612, 0.28551032327846126),
    c64::new(0.9582660714080177, 0.2858778347270806),
    c64::new(0.9581563681687588, 0.2862453041320571),
    c64::new(0.9580465240148186, 0.2866127314393478),
    c64::new(0.9579365389623514, 0.28698011659491557),
    c64::new(0.9578264130275329, 0.2873474595447295),
    c64::new(0.9577161462265589, 0.28771476023476517),
    c64::new(0.9576057385756463, 0.28808201861100413),
    c64::new(0.9574951900910326, 0.2884492346194342),
    c64::new(0.9573845007889759, 0.2888164082060495),
    c64::new(0.9572736706857552, 0.2891835393168502),
    c64::new(0.9571626997976701, 0.289550627897843),
    c64::new(0.957051588141041, 0.28991767389504075),
    c64::new(0.9569403357322088, 0.29028467725446233),
    c64::new(0.9568289425875354, 0.2906516379221332),
    c64::new(0.956717408723403, 0.2910185558440851),
    c64::new(0.9566057341562151, 0.29138543096635566),
    c64::new(0.9564939189023951, 0.29175226323498926),
    c64::new(0.9563819629783877, 0.2921190525960364),
    c64::new(0.956269866400658, 0.2924857989955539),
    c64::new(0.9561576291856921, 0.2928525023796048),
    c64::new(0.9560452513499964, 0.29321916269425863),
    c64::new(0.9559327329100983, 0.2935857798855912),
    c64::new(0.9558200738825454, 0.29395235389968466),
    c64::new(0.9557072742839066, 0.2943188846826274),
    c64::new(0.9555943341307711, 0.2946853721805143),
    c64::new(0.9554812534397488, 0.2950518163394467),
    c64::new(0.9553680322274704, 0.295418217105532),
    c64::new(0.955254670510587, 0.29578457442488426),
    c64::new(0.9551411683057707, 0.2961508882436238),
    c64::new(0.9550275256297142, 0.29651715850787747),
    c64::new(0.9549137424991305, 0.29688338516377827),
    c64::new(0.9547998189307537, 0.29724956815746584),
    c64::new(0.9546857549413383, 0.2976157074350862),
    c64::new(0.9545715505476597, 0.2979818029427918),
    c64::new(0.9544572057665135, 0.2983478546267414),
    c64::new(0.9543427206147165, 0.29871386243310033),
    c64::new(0.9542280951091057, 0.2990798263080405),
    c64::new(0.9541133292665388, 0.2994457461977399),
    c64::new(0.9539984231038945, 0.29981162204838335),
    c64::new(0.9538833766380718, 0.30017745380616195),
    c64::new(0.9537681898859903, 0.30054324141727345),
    c64::new(0.9536528628645905, 0.3009089848279219),
    c64::new(0.9535373955908333, 0.30127468398431795),
    c64::new(0.9534217880817004, 0.30164033883267877),
    c64::new(0.9533060403541939, 0.3020059493192281),
    c64::new(0.9531901524253367, 0.30237151539019597),
    c64::new(0.9530741243121722, 0.30273703699181914),
    c64::new(0.9529579560317647, 0.30310251407034106),
    c64::new(0.9528416476011987, 0.3034679465720113),
    c64::new(0.9527251990375796, 0.30383333444308636),
    c64::new(0.9526086103580333, 0.3041986776298291),
    c64::new(0.9524918815797063, 0.3045639760785091),
    c64::new(0.9523750127197659, 0.3049292297354024),
    c64::new(0.9522580037953996, 0.30529443854679167),
    c64::new(0.9521408548238158, 0.3056596024589661),
    c64::new(0.9520235658222436, 0.3060247214182218),
    c64::new(0.9519061368079323, 0.3063897953708609),
    c64::new(0.9517885677981521, 0.3067548242631928),
    c64::new(0.9516708588101939, 0.3071198080415331),
    c64::new(0.9515530098613686, 0.3074847466522041),
    c64::new(0.9514350209690083, 0.30784964004153487),
    c64::new(0.9513168921504656, 0.30821448815586105),
    c64::new(0.9511986234231132, 0.30857929094152503),
    c64::new(0.951080214804345, 0.3089440483448757),
    c64::new(0.9509616663115751, 0.3093087603122687),
    c64::new(0.9508429779622382, 0.3096734267900664),
    c64::new(0.9507241497737896, 0.3100380477246379),
    c64::new(0.9506051817637053, 0.3104026230623587),
    c64::new(0.9504860739494817, 0.3107671527496115),
    c64::new(0.9503668263486358, 0.31113163673278527),
    c64::new(0.9502474389787052, 0.3114960749582759),
    c64::new(0.9501279118572481, 0.311860467372486),
    c64::new(0.950008245001843, 0.3122248139218249),
    c64::new(0.9498884384300893, 0.3125891145527087),
    c64::new(0.9497684921596067, 0.3129533692115602),
    c64::new(0.9496484062080355, 0.313317577844809),
    c64::new(0.9495281805930367, 0.3136817403988915),
    c64::new(0.9494078153322916, 0.3140458568202507),
    c64::new(0.9492873104435021, 0.31440992705533666),
    c64::new(0.9491666659443907, 0.31477395105060607),
    c64::new(0.9490458818527006, 0.31513792875252244),
    c64::new(0.9489249581861952, 0.315501860107556),
    c64::new(0.9488038949626585, 0.31586574506218396),
    c64::new(0.9486826921998951, 0.3162295835628903),
    c64::new(0.9485613499157303, 0.31659337555616585),
    c64::new(0.9484398681280096, 0.31695712098850815),
    c64::new(0.9483182468545992, 0.31732081980642174),
    c64::new(0.9481964861133856, 0.31768447195641797),
    c64::new(0.9480745859222762, 0.31804807738501495),
    c64::new(0.9479525462991987, 0.3184116360387378),
    c64::new(0.947830367262101, 0.3187751478641185),
    c64::new(0.9477080488289521, 0.3191386128076959),
    c64::new(0.9475855910177411, 0.3195020308160157),
    c64::new(0.9474629938464777, 0.3198654018356305),
    c64::new(0.947340257333192, 0.32022872581309986),
    c64::new(0.9472173814959348, 0.32059200269499033),
    c64::new(0.9470943663527772, 0.3209552324278752),
    c64::new(0.9469712119218109, 0.32131841495833485),
    c64::new(0.946847918221148, 0.3216815502329566),
    c64::new(0.9467244852689212, 0.3220446381983345),
    c64::new(0.9466009130832835, 0.32240767880106985),
    c64::new(0.9464772016824087, 0.3227706719877707),
    c64::new(0.9463533510844906, 0.32313361770505233),
    c64::new(0.9462293613077438, 0.3234965158995367),
    c64::new(0.9461052323704034, 0.32385936651785285),
    c64::new(0.9459809642907248, 0.32422216950663696),
    c64::new(0.9458565570869839, 0.32458492481253215),
    c64::new(0.9457320107774772, 0.32494763238218843),
    c64::new(0.9456073253805213, 0.3253102921622629),
    c64::new(0.9454825009144537, 0.32567290409941985),
    c64::new(0.9453575373976323, 0.32603546814033024),
    c64::new(0.945232434848435, 0.3263979842316725),
    c64::new(0.9451071932852606, 0.32676045232013173),
    c64::new(0.9449818127265281, 0.3271228723524005),
    c64::new(0.9448562931906772, 0.327485244275178),
    c64::new(0.9447306346961678, 0.32784756803517084),
    c64::new(0.9446048372614803, 0.3282098435790925),
    c64::new(0.9444789009051155, 0.32857207085366374),
    c64::new(0.9443528256455948, 0.3289342498056122),
    c64::new(0.9442266115014598, 0.32929638038167275),
    c64::new(0.9441002584912727, 0.3296584625285875),
    c64::new(0.943973766633616, 0.3300204961931054),
    c64::new(0.9438471359470927, 0.3303824813219828),
    c64::new(0.9437203664503262, 0.3307444178619829),
    c64::new(0.9435934581619604, 0.33110630575987643),
    c64::new(0.9434664111006593, 0.33146814496244087),
    c64::new(0.9433392252851077, 0.3318299354164611),
    c64::new(0.9432119007340106, 0.33219167706872915),
    c64::new(0.9430844374660935, 0.3325533698660442),
    c64::new(0.9429568355001021, 0.33291501375521265),
    c64::new(0.9428290948548027, 0.3332766086830479),
    c64::new(0.942701215548982, 0.33363815459637086),
    c64::new(0.9425731976014469, 0.3339996514420094),
    c64::new(0.9424450410310249, 0.33436109916679874),
    c64::new(0.9423167458565638, 0.3347224977175812),
    c64::new(0.9421883120969318, 0.3350838470412066),
    c64::new(0.9420597397710173, 0.3354451470845316),
    c64::new(0.9419310288977296, 0.33580639779442045),
    c64::new(0.9418021794959976, 0.3361675991177445),
    c64::new(0.9416731915847714, 0.3365287510013824),
    c64::new(0.9415440651830208, 0.33688985339222005),
    c64::new(0.9414148003097363, 0.3372509062371506),
    c64::new(0.9412853969839287, 0.3376119094830746),
    c64::new(0.9411558552246292, 0.3379728630768997),
    c64::new(0.9410261750508893, 0.3383337669655411),
    c64::new(0.9408963564817808, 0.3386946210959212),
    c64::new(0.9407663995363961, 0.33905542541496964),
    c64::new(0.9406363042338476, 0.33941617986962336),
    c64::new(0.9405060705932683, 0.33977688440682685),
    c64::new(0.9403756986338115, 0.3401375389735317),
    c64::new(0.9402451883746509, 0.34049814351669716),
    c64::new(0.9401145398349803, 0.34085869798328944),
    c64::new(0.939983753034014, 0.34121920232028236),
    c64::new(0.9398528279909867, 0.34157965647465716),
    c64::new(0.9397217647251533, 0.3419400603934022),
    c64::new(0.9395905632557893, 0.3423004140235135),
    c64::new(0.9394592236021899, 0.3426607173119944),
    c64::new(0.9393277457836714, 0.34302097020585554),
    c64::new(0.9391961298195699, 0.34338117265211504),
    c64::new(0.939064375729242, 0.3437413245977985),
    c64::new(0.9389324835320645, 0.3441014259899388),
    c64::new(0.9388004532474348, 0.34446147677557654),
    c64::new(0.9386682848947702, 0.3448214769017593),
    c64::new(0.9385359784935086, 0.34518142631554255),
    c64::new(0.9384035340631081, 0.3455413249639891),
    c64::new(0.9382709516230472, 0.345901172794169),
    c64::new(0.9381382311928244, 0.34626096975316),
    c64::new(0.9380053727919588, 0.3466207157880473),
    c64::new(0.9378723764399899, 0.3469804108459237),
    c64::new(0.937739242156477, 0.34734005487388914),
    c64::new(0.937605969961, 0.3476996478190514),
    c64::new(0.9374725598731593, 0.3480591896285256),
    c64::new(0.937339011912575, 0.34841868024943456),
    c64::new(0.937205326098888, 0.3487781196289084),
    c64::new(0.9370715024517592, 0.34913750771408497),
    c64::new(0.9369375409908699, 0.34949684445210955),
    c64::new(0.9368034417359216, 0.3498561297901349),
    c64::new(0.9366692047066362, 0.3502153636753216),
    c64::new(0.9365348299227555, 0.3505745460548375),
    c64::new(0.9364003174040421, 0.35093367687585836),
    c64::new(0.9362656671702783, 0.3512927560855671),
    c64::new(0.936130879241267, 0.35165178363115457),
    c64::new(0.9359959536368314, 0.3520107594598191),
    c64::new(0.9358608903768146, 0.35236968351876663),
    c64::new(0.9357256894810804, 0.3527285557552107),
    c64::new(0.9355903509695124, 0.3530873761163725),
    c64::new(0.9354548748620146, 0.3534461445494808),
    c64::new(0.9353192611785116, 0.35380486100177205),
    c64::new(0.9351835099389476, 0.35416352542049034),
    c64::new(0.9350476211632874, 0.35452213775288743),
    c64::new(0.9349115948715161, 0.3548806979462228),
    c64::new(0.9347754310836387, 0.3552392059477633),
    c64::new(0.9346391298196808, 0.35559766170478385),
    c64::new(0.9345026910996879, 0.35595606516456685),
    c64::new(0.9343661149437259, 0.3563144162744024),
    c64::new(0.9342294013718808, 0.35667271498158826),
    c64::new(0.934092550404259, 0.35703096123343),
    c64::new(0.9339555620609867, 0.35738915497724094),
    c64::new(0.933818436362211, 0.3577472961603419),
    c64::new(0.9336811733280984, 0.3581053847300616),
    c64::new(0.9335437729788362, 0.35846342063373654),
    c64::new(0.9334062353346315, 0.35882140381871086),
    c64::new(0.933268560415712, 0.3591793342323365),
    c64::new(0.9331307482423252, 0.35953721182197307),
    c64::new(0.932992798834739, 0.3598950365349881),
    c64::new(0.9328547122132411, 0.3602528083187569),
    c64::new(0.9327164883981403, 0.36061052712066227),
    c64::new(0.9325781274097644, 0.36096819288809523),
    c64::new(0.9324396292684624, 0.3613258055684543),
    c64::new(0.9323009939946028, 0.36168336510914584),
    c64::new(0.9321622216085744, 0.3620408714575842),
    c64::new(0.9320233121307865, 0.3623983245611913),
    c64::new(0.9318842655816681, 0.3627557243673972),
    c64::new(0.9317450819816687, 0.36311307082363947),
    c64::new(0.9316057613512578, 0.36347036387736376),
    c64::new(0.9314663037109251, 0.3638276034760235),
    c64::new(0.9313267090811804, 0.3641847895670799),
    c64::new(0.9311869774825539, 0.3645419220980021),
    c64::new(0.9310471089355953, 0.36489900101626727),
    c64::new(0.9309071034608751, 0.36525602626936027),
    c64::new(0.9307669610789837, 0.36561299780477385),
    c64::new(0.9306266818105318, 0.36596991557000874),
    c64::new(0.9304862656761497, 0.3663267795125736),
    c64::new(0.9303457126964885, 0.3666835895799849),
    c64::new(0.9302050228922191, 0.3670403457197672),
    c64::new(0.9300641962840324, 0.3673970478794527),
    c64::new(0.9299232328926397, 0.367753696006582),
    c64::new(0.9297821327387721, 0.36811029004870305),
    c64::new(0.9296408958431812, 0.3684668299533723),
    c64::new(0.9294995222266386, 0.3688233156681539),
    c64::new(0.9293580119099355, 0.36917974714062),
    c64::new(0.929216364913884, 0.36953612431835065),
    c64::new(0.9290745812593157, 0.3698924471489341),
    c64::new(0.9289326609670828, 0.37024871557996636),
    c64::new(0.928790604058057, 0.37060492955905167),
    c64::new(0.9286484105531305, 0.370961089033802),
    c64::new(0.9285060804732156, 0.37131719395183754),
    c64::new(0.9283636138392444, 0.3716732442607865),
    c64::new(0.9282210106721694, 0.372029239908285),
    c64::new(0.9280782709929631, 0.37238518084197736),
    c64::new(0.9279353948226179, 0.37274106700951576),
    c64::new(0.9277923821821463, 