use super::Tensor;

macro_rules! save_load_unsigned {
    ($T:ty, $name: ident) => {
        mod $name {
            use super::super::Tensor;
            use crate::backends::core::private::math::random;
            use rand;

            #[test]
            fn test_save_load() {
                let mut generator = random::RandomGenerator::new(None);
                let tensor_in: Tensor<_> = generator.random_uniform_tensor::<$T>(10_000);
                let path = format!("/tmp/{}", rand::random::<usize>());
                tensor_in.save_to_file(&path).unwrap();
                println!("saved");
                let tensor_out: Tensor<Vec<$T>> = Tensor::load_from_file(&path).unwrap();
                // test
                assert_eq!(tensor_in, tensor_out);
            }
        }
    };
}

save_load_unsigned!(u8, tests_u8);
save_load_unsigned!(u16, tests_u16);
save_load_unsigned!(u32, tests_u32);
save_load_unsigned!(u64, tests_u64);
save_load_unsigned!(u128, tests_u128);

#[test]
fn test_add_u32() {
    let t_1 = Tensor::from_container(vec![
        2_614_422_625_u32,
        1_347_010_255_u32,
        1_755_118_555_u32,
        3_348_067_670_u32,
        3_896_589_259_u32,
        97_617_327_u32,
        1_545_053_739_u32,
        1_211_085_433_u32,
        2_684_538_667_u32,
        202_832_626_u32,
        1_638_508_087_u32,
        879_523_200_u32,
        2_456_511_176_u32,
        2_648_745_580_u32,
        967_205_272_u32,
        54_854_762_u32,
        2_609_115_771_u32,
        1_725_392_344_u32,
        2_314_671_715_u32,
        1_840_995_902_u32,
        4_041_278_880_u32,
        275_079_767_u32,
        2_300_142_423_u32,
        2_333_095_686_u32,
        3_026_580_357_u32,
        21_931_374_u32,
        372_535_067_u32,
        6_439_834_u32,
        762_787_515_u32,
        2_734_668_397_u32,
        3_013_991_526_u32,
        579_324_780_u32,
        916_175_967_u32,
        850_321_436_u32,
        978_826_112_u32,
        1_360_938_704_u32,
        2_363_410_736_u32,
        353_572_296_u32,
        2_196_029_604_u32,
        1_676_698_573_u32,
        71_702_920_u32,
        433_353_586_u32,
        3_336_662_792_u32,
        3_815_644_954_u32,
        2_974_299_797_u32,
        1_990_548_820_u32,
        1_683_843_869_u32,
        2_152_628_932_u32,
        3_625_450_751_u32,
        2_366_853_676_u32,
        1_798_342_904_u32,
        2_869_368_979_u32,
        1_185_695_639_u32,
        3_173_469_147_u32,
        1_531_916_725_u32,
        3_326_214_024_u32,
        2_067_990_523_u32,
        976_120_805_u32,
        3_535_693_006_u32,
        4_223_913_473_u32,
        2_143_410_133_u32,
        187_637_181_u32,
        2_370_649_336_u32,
        3_155_284_399_u32,
        3_282_898_811_u32,
        3_068_767_567_u32,
        3_033_732_496_u32,
        3_278_852_653_u32,
        1_988_815_405_u32,
        3_318_268_258_u32,
        402_934_292_u32,
        3_162_645_643_u32,
        2_103_209_800_u32,
        4_253_170_701_u32,
        2_489_673_789_u32,
        2_224_135_091_u32,
        1_848_398_457_u32,
        3_159_326_514_u32,
        3_865_725_686_u32,
        674_027_046_u32,
        3_191_092_214_u32,
        356_413_912_u32,
        682_734_067_u32,
        2_368_555_344_u32,
        614_314_161_u32,
        3_515_266_737_u32,
        949_414_245_u32,
        2_046_032_417_u32,
        1_495_462_201_u32,
        2_307_315_576_u32,
        1_960_455_472_u32,
        917_911_666_u32,
        1_518_075_072_u32,
        2_925_772_427_u32,
        298_590_050_u32,
        1_441_972_928_u32,
        666_987_301_u32,
        2_167_997_170_u32,
        3_413_359_382_u32,
        3_526_531_810_u32,
    ]);
    let t_2 = Tensor::from_container(vec![
        3_428_858_567_u32,
        827_447_270_u32,
        959_110_479_u32,
        4_184_350_429_u32,
        1_820_415_259_u32,
        2_322_099_741_u32,
        1_328_906_591_u32,
        1_664_312_159_u32,
        549_610_931_u32,
        2_945_591_302_u32,
        295_342_634_u32,
        1_589_486_080_u32,
        1_359_822_125_u32,
        1_285_568_394_u32,
        1_881_925_871_u32,
        3_058_045_327_u32,
        1_773_709_235_u32,
        3_813_730_789_u32,
        823_940_101_u32,
        2_480_100_080_u32,
        3_639_129_118_u32,
        759_351_495_u32,
        1_301_750_125_u32,
        1_054_832_776_u32,
        3_245_556_275_u32,
        2_800_997_186_u32,
        1_256_287_364_u32,
        2_573_603_461_u32,
        2_328_221_582_u32,
        1_633_069_253_u32,
        102_853_950_u32,
        2_716_685_335_u32,
        503_267_884_u32,
        2_202_048_416_u32,
        1_602_161_938_u32,
        1_927_466_558_u32,
        858_392_614_u32,
        956_183_465_u32,
        4_135_389_917_u32,
        951_071_347_u32,
        2_318_567_902_u32,
        2_004_258_446_u32,
        1_797_038_763_u32,
        1_610_761_714_u32,
        3_236_519_313_u32,
        316_586_765_u32,
        307_967_731_u32,
        3_588_485_359_u32,
        1_947_118_682_u32,
        2_002_927_095_u32,
        1_136_304_281_u32,
        1_065_157_362_u32,
        3_714_003_080_u32,
        786_946_775_u32,
        2_441_787_699_u32,
        832_944_437_u32,
        3_651_539_633_u32,
        798_050_864_u32,
        669_130_367_u32,
        2_000_552_570_u32,
        2_700_875_050_u32,
        1_118_200_520_u32,
        708_688_839_u32,
        3_463_285_323_u32,
        2_270_978_169_u32,
        237_144_352_u32,
        660_096_080_u32,
        4_230_221_095_u32,
        3_280_398_471_u32,
        3_354_541_293_u32,
        833_549_915_u32,
        1_136_697_343_u32,
        1_096_316_847_u32,
        2_476_951_406_u32,
        3_971_141_200_u32,
        12_788_335_u32,
        1_580_197_299_u32,
        2_444_226_867_u32,
        387_836_375_u32,
        4_067_693_575_u32,
        3_918_490_972_u32,
        3_704_639_326_u32,
        664_901_677_u32,
        1_847_150_125_u32,
        476_514_752_u32,
        466_866_141_u32,
        704_667_620_u32,
        2_652_242_441_u32,
        3_683_680_188_u32,
        2_589_696_574_u32,
        571_587_908_u32,
        953_792_331_u32,
        3_677_654_843_u32,
        2_056_915_677_u32,
        3_272_850_239_u32,
        1_669_788_385_u32,
        2_311_731_086_u32,
        162_748_026_u32,
        907_347_406_u32,
        2_760_784_143_u32,
    ]);
    let ground_truth_t_3 = Tensor::from_container(vec![
        1_748_313_896_u32,
        2_174_457_525_u32,
        2_714_229_034_u32,
        3_237_450_803_u32,
        1_422_037_222_u32,
        2_419_717_068_u32,
        2_873_960_330_u32,
        2_875_397_592_u32,
        3_234_149_598_u32,
        3_148_423_928_u32,
        1_933_850_721_u32,
        2_469_009_280_u32,
        3_816_333_301_u32,
        3_934_313_974_u32,
        2_849_131_143_u32,
        3_112_900_089_u32,
        87_857_710_u32,
        1_244_155_837_u32,
        3_138_611_816_u32,
        26_128_686_u32,
        3_385_440_702_u32,
        1_034_431_262_u32,
        3_601_892_548_u32,
        3_387_928_462_u32,
        1_977_169_336_u32,
        2_822_928_560_u32,
        1_628_822_431_u32,
        2_580_043_295_u32,
        3_091_009_097_u32,
        72_770_354_u32,
        3_116_845_476_u32,
        3_296_010_115_u32,
        1_419_443_851_u32,
        3_052_369_852_u32,
        2_580_988_050_u32,
        3_288_405_262_u32,
        3_221_803_350_u32,
        1_309_755_761_u32,
        2_036_452_225_u32,
        2_627_769_920_u32,
        2_390_270_822_u32,
        2_437_612_032_u32,
        838_734_259_u32,
        1_131_439_372_u32,
        1_915_851_814_u32,
        2_307_135_585_u32,
        1_991_811_600_u32,
        1_446_146_995_u32,
        1_277_602_137_u32,
        74_813_475_u32,
        2_934_647_185_u32,
        3_934_526_341_u32,
        604_731_423_u32,
        3_960_415_922_u32,
        3_973_704_424_u32,
        4_159_158_461_u32,
        1_424_562_860_u32,
        1_774_171_669_u32,
        4_204_823_373_u32,
        1_929_498_747_u32,
        549_317_887_u32,
        1_305_837_701_u32,
        3_079_338_175_u32,
        2_323_602_426_u32,
        1_258_909_684_u32,
        3_305_911_919_u32,
        3_693_828_576_u32,
        3_214_106_452_u32,
        974_246_580_u32,
        2_377_842_255_u32,
        1_236_484_207_u32,
        4_375_690_u32,
        3_199_526_647_u32,
        2_435_154_811_u32,
        2_165_847_693_u32,
        2_236_923_426_u32,
        3_428_595_756_u32,
        1_308_586_085_u32,
        4_253_562_061_u32,
        446_753_325_u32,
        2_814_615_890_u32,
        4_061_053_238_u32,
        1_347_635_744_u32,
        4_215_705_469_u32,
        1_090_828_913_u32,
        3_982_132_878_u32,
        1_654_081_865_u32,
        403_307_562_u32,
        884_175_093_u32,
        602_044_854_u32,
        2_532_043_380_u32,
        1_871_703_997_u32,
        900_762_619_u32,
        687_720_808_u32,
        3_571_440_289_u32,
        3_111_761_313_u32,
        2_978_718_387_u32,
        2_330_745_196_u32,
        25_739_492_u32,
        1_992_348_657_u32,
    ]);
    let mut t_3 = Tensor::allocate(0_u32, 100);
    t_3.fill_with_wrapping_add(&t_1, &t_2);

    assert_eq!(t_3, ground_truth_t_3, "we are testing addition");
}

#[test]
fn test_sub_u64() {
    let t_1 = Tensor::from_container(vec![
        5_682_232_049_849_203_449_u64,
        1_744_272_140_419_931_610_u64,
        6_524_694_120_235_710_248_u64,
        501_685_223_587_207_708_u64,
        7_454_825_121_449_404_861_u64,
        1_452_049_147_138_516_728_u64,
        3_744_089_800_795_951_655_u64,
        2_900_714_251_440_266_072_u64,
        2_885_003_742_441_599_873_u64,
        5_127_037_330_303_939_263_u64,
        3_942_793_256_559_402_137_u64,
        2_938_215_163_794_025_597_u64,
        3_194_270_088_293_124_907_u64,
        3_798_617_854_173_374_109_u64,
        2_281_550_512_455_919_685_u64,
        1_378_021_925_594_404_903_u64,
        6_273_819_789_066_539_195_u64,
        5_891_518_315_759_560_031_u64,
        6_569_862_020_994_290_872_u64,
        2_312_304_860_409_402_175_u64,
        3_768_205_285_282_560_447_u64,
        2_813_718_090_332_844_130_u64,
        4_741_992_406_149_366_296_u64,
        2_862_912_615_999_044_257_u64,
        2_711_698_756_636_236_379_u64,
        3_105_025_607_153_753_493_u64,
        3_280_659_296_609_069_569_u64,
        1_621_356_564_053_932_659_u64,
        244_394_454_277_671_115_u64,
        1_370_168_407_221_172_838_u64,
        384_807_778_723_441_456_u64,
        5_421_384_837_838_501_695_u64,
        3_524_866_043_795_281_573_u64,
        273_224_951_302_481_390_u64,
        8_874_399_707_947_016_287_u64,
        5_042_853_686_974_107_712_u64,
        8_593_762_746_401_730_055_u64,
        4_298_169_213_116_104_086_u64,
        1_043_682_735_183_771_811_u64,
        8_271_963_865_357_943_237_u64,
        2_866_933_850_832_375_526_u64,
        3_680_273_731_625_120_587_u64,
        5_594_513_115_859_518_166_u64,
        1_643_917_283_539_244_290_u64,
        3_172_178_086_476_235_900_u64,
        6_964_486_530_272_725_036_u64,
        6_025_940_910_517_479_800_u64,
        8_277_718_434_101_601_483_u64,
        8_184_281_612_310_786_511_u64,
        5_373_031_274_997_880_981_u64,
        443_782_149_988_086_463_u64,
        9_185_207_564_855_550_126_u64,
        3_175_405_486_723_930_612_u64,
        538_795_803_601_238_624_u64,
        1_842_522_998_755_997_387_u64,
        756_815_213_533_913_513_u64,
        4_792_029_986_473_993_888_u64,
        4_782_811_555_589_976_751_u64,
        4_765_160_184_182_081_015_u64,
        6_870_421_860_884_204_987_u64,
        6_644_609_928_302_751_438_u64,
        9_205_665_417_060_638_521_u64,
        4_422_362_498_965_857_329_u64,
        3_911_541_231_075_340_397_u64,
        714_780_100_332_094_572_u64,
        854_285_349_422_025_761_u64,
        7_998_144_870_496_815_069_u64,
        4_601_820_771_957_226_501_u64,
        4_668_015_978_555_069_529_u64,
        3_107_134_174_330_286_017_u64,
        8_556_643_770_851_938_756_u64,
        7_603_022_701_719_395_789_u64,
        9_061_759_085_783_731_100_u64,
        335_871_293_124_179_717_u64,
        578_609_166_965_025_587_u64,
        8_344_077_009_920_132_885_u64,
        5_890_072_533_484_701_885_u64,
        4_572_233_892_255_728_435_u64,
        6_510_971_065_603_537_789_u64,
        2_119_489_420_934_143_588_u64,
        7_384_712_968_731_389_043_u64,
        5_631_603_782_423_650_945_u64,
        2_426_176_736_130_500_836_u64,
        8_725_885_473_278_349_136_u64,
        6_998_312_559_885_650_695_u64,
        1_747_649_994_418_612_192_u64,
        5_557_047_201_979_978_882_u64,
        4_330_564_741_999_955_015_u64,
        1_423_746_095_735_226_283_u64,
        6_729_353_041_636_611_170_u64,
        3_912_555_288_358_270_774_u64,
        6_236_800_801_119_694_988_u64,
        1_119_102_165_244_657_550_u64,
        5_444_700_680_136_175_568_u64,
        6_107_520_479_033_799_392_u64,
        6_092_621_673_178_322_094_u64,
        2_613_801_610_897_795_471_u64,
        7_958_414_627_268_864_059_u64,
        1_701_360_089_741_291_949_u64,
        8_900_744_252_335_003_997_u64,
    ]);
    let t_2 = Tensor::from_container(vec![
        8_256_890_089_369_290_096_u64,
        6_729_858_587_364_974_993_u64,
        7_847_985_733_087_156_225_u64,
        4_256_288_592_723_368_540_u64,
        1_794_053_349_452_132_041_u64,
        6_010_968_597_662_138_399_u64,
        6_274_700_101_275_637_475_u64,
        3_672_569_542_766_325_569_u64,
        7_783_627_030_003_669_629_u64,
        249_357_646_255_069_879_u64,
        5_557_476_119_820_039_974_u64,
        8_042_948_614_404_456_368_u64,
        4_654_915_497_230_252_172_u64,
        7_722_972_477_579_752_886_u64,
        258_964_119_735_943_544_u64,
        3_661_700_972_414_689_603_u64,
        5_780_010_438_965_763_305_u64,
        5_399_007_971_131_851_993_u64,
        9_009_523_661_328_089_448_u64,
        670_837_492_260_568_551_u64,
        8_553_265_509_497_774_774_u64,
        475_007_578_406_922_623_u64,
        1_656_958_878_392_217_405_u64,
        3_145_284_643_778_286_187_u64,
        6_211_468_814_998_169_736_u64,
        7_898_586_816_448_146_424_u64,
        6_385_644_578_140_856_445_u64,
        6_278_113_144_098_235_027_u64,
        5_508_031_993_944_422_488_u64,
        2_541_351_454_611_805_754_u64,
        253_476_817_899_518_218_u64,
        4_042_272_828_677_076_320_u64,
        6_273_812_701_503_178_622_u64,
        7_154_361_991_326_158_245_u64,
        4_812_968_649_666_322_424_u64,
        8_058_877_626_669_330_796_u64,
        2_570_559_734_648_418_432_u64,
        3_260_085_933_573_705_643_u64,
        1_282_517_144_950_793_850_u64,
        1_370_863_113_856_127_345_u64,
        7_751_961_484_782_528_551_u64,
        2_576_515_167_053_557_195_u64,
        6_023_795_786_532_458_230_u64,
        6_726_030_942_349_870_732_u64,
        7_466_418_281_703_253_736_u64,
        8_567_435_608_821_064_654_u64,
        1_678_961_003_340_349_987_u64,
        3_502_334_064_042_353_274_u64,
        3_731_187_845_427_012_882_u64,
        5_317_359_253_712_576_816_u64,
        6_534_183_265_395_520_755_u64,
        3_251_278_594_118_653_876_u64,
        8_455_470_979_973_987_894_u64,
        1_134_450_355_974_787_411_u64,
        2_087_289_461_344_972_800_u64,
        898_091_164_345_629_933_u64,
        1_383_688_945_649_969_441_u64,
        6_412_373_125_771_730_589_u64,
        3_137_727_871_406_467_282_u64,
        2_531_450_854_507_130_283_u64,
        8_942_523_860_499_484_955_u64,
        3_053_185_116_942_316_003_u64,
        7_573_298_098_522_728_453_u64,
        7_850_035_594_752_589_513_u64,
        7_609_365_690_458_693_792_u64,
        3_979_440_714_450_645_544_u64,
        8_679_308_680_362_097_737_u64,
        3_937_728_290_719_953_722_u64,
        3_848_494_478_551_479_774_u64,
        3_384_383_891_744_980_023_u64,
        7_516_977_367_724_693_326_u64,
        435_538_850_065_011_084_u64,
        2_232_114_847_229_197_016_u64,
        8_939_199_010_658_684_319_u64,
        2_450_683_567_053_287_115_u64,
        7_734_458_772_215_536_274_u64,
        8_218_782_583_431_213_252_u64,
        8_553_066_689_779_351_731_u64,
        3_832_186_301_178_121_773_u64,
        6_381_512_211_621_916_311_u64,
        1_300_796_182_487_056_551_u64,
        8_878_587_019_650_156_826_u64,
        8_211_502_017_418_832_896_u64,
        8_604_481_095_678_650_971_u64,
        5_587_624_902_285_300_211_u64,
        3_426_416_825_207_801_687_u64,
        6_489_160_959_956_510_743_u64,
        7_128_903_513_419_063_730_u64,
        4_040_914_739_727_604_681_u64,
        2_688_309_576_123_447_655_u64,
        6_492_809_116_044_763_762_u64,
        7_116_289_826_504_895_799_u64,
        3_531_910_189_811_123_524_u64,
        789_365_259_912_898_159_u64,
        6_469_517_349_990_767_948_u64,
        1_189_741_323_354_180_502_u64,
        1_445_291_182_187_601_512_u64,
        8_147_987_973_554_022_701_u64,
        2_888_875_140_678_677_703_u64,
        7_057_288_198_514_267_233_u64,
    ]);
    let ground_truth_t_3 = Tensor::from_container(vec![
        15_872_086_034_189_464_969_u64,
        13_461_157_626_764_508_233_u64,
        17_123_452_460_858_105_639_u64,
        14_692_140_704_573_390_784_u64,
        5_660_771_771_997_272_820_u64,
        13_887_824_623_185_929_945_u64,
        15_916_133_773_229_865_796_u64,
        17_674_888_782_383_492_119_u64,
        13_548_120_786_147_481_860_u64,
        4_877_679_684_048_869_384_u64,
        16_832_061_210_448_913_779_u64,
        13_342_010_623_099_120_845_u64,
        16_986_098_664_772_424_351_u64,
        14_522_389_450_303_172_839_u64,
        2_022_586_392_719_976_141_u64,
        16_163_065_026_889_266_916_u64,
        493_809_350_100_775_890_u64,
        492_510_344_627_708_038_u64,
        16_007_082_433_375_753_040_u64,
        1_641_467_368_148_833_624_u64,
        13_661_683_849_494_337_289_u64,
        2_338_710_511_925_921_507_u64,
        3_085_033_527_757_148_891_u64,
        18_164_372_045_930_309_686_u64,
        14_946_974_015_347_618_259_u64,
        13_653_182_864_415_158_685_u64,
        15_341_758_792_177_764_740_u64,
        13_789_987_493_665_249_248_u64,
        13_183_106_534_042_800_243_u64,
        17_275_561_026_318_918_700_u64,
        131_330_960_823_923_238_u64,
        1_379_112_009_161_425_375_u64,
        15_697_797_416_001_654_567_u64,
        11_565_607_033_685_874_761_u64,
        4_061_431_058_280_693_863_u64,
        15_430_720_134_014_328_532_u64,
        6_023_203_011_753_311_623_u64,
        1_038_083_279_542_398_443_u64,
        18_207_909_663_942_529_577_u64,
        6_901_100_751_501_815_892_u64,
        13_561_716_439_759_398_591_u64,
        1_103_758_564_571_563_392_u64,
        18_017_461_403_036_611_552_u64,
        13_364_630_414_898_925_174_u64,
        14_152_503_878_482_533_780_u64,
        16_843_794_995_161_211_998_u64,
        4_346_979_907_177_129_813_u64,
        4_775_384_370_059_248_209_u64,
        4_453_093_766_883_773_629_u64,
        55_672_021_285_304_165_u64,
        12_356_342_958_302_117_324_u64,
        5_933_928_970_736_896_250_u64,
        13_166_678_580_459_494_334_u64,
        17_851_089_521_336_002_829_u64,
        18_201_977_611_120_576_203_u64,
        18_305_468_122_897_835_196_u64,
        3_408_341_040_824_024_447_u64,
        16_817_182_503_527_797_778_u64,
        1_627_432_312_775_613_733_u64,
        4_338_971_006_377_074_704_u64,
        16_148_830_141_512_818_099_u64,
        6_152_480_300_118_322_518_u64,
        15_295_808_474_152_680_492_u64,
        14_508_249_710_032_302_500_u64,
        11_552_158_483_582_952_396_u64,
        15_321_588_708_680_931_833_u64,
        17_765_580_263_844_268_948_u64,
        664_092_481_237_272_779_u64,
        819_521_500_003_589_755_u64,
        18_169_494_356_294_857_610_u64,
        1_039_666_403_127_245_430_u64,
        7_167_483_851_654_384_705_u64,
        6_829_644_238_554_534_084_u64,
        9_843_416_356_175_047_014_u64,
        16_574_669_673_621_290_088_u64,
        609_618_237_704_596_611_u64,
        16_118_034_023_763_040_249_u64,
        14_465_911_276_185_928_320_u64,
        2_678_784_764_425_416_016_u64,
        14_184_721_283_021_778_893_u64,
        6_083_916_786_244_332_492_u64,
        15_199_760_836_483_045_735_u64,
        12_661_418_792_421_219_556_u64,
        121_404_377_599_698_165_u64,
        1_410_687_657_600_350_484_u64,
        16_767_977_242_920_362_121_u64,
        17_514_630_315_733_019_755_u64,
        15_648_405_302_290_442_901_u64,
        15_829_575_429_717_173_218_u64,
        4_041_043_465_513_163_515_u64,
        15_866_490_246_023_058_628_u64,
        17_567_255_048_324_350_805_u64,
        16_033_936_049_143_085_642_u64,
        4_655_335_420_223_277_409_u64,
        18_084_747_202_752_583_060_u64,
        4_902_880_349_824_141_592_u64,
        1_168_510_428_710_193_959_u64,
        18_257_170_727_424_392_974_u64,
        17_259_229_022_772_165_862_u64,
        1_843_456_053_820_736_764_u64,
    ]);
    let mut t_3 = Tensor::allocate(0_u64, 100);
    t_3.fill_with_wrapping_sub(&t_1, &t_2);

    assert_eq!(t_3, ground_truth_t_3, "we are testing u-64 sub ");
}

#[test]
fn test_sub_u32() {
    let t_1 = Tensor::from_container(vec![
        2_614_422_625_u32,
        1_347_010_255_u32,
        1_755_118_555_u32,
        3_348_067_670_u32,
        3_896_589_259_u32,
        97_617_327_u32,
        1_545_053_739_u32,
        1_211_085_433_u32,
        2_684_538_667_u32,
        202_832_626_u32,
        1_638_508_087_u32,
        879_523_200_u32,
        2_456_511_176_u32,
        2_648_745_580_u32,
        967_205_272_u32,
        54_854_762_u32,
        2_609_115_771_u32,
        1_725_392_344_u32,
        2_314_671_715_u32,
        1_840_995_902_u32,
        4_041_278_880_u32,
        275_079_767_u32,
        2_300_142_423_u32,
        2_333_095_686_u32,
        3_026_580_357_u32,
        21_931_374_u32,
        372_535_067_u32,
        6_439_834_u32,
        762_787_515_u32,
        2_734_668_397_u32,
        3_013_991_526_u32,
        579_324_780_u32,
        916_175_967_u32,
        850_321_436_u32,
        978_826_112_u32,
        1_360_938_704_u32,
        2_363_410_736_u32,
        353_572_296_u32,
        2_196_029_604_u32,
        1_676_698_573_u32,
        71_702_920_u32,
        433_353_586_u32,
        3_336_662_792_u32,
        3_815_644_954_u32,
        2_974_299_797_u32,
        1_990_548_820_u32,
        1_683_843_869_u32,
        2_152_628_932_u32,
        3_625_450_751_u32,
        2_366_853_676_u32,
        1_798_342_904_u32,
        2_869_368_979_u32,
        1_185_695_639_u32,
        3_173_469_147_u32,
        1_531_916_725_u32,
        3_326_214_024_u32,
        2_067_990_523_u32,
        976_120_805_u32,
        3_535_693_006_u32,
        4_223_913_473_u32,
        2_143_410_133_u32,
        187_637_181_u32,
        2_370_649_336_u32,
        3_155_284_399_u32,
        3_282_898_811_u32,
        3_068_767_567_u32,
        3_033_732_496_u32,
        3_278_852_653_u32,
        1_988_815_405_u32,
        3_318_268_258_u32,
        402_934_292_u32,
        3_162_645_643_u32,
        2_103_209_800_u32,
        4_253_170_701_u32,
        2_489_673_789_u32,
        2_224_135_091_u32,
        1_848_398_457_u32,
        3_159_326_514_u32,
        3_865_725_686_u32,
        674_027_046_u32,
        3_191_092_214_u32,
        356_413_912_u32,
        682_734_067_u32,
        2_368_555_344_u32,
        614_314_161_u32,
        3_515_266_737_u32,
        949_414_245_u32,
        2_046_032_417_u32,
        1_495_462_201_u32,
        2_307_315_576_u32,
        1_960_455_472_u32,
        917_911_666_u32,
        1_518_075_072_u32,
        2_925_772_427_u32,
        298_590_050_u32,
        1_441_972_928_u32,
        666_987_301_u32,
        2_167_997_170_u32,
        3_413_359_382_u32,
        3_526_531_810_u32,
    ]);
    let t_2 = Tensor::from_container(vec![
        3_428_858_567_u32,
        827_447_270_u32,
        959_110_479_u32,
        4_184_350_429_u32,
        1_820_415_259_u32,
        2_322_099_741_u32,
        1_328_906_591_u32,
        1_664_312_159_u32,
        549_610_931_u32,
        2_945_591_302_u32,
        295_342_634_u32,
        1_589_486_080_u32,
        1_359_822_125_u32,
        1_285_568_394_u32,
        1_881_925_871_u32,
        3_058_045_327_u32,
        1_773_709_235_u32,
        3_813_730_789_u32,
        823_940_101_u32,
        2_480_100_080_u32,
        3_639_129_118_u32,
        759_351_495_u32,
        1_301_750_125_u32,
        1_054_832_776_u32,
        3_245_556_275_u32,
        2_800_997_186_u32,
        1_256_287_364_u32,
        2_573_603_461_u32,
        2_328_221_582_u32,
        1_633_069_253_u32,
        102_853_950_u32,
        2_716_685_335_u32,
        503_267_884_u32,
        2_202_048_416_u32,
        1_602_161_938_u32,
        1_927_466_558_u32,
        858_392_614_u32,
        956_183_465_u32,
        4_135_389_917_u32,
        951_071_347_u32,
        2_318_567_902_u32,
        2_004_258_446_u32,
        1_797_038_763_u32,
        1_610_761_714_u32,
        3_236_519_313_u32,
        316_586_765_u32,
        307_967_731_u32,
        3_588_485_359_u32,
        1_947_118_682_u32,
        2_002_927_095_u32,
        1_136_304_281_u32,
        1_065_157_362_u32,
        3_714_003_080_u32,
        786_946_775_u32,
        2_441_787_699_u32,
        832_944_437_u32,
        3_651_539_633_u32,
        798_050_864_u32,
        669_130_367_u32,
        2_000_552_570_u32,
        2_700_875_050_u32,
        1_118_200_520_u32,
        708_688_839_u32,
        3_463_285_323_u32,
        2_270_978_169_u32,
        237_144_352_u32,
        660_096_080_u32,
        4_230_221_095_u32,
        3_280_398_471_u32,
        3_354_541_293_u32,
        833_549_915_u32,
        1_136_697_343_u32,
        1_096_316_847_u32,
        2_476_951_406_u32,
        3_971_141_200_u32,
        12_788_335_u32,
        1_580_197_299_u32,
        2_444_226_867_u32,
        387_836_375_u32,
        4_067_693_575_u32,
        3_918_490_972_u32,
        3_704_639_326_u32,
        664_901_677_u32,
        1_847_150_125_u32,
        476_514_752_u32,
        466_866_141_u32,
        704_667_620_u32,
        2_652_242_441_u32,
        3_683_680_188_u32,
        2_589_696_574_u32,
        571_587_908_u32,
        953_792_331_u32,
        3_677_654_843_u32,
        2_056_915_677_u32,
        3_272_850_239_u32,
        1_669_788_385_u32,
        2_311_731_086_u32,
        162_748_026_u32,
        907_347_406_u32,
        2_760_784_143_u32,
    ]);
    let ground_truth_t_3 = Tensor::from_container(vec![
        3_480_531_354_u32,
        519_562_985_u32,
        796_008_076_u32,
        3_458_684_537_u32,
        2_076_174_000_u32,
        2_070_484_882_u32,
        216_147_148_u32,
        3_841_740_570_u32,
        2_134_927_736_u32,
        1_552_208_620_u32,
        1_343_165_453_u32,
        3_585_004_416_u32,
        1_096_689_051_u32,
        1_363_177_186_u32,
        3_380_246_697_u32,
        1_291_776_731_u32,
        835_406_536_u32,
        2_206_628_851_u32,
        1_490_731_614_u32,
        3_655_863_118_u32,
        402_149_762_u32,
        3_810_695_568_u32,
        998_392_298_u32,
        1_278_262_910_u32,
        4_075_991_378_u32,
        1_515_901_484_u32,
        3_411_214_999_u32,
        1_727_803_669_u32,
        2_729_533_229_u32,
        1_101_599_144_u32,
        2_911_137_576_u32,
        2_157_606_741_u32,
        412_908_083_u32,
        2_943_240_316_u32,
        3_671_631_470_u32,
        3_728_439_442_u32,
        1_505_018_122_u32,
        3_692_356_127_u32,
        2_355_606_983_u32,
        725_627_226_u32,
        2_048_102_314_u32,
        2_724_062_436_u32,
        1_539_624_029_u32,
        2_204_883_240_u32,
        4_032_747_780_u32,
        1_673_962_055_u32,
        1_375_876_138_u32,
        2_859_110_869_u32,
        1_678_332_069_u32,
        363_926_581_u32,
        662_038_623_u32,
        1_804_211_617_u32,
        1_766_659_855_u32,
        2_386_522_372_u32,
        3_385_096_322_u32,
        2_493_269_587_u32,
        2_711_418_186_u32,
        178_069_941_u32,
        2_866_562_639_u32,
        2_223_360_903_u32,
        3_737_502_379_u32,
        3_364_403_957_u32,
        1_661_960_497_u32,
        3_986_966_372_u32,
        1_011_920_642_u32,
        2_831_623_215_u32,
        2_373_636_416_u32,
        3_343_598_854_u32,
        3_003_384_230_u32,
        4_258_694_261_u32,
        3_864_351_673_u32,
        2_025_948_300_u32,
        1_006_892_953_u32,
        1_776_219_295_u32,
        2_813_499_885_u32,
        2_211_346_756_u32,
        268_201_158_u32,
        715_099_647_u32,
        3_477_889_311_u32,
        901_300_767_u32,
        3_567_568_538_u32,
        946_741_882_u32,
        17_832_390_u32,
        521_405_219_u32,
        137_799_409_u32,
        3_048_400_596_u32,
        244_746_625_u32,
        3_688_757_272_u32,
        2_106_749_309_u32,
        4_012_586_298_u32,
        1_388_867_564_u32,
        4_259_086_631_u32,
        2_135_387_525_u32,
        868_856_750_u32,
        1_320_707_107_u32,
        4_067_151_839_u32,
        2_650_223_511_u32,
        2_005_249_144_u32,
        2_506_011_976_u32,
        765_747_667_u32,
    ]);
    let mut t_3 = Tensor::allocate(0_u32, 100);
    t_3.fill_with_wrapping_sub(&t_1, &t_2);

    assert_eq!(t_3, ground_truth_t_3, "we are testing substraction");
}

#[test]
fn test_add_u64() {
    let t_1 = Tensor::from_container(vec![
        5_682_232_049_849_203_449_u64,
        1_744_272_140_419_931_610_u64,
        6_524_694_120_235_710_248_u64,
        501_685_223_587_207_708_u64,
        7_454_825_121_449_404_861_u64,
        1_452_049_147_138_516_728_u64,
        3_744_089_800_795_951_655_u64,
        2_900_714_251_440_266_072_u64,
        2_885_003_742_441_599_873_u64,
        5_127_037_330_303_939_263_u64,
        3_942_793_256_559_402_137_u64,
        2_938_215_163_794_025_597_u64,
        3_194_270_088_293_124_907_u64,
        3_798_617_854_173_374_109_u64,
        2_281_550_512_455_919_685_u64,
        1_378_021_925_594_404_903_u64,
        6_273_819_789_066_539_195_u64,
        5_891_518_315_759_560_031_u64,
        6_569_862_020_994_290_872_u64,
        2_312_304_860_409_402_175_u64,
        3_768_205_285_282_560_447_u64,
        2_813_718_090_332_844_130_u64,
        4_741_992_406_149_366_296_u64,
        2_862_912_615_999_044_257_u64,
        2_711_698_756_636_236_379_u64,
        3_105_025_607_153_753_493_u64,
        3_280_659_296_609_069_569_u64,
        1_621_356_564_053_932_659_u64,
        244_394_454_277_671_115_u64,
        1_370_168_407_221_172_838_u64,
        384_807_778_723_441_456_u64,
        5_421_384_837_838_501_695_u64,
        3_524_866_043_795_281_573_u64,
        273_224_951_302_481_390_u64,
        8_874_399_707_947_016_287_u64,
        5_042_853_686_974_107_712_u64,
        8_593_762_746_401_730_055_u64,
        4_298_169_213_116_104_086_u64,
        1_043_682_735_183_771_811_u64,
        8_271_963_865_357_943_237_u64,
        2_866_933_850_832_375_526_u64,
        3_680_273_731_625_120_587_u64,
        5_594_513_115_859_518_166_u64,
        1_643_917_283_539_244_290_u64,
        3_172_178_086_476_235_900_u64,
        6_964_486_530_272_725_036_u64,
        6_025_940_910_517_479_800_u64,
        8_277_718_434_101_601_483_u64,
        8_184_281_612_310_786_511_u64,
        5_373_031_274_997_880_981_u64,
        443_782_149_988_086_463_u64,
        9_185_207_564_855_550_126_u64,
        3_175_405_486_723_930_612_u64,
        538_795_803_601_238_624_u64,
        1_842_522_998_755_997_387_u64,
        756_815_213_533_913_513_u64,
        4_792_029_986_473_993_888_u64,
        4_782_811_555_589_976_751_u64,
        4_765_160_184_182_081_015_u64,
        6_870_421_860_884_204_987_u64,
        6_644_609_928_302_751_438_u64,
        9_205_665_417_060_638_521_u64,
        4_422_362_498_965_857_329_u64,
        3_911_541_231_075_340_397_u64,
        714_780_100_332_094_572_u64,
        854_285_349_422_025_761_u64,
        7_998_144_870_496_815_069_u64,
        4_601_820_771_957_226_501_u64,
        4_668_015_978_555_069_529_u64,
        3_107_134_174_330_286_017_u64,
        8_556_643_770_851_938_756_u64,
        7_603_022_701_719_395_789_u64,
        9_061_759_085_783_731_100_u64,
        335_871_293_124_179_717_u64,
        578_609_166_965_025_587_u64,
        8_344_077_009_920_132_885_u64,
        5_890_072_533_484_701_885_u64,
        4_572_233_892_255_728_435_u64,
        6_510_971_065_603_537_789_u64,
        2_119_489_420_934_143_588_u64,
        7_384_712_968_731_389_043_u64,
        5_631_603_782_423_650_945_u64,
        2_426_176_736_130_500_836_u64,
        8_725_885_473_278_349_136_u64,
        6_998_312_559_885_650_695_u64,
        1_747_649_994_418_612_192_u64,
        5_557_047_201_979_978_882_u64,
        4_330_564_741_999_955_015_u64,
        1_423_746_095_735_226_283_u64,
        6_729_353_041_636_611_170_u64,
        3_912_555_288_358_270_774_u64,
        6_236_800_801_119_694_988_u64,
        1_119_102_165_244_657_550_u64,
        5_444_700_680_136_175_568_u64,
        6_107_520_479_033_799_392_u64,
        6_092_621_673_178_322_094_u64,
        2_613_801_610_897_795_471_u64,
        7_958_414_627_268_864_059_u64,
        1_701_360_089_741_291_949_u64,
        8_900_744_252_335_003_997_u64,
    ]);
    let t_2 = Tensor::from_container(vec![
        8_256_890_089_369_290_096_u64,
        6_729_858_587_364_974_993_u64,
        7_847_985_733_087_156_225_u64,
        4_256_288_592_723_368_540_u64,
        1_794_053_349_452_132_041_u64,
        6_010_968_597_662_138_399_u64,
        6_274_700_101_275_637_475_u64,
        3_672_569_542_766_325_569_u64,
        7_783_627_030_003_669_629_u64,
        249_357_646_255_069_879_u64,
        5_557_476_119_820_039_974_u64,
        8_042_948_614_404_456_368_u64,
        4_654_915_497_230_252_172_u64,
        7_722_972_477_579_752_886_u64,
        258_964_119_735_943_544_u64,
        3_661_700_972_414_689_603_u64,
        5_780_010_438_965_763_305_u64,
        5_399_007_971_131_851_993_u64,
        9_009_523_661_328_089_448_u64,
        670_837_492_260_568_551_u64,
        8_553_265_509_497_774_774_u64,
        475_007_578_406_922_623_u64,
        1_656_958_878_392_217_405_u64,
        3_145_284_643_778_286_187_u64,
        6_211_468_814_998_169_736_u64,
        7_898_586_816_448_146_424_u64,
        6_385_644_578_140_856_445_u64,
        6_278_113_144_098_235_027_u64,
        5_508_031_993_944_422_488_u64,
        2_541_351_454_611_805_754_u64,
        253_476_817_899_518_218_u64,
        4_042_272_828_677_076_320_u64,
        6_273_812_701_503_178_622_u64,
        7_154_361_991_326_158_245_u64,
        4_812_968_649_666_322_424_u64,
        8_058_877_626_669_330_796_u64,
        2_570_559_734_648_418_432_u64,
        3_260_085_933_573_705_643_u64,
        1_282_517_144_950_793_850_u64,
        1_370_863_113_856_127_345_u64,
        7_751_961_484_782_528_551_u64,
        2_576_515_167_053_557_195_u64,
        6_023_795_786_532_458_230_u64,
        6_726_030_942_349_870_732_u64,
        7_466_418_281_703_253_736_u64,
        8_567_435_608_821_064_654_u64,
        1_678_961_003_340_349_987_u64,
        3_502_334_064_042_353_274_u64,
        3_731_187_845_427_012_882_u64,
        5_317_359_253_712_576_816_u64,
        6_534_183_265_395_520_755_u64,
        3_251_278_594_118_653_876_u64,
        8_455_470_979_973_987_894_u64,
        1_134_450_355_974_787_411_u64,
        2_087_289_461_344_972_800_u64,
        898_091_164_345_629_933_u64,
        1_383_688_945_649_969_441_u64,
        6_412_373_125_771_730_589_u64,
        3_137_727_871_406_467_282_u64,
        2_531_450_854_507_130_283_u64,
        8_942_523_860_499_484_955_u64,
        3_053_185_116_942_316_003_u64,
        7_573_298_098_522_728_453_u64,
        7_850_035_594_752_589_513_u64,
        7_609_365_690_458_693_792_u64,
        3_979_440_714_450_645_544_u64,
        8_679_308_680_362_097_737_u64,
        3_937_728_290_719_953_722_u64,
        3_848_494_478_551_479_774_u64,
        3_384_383_891_744_980_023_u64,
        7_516_977_367_724_693_326_u64,
        435_538_850_065_011_084_u64,
        2_232_114_847_229_197_016_u64,
        8_939_199_010_658_684_319_u64,
        2_450_683_567_053_287_115_u64,
        7_734_458_772_215_536_274_u64,
        8_218_782_583_431_213_252_u64,
        8_553_066_689_779_351_731_u64,
        3_832_186_301_178_121_773_u64,
        6_381_512_211_621_916_311_u64,
        1_300_796_182_487_056_551_u64,
        8_878_587_019_650_156_826_u64,
        8_211_502_017_418_832_896_u64,
        8_604_481_095_678_650_971_u64,
        5_587_624_902_285_300_211_u64,
        3_426_416_825_207_801_687_u64,
        6_489_160_959_956_510_743_u64,
        7_128_903_513_419_063_730_u64,
        4_040_914_739_727_604_681_u64,
        2_688_309_576_123_447_655_u64,
        6_492_809_116_044_763_762_u64,
        7_116_289_826_504_895_799_u64,
        3_531_910_189_811_123_524_u64,
        789_365_259_912_898_159_u64,
        6_469_517_349_990_767_948_u64,
        1_189_741_323_354_180_502_u64,
        1_445_291_182_187_601_512_u64,
        8_147_987_973_554_022_701_u64,
        2_888_875_140_678_677_703_u64,
        7_057_288_198_514_267_233_u64,
    ]);
    let ground_truth_t_3 = Tensor::from_container(vec![
        13_939_122_139_218_493_545_u64,
        8_474_130_727_784_906_603_u64,
        14_372_679_853_322_866_473_u64,
        4_757_973_816_310_576_248_u64,
        9_248_878_470_901_536_902_u64,
        7_463_017_744_800_655_127_u64,
        10_018_789_902_071_589_130_u64,
        6_573_283_794_206_591_641_u64,
        10_668_630_772_445_269_502_u64,
        5_376_394_976_559_009_142_u64,
        9_500_269_376_379_442_111_u64,
        10_981_163_778_198_481_965_u64,
        7_849_185_585_523_377_079_u64,
        11_521_590_331_753_126_995_u64,
        2_540_514_632_191_863_229_u64,
        5_039_722_898_009_094_506_u64,
        12_053_830_228_032_302_500_u64,
        11_290_526_286_891_412_024_u64,
        15_579_385_682_322_380_320_u64,
        2_983_142_352_669_970_726_u64,
        12_321_470_794_780_335_221_u64,
        3_288_725_668_739_766_753_u64,
        6_398_951_284_541_583_701_u64,
        6_008_197_259_777_330_444_u64,
        8_923_167_571_634_406_115_u64,
        11_003_612_423_601_899_917_u64,
        9_666_303_874_749_926_014_u64,
        7_899_469_708_152_167_686_u64,
        5_752_426_448_222_093_603_u64,
        3_911_519_861_832_978_592_u64,
        638_284_596_622_959_674_u64,
        9_463_657_666_515_578_015_u64,
        9_798_678_745_298_460_195_u64,
        7_427_586_942_628_639_635_u64,
        13_687_368_357_613_338_711_u64,
        13_101_731_313_643_438_508_u64,
        11_164_322_481_050_148_487_u64,
        7_558_255_146_689_809_729_u64,
        2_326_199_880_134_565_661_u64,
        9_642_826_979_214_070_582_u64,
        10_618_895_335_614_904_077_u64,
        6_256_788_898_678_677_782_u64,
        11_618_308_902_391_976_396_u64,
        8_369_948_225_889_115_022_u64,
        10_638_596_368_179_489_636_u64,
        15_531_922_139_093_789_690_u64,
        7_704_901_913_857_829_787_u64,
        11_780_052_498_143_954_757_u64,
        11_915_469_457_737_799_393_u64,
        10_690_390_528_710_457_797_u64,
        6_977_965_415_383_607_218_u64,
        12_436_486_158_974_204_002_u64,
        11_630_876_466_697_918_506_u64,
        1_673_246_159_576_026_035_u64,
        3_929_812_460_100_970_187_u64,
        1_654_906_377_879_543_446_u64,
        6_175_718_932_123_963_329_u64,
        11_195_184_681_361_707_340_u64,
        7_902_888_055_588_548_297_u64,
        9_401_872_715_391_335_270_u64,
        15_587_133_788_802_236_393_u64,
        12_258_850_534_002_954_524_u64,
        11_995_660_597_488_585_782_u64,
        11_761_576_825_827_929_910_u64,
        8_324_145_790_790_788_364_u64,
        4_833_726_063_872_671_305_u64,
        16_677_453_550_858_912_806_u64,
        8_539_549_062_677_180_223_u64,
        8_516_510_457_106_549_303_u64,
        6_491_518_066_075_266_040_u64,
        16_073_621_138_576_632_082_u64,
        8_038_561_551_784_406_873_u64,
        11_293_873_933_012_928_116_u64,
        9_275_070_303_782_864_036_u64,
        3_029_292_734_018_312_702_u64,
        16_078_535_782_135_669_159_u64,
        14_108_855_116_915_915_137_u64,
        13_125_300_582_035_080_166_u64,
        10_343_157_366_781_659_562_u64,
        8_501_001_632_556_059_899_u64,
        8_685_509_151_218_445_594_u64,
        14_510_190_802_073_807_771_u64,
        10_637_678_753_549_333_732_u64,
        17_330_366_568_957_000_107_u64,
        12_585_937_462_170_950_906_u64,
        5_174_066_819_626_413_879_u64,
        12_046_208_161_936_489_625_u64,
        11_459_468_255_419_018_745_u64,
        5_464_660_835_462_830_964_u64,
        9_417_662_617_760_058_825_u64,
        10_405_364_404_403_034_536_u64,
        13_353_090_627_624_590_787_u64,
        4_651_012_355_055_781_074_u64,
        6_234_065_940_049_073_727_u64,
        12_577_037_829_024_567_340_u64,
        7_282_362_996_532_502_596_u64,
        4_059_092_793_085_396_983_u64,
        16_106_402_600_822_886_760_u64,
        4_590_235_230_419_969_652_u64,
        15_958_032_450_849_271_230_u64,
    ]);
    let mut t_3 = Tensor::allocate(0_u64, 100);
    t_3.fill_with_wrapping_add(&t_1, &t_2);

    assert_eq!(t_3, ground_truth_t_3, "we are testing u64 add");
}
