#[macro_use]
extern crate compact_sql;

#[test]
fn single_select() {
	let sql_str = pg_sql!(SELECT);
	assert_eq!(sql_str, "SELECT");
}

#[test]
fn single_select_literal() {
	let sql_str = pg_sql!(SELECT 1);
	assert_eq!(sql_str, "SELECT 1");
}


#[test]
fn multi_literals() {
	let sql_str = pg_sql!(SELECT FROM pg_class);
	assert_eq!(sql_str, "SELECT FROM pg_class");
}

#[test]
fn multi_punct() {
	let sql_str = pg_sql!(SELECT + -1);
	assert_eq!(sql_str, "SELECT+-1"); // no space is correct!
}

#[test]
fn punct_before_placeholder() {
	#[rustfmt::skip]
	let sql_str = pg_sql!(SELECT - {arg});
	assert_eq!(sql_str, "SELECT-$1"); // no space is correct!
}

#[test]
fn ident_and_numeric() {
	let sql_str = pg_sql!(SELECT .1 FROM pg_class);
	assert_eq!(sql_str, "SELECT.1 FROM pg_class"); // TODO: no space around .1 SHOULD be correct!
	let sql_str = pg_sql!(SELECT 0.1 FROM pg_class);
	assert_eq!(sql_str, "SELECT 0.1 FROM pg_class"); // TODO: no space after 0.1 SHOULD be correct!
}

#[test]
fn ident_and_string() {
	let sql_str = pg_sql!(SELECT "x" FROM pg_class);
	assert_eq!(sql_str, "SELECT 'x' FROM pg_class"); // TODO: no space around 'x' SHOULD be correct!
}

#[test]
fn ident_and_int() {
	// Space between digits and idents must be present
	let sql_str = pg_sql!(SELECT 1 FROM pg_class);
	assert_eq!(sql_str, "SELECT 1 FROM pg_class");
}
#[test]
fn star_as_multiplication() {
	// Space between digits and idents must be present
	let sql_str = pg_sql!(
		UPDATE limits.main m SET
			withdraw = withdraw + p.amount_full * is_accept,
			hold     = hold     - p.amount_full,
	);
	assert_eq!(
		sql_str,
		"UPDATE limits.main m SET withdraw=withdraw+p.amount_full*is_accept,hold=hold-p.amount_full"
	);
}

#[test]
fn multi_literal() {
	let sql_str = pg_sql!(SELECT ROW( 1, "ww" ) );
	assert_eq!(sql_str, "SELECT ROW(1,'ww')");
}

#[test]
fn ending_comma() {
	let sql_str = pg_sql!(
	SELECT
		ROW( 1, "ww", ),
		ARRAY[ 1, 5, 22, 1],
	FROM pg_class, pg_namespace,
	WHERE TRUE
	);
	assert_eq!(
		sql_str,
		"SELECT ROW(1,'ww'),ARRAY[1,5,22,1]FROM pg_class,pg_namespace WHERE TRUE"
	);
	let sql_str = pg_sql!(SELECT a,b,FROM c);
	assert_eq!(sql_str, "SELECT a,b FROM c"); // space before "FROM"
}

#[test]
fn ident_and_placeholder() {
	let sql_str = pg_sql!(SELECT {arg} FROM pg_class);
	assert_eq!(sql_str, "SELECT $1FROM pg_class");
	let sql_str = pg_sql!(SELECT {arg} IN (1,2));
	assert_eq!(sql_str, "SELECT $1IN(1,2)");
}

#[test]
fn complex_string_literal() {
	// Space between digits and idents must be present
	// TODO: add a check where "_" is replaced with "\x00"!
	let sql_str = pg_sql!(SELECT "a_bcd\n\\e\\\\\nf  sngl:' end:'" FROM pg_class);
	assert_eq!(
		sql_str,
		"SELECT \'a_bcd\n\\e\\\\\nf  sngl:\'\' end:\'\'\' FROM pg_class"
	);
	// The same as above but as the raw string (avoid reverse quoting)
	assert_eq!(
		sql_str,
		r#"SELECT 'a_bcd
\e\\
f  sngl:'' end:''' FROM pg_class"#
	);
}

#[test]
fn complex_string_literal2() {
	// Space between digits and idents must be present
	// TODO: add a check where "_" is replaced with "\x00"!
	let sql_str = pg_sql!(
		SELECT "quoted'x'",
		r#"quoted"y"+'x'"#,
		"\\quoted\"'z\"\\",
		'c',
		'\x33'  // TODO: add test for "\x33" (in double quotes)
	);
	assert_eq!(
		sql_str,
		"SELECT \'quoted\'\'x\'\'\',\'quoted\"y\"+\'\'x\'\'\',\'\\quoted\"\'\'z\"\\\',\'c\',\'3\'"
	);
	// The same as above but as the raw string (avoid reverse quoting)
	assert_eq!(
		sql_str,
		r#"SELECT 'quoted''x''','quoted"y"+''x''','\quoted"''z"\','c','3'"#
	);
}


#[test]
fn complex_query() {
	let sql_str = pg_sql!(
		SELECT
			{arg},
			1,
			0.2 AS t,
			{arg2},
			unnest(ARRAY[1,
				2,
				3])
		FROM pg_class p
		JOIN nspname AS n ON p.namespace = n.oid
		WHERE
			{arg2}::int = {arg}
			AND
			(p.relkind='r' OR z=ANY(any_arr))
			// commented row
	);
	const EXP: &str = concat!(
		"SELECT $1,1,0.2 AS t,$2,unnest(ARRAY[1,2,3])",
		"FROM pg_class p JOIN nspname AS n ON p.namespace=n.oid",
		" WHERE $2::int=$1AND(p.relkind=\'r\' OR z=ANY(any_arr))",
	);
	assert_eq!(sql_str, EXP);
}
