// We need it because there is a normal correct code after "ERROR" macros when "pretty-errors"
// but without that feature we fail in "panic" after which the code mentioned above is "dead"...
#![cfg_attr(not(feature = "pretty-errors"), allow(unreachable_code))]

use std::{
	convert::TryInto,
	error::Error,
	iter::Peekable,
	str::FromStr,
};


#[cfg(feature = "pretty-errors")]
use proc_macro::Level;

use proc_macro::{
	Delimiter,
	Group,
	Ident,
	Literal,
	Punct,
	Spacing,
	Span,
	TokenStream,
	TokenTree,
};
use proc_macro2::{
	Ident as Ident2,
	TokenStream as TokenStream2,
};

#[cfg(feature = "pretty-errors")]
use syn::spanned::Spanned;

use syn::{
	parse_macro_input,
	Lit,
	Type,
};
use tokio_postgres::error::{
	DbError,
	ErrorPosition,
};


pub(crate) enum QueryNumRows {
	ExactOne,
	ZeroOrOne,
	Many,
	Stream,
	Execute,
}

impl From<char> for QueryNumRows {
	fn from(ch: char) -> Self {
		match ch {
			'-' => Self::ExactOne,
			'?' => Self::ZeroOrOne,
			'*' => Self::Many,
			'#' => Self::Stream,
			'!' => Self::Execute,
			_ => unimplemented!(),
		}
	}
}

pub(crate) struct QueryData {
	query_enum:  Option<Vec<TokenTree>>,
	struct_type: Vec<TokenTree>,
	num_rows:    Option<QueryNumRows>,
	row_type:    Option<Vec<TokenTree>>,
}

struct Placeholder {
	name: String,
	pos:  String,
	idnt: Ident,
	typ:  Option<Box<syn::Type>>,
}
#[derive(Default)]
struct SqlString {
	sql:   String,
	spans: Vec<(std::ops::Range<usize>, Span)>,
}
impl SqlString {
	fn push_space(&mut self) {
		self.sql.push(' ')
	}

	fn push_ch(&mut self, ch: char, span: Span) {
		let min = self.sql.len();
		self.sql.push(ch);
		let max = self.sql.len();
		self.spans.push((min..max, span));
	}

	fn push_ch_groups(&mut self, ch: char) {
		// TODO: keep a separate _tree_ (!!!) of Span's since braces can be nested (intersected).
		self.sql.push(ch)
	}

	fn push_str(&mut self, str: &str, span: Span) {
		let min = self.sql.len();
		self.sql.push_str(str);
		let max = self.sql.len();
		self.spans.push((min..max, span));
	}
}
pub(crate) struct SqlHandler {
	sql_string:   SqlString,
	query_data:   Option<QueryData>,
	placeholders: Vec<Placeholder>,
}


#[cfg(not(feature = "pretty-errors"))]
macro_rules! ERROR {
	(<= > : $span_bgn:ident.. $span_end:ident, $msg:expr) => {
		panic!("{:?}", $msg)
	};
	($lvl:ident => $span:expr, $msg:expr,) => {
		panic!("{:?}", $msg)
	};

	($span:expr, $msg:expr $(,)?) => {
		panic!("{:?}", $msg)
	};
	($span:expr, $msg:expr, $help:expr $(,)?) => {
		panic!("{:?}", $msg)
	};
}

#[cfg(feature = "pretty-errors")]
macro_rules! ERROR {
	(<= > : $span_bgn:ident.. $span_end:ident, $msg:expr) => {
		proc_macro::Diagnostic::spanned(
			$span_bgn.span().join($span_end).unwrap(),
			Level::Error,
			$msg,
		)
		.emit()
	};
	($lvl:ident => $span:expr, $msg:expr,) => {
		proc_macro::Diagnostic::spanned($span, Level::$lvl, $msg).emit()
	};
	($span:expr, $msg:expr $(,)?) => {
		proc_macro::Diagnostic::spanned($span, Level::Error, $msg).emit()
	};
	($span:expr, $msg:expr, $help:expr $(,)?) => {
		proc_macro::Diagnostic::spanned($span, Level::Error, $msg)
			.help($help)
			.emit()
	};
}

impl SqlHandler {
	pub(crate) fn new() -> Self {
		Self {
			sql_string:   SqlString::default(),
			query_data:   None,
			placeholders: Vec::default(),
		}
	}

	pub(crate) fn finish(self) -> TokenStream {
		let Self {
			sql_string,
			query_data,
			placeholders,
		} = self;

		let rt = tokio::runtime::Runtime::new().unwrap();
		let maybe_statement = rt.block_on(Self::test_sql(&sql_string));

		let sql_string = &sql_string.sql;

		let sql = TokenStream::from(quote!(
			#sql_string
		));

		match query_data {
			None => sql,
			Some(d) => {
				Self::finish_query_data(sql.into(), d, placeholders, maybe_statement)
					.map_or_else(TokenStream::from, TokenStream::from)
			},
		}
	}

	async fn test_sql(sql_string: &SqlString) -> Option<tokio_postgres::Statement> {
		let host = option_env!("TEST_PG_HOST");
		#[rustfmt::skip]
		let port = option_env!("TEST_PG_PORT").map_or(Ok(5432), u16::from_str).unwrap();
		let user = option_env!("TEST_PG_USER").unwrap_or("postgres");
		let pass = option_env!("TEST_PG_PASS").unwrap_or("");
		let datb = option_env!("TEST_PG_DATB");

		let (host, datb) = match (host, datb) {
			(None, None) => return None,
			(Some(host), Some(datb)) => (host, datb),
			_ => {
				ERROR!(
					Span::call_site(),
					"For testing both \"TEST_PG_HOST\" and \"TEST_PG_DATB\" must be set",
				);
				return None;
			},
		};

		let connection = tokio_postgres::Config::new()
			.host(host)
			.port(port)
			.user(user)
			.password(pass)
			.dbname(datb)
			.application_name("compact_sql macro testing")
			.connect(tokio_postgres::NoTls)
			.await;

		let (client, conn) = match connection {
			Ok(v) => v,
			Err(e) => {
				// We do not want to ruin the whole process via panic!(), but since the user set
				// TEST_PG_HOST and TEST_PG_DATB emit Error as a marker we can not test it.
				ERROR!(
					Span::call_site(),
					format!("Error on connecting to the DB: {}", e),
				);
				return None;
			},
		};

		tokio::spawn(async move {
			if let Err(e) = conn.await {
				eprintln!("connection error: {}", e);
			}
		});

		let err = match client.prepare(&sql_string.sql).await {
			Ok(stmnt) => return Some(stmnt),
			Err(e) => e,
		};

		let err = match err.source().and_then(|e| e.downcast_ref::<DbError>()) {
			Some(e) => e,
			#[rustfmt::skip]
			None => {
				ERROR!(Span::call_site(), format!("Unknown error from PG lib: {}", err));
				return None;
			},
		};

		let err_pos: usize = match err.position() {
			None => {
				ERROR!(
					Span::call_site(),
					format!("Error (no position) {:?}: {}", err.code(), err.message()),
					format!("{:?}", err),
				);
				return None;
			},
			Some(ErrorPosition::Original(pos)) => (*pos).try_into().unwrap(),
			#[rustfmt::skip]
			Some(ErrorPosition::Internal{query, ..}) => {
				let _ = &query; // suppress "unusable variable"
				ERROR!(
					Span::call_site(),
					format!("Error (internal query): {:?}", err),
					format!("The internal query: {:?}", query),
				);
				return None;
			},
		};

		let err_pos = err_pos.saturating_sub(1); // If somehow err_pos == 0, it remains 0.

		let search_res = sql_string
			.spans
			.binary_search_by_key(&err_pos, |v| v.0.start);

		let err_span = match search_res {
			Ok(pos) => vec![sql_string.spans[pos].1], // Exact matching
			#[rustfmt::skip]
			Err(pos) if pos == 0 => {
				ERROR!(Span::call_site(), format!("Error with undetected position: {:?}", err));
				return None;
			},
			Err(pos) if pos < sql_string.spans.len() => {
				// Since we have not found exact position,
				// try to create span with two neighbour spans.
				vec![sql_string.spans[pos - 1].1, sql_string.spans[pos].1]
			},
			// Here we know the previous value exists and the next does not:
			Err(pos) => vec![sql_string.spans[pos - 1].1],
		};

		let _ = &err_span; // suppress "unusable variable"
		ERROR!(
			err_span,
			format!("Error {:?}: {}", err.code(), err.message()),
		);
		None
	}

	fn finish_query_data(
		sql: TokenStream2,
		query: QueryData,
		ph: Vec<Placeholder>,
		maybe_statement: Option<tokio_postgres::Statement>,
	) -> Result<TokenStream2, TokenStream2> {
		let QueryData {
			query_enum,
			struct_type,
			num_rows,
			row_type,
		} = query;

		// Const-like
		let sync_send = quote!(::std::marker::Send + ::std::marker::Sync);
		let enum_param;
		let uniqueness_check;

		let struct_type_last = struct_type.last().unwrap();
		let query_name = Ident2::new(
			&struct_type_last.to_string(),
			struct_type_last.span().into(),
		);

		match query_enum {
			None => {
				enum_param = quote!();
				uniqueness_check = quote!();
			},
			Some(v) => {
				let _enum_param: syn::TypePath =
					syn::parse(v.into_iter().collect()).map_err(|e| e.into_compile_error())?;

				#[cfg(not(feature = "check-name-uniq"))]
				{
					uniqueness_check = quote!();
				}
				#[cfg(feature = "check-name-uniq")]
				{
					let const_sql_name =
						Ident2::new(&format!("_sql_{}", query_name), query_name.span());
					uniqueness_check = quote! {
						impl #_enum_param {
							// Keep it here to avoid multiple definition
							// (with this row an error will be emitted)!
							#[allow(non_upper_case_globals)]
							pub(crate) const #const_sql_name: &'static str = #sql;
						}
					};
				}
				enum_param = quote!( #_enum_param );
			},
		};


		let mut params_body = vec![];
		let mut params_none = vec![];
		let mut params_list = vec![];

		for placeholder in ph.iter() {
			let Placeholder {
				idnt,
				typ,
				..
			} = placeholder;

			let idnt = Ident2::new(&idnt.to_string(), idnt.span().into());
			let typ = typ
				.as_ref()
				.map_or_else(|| quote!( (dyn ToSql + #sync_send) ), |v| quote!( #v ));

			// Type inferring:
			params_body.push(quote! {
				{ let #idnt: &#typ = #idnt; #idnt }
			});
			params_none.push(quote! {
				::std::boxed::Box::new(none.map(|v| &v.#idnt))
			});
			params_list.push(idnt);
		}

		let num_items = ph.len();

		let trait_name;
		let trait_row_type;
		let ret_row_type;
		match (&num_rows, &row_type) {
			(None, None) => {
				// We will not use it but create fake ident to simplify the code below
				ret_row_type = Ident2::new("fake_type_name", Span::call_site().into());
				trait_row_type = quote!();

				trait_name = quote!(SqlParamsMany);
			},
			(Some(num_rows), Some(ty)) => {
				// ret_row_type = Ident2::new(&ty.to_string(), ty.span().into());
				ret_row_type = Ident2::new(&ty[0].to_string(), ty[0].span().into());
				trait_row_type = quote!( type RowType = #ret_row_type; );

				// TODO: move it to a separate function!
				// TODO: generate enum to be able to detect what to do from
				//       shared code (will be written later)...
				trait_name = match num_rows {
					QueryNumRows::Many => quote!(SqlFetchManyRows),
					QueryNumRows::ZeroOrOne => quote!(SqlFetchMaybeRow),
					QueryNumRows::ExactOne => quote!(SqlFetchOneRow),
					QueryNumRows::Stream => quote!(SqlFetchRowsStream),
					QueryNumRows::Execute => {
						ERROR!(ty[0].span(), "Execute does not return any row result.");
						quote!(SqlExecuteError)
					},
				};
			},
			(Some(QueryNumRows::Execute), None) => {
				ret_row_type = Ident2::new("fake_type_name", Span::call_site().into());
				trait_row_type = quote!();
				trait_name = quote!(SqlExecute);
			},
			_ => {
				// Somehow only one of [num_rows, row_type] (except for `Execute`) is set...
				unimplemented!();
			},
		}

		let struct_type: syn::TypePath =
			syn::parse(struct_type.into_iter().collect()).map_err(|e| e.into_compile_error())?;

		let typ_struct = quote! {
			impl #struct_type
			{
				const SQL_ID: #enum_param = #enum_param::#query_name;
				const SQL: &'static str = #sql;

				#[allow(non_snake_case)]
				fn params(&self) -> [&(dyn ToSql + #sync_send); #num_items]
				{
					let #query_name { #( #params_list, )* } = &self;
					[ #( #params_body, )* ]
				}

				// Useful for EXPLAIN (params are required for explain query).
				fn all_none() -> [::std::boxed::Box<(dyn ToSql + #sync_send)>; #num_items]
				{
					let none: Option<&Self> = None;
					[ #( #params_none, )* ]
				}
			}

			impl #trait_name<#enum_param> for #struct_type
			{
				const SQL_ID: #enum_param = #enum_param::#query_name;
				const SQL: &'static str = #sql;
				#trait_row_type

				fn params(&self) -> ::std::vec::Vec<&(dyn ToSql + #sync_send)>
				{
					#struct_type::params(self).to_vec()
				}

				fn all_none() -> ::std::vec::Vec<::std::boxed::Box<(dyn ToSql + #sync_send)>> {
					#struct_type::all_none().into_iter().collect()
				}
			}
		};

		// If we've successfully "prepared" the query, create a check function
		// for the struct (per-query) of row result.
		let const_check_impl = row_type.and(maybe_statement).map_or(quote!(), |stmnt| {
			Self::gen_check_columns(stmnt, &query_name, ret_row_type)
		});

		Ok(quote!(
			#uniqueness_check
			#typ_struct

			#const_check_impl
		))
	}

	#[allow(unused)] // WIP: temporary disable until `RowFromDB` is returned...
	fn gen_check_columns(
		stmnt: tokio_postgres::Statement,
		query_name: &Ident2,
		ret_row_type: Ident2,
	) -> TokenStream2 {
		let num_columns = stmnt.columns().len();
		let column_idx = 0..num_columns;
		let columns_list: Vec<_> = stmnt.columns().iter().map(|c| c.name()).collect();

		// It is better to use per-query name of function inside "impl `ret_row_type`" to be
		// able to use the same `ret_row_type` in many SQL queries.

		let chk_func_name_str = format!("_check_column_names__{}", query_name);
		let chk_func_name = Ident2::new(&chk_func_name_str, query_name.span());

		// WIP: temporary disable until `RowFromDB` is returned...
		return quote!();

		// Use "quote_spanned" here to avoid error as the whole (possibly huge) SQL query.
		// Limit it the returning type only.
		quote_spanned! {
			ret_row_type.span()=>
			// Columns check:
			const _: [(); #ret_row_type::#chk_func_name()] = [];
			impl #ret_row_type {
				#[allow(non_snake_case)]
				#[allow(non_fmt_panic)]
				const fn #chk_func_name() -> usize {
					const COL_STRUCT: &[&'static str] = &#ret_row_type::COLUMN_NAMES;
					const COL_QUERY: [&'static str; #num_columns] = [
						#( #columns_list, )*
					];

					// Caught from:
					// https://play.rust-lang.org/?gist=1555757f31d8b98f514806c4f94b634
					// ...&version=nightly&edition=2018&mode=debug
					macro_rules! concat_strs {
						($($strs: expr),* $(,)?)=>{{
							const _STRS : &[&str] = &[$($strs),*];
							{
								#[allow(unused)]
								const ADDED_LEN: usize = {
									let mut len = 0;
									for_range!{ i = 0.._STRS.len() =>
										len += _STRS[i].len();
									}
									len
								};
								#[allow(unused)]
								const CONCAT_ARR: [u8; ADDED_LEN] = {
									let mut out = [0u8; ADDED_LEN];
									let mut out_i = 0;
									for_range!{ outer_i = 0.._STRS.len() =>
										let string = _STRS[outer_i].as_bytes();
										for_range!{ inner_i = 0..string.len() =>
											out[out_i] = string[inner_i];
											out_i+=1;
										}
									}
									out
								};

								union Transmute<F: Copy, T: Copy>{
									from: F,
									to: T,
								}

								#[allow(unused)]
								const CONCAT_SLICE: &[u8] = &CONCAT_ARR;
								#[allow(unused)]
								const CONCAT_STR: &str = unsafe{
									Transmute{from: CONCAT_SLICE}.to
								};
								CONCAT_STR
							}
						}}
					}

					macro_rules! for_range{
						( $var:ident = $range:expr => $($for_body:tt)* )=>{{
							let std::ops::Range{mut start,end} = $range;
							while start < end {
								let $var = start;
								$($for_body)*
								start+=1;
							}
						}};
						( $range:expr => $($for_body:tt)* )=>{
							for_range!(start = $range => $($for_body)*)
						}
					}

					macro_rules! check_query_columns {
						(@safe $COL_STRUCT:ident[$idx_col:literal]) => {
							$COL_STRUCT[
								$idx_col
								- ($idx_col as usize)
									.saturating_sub($COL_STRUCT.len().saturating_sub(1))
							]
						};
						( $COL_STRUCT:ident, $COL_QUERY:ident, $idx_col:literal ) => {
							match ($idx_col >= $COL_STRUCT.len(), $idx_col >= $COL_QUERY.len()) {
								(true, true) => {
									// Nothing more; just exit
									return 0;
								},
								(true, false) => {
									panic!(concat_strs!(
										"Extra column from the query: ", $COL_QUERY[$idx_col]
									));
								},
								(false, true) => {
									// unreachable!()  // But try to generate "panic!" instead:
									panic!(concat_strs!(
										"No return column from the query: ",
										check_query_columns!(@safe $COL_STRUCT[$idx_col]),
									));
								},
								(false, false) => {/* nothing to do; continue to check below*/},
							}

							#[allow(unconditional_panic)]  // All checks are above
							let name_s = $COL_STRUCT[$idx_col].as_bytes();
							#[allow(unconditional_panic)]  // All checks are above
							let name_e = $COL_QUERY[$idx_col].as_bytes();

							// Until string comparison is implemented in pure Rust,
							// do it byte-for-byte manually:
							let mut idx: usize = 0;
							loop {
								match (idx >= name_s.len(), idx >= name_e.len()) {
									(true, true) => {
										// Nothing more; just continue with the next column name
										break;
									},
									(false, false) if name_s[idx] == name_e[idx] => {
										idx += 1;
										continue;
									},
									_ => {
										panic!(concat_strs!(
											"Column names are different! In struct: ",
											check_query_columns!(@safe $COL_STRUCT[$idx_col]),
											"; from SQL query: ", $COL_QUERY[$idx_col],
										));
									},
								}
							}
						};
					}

					#( check_query_columns!(COL_STRUCT, COL_QUERY, #column_idx); )*
					if #num_columns < COL_STRUCT.len() {
						panic!(concat_strs!("No required return column from the query: ",
							check_query_columns!(@safe COL_STRUCT[#num_columns]),
							));
					}
					0
				}
			}
		}
	}

	fn add_space_if(&mut self, token_cur: &TokenTree, token_next: Option<&TokenTree>) {
		let token_next = match token_next {
			Some(v) => v,
			None => return, // We do not needed spaces in the end of a stream
		};
		macro_rules! add {
			() => {
				self.sql_string.push_space()
			};
		}
		const fn skip() {} // a semantic "do nothing" to be sure we did not forget something.
		match (token_cur, token_next) {
			// First of all: special cases!

			// In fact it is a lifetime which will be converted to quoted PG's identifier.
			(TokenTree::Ident(_), TokenTree::Punct(p)) if is_quoted_indent(p) => add!(),
			// Punctuation just before placeholder is OK: "SELECT -$1"
			(TokenTree::Punct(_), TokenTree::Group(g)) if is_placeholder(g) => skip(),
			// We convert braces "{}" as a placeholder which must be separated from idents
			(_, TokenTree::Group(g)) if is_placeholder(g) => add!(),

			// Then handle regular cases:
			(TokenTree::Ident(_), TokenTree::Ident(_)) => add!(),
			(TokenTree::Ident(_), TokenTree::Punct(_)) => skip(),
			(TokenTree::Ident(_), TokenTree::Literal(_)) => add!(), // Avoid "SELECT1";

			(TokenTree::Punct(_), TokenTree::Ident(_)) => skip(), // "... =ANY..."
			(TokenTree::Punct(_), TokenTree::Punct(_)) => skip(), // "SELECT + -1" is correct!
			(TokenTree::Punct(_), TokenTree::Literal(_)) => skip(),

			(TokenTree::Literal(_), TokenTree::Punct(_)) => skip(),
			(TokenTree::Literal(_), TokenTree::Ident(_)) => add!(), // Avoid "SELECT 1FROM";

			(TokenTree::Literal(ref a), TokenTree::Literal(ref b)) => {
				let _ = (&a, &b); // suppress "unusable variable"
				ERROR!(
					vec![a.span(), b.span()],
					"Duplication of literals is not allowed",
				);
			},

			// We know there is no sense to insert spaces after any group.
			(TokenTree::Group(_), _) => skip(),
			(_, TokenTree::Group(_)) => skip(),
		};
	}

	pub(crate) fn parse(&mut self, input: TokenStream) -> Result<(), ()> {
		let mut input = input.into_iter().peekable();

		match input.peek() {
			Some(TokenTree::Ident(i)) if i.to_string() == "impl" => self.handle_impl(&mut input)?,
			_ => self.parse_sql(input),
		};
		Ok(())
	}

	#[allow(non_upper_case_globals)]
	fn read_full_ident(
		input: &mut Peekable<proc_macro::token_stream::IntoIter>,
	) -> Result<Vec<TokenTree>, ()> {
		// TODO: try to use `syn` parsing?
		// https://docs.rs/syn/1.0.81/syn/parse/discouraged/trait.Speculative.html
		let mut ret = vec![];
		#[derive(PartialEq)]
		enum Expect {
			Ident,
			Colon1,
			Colon2,
		}
		const WantIdent: Expect = Expect::Ident;
		const WantColon1: Expect = Expect::Colon1;
		const WantColon2: Expect = Expect::Colon2;
		let mut expect = WantIdent;
		loop {
			match input.peek() {
				// TODO: check for the "for" name: it must be error...
				Some(TokenTree::Ident(_)) if expect == WantIdent => {
					expect = WantColon1;
					ret.push(input.next().unwrap());
				},
				Some(TokenTree::Punct(p))
					if expect == WantColon1
						&& p.as_char() == ':' && p.spacing() == Spacing::Joint =>
				{
					expect = WantColon2;
					ret.push(input.next().unwrap());
				}
				Some(TokenTree::Punct(p))
					if expect == WantColon2
						&& p.as_char() == ':' && p.spacing() == Spacing::Alone =>
				{
					expect = WantIdent;
					ret.push(input.next().unwrap());
				}

				_ => break,
			}
		}
		// The `WantColon1` means we've stopped at Ident and have at least one of it:
		if expect != WantColon1 {
			return Err(());
		}
		Ok(ret)
	}

	fn handle_impl(
		&mut self,
		input: &mut Peekable<proc_macro::token_stream::IntoIter>,
	) -> Result<(), ()> {
		const EXPECT_MSG: &str =
			"Expect \"impl [path_to::]ParamsStruct [-> [path_to::]ReturnRowType]\" pattern";

		let ident_1st = input.next().unwrap(); // Also for beginning of a new span.

		assert!(ident_1st.to_string() == "impl"); // It should be checked before.
		let mut end_span = ident_1st.span();
		let num_rows;

		let struct_type = match Self::read_full_ident(input) {
			Ok(name) => {
				// Add a comment to an SQL to be able to see the name in pg_stat_activity:
				let last_part = name.last().unwrap();
				self.sql_string.sql.push_str(&format!("-- {}\n", last_part));

				end_span = last_part.span();
				name
			},
			_ => {
				let _ = &end_span; // suppress "unusable variable"
				ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
				return Err(());
			},
		};

		let row_type = match input.peek() {
			Some(TokenTree::Punct(p)) if "-?*#".contains(p.as_char()) => {
				// - : Exact one row (RetType)
				// ? : At most one row (Option<RetType>)
				// * : Any number of rows (Vec<RetType>)
				// # : Stream of rows (RowStream<RetType>)
				num_rows = Some(p.as_char().into());
				end_span = p.span();
				input.next(); // Consume just checked char.

				match input.next() {
					Some(TokenTree::Punct(p)) if p.as_char() == '>' => {
						end_span = p.span();
					},
					_ => {
						let _ = &end_span; // suppress "unusable variable"
						ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
						return Err(());
					},
				}

				match Self::read_full_ident(input) {
					Err(_) => {
						let _ = &end_span; // suppress "unusable variable"
						ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
						return Err(());
					},
					Ok(v) => Some(v),
				}
			},
			Some(TokenTree::Punct(p)) if p.as_char() == '!' => {
				// ! : Execute (no rows expected) like "SAVEPOINT x"
				num_rows = Some(p.as_char().into());
				// leave row_type as None
				end_span = p.span();
				input.next(); // Consume the char!
				None
			},
			Some(el) => {
				end_span = el.span();
				let _ = &end_span; // suppress "unusable variable"
				ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
				return Err(());
			},
			None => {
				let _ = &end_span; // suppress "unusable variable"
				ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
				return Err(());
			},
		};

		let query_enum = match input.peek() {
			Some(TokenTree::Ident(i)) if i.to_string() == "for" => {
				input.next();
				Self::read_full_ident(input).map(Some).map_err(|e| {
					let _ = &e; // suppress "unusable variable"
					ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
					e
				})?
			},
			_ => None,
		};

		match input.peek() {
			Some(TokenTree::Group(g)) if g.delimiter() == Delimiter::Brace => {
				self.parse_sql(g.stream().into_iter().peekable());
				input.next(); // Consume just checked char.
			},
			Some(g) => {
				let _ = &g; // suppress "unusable variable"
				ERROR!(g.span(), "Expect curly braces for \"impl\" block");
				return Err(());
			},
			_ => {
				let _ = &end_span; // suppress "unusable variable"
				ERROR!(<=>: ident_1st..end_span, EXPECT_MSG);
				return Err(());
			},
		};

		if let Some(el) = input.next() {
			let _ = &el; // suppress "unusable variable"
			ERROR!(
				el.span(),
				"Unsupported extra element after the \"impl\" block",
			);
			return Err(());
		}

		self.query_data = Some(QueryData {
			query_enum,
			struct_type,
			num_rows,
			row_type,
		});

		Ok(())
	}

	fn parse_sql(&mut self, mut input: Peekable<proc_macro::token_stream::IntoIter>) {
		let mut prev_token = None;

		const ASTERISK_MSG: &str = concat!(
			"The asterisk expander (\"*\") is not allowed. ",
			"Use direct list of column names",
		);

		while let Some(token) = input.next() {
			match &token {
				TokenTree::Ident(i) => {
					chk_capitalize(i);
					self.sql_string.push_str(&i.to_string(), i.span());
					self.add_space_if(&token, input.peek());
				},
				TokenTree::Literal(l) => {
					self.sql_string
						.push_str(&literal_fix(&prev_token, l), l.span());
					self.add_space_if(&token, input.peek());
				},
				TokenTree::Punct(p) => {
					match (p.as_char(), input.peek(), &prev_token) {
						('\'', _, _) => {
							// See the comment [1] below (moved to make rustfmt be happy).
							ERROR!(p.span(), "Single quote (lifetime) is not acceptable")
						},
						('$', _, _) => {
							ERROR!(
								p.span(),
								concat!(
									"Neither direct insertion of a placeholder",
									" nor Dollar-Quoted String are not allowed"
								),
							);
						},
						('!', Some(TokenTree::Group(_)), Some(TokenTree::Ident(_))) => {
							// See: RFC: Eager Macro Expansion
							// https://github.com/rust-lang/rfcs/pull/2320
							ERROR!(
								p.span(),
								concat!(
									"It seems like a macro invocation is inside,",
									" but it does not work (see #2320)",
								),
							);
						},
						('*', _, Some(TokenTree::Ident(prv))) if is_reserved_word(prv) => {
							ERROR!(p.span(), ASTERISK_MSG)
						},
						(',', Some(TokenTree::Punct(p)), _) if p.as_char() == '*' => {
							ERROR!(p.span(), ASTERISK_MSG)
						},
						('.', Some(TokenTree::Punct(p)), _) if p.as_char() == '*' => {
							ERROR!(p.span(), ASTERISK_MSG)
							// We use second almost-all block (except matching char ','/".")
							// because or-pattern syntax `',' | '.'` is not stabilized yet (#54883)
							// and rustfmt result is weird if we try to use stable "or" block.
						},
						('\\', _, _) => {
							ERROR!(p.span(), "Backslash is not allowed")
						},
						(';', _, _) => {
							ERROR!(p.span(), "Multiquery is not allowed")
						},
						_ => {},
					}
					// Do not add the last comma before the end of a stream.
					// It is usually the final one in forms like "ARRAY[1,2,]" or "ROW(1,2)"
					// The SQL standard does not allow this, but for better GIT DIFFs we can allow
					// it in Rust source and remove when we generate SQL for PG.
					match (p.as_char(), p.spacing(), input.peek()) {
						(',', _, None) => { /* Do nothing */ },
						(',', _, Some(TokenTree::Punct(p2))) if p2.as_char() == ',' => {
							ERROR!(
								vec![p.span(), p2.span()],
								"Duplication of commas is an error",
							);
						},
						(',', _, Some(TokenTree::Ident(i))) if can_skip_comma(i) => {
							// Skip coma but add space if necessary between the prev and the next:
							if let Some(prev) = &prev_token {
								self.add_space_if(prev, input.peek());
							}
						},
						(_, Spacing::Alone, _) => {
							self.sql_string.push_ch(p.as_char(), p.span());
							self.add_space_if(&token, input.peek());
						},
						(_, Spacing::Joint, _) => self.sql_string.push_ch(p.as_char(), p.span()),
					}

					// [1] (see above; moved to make rustfmt be happy)
					// Note: "Additionally, single quote ' can join with identifiers
					//              to form lifetimes 'ident"
					// I.e. it is not starting of a quoting (quotes are already parsed and received
					// from the TokenStream as TokenTree::Punct).
					// https://doc.rust-lang.org/stable/proc_macro/enum.Spacing.html#variant.Joint
				},
				TokenTree::Group(g) => {
					match g.delimiter() {
						Delimiter::Parenthesis => {
							self.sql_string.push_ch_groups('(');
							self.parse_sql(g.stream().into_iter().peekable());
							self.sql_string.push_ch_groups(')');
						},
						Delimiter::Bracket => {
							self.sql_string.push_ch_groups('[');
							self.parse_sql(g.stream().into_iter().peekable());
							self.sql_string.push_ch_groups(']');
						},
						Delimiter::None => {
							ERROR!(g.span(), "Unsupported implicit delimiter is found")
						},
						Delimiter::Brace => {
							const MSG: &str =
								"Only single ident (optionally typed) is allowed here";
							// PG (SQL) does not have braces in its grammar.
							// The only allowed internal token is a single Ident which is replaced
							// with "$n" where "n" is a sequence of its unique entry
							let mut inner = g.stream().into_iter();
							let inner_1st = inner.next();
							match (&inner_1st, inner.next()) {
								// At first check there is only single ident (untyped arg):
								(Some(TokenTree::Ident(i)), None) => {
									self.handle_placeholder(i, None);
								},

								// Then check whether it is ExprType (typed arg):
								// The only case when it is "Single ident then its type" is
								// when there is Ident, then Punct which is not Joined.
								(Some(TokenTree::Ident(i)), Some(TokenTree::Punct(p)))
									if p.as_char() == ':' && p.spacing() == Spacing::Alone =>
								{
									let row_type = inner.collect();
									// Type can be tricky. Do not check internals until we
									// understand it is necessary.
									match parse_macro_input::parse::<Type>(row_type) {
										Err(e) => return ERROR!(g.span(), e.to_string(), MSG),
										Ok(ty) => {
											self.handle_placeholder(i, Some(ty.into()));
										},
									}
								}
								_ => {
									ERROR!(g.span(), MSG)
								},
							}
						},
					}
					// We know there is no sense to insert spaces after any group.
					// DO NOT call "self.add_space_if(token, input.peek());"
				},
			};
			prev_token = Some(token);
		}
	}

	fn handle_placeholder(&mut self, idnt: &Ident, ty: Option<Box<Type>>) {
		let name = idnt.to_string();
		let pos_data = self.placeholders.iter_mut().find(|ph| ph.name == name);

		let pos_str = match pos_data {
			Some(ph) => {
				// If we've find another entry of the same name which has a type:
				match (&ph.typ, ty) {
					// 1. Replace untyped with typed
					(None, Some(ty_inn)) => {
						ph.typ.replace(ty_inn);
					},

					// 2. Raise error if types in both entries are different
					(Some(typ_inn), Some(ty_inn)) if typ_inn != &ty_inn => {
						ERROR!(
							vec![ty_inn.span().unwrap(), typ_inn.span().unwrap()],
							"Params with the same names have different types",
						);
					},

					// 3. Otherwise (types are equal or new is untyped) leave "as is"
					_ => {},
				}
				&ph.pos
			},
			None => {
				// If we have not found the param name, insert it there.
				let len = self.placeholders.len() + 1;
				self.placeholders.push(Placeholder {
					name,
					idnt: idnt.clone(),
					pos: len.to_string(),
					typ: ty,
				});
				&self.placeholders.last().unwrap().pos
			},
		};

		let ph = format!("${}", pos_str);
		self.sql_string.push_str(&ph, idnt.span());
	}
}


fn can_skip_comma(ident: &Ident) -> bool {
	// We can not rely on PG_RESERVED because we fail in case "SELECT 1, ARRAY[1]"
	let upper = ident.to_string().to_uppercase();
	PG_SELECT_KEWYORD.binary_search(&&upper[..]).is_ok()
}


fn is_reserved_word(token: &Ident) -> bool {
	let upper = token.to_string().to_uppercase();
	if PG_RESERVED.binary_search(&&upper[..]).is_err() {
		// Just in case check for "IS" and "BY": it allows to insist
		// on uppercased "IS NOT NULL" & "GROUP BY":
		if !["BY", "IS"].contains(&&upper[..]) {
			return false;
		}
	}
	true
}

#[rustfmt::skip]  // Skip long URL inside.
fn chk_capitalize(token: &Ident) {
	if !is_reserved_word(token) {
		return;
	}

	let word = token.to_string();
	let upper = word.to_uppercase();
	// If reserved and already uppercased, do not emit warning:
	if upper == token.to_string() {
		return;
	}

	// WIP: sorry, I could not find a way to emit something like
	// Span.suggested_replacement
	// https://github.com/rust-lang/rls/blob/master/rls/src/actions/diagnostics.rs#L177
	// https://doc.rust-lang.org/nightly/nightly-rustc/rustc_errors/diagnostic/struct.Diagnostic.html#method.span_suggestion
	ERROR!(
		Warning => token.span(),
		format!("capitalize keyword: {} => {}", word, upper),
	);
}

// This list is as of PostgreSQL-9.6 by the query:
// SELECT upper(word) FROM pg_get_keywords() WHERE catcode='R' ORDER BY 1
const PG_RESERVED: &[&str] = &[
	"ALL",
	"ANALYSE",
	"ANALYZE",
	"AND",
	"ANY",
	"ARRAY",
	"AS",
	"ASC",
	"ASYMMETRIC",
	"BOTH",
	"CASE",
	"CAST",
	"CHECK",
	"COLLATE",
	"COLUMN",
	"CONSTRAINT",
	"CREATE",
	"CURRENT_CATALOG",
	"CURRENT_DATE",
	"CURRENT_ROLE",
	"CURRENT_TIME",
	"CURRENT_TIMESTAMP",
	"CURRENT_USER",
	"DEFAULT",
	"DEFERRABLE",
	"DESC",
	"DISTINCT",
	"DO",
	"ELSE",
	"END",
	"EXCEPT",
	"FALSE",
	"FETCH",
	"FOR",
	"FOREIGN",
	"FROM",
	"GRANT",
	"GROUP",
	"HAVING",
	"IN",
	"INITIALLY",
	"INTERSECT",
	"INTO",
	"LATERAL",
	"LEADING",
	"LIMIT",
	"LOCALTIME",
	"LOCALTIMESTAMP",
	"NOT",
	"NULL",
	"OFFSET",
	"ON",
	"ONLY",
	"OR",
	"ORDER",
	"PLACING",
	"PRIMARY",
	"REFERENCES",
	"RETURNING",
	"SELECT",
	"SESSION_USER",
	"SOME",
	"SYMMETRIC",
	"TABLE",
	"THEN",
	"TO",
	"TRAILING",
	"TRUE",
	"UNION",
	"UNIQUE",
	"USER",
	"USING",
	"VARIADIC",
	"WHEN",
	"WHERE",
	"WINDOW",
	"WITH",
];

// First words in grammar before which "[, ...]" is present and not inside parenthesis.
// https://www.postgresql.org/docs/13/sql-select.html
const PG_SELECT_KEWYORD: &[&str] = &[
	// NOTE: keep the list sorted!
	"FETCH",     // after "ORDER BY"
	"FOR",       // after "ORDER BY"
	"FROM",      // after "SELECT"
	"HAVING",    // after "GROUP"
	"LIMIT",     // after "ORDER BY"
	"NOWAIT",    // after "OF table_name"
	"OFFSET",    // after "ORDER BY"
	"ORDER",     // after "WINDOW"
	"RETURNING", // after all (at the end)
	"SKIP",      // after "OF table_name"
	"WHERE",     // after "FROM"
	"WINDOW",    // after "GROUP BY"
];

// Helper functions to make the main code be shorter
fn is_placeholder(grp: &Group) -> bool {
	grp.delimiter() == Delimiter::Brace
}

fn is_quoted_indent(pct: &Punct) -> bool {
	pct.as_char() == '\''
}

fn literal_fix(prev_token: &Option<TokenTree>, literal: &Literal) -> String {
	// WIP: it can be weird because Rust accepts (and emits) literals like "2.4f64"
	// It seems (as of now) we can only check by generating debug string and check
	// for the "suffix: None," there.

	// TODO: We can try to use string literal suffixes as a marker of PG ident escaping
	// (e.g. 'SELECT abc FROM "CaseSensitivitySchma"I."CaseSensitivityTable"I'), but
	// keep in mind there are several checks in the `parse_inner` for the next token is "Ident".
	// We need a separate function to check for that (and custom pushing to SqlString).

	// We reconstruct it from existing parsed literal to have access to parsed "value" and "suffix"
	// fields which are not accessible in the std::proc_macro::Literal.
	let lit = syn::parse_str::<Lit>(&literal.to_string()).unwrap();

	if !lit.suffix().is_empty() {
		ERROR!(literal.span(), "Suffixed literal is not supported");
	}

	// There is PG's possibility to use syntax E'...' to be able to use custom chars via
	// symbol escaping (like "\n"). Since we already have ability to use escaping via Rust's
	// syntax system. Therefore we forbid this syntax.
	// We get Ident token as a single "E" char followed by Literal (string).
	match (prev_token, &lit) {
		(Some(TokenTree::Ident(i)), Lit::Str(s)) if i.to_string() == "E" => {
			let _ = &s; // suppress "unusable variable"
			ERROR!(
				vec![i.span(), s.span().unwrap()],
				"PG's syntax E\"...\" is forbidden, use Rust's escaping syntax",
			)
		},
		_ => {},
	}

	const NULLCHR_MSG: &str = "Null byte is not supported in PG's TEXT literal";
	match &lit {
		// A UTF-8 string literal: "foo".
		Lit::Str(v) => {
			// Here should be already parsed data (like slashes and 0xNN), just check there is no
			// 0x00 (unsupported TEXT char in PostgreSQL) and single quote with another one.

			// We even do not need to escape resulting '\' for PG! Check it: SELECT '12\'::text
			// Without the "E" letter before the string (like `SELECT E'a\x23b'` => 'a#b') which
			// can not be parsed by Rust it is just PG's literal which required to escape only
			// single quotes by just duplicate them. Do not bother other chars there!

			let v = v.value().replace('\'', "''");
			// P.S.: But keep in mind the 'null' char (0x00) still not acceptable for TEXT type!
			if v.contains('\x00') {
				ERROR!(literal.span(), NULLCHR_MSG);
			}
			format!("'{}'", v)
		},

		// A byte string literal: b"foo".
		Lit::ByteStr(_) => {
			ERROR!(
				literal.span(),
				"ByteString literal (b\"foo\") is not supported",
			);
			literal.to_string()
		},

		// A byte literal: b'f'.
		Lit::Byte(_) => {
			ERROR!(literal.span(), "Byte literal (b'f') is not supported");
			literal.to_string()
		},

		// A character literal: 'a'.
		Lit::Char(v) => {
			// The same comment as for Lit::Str above
			match v.value() {
				'\'' => "''''".to_string(), // manually handle this simple case
				// P.S.: But keep in mind the 'null' char (0x00) still not acceptable for TEXT type!
				'\x00' => {
					ERROR!(literal.span(), NULLCHR_MSG);
					literal.to_string()
				},
				_ => format!("'{}'", v.value()),
			}
		},

		// An integer literal: 1 or 1u16.
		Lit::Int(_) => {
			// TODO: check input for onr of: [0x8000, 0x8000_0000, -0x8000_0000_0000_0000]
			// these values (being negatives: -32768 must be wrapped: "(-32768)::int2), otherwise
			// it is an error.
			literal.to_string()
		},

		// A floating point literal: 1f64 or 1.0e10f64. Must be finite. May not be infinite or NaN.
		Lit::Float(_) => literal.to_string(),

		// A boolean literal: true or false.
		Lit::Bool(_) => {
			// Must not happen (we handle it as an Ident), but leave it just here in case.
			ERROR!(literal.span(), "Lit::Bool is not supported");
			literal.to_string().to_uppercase()
		},

		// A raw token literal not interpreted by Syn.
		Lit::Verbatim(_) => {
			ERROR!(literal.span(), "Literal can not be parsed");
			literal.to_string().to_uppercase()
		},
	}
}
