#![cfg_attr(
	feature = "pretty-errors",
	feature(proc_macro_diagnostic, proc_macro_span)
)]

//! This crate provides the [`pg_sql!`] macro for writing SQL queries (for [PostgreSQL]) in
//! Rust syntax (not Rust code), i.e. you can use convenient indents, comments, Rust's
//! string escaping etc. without putting the query into a single string and get some
//! syntax highlighting (functions call, capitalized reserved words, ...).
//!
//! [PostgreSQL]: https://www.postgresql.org/
//!
//! On build the macro creates an SQL string with minimum spaces and replace identifiers in curly
//! braces "SELECT {arg1}::bigint" into sequenced argument number ("$1", "$2", etc.). If there are
//! several entries of the same named argument, the same numbered argument is used.
//!
//! There are two modes: the first is just returns a minificated SQL-string, the second generates
//! additional code to be used with a query arguments struct (fields must match names and types in
//! named arguments in the SQL query) and expected return row numbers defined as a specific trait
//! (to be polished and documented).
//!
//! You can find examples in the [`pg_sql!`] description.
//!
//! ```toml
//! [dependencies]
//! compact_sql = "0.0.1"
//! ```
//!
//! # Features:
//!
//! | Feature | Description |
//! | ------- | ----------- |
//! | pretty-errors | (requires nightly Rust) gives better errors pointing to a concrete span in the SQL query text. |
//!
//! # Constraints:
//! There are several artificial constraints:
//! 1. expanded asterisks (`SELECT * FROM...`), direct numbered arguments are forbidden;
//! 1. DBMS objects (schema/table/view/index/etc. names) can not be quoted
//! (like `SELECT ... FROM "CaseSensitiveTable" ...`).
//!
//! Instead you can use leading commas (which are forbidden by the SQL standard) which are not
//! collected in resulting SQL query.
//!
//! # Test query in real DBMS
//!
//! Your queries can be tested (via "prepare" feature of the PG's protocol, i.e. does not run any query at all;
//! totally safe) in a real DBMS if you set environment variables for `cargo`:
//! * `TEST_PG_HOST`
//! * `TEST_PG_PORT` (optional; default: 5432)
//! * `TEST_PG_USER` (optional; default: "postgres")
//! * `TEST_PG_PASS` (optional; default: "")
//! * `TEST_PG_DATB`
//!
//! It allows to detect wrong SQL for your DB schema. E.g.:
//!
//! ```shell
//! TEST_PG_HOST=127.0.0.1 TEST_PG_DATB=test_db cargo check
//! ```


#[macro_use]
extern crate quote;

use proc_macro::TokenStream;

mod sql;

/// The whole point
///
/// # Examples:
///
/// ## Generate just SQL string
/// ```
/// # use compact_sql::pg_sql;
/// #
/// # fn main() {
/// let sql = pg_sql! {
///     SELECT
///         relname,
///         relnamespace,
///         format("%I", oid),
///     FROM
///         pg_catalog.pg_class
///     WHERE
///         oid = {var_oid}::oid
/// };
/// assert_eq!(
///     sql,
///     "SELECT relname,relnamespace,format('%I',oid)FROM pg_catalog.pg_class WHERE oid=$1::oid"
/// );
/// # }
/// ```
#[proc_macro]
pub fn pg_sql(input: TokenStream) -> TokenStream {
	let mut ret = sql::SqlHandler::new();
	match ret.parse(input) {
		Ok(()) => ret.finish(),
		Err(()) => TokenStream::default(),
	}
}
