use super::*;
use crate::*;
use std::{
    cell::UnsafeCell,
    future::Future,
    pin::Pin,
    sync::atomic::{AtomicI32, Ordering},
    task::{Context, Poll, Waker},
};

pub struct Cond {
    mutx: Autex,
    head: UnsafeCell<Box<ListHead>>,
}
make!(Cond: Send, Sync, Default);

impl Cond {
    pub fn new() -> Self {
        let mut headp = Box::new(ListHead::new());
        headp.init_list_head();
        Cond {
            mutx: Autex::new(),
            head: UnsafeCell::new(headp),
        }
    }

    pub async fn notify_one(&self) {
        let (ent, _) = unsafe {
            let headp = self.head.get();
            let g = self.mutx.lock().await;
            if (*headp).list_empty() {
                return;
            }

            let ent = (*headp).next;
            (*ent).list_del();
            (ent, g)
        };

        let wn = container_of!(ent, WakeNode, entry);
        wn.wake();
    }

    pub async fn notify_all(&self) {
        let g = self.mutx.lock().await;
        InitListHead!(head);
        unsafe {
            let headp = self.head.get();
            (*headp).list_add(&mut head);
            (*headp).list_del_init();
        }
        drop(g);

        while !head.list_empty() {
            let ent = head.next;
            unsafe {
                (*ent).list_del();
            }
            let wn = container_of!(ent, WakeNode, entry);
            wn.wake();
        }
    }

    pub async fn wait<'a, 'b>(&'a self, g1: Guard<'b>) -> Guard<'b> {
        let autx = g1.0;
        let mut wn = WakeNode::new(context.await);

        let g2 = self.mutx.lock().await;
        unsafe {
            let headp = self.head.get();
            (*headp).list_add_tail(&mut wn.entry);
        }
        drop(g2);
        drop(g1);

        struct Futwn<'a>(&'a AtomicI32);
        impl<'a> Future for Futwn<'a> {
            type Output = ();
            fn poll(self: Pin<&mut Self>, _: &mut Context<'_>) -> Poll<Self::Output> {
                let waked = self.0.load(Ordering::Relaxed);
                if waked == 0 {
                    return Poll::Pending;
                }
                Poll::Ready(())
            }
        }
        Futwn(&wn.waked).await;

        autx.lock().await
    }
}

struct WakeNode {
    waked: AtomicI32,
    entry: ListHead,
    waker: Waker,
}

unsafe impl Sync for WakeNode {}
unsafe impl Send for WakeNode {}

impl WakeNode {
    fn new(waker: Waker) -> Self {
        WakeNode {
            waked: AtomicI32::new(0),
            entry: ListHead::new(),
            waker: waker,
        }
    }

    fn wake(&self) {
        self.waked.store(1, Ordering::Relaxed);
        self.waker.wake_by_ref();
    }
}
