use super::*;
use crate::{
    atomic::{AtomicN, AtomicP},
    *,
};
use std::{
    cell::UnsafeCell,
    mem::forget,
    ops::Fn,
    sync::{
        atomic::{compiler_fence, Ordering},
        Arc,
    },
};

struct CondVar<T> {
    cv: Cond,
    uptr: UnsafeCell<Option<Arc<T>>>,
}
make!(CondVar<T>: Sync);

impl<T> CondVar<T> {
    fn new() -> Self {
        CondVar {
            cv: Cond::new(),
            uptr: UnsafeCell::new(None),
        }
    }

    async fn wait<'a, 'b>(&'a self, mut g: Guard<'b>) -> Arc<T> {
        let x = || unsafe { &mut *self.uptr.get() };
        let uptr = x();
        while uptr.is_none() {
            g = self.cv.wait(g).await;
        }

        match uptr {
            Some(ref uptr) => uptr.clone(),
            None => unreachable!(),
        }
    }

    async fn wake(&self, any: Arc<T>) {
        let x = || {
            let uptr = unsafe { &mut *self.uptr.get() };
            *uptr = Some(any);
        };
        x();
        self.cv.notify_all().await;
    }
}

pub enum R<'a, T> {
    G(Guard<'a>),
    V(Arc<T>),
}

pub struct LeadLock<T> {
    nr: i32,
    mux: Autex,
    mux_cond: Autex,
    cond: *const CondVar<T>,
    holder: *const CondVar<T>,
}
make!(LeadLock<T>: Sync, Send, Default);

impl<T> LeadLock<T> {
    pub fn new() -> Self {
        let cond = Arc::new(CondVar::new());
        LeadLock {
            nr: 0,
            cond: Arc::into_raw(cond),
            holder: nil!(),
            mux: Autex::new(),
            mux_cond: Autex::new(),
        }
    }

    pub async fn lock(&self) -> R<'_, T> {
        let mut cond: sb<*const CondVar<T>>;
        let n = self.nr.atomic_fetch_add(1, Ordering::Relaxed);
        if n != 0 {
            let g = self.mux_cond.lock().await;
            cond = sb(self.cond.atomic_load(Ordering::Relaxed));
            compiler_fence(Ordering::Release);

            if self.nr.atomic_fetch_add(1, Ordering::Relaxed) != 0 {
                let cond = unsafe {
                    let cond = Arc::from_raw(cond.0);
                    let c = cond.clone();
                    forget(cond);
                    c
                };
                return R::V(cond.wait(g).await);
            }
        }

        let g = self.mux.lock().await;
        self.nr.atomic_store(0, Ordering::Relaxed);
        compiler_fence(Ordering::Release);

        cond = sb(self
            .cond
            .atomic_swap(Arc::into_raw(Arc::new(CondVar::new())), Ordering::Relaxed));
        self.holder.atomic_store(cond.0, Ordering::Relaxed);
        R::G(g)
    }

    /// # Safety
    /// the function is unsafe because g must be the one returned from lock()
    pub async unsafe fn unlock(&self, g: Guard<'_>, uptr: Arc<T>) {
        let holder = sb(self.holder.atomic_swap(nil!(), Ordering::Relaxed));
        if holder.0 == nil!() {
            return;
        }

        drop(g);
        // no new follower can reach holder
        drop(self.mux_cond.lock().await);
        let holder = Arc::from_raw(holder.0);
        holder.wake(uptr).await;
    }

    pub async fn single_flight<F: Future<Output = Arc<T>>>(&self, f: F) -> Arc<T> {
        match self.lock().await {
            R::G(g) => {
                let any = f.await;
                unsafe { self.unlock(g, any.clone()).await };
                any
            }
            R::V(any) => any,
        }
    }
}
