use std::fmt::Display;

#[macro_export]
macro_rules! suffx {
    ($($token:tt)*) => {
        $crate::_suffix!({$($token)*});
    };
}

#[macro_export]
macro_rules! nil {
    () => {
        std::ptr::null_mut::<u8>().cast()
    };
}

#[macro_export]
macro_rules! cptr {
    ($ptr:expr) => {
        ($ptr) as *const _
    };
}

#[macro_export]
macro_rules! mptr {
    ($ptr:expr) => {
        cptr!($ptr) as *mut _
    };
}

#[macro_export]
macro_rules! offset_of {
    ($type:ty, $member:ident) => {{
        let addr = ::std::mem::align_of::<$type>();
        std::ptr::addr_of!((*(addr as *const $type)).$member) as *const _ as usize - addr
    }};
}

#[macro_export]
macro_rules! container_of {
    ($ptr:expr, $type:ty, $member:ident) => {{
        unsafe {
            let addr = $ptr as *const _ as usize;
            let offset = $crate::offset_of!($type, $member);
            &*((addr - offset) as *const $type)
        }
    }};
}

#[derive(Debug)]
pub struct CallPos {
    pub line: u32,
    pub file: &'static str,
}

impl Display for CallPos {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "({}:{})", self.file, self.line)
    }
}

#[macro_export]
macro_rules! callpos {
    () => {{
        static mut POS: $crate::CallPos = $crate::CallPos {
            line: line!(),
            file: file!(),
        };
        &mut POS
    }};
}
