use crate::*;
use std::{
    mem::transmute,
    ptr::addr_of_mut,
    sync::atomic::{AtomicPtr, Ordering},
};

#[repr(C)]
#[derive(Debug)]
pub struct LkfNode(AtomicPtr<LkfNode>, AtomicPtr<CallPos>);

impl LkfNode {
    #[inline]
    pub fn new() -> LkfNode {
        LkfNode(AtomicPtr::new(nil!()), AtomicPtr::new(nil!()))
    }

    #[inline]
    pub unsafe fn next(&mut self) -> *mut LkfNode {
        let ptr = self.0.load(Ordering::Relaxed);
        let next = (*ptr).0.load(Ordering::Relaxed);
        if ptr == nil!() || next == nil!() {
            return nil!();
        }

        if ptr != self {
            self.0 = AtomicPtr::new(next);
        }
        (*ptr).0 = AtomicPtr::new(nil!());
        (*ptr).1 = AtomicPtr::new(nil!());
        ptr
    }
}

impl Drop for LkfNode {
    fn drop(&mut self) {
        let pos = self.1.load(Ordering::Relaxed);
        if pos != nil!() {
            let pos: &CallPos = unsafe { &*pos };
            panic!("still linked: {}", pos);
        }
    }
}

pub struct Lkf {
    root: LkfNode,
    tail: AtomicPtr<AtomicPtr<LkfNode>>,
}
make!(Lkf: Sync, Send, Default);

impl Lkf {
    #[inline]
    pub fn new() -> Lkf {
        Lkf {
            root: LkfNode::new(),
            tail: AtomicPtr::new(nil!()),
        }
    }

    /// # Safety
    /// This fn is unsafe because it does not care about the lifetime of `node`.
    /// Caller must unlink `node` before its lifetime ends.
    #[inline]
    pub unsafe fn put(
        &self,
        node: *mut LkfNode,
        pos: *mut CallPos,
    ) -> Result<(), &'static CallPos> {
        let x = (*node)
            .1
            .compare_exchange(nil!(), pos, Ordering::Relaxed, Ordering::Relaxed);
        if let Err(pos) = x {
            return Err(&*pos);
        }

        let mut next = addr_of_mut!((*node).0);
        next = self.tail.swap(next, Ordering::Relaxed);
        if next == nil!() {
            self.root.0.store(node, Ordering::Relaxed);
        } else {
            (*next).store(node, Ordering::Relaxed);
        }
        Ok(())
    }

    #[inline]
    pub fn get(&self) -> *mut LkfNode {
        let node = self.root.0.swap(nil!(), Ordering::Relaxed);
        if node == nil!() {
            return node;
        }

        let last = self.tail.swap(nil!(), Ordering::Relaxed);
        unsafe {
            (*last).store(node, Ordering::Relaxed);
        }
        mptr!(last)
    }
}

#[macro_export]
macro_rules! lkf_put_unsafe {
    ($list:expr, $node:expr) => {
        unsafe { ($list).put($node, callpos!()) }
    };
}

#[macro_export]
macro_rules! lkf_get {
    ($list:expr) => {
        ($list).get()
    };
}

#[macro_export]
macro_rules! lkf_next_unsafe {
    ($node:expr) => {
        unsafe { (*$node).next() }
    };
}
