use std::path::PathBuf;
use std::fs::File;
use std::io::{prelude::*, BufReader};
use structopt::StructOpt;
use encoding_rs_io::DecodeReaderBytesBuilder;

/// A CLI tool that comments out parts of a file.
#[derive(StructOpt, Debug)]
#[structopt(name = "comment-by")]
struct Opt {

    /// Appended comment character
    #[structopt(short = "cc", long = "comment-character", default_value  = "--")]
    comment_char: String,

    /// String used to split the file
    #[structopt(short, long, default_value = "/****** Object:")]
    split_by: String,
    
    /// String used to split the file
    #[structopt(short = "i", long, default_value = "test")]
    comment_if: String,

    /// Output file, stdout if not present
    #[structopt(parse(from_os_str))]
    output: PathBuf,

    /// Files to process
    #[structopt(name = "FILE", parse(from_os_str))]
    input: PathBuf,
}

#[derive(Debug)]
struct Chunk {
    comment_out: bool,
    lines: Vec<String>
}

impl Chunk {
    fn new() -> Chunk {
        Chunk {
            comment_out: false,
            lines: Vec::new()
        }
    }

    fn process_lines(self, comment_char: &str) -> Vec<String> {
        let mut lines = self.lines;
        if self.comment_out {
            lines = lines.into_iter().map(|line| format!("{} {}", comment_char, line)).collect()
        };
        lines.iter().map(|line| format!("{}\n", line)).collect()
    }
}

fn main() {
    let opt = Opt::from_args();

    let file_string = opt.input.to_str().expect("could not parse input file");
    let output_file = opt.output.to_str().expect("could not parse output file");
    comment_chunks(
        &file_string,
        &output_file,
        &opt.split_by,
        &opt.comment_if,
        &opt.comment_char
    ).expect("Could not create commented file");
}

fn comment_chunks(
    input_file: &str,
    output_file: &str,
    split_by: &str,
    comment_if: &str,
    comment_char: &str
) -> Result<(), std::io::Error> {
    let file = File::open(input_file)?;
    let reader = BufReader::new(DecodeReaderBytesBuilder::new().build(file));

    let mut file_out = File::create(output_file)?;
    let mut chunk = Chunk::new();
    
    for (i, line) in reader.lines().enumerate() {
        let new_line = line?.clone();
        if new_line.contains(&split_by) && i > 0 {
            for chunk_line in chunk.process_lines(comment_char) {
                file_out.write_all(chunk_line.as_bytes())?;
            }
            chunk = Chunk::new();
        }
        if new_line.contains(&comment_if) {
            chunk.comment_out = true;
        };
        chunk.lines.push(new_line)
    };
    for chunk_line in chunk.process_lines(comment_char) {
        file_out.write_all(chunk_line.as_bytes())?;
    };

    Ok(())
}

