use std::error::Error;
use std::fs;
use std::env;

pub struct Config {
    pub pattern: String,
    pub filename: String,
    pub case_sensitive: bool,
}

impl Config {
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        args.next();

        let pattern = match args.next() {
            Some(p) => p,
            None => return Err("Lack of pattern argument"),
        };
        let filename = match args.next() {
            Some(f) => f,
            None => return Err("Lack of filename argument"),
        };
        let case_sensitive = env::var("CASE_INSENSITIVE").is_err();

        Ok(Config {
            pattern,
            filename,
            case_sensitive,
        })
    }
}

pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let contents = fs::read_to_string(&config.filename)?;
    println!("File contents:\n {}", contents);

    println!("Search results:");
    let results: Vec<&str>;
    if config.case_sensitive {
        results = search(&config.pattern, &contents);
    }
    else {
        results = search_case_insensitive(&config.pattern, &contents);
    }
    for line in results {
        println!("{}", line);
    }
    Ok(())
}

fn search<'a>(pattern: &str, contents: &'a str) -> Vec<&'a str> {
    contents.lines().filter(|line| line.contains(pattern)).collect()
}

fn search_case_insensitive<'a>(pattern: &str, contents: &'a str) -> Vec<&'a str> {
    let lowercase_pattern = pattern.to_lowercase();
    contents.lines().filter(|line| line.to_lowercase().contains(&lowercase_pattern)).collect()
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn one_result() {
        let pattern = "rea";
        let contents = "How dreary to be somebody!
        How public, like a frog
        To tell your name the livelong day";
        let results = search(pattern, contents);
        assert!(results.len() == 1 && results.get(0).unwrap().contains(pattern));
    }

    #[test]
    fn case_sensitive() {
        let pattern = "To";
        let contents = "How dreary to be somebody!
How public, like a frog
To tell your name the livelong day";
        let results = search(pattern, contents);
        assert_eq!(vec!["To tell your name the livelong day"], results);
    }

    #[test]
    fn case_insensitive() {
        let pattern = "lIkE";
        let contents = "How dreary to be somebody!
How public, like a frog
To tell your name the livelong day";
        let results = search_case_insensitive(pattern, contents);
        assert_eq!(vec!["How public, like a frog"], results);
    }
}