/*
 * Copyright (C) 2020-2021 David Sugar <tychosoft@gmail.com>.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

const VERSION: &'static str = env!("CARGO_PKG_VERSION");
const AUTHORS: &'static str = env!("CARGO_PKG_AUTHORS");

#[macro_use]
extern crate clap;

use os_display::Quotable;
use std::fs;
use std::path::Path;

struct App {
    delim: String,
    quote: String,
    total: u32,
    basename: bool,
    recursive: bool,
    directory: bool
}

fn output(arg: &str, app: &mut App) {
    let mut path = Path::new(arg);

    if app.basename {
        path = match path.file_name() {
            Some(part) => Path::new(part),
            None => path,
        };
    }

    if app.total > 0 {
        print!("{}", app.delim);
    }
    app.total += 1;
    print!("{}{}{}", app.quote, path.maybe_quote(), app.quote);
}

fn exam(path: &str, count: u32, app: &mut App) {
    let meta = fs::metadata(&path);
    if !meta.is_ok() {
        output(path, app);
        return;
    }

    let inode = meta.unwrap();
    if !inode.is_dir() {
        output(path, app);
        return;
    }

    if !app.basename || (count > 0 && !app.recursive) {
        output(path, app);
    }

    if (!app.directory && !app.recursive) || (count > 0 && !app.recursive) {
        return;
    }

    let dir = fs::read_dir(path);
    if dir.is_err() {
        return;
    }

    let paths = dir.unwrap();
    let names: Vec<String> = paths.map(|entry| {
        let entry = entry.unwrap();
        let entry_path = entry.path();
        let entry_name = entry_path.file_name().unwrap();
        let file_name = entry_name.to_str().unwrap();
        return String::from(file_name);
    }).collect();
    for name in names {
        exam(&(path.to_string() + "/" + &name), count + 1, app);
    }
}

fn main() {
    // This I do really like...
    let args = clap_app!(paths =>
        (version: VERSION)
        (author: AUTHORS)
        (about: "Expands and echos path arguments")
        (@arg DIR: -D --directory "Expand directory into files")
        (@arg RECURSE: -R --recursive "Recursive directory scan")
        (@arg BASENAME: -b --basename "Reduce paths to filename")
        (@arg DELIM: -d --delim +takes_value "Deliminator between arguments")
        (@arg LINE: -l --line "Separate arguments with newline")
        (@arg QUOTE: -q --quote +takes_value "Quote for each argument")
        (@arg PATH: +multiple)
    ).get_matches();

    // Define application context and flags...
    let mut app = App{
        delim: match args.is_present("LINE") {
            true => args.value_of("DELIM").unwrap_or("\n").to_string(),
            false => args.value_of("DELIM").unwrap_or(" ").to_string(),
        },
        quote: args.value_of("QUOTE").unwrap_or("").to_string(),
        basename: args.is_present("BASENAME"),
        directory: args.is_present("DIR"),
        recursive: args.is_present("RECURSE"),
        total: 0,
    };

    // Evaluate path arguments...
    let paths = args.values_of("PATH");
    if paths.is_some() {
        for path in paths.unwrap() {
            exam(path, 0, &mut app);
        }
        println!();
    }
}
