/*
 * Copyright (C) 2020-2021 David Sugar <tychosoft@gmail.com>.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

const VERSION: &'static str = env!("CARGO_PKG_VERSION");
const AUTHORS: &'static str = env!("CARGO_PKG_AUTHORS");

#[macro_use]
extern crate clap;

use std::fs;
use std::io;
use std::process;
use std::path::Path;
use std::ffi::OsStr;
use os_display::Quotable;

enum Algorithm {
    SHA256,
    SHA512,
    SHA1,
    RIPEMD160,
    MD5
}

struct App {
    recursive: bool,
    algorithm: Algorithm,
}

fn compute<R: io::Read>(algorithm: &Algorithm, mut from: R, name: &OsStr)
{
    let mut buffer: [u8; 4096] = [0; 4096];

    match algorithm {
        Algorithm::SHA256 => {
            use sha2::{Digest, Sha256};
            let mut digest = Sha256::new();
            loop {
                let count: usize = match from.read(&mut buffer) {
                    Ok(value) => value,
                    Err(_) => 0,
                };
                if count < 1 {
                    break;
                };
                digest.update(&buffer[0 .. count]);
            }
            println!("{:x} {}", digest.finalize(), name.maybe_quote());
        },
        Algorithm::SHA512 => {
            use sha2::{Digest, Sha512};
            let mut digest = Sha512::new();
            loop {
                let count: usize = match from.read(&mut buffer) {
                    Ok(value) => value,
                    Err(_) => 0,
                };
                if count < 1 {
                    break;
                };
                digest.update(&buffer[0 .. count]);
            }
            println!("{:x} {}", digest.finalize(), name.maybe_quote());
        },
        Algorithm::SHA1 => {
            use sha1::Sha1;
            let mut digest = Sha1::new();
            loop {
                let count: usize = match from.read(&mut buffer) {
                    Ok(value) => value,
                    Err(_) => 0,
                };
                if count < 1 {
                    break;
                };
                digest.update(&buffer[0 .. count]);
            }
            println!("{} {}", digest.digest().to_string(), name.maybe_quote());
        },
        Algorithm::RIPEMD160 => {
            use ripemd160::{Digest, Ripemd160};
            let mut digest = Ripemd160::new();
            loop {
                let count: usize = match from.read(&mut buffer) {
                    Ok(value) => value,
                    Err(_) => 0,
                };
                if count < 1 {
                    break;
                };
                digest.update(&buffer[0 .. count]);
            }
            println!("{:x} {}", digest.finalize(), name.maybe_quote());
        },
        Algorithm::MD5 => {
            use md5::{Digest, Md5};
            let mut digest = Md5::new();
            loop {
                let count: usize = match from.read(&mut buffer) {
                    Ok(value) => value,
                    Err(_) => 0,
                };
                if count < 1 {
                    break;
                };
                digest.update(&buffer[0 .. count]);
            }
            println!("{:x} {}", digest.finalize(), name.maybe_quote());
        },
    }
}

fn access(app: &App, name: &str)
{
    let path = Path::new(name);
    compute(&app.algorithm,
        match fs::File::open(path) {
            Err(err) => {
                eprintln!("*** fdsum: {}:{}", path.display(), err);
                process::exit(2);
            },
            Ok(file) => file,
        },
        match path.file_name() {
            Some(part) => part,
            None => path.as_os_str(),
        });
}

fn scan(path: &str, count: u32, app: &App) {
    let meta = fs::metadata(&path);
    if !meta.is_ok() {
        eprintln!("*** fdsum: {}: not found", path);
        process::exit(3);
    }

    let inode = meta.unwrap();
    if !inode.is_dir() {
        access(app, path);
        return;
    }

    if (count > 0) && !app.recursive {
        return;
    }

    let dir = fs::read_dir(path);
    if dir.is_err() {
        return;
    }

    let paths = dir.unwrap();
    let names: Vec<String> = paths.map(|entry| {
        let entry = entry.unwrap();
        let entry_path = entry.path();
        let entry_name = entry_path.file_name().unwrap();
        let file_name = entry_name.to_str().unwrap();
        return String::from(file_name);
    }).collect();
    for name in names {
        if name.chars().next().unwrap() != '.' {
            scan(&(path.to_string() + "/" + &name), count + 1, app);
        }
    }
}


fn main() {
    // This I do really like...
    let args = clap_app!(mdsum =>
        (version: VERSION)
        (author: AUTHORS)
        (about: "Compute digest for files")
        (@arg RECURSE: -R --recursive "Recursive directory scan")
        (@arg HASH: -a --algo +takes_value "Digest algorithm")
        (@arg PATH: +multiple)
    ).get_matches();

    // Define application context and flags...
    let app = App{
        recursive: args.is_present("RECURSE"),
        algorithm: match args.value_of("HASH").unwrap_or("sha1") {
            "ripemd160" => Algorithm::RIPEMD160,
            "sha512" => Algorithm::SHA512,
            "sha256" => Algorithm::SHA256,
            "sha2" => Algorithm::SHA256,
            "sha1" => Algorithm::SHA1,
            "md5" => Algorithm::MD5,
            _ => {
                eprintln!("*** fdsum: unknown algorithm {}", args.value_of("HASH").unwrap());
                process::exit(1)
            }
        },
    };

    // Evaluate path arguments...
    let paths = args.values_of("PATH");

    // if empty path list, compute for stdin
    if !paths.is_some() {
        compute(&app.algorithm, io::stdin().lock(), OsStr::new("-"));
        return;
    }

    // scan arguments...
    for path in paths.unwrap() {
        scan(path, 0, &app)
    }
}
