use std::error::Error;
use std::io::{self, BufRead};
use rargsxd::*;

const OFF: &'static str = "\x1b[m";

/// Trait with a set of functions of colouring a string.
pub trait StyledString {
    /// Every other function uses this to access the string itself.
    fn get_str(&self) -> String;

    fn do_off(&self) -> &str {
        if self.get_str().ends_with(OFF) {""}
        else {OFF}
    }

    fn remove_off(&self) -> String {
        let mut cur = self.get_str();
        while let Some(new) = cur.strip_suffix(OFF) {
            cur = new.to_string();
        }
        cur
    }

    fn fgblack(&self) -> String     {format!("\x1b[30m{}{}", self.get_str(), self.do_off())}
    fn fgred(&self) -> String       {format!("\x1b[31m{}{}", self.get_str(), self.do_off())}
    fn fggreen(&self) -> String     {format!("\x1b[32m{}{}", self.get_str(), self.do_off())}
    fn fgyellow(&self) -> String    {format!("\x1b[33m{}{}", self.get_str(), self.do_off())}
    fn fgblue(&self) -> String      {format!("\x1b[34m{}{}", self.get_str(), self.do_off())}
    fn fgmagenta(&self) -> String   {format!("\x1b[35m{}{}", self.get_str(), self.do_off())}
    fn fgcyan(&self) -> String      {format!("\x1b[36m{}{}", self.get_str(), self.do_off())}
    fn fgwhite(&self) -> String     {format!("\x1b[37m{}{}", self.get_str(), self.do_off())}

    fn bgblack(&self) -> String     {format!("\x1b[40m{}{}", self.get_str(), self.do_off())}
    fn bgred(&self) -> String       {format!("\x1b[41m{}{}", self.get_str(), self.do_off())}
    fn bggreen(&self) -> String     {format!("\x1b[42m{}{}", self.get_str(), self.do_off())}
    fn bgyellow(&self) -> String    {format!("\x1b[43m{}{}", self.get_str(), self.do_off())}
    fn bgblue(&self) -> String      {format!("\x1b[44m{}{}", self.get_str(), self.do_off())}
    fn bgmagenta(&self) -> String   {format!("\x1b[45m{}{}", self.get_str(), self.do_off())}
    fn bgcyan(&self) -> String      {format!("\x1b[46m{}{}", self.get_str(), self.do_off())}
    fn bgwhite(&self) -> String     {format!("\x1b[47m{}{}", self.get_str(), self.do_off())}

    fn rgb(&self, rgb: &str) -> String    {format!("\x1b[38;2;{}m{}{}", rgb, self.get_str(), self.do_off())}

    fn bold(&self) -> String        {format!("\x1b[1m{}{}", self.get_str(), self.do_off())}
    fn italics(&self) -> String     {format!("\x1b[3m{}{}", self.get_str(), self.do_off())}
    fn underline(&self) -> String   {format!("\x1b[4m{}{}", self.get_str(), self.do_off())}
    fn strike(&self) -> String      {format!("\x1b[9m{}{}", self.get_str(), self.do_off())}
}

impl StyledString for String {
    fn get_str(&self) -> String {
        self.clone()
    }
}

impl StyledString for &str {
    fn get_str(&self) -> String {
        self.to_string()
    }
}

pub fn strip_styles(input: String) -> String {
    let mut output = String::new();
    let mut skip = false;
    for ch in input.chars() {
        if ch == '\x1b' {
            skip = true;
            continue;
        } else if ch == 'm' && skip {
            skip = false;
            continue;
        } else if skip {
            continue;
        } else {
            output.push(ch);
        }
    }
    output
}

pub fn run(args: ArgParser) -> Result<(), Box<dyn Error>> {
    let mut modifier = String::new();

    if args.get_bool("bold")          {modifier = modifier.bold()}
    if args.get_bool("italics")       {modifier = modifier.italics()}
    if args.get_bool("underline")     {modifier = modifier.underline()}
    if args.get_bool("strike")      {modifier = modifier.strike()}

    if args.get_str("fg") != "" {
        modifier = match args.get_str("fg").as_str() {
            "black"     => modifier.fgblack(),
            "red"       => modifier.fgred(),
            "green"     => modifier.fggreen(),
            "yellow"    => modifier.fgyellow(),
            "blue"      => modifier.fgblue(),
            "magenta"   => modifier.fgmagenta(),
            "cyan"      => modifier.fgcyan(),
            "white"     => modifier.fgwhite(),
            _ => "".to_string(),
        }
    }

    if args.get_str("bg") != "" {
        modifier = match args.get_str("bg").as_str() {
            "black"     => modifier.bgblack(),
            "red"       => modifier.bgred(),
            "green"     => modifier.bggreen(),
            "yellow"    => modifier.bgyellow(),
            "blue"      => modifier.bgblue(),
            "magenta"   => modifier.bgmagenta(),
            "cyan"      => modifier.bgcyan(),
            "white"     => modifier.bgwhite(),
            _ => "".to_string(),
        }
    }

    if args.get_str("rgb") != "" {modifier = modifier.rgb(&args.get_str("rgb"))}

    modifier = modifier.remove_off();
    if args.extra.len() > 0 {
        let string = args.extra.join(" ");
        println!("{}{}", modifier, string);
    } else {
        let stdin = io::stdin();
        // Need to do this each line so that things like ping work
        for line in stdin.lock().lines() {
            let mut string = line? + "\n";
            if !args.get_bool("dontforce") {
                string = strip_styles(string);
            }
            print!("{}{}", modifier, string);
        }
    }
    Ok(())
}
