use std::collections::HashMap;
use std::io::{self, BufRead};
use rargsxd::*;

fn strip_styles(input: String) -> String {
    let mut output = String::new();
    let mut skip = false;
    for ch in input.chars() {
        if ch == '\x1b' {
            skip = true;
            continue;
        } else if ch == 'm' && skip {
            skip = false;
            continue;
        } else if skip {
            continue;
        } else {
            output.push(ch);
        }
    }
    output
}

fn main() {
    let mut parser = ArgParser::from_argv0();
    parser.author("BubbyRoosh")
        .info("stdin colourizer")
        .version("0.1.0")
        .args(vec!(
                Arg::bool("dontforce", false)
                    .short('d')
                    .help("Prevents colourizer from overriding the current input's styles"),

                Arg::bool("bold", false)
                    .short('b')
                    .help("Boldens the output"),
                Arg::bool("italics", false)
                    .short('i')
                    .help("Italicises the output"),
                Arg::bool("underline", false)
                    .short('u')
                    .help("Underlines the output"),
                Arg::bool("strikethrough", false)
                    .short('s')
                    .help("Strikes through the output"),

                Arg::str("fg", "")
                    .short('c')
                    .help("Sets the foreground colour for the output"),
                Arg::str("bg", "")
                    .short('g') // b is already taken for **bold**
                    .help("Sets the background colour for the output"),

                Arg::str("rgb", "")
                    .help("Sets the colour for the output according to \"r;g;b\""),

        ))
        .parse();

    // Moment
    let mut fgs = HashMap::new();
    fgs.insert("black".to_string(),     "\x1b[30m");
    fgs.insert("red".to_string(),       "\x1b[31m");
    fgs.insert("green".to_string(),     "\x1b[32m");
    fgs.insert("yellow".to_string(),    "\x1b[33m");
    fgs.insert("blue".to_string(),      "\x1b[34m");
    fgs.insert("magenta".to_string(),   "\x1b[35m");
    fgs.insert("cyan".to_string(),      "\x1b[36m");
    fgs.insert("white".to_string(),     "\x1b[37m");

    let mut bgs = HashMap::new();
    bgs.insert("black".to_string(),     "\x1b[40m");
    bgs.insert("red".to_string(),       "\x1b[41m");
    bgs.insert("green".to_string(),     "\x1b[42m");
    bgs.insert("yellow".to_string(),    "\x1b[43m");
    bgs.insert("blue".to_string(),      "\x1b[44m");
    bgs.insert("magenta".to_string(),   "\x1b[45m");
    bgs.insert("cyan".to_string(),      "\x1b[46m");
    bgs.insert("white".to_string(),     "\x1b[47m");

    let mut modifier = String::new();

    if parser.get_bool("bold")          {modifier += "\x1b[1m"}
    if parser.get_bool("italics")       {modifier += "\x1b[3m"}
    if parser.get_bool("underline")     {modifier += "\x1b[4m"}
    if parser.get_bool("strikethrough") {modifier += "\x1b[9m"}

    if parser.get_str("fg") != "" {
        if let Some(code) = fgs.get(&parser.get_str("fg")) {modifier += code}
    }
    if parser.get_str("bg") != "" {
        if let Some(code) = bgs.get(&parser.get_str("bg")) {modifier += code}
    }

    if parser.get_str("rgb") != "" {modifier += &format!("\x1b[38;2;{}m", parser.get_str("rgb"))}

    if parser.extra.len() > 0 {
        let string = parser.extra.join(" ");
        println!("{}{}", modifier, string);
    } else {
        let stdin = io::stdin();
        // Need to do this each line so that things like ping work
        for line in stdin.lock().lines() {
            let mut string = line.unwrap() + "\n";
            if !parser.get_bool("dontforce") {
                string = strip_styles(string);
            }
            print!("{}{}", modifier, string);
        }
    }
}
