#![deny(missing_docs)]

/*!
This crate defines an infrastructure to handle collision detection.

It's based on the `collide` crate, so your colliders need need to implement the `Collider` trait.

The `CollisionManager`, which can only contain one type of collider.
When you want to use multiple colliders, you currently need to use trait objects or enums.

Then you can add bodies and colliders to the manager. You should store their indices, which will be used when you modify or remove them.
When you're done, you can compute the collisions, which you can use to update your objects.
Before computing the collisions again, you should update your bodies and colliders.
**/

use array_linked_list::ArrayLinkedList;
use collide::{Collider, CollisionInfo};
use vector_space::VectorSpace;

/// An object which might consist of multiple colliders.
/// Collisions between multiple colliders of the same body are not checked.
pub struct Body<C: Collider> {
    /// All the colliders of the body.
    pub colliders: Vec<C>,
}

impl<C: Collider> Body<C> {
    /// Creates a new body.
    pub fn new() -> Self {
        Self {
            colliders: Vec::new(),
        }
    }

    /// Adds a collider to the body.
    pub fn with_collider(mut self, collider: C) -> Self {
        self.colliders.push(collider);
        self
    }
}

/// Represents the index of a static or dynamic inserted object.
#[derive(Copy, Clone)]
pub enum ObjectIndex<S, D> {
    /// The index of the static object represented by the collider.
    Static(S),
    /// The index of the dynamic object represented by the body.
    Dynamic(D),
}

/// The collision info with the index of the collider or body.
pub struct IndexedCollisionInfo<V: VectorSpace, S, D> {
    /// Index of the collider or body.
    pub index: ObjectIndex<S, D>,
    /// The actual collision info.
    pub info: CollisionInfo<V>,
}

struct Static<C: Collider, I> {
    collider: C,
    index: I,
}

struct Dynamic<C: Collider, I> {
    body: Body<C>,
    index: I,
}

/// The collision manager, which manages collisions.
/// Can contain multiple bodies and other colliders.
/// Collisions between colliders are not checked.
pub struct CollisionManager<C: Collider, S, D> {
    bodies: ArrayLinkedList<Dynamic<C, D>>,
    colliders: ArrayLinkedList<Static<C, S>>,
}

impl<C: Collider, S: Copy, D: Copy> CollisionManager<C, S, D> {
    /// Creates a new collision manager.
    pub fn new() -> Self {
        Self {
            bodies: ArrayLinkedList::new(),
            colliders: ArrayLinkedList::new(),
        }
    }

    /// Creates a new collision manager with the capacity of expected bodies specified.
    pub fn with_capacity(capacity: usize) -> Self {
        Self {
            bodies: ArrayLinkedList::with_capacity(capacity),
            colliders: ArrayLinkedList::new(),
        }
    }

    /// Inserts a new collider and returns its index.
    /// An object index needs to be specified.
    pub fn insert_collider(&mut self, collider: C, index: S) -> usize {
        self.colliders.push_back(Static { collider, index })
    }

    /// Inserts a new body and returns its index.
    /// An object index needs to be specified.
    pub fn insert_body(&mut self, body: Body<C>, index: D) -> usize {
        self.bodies.push_back(Dynamic { body, index })
    }

    /// Replaces the existing collider at the specified index by a new collider.
    pub fn replace_collider(&mut self, index: usize, collider: C) {
        self.colliders[index].as_mut().unwrap().collider = collider;
    }

    /// Replaces the existing body at the specified index by a new body.
    pub fn replace_body(&mut self, index: usize, body: Body<C>) {
        self.bodies[index].as_mut().unwrap().body = body;
    }

    /// Removes an existing collider, so the index is usable for a new collect again.
    pub fn remove_collider(&mut self, index: usize) {
        self.colliders.remove(index);
    }

    /// Removes an existing body, so the index is usable for a new body again.
    pub fn remove_body(&mut self, index: usize) {
        self.bodies.remove(index);
    }

    /// Get the collider at the sepcified index.
    pub fn collider(&self, index: usize) -> &C {
        &self.colliders[index].as_ref().unwrap().collider
    }

    /// Get the collider at the sepcified index as mutable.
    pub fn collider_mut(&mut self, index: usize) -> &mut C {
        &mut self.colliders[index].as_mut().unwrap().collider
    }

    /// Get the body at the sepcified index.
    pub fn body(&self, index: usize) -> &Body<C> {
        &self.bodies[index].as_ref().unwrap().body
    }

    /// Get the body at the sepcified index as mutable.
    pub fn body_mut(&mut self, index: usize) -> &mut Body<C> {
        &mut self.bodies[index].as_mut().unwrap().body
    }

    /// Checks for a static collision with collider and returns the index.
    pub fn find_static_collision(&self, collider: &C) -> Option<S> {
        for collider_static in &self.colliders {
            if collider.check_collision(&collider_static.collider) {
                return Some(collider_static.index);
            }
        }
        None
    }

    /// Checks for a dynamic collision with collider and returns the index.
    pub fn find_dynamic_collision(&self, collider: &C) -> Option<D> {
        for body_dynamic in &self.bodies {
            for dynamic_collider in &body_dynamic.body.colliders {
                if collider.check_collision(dynamic_collider) {
                    return Some(body_dynamic.index);
                }
            }
        }
        None
    }

    /// Checks for any collision with collider and returns the index.
    pub fn find_collision(&self, collider: &C) -> Option<ObjectIndex<S, D>> {
        if let Some(index) = self.find_dynamic_collision(collider) {
            Some(ObjectIndex::Dynamic(index))
        } else if let Some(index) = self.find_static_collision(collider) {
            Some(ObjectIndex::Static(index))
        } else {
            None
        }
    }

    /// Computes the collisions between all colliders of a body with all other colliders.
    /// Returns a list for each, which can be indexed by the specified index.
    pub fn compute_collisions(
        &self,
    ) -> ArrayLinkedList<Vec<IndexedCollisionInfo<C::Vector, S, D>>> {
        let mut result = ArrayLinkedList::with_capacity(self.bodies.capacity());
        for body_index in self.bodies.indices() {
            result.replace_front(body_index, Vec::new());
        }
        for (body_index, body_dynamic) in self.bodies.indexed() {
            let infos = result[body_index].as_mut().unwrap();
            // check static collisions
            for collider_static in &self.colliders {
                for body_collider in &body_dynamic.body.colliders {
                    if let Some(info) = body_collider.collision_info(&collider_static.collider) {
                        infos.push(IndexedCollisionInfo {
                            index: ObjectIndex::Static(collider_static.index),
                            info,
                        });
                    }
                }
            }
            let infos = infos as *mut _;
            // check dynamic collisions
            for (other_index, other_dynamic) in self.bodies.indexed_after(body_index) {
                let other_infos = result[other_index].as_mut().unwrap();
                for dynamic_collider in &other_dynamic.body.colliders {
                    for body_collider in &body_dynamic.body.colliders {
                        if let Some(info) = body_collider.collision_info(dynamic_collider) {
                            other_infos.push(IndexedCollisionInfo {
                                index: ObjectIndex::Dynamic(body_dynamic.index),
                                info: -info,
                            });
                            unsafe {
                                std::mem::transmute::<
                                    _,
                                    &mut Vec<IndexedCollisionInfo<C::Vector, S, D>>,
                                >(infos)
                            }
                            .push(IndexedCollisionInfo {
                                index: ObjectIndex::Dynamic(other_dynamic.index),
                                info,
                            });
                        }
                    }
                }
            }
        }
        result
    }
}
