//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
//

use std::fmt::Debug;
use std::hash::Hash;
use std::iter::FromIterator;
use std::iter::IntoIterator;
use std::collections::HashMap;

#[derive(Debug, thiserror::Error)]
pub enum Error<K>
    where K: Debug,
{
    #[error("duplicated key in iterator: {:?}", _0)]
    DuplicatedKey(K),

    #[error("unknown error")]
    Unknown,
}

pub struct CollectOnceHashMap<K, V, S = std::collections::hash_map::RandomState>
    where K: Eq + Hash + Debug
{
    inner: std::result::Result<HashMap<K, V, S>, Error<K>>
}

impl<K, V, S> CollectOnceHashMap<K, V, S>
    where K: Eq + Hash + Debug
{
    pub fn into_inner(self) -> std::result::Result<HashMap<K, V, S>, Error<K>> {
        self.inner
    }
}

impl<K, V> FromIterator<(K, V)> for CollectOnceHashMap<K, V, std::collections::hash_map::RandomState>
    where K: Eq + Hash + Debug
{
    fn from_iter<T>(iter: T) -> Self
        where T: IntoIterator<Item = (K, V)>
    {
        let iter = iter.into_iter();
        let iter_size = {
            let iter_size = iter.size_hint();
            if let Some(s) = iter_size.1 {
                s
            } else {
                iter_size.0
            }
        };

        let mut hm = HashMap::with_capacity(iter_size);

        for (element_key, element_value) in iter.into_iter() {
            if hm.contains_key(&element_key) {
                return CollectOnceHashMap {
                    inner: Err(Error::DuplicatedKey(element_key))
                }
            } else {
                hm.insert(element_key, element_value);
            }
        }

        CollectOnceHashMap {
            inner: Ok(hm)
        }
    }
}


#[cfg(test)]
mod tests {
    use super::CollectOnceHashMap;
    use super::Error;

    #[test]
    fn test_empty() {
        let hm = vec![]
            .into_iter()
            .collect::<CollectOnceHashMap<u8, u8>>()
            .into_inner()
            .unwrap();

        assert_eq!(0, hm.keys().count());
    }

    #[test]
    fn test_simple() {
        let hm = vec![(1, 1)]
            .into_iter()
            .collect::<CollectOnceHashMap<u8, u8>>()
            .into_inner()
            .unwrap();

        assert_eq!(1, hm.keys().count());
        assert!(hm.contains_key(&1));
        assert!(hm.keys().all(|k| *k == 1));
        assert!(hm.values().all(|k| *k == 1));
    }

    #[test]
    fn test_duplicate() {
        let hm = vec![(1, 1), (1, 2)]
            .into_iter()
            .collect::<CollectOnceHashMap<u8, u8>>()
            .into_inner();

        assert!(hm.is_err());
        assert!(std::matches!(hm, Err(Error::DuplicatedKey(1))));
    }
}

