use std::cmp::Ord;
use std::collections::BTreeMap;
use std::fmt::Debug;
use std::hash::Hash;
use std::iter::FromIterator;
use std::iter::IntoIterator;

#[derive(Debug, thiserror::Error)]
pub enum Error<K>
    where K: Debug,
{
    #[error("duplicated key in iterator: {:?}", _0)]
    DuplicatedKey(K),

    #[error("unknown error")]
    Unknown,
}

pub struct CollectOnceBTreeMap<K, V>
    where K: Eq + Ord + Hash + Debug
{
    inner: std::result::Result<BTreeMap<K, V>, Error<K>>
}

impl<K, V> CollectOnceBTreeMap<K, V>
    where K: Eq + Ord + Hash + Debug
{
    pub fn into_inner(self) -> std::result::Result<BTreeMap<K, V>, Error<K>> {
        self.inner
    }
}

impl<K, V> FromIterator<(K, V)> for CollectOnceBTreeMap<K, V>
    where K: Eq + Ord + Hash + Debug
{
    fn from_iter<T>(iter: T) -> Self
        where T: IntoIterator<Item = (K, V)>
    {
        let mut hm = BTreeMap::new();

        for (element_key, element_value) in iter.into_iter() {
            if hm.contains_key(&element_key) {
                return CollectOnceBTreeMap {
                    inner: Err(Error::DuplicatedKey(element_key))
                }
            } else {
                hm.insert(element_key, element_value);
            }
        }

        CollectOnceBTreeMap {
            inner: Ok(hm)
        }
    }
}


#[cfg(test)]
mod tests {
    use super::CollectOnceBTreeMap;
    use super::Error;

    #[test]
    fn test_empty() {
        let hm = vec![]
            .into_iter()
            .collect::<CollectOnceBTreeMap<u8, u8>>()
            .into_inner()
            .unwrap();

        assert_eq!(0, hm.keys().count());
    }

    #[test]
    fn test_simple() {
        let hm = vec![(1, 1)]
            .into_iter()
            .collect::<CollectOnceBTreeMap<u8, u8>>()
            .into_inner()
            .unwrap();

        assert_eq!(1, hm.keys().count());
        assert!(hm.contains_key(&1));
        assert!(hm.keys().all(|k| *k == 1));
        assert!(hm.values().all(|k| *k == 1));
    }

    #[test]
    fn test_duplicate() {
        let hm = vec![(1, 1), (1, 2)]
            .into_iter()
            .collect::<CollectOnceBTreeMap<u8, u8>>()
            .into_inner();

        assert!(hm.is_err());
        assert!(std::matches!(hm, Err(Error::DuplicatedKey(1))));
    }
}

