use std::fmt::{Display, Formatter};

/// Responsible for buffering code.
#[derive(Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct CodeBuffer {
    indent: String,
    line_ending: String,
    code: String,
}

impl CodeBuffer {
    //! Constructors

    /// Creates a new code buffer.
    pub fn new(indent: &str, line_ending: &str, buffer_size: usize) -> Self {
        Self {
            indent: indent.to_string(),
            line_ending: line_ending.to_string(),
            code: String::with_capacity(buffer_size),
        }
    }
}

impl Default for CodeBuffer {
    fn default() -> Self {
        Self::new("\t", "\n", 16 * 1024)
    }
}

impl CodeBuffer {
    //! Write

    /// Writes the string.
    pub fn write(&mut self, s: &str) {
        self.code.push_str(s);
    }

    /// Writes the indent level.
    pub fn indent(&mut self, level: usize) {
        for _ in 0..level {
            self.code.push_str(&self.indent);
        }
    }

    /// Writes a line-ending.
    pub fn end_line(&mut self) {
        self.code.push_str(&self.line_ending);
    }

    /// Writes a single space.
    pub fn space(&mut self) {
        self.code.push_str(" ");
    }
}

impl CodeBuffer {
    //! Export

    /// Peeks at the buffered code.
    pub fn peek(&self) -> &str {
        self.code.as_str()
    }

    /// Exports the buffered code.
    pub fn export(self) -> String {
        self.code
    }
}

impl Display for CodeBuffer {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.code)
    }
}
