/// Trait for any integer type that can be converted to a number of octets in network byte order
/// for CoAP.
///
/// These are used to back the [crate::MinimalWritableMessage::add_option_uint] functions.
///
/// Implemented as a sealed trait over u8, u16, u32 and u64 to be later moved to
/// num_traits::PrimInt if <https://github.com/rust-num/num-traits/issues/189> works out, or a
/// similar generic trait.
pub trait Ux: sealed::Sealed + Copy + Clone {
    /// Index is used to get a slice out of it, Default and IndexMut in value_uint. Given it's
    /// sealed, this can easily be extended if the add_option_uint or value_uint implementations
    /// chanage.
    type Bytes: AsRef<[u8]> + Default + AsMut<[u8]>;

    fn to_be_bytes(self) -> Self::Bytes;
    fn from_be_bytes(bytes: Self::Bytes) -> Self;
}

mod sealed {
    pub trait Sealed {}

    impl Sealed for u8 {}
    impl Sealed for u16 {}
    impl Sealed for u32 {}
    impl Sealed for u64 {}
}

impl Ux for u8 {
    type Bytes = [u8; 1];

    fn to_be_bytes(self) -> Self::Bytes {
        self.to_be_bytes()
    }

    fn from_be_bytes(bytes: Self::Bytes) -> Self {
        Self::from_be_bytes(bytes)
    }
}

impl Ux for u16 {
    type Bytes = [u8; 2];

    fn to_be_bytes(self) -> Self::Bytes {
        self.to_be_bytes()
    }

    fn from_be_bytes(bytes: Self::Bytes) -> Self {
        Self::from_be_bytes(bytes)
    }
}

impl Ux for u32 {
    type Bytes = [u8; 4];

    fn to_be_bytes(self) -> Self::Bytes {
        self.to_be_bytes()
    }

    fn from_be_bytes(bytes: Self::Bytes) -> Self {
        Self::from_be_bytes(bytes)
    }
}

impl Ux for u64 {
    type Bytes = [u8; 8];

    fn to_be_bytes(self) -> Self::Bytes {
        self.to_be_bytes()
    }

    fn from_be_bytes(bytes: Self::Bytes) -> Self {
        Self::from_be_bytes(bytes)
    }
}
