/// A message code
/// A CoAP code
///
/// For the code's meaning, see the [CoAP Codes IANA subregistry] or the [coap_numbers::code]
/// module that lists them.
///
/// [CoAP Codes IANA subregistry]: https://www.iana.org/assignments/core-parameters/core-parameters.xhtml#codes
/// [coap_numbers::code]: https://docs.rs/coap-numbers/0.1.2/coap_numbers/code/
///
/// All code implementations can be converted to u8 (as they correspond to an 8-bit unsigned
/// integer in the CoAP protocol). The conversion back is fallible to allow strict message
/// implementations to only accept codes supported by the implementation, and/or to limit the
/// values by type state. For example, a strict implementation may distinguish between request and
/// response messages, and only allow the respective code class into their objects (simultaneously
/// creating usable niches in the in-memory representations).
// It'd be convenient to demand TryFrom<u8, Error: core::fmt::Debug>, but RFC 2289 hasn't landed in
// stable yet
pub trait Code: Into<u8> + core::convert::TryFrom<u8> {}

impl Code for u8 {}

/// A CoAP option number
///
/// For its meaning, see the [CoAP Option Numbers IANA subregistry] or the
/// [coap_numbers::option] module that lists them, and provides helpers for decoding their
/// properties.
///
/// [CoAP Option Numbers IANA subregistry]: https://www.iana.org/assignments/core-parameters/core-parameters.xhtml#option-numbers
/// [coap_numbers::option]: https://docs.rs/coap-numbers/0.1.2/coap_numbers/option/
///
/// In analogy to [Code], these are convertible to [u16] and fallibly convertible from there.
pub trait OptionNumber: Into<u16> + core::convert::TryFrom<u16> {}

impl OptionNumber for u16 {}

/// Experimental trait for conversion between option representation, with the goal to later avoid
/// the conversion step via u16 and specialize for T->T conversions.
pub trait FromOtherOption<O: OptionNumber> {
    type Error;

    fn convert_from_other(other: O) -> Result<Self, Self::Error>
    where
        Self: Sized;
}

impl<O: OptionNumber, T: OptionNumber> FromOtherOption<O> for T {
    type Error = <Self as core::convert::TryFrom<u16>>::Error;

    /* default */
    fn convert_from_other(other: O) -> Result<Self, Self::Error>
    where
        Self: Sized,
    {
        let num: u16 = other.into();
        use core::convert::TryInto;
        num.try_into()
    }
}

/*

needs
#![feature(specialization)]
and the above default to be active

impl<T: OptionNumber> FromOtherOption<T> for T
{
//     type Error = <Self as core::convert::TryFrom<u16>>::Error; // lont term I'd like to have `!` here
//
    fn convert_from_other(other: T) -> Result<Self, Self::Error>
    where
        Self: Sized
    {
        Ok(other)
    }
}
*/
