use anyhow::{Context, Result};
use clap::{App, Arg};

mod cli_args {
    pub const HOOKS: &str = "hooks";
    pub const COMMAND: &str = "command";
}

fn main() -> Result<()> {
    let matches = App::new(clap::crate_name!())
        .author(clap::crate_authors!())
        .name(clap::crate_name!())
        .version(clap::crate_version!())
        .about(clap::crate_description!())
        .setting(clap::AppSettings::ArgRequiredElseHelp)
        .setting(clap::AppSettings::TrailingVarArg)
        .arg(
            Arg::new(cli_args::HOOKS)
                .long(cli_args::HOOKS)
                .help("Print/install shell hooks for automating calls to cnf")
                .value_name("SHELL")
                .possible_values(["bash", "zsh", "help"])
                .default_missing_value("help"),
        )
        .arg(
            Arg::new(cli_args::COMMAND)
                .required(false)
                .multiple_values(true),
        )
        .get_matches();

    if matches.is_present(cli_args::HOOKS) {
        match matches.value_of(cli_args::HOOKS) {
            Some("help") => {
                println!(
                    "
                == Installing Shell Hooks ==

cnf on its own isn't nearly half as useful as it could be.
To unleash its full potential, you will likely want to install
hooks for your current shell. This will make sure cnf is
always called when your shell doesn't find a command, and
passes the command along with all its arguments to cnf instead.

To get started, run this command again, giving it either 'bash'
or 'zsh' as argument, depending on your current shell."
                );
            }
            Some("bash") => println!(
                "
Add the following snippet to the bottom of your `~/.bashrc`:

    function command_not_found_handle {{
        if command -v cnf 1>/dev/null; then
            cnf \"$@\"
        fi
    }}"
            ),
            Some("zsh") => println!(
                "
Add the following snippet to the bottom of your `~/.zshrc`:

    function command_not_found_handler {{
        if command -v cnf 1>/dev/null; then
            cnf \"$@\"
        fi
    }}"
            ),
            _ => unreachable!(),
        };
        // end here
        return Ok(());
    };

    let args: Vec<&str> = matches.values_of(cli_args::COMMAND).unwrap().collect();
    cnf::run(&args).context("Something terrible happened...".to_string())
}
