//! # Toolbx Environment Handler.
//!
//! This module handles command-not-found errors that occur while executing inside a Toolbx
//! container. Currently, this means that commands are forwarded to the host using `flatpak-spawn`.
//! If `flatpak-spawn` isn't present, an error is thrown instead.
use crate::config;
use crate::error::CnfError;
use std::{path::Path, process::Command};

const TOOLBX_ENV: &str = "/run/.toolboxenv";
const FLATPAK_SPAWN_EXEC: &str = "flatpak-spawn";

/// Detect if the current execution environment is a Toolbx container.
///
/// Checks for the presence of the `.toolboxenv` files.
pub fn detect() -> bool {
    Path::new(TOOLBX_ENV).exists()
}

/// Executes given command on the host.
///
/// Takes as argument a slice of the command that wasn't found. The command is then forwarded to
/// the host via `flatpak-spawn`.
pub fn run(args: &[&str]) -> Result<(), CnfError> {
    let flatpak_spawn = which::which(FLATPAK_SPAWN_EXEC)
        .map_err(|_| CnfError::NotFound(FLATPAK_SPAWN_EXEC.to_string()))?;

    let cfg = config::get();

    // Assemble arguments
    let mut hostargs = vec!["--host"];
    hostargs.push(args[0]);
    if args[0] == "sudo" {
        // Needed for sudo to correctly query passwords
        hostargs.push("-S");
    }
    for s in &args[1..] {
        hostargs.push(s.as_ref());
    }

    if cfg.verbose {
        println!("Args: {:?}", hostargs);
        println!(
            "Running command '{}' on host via '{}'",
            &args[0],
            flatpak_spawn.display()
        );
    }

    let status = Command::new(flatpak_spawn)
        .args(&hostargs)
        .status()
        .map_err(CnfError::ApplicationError)?;
    if status.success() {
        Ok(())
    } else {
        Err(CnfError::Execution {
            env: "host".to_string(),
            error: status,
        })
    }
}
