//! Configuration handler for CNF
//!
//! Reads the user configuration from the configuration file `~/.config/cnf/cnf.toml` and
//! deserializes it into a struct for convenient handling.
use serde_derive::{Deserialize, Serialize};

pub static APP_NAME: &str = "cnf";

/// Configuration struct.
#[derive(Serialize, Deserialize, Debug)]
pub struct Config {
    /// Name of the Toolbx container to forward commands to that weren't found on the host.
    pub toolbx_name: String,
    /// Set to true to print verbose command outputs.
    pub verbose: bool,
}

/// Default configuration.
impl std::default::Default for Config {
    fn default() -> Self {
        Self {
            toolbx_name: "".into(),
            verbose: false,
        }
    }
}

/// Read program configuration.
///
/// Recovers from errors caused by invalid TOML formatting by printing an error message and
/// returning the default config instead.
pub fn get() -> Config {
    match confy::load(APP_NAME) {
        Ok(c) => c,
        Err(e) => match e {
            confy::ConfyError::BadTomlData(_) => {
                eprintln!("Bad TOML data found in config, please regenerate config. Using default instead.");
                Config::default()
            }
            _ => panic!("Error '{}' reading config", e),
        },
    }
}
