//! Test linking/unlinking and connecting/disconnecting nodes.
use cnetworks::*;

#[test]
fn test_link() {
    let mut net = Network::new(10, Model::None, Weight::default());

    assert!(net.links().count() == 0);
    for i in net.nodes() {
        for j in net.nodes() {
            assert!(net.get_link(i, j).unwrap_or(None).is_none())
        }
    }
    for node in net.nodes() {
        assert!(net.links_of(node).unwrap().is_empty());
    }

    net.link(1, 2).unwrap();
    assert!(net.links().count() == 1);
    assert!((net.get_link(1, 2).unwrap().unwrap() - 1.0).abs() < f64::EPSILON);
    assert!((net.get_link(2, 1).unwrap().unwrap() - 1.0).abs() < f64::EPSILON);

    net.link_exact(2, 3, 0.25).unwrap();
    assert!(net.links().count() == 2);
    assert!((net.get_link(2, 3).unwrap().unwrap() - 0.25).abs() < f64::EPSILON);
    assert!((net.get_link(3, 2).unwrap().unwrap() - 0.25).abs() < f64::EPSILON);

    assert_eq!(net.link(1, 100).unwrap_err(), cn::Err::NoSuchNode(100));
    assert_eq!(net.get_link(1, 100).unwrap_err(), cn::Err::NoSuchNode(100));
    assert!(net.links().count() == 2);
}

#[test]
fn test_unlink() {
    let mut net = Network::new(3, Model::None, Weight::default());
    assert!(net.links().count() == 0);

    net.link(0, 1).unwrap();
    assert!(net.links().count() == 1);
    assert_eq!(net.unlink_safe(0, 1).unwrap(), None);
    assert!((net.unlink(0, 1).unwrap().unwrap() - 1.0).abs() < f64::EPSILON);

    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.get_link(1, 0).unwrap(), None);
    assert!(net.links().count() == 0);
    assert!(!net.links().any(|(i, j)| i == 0 && j == 1 || i == 1 && j == 0));

    assert_eq!(net.unlink(0, 100).unwrap_err(), cn::Err::NoSuchNode(100));
    assert_eq!(
        net.unlink_safe(0, 100).unwrap_err(),
        cn::Err::NoSuchNode(100)
    );

    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(0, 2).unwrap();
    assert!(net.links().count() == 3);
    assert!((net.unlink_safe(0, 1).unwrap().unwrap() - 1.0).abs() < f64::EPSILON);

    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.get_link(1, 0).unwrap(), None);
    assert!(net.links().count() == 2);
    assert!(!net.links().any(|(i, j)| i == 0 && j == 1 || i == 1 && j == 0));
}

#[test]
fn test_disconnect() {
    let mut net = Network::new(3, Model::None, Weight::default());
    let idxes: Vec<_> = net.nodes().collect();
    for &i in &idxes {
        for &j in &idxes {
            // Ignore the "self link" errors
            net.link(i, j).unwrap_or(None);
        }
    }
    assert_eq!(net.links().count(), 3);

    for (target, weight) in net.disconnect_safe(0).unwrap().unwrap() {
        assert!(vec![1, 2].contains(&target));
        assert!((weight - 1.0).abs() < f64::EPSILON);
    }
    assert_eq!(net.links().count(), 1);

    assert_eq!(net.get_link(0, 1).unwrap(), None);
    assert_eq!(net.disconnect(0).unwrap(), cn::IndexMap::default());
    assert!((net.disconnect(1).unwrap()[&2] - 1.0).abs() < f64::EPSILON);
    assert_eq!(net.disconnect(2).unwrap(), cn::IndexMap::default());
    assert_eq!(net.disconnect(100).unwrap_err(), cn::Err::NoSuchNode(100));
}

#[test]
fn test_attach_detach() {
    let mut net = Network::new(0, Model::None, Weight::default());
    assert_eq!(net.size(), 0);

    for i in [0, 1, 2] {
        net.attach(i).unwrap();
    }
    assert_eq!(net.size(), 3);
    assert_eq!(net.size_connected(), 0);

    for i in 0..3 {
        assert!(net.exists(i));
    }

    assert_eq!(net.attach(0).unwrap_err(), cn::Err::NodeExists(0));

    net.detach(2).unwrap();
    assert_eq!(net.size(), 2);
    assert_eq!(net.nodes().collect::<Vec<_>>(), vec![0, 1]);
    for i in 0..2 {
        assert!(net.exists(i));
    }
    assert!(!net.exists(2));
}
