//! Test common network algorithms.
use cnetworks::*;

#[test]
fn test_components() {
    let mut net = Network::new(10, Model::None, Weight::default());
    assert_eq!(net.components().len(), net.size());
    net.link(0, 1).unwrap();
    net.link(0, 2).unwrap();
    net.link(0, 3).unwrap();
    net.link(4, 5).unwrap();
    net.link(4, 6).unwrap();
    net.link(4, 7).unwrap();
    net.link(4, 8).unwrap();
    assert_eq!(net.components().len(), 3);
    let component = net.into_largest_component();
    assert_eq!(component.components().len(), 1);
    assert_eq!(component.nodes().collect::<Vec<_>>(), vec![4, 5, 6, 7, 8]);
}

#[test]
fn test_stitch() {
    let mut net = Network::new(4, Model::None, Weight::default());
    net.stitch_together();
    for start in [0, 1, 2, 3] {
        let (_, unexpl) = bfs::explore(&net, start).unwrap();
        assert!(unexpl.is_empty());
    }
}

#[test]
fn test_hide_links() {
    let mut net = Network::with_seed(
        100,
        Model::BA { m0: 4, m: 4 },
        Weight::default(),
        "test seed",
    );
    let initial_links = net.links().count() as f64;
    assert!((misc::hide_links(&mut net, 0.2, true) - 0.2).abs() < 0.01);
    assert_eq!(net.links().count(), (initial_links * 0.8) as usize);
    assert!(net.is_whole());
    assert!((misc::hide_links(&mut net, 0.2, false) - 0.2).abs() < 0.01);
    assert_eq!(
        net.links().count(),
        (initial_links * 0.8 * 0.8).round() as usize
    );
    misc::hide_links(&mut net, 0.95, false);
    assert!(!net.is_whole());
}

#[test]
fn test_hide_nodes() {
    let mut net = Network::with_seed(
        100,
        Model::ER {
            p: 8.0 / 100.0,
            whole: true,
        },
        Weight::default(),
        "test seed",
    );
    assert!((misc::hide_nodes(&mut net, 0.2, &[], true) - 0.2).abs() < 0.01);
    assert_eq!(net.size(), (100.0 * 0.8) as usize);
    assert!(net.is_whole());
    assert!((misc::hide_nodes(&mut net, 0.2, &[], false) - 0.2).abs() < 0.01);
    assert_eq!(net.size(), (100.0 * 0.8 * 0.8) as usize);
    misc::hide_nodes(&mut net, 0.75, &[], false);
    assert!(!net.is_whole());
}

#[test]
fn test_init_from_file() {
    let fname = "tests/test_network";
    let net = misc::init_from_file(fname, Weight::Constant { c: 0.5 });
    assert_eq!(net.size(), 4);
    assert_eq!(net.links().count(), 3);
    assert!(net.get_link(0, 1).unwrap_or(None).is_some());
    assert!(net.get_link(2, 5).unwrap_or(None).is_some());
    assert!(net.get_link(1, 2).unwrap_or(None).is_some());
}
