use crate::{cn, Model, Network, Weight};
use rand::prelude::*;
use std::fs::File;
use std::io::{BufRead, BufReader};
use std::path::Path;

pub fn hide_links(net: &mut Network, h_e: f64, ensure_whole: bool) -> f64 {
    let mut possible: cn::IndexSet<_> = net.links().collect();
    let initial_links = possible.len() as f64;
    let mut to_remove = (h_e * initial_links).round() as usize;
    while to_remove > 0 {
        if let Some(&(i, j)) = possible.iter().choose(&mut *net.rng().unwrap()) {
            possible.remove(&(i, j));
            if !ensure_whole {
                net.unlink(i, j).unwrap();
                to_remove -= 1;
            } else if net.unlink_safe(i, j).unwrap().is_some() {
                to_remove -= 1;
            }
        } else {
            break;
        }
    }
    1.0 - net.links().count() as f64 / initial_links
}

pub fn hide_nodes(net: &mut Network, h_v: f64, exclude: &[usize], ensure_whole: bool) -> f64 {
    let initial_nodes = net.size() as f64;
    let mut to_remove = (h_v * initial_nodes).round() as usize;
    let mut possible: cn::IndexSet<_> =
        net.nodes().filter(|node| !exclude.contains(node)).collect();
    while to_remove > 0 {
        if let Some(&i) = possible.iter().choose(&mut *net.rng().unwrap()) {
            possible.remove(&i);
            if !ensure_whole || net.disconnect_safe(i).unwrap().is_some() {
                net.detach(i).unwrap();
                to_remove -= 1;
            }
        } else {
            break;
        }
    }
    1.0 - net.size() as f64 / initial_nodes
}

pub fn init_from_file(model_file: &str, weight: Weight) -> Network {
    let model_file = Path::new(model_file);
    let name = model_file
        .file_name()
        .map(|n| n.to_str().unwrap().to_owned())
        .unwrap();
    let file = File::open(model_file).unwrap();
    let reader = BufReader::new(file);
    let mut net = Network::new(0, Model::Custom(name), weight);

    for line in reader.lines() {
        let line: Vec<String> = line.unwrap().split(' ').map(|s| s.to_string()).collect();
        let i: usize = line[0].parse().expect("Cannot parse number");
        let j: usize = line[1].parse().expect("Cannot parse number");
        net.attach_skip(&[i, j]);
        net.link(i, j).unwrap();
    }
    net
}
