/*!
This is a library of tools for creating and manipulating complex networks.

From [Wikipedia](https://en.wikipedia.org/wiki/Complex_network):

> Complex network is a graph-like structure with non-trivial characteristics.
> A complex network is a graph (network) with non-trivial topological features—features that do not
> occur in simple networks such as lattices or random graphs but often occur in networks representing
> real systems. The study of complex networks is a young and active area of scientific
> research inspired largely by empirical findings of real-world networks such
> as computer networks, biological networks, technological networks, brain networks, climate networks
> and social networks.
*/
mod network;
mod model;
mod weight;

pub mod bfs;
pub mod locate;
pub mod si;

pub use network::Network;
pub use weight::Weight;
pub use model::Model;

pub mod cn {
    //! Library-specific utility module.
    //!
    //! This module serves as a namespace for the various utilities with generic names.

    use std::hash::BuildHasherDefault as Builder;

    /// Re-export of
    /// [`rustc_hash::FxHasher`](https://docs.rs/rustc-hash/1/rustc_hash/struct.FxHasher.html).
    pub type FxHasher = rustc_hash::FxHasher;

    /// Variant of [`indexmap::IndexMap`] using [`FxHasher`].
    pub type IndexMap<K, V> = indexmap::IndexMap<K, V, Builder<FxHasher>>;

    /// Variant of [`std::collections::HashMap`] using [`FxHasher`].
    pub type HashMap<K, V> = std::collections::HashMap<K, V, Builder<FxHasher>>;

    /// Variant of [`std::collections::HashSet`] using [`FxHasher`].
    pub type HashSet<V> = std::collections::HashSet<V, Builder<FxHasher>>;

    /// Shorthand trait for **exact size**, **double-ended** iterator.
    pub trait Iter<I>: DoubleEndedIterator<Item = I> + ExactSizeIterator<Item = I> {}
    impl<T: DoubleEndedIterator + ExactSizeIterator + Iterator<Item = I>, I> Iter<I> for T {}

    /// Error enum used for various recoverable errors
    #[derive(Debug, PartialEq)]
    pub enum Err {
        /// Emitted whenever a change involving a non-existent node has been requested.
        NoSuchNode(usize),
        /// Emitted when the link weight is not in `(0, 1]`.
        BadWeight(f64),
        /// Emitted when an attempt is made to link a node to itself.
        LinkToSelf(usize),
        /// Emitted when constant probability is not in `(0, 1]`.
        BadProbability(f64),
        /// Emitted when an attempt is made to attach a node which already exists.
        NodeExists(usize),
        /// Emitted when an attempt is made to run an algorithm requiring at least 1 target without
        /// targets.
        NoTarget,
    }

    /// A variant of [`std::result::Result`] which returns [`cn::Err`](crate::cn::Err).
    pub type Result<T> = std::result::Result<T, Err>;

    impl std::fmt::Display for Err {
        fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
            let message = match self {
                Err::NoSuchNode(i) => format!("No such node {}", i),
                Err::BadWeight(w) => format!("Link weight cannot be {}", w),
                Err::LinkToSelf(i) => format!("Attempted linking node {} to itself", i),
                Err::BadProbability(p) => format!("Probability cannot be {}", p),
                Err::NodeExists(i) => format!("Node {} already exists", i),
                Err::NoTarget => "No search targets were specified".to_string(),
            };
            write!(f, "{}", message)
        }
    }

    impl std::error::Error for Err {}
}
