//! Test linking/unlinking and connecting/disconnecting nodes.
use cnetworks::*;
use rustc_hash::FxHashSet as HashSet;

#[test]
fn test_bfs() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    assert_eq!(bfs(&net, 0, 0).unwrap(), Some(0),);

    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(bfs(&net, 0, 100).unwrap_err(), cn::Err::NoSuchNode(100));

    assert_eq!(bfs(&net, 0, 2).unwrap(), Some(2),);

    net.unlink(0, 1).unwrap();
    assert_eq!(bfs(&net, 0, 2).unwrap(), None);
}

#[test]
fn test_bfs_many() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    assert_eq!(
        bfs_many(&net, 0, &[1, 100]).unwrap_err(),
        cn::Err::NoSuchNode(100)
    );

    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(2, 3).unwrap();

    assert_eq!(
        bfs_many(&net, 0, &[1, 2, 3]).unwrap(),
        vec![(1, Some(1)), (2, Some(2)), (3, Some(3))],
    );

    net.unlink(1, 2).unwrap();
    assert_eq!(
        bfs_many(&net, 0, &[1, 2, 3]).unwrap(),
        vec![(1, Some(1)), (2, None), (3, None)],
    );

    assert_eq!(bfs(&net, 0, 2).unwrap(), None);
}

#[test]
fn test_bfs_callback() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();

    let mut visited: Vec<usize> = Vec::new();

    assert_eq!(
        bfs_callback(&net, 0, &[1, 100], |current, _, _| visited.push(current)).unwrap_err(),
        cn::Err::NoSuchNode(100)
    );
    assert_eq!(visited, Vec::new());

    assert_eq!(
        bfs_callback(&net, 0, &[1, 2], |current, _, _| visited.push(current)).unwrap(),
        vec![(1, Some(1)), (2, Some(2))],
    );
    assert_eq!(visited, vec![0, 1, 2]);
}

#[test]
fn test_bfs_build_tree() {
    let mut net = Network::new(6, Model::None, Weight::Constant { c: 1.0 });
    assert_eq!(bfs_build_tree(&net, 0, &[1, 100]).unwrap_err(), cn::Err::NoSuchNode(100));

    for i in [1, 2, 3] {
        net.link(0, i).unwrap();
    }
    for i in [1, 2, 4, 5] {
        net.link(3, i).unwrap();
    }

    let tree = bfs_build_tree(&net, 0, &[]).unwrap();
    for i in tree.indexes() {
        let mut links = tree[i].links().keys().cloned().collect::<Vec<usize>>();
        links.sort_unstable();
        match i {
            0 => assert_eq!(links, vec![1, 2, 3]),
            1 | 2 => assert_eq!(links, vec![0]),
            3 => assert_eq!(links, vec![0, 4, 5]),
            4 | 5 => assert_eq!(links, vec![3]),
            _ => panic!("There should not be a node with index {}", i),
        }
    }
}

#[test]
fn test_explore() {
    let mut net = Network::new(3, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    for node in [0, 1, 2] {
        assert_eq!(
            explore(&net, node).unwrap(),
            (vec![0, 1, 2].into_iter().collect(), HashSet::default())
        );
    }
    net.unlink(1, 2).unwrap();
    for node in [0, 1] {
        assert_eq!(
            explore(&net, node).unwrap(),
            (
                vec![0, 1].into_iter().collect(),
                vec![2].into_iter().collect()
            )
        );
    }
    assert_eq!(
        explore(&net, 2).unwrap(),
        (
            vec![2].into_iter().collect(),
            vec![1, 0].into_iter().collect()
        )
    );
}

#[test]
fn test_clusters() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    assert_eq!(
        clusters(&net),
        vec![
            vec![0, 1, 2].into_iter().collect(),
            vec![3].into_iter().collect()
        ]
    );
    net.link(1, 3).unwrap();
    let clst = clusters(&net);
    assert_eq!(clst, vec![vec![0, 1, 2, 3].into_iter().collect()]);
}

#[test]
fn test_spread_si() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    net.link(0, 1).unwrap();
    net.link(1, 2).unwrap();
    net.link(2, 3).unwrap();
    let root = spread_si(&mut net, 1);
    let mut infected = net
        .flagged("INFECTED")
        .keys()
        .cloned()
        .collect::<Vec<usize>>();
    infected.sort_unstable();
    match root {
        0 => assert_eq!(infected, vec![0, 1]),
        1 => assert_eq!(infected, vec![0, 1, 2]),
        2 => assert_eq!(infected, vec![1, 2, 3]),
        3 => assert_eq!(infected, vec![2, 3]),
        _ => panic!("Root should not be {}", root),
    }
    net.clear_flag("INFECTED");
    let root = spread_si(&mut net, 3);
    assert!(vec![0, 1, 2, 3].contains(&root));
    let mut infected = net
        .flagged("INFECTED")
        .keys()
        .cloned()
        .collect::<Vec<usize>>();
    infected.sort_unstable();
    assert_eq!(infected, vec![0, 1, 2, 3])
}

#[test]
fn test_stitch() {
    let mut net = Network::new(4, Model::None, Weight::Constant { c: 1.0 });
    stitch_together(&mut net);
    for start in [0, 1, 2, 3] {
        let (_, unexpl) = explore(&net, start).unwrap();
        assert!(unexpl.is_empty());
    }
}
